;;; GNU Guix --- Functional package management for GNU
;;; Copyright © 2025 Hilton Chain <hako@ultrarare.space>
;;;
;;; This file is part of GNU Guix.
;;;
;;; GNU Guix is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; GNU Guix is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Guix.  If not, see <http://www.gnu.org/licenses/>.

(define-module (gnu packages rust-crates)
  #:use-module (guix gexp)
  #:use-module (guix packages)
  #:use-module (guix download)
  #:use-module (guix git-download)
  #:use-module (guix build-system cargo)
  #:use-module (gnu packages rust-sources)
  #:export (lookup-cargo-inputs))

;;;
;;; This file is managed by ‘guix import’.  Do NOT add definitions manually.
;;;
;;; Rust libraries fetched from crates.io and non-workspace development
;;; snapshots.
;;;

(define qqqq-separator 'begin-of-crates)

(define rust-ab-glyph-rasterizer-0.1.8
  (crate-source "ab_glyph_rasterizer" "0.1.8"
                "0ikhgzig59q8b1a1iw83sxfnvylg5gx6w2y8ynbnf231xs9if6y7"))

(define rust-abscissa-core-0.8.2
  (crate-source "abscissa_core" "0.8.2"
                "0fvpm79dnwh1lj5kpw1bs3sfs8drcwbvbic6vrmjsh34v1x1i0rh"))

(define rust-abscissa-derive-0.8.2
  (crate-source "abscissa_derive" "0.8.2"
                "1qkzz20v71y26id2sfcdfc3jhgzf4gihf6g07x1xmx1f3mi19n88"))

(define rust-accesskit-0.21.0
  (crate-source "accesskit" "0.21.0"
                "0z7287v7vczkd1nhr5gx782cvxbdiigcklwv8fw9g5bgdsnr01lw"))

(define rust-accesskit-atspi-common-0.14.0
  (crate-source "accesskit_atspi_common" "0.14.0"
                "01r65z92h3zpxpka62136bm5kzvx10i0797v1sp3qvc9jgh13dcg"))

(define rust-accesskit-consumer-0.30.0
  (crate-source "accesskit_consumer" "0.30.0"
                "0f2hd7dah3jq95wrbi0xx1hj8q3bnik4a69ah1y79dxf3d65f9zc"))

(define rust-accesskit-unix-0.17.0
  (crate-source "accesskit_unix" "0.17.0"
                "1x0mfn5fga1flnhyn70pzba707kwbf3dbjkaxar5pjjc2hbgpfra"))

(define rust-actix-codec-0.5.2
  (crate-source "actix-codec" "0.5.2"
                "12m2jxysk2xpxi193340zv4w215cv9fyyna7rxvzh6wck0hhlysz"))

(define rust-actix-http-3.10.0
  (crate-source "actix-http" "3.10.0"
                "1jblgim18lhbbxyi3myfky9lc0z7a5066b0mwi2n15k7ddjq5a0g"))

(define rust-actix-macros-0.2.4
  (crate-source "actix-macros" "0.2.4"
                "1jsmhq9k5nsms8sci2lqkihym5nrhlpfv8dgd0n4539g1cad67p0"))

(define rust-actix-router-0.5.3
  (crate-source "actix-router" "0.5.3"
                "1y1n086zgfgf6483vlm18651n5ga6rcvlwvynmkkixji9hb29lqk"))

(define rust-actix-rt-2.10.0
  (crate-source "actix-rt" "2.10.0"
                "022jj938jdhs3r0xg0yg1vdbblsjw0m8lhxcam7alhp0lvia9v94"))

(define rust-actix-server-2.5.1
  (crate-source "actix-server" "2.5.1"
                "092ycxvlpc8vvv521zvbc70gvddvxw0mg5h7mxl4fkr8si7rg633"))

(define rust-actix-service-2.0.3
  (crate-source "actix-service" "2.0.3"
                "0zyw2178kcyy3r775klvkb3riai1sjxnzcxxqjyl9bz5y1mz6ily"))

(define rust-actix-utils-3.0.1
  (crate-source "actix-utils" "3.0.1"
                "1n05nzwdkx6jhmzr6f9qsh57a8hqlwv5rjz1i0j3qvj6y7gxr8c8"))

(define rust-actix-web-4.10.2
  (crate-source "actix-web" "4.10.2"
                "0kcv8z8hgwxmhpzgp4hy5c0apm29k2g3hcj0dscyvin67mdv3qzj"))

(define rust-actix-web-codegen-4.3.0
  (crate-source "actix-web-codegen" "4.3.0"
                "1f12ss948vpanh98a1v7f2x893g7xfh1mpgiz9fhnjb85q73i4gm"))

(define rust-adblock-0.9.2
  (crate-source "adblock" "0.9.2"
                "14k577a3xj2l973xvll884p1ixj9j42q1l1wwdxna28g0aw42c1q"))

(define rust-addr-0.15.6
  (crate-source "addr" "0.15.6"
                "1vrbnzv4r7sx65mmi011aiafc7b1866iywnchx8asc72vd0qlfx9"))

(define rust-addr2line-0.21.0
  (crate-source "addr2line" "0.21.0"
                "1jx0k3iwyqr8klqbzk6kjvr496yd94aspis10vwsj5wy7gib4c4a"))

(define rust-addr2line-0.22.0
  (crate-source "addr2line" "0.22.0"
                "0y66f1sa27i9kvmlh76ynk60rxfrmkba9ja8x527h32wdb206ibf"))

(define rust-addr2line-0.24.2
  (crate-source "addr2line" "0.24.2"
                "1hd1i57zxgz08j6h5qrhsnm2fi0bcqvsh389fw400xm3arz2ggnz"))

(define rust-addr2line-0.25.1
  (crate-source "addr2line" "0.25.1"
                "0jwb96gv17vdr29hbzi0ha5q6jkpgjyn7rjlg5nis65k41rk0p8v"))

(define rust-adler-1.0.2
  (crate-source "adler" "1.0.2"
                "1zim79cvzd5yrkzl3nyfx0avijwgk9fqv3yrscdy1cc79ih02qpj"))

(define rust-adler2-2.0.0
  (crate-source "adler2" "2.0.0"
                "09r6drylvgy8vv8k20lnbvwq8gp09h7smfn6h1rxsy15pgh629si"))

(define rust-adler2-2.0.1
  (crate-source "adler2" "2.0.1"
                "1ymy18s9hs7ya1pjc9864l30wk8p2qfqdi7mhhcc5nfakxbij09j"))

(define rust-adler32-1.2.0
  (crate-source "adler32" "1.2.0"
                "0d7jq7jsjyhsgbhnfq5fvrlh9j0i9g1fqrl2735ibv5f75yjgqda"))

(define rust-aead-0.5.2
  (crate-source "aead" "0.5.2"
                "1c32aviraqag7926xcb9sybdm36v5vh9gnxpn4pxdwjc50zl28ni"))

(define rust-aes-0.7.5
  (crate-source "aes" "0.7.5"
                "1f0sdx2fsa8w3l7xzsyi9ry3shvnnsgc0znh50if9fm95vslg2wy"
                #:snippet '(delete-file-recursively "tests")))

(define rust-aes-0.8.4
  (crate-source "aes" "0.8.4"
                "1853796anlwp4kqim0s6wm1srl4ib621nm0cl2h3c8klsjkgfsdi"
                #:snippet '(delete-file-recursively "tests")))

(define rust-aes-gcm-0.10.3
  (crate-source "aes-gcm" "0.10.3"
                "1lgaqgg1gh9crg435509lqdhajg1m2vgma6f7fdj1qa2yyh10443"
                #:snippet '(delete-file-recursively "tests")))

(define rust-ahash-0.7.8
  (crate-source "ahash" "0.7.8"
                "1y9014qsy6gs9xld4ch7a6xi9bpki8vaciawxq4p75d8qvh7f549"))

(define rust-ahash-0.8.11
  (crate-source "ahash" "0.8.11"
                "04chdfkls5xmhp1d48gnjsmglbqibizs3bpbj6rsj604m10si7g8"))

(define rust-ahash-0.8.12
  (crate-source "ahash" "0.8.12"
                "0xbsp9rlm5ki017c0w6ay8kjwinwm8knjncci95mii30rmwz25as"))

(define rust-ahash-0.8.3
  (crate-source "ahash" "0.8.3"
                "0bzcsxdl2wd6j2p4214qh9sqkqn69gi7f9lk1xi8yj063r6zd69c"))

(define rust-aho-corasick-0.7.20
  (crate-source "aho-corasick" "0.7.20"
                "1b3if3nav4qzgjz9bf75b2cv2h2yisrqfs0np70i38kgz4cn94yc"))

(define rust-aho-corasick-1.1.1
  (crate-source "aho-corasick" "1.1.1"
                "1aqqalh66jygy54fbnpglzrb9dwlrvn6zl1nhncdvynl8w376pga"))

(define rust-aho-corasick-1.1.3
  (crate-source "aho-corasick" "1.1.3"
                "05mrpkvdgp5d20y2p989f187ry9diliijgwrs254fs9s1m1x6q4f"))

(define rust-alga-0.9.3
  (crate-source "alga" "0.9.3"
                "1wl4z8ini9269x04g8wwdz1nn3hmmvaaysq4jwhymikyg81kv0jg"))

(define rust-aliasable-0.1.3
  (crate-source "aliasable" "0.1.3"
                "1z8548zdjlm4ps1k0d7x68lfdyji02crwcc9rw3q3bb106f643r5"))

(define rust-aligned-0.4.2
  (crate-source "aligned" "0.4.2"
                "08srg0rbz21s1kk97b1canffly7icr31sp7l1nqk2i1ym05lqzip"))

(define rust-aligned-vec-0.5.0
  (crate-source "aligned-vec" "0.5.0"
                "1lb8qjqfap028ylf8zap6rkwrnrqimc3v6h3cixycjrdx1y0vaaa"))

(define rust-aligned-vec-0.6.4
  (crate-source "aligned-vec" "0.6.4"
                "16vnf78hvfix5cwzd5xs5a2g6afmgb4h7n6yfsc36bv0r22072fw"))

(define rust-alloc-no-stdlib-2.0.4
  (crate-source "alloc-no-stdlib" "2.0.4"
                "1cy6r2sfv5y5cigv86vms7n5nlwhx1rbyxwcraqnmm1rxiib2yyc"))

(define rust-alloc-stdlib-0.2.2
  (crate-source "alloc-stdlib" "0.2.2"
                "1kkfbld20ab4165p29v172h8g0wvq8i06z8vnng14whw0isq5ywl"))

(define rust-alloca-0.4.0
  (crate-source "alloca" "0.4.0"
                "1x6p4387rz6j7h342kp3b7bgvqzyl9mibf959pkfk9xflrgd19z5"))

(define rust-allocator-api2-0.2.18
  (crate-source "allocator-api2" "0.2.18"
                "0kr6lfnxvnj164j1x38g97qjlhb7akppqzvgfs0697140ixbav2w"))

(define rust-allocator-api2-0.2.21
  (crate-source "allocator-api2" "0.2.21"
                "08zrzs022xwndihvzdn78yqarv2b9696y67i6h78nla3ww87jgb8"))

(define rust-alphanumeric-sort-1.5.3
  (crate-source "alphanumeric-sort" "1.5.3"
                "13vyx3cqpylvc0md4563rd42b7dvk3fv4wny0kpcc48gy72n0z6n"))

(define rust-alsa-0.9.1
  (crate-source "alsa" "0.9.1"
                "0hvxc447bsynyhzhmznw6w2kwbid83p712dls4h1x8w3pavp4xgd"))

(define rust-alsa-sys-0.3.1
  (crate-source "alsa-sys" "0.3.1"
                "09qmmnpmlcj23zcgx2xsi4phcgm5i02g9xaf801y7i067mkfx3yv"))

(define rust-amd-0.2.2
  (crate-source "amd" "0.2.2"
                "15ib7hhzdifb2m099561ip0fq627gaszw4sq36ys75snaw0y0yd6"))

(define rust-ammonia-4.0.0
  (crate-source "ammonia" "4.0.0"
                "0nd432yg1cl9kj4i9c37a9hvwffayqh6zsvb4fmh31g5bsp9xf8s"))

(define rust-ammonia-4.1.2
  (crate-source "ammonia" "4.1.2"
                "1rnx4v8qiw4nvqzvnnbagb0lp5cc9q9q1ja6cys2888sgq4i7s8p"))

(define rust-amplify-4.9.0
  (crate-source "amplify" "4.9.0"
                "0c9v4whgyh08a6n8jv4bjnyab8hjc6dkjph1wyl587l8gjnb8zrz"))

(define rust-amplify-derive-4.0.1
  (crate-source "amplify_derive" "4.0.1"
                "0a6lwinr627v4j547xm0494vk0rm17xaidsrz6wkd6yqp3k0jqra"))

(define rust-amplify-num-0.5.3
  (crate-source "amplify_num" "0.5.3"
                "1bg9h12p9cm3vs80pngpzh6lc3wcjv1jy1276mrpy14255dbgg4r"))

(define rust-amplify-syn-2.0.1
  (crate-source "amplify_syn" "2.0.1"
                "0annky4dq5h5wg58pkbm0d3j07jndbgl9b2vic4q639w8y6zndkp"))

(define rust-ammonia-4.1.1
  (crate-source "ammonia" "4.1.1"
                "13wpycwpx0c5qxqlf6mvd7pbqr9hw2gqkgwavq2li0fh9mv4dcyn"))

(define rust-android-activity-0.6.0
  (crate-source "android-activity" "0.6.0"
                "0inh88x8x2fh62jg739s9hwyvdh8i920qf0qw7bhr802j9c7hsgg"))

(define rust-android-properties-0.2.2
  (crate-source "android-properties" "0.2.2"
                "016slvg269c0y120p9qd8vdfqa2jbw4j0g18gfw6p3ain44v4zpw"))

(define rust-android-system-properties-0.1.5
  (crate-source "android_system_properties" "0.1.5"
                "04b3wrz12837j7mdczqd95b732gw5q7q66cv4yn4646lvccp57l1"))

(define rust-android-tzdata-0.1.1
  (crate-source "android-tzdata" "0.1.1"
                "1w7ynjxrfs97xg3qlcdns4kgfpwcdv824g611fq32cag4cdr96g9"))

(define rust-anes-0.1.6
  (crate-source "anes" "0.1.6"
                "16bj1ww1xkwzbckk32j2pnbn5vk6wgsl3q4p3j9551xbcarwnijb"))

(define rust-annotate-snippets-0.11.5
  (crate-source "annotate-snippets" "0.11.5"
                "1i1bmr5vy957l8fvivj9x1xs24np0k56rdgwj0bxqk45b2p8w3ki"))

(define rust-annotate-snippets-0.6.1
  (crate-source "annotate-snippets" "0.6.1"
                "19x7ldklprdgf8pam8b3lfhrxqw5yldcvk5j0bw2agsajbj1q0n7"))

(define rust-annotate-snippets-0.9.2
  (crate-source "annotate-snippets" "0.9.2"
                "07p8r6jzb7nqydq0kr5pllckqcdxlyld2g275v425axnzffpxbyc"))

(define rust-ansi-colours-1.2.3
  (crate-source "ansi_colours" "1.2.3"
                "1zimwh84gs1r0g0chy6x5lm9v0ksxxlzwy8nyj80f6cq08zc9vhl"))

(define rust-ansi-escapes-0.1.1
  (crate-source "ansi-escapes" "0.1.1"
                "1byp22h6cnq5wf5pdg2vph0j384ggxi8kdilayczakg2mam0sg3y"))

(define rust-ansi-parser-0.6.5
  (crate-source "ansi-parser" "0.6.5"
                "152idb8a6gwdxzj6m099h3xgx8vw0sjc6skgw94nm2k3y5swc6kn"))

(define rust-ansi-str-0.8.0
  (crate-source "ansi-str" "0.8.0"
                "07ddhqynv05xjyhw295w29qy77fi84sh5p2mm46ap0d94s4mgx0w"))

(define rust-ansi-str-0.9.0
  (crate-source "ansi-str" "0.9.0"
                "03c9j3870slj40lkdkrpav2p4kig2f1g6x42n8267x397d2y2386"))

(define rust-ansi-term-0.11.0
  (crate-source "ansi_term" "0.11.0"
                "16wpvrghvd0353584i1idnsgm0r3vchg8fyrm0x8ayv1rgvbljgf"))

(define rust-ansi-term-0.12.1
  (crate-source "ansi_term" "0.12.1"
                "1ljmkbilxgmhavxvxqa7qvm6f3fjggi7q2l3a72q9x0cxjvrnanm"))

(define rust-ansi-to-html-0.1.3
  (crate-source "ansi-to-html" "0.1.3"
                "18kwlgr3vfsij8gvl7vxw11yl628b1s8z2pldh73z4zzq2693gf7"))

(define rust-ansi-to-tui-7.0.0
  (crate-source "ansi-to-tui" "7.0.0"
                "0b4iynqcqaav8i55w8lk7ypm6xr845vh32lcw8vxffff3qgmwmb7"))

(define rust-ansi-width-0.1.0
  (crate-source "ansi-width" "0.1.0"
                "023693dwvci91456nfi0aygc44i1f098l2f23ssq67b1ybk3r7i1"))

(define rust-ansiterm-0.12.2
  (crate-source "ansiterm" "0.12.2"
                "1k14pywvgd829swxzji0nchk4n6yvr9xz6lkwv96v8ax77sqgdaa"))

(define rust-ansitok-0.2.0
  (crate-source "ansitok" "0.2.0"
                "10vc2d1325qsbvbnqnj48zg55wv7jz929drx9vpdscdvl7k48012"))

(define rust-ansitok-0.3.0
  (crate-source "ansitok" "0.3.0"
                "1vjrlvmwrq5v72rcmfqhdyspvabcz5mx531am7q6071gikmara60"))

(define rust-anstream-0.3.2
  (crate-source "anstream" "0.3.2"
                "0qzinx9c8zfq3xqpxzmlv6nrm3ymccr4n8gffkdmj31p50v4za0c"))

(define rust-anstream-0.6.18
  (crate-source "anstream" "0.6.18"
                "16sjk4x3ns2c3ya1x28a44kh6p47c7vhk27251i015hik1lm7k4a"))

(define rust-anstream-0.6.19
  (crate-source "anstream" "0.6.19"
                "0crr9a207dyn8k66xgvhvmlxm9raiwpss3syfa35c6265s9z26ih"))

(define rust-anstream-0.6.20
  (crate-source "anstream" "0.6.20"
                "14k1iqdf3dx7hdjllmql0j9sjxkwr1lfdddi3adzff0r7mjn7r9s"))

(define rust-anstyle-1.0.1
  (crate-source "anstyle" "1.0.1"
                "1kff80219d5rvvi407wky2zdlb0naxvbbg005s274pidbxfdlc1s"))

(define rust-anstyle-1.0.10
  (crate-source "anstyle" "1.0.10"
                "1yai2vppmd7zlvlrp9grwll60knrmscalf8l2qpfz8b7y5lkpk2m"))

(define rust-anstyle-1.0.11
  (crate-source "anstyle" "1.0.11"
                "1gbbzi0zbgff405q14v8hhpi1kz2drzl9a75r3qhks47lindjbl6"))

(define rust-anstyle-1.0.13
  (crate-source "anstyle" "1.0.13"
                "0y2ynjqajpny6q0amvfzzgw0gfw3l47z85km4gvx87vg02lcr4ji"))

(define rust-anstyle-lossy-1.1.3
  (crate-source "anstyle-lossy" "1.1.3"
                "16z58pw4z6ncypv6hl49sgcwxhy16saycqyg90x05lpzkrqzhkwk"))

(define rust-anstyle-parse-0.2.1
  (crate-source "anstyle-parse" "0.2.1"
                "0cy38fbdlnmwyy6q8dn8dcfrpycwnpjkljsjqn3kmc40b7zp924k"))

(define rust-anstyle-parse-0.2.6
  (crate-source "anstyle-parse" "0.2.6"
                "1acqayy22fwzsrvr6n0lz6a4zvjjcvgr5sm941m7m0b2fr81cb9v"))

(define rust-anstyle-parse-0.2.7
  (crate-source "anstyle-parse" "0.2.7"
                "1hhmkkfr95d462b3zf6yl2vfzdqfy5726ya572wwg8ha9y148xjf"))

(define rust-anstyle-query-1.0.0
  (crate-source "anstyle-query" "1.0.0"
                "0js9bgpqz21g0p2nm350cba1d0zfyixsma9lhyycic5sw55iv8aw"))

(define rust-anstyle-query-1.1.2
  (crate-source "anstyle-query" "1.1.2"
                "036nm3lkyk43xbps1yql3583fp4hg3b1600is7mcyxs1gzrpm53r"))

(define rust-anstyle-query-1.1.3
  (crate-source "anstyle-query" "1.1.3"
                "1sgs2hq54wayrmpvy784ww2ccv9f8yhhpasv12z872bx0jvdx2vc"))

(define rust-anstyle-query-1.1.4
  (crate-source "anstyle-query" "1.1.4"
                "1qir6d6fl5a4y2gmmw9a5w93ckwx6xn51aryd83p26zn6ihiy8wy"))

(define rust-anstyle-svg-0.1.7
  (crate-source "anstyle-svg" "0.1.7"
                "0ddvwk08qaihhy2y95a7ybmi6dnp9xg2zqdg9gm4kppnx54pjq6k"))

(define rust-anstyle-wincon-1.0.2
  (crate-source "anstyle-wincon" "1.0.2"
                "0k6gcx8qih4gzb5glrl37rqvj5qc893cvkkssdnjjm4iw02snxy6"))

(define rust-anstyle-wincon-3.0.10
  (crate-source "anstyle-wincon" "3.0.10"
                "0ajz9wsf46a2l3pds7v62xbhq2cffj7wrilamkx2z8r28m0k61iy"))

(define rust-anstyle-wincon-3.0.7
  (crate-source "anstyle-wincon" "3.0.7"
                "0kmf0fq4c8yribdpdpylzz1zccpy84hizmcsac3wrac1f7kk8dfa"))

(define rust-anstyle-wincon-3.0.8
  (crate-source "anstyle-wincon" "3.0.8"
                "1ykkvih20kykgfix7j8y74av90m2v8ji72hv373f8vmx659dx036"))

(define rust-anstyle-wincon-3.0.9
  (crate-source "anstyle-wincon" "3.0.9"
                "10n8mcgr89risdf35i73zc67aaa392bhggwzqlri1fv79297ags0"))

(define rust-any-ascii-0.1.7
  ;; TODO: Adapt the build scirpt to Guile.
  ;; https://github.com/anyascii/anyascii/blob/master/src/main/java/com/anyascii/build/gen/Rust.kt
  (crate-source "any_ascii" "0.1.7"
                "07kyb9is518jr1sbc6804kgg5pnx8djl328q3al28lcbxdvkf0vh"))

(define rust-any-ascii-0.3.2
  ;; TODO: Adapt the build scirpt to Guile.
  ;; https://github.com/anyascii/anyascii/blob/master/src/main/java/com/anyascii/build/gen/Rust.kt
  (crate-source "any_ascii" "0.3.2"
                "0b6c2vb3wlzcsd7s5l6s9fyq4r07a8yab9r7qvw474sbg95v2l7a"))

(define rust-anyhow-1.0.100
  (crate-source "anyhow" "1.0.100"
                "0qbfmw4hhv2ampza1csyvf1jqjs2dgrj29cv3h3sh623c6qvcgm2"))

(define rust-anyhow-1.0.72
  (crate-source "anyhow" "1.0.72"
                "0m08x5pwz8ndyjdxmyy41ix8mvjlrr5pihw0gdsapizch0nw64rv"))

(define rust-anyhow-1.0.75
  (crate-source "anyhow" "1.0.75"
                "1rmcjkim91c5mw7h9wn8nv0k6x118yz0xg0z1q18svgn42mqqrm4"))

(define rust-anyhow-1.0.97
  (crate-source "anyhow" "1.0.97"
                "0kvspbiwncmmkdgrwjrimsmbmhzxc641p5ql99l2rjq6smmdbznw"))

(define rust-anyhow-1.0.98
  (crate-source "anyhow" "1.0.98"
                "11ylvjdrcjs0q9jgk1af4r5cx1qppj63plxqkq595vmc24rjsvg1"))

(define rust-anyhow-1.0.99
  (crate-source "anyhow" "1.0.99"
                "001icqvkfl28rxxmk99rm4gvdzxqngj5v50yg2bh3dzcvqfllrxh"))

(define rust-anyinput-0.1.8
  (crate-source "anyinput" "0.1.8"
                "08dkks0pg097vcjj3a43fxzazs2cnmf1jd7kcj8s3y6lfxj80n12"))

(define rust-anyinput-core-0.1.8
  (crate-source "anyinput-core" "0.1.8"
                "17y761xzk0xy7n5q0d76fb29f40cd59qa3b4kv58g9n8k2qivks9"))

(define rust-anyinput-derive-0.1.8
  (crate-source "anyinput-derive" "0.1.8"
                "1xcgkdkskmh8xxxyr1chfapn9avk8d8s7jjph5zy4ff22dri4m7y"))

(define rust-aom-sys-0.3.3
  (crate-source "aom-sys" "0.3.3"
                "0bc1dzl3c95s44q7c1i0vnj7fhiqf44in8w22nw5vmp1vgbpadk2"))

(define rust-aperture-0.3.2
  (crate-source "aperture" "0.3.2"
                "02bjzskxp91br91yvf5f32wakp1i9948sxbsy9hdrxs52w38hr61"))

(define rust-app-dirs2-2.5.5
  (crate-source "app_dirs2" "2.5.5"
                "0bimzwj54fv8dvailkfb4l9sdsmnsn6hi183p76c3a736dbv7rx7"))

(define rust-appendlist-1.4.0
  (crate-source "appendlist" "1.4.0"
                "1lnbl7mc7capcqj1z1ylxvm4h492sb9sr8pzww3q6lrhrmrxqjg1"))

(define rust-approx-0.3.2
  (crate-source "approx" "0.3.2"
                "1hx580xjdxl3766js9b49rnbnmr8gw8c060809l43k9f0xshprph"))

(define rust-approx-0.4.0
  (crate-source "approx" "0.4.0"
                "0y52dg58lapl4pp1kqlznfw1blbki0nx6b0aw8kja2yi3gyhaaiz"))

(define rust-approx-0.5.1
  (crate-source "approx" "0.5.1"
                "1ilpv3dgd58rasslss0labarq7jawxmivk17wsh8wmkdm3q15cfa"))

(define rust-ar-0.9.0
  (crate-source "ar" "0.9.0"
                "0sdvvrf4inrkmrh6lzwg4z8x38b3gncbb8yqrgayqcd9d1yzfynn"))

(define rust-arbitrary-1.3.0
  (crate-source "arbitrary" "1.3.0"
                "0km5cj0sxfzv863blfjpz49mlikaxbaslyzk463i9jn1fgzril72"))

(define rust-arbitrary-1.4.1
  (crate-source "arbitrary" "1.4.1"
                "08zj2yanll5s5gsbmvgwvbq39iqzy3nia3yx3db3zwba08yhpqnx"))

(define rust-arbitrary-1.4.2
  (crate-source "arbitrary" "1.4.2"
                "1wcbi4x7i3lzcrkjda4810nqv03lpmvfhb0a85xrq1mbqjikdl63"))

(define rust-arboard-3.5.0
  (crate-source "arboard" "3.5.0"
                "0w1yqcx51153hy5w3y0702xjc9nmlhncw9f5l0rdwbl62pvj3py1"))

(define rust-arboard-3.6.1
  (crate-source "arboard" "3.6.1"
                "1byx6q5iipxkb0pyjp80k7c4akp4n5m7nsmqdbz4n7s9ak0a2j03"))

(define rust-arc-swap-1.7.1
  (crate-source "arc-swap" "1.7.1"
                "0mrl9a9r9p9bln74q6aszvf22q1ijiw089jkrmabfqkbj31zixv9"))

(define rust-archery-1.2.1
  (crate-source "archery" "1.2.1"
                "0sdqlmybcvd0rzv22ac3k3xxm5anr1gpm03sf02iy0jmrlhyvqpa"))

(define rust-arcstr-1.2.0
  (crate-source "arcstr" "1.2.0"
                "0vbyslhqr5fh84w5dd2hqck5y5r154p771wqddfah0bpplyqr483"))

(define rust-arg-enum-proc-macro-0.3.4
  (crate-source "arg_enum_proc_macro" "0.3.4"
                "1sjdfd5a8j6r99cf0bpqrd6b160x9vz97y5rysycsjda358jms8a"))

(define rust-argfile-0.2.1
  (crate-source "argfile" "0.2.1"
                "1s756chhwp69ha23i17hvqfriqxf5k7zfrjccx0dnmyyd6xc070a"))

(define rust-argh-0.1.13
  (crate-source "argh" "0.1.13"
                "0h6jzj4aqswk9x6w3lbb8kdskyf93v73wlrfk4pvhdlabhr1izrl"))

(define rust-argh-derive-0.1.13
  (crate-source "argh_derive" "0.1.13"
                "00vqfqgxqq6dd9jgbg9qhn12hh06qzsj1incv3ajklsh7awb5dxd"))

(define rust-argh-shared-0.1.13
  (crate-source "argh_shared" "0.1.13"
                "1xplhinnv139x2w2wknvnms7css6c99l8dw7jb1wvv9dr0y18r54"))

(define rust-argmax-0.3.1
  (crate-source "argmax" "0.3.1"
                "0ricjx82rdnycjndlmaiwxc85lq3gh4av44xlkjwbwm7wgskwzjv"))

(define rust-argon2-0.5.3
  (crate-source "argon2" "0.5.3"
                "0wn0kk97k49wxidfigmz1pdqmygqzi4h6w72ib7cpq765s4i0diw"))

(define rust-array-macro-1.0.5
  (crate-source "array-macro" "1.0.5"
                "19mdx2xlppnqwl6rhsbzylx61a0kkp2ql8q16195b7iga977ps86"))

(define rust-arraydeque-0.5.1
  (crate-source "arraydeque" "0.5.1"
                "0dn2xdfg3rkiqsh8a6achnmvf5nf11xk33xgjzpksliab4yjx43x"))

(define rust-arrayref-0.3.9
  (crate-source "arrayref" "0.3.9"
                "1jzyp0nvp10dmahaq9a2rnxqdd5wxgbvp8xaibps3zai8c9fi8kn"))

(define rust-arrayvec-0.5.2
  (crate-source "arrayvec" "0.5.2"
                "12q6hn01x5435bprwlb7w9m7817dyfq55yrl4psygr78bp32zdi3"))

(define rust-arrayvec-0.7.4
  (crate-source "arrayvec" "0.7.4"
                "04b7n722jij0v3fnm3qk072d5ysc2q30rl9fz33zpfhzah30mlwn"))

(define rust-arrayvec-0.7.6
  (crate-source "arrayvec" "0.7.6"
                "0l1fz4ccgv6pm609rif37sl5nv5k6lbzi7kkppgzqzh1vwix20kw"))

(define rust-as-raw-xcb-connection-1.0.1
  (crate-source "as-raw-xcb-connection" "1.0.1"
                "0sqgpz2ymv5yx76r5j2npjq2x5qvvqnw0vrs35cyv30p3pfp2m8p"))

(define rust-as-slice-0.2.1
  (crate-source "as-slice" "0.2.1"
                "05j52y1ws8kir5zjxnl48ann0if79sb56p9nm76hvma01r7nnssi"))

(define rust-ascii-1.1.0
  (crate-source "ascii" "1.1.0"
                "05nyyp39x4wzc1959kv7ckwqpkdzjd9dw4slzyjh73qbhjcfqayr"))

(define rust-ascii-canvas-3.0.0
  (crate-source "ascii-canvas" "3.0.0"
                "1in38ziqn4kh9sw89ys4naaqzvvjscfs0m4djqbfq7455v5fq948"))

(define rust-ascii-canvas-4.0.0
  (crate-source "ascii-canvas" "4.0.0"
                "14g8n6fqbhz0bjz07hgakbl4m8va21f0q0d1244ipaw4kmlkw7pg"))

(define rust-ash-0.38.0+1.3.281
  (crate-source "ash" "0.38.0+1.3.281"
                "0vx4yf689v1rc680jvy8bnysx5sgd8f33wnp2vqaizh0v0v4kd0b"))

(define rust-ashpd-0.6.8
  (crate-source "ashpd" "0.6.8"
                "109d7w6v0rnpy9lv4kmhwgh0sff0440s2vybj1k0ik4ib3d2xhja"))

(define rust-askama-0.12.1
  (crate-source "askama" "0.12.1"
                "0a1cmp0f1b01zzbzzp81ppa6r754zpax5372aykswz5933gr345p"))

(define rust-askama-0.13.1
  (crate-source "askama" "0.13.1"
                "19z1zjabw7xbzf4491bwn8mdx6k8b5a8y7a43f1la9pg5vnl8isx"))

(define rust-askama-0.14.0
  (crate-source "askama" "0.14.0"
                "1i3m3dzshx46v94w24chl6xg7xjyf350gqzzyijy46vp9f3n6lzp"))

(define rust-askama-derive-0.12.5
  (crate-source "askama_derive" "0.12.5"
                "10qxszzrwbabpd3jh6lvk3q1a81ryfba8bh75jb18irwn5n8vzhr"))

(define rust-askama-derive-0.13.1
  (crate-source "askama_derive" "0.13.1"
                "1b26ijv1b3gxyalwqsgj32v0qzp6268d0y4gqha5qsp3ggsy0qfn"))

(define rust-askama-derive-0.14.0
  (crate-source "askama_derive" "0.14.0"
                "0kx9sfych8m7cswcs75jhq0cy9pqn7iah1w4lvl8hc781wh9g4qj"))

(define rust-askama-escape-0.10.3
  (crate-source "askama_escape" "0.10.3"
                "0hg3rz0cma5f6385z7qmqw3jbir76jndwd5s7dqfk92v9gil75v1"))

(define rust-askama-parser-0.13.0
  (crate-source "askama_parser" "0.13.0"
                "0kqd9pg96dd6w9pm4q7zdhmchhkdrwsljygz56qpp1acabk5qcfg"
                #:snippet '(delete-file-recursively "tests")))

(define rust-askama-parser-0.14.0
  (crate-source "askama_parser" "0.14.0"
                "0n235ljbvbvlhwr54s675x1z6lgbjmzrfrq1c8rg5snmncq5dayn"
                #:snippet '(delete-file-recursively "tests")))

(define rust-askama-parser-0.2.1
  (crate-source "askama_parser" "0.2.1"
                "1h00vcnqq9qqlayx1ass4an458rk4lm3q88867cc7lb4dcf1dcdc"
                #:snippet '(delete-file-recursively "tests")))

(define rust-asn1-0.20.0
  (crate-source "asn1" "0.20.0"
                "0ckg83ingvagwjvmxadjjmkgna5kvlvrfx9arlfvzqhxxas892rd"))

(define rust-asn1-derive-0.20.0
  (crate-source "asn1_derive" "0.20.0"
                "1b88xsqmxpxjq4p2mrn1icj7c3k2s041v7wqp8yhnqiq06fq0052"))

(define rust-assert-cli-0.6.3
  (crate-source "assert_cli" "0.6.3"
                "0jc1bh3cvnl66bl7s5gr1xnm0hl8d2l3gmil0pmhp5v2xp0bg6m2"))

(define rust-assert-cmd-2.0.16
  (crate-source "assert_cmd" "2.0.16"
                "0gdj0710k3lnvyjmpv8a4dgwrk9ib85l2wfw4n2xwy3qyavka66w"))

(define rust-assert-cmd-2.0.17
  (crate-source "assert_cmd" "2.0.17"
                "0rhb6b0w23pbqcj1mkgdv8j3g9602d4jjmg45ql022lpnaj8klrb"))

(define rust-assert-cmd-2.0.8
  (crate-source "assert_cmd" "2.0.8"
                "0pjpsbhlg1b7ca94c3sn3gqyk8ylcximhc020553jx085v1gqd4q"))

(define rust-assert-fs-1.1.2
  (crate-source "assert_fs" "1.1.2"
                "0x3nj817l5kbpmr42habqv5i49rpxdpncmr86ix840knnkyv3zby"))

(define rust-assert-json-diff-2.0.2
  (crate-source "assert-json-diff" "2.0.2"
                "04mg3w0rh3schpla51l18362hsirl23q93aisws2irrj32wg5r27"))

(define rust-assert-matches-1.5.0
  (crate-source "assert_matches" "1.5.0"
                "1a9b3p9vy0msylyr2022sk5flid37ini1dxji5l3vwxsvw4xcd4v"))

(define rust-associative-cache-1.0.1
  (crate-source "associative-cache" "1.0.1"
                "05lg0mwpqfqb9zh958x0358x1k5ngmmmbzjnp0imrd8vzhrn40a6"))

(define rust-astral-tokio-tar-0.5.2
  (crate-source "astral-tokio-tar" "0.5.2"
                "16wip9bzzbjkpf1rgs8cjbfmnhxflrdi35xpb53yrncrl7xjpfqs"
                #:snippet '(delete-file-recursively "tests")))

(define rust-astral-tokio-tar-0.5.3
  (crate-source "astral-tokio-tar" "0.5.3"
                "1qg1g8bb61xasmxm32c1bkamn551sdkxd3pcsj8i5brc2irsydh0"
                #:snippet '(delete-file-recursively "tests")))

(define rust-async-attributes-1.1.2
  (crate-source "async-attributes" "1.1.2"
                "1rcnypqgmlcw9vwh0fk8bivvz8p5v8acy0zd2njdv6yxyiwkw853"))

(define rust-async-broadcast-0.5.1
  (crate-source "async-broadcast" "0.5.1"
                "0avdqbci1qdlfc4glc3wqrb0wi5ffc7bqv2q1wg14syayvdwqj3w"))

(define rust-async-broadcast-0.7.2
  (crate-source "async-broadcast" "0.7.2"
                "0ckmqcwyqwbl2cijk1y4r0vy60i89gqc86ijrxzz5f2m4yjqfnj3"))

(define rust-async-channel-1.9.0
  (crate-source "async-channel" "1.9.0"
                "0dbdlkzlncbibd3ij6y6jmvjd0cmdn48ydcfdpfhw09njd93r5c1"))

(define rust-async-channel-2.3.1
  (crate-source "async-channel" "2.3.1"
                "0skvwxj6ysfc6d7bhczz9a2550260g62bm5gl0nmjxxyn007id49"))

(define rust-async-channel-2.5.0
  (crate-source "async-channel" "2.5.0"
                "1ljq24ig8lgs2555myrrjighycpx2mbjgrm3q7lpa6rdsmnxjklj"))

(define rust-async-compression-0.3.15
  (crate-source "async-compression" "0.3.15"
                "0fnnpbsljngmrj8amhw6yg164bl6x4r43mr093g1ps9rmvbpqb4l"))

(define rust-async-compression-0.4.19
  (crate-source "async-compression" "0.4.19"
                "0g006fvpri6drd0k8ds144gg9yxzm9xi14hwcr90yn3kjrm5wmq6"
                #:snippet '(delete-file-recursively "tests")))

(define rust-async-compression-0.4.22
  (crate-source "async-compression" "0.4.22"
                "0r6shv717rl3qzccjc9qlxmnaj3l22rr9197jsahkn33v7wr98ar"
                #:snippet '(delete-file-recursively "tests")))

(define rust-async-executor-1.13.1
  (crate-source "async-executor" "1.13.1"
                "1v6w1dbvsmw6cs4dk4lxj5dvrikc6xi479wikwaab2qy3h09mjih"))

(define rust-async-executor-1.13.2
  (crate-source "async-executor" "1.13.2"
                "1yjgdh62iqn2bvp792j3q980kk015x79fz8dhq8dsjsjb3xjz0dv"))

(define rust-async-executor-1.13.3
  (crate-source "async-executor" "1.13.3"
                "1f3za9v8wkqzv6rz69g0qzvdcmghwbixijwzldwjm9w3zph00z29"))

(define rust-async-fs-1.6.0
  (crate-source "async-fs" "1.6.0"
                "01if2h77mry9cnm91ql2md595108i2c1bfy9gaivzvjfcl2gk717"))

(define rust-async-fs-2.1.2
  (crate-source "async-fs" "2.1.2"
                "0jp0p7lg9zqy2djgdmivbzx0yqmfn9sm2s9dkhaws3zlharhkkgb"))

(define rust-async-generic-1.1.2
  (crate-source "async-generic" "1.1.2"
                "0wqnsinxaw6cj6wfbrj96qfrbc1gfgsm95qm6cw8gypfcs2p5wyx"))

(define rust-async-global-executor-2.4.1
  (crate-source "async-global-executor" "2.4.1"
                "1762s45cc134d38rrv0hyp41hv4iv6nmx59vswid2p0il8rvdc85"))

(define rust-async-http-range-reader-0.9.1
  (crate-source "async_http_range_reader" "0.9.1"
                "15s16da73xw2vl8z3iyh1y01na6dijzwmmzm0qz98gwy4q07qlrb"))

(define rust-async-io-1.13.0
  (crate-source "async-io" "1.13.0"
                "1byj7lpw0ahk6k63sbc9859v68f28hpaab41dxsjj1ggjdfv9i8g"))

(define rust-async-io-2.4.0
  (crate-source "async-io" "2.4.0"
                "0n8h0vy53n4vdkq529scqnkzm9vcl3r73za9nj81s2nfrhiv78j3"))

(define rust-async-io-2.4.1
  (crate-source "async-io" "2.4.1"
                "1cybmbq3gwv26bi7sk1gmcwynbm1s36zj3wib1gpdwx0fnpc0dqj"))

(define rust-async-io-2.5.0
  (crate-source "async-io" "2.5.0"
                "1ji3y970jdnc6xa3905zbhlln62wrrl13lwzy0hg57h16rilsqqr"))

(define rust-async-io-2.6.0
  (crate-source "async-io" "2.6.0"
                "1z16s18bm4jxlmp6rif38mvn55442yd3wjvdfhvx4hkgxf7qlss5"))

(define rust-async-lock-2.8.0
  (crate-source "async-lock" "2.8.0"
                "0asq5xdzgp3d5m82y5rg7a0k9q0g95jy6mgc7ivl334x7qlp4wi8"))

(define rust-async-lock-3.4.0
  (crate-source "async-lock" "3.4.0"
                "060vh45i809wcqyxzs5g69nqiqah7ydz0hpkcjys9258vqn4fvpz"))

(define rust-async-lock-3.4.1
  (crate-source "async-lock" "3.4.1"
                "1p6i1sw3mwv1msdx9jqkr0h0a2jlrp3717yyx5n9pvkw0h23dl2z"))

(define rust-async-net-2.0.0
  (crate-source "async-net" "2.0.0"
                "1xyc5a5vcp3a7h1q2lbfh79wz8136dig4q4x6g4w2ws8ml7h0j5r"))

(define rust-async-object-pool-0.1.5
  (crate-source "async-object-pool" "0.1.5"
                "0b3546r13bdhcigbfbmpihh8gxxpad165cp80h3dbwn3jxmlag1k"))

(define rust-async-pidfd-next-0.1.0
  (crate-source "async-pidfd-next" "0.1.0"
                "0p0l5k79d7379bzxfjvpygsybc8xjzwf6m1b3v03zvwgwvm5qwwd"))

(define rust-async-priority-channel-0.1.0
  (crate-source "async-priority-channel" "0.1.0"
                "0h36m0avgs86pgh286xkvbnhdhb8bxgsnlxwwazvw88v5scph5n2"))

(define rust-async-priority-channel-0.2.0
  (crate-source "async-priority-channel" "0.2.0"
                "0n4pk6dgx0w3i43dsjgfn8f3wzdrhv3kvi65c3vk246k8ks9dpmc"))

(define rust-async-process-1.8.1
  (crate-source "async-process" "1.8.1"
                "126s968lvhg9rlwsnxp7wfzkfn7rl87p0dlvqqlibn081ax3hr7a"))

(define rust-async-process-2.3.0
  (crate-source "async-process" "2.3.0"
                "1fr6cpqdw7hrmzns1983lgx86cg8vyz7nlrn0h0125iqq8fmy9b3"))

(define rust-async-process-2.3.1
  (crate-source "async-process" "2.3.1"
                "1p459ckyndgl69cdxl8375cwnm6wsv5ra069zyndf8b01vjg9qyd"))

(define rust-async-process-2.4.0
  (crate-source "async-process" "2.4.0"
                "004zcsyvny8fl77j6y9kbc9pbg22j4lh370v3amycldd48vs3nk5"))

(define rust-async-process-2.5.0
  (crate-source "async-process" "2.5.0"
                "0xfswxmng6835hjlfhv7k0jrfp7czqxpfj6y2s5dsp05q0g94l7w"))

(define rust-async-recursion-1.1.1
  (crate-source "async-recursion" "1.1.1"
                "04ac4zh8qz2xjc79lmfi4jlqj5f92xjvfaqvbzwkizyqd4pl4hrv"))

(define rust-async-scoped-0.9.0
  (crate-source "async-scoped" "0.9.0"
                "0h47p4is1i5bvsy1i5mdlk521cdjznclxwgfab2gzvckln70fhj0"))

(define rust-async-signal-0.2.10
  (crate-source "async-signal" "0.2.10"
                "1wxrq3871l00mil43nmh0akvwjjjnv0bn7n2pzwbvh00k0s00zk3"))

(define rust-async-signal-0.2.11
  (crate-source "async-signal" "0.2.11"
                "03ffmhkvr4si21rlnhxhb68fslfybyzp0nldi7rnvc6la175lq6p"))

(define rust-async-signal-0.2.12
  (crate-source "async-signal" "0.2.12"
                "1hg3wchajq47frfsh54jpdpkwxihl87cwq5mq991v7pn1qkayrzm"))

(define rust-async-signal-0.2.13
  (crate-source "async-signal" "0.2.13"
                "0k66mpb3xp86hj4vxs7w40v7qz2jfbblrm9ddc5mglwwynxp1h23"))

(define rust-async-std-1.13.1
  (crate-source "async-std" "1.13.1"
                "094azlylq90vki8pv8k6lyhdbw1kcf7m344mfn2hyblcq30r80kk"))

(define rust-async-std-1.13.2
  (crate-source "async-std" "1.13.2"
                "0fzrg0ainb5h87z0q09adnrnspc13132wqq3fhmyaymn9ad0g3ic"))

(define rust-async-stream-0.3.6
  (crate-source "async-stream" "0.3.6"
                "0xl4zqncrdmw2g6241wgr11dxdg4h7byy6bz3l6si03qyfk72nhb"))

(define rust-async-stream-impl-0.3.6
  (crate-source "async-stream-impl" "0.3.6"
                "0kaplfb5axsvf1gfs2gk6c4zx6zcsns0yf3ssk7iwni7bphlvhn7"))

(define rust-async-task-4.7.1
  (crate-source "async-task" "4.7.1"
                "1pp3avr4ri2nbh7s6y9ws0397nkx1zymmcr14sq761ljarh3axcb"))

(define rust-async-trait-0.1.88
  (crate-source "async-trait" "0.1.88"
                "1dgxvz7g75cmz6vqqz0mri4xazc6a8xfj1db6r9fxz29lzyd6fg5"))

(define rust-async-trait-0.1.89
  (crate-source "async-trait" "0.1.89"
                "1fsxxmz3rzx1prn1h3rs7kyjhkap60i7xvi0ldapkvbb14nssdch"))

(define rust-async-zip-0.0.12
  (crate-source "async_zip" "0.0.12"
                "1l1l6iyg8hkbnjilj50vndn9ln4l6dav10ywvawh6qlwvd15245j"
                #:snippet '(for-each delete-file-recursively '("src/tests" "tests"))))

(define rust-async-zip-0.0.17.c909fda
  (origin
    (method git-fetch)
    (uri (git-reference (url "https://github.com/charliermarsh/rs-async-zip")
                        (commit "c909fda63fcafe4af496a07bfda28a5aae97e58d")))
    (file-name (git-file-name "rust-async-zip" "0.0.17.c909fda"))
    (sha256 (base32 "1nk16a56fjjjd12rpg92d2az8y0cyvbsw6c4l42nf5rrjvcj1x2m"))
    (modules '((guix build utils)))
    (snippet '(for-each delete-file-recursively '("src/tests" "tests")))))

(define rust-asyncgit-0.27.0
  (crate-source "asyncgit" "0.27.0"
                "16yx5lga0ays5av92h2vf77qzzcxam5ph5w1dl3rj24s5f15idhy"))

(define rust-atk-0.8.0
  (crate-source "atk" "0.8.0"
                "0gk6ijqsd6kh8cki1wznw570499psbppg3d5bqaayagjapxawka4"))

(define rust-atk-sys-0.10.0
  (crate-source "atk-sys" "0.10.0"
                "1knzvq2jdkx1nav619jbqsx2ivzh901rsp2wl57wr50x2fpy8c7m"))

(define rust-atk-sys-0.9.1
  (crate-source "atk-sys" "0.9.1"
                "0vzcm1inhax78bcsbapr6mrp4z7lk783csyh200wi91pcxvw2lp5"))

(define rust-atoi-2.0.0
  (crate-source "atoi" "2.0.0"
                "0a05h42fggmy7h0ajjv6m7z72l924i7igbx13hk9d8pyign9k3gj"))

(define rust-atomic-0.6.0
  (crate-source "atomic" "0.6.0"
                "15193mfhmrq3p6vi1a10hw3n6kvzf5h32zikhby3mdj0ww1q10cd"))

(define rust-atomic-0.6.1
  (crate-source "atomic" "0.6.1"
                "0h43ljcgbl6vk62hs6yk7zg7qn3myzvpw8k7isb9nzhkbdvvz758"))

(define rust-atomic-float-1.1.0
  (crate-source "atomic_float" "1.1.0"
                "02j85l9wf0pycq1ad8rwq6h681nk373jqdchwlpvihwaj67j53b2"))

(define rust-atomic-polyfill-1.0.3
  (crate-source "atomic-polyfill" "1.0.3"
                "1x00ndablb89zvbr8m03cgjzgajg86fqn8pgz85yy2gy1pivrwlc"))

(define rust-atomic-refcell-0.1.13
  (crate-source "atomic_refcell" "0.1.13"
                "0z04ng59y22mwf315wamx78ybhjag0x6k7isc36hdgcv63c7rrj1"))

(define rust-atomic-take-1.1.0
  (crate-source "atomic-take" "1.1.0"
                "1hzvfqiy0ixsawkh7ci9visn95kx2j6yvnqxz536x5wpzranpax8"))

(define rust-atomic-waker-1.1.2
  (crate-source "atomic-waker" "1.1.2"
                "1h5av1lw56m0jf0fd3bchxq8a30xv0b4wv8s4zkp4s0i7mfvs18m"))

(define rust-atomicwrites-0.4.4
  (crate-source "atomicwrites" "0.4.4"
                "1qrkr9jz43y8g767c3249g4w4pzv43bk3hyza66y6pv43f6vpw9y"))

(define rust-atspi-0.25.0
  (crate-source "atspi" "0.25.0"
                "0p412rz8cnsqh1l3wx5zq0ahxvhyg406qcazmy68623m5rc4fcn8"))

(define rust-atspi-common-0.9.0
  (crate-source "atspi-common" "0.9.0"
                "1yzxdkkzzs43aslyysaar7vr93vqyljby0vq3659i46zgigc1prk"))

(define rust-atspi-connection-0.9.0
  (crate-source "atspi-connection" "0.9.0"
                "0f29g39w06dk15hmap2scfv4csr52i3h1q3a0l226cyq0c9xb4s1"))

(define rust-atspi-proxies-0.9.0
  (crate-source "atspi-proxies" "0.9.0"
                "073msx1xrf0xjy56kifvpqrny7ndw6ah4vzxpk82cvz7wywvrvnj"))

(define rust-atty-0.2.14
  (crate-source "atty" "0.2.14"
                "1s7yslcs6a28c5vz7jwj63lkfgyx8mx99fdirlhi9lbhhzhrpcyr"))

(define rust-auditable-extract-0.3.5
  (crate-source "auditable-extract" "0.3.5"
                "0yiy1ph2620qlwj4anijmzi6w8gaci6gximn8afa9pjrjygiwds4"))

(define rust-auditable-info-0.8.0
  (crate-source "auditable-info" "0.8.0"
                "1kbhc89fg4ykywwp2z810if1i48pqng6va2rnf79qwbxcq2ffscq"))

(define rust-auditable-serde-0.7.0
  (crate-source "auditable-serde" "0.7.0"
                "0qqnraspp6zvncmda8i6mx8lsj78gxqx9qwk50nmpibnmh660zpc"))

(define rust-auth-git2-0.5.7
  (crate-source "auth-git2" "0.5.7"
                "1p0x69w81lkr11l6j4g7bkkp991hb2wgk74267skcc69438ylpnm"))

(define rust-auth-git2-0.5.8
  (crate-source "auth-git2" "0.5.8"
                "1ziy9c9apj1ah426bbdv9ad1fdnnnl90ybaif0bayfz6rj8vz228"))

(define rust-auto-impl-0.5.0
  (crate-source "auto_impl" "0.5.0"
                "195d6s9bfcchwizf3km1g69l74f6xvm5gl9505js2r9xi4ff4qkq"))

(define rust-autocfg-0.1.8
  (crate-source "autocfg" "0.1.8"
                "0y4vw4l4izdxq1v0rrhvmlbqvalrqrmk60v1z0dqlgnlbzkl7phd"))

(define rust-autocfg-1.1.0
  (crate-source "autocfg" "1.1.0"
                "1ylp3cb47ylzabimazvbz9ms6ap784zhb6syaz6c1jqpmcmq0s6l"))

(define rust-autocfg-1.3.0
  (crate-source "autocfg" "1.3.0"
                "1c3njkfzpil03k92q0mij5y1pkhhfr4j3bf0h53bgl2vs85lsjqc"))

(define rust-autocfg-1.4.0
  (crate-source "autocfg" "1.4.0"
                "09lz3by90d2hphbq56znag9v87gfpd9gb8nr82hll8z6x2nhprdc"))

(define rust-autocfg-1.5.0
  (crate-source "autocfg" "1.5.0"
                "1s77f98id9l4af4alklmzq46f21c980v13z2r1pcxx6bqgw0d1n0"))

(define rust-autocompress-0.6.0
  (crate-source "autocompress" "0.6.0"
                "1rjpqvf5if087n1cijcxf5shw9cjv7nyhkxigr0zg9446p8hg2bi"))

(define rust-automod-1.0.15
  (crate-source "automod" "1.0.15"
                "0w76lzg8vbrvx4cl8x63kdaxkwpbp17c4cg1bcgwmqmj3lqbvd7b"))

(define rust-av-metrics-0.9.1
  (crate-source "av-metrics" "0.9.1"
                "0c1m5rrrx88y1hm4i17qh0fd2rqd3jwck86lj5dkw85hpmdyjv4r"))

(define rust-av1-grain-0.2.3
  (crate-source "av1-grain" "0.2.3"
                "1gvqdh21bm1cfqiwyiinbqi0mg7x2lg2fwgmphma8ijxijfr0y36"))

(define rust-av1-grain-0.2.4
  (crate-source "av1-grain" "0.2.4"
                "1j409cf560kvhxzjs65ksnzbxz7k6sm5lywizan11ijvm0ngngjg"))

(define rust-avif-parse-1.4.0
  (crate-source "avif-parse" "1.4.0"
                "1larr51k8a2g3sghyh99l2wawri4w9fi36n21niw0jnighmcx19z"))

(define rust-avif-serialize-0.8.3
  (crate-source "avif-serialize" "0.8.3"
                "13k0sy5qd6pyvfqzbd06zadz5cavq36fxn391j10ijzv9im2v4lq"))

(define rust-avif-serialize-0.8.5
  (crate-source "avif-serialize" "0.8.5"
                "0hkb94flydcv8a5g9y1kk5nqfc6qa1agb5w1q28r2azdmi8yza1f"))

(define rust-avif-serialize-0.8.6
  (crate-source "avif-serialize" "0.8.6"
                "0gx2yqlh0j5063vk70hv0f1yq40lm6kvc45qx2gm3x1iz30gpj27"))

(define rust-aws-lc-fips-sys-0.13.5
  ;; TODO: Unbundle aws-lc.
  (crate-source "aws-lc-fips-sys" "0.13.5"
                "15fpxrq5ckix6vkabh5rx160rrrrhsb8l1cbqz5yhmqz5aajx71d"))

(define rust-aws-lc-rs-1.13.0
  (crate-source "aws-lc-rs" "1.13.0"
                "0y08lzplcfcyqhlb1xvzmsf8whp2wq2xbp561a8dry5jkj9mddqr"))

(define rust-aws-lc-rs-1.13.1
  (crate-source "aws-lc-rs" "1.13.1"
                "1ix87hcaiala8r40xd76fzb044jvxrdc8ksinm6q6v4kcprwiz4k"))

(define rust-aws-lc-rs-1.14.0
  (crate-source "aws-lc-rs" "1.14.0"
                "11i7l7pddlw808xidvm5fb0qi2qnc2lclh9nlmnv2myd15ngzf4l"))

(define rust-aws-lc-sys-0.28.0
  ;; TODO: Unbundle aws-lc.
  (crate-source "aws-lc-sys" "0.28.0"
                "0zzmrwlh41g18zllmbkggw9s0dy6v0gsfs87z5vwla7dfh5p5xxr"
                #:snippet '(delete-file-recursively "builder/prebuilt-nasm")))

(define rust-aws-lc-sys-0.28.1
  ;; TODO: Unbundle aws-lc.
  (crate-source "aws-lc-sys" "0.28.1"
                "0s9p1mwhxkw4cbdfvajp15v8aqpz19kb1rbir37nxcbcx2gb3phd"
                #:snippet '(delete-file-recursively "builder/prebuilt-nasm")))

(define rust-aws-lc-sys-0.29.0
  ;; TODO: Unbundle aws-lc.
  (crate-source "aws-lc-sys" "0.29.0"
                "0yfhmn3vcbfyg07ic2mn43i187pssi0sp6sn3fzf3vq5fxpdicb1"
                #:snippet '(delete-file-recursively "builder/prebuilt-nasm")))

(define rust-aws-lc-sys-0.31.0
  ;; TODO: Unbundle aws-lc.
  (crate-source "aws-lc-sys" "0.31.0"
                "1kcxz4rdglzmw7r6ni45ys1hnn35pp7956w9r6g6mbxcg1kx2i0f"
                #:snippet '(delete-file-recursively "builder/prebuilt-nasm")))

(define rust-axoasset-1.2.0
  (crate-source "axoasset" "1.2.0"
                "0hql04vlw1z5dnrdij1mkfi82gnvpq3ywrmps9rhjzx1za60k8av"
                #:snippet '(delete-file-recursively "tests")))

(define rust-axoprocess-0.2.0
  (crate-source "axoprocess" "0.2.0"
                "0zfs9fkn04jj8r30i470mqv28m6dp8q636cphxjrbvwab0h6kr2d"))

(define rust-axotag-0.2.0
  (crate-source "axotag" "0.2.0"
                "0m3c38gbz2dk42s5qlspqdayjnmg4mdgqhx76vgwnr1ynz0gm26q"))

(define rust-axoupdater-0.9.0
  (crate-source "axoupdater" "0.9.0"
                "1r6cmppqv57059vg22z43wa25ahnhx7s2gxy517vrpd8c3wll6dw"))

(define rust-axum-0.7.9
  (crate-source "axum" "0.7.9"
                "07z7wqczi9i8xb4460rvn39p4wjqwr32hx907crd1vwb2fy8ijpd"))

(define rust-axum-0.8.4
  (crate-source "axum" "0.8.4"
                "1d99kb3vcjnhbgrf6hysllf25hzagw7m1i1nidjpgsaa30n8c7h2"))

(define rust-axum-core-0.4.5
  (crate-source "axum-core" "0.4.5"
                "16b1496c4gm387q20hkv5ic3k5bd6xmnvk50kwsy6ymr8rhvvwh9"))

(define rust-axum-core-0.5.2
  (crate-source "axum-core" "0.5.2"
                "19kwzksb4hwr3qfbrhjbqf83z6fjyng14wrkzck6fj1g8784qik8"))

(define rust-az-1.2.1
  (crate-source "az" "1.2.1"
                "0ww9k1w3al7x5qmb7f13v3s9c2pg1pdxbs8xshqy6zyrchj4qzkv"))

(define rust-backon-1.4.1
  (crate-source "backon" "1.4.1"
                "1drv0gvhjs3g0q88f1mknqjdyhh6qg8pvb9nkfasba011ibr23cp"))

(define rust-backon-1.5.0
  (crate-source "backon" "1.5.0"
                "15k4p6xyxi4lkiyw5yxrmcws3wwnwjacgcqqmd2dvfldnyqm02zx"))

(define rust-backtrace-0.3.71
  (crate-source "backtrace" "0.3.71"
                "17bgd7pbjb9gc8q47qwsg2lmy9i62x3bsjmmnjrwh5z8s805ic16"))

(define rust-backtrace-0.3.73
  (crate-source "backtrace" "0.3.73"
                "02iffg2pkg5nc36pgml8il7f77s138hhjw9f9l56v5zqlilk5hjw"))

(define rust-backtrace-0.3.74
  (crate-source "backtrace" "0.3.74"
                "06pfif7nwx66qf2zaanc2fcq7m64i91ki9imw9xd3bnz5hrwp0ld"))

(define rust-backtrace-0.3.75
  (crate-source "backtrace" "0.3.75"
                "00hhizz29mvd7cdqyz5wrj98vqkihgcxmv2vl7z0d0f53qrac1k8"))

(define rust-backtrace-0.3.76
  (crate-source "backtrace" "0.3.76"
                "1mibx75x4jf6wz7qjifynld3hpw3vq6sy3d3c9y5s88sg59ihlxv"))

(define rust-backtrace-ext-0.2.1
  (crate-source "backtrace-ext" "0.2.1"
                "0l4xacjnx4jrn9k14xbs2swks018mviq03sp7c1gn62apviywysk"))

(define rust-base-x-0.2.11
  (crate-source "base-x" "0.2.11"
                "0w02sdqvg7zwm91agb2phikw4ri8jmncw32paxsv8ra1jv8ckfsc"))

(define rust-base16ct-0.2.0
  (crate-source "base16ct" "0.2.0"
                "1kylrjhdzk7qpknrvlphw8ywdnvvg39dizw9622w3wk5xba04zsc"))

(define rust-base32-0.4.0
  (crate-source "base32" "0.4.0"
                "1ykwx8jhksqxghfgyw2pzikzjf4n9wqm1x2ww5wqyn68ssf6dki3"))

(define rust-base32-0.5.1
  (crate-source "base32" "0.5.1"
                "0xp0a3xml25xw2bp5pyac2nld7vmmfjl02qynnyfn6aznfggwb82"))

(define rust-base62-2.2.1
  (crate-source "base62" "2.2.1"
                "07xb6xj6f3h46kjhjj8sxnkvgjz3z4x0bd8zwazfwsqxrdxjmr8h"))

(define rust-base64-0.13.1
  (crate-source "base64" "0.13.1"
                "1s494mqmzjb766fy1kqlccgfg2sdcjb6hzbvzqv2jw65fdi5h6wy"))

(define rust-base64-0.20.0
  (crate-source "base64" "0.20.0"
                "1r855djiv8rirg37w5arazk42ya5gm5gd2bww75v14w0sy02i8hf"))

(define rust-base64-0.21.7
  (crate-source "base64" "0.21.7"
                "0rw52yvsk75kar9wgqfwgb414kvil1gn7mqkrhn9zf1537mpsacx"))

(define rust-base64-0.22.1
  (crate-source "base64" "0.22.1"
                "1imqzgh7bxcikp5vx3shqvw9j09g9ly0xr0jma0q66i52r7jbcvj"))

(define rust-base64-simd-0.7.0
  (crate-source "base64-simd" "0.7.0"
                "1mg5ayj5z7imfyv06fhzi5rw289gv5yrfakxzsad22zz786d47bq"))

(define rust-base64ct-1.7.3
  (crate-source "base64ct" "1.7.3"
                "18scrpjl145msdb64q4nbb0plm4wbmp5lml134nz0c5rvxm5pql9"))

(define rust-base64ct-1.8.0
  (crate-source "base64ct" "1.8.0"
                "1fj4vc6ghy3j1120r7dwn4xw90crfy46b448g5pm9w6an13qn92m"))

(define rust-basic-cookies-0.1.5
  (crate-source "basic-cookies" "0.1.5"
                "1xwnmmcn32m18nis7azfxylkqyhirkqcag94i23b1g8n5ka8zgb7"))

(define rust-basic-toml-0.1.10
  (crate-source "basic-toml" "0.1.10"
                "12hp59jl28kk229q4sqx6v4fc9p66v8i2byi0vlc9922h9g6fqms"))

(define rust-bat-0.24.0
  ;; TODO: Generate files in assets.
  (crate-source "bat" "0.24.0"
                "11nc2iv2qhd1bs16yijqq934864ybnmg485rny70scy26xb9xk4x"
                #:snippet '(for-each delete-file-recursively '("doc" "tests"))))

(define rust-battery-0.7.8
  (crate-source "battery" "0.7.8"
                "1r1641dyks76p39i1iihswhc6iz5z51pihmpxniy1h1pi4k29dml"))

(define rust-bcrypt-pbkdf-0.10.0
  (crate-source "bcrypt-pbkdf" "0.10.0"
                "18pjhsy3m2v0silsp4mjzz8i92zrpqxk9b059zrnk1w8zvhw5ska"))

(define rust-bcrypt-pbkdf-0.6.2
  (crate-source "bcrypt-pbkdf" "0.6.2"
                "1ms9c5z90n5szx5nbxrqaihny5fs3sl6a1pm3szr5g86jlxw0f3w"))

(define rust-beef-0.5.2
  (crate-source "beef" "0.5.2"
                "1c95lbnhld96iwwbyh5kzykbpysq0fnjfhwxa1mhap5qxgrl30is"))

(define rust-bgzip-0.3.1
  (crate-source "bgzip" "0.3.1"
                "16zr2nclis3kgz0jxi7ayyk510ar5dvyfpf03fazajmn1ycdhkxn"))

(define rust-biblatex-0.10.0
  (crate-source "biblatex" "0.10.0"
                "1hpkxwkyby2bx8gn6jnfryn3da8ihxjhmpfhc15zkgmxzhbp6nm3"))

(define rust-bincode-1.3.3
  (crate-source "bincode" "1.3.3"
                "1bfw3mnwzx5g1465kiqllp5n4r10qrqy88kdlp3jfwnq2ya5xx5i"))

(define rust-bindgen-0.48.1
  (crate-source "bindgen" "0.48.1"
                "1jpk0p4l4dg1lpvciq9q8wm94sjsflb1vb5x2gk9dlizv4gl2gcx"))

(define rust-bindgen-0.63.0
  (crate-source "bindgen" "0.63.0"
                "119qkny3swlx8zm0jxp19hizpkym6arh8mlv9ylskch030961n1n"))

(define rust-bindgen-0.65.1
  (crate-source "bindgen" "0.65.1"
                "1i9wci1h3xnk8hi7cf06capgifnmpk9dd59zqznh6jcsdx37ppyg"))

(define rust-bindgen-0.66.1
  (crate-source "bindgen" "0.66.1"
                "19yj6fsb08x0l1pg871vvfvlx1mglamz8hyjpazhfc90zh34xf7j"))

(define rust-bindgen-0.69.5
  (crate-source "bindgen" "0.69.5"
                "1240snlcfj663k04bjsg629g4wx6f83flgbjh5rzpgyagk3864r7"))

(define rust-bindgen-0.70.1
  (crate-source "bindgen" "0.70.1"
                "0vyf0jp6apcy9kjyz4s8vldj0xqycnbzb6zv3skkwiqdi3nqz7gl"))

(define rust-bindgen-0.71.1
  (crate-source "bindgen" "0.71.1"
                "1cynz43s9xwjbd1y03rx9h37xs0isyl8bi6g6yngp35nglyvyn2z"))

(define rust-bindgen-0.72.1
  (crate-source "bindgen" "0.72.1"
                "15bq73y3wd3x3vxh3z3g72hy08zs8rxg1f0i1xsrrd6g16spcdwr"))

(define rust-binfarce-0.2.1
  (crate-source "binfarce" "0.2.1"
                "18hnqqir3gk5sx1mlndzgpxs0l4rviv7dk3h1piyspayp35lqihq"))

(define rust-bio-0.33.0
  (crate-source "bio" "0.33.0"
                "1zaghvazh243x2pg866hjx1zwwpp9qzwkf45iz2fs9af5y661gz5"))

(define rust-bio-0.39.2
  (crate-source "bio" "0.39.2"
                "00k0zv4yyiipzg2arnkrplfi0digwnv89bczfzhyva68yfv8scvb"))

(define rust-bio-1.6.0
  (crate-source "bio" "1.6.0"
                "1hpnfwjyqg23dlk75frydf9kgilccyn3haaw6xdwh25zpa9wnwks"))

(define rust-bio-types-0.13.0
  (crate-source "bio-types" "0.13.0"
                "102cn2kpp4ivj0ixcnc98m0nnbp6ff6xagfwk2jmywr81bs91afz"))

(define rust-bio-types-1.0.4
  (crate-source "bio-types" "1.0.4"
                "0zmdcvj44a088larkahcic5z61cwn2x80iym0w14albzid7zbp7l"))

(define rust-biquad-0.4.2
  (crate-source "biquad" "0.4.2"
                "0gpc13lag439nmq077wfwz055qbjaxbpk7znvnbddbg3wgsj81c2"))

(define rust-bisection-0.1.0
  (crate-source "bisection" "0.1.0"
                "1hx80j7lmj3mg093psrnf5llmyksqg078jsbrzkcq3mb3fd0f7h2"))

(define rust-bit-field-0.10.2
  (crate-source "bit_field" "0.10.2"
                "0qav5rpm4hqc33vmf4vc4r0mh51yjx5vmd9zhih26n9yjs3730nw"))

(define rust-bit-field-0.10.3
  (crate-source "bit_field" "0.10.3"
                "1ikhbph4ap4w692c33r8bbv6yd2qxm1q3f64845grp1s6b3l0jqy"))

(define rust-bit-set-0.5.3
  (crate-source "bit-set" "0.5.3"
                "1wcm9vxi00ma4rcxkl3pzzjli6ihrpn9cfdi0c5b4cvga2mxs007"))

(define rust-bit-set-0.8.0
  (crate-source "bit-set" "0.8.0"
                "18riaa10s6n59n39vix0cr7l2dgwdhcpbcm97x1xbyfp1q47x008"))

(define rust-bit-vec-0.6.3
  (crate-source "bit-vec" "0.6.3"
                "1ywqjnv60cdh1slhz67psnp422md6jdliji6alq0gmly2xm9p7rl"))

(define rust-bit-vec-0.8.0
  (crate-source "bit-vec" "0.8.0"
                "1xxa1s2cj291r7k1whbxq840jxvmdsq9xgh7bvrxl46m80fllxjy"))

(define rust-bitfield-0.14.0
  (crate-source "bitfield" "0.14.0"
                "1b26k9acwss4qvrl60lf9n83l17d4hj47n5rmpd3iigf9j9n0zid"))

(define rust-bitfield-0.17.0
  (crate-source "bitfield" "0.17.0"
                "1q4n13japrj852yzidhjfcq702yxkvrpv5mhmacsliz5az8x567p"))

(define rust-bitflags-1.3.2
  (crate-source "bitflags" "1.3.2"
                "12ki6w8gn1ldq7yz9y680llwk5gmrhrzszaa17g1sbrw2r2qvwxy"))

(define rust-bitflags-2.4.0
  (crate-source "bitflags" "2.4.0"
                "0dc6xa7flfl59makmhixjcrslwlvdxxwrgxbr8p7bkvz53k2ls5l"))

(define rust-bitflags-2.9.0
  (crate-source "bitflags" "2.9.0"
                "1gb5w7pxnmx8l2bjz1i6rkbwbm2167k294rhy6cl1y3vbc8i90jw"))

(define rust-bitflags-2.9.1
  (crate-source "bitflags" "2.9.1"
                "0rz9rpp5wywwqb3mxfkywh4drmzci2fch780q7lifbf6bsc5d3hv"))

(define rust-bitflags-2.9.3
  (crate-source "bitflags" "2.9.3"
                "0pgjwsd9qgdmsmwpvg47p9ccrsc26yfjqawbhsi9qds5sg6brvrl"))

(define rust-bitflags-2.9.4
  (crate-source "bitflags" "2.9.4"
                "157kkcv8s7vk6d17dar1pa5cqcz4c8pdrn16wm1ld7jnr86d2q92"))

(define rust-bitmaps-2.1.0
  (crate-source "bitmaps" "2.1.0"
                "18k4mcwxl96yvii5kcljkpb8pg5j4jj1zbsdn26nsx4r83846403"))

(define rust-bitreader-0.3.11
  (crate-source "bitreader" "0.3.11"
                "0rn010p46g7fv08la915yq4q1vjfpgz8baf3bdv6rmb3w6qmjrc8"))

(define rust-bitstream-io-2.6.0
  (crate-source "bitstream-io" "2.6.0"
                "1cli390l1dhp9skygyjjnqvczp36b7f31mkx9ry3dg26330cv6b0"))

(define rust-bitvec-1.0.1
  (crate-source "bitvec" "1.0.1"
                "173ydyj2q5vwj88k6xgjnfsshs4x9wbvjjv7sm0h36r34hn87hhv"))

(define rust-blake2-0.10.6
  (crate-source "blake2" "0.10.6"
                "1zlf7w7gql12v61d9jcbbswa3dw8qxsjglylsiljp9f9b3a2ll26"
                #:snippet '(delete-file-recursively "tests")))

(define rust-blake2-0.9.2
  (crate-source "blake2" "0.9.2"
                "0x7i67c0hn3bzcwny08rgjrrnarqnqw10qpmh2blbx9hd78kfkha"
                #:snippet '(delete-file-recursively "tests")))

(define rust-blake2b-simd-1.0.3
  (crate-source "blake2b_simd" "1.0.3"
                "16cxmm4pr5jx9xc0msj1qn018ram3vz9k17cj57r97hm1fi07s86"))

(define rust-blake3-1.8.1
  (crate-source "blake3" "1.8.1"
                "1czffygg8dhdsjyzydsrf50harfrralrkm10ckhkja1i6jdhk6iq"))

(define rust-blake3-1.8.2
  (crate-source "blake3" "1.8.2"
                "1854x65zmjh9w9cfhyyyg0wmm2k5d87l13l4m7y40ajbkslam21q"))

(define rust-blakeout-0.3.0
  (crate-source "blakeout" "0.3.0"
                "0dxcg3sjxd82mn7mq4ammrfpidqf9zsagvhfzgblsi8g4b2fgvw1"))

(define rust-blas-0.22.0
  (crate-source "blas" "0.22.0"
                "1p1rs9y8fpxmrh9jj1rf4q517x5h960v4jf30f1gwnr1qdshz65f"))

(define rust-blas-sys-0.7.1
  (crate-source "blas-sys" "0.7.1"
                "0h14zjycwc76v15h8qll9z1xiryvmpvsb5gam97pqpdjrrwv5c8k"))

(define rust-block-0.1.6
  (crate-source "block" "0.1.6"
                "16k9jgll25pzsq14f244q22cdv0zb4bqacldg3kx6h89d7piz30d"))

(define rust-block-buffer-0.10.4
  (crate-source "block-buffer" "0.10.4"
                "0w9sa2ypmrsqqvc20nhwr75wbb5cjr4kkyhpjm1z1lv2kdicfy1h"))

(define rust-block-buffer-0.7.3
  (crate-source "block-buffer" "0.7.3"
                "12v8wizynqin0hqf140kmp9s38q223mp1b0hkqk8j5pk8720v560"))

(define rust-block-buffer-0.9.0
  (crate-source "block-buffer" "0.9.0"
                "1r4pf90s7d7lj1wdjhlnqa26vvbm6pnc33z138lxpnp9srpi2lj1"))

(define rust-block-modes-0.8.1
  (crate-source "block-modes" "0.8.1"
                "13id7rw1lhi83i701za8w5is3a8qkf4vfigqw3f8jp8mxldkvc1c"
                #:snippet '(delete-file-recursively "tests")))

(define rust-block-padding-0.1.5
  (crate-source "block-padding" "0.1.5"
                "1xbkmysiz23vimd17rnsjpw9bgjxipwfslwyygqlkx4in3dxwygs"))

(define rust-block-padding-0.2.1
  (crate-source "block-padding" "0.2.1"
                "1bickjlmfw9iv63dp781n589rfybw879mik1va59833m1hvnqscd"))

(define rust-block-padding-0.3.3
  (crate-source "block-padding" "0.3.3"
                "14wdad0r1qk5gmszxqd8cky6vx8qg7c153jv981mixzrpzmlz2d8"))

(define rust-block2-0.5.1
  (crate-source "block2" "0.5.1"
                "0pyiha5his2grzqr3mynmq244laql2j20992i59asp0gy7mjw4rc"))

(define rust-block2-0.6.0
  (crate-source "block2" "0.6.0"
                "0dx0j9bykzryc993spv7jym4239a8d2gynp9l82h8vz1f30v8n8x"))

(define rust-block2-0.6.1
  (crate-source "block2" "0.6.1"
                "1wnwha7wjjqiamj9abq5l45fyzdxna2k2la0rp9w2hravc5jy39l"))

(define rust-blocking-1.6.1
  (crate-source "blocking" "1.6.1"
                "1si99l8zp7c4zq87y35ayjgc5c9b60jb8h0k14zfcs679z2l2gvh"))

(define rust-blocking-1.6.2
  (crate-source "blocking" "1.6.2"
                "08bz3f9agqlp3102snkvsll6wc9ag7x5m1xy45ak2rv9pq18sgz8"))

(define rust-bloomy-1.2.0
  (crate-source "bloomy" "1.2.0"
                "1p9vxbv0kws99ilkk97kd8dhc4pr3rmaqww28y38mdsjg3sjm7a8"))

(define rust-blosc2-rs-0.4.0+2.15.2
  (crate-source "blosc2-rs" "0.4.0+2.15.2"
                "1crw6sc8d6pmah7y2jxmd94wldz6dvjlinmgkzagcsy67asvllm7"))

(define rust-blosc2-sys-0.4.0+2.15.2
  (crate-source "blosc2-sys" "0.4.0+2.15.2"
                "0wcq2dcgwml7q6n5v8qh0cibv86b5qcqv1c7gmfjsddi2bn0yy2s"
                #:snippet '(delete-file-recursively "c-blosc2")))

(define rust-blowfish-0.8.0
  (crate-source "blowfish" "0.8.0"
                "1ax736islxcbghc2lqq4vy7zn6qdigrls71lwg11m3743pyg6gzy"
                #:snippet '(delete-file-recursively "tests")))

(define rust-blowfish-0.9.1
  (crate-source "blowfish" "0.9.1"
                "1mw7bvj3bg5w8vh9xw9xawqh7ixk2xwsxkj34ph96b9b1z6y44p4"
                #:snippet '(delete-file-recursively "tests")))

(define rust-bon-3.7.2
  (crate-source "bon" "3.7.2"
                "1yzhhczsfwvc6apl6xm1q00a08bci8kr4n5l3s2bw0kl04qrqln2"))

(define rust-bon-macros-3.7.2
  (crate-source "bon-macros" "3.7.2"
                "01bhvix3pqbhhi06s2waq72qvyklbnvasqqqb52s6idqvkd2086q"))

(define rust-borrow-or-share-0.2.2
  (crate-source "borrow-or-share" "0.2.2"
                "0ciski5i69a8mx6f0fh901hn73fii3g39lpl8k3xgi88651b9siy"))

(define rust-borsh-1.5.7
  (crate-source "borsh" "1.5.7"
                "1kikljm5yr3l9qsw5xvdccragxj4445s4s3fqsgy6hmmipwld1md"))

(define rust-borsh-derive-1.5.7
  (crate-source "borsh-derive" "1.5.7"
                "1hwsznww7d01bh1fzpndzmcpyp00xpl2y9bg70i3z0zmqb0d7lgx"))

(define rust-bosion-1.1.2
  (crate-source "bosion" "1.1.2"
                "1kb2jhlmr6ppznpk45mx5lv4dv7hf7gclv1396dgb0zd5i9m10c5"))

(define rust-bosion-1.1.3
  (crate-source "bosion" "1.1.3"
                "0pqq9ydi8hsnn3x297ad14bwcv6mxln7f8lgjx913g2247n288w1"))

(define rust-botan-0.11.1
  (crate-source "botan" "0.11.1"
                "0xq315pvid6vd2c9msdzyap8122hdi640h07zaa33ib7gmjcgm14"))

(define rust-botan-sys-0.11.1
  (crate-source "botan-sys" "0.11.1"
                "0f6swv09cqdahgj2rn2ahja97nw3j4kinns3zrhrkk4lq2h5ya04"))

(define rust-box-drawing-0.1.2
  (crate-source "box_drawing" "0.1.2"
                "0jx4rrxy4xmgmplmgl398vrng67sfl8qny7n7d91fyw6zpaxh9za"))

(define rust-boxcar-0.2.11
  (crate-source "boxcar" "0.2.11"
                "1gip12s87j97dxh94q4jjdcykbi6714wf52jqdbzlq33zkicch37"))

(define rust-boxfnonce-0.1.1
  (crate-source "boxfnonce" "0.1.1"
                "09ilf4zyx92hyhkxlsxksfyprzr9iwq5gqqb22aaqr32c8fwp22r"
                #:snippet '(delete-file-recursively ".travis")))

(define rust-boxxy-0.13.1
  (crate-source "boxxy" "0.13.1"
                "0rhzv4c6xn6va4dikbq2a2cmd5sa3svl13lhpcrchdn23y744aq0"))

(define rust-bpaf-0.9.20
  (crate-source "bpaf" "0.9.20"
                "0caacci746fr7vh3x64fxi7zwqr3gi089lfddphb22v2m3bpcfa7"))

(define rust-bpaf-derive-0.5.17
  (crate-source "bpaf_derive" "0.5.17"
                "12k1w518xkly98fcsxifrmj7rmxah5hg48l9jc2if2csxkp4zyzy"))

(define rust-bracoxide-0.1.5
  (crate-source "bracoxide" "0.1.5"
                "0alnn0lkfyfk5i1dd0ykqpw4zli6az7bdkslh98xga8s90f9jlkz"))

(define rust-brotli-7.0.0
  (crate-source "brotli" "7.0.0"
                "1g99xay61mds9d23fnfj5gfbd6g11gihfgs3y1abljwldzqvi5yc"
                #:snippet '(delete-file-recursively "testdata")))

(define rust-brotli-decompressor-4.0.2
  (crate-source "brotli-decompressor" "4.0.2"
                "0dqb0vbhrc77c09qf6qmbylgkfnbjaq8629qp0z42gc0gnnhbykl"))

(define rust-brotli-decompressor-4.0.3
  (crate-source "brotli-decompressor" "4.0.3"
                "1zdxrw6plcnfwirb6bnmd9h3xabx0c4x734ffk7g1ar3kryfyd53"))

(define rust-browserslist-rs-0.6.0
  (crate-source "browserslist-rs" "0.6.0"
                "19l0prgnbx0cyqvjsf30ni7d87nkn8j0fj56fg259yvfh50if1ri"))

(define rust-brunch-0.8.1
  (crate-source "brunch" "0.8.1"
                "0qlvavpczq7dhgw6aqfp44yr3wa91wsgw6ry1gx285ii8c79av81"))

(define rust-bs58-0.4.0
  (crate-source "bs58" "0.4.0"
                "1lr3vwzhhyica4y7rbkf26vr1h7vpjb1m6rml8zcqgw81c2y07vp"))

(define rust-bsd-kvm-0.1.5
  (crate-source "bsd-kvm" "0.1.5"
                "1gncwfwlx6mq47qc1siwaiqlsaccy7vsc1v39ybs4xvvn4lfpd4l"))

(define rust-bsd-kvm-sys-0.2.0
  (crate-source "bsd-kvm-sys" "0.2.0"
                "0cn90d0kkfcs36v3sq3lpckyy0pdpdq0m7ihjlancripdn98yh35"))

(define rust-bstr-1.11.3
  (crate-source "bstr" "1.11.3"
                "1q3g2wmrvclgx7lk2p6mpzhqxzx41hyg962gkmlyxql1liar26jk"))

(define rust-bstr-1.12.0
  (crate-source "bstr" "1.12.0"
                "195i0gd7r7jg7a8spkmw08492n7rmiabcvz880xn2z8dkp8i6h93"))

(define rust-btoi-0.4.3
  (crate-source "btoi" "0.4.3"
                "1bg02zgsv5njbhng9sq2b70przbazsczjbla5lbbdf59fdzl1mlx"))

(define rust-buffer-redux-1.0.2
  (crate-source "buffer-redux" "1.0.2"
                "1waq39blrj7j6qp1sp2fvplwmq10yhks7fgbsdy8kxdrqn3wz2jf"))

(define rust-buffered-reader-1.4.0
  (crate-source "buffered-reader" "1.4.0"
                "0bs1q8vg61qgrs39ykhcybrvdbhm14ljzgmkb9dy1m9g14gvy9nv"))

(define rust-bufstream-0.1.4
  (crate-source "bufstream" "0.1.4"
                "1j7f52rv73hd1crzrrfb9dr50ccmi3hb1ybd6s5dyg6jmllqkqs0"))

(define rust-bugreport-0.5.1
  (crate-source "bugreport" "0.5.1"
                "0qn1d7wivxg29kwfbcnwxr8i4a9h94hcpgwv6hchk22vx1fgd07j"))

(define rust-build-const-0.2.2
  (crate-source "build_const" "0.2.2"
                "1dryhsf4vfi1plljgv069sgfr8m1rsg04qy76x36kh6swqsl5bml"))

(define rust-built-0.7.7
  (crate-source "built" "0.7.7"
                "0ywn0m11xm80pg6zrzq3sdj3vmzg3qs6baqnvfmkd377ly8n3van"))

(define rust-bumpalo-3.16.0
  (crate-source "bumpalo" "3.16.0"
                "0b015qb4knwanbdlp1x48pkb4pm57b8gidbhhhxr900q2wb6fabr"))

(define rust-bumpalo-3.17.0
  (crate-source "bumpalo" "3.17.0"
                "1gxxsn2fsjmv03g8p3m749mczv2k4m8xspifs5l7bcx0vx3gna0n"))

(define rust-bumpalo-3.18.1
  (crate-source "bumpalo" "3.18.1"
                "1vmfniqr484l4ffkf0056g6hakncr7kdh11hyggh9kc7c5nvfgbr"))

(define rust-bumpalo-3.19.0
  (crate-source "bumpalo" "3.19.0"
                "0hsdndvcpqbjb85ghrhska2qxvp9i75q2vb70hma9fxqawdy9ia6"))

(define rust-bv-0.11.1
  (crate-source "bv" "0.11.1"
                "0h5kbl54fsccznfixw83xndbripw39y2qkqjwf709p75iqfvnd48"))

(define rust-bwrap-1.3.0
  (crate-source "bwrap" "1.3.0"
                "15dv9mawgmwgpj88k832qjdi94w70advg9zv28bpgig5k8qs6gfi"
                #:snippet '(for-each delete-file (find-files "." "\\.png$"))))

(define rust-by-address-1.2.1
  (crate-source "by_address" "1.2.1"
                "01idmag3lcwnnqrnnyik2gmbrr34drsi97q15ihvcbbidf2kryk4"))

(define rust-byte-tools-0.3.1
  (crate-source "byte-tools" "0.3.1"
                "1mqi29wsm8njpl51pfwr31wmpzs5ahlcb40wsjyd92l90ixcmdg3"))

(define rust-byte-unit-4.0.19
  (crate-source "byte-unit" "4.0.19"
                "0v4v4z8qilnbg0lv16icbbbdq5kjpbp8yw044lszrzdqawhb6y6s"))

(define rust-byte-unit-5.1.6
  (crate-source "byte-unit" "5.1.6"
                "0x3ifzw0isgsiffg4yhvqa28rzbyxpivy2bkph69n845qp1jkkg1"))

(define rust-bytecheck-0.6.12
  (crate-source "bytecheck" "0.6.12"
                "1hmipv4yyxgbamcbw5r65wagv9khs033v9483s9kri9sw9ycbk93"))

(define rust-bytecheck-0.8.1
  (crate-source "bytecheck" "0.8.1"
                "18yisf4zvvhhpv987ic5js6arwj69h44c9rpa0szxf8g6yrhysah"))

(define rust-bytecheck-derive-0.6.12
  (crate-source "bytecheck_derive" "0.6.12"
                "0ng6230brd0hvqpbgcx83inn74mdv3abwn95x515bndwkz90dd1x"))

(define rust-bytecheck-derive-0.8.1
  (crate-source "bytecheck_derive" "0.8.1"
                "0wfbjc1vqkmszx99y6hghlcnk8xgxlhyfsgc5mf3b05i1ip89dzg"))

(define rust-bytecount-0.6.3
  (crate-source "bytecount" "0.6.3"
                "173wsvyagflb7ic3hpvp1db6q3dsigr452inslnzmsb3ix3nlrrc"))

(define rust-bytecount-0.6.8
  (crate-source "bytecount" "0.6.8"
                "1klqfjwn41fwmcqw4z03v6i4imgrf7lmf3b5s9v74hxir8hrps2w"))

(define rust-bytecount-0.6.9
  (crate-source "bytecount" "0.6.9"
                "0pinq0n8zza8qr2lyc3yf17k963129kdbf0bwnmvdk1bpvh14n0p"))

(define rust-bytelines-2.5.0
  (crate-source "bytelines" "2.5.0"
                "1jxacxpb7v0qgh325s5b7mfk90fr63jpr90dar8m47r27imnb5qj"))

(define rust-bytemuck-1.22.0
  (crate-source "bytemuck" "1.22.0"
                "0h6m8wh7iw98cn69k53plbyqff78c2yrs32l0fy4wqdcvc8grcdn"))

(define rust-bytemuck-1.23.0
  (crate-source "bytemuck" "1.23.0"
                "134mywp7bqm252c851h3bny75g0l5kw99738a1mkcjyf07pscd4i"))

(define rust-bytemuck-1.23.1
  (crate-source "bytemuck" "1.23.1"
                "08ilqv7x5lckj82i41j5lx2is9krcxiibgrs9pispr245rwsaxjw"))

(define rust-bytemuck-1.23.2
  (crate-source "bytemuck" "1.23.2"
                "0xs637lsr9p73ackjkmbjw80dp1dfdw0ydhdk0gzjcnzpkpfm59r"))

(define rust-bytemuck-derive-1.10.1
  (crate-source "bytemuck_derive" "1.10.1"
                "0a9dczfzn2c1lgg7afhqrh2drmg34w49hxhipni6pjri49blw5ag"))

(define rust-bytemuck-derive-1.9.3
  (crate-source "bytemuck_derive" "1.9.3"
                "18g1r1zgwiz5px2kf1n55ibjb2aqm86nkw28ss1mn85k94xjgk3y"))

(define rust-byteorder-1.5.0
  (crate-source "byteorder" "1.5.0"
                "0jzncxyf404mwqdbspihyzpkndfgda450l0893pz5xj685cg5l0z"))

(define rust-byteorder-lite-0.1.0
  (crate-source "byteorder-lite" "0.1.0"
                "15alafmz4b9az56z6x7glcbcb6a8bfgyd109qc3bvx07zx4fj7wg"))

(define rust-bytes-0.4.12
  (crate-source "bytes" "0.4.12"
                "0768a55q2fsqdjsvcv98ndg9dq7w2g44dvq1avhwpxrdzbydyvr0"))

(define rust-bytes-1.10.1
  (crate-source "bytes" "1.10.1"
                "0smd4wi2yrhp5pmq571yiaqx84bjqlm1ixqhnvfwzzc6pqkn26yp"))

(define rust-bytes-1.7.1
  (crate-source "bytes" "1.7.1"
                "0l5sf69avjxcw41cznyzxsnymwmkpmk08q0sm7fgicvvn0ysa643"))

(define rust-bytesize-1.3.3
  (crate-source "bytesize" "1.3.3"
                "0nb645ma48nwsv1piylzcza0avjp435sl8krhyws3q18kv5ap4rf"))

(define rust-bytesize-2.0.1
  (crate-source "bytesize" "2.0.1"
                "1fmchbqbk4pvzh0g4qmsgxlq2scdn4qgfg5fxvqaqjj114rgij53"))

(define rust-bytestring-1.4.0
  (crate-source "bytestring" "1.4.0"
                "0pwf6x2hxy6imy7ph15c5jxx51js5p5xwl1zfnq26a1vw9x68rg4"))

(define rust-bzip2-0.4.4
  (crate-source "bzip2" "0.4.4"
                "1y27wgqkx3k2jmh4k26vra2kqjq1qc1asww8hac3cv1zxyk1dcdx"
                #:snippet '(delete-file-recursively "tests")))

(define rust-bzip2-0.5.2
  (crate-source "bzip2" "0.5.2"
                "0iya6nbj0p2y8jss0z05yncc5hadry164fw3zva01y06v4igpv29"
                #:snippet '(delete-file-recursively "tests")))

(define rust-bzip2-rs-0.1.2
  (crate-source "bzip2-rs" "0.1.2"
                "0dgp83kixqrqj6q6574qr5zsfpbsiiwhqs3krhvsn4f8wkkmksxy"
                #:snippet '(delete-file-recursively "tests")))

(define rust-bzip2-sys-0.1.13+1.0.8
  (crate-source "bzip2-sys" "0.1.13+1.0.8"
                "056c39pgjh4272bdslv445f5ry64xvb0f7nph3z7860ln8rzynr2"
                #:snippet
                '(begin
                   (delete-file-recursively "bzip2-1.0.8")
                   (delete-file "build.rs")
                   ;; Inspired by Debian's patch.
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main() {~@
                        println!(\"cargo:rustc-link-lib=bz2\");~@
                        }~%"))))))

(define rust-bzip3-0.9.0
  (crate-source "bzip3" "0.9.0"
                "0351ka1qm20hdvlxy5d536d2v777586ygg77jj417wan70bch279"))

(define rust-c-gull-0.21.0
  (crate-source "c-gull" "0.21.0"
                "1q4qrk4mi9pw21dy4fcml4hhmm824rwya4k36qi7952665pvxagc"))

(define rust-c-gull-0.22.1
  (crate-source "c-gull" "0.22.1"
                "0yv0v1bz6xb6rz9klnsp30crw2b7wmnydx6xifwf3b6ady8hxxq5"))

(define rust-c-scape-0.21.0
  (crate-source "c-scape" "0.21.0"
                "1vq4w94f4ryb3s7v2fs48plv5p63b72v3zbffyh29xbq6l8cp8pa"
                #:snippet '(delete-file-recursively "empty")))

(define rust-c-scape-0.22.1
  (crate-source "c-scape" "0.22.1"
                "0mbip0ipx9idpw1iiz3kdvfgchlbxxy9jz797hnk3k7iw8740kpz"
                #:snippet '(delete-file-recursively "empty")))

(define rust-c2rust-ast-builder-0.20.0
  (crate-source "c2rust-ast-builder" "0.20.0"
                "1kk7vm9zmsnavslmblcp3pgl4rc9kbsynm6yykvhjfjcd0fgd5vs"))

(define rust-c2rust-ast-exporter-0.20.0
  (crate-source "c2rust-ast-exporter" "0.20.0"
                "0np9n0k2fw10l07w23yzi58vv15npqcr01rayf5n4rlmhafm6s48"))

(define rust-c2rust-ast-printer-0.20.0
  (crate-source "c2rust-ast-printer" "0.20.0"
                "13n6krmvrrmkrk7sq65iwndbqvxsjri7ipk6hdzc3gv151w6rmv4"))

(define rust-c2rust-bitfields-0.20.0
  (crate-source "c2rust-bitfields" "0.20.0"
                "1pqp22mr43m1v3vcjakc8j0q51b66jbnkp5jgvab7l50zwmpvp26"))

(define rust-c2rust-bitfields-derive-0.20.0
  (crate-source "c2rust-bitfields-derive" "0.20.0"
                "1y1kad9kim7f4w1khx9g4m585lafx2hizqrl00lcwdsrz9x13qgb"))

(define rust-c2rust-build-paths-0.20.0
  (crate-source "c2rust-build-paths" "0.20.0"
                "1y7dljrgrfjn5mgydlqh5f0m1576brc133pjqzw2c19vly1rba16"))

(define rust-c2rust-transpile-0.20.0
  (crate-source "c2rust-transpile" "0.20.0"
                "0801i05pmsp8nvw0r4nx8pnircsdsjb9764bc5gv7sh6wmaavdyz"))

(define rust-cairo-rs-0.18.2
  (crate-source "cairo-rs" "0.18.2"
                "0k8wfv2ri8i417pwb1mvdl51x02nsx52f31rxzg7ivn0m3gnc10w"))

(define rust-cairo-rs-0.18.5
  (crate-source "cairo-rs" "0.18.5"
                "1qjfkcq3mrh3p01nnn71dy3kn99g21xx3j8xcdvzn8ll2pq6x8lc"))

(define rust-cairo-rs-0.19.4
  (crate-source "cairo-rs" "0.19.4"
                "0qp5rixgipdj9d8yd5458hzfxam1rgpzcxi90vq6q0v91r6jmb5j"))

(define rust-cairo-rs-0.20.10
  (crate-source "cairo-rs" "0.20.10"
                "0m9z0hackmqni1zfkjwp2ha46v3yvbyk3fw27zvb6byhgji653mm"))

(define rust-cairo-rs-0.20.12
  (crate-source "cairo-rs" "0.20.12"
                "1l5d1rgvagvvs4a99i28ciyhdygf9v8hhy8mpk5akbr59q7vvqwi"))

(define rust-cairo-rs-0.8.1
  (crate-source "cairo-rs" "0.8.1"
                "11303v1fv6hsc9n70ak380gknkf0098phpcxqdhkmahqjsx4jw0m"))

(define rust-cairo-sys-rs-0.10.0
  (crate-source "cairo-sys-rs" "0.10.0"
                "19wch8zc11hbi724mn16hhqyff8kw5c5bsbdlzpxdwfmkadn7lif"))

(define rust-cairo-sys-rs-0.18.2
  (crate-source "cairo-sys-rs" "0.18.2"
                "0lfsxl7ylw3phbnwmz3k58j1gnqi6kc2hdc7g3bb7f4hwnl9yp38"))

(define rust-cairo-sys-rs-0.19.2
  (crate-source "cairo-sys-rs" "0.19.2"
                "0r0yp0lph77lm4blrn6fvdmz2i3r8ibkkjg6nmwbvvv4jq8v6fzx"))

(define rust-cairo-sys-rs-0.20.10
  (crate-source "cairo-sys-rs" "0.20.10"
                "12sgv9mimxy5nsxm4ipga1k7an59wn444xa7kbywp64qai3cg705"))

(define rust-cairo-sys-rs-0.9.2
  (crate-source "cairo-sys-rs" "0.9.2"
                "0qsdy6s57yvscg2rfm7wdqrlhzbn1aq9lhk3dy1vw5f7r81blrgz"))

(define rust-calamine-0.26.1
  (crate-source "calamine" "0.26.1"
                "1cbfjwb37c28gkb42wsgpp93fxsrzhxgjjza0hc7yp9cmywld1hk"))

(define rust-calendrical-calculations-0.1.3
  (crate-source "calendrical_calculations" "0.1.3"
                "0f1xix1ph9v37ngxw1zq1id7hwv76bk74c4anb4jb1k6bplp6zz9"))

(define rust-calibright-0.1.11
  (crate-source "calibright" "0.1.11"
                "1sf9nl6x89cy478nplll7jqjnlgy68zr7sl14acbsb7rgn0n4jni"))

(define rust-calloop-0.13.0
  (crate-source "calloop" "0.13.0"
                "1v5zgidnhsyml403rzr7vm99f8q6r5bxq5gxyiqkr8lcapwa57dr"))

(define rust-calloop-0.14.2
  (crate-source "calloop" "0.14.2"
                "1jzx8rmgndj1br4gnd4iaxayqi79g897lz6qdy2l670xcqj9g4hh"))

(define rust-calloop-0.14.3
  (crate-source "calloop" "0.14.3"
                "17ih3c840cqksv9ms7i2ynnkiabpvqvpxakbr3922imxd09nx7yb"))

(define rust-calloop-0.9.3
  (crate-source "calloop" "0.9.3"
                "10mbcsd7fj3cg0a463h3003wycv955cnj4pm2gla2sp5xxhyqbmz"))

(define rust-calloop-wayland-source-0.3.0
  (crate-source "calloop-wayland-source" "0.3.0"
                "086x5mq16prrcwd9k6bw9an0sp8bj9l5daz4ziz5z4snf2c6m9lm"))

(define rust-calloop-wayland-source-0.4.0
  (crate-source "calloop-wayland-source" "0.4.0"
                "1bsxx4dz4k4icza63w108n8s1agm7890nl3syigaa9p0pcfplsl7"))

(define rust-camellia-0.1.0
  (crate-source "camellia" "0.1.0"
                "0c6f61rf0gzq7x9d2qmp0330pb397aldwdpmwqybbwly9rby4r1j"
                #:snippet '(delete-file-recursively "tests")))

(define rust-camino-1.1.10
  (crate-source "camino" "1.1.10"
                "1asw3160i5x2r98lsfym3my8dps0fyk25qi206bddn3i271mp90d"))

(define rust-camino-1.1.11
  (crate-source "camino" "1.1.11"
                "1h2150limbipfx5w59cw797nsgwjxn5mjpf33gvpc35hjfdal1sx"))

(define rust-camino-1.1.9
  (crate-source "camino" "1.1.9"
                "1lqszl12l1146jf8g01rvjmapif82mhzih870ln3x0dmcr4yr5lb"))

(define rust-camino-1.2.1
  (crate-source "camino" "1.2.1"
                "0296bpl9smby45fc2583cgb7ybdiqpqcjh4k2f47g5ic5fzmjsi7"))

(define rust-camino-tempfile-1.4.1
  (crate-source "camino-tempfile" "1.4.1"
                "081bgm4zisasfn05gfrvav63sj61imrqipsxjiwqdhx5h968qc34"))

(define rust-canonical-path-2.0.2
  (crate-source "canonical-path" "2.0.2"
                "0vvsjda6ka5nz8zvx6r08zqi0j59sjccgcbjxj96xj764w9y1sg6"))

(define rust-capnp-0.19.8
  (crate-source "capnp" "0.19.8"
                "0vk3vxcq11v4nzxlrdfxmyr4339i1jqi4zcmi919mbnsddb5m62f"))

(define rust-capnp-futures-0.19.1
  (crate-source "capnp-futures" "0.19.1"
                "0ryqnn2v1ipss96i55rq484m1p9cffn4i102a674k41q1f0yxwzq"))

(define rust-capnp-rpc-0.19.5
  (crate-source "capnp-rpc" "0.19.5"
                "00f5vwh0hqx5znkbmwngif4axfda2isq6vpkvbyyc8g1llianmzy"))

(define rust-capnpc-0.19.0
  (crate-source "capnpc" "0.19.0"
                "1v49w7zsv4bkdn88dfmi2hk5dzv5pgs0qwgkq99jsn081w7a6ny7"))

(define rust-caps-0.5.5
  (crate-source "caps" "0.5.5"
                "02vk0w48rncgvfmj2mz2kpzvdgc14z225451w7lvvkwvaansl2qr"))

(define rust-capstone-0.13.0
  (crate-source "capstone" "0.13.0"
                "16mcsipj3x9fbk8lhvp4zqy551i8pnk9nl7r9bwy6hqprbazaph1"
                #:snippet '(delete-file-recursively "test-inputs")))

(define rust-capstone-sys-0.17.0
  (crate-source "capstone-sys" "0.17.0"
                "1809b0is1415n13ayn3wwpknzhmfs7zq8hpccdw1kr512s6wnrr2"
                #:snippet '(delete-file-recursively "capstone")))

(define rust-captur-0.1.0
  (crate-source "captur" "0.1.0"
                "10mi8mz5l9wq6pv1p26237pw4bywn90nra076ghh4ini90jv7ykh"))

(define rust-card-backend-0.2.0
  (crate-source "card-backend" "0.2.0"
                "1ra2zfcs0n4msw4j0hmv8f3bpfz49x5c704i93f6a844k2if6gmx"))

(define rust-card-backend-pcsc-0.5.0
  (crate-source "card-backend-pcsc" "0.5.0"
                "0ddv3jkcyy2vfc6jmlsh87yxcgkhcppp1g9sv670asqvgdq0pfv8"))

(define rust-cargo-0.83.0
  (crate-source "cargo" "0.83.0"
                "0yvb3vh87ngrfsgvl6r2drjrq5h1yw07yhq6c6kc39wmigc1z4l2"
                #:snippet '(delete-file-recursively "benches")))

(define rust-cargo-0.85.0
  (crate-source "cargo" "0.85.0"
                "05n42kxzxhkfj4s2jg2qcw759h2b3piai6p1fm90kx17jhlg9vxv"
                #:snippet '(for-each delete-file-recursively '("benches" "tests"))))

(define rust-cargo-config2-0.1.32
  (crate-source "cargo-config2" "0.1.32"
                "0qf4kkbh3m47n6s3scaqjr41ysn3n153wyhy3yckqhp06sd79hvd"))

(define rust-cargo-credential-0.4.8
  (crate-source "cargo-credential" "0.4.8"
                "0anzvfk11fc1l72h2cm2q9b0i680qk98864h1qcxpqfx184ga7mc"))

(define rust-cargo-credential-libsecret-0.4.12
  (crate-source "cargo-credential-libsecret" "0.4.12"
                "1f2ijp1k4rr7yjk9rppwgga6f2ppsmm6yx9yh0qb85g8dk1dpn78"))

(define rust-cargo-credential-libsecret-0.4.13
  (crate-source "cargo-credential-libsecret" "0.4.13"
                "12vv8y0bq2nr0kf1ah4c2sxbkfqah878xbnnj37nfffxjgjyim02"))

(define rust-cargo-credential-macos-keychain-0.4.12
  (crate-source "cargo-credential-macos-keychain" "0.4.12"
                "0a5ggaba8wb9fj785fq5v2y67r95fw7ips2z1c22rqrvwvdcp2dl"))

(define rust-cargo-credential-macos-keychain-0.4.13
  (crate-source "cargo-credential-macos-keychain" "0.4.13"
                "1k1810bg3g6s21qymfg8gvjycapxv4i0lfhlh865i0nn9fpyads0"))

(define rust-cargo-credential-wincred-0.4.12
  (crate-source "cargo-credential-wincred" "0.4.12"
                "0jrg7d3pnx6rrj2brp53jwfjcb2svcvc8yq7316pl9m5alicpij9"))

(define rust-cargo-credential-wincred-0.4.13
  (crate-source "cargo-credential-wincred" "0.4.13"
                "0f574vad13zc6ygnm8ji5z9m19pbfnpl9j2wd3l3yw235pyr211j"))

(define rust-cargo-husky-1.5.0
  (crate-source "cargo-husky" "1.5.0"
                "1b9jx720dzw9s7rl82bywz4d089c9rb0j526c1jfzs1g4llvc0kv"))

(define rust-cargo-lock-10.1.0
  (crate-source "cargo-lock" "10.1.0"
                "0m74y8w9wn7rl5mpzr0436r6fshf3qhm7d3wl02s4ys0f57wnsn0"))

(define rust-cargo-manifest-0.13.0
  (crate-source "cargo-manifest" "0.13.0"
                "13a0dgqchxjmhr4idswpri2l3lwv7fxga69yj5hvylla0adg8vxz"))

(define rust-cargo-metadata-0.14.2
  (crate-source "cargo_metadata" "0.14.2"
                "1yl1y40vby9cas4dlfc44szrbl4m4z3pahv3p6ckdqp8ksfv1jsa"))

(define rust-cargo-metadata-0.15.4
  (crate-source "cargo_metadata" "0.15.4"
                "0fm767r349p5rh50k21g8w2hpcg2970kjx0ff517mz163wzj9r7f"))

(define rust-cargo-metadata-0.18.1
  (crate-source "cargo_metadata" "0.18.1"
                "0drh0zndl4qgndy6kg6783cydbvhxgv0hcg7d9hhqx0zwi3nb21d"))

(define rust-cargo-metadata-0.19.2
  (crate-source "cargo_metadata" "0.19.2"
                "1fkr8jp6vhva4kc3rhx13yrnl8g3zch463j20vbwa9scxlabcpnx"))

(define rust-cargo-metadata-0.20.0
  (crate-source "cargo_metadata" "0.20.0"
                "00i5m052f9p01q2fcsrsvh1w9dfrwm9jpswm1q3r6l0kqv7kay2g"))

(define rust-cargo-options-0.7.5
  (crate-source "cargo-options" "0.7.5"
                "0wc1qy1plwp6i0g5p74bnvy545xk7hccvv68lmmg4739g0ay923l"
                #:snippet '(delete-file-recursively "tests")))

(define rust-cargo-platform-0.1.9
  (crate-source "cargo-platform" "0.1.9"
                "1sinpmqjdk3q9llbmxr0h0nyvqrif1r5qs34l000z73b024z2np3"))

(define rust-cargo-platform-0.2.0
  (crate-source "cargo-platform" "0.2.0"
                "1m7bk5ry59lz52kwm0xir0skflb5z7gdxrjf79d66hz319n2r644"))

(define rust-cargo-test-macro-0.3.3
  (crate-source "cargo-test-macro" "0.3.3"
                "0bvxw120y5pdy7p6kqzxf473q0kwwrbw0pk45rdmzxr72g0kr1vc"))

(define rust-cargo-test-support-0.6.0
  (crate-source "cargo-test-support" "0.6.0"
                "01l0c1wksfksj5mj0fb51hv70gsrnj2wibl0dyj5mpcx1nl24wcy"))

(define rust-cargo-toml-0.21.0
  (crate-source "cargo_toml" "0.21.0"
                "0wp4qa1f6iwm149i61ng4njddhh40mxszbb0j2w73gryvglizgaz"))

(define rust-cargo-util-0.2.19
  (crate-source "cargo-util" "0.2.19"
                "02m2xjvq02iiz9dp9v6m792w55jrka3vwll0ca82wjc09qm6wzsj"))

(define rust-cargo-util-0.2.20
  (crate-source "cargo-util" "0.2.20"
                "0xixfl8hfvaj02l226km6rvfxmn0ymb0nhvjc0x4ixk7yf2vqryp"))

(define rust-cargo-util-schemas-0.2.0
  (crate-source "cargo-util-schemas" "0.2.0"
                "1jncrbsh5w8pl968a04893mwkaq0cgav19zyy6wpwj4lmj02fgg6"))

(define rust-cargo-util-schemas-0.6.0
  (crate-source "cargo-util-schemas" "0.6.0"
                "163x9cfg5g9ilz1ra3zr5nzk34qg8h4bxgsbrpgff6saaja2f19w"))

(define rust-cargo-util-schemas-0.7.2
  (crate-source "cargo-util-schemas" "0.7.2"
                "014qha8zp604n7yh2scbn2jz0q2ki6jmi61db61832nbz1l5z44z"))

(define rust-caseless-0.2.2
  (crate-source "caseless" "0.2.2"
                "1n75pz65q9zn5rqq2xlw43dkq69rljnsd73nzpnwi1j08l3xavwb"))

(define rust-cassowary-0.3.0
  (crate-source "cassowary" "0.3.0"
                "0lvanj0gsk6pc1chqrh4k5k0vi1rfbgzmsk46dwy3nmrqyw711nz"))

(define rust-cast-0.3.0
  (crate-source "cast" "0.3.0"
                "1dbyngbyz2qkk0jn2sxil8vrz3rnpcj142y184p9l4nbl9radcip"))

(define rust-cast5-0.11.1
  (crate-source "cast5" "0.11.1"
                "04crg8dj6lxbp3lmdc3filsahxcyvccvhm0gx40g1k5i7mkpvc16"))

(define rust-castaway-0.2.2
  (crate-source "castaway" "0.2.2"
                "1k1z4v61vq7la56js1azkr0k9b415vif2kaxiqk3d1gw6mbfs5wa"))

(define rust-castaway-0.2.3
  (crate-source "castaway" "0.2.3"
                "1mf0wypwnkpa1hi0058vp8g7bjh2qraip2qv7dmak7mg1azfkfha"))

(define rust-castaway-0.2.4
  (crate-source "castaway" "0.2.4"
                "0nn5his5f8q20nkyg1nwb40xc19a08yaj4y76a8q2y3mdsmm3ify"))

(define rust-catppuccin-2.5.1
  (crate-source "catppuccin" "2.5.1"
                "0zdc80bghdndr1ii78y9jjyp55biawn1a7m9ncdprpah5icmc8ym"
                #:snippet '(delete-file-recursively "assets")))

(define rust-cbc-0.1.2
  (crate-source "cbc" "0.1.2"
                "19l9y9ccv1ffg6876hshd123f2f8v7zbkc4nkckqycxf8fajmd96"
                #:snippet '(delete-file-recursively "tests")))

(define rust-cbindgen-0.25.0
  (crate-source "cbindgen" "0.25.0"
                "1jl82k816rxwi2ks9smnzlqv02fbq0sqy6r1g5d9f9x7wn9sdsps"))

(define rust-cbindgen-0.27.0
  (crate-source "cbindgen" "0.27.0"
                "1sqm3axr678d72yihgmpr9d17mj99ccibxfqhw53mgzwzkbqvkiz"))

(define rust-cbindgen-0.28.0
  (crate-source "cbindgen" "0.28.0"
                "1zyiaifg6mcd4wwhhbxk8adzhph6qz4wxzgagvg3ijp95j58dpga"))

(define rust-cc-1.0.82
  (crate-source "cc" "1.0.82"
                "00cgp2xjw0jbryp2xqajgb9rh9s23nk6nwmnm07jli61xm2ycprh"))

(define rust-cc-1.0.83
  (crate-source "cc" "1.0.83"
                "1l643zidlb5iy1dskc5ggqs4wqa29a02f44piczqc8zcnsq4y5zi"))

(define rust-cc-1.2.1
  (crate-source "cc" "1.2.1"
                "0izavwg0w3ahhp8jhajy97adv53r6ggv11g6czyz6pjx43rfk7gx"))

(define rust-cc-1.2.16
  (crate-source "cc" "1.2.16"
                "131bhgafc1i86vvjipkj0kwzz0hlpwrkl8mdbmzyq2g69calqwdy"))

(define rust-cc-1.2.18
  (crate-source "cc" "1.2.18"
                "0p6d2pfyrjgqpf2w399wzj4hmyffj6g0gyzg3pdy6xl3gmhlcl2j"))

(define rust-cc-1.2.19
  (crate-source "cc" "1.2.19"
                "0qpk2cnbfpp5r3sb5c9clb49n39x0y64b6xk2cf6hn69g9q16flf"))

(define rust-cc-1.2.22
  (crate-source "cc" "1.2.22"
                "1lfg9bd56iq6mf5rwndjnj3mrc6r9kwk5jc1g360licqz7nrbnrj"))

(define rust-cc-1.2.24
  (crate-source "cc" "1.2.24"
                "1irvbn8y9sg6f1070yg5469fxk5c3ximh24ds04kph21w0xmsn8n"))

(define rust-cc-1.2.25
  (crate-source "cc" "1.2.25"
                "0la999i153miffa4maj2wk5z679zkmnq6np0q1kwqrg8q5yqkz6h"))

(define rust-cc-1.2.26
  (crate-source "cc" "1.2.26"
                "1b5g9ln7a2imwhrvfi77qbmj7gxsg0xihrlvarrg71wbk0hmwslm"))

(define rust-cc-1.2.27
  (crate-source "cc" "1.2.27"
                "1p5zfsl2mw3j46w58j2sxqkbfi49azilis5335pxlr2z3c3sm1yl"))

(define rust-cc-1.2.29
  (crate-source "cc" "1.2.29"
                "0qlkaspjmywvjyfqhpv2x4kwrqs6b69zg33wfi2l8fg2im9rj5aw"))

(define rust-cc-1.2.34
  (crate-source "cc" "1.2.34"
                "1p5ycww65h7xca03lwdp264qalw8v357rg5h17s7naq3h3m4mg22"))

(define rust-cc-1.2.35
  (crate-source "cc" "1.2.35"
                "18vfhd6njr0lhfvfvxchj3bay4fw6hcpyy4130sl134alqj903sr"))

(define rust-cc-1.2.36
  (crate-source "cc" "1.2.36"
                "0m3dg927zvvfvnxzifwk3rf041syg4y1wl3g3ayfsplfck9b6ljj"))

(define rust-cc-1.2.37
  (crate-source "cc" "1.2.37"
                "0i5xlxsgd7jif1ry9k3ysnpsmbrckapqwq8d8l5vhkj0qs4ka6b5"))

(define rust-cc-1.2.38
  (crate-source "cc" "1.2.38"
                "1sg7gd94611qhryvb0iip0zibjnhf1yha2wnp0pw2mgrd3himx40"))

(define rust-cc-1.2.39
  (crate-source "cc" "1.2.39"
                "0py3546wz3k5qi6pbfz80jvg0g3qgzr21c7a1p5wjvscjm4l6dg1"))

(define rust-cedarwood-0.4.6
  (crate-source "cedarwood" "0.4.6"
                "142fw2aj0c34v56bsml95v9cjlq88r3x5gnhccr7691csvnhp4bd"))

(define rust-cesu8-1.1.0
  (crate-source "cesu8" "1.1.0"
                "0g6q58wa7khxrxcxgnqyi9s1z2cjywwwd3hzr5c55wskhx6s0hvd"))

(define rust-cexpr-0.3.6
  (crate-source "cexpr" "0.3.6"
                "07fdfj4ff2974y33yixrb657riq9zl9b9h9lr0h7ridhhvxvbrgw"))

(define rust-cexpr-0.6.0
  (crate-source "cexpr" "0.6.0"
                "0rl77bwhs5p979ih4r0202cn5jrfsrbgrksp40lkfz5vk1x3ib3g"))

(define rust-cfb-0.7.3
  (crate-source "cfb" "0.7.3"
                "03y6p3dlm7gfds19bq4ba971za16rjbn7q2v0vqcri52l2kjv3yk"
                #:snippet '(delete-file-recursively "tests")))

(define rust-cfb-mode-0.8.2
  (crate-source "cfb-mode" "0.8.2"
                "0c6kd34jk4p52vr0qgn9slj6zdgmc42gfcqr6mqhmy37g138v2vk"
                #:snippet '(delete-file-recursively "tests")))

(define rust-cfg-aliases-0.1.1
  (crate-source "cfg_aliases" "0.1.1"
                "17p821nc6jm830vzl2lmwz60g3a30hcm33nk6l257i1rjdqw85px"))

(define rust-cfg-aliases-0.2.1
  (crate-source "cfg_aliases" "0.2.1"
                "092pxdc1dbgjb6qvh83gk56rkic2n2ybm4yvy76cgynmzi3zwfk1"))

(define rust-cfg-expr-0.15.5
  (crate-source "cfg-expr" "0.15.5"
                "1cqicd9qi8mzzgh63dw03zhbdihqfl3lbiklrkynyzkq67s5m483"))

(define rust-cfg-expr-0.15.8
  (crate-source "cfg-expr" "0.15.8"
                "00lgf717pmf5qd2qsxxzs815v6baqg38d6m5i6wlh235p14asryh"))

(define rust-cfg-expr-0.18.0
  (crate-source "cfg-expr" "0.18.0"
                "03biza6hs2nj4pzqmr5rbvxq1zcl0vnbs21ava90z62rc4938aqs"))

(define rust-cfg-expr-0.20.0
  (crate-source "cfg-expr" "0.20.0"
                "0ibznd2fv4qnlmagjlb4cb03h2b7c4dczjdm2n1mxsy7j4g24kp3"))

(define rust-cfg-expr-0.20.2
  (crate-source "cfg-expr" "0.20.2"
                "01l25g563n04ci6cjv0509d8ig91djvwidx9iln4h3qg7zb5im68"))

(define rust-cfg-if-0.1.10
  (crate-source "cfg-if" "0.1.10"
                "08h80ihs74jcyp24cd75wwabygbbdgl05k6p5dmq8akbr78vv1a7"))

(define rust-cfg-if-1.0.0
  (crate-source "cfg-if" "1.0.0"
                "1za0vb97n4brpzpv8lsbnzmq5r8f2b0cpqqr0sy8h5bn751xxwds"))

(define rust-cfg-if-1.0.1
  (crate-source "cfg-if" "1.0.1"
                "0s0jr5j797q1vqjcd41l0v5izlmlqm7lxy512b418xz5r65mfmcm"))

(define rust-cfg-if-1.0.3
  (crate-source "cfg-if" "1.0.3"
                "1afg7146gbxjvkbjx7i5sdrpqp9q5akmk9004fr8rsm90jf2il9g"))

(define rust-cgl-0.3.2
  (crate-source "cgl" "0.3.2"
                "1zs7skrsyrsm759vfy2cygkx52fx91b567a12bpaz1sf4d8hbv8c"))

(define rust-cgmath-0.18.0
  (crate-source "cgmath" "0.18.0"
                "05sk7c1c1jg5ygqvc3y77kxddp177gwazfibhd864ag3800x760s"))

(define rust-chacha20-0.9.1
  (crate-source "chacha20" "0.9.1"
                "0678wipx6kghp71hpzhl2qvx80q7caz3vm8vsvd07b1fpms3yqf3"
                #:snippet '(delete-file-recursively "tests")))

(define rust-chacha20poly1305-0.10.1
  (crate-source "chacha20poly1305" "0.10.1"
                "0dfwq9ag7x7lnd0znafpcn8h7k4nfr9gkzm0w7sc1lcj451pkk8h"
                #:snippet '(delete-file-recursively "tests")))

(define rust-chan-0.1.23
  (crate-source "chan" "0.1.23"
                "1n0y992mqfk5zpxzvrv14g9qivacmd4fiv4j1nmgyrg0vaimcjfi"))

(define rust-chardetng-0.1.17
  (crate-source "chardetng" "0.1.17"
                "1spikjcnblwa5n1nnk46fxkwn86yfiqxgs47h4yaw23vbfvg1f0l"))

(define rust-charset-0.1.5
  (crate-source "charset" "0.1.5"
                "0zkwcw525qwcqsdf74l9d2r6m69yxfxb4kgywp3q9fklgjq2gygi"))

(define rust-checked-int-cast-1.0.0
  (crate-source "checked_int_cast" "1.0.0"
                "06brva5agm6g12q15f8fidz17akb85q211496p1k2qxhb9mmxk0p"))

(define rust-chic-1.2.2
  (crate-source "chic" "1.2.2"
                "037pkdccj25gr4my8fq1qni9v87rydpyhfi2naf86mimkxhxpdd5"))

(define rust-chinese-number-0.7.7
  (crate-source "chinese-number" "0.7.7"
                "0v5va8drix8gs2kv6pmv5yzdxhlpzrwkp3ch86kxdxj6cgpwmz29"))

(define rust-chinese-variant-1.1.3
  (crate-source "chinese-variant" "1.1.3"
                "12s91vg2m9wfs9b3f0q2alj9am08y7r2prb0szg3fwjh8m8lg23m"))

(define rust-chrono-0.4.30
  (crate-source "chrono" "0.4.30"
                "0xz8bldv1kqb82ak3b54lv2animrzk3rkqciginbmpfvfdw4xzfy"))

(define rust-chrono-0.4.40
  (crate-source "chrono" "0.4.40"
                "0z334kqnvq5zx6xsq1k6zk8g9z14fgk2w3vkn4n13pvi3mhn8y8s"))

(define rust-chrono-0.4.41
  (crate-source "chrono" "0.4.41"
                "0k8wy2mph0mgipq28vv3wirivhb31pqs7jyid0dzjivz0i9djsf4"))

(define rust-chrono-0.4.42
  (crate-source "chrono" "0.4.42"
                "1lp8iz9js9jwxw0sj8yi59v54lgvwdvm49b9wch77f25sfym4l0l"))

(define rust-chrono-humanize-0.2.3
  (crate-source "chrono-humanize" "0.2.3"
                "0fq25fcdqd7s39dx81hq123210q4lpcbjdz82jl2fy6jnkk2g5kr"))

(define rust-chrono-tz-0.10.3
  (crate-source "chrono-tz" "0.10.3"
                "1wxl6jm6x5z2m6bh10dhi9s8hyqb6kmgd3naf6h37wbhqd4y3p7g"))

(define rust-chrono-tz-0.9.0
  (crate-source "chrono-tz" "0.9.0"
                "1fvicqrlmdsjkrgxr7bxfd62i9w2qi2b6iv4w85av5syvqlqnsck"))

(define rust-chrono-tz-build-0.3.0
  (crate-source "chrono-tz-build" "0.3.0"
                "1c8ixwwwsn9kgs1dr5mz963p0fgw9j9p7fzb3w2c7y8xhkp8l20c"))

(define rust-chrono-tz-build-0.4.1
  (crate-source "chrono-tz-build" "0.4.1"
                "00llaqz0mhzhayswlxxx9d3qxd50akdzr1gq1w9gqc8f6k4zh44g"))

(define rust-chumsky-0.9.3
  (crate-source "chumsky" "0.9.3"
                "1jcnafc8rjfs1al08gqzyn0kpbaizgdwrd0ajqafspd18ikxdswf"))

(define rust-chunked-transfer-1.5.0
  (crate-source "chunked_transfer" "1.5.0"
                "00a9h3csr1xwkqrzpz5kag4h92zdkrnxq4ppxidrhrx29syf6kbf"))

(define rust-ci-info-0.10.2
  (crate-source "ci_info" "0.10.2"
                "0pn4lfq630bf6aki3kz41555b4fs8iyc1a69biwm6mwc1v3kixi4"
                #:snippet '(delete-file-recursively "docs")))

(define rust-ciborium-0.2.2
  (crate-source "ciborium" "0.2.2"
                "03hgfw4674im1pdqblcp77m7rc8x2v828si5570ga5q9dzyrzrj2"))

(define rust-ciborium-io-0.2.2
  (crate-source "ciborium-io" "0.2.2"
                "0my7s5g24hvp1rs1zd1cxapz94inrvqpdf1rslrvxj8618gfmbq5"))

(define rust-ciborium-ll-0.2.2
  (crate-source "ciborium-ll" "0.2.2"
                "1n8g4j5rwkfs3rzfi6g1p7ngmz6m5yxsksryzf5k72ll7mjknrjp"))

(define rust-cipher-0.3.0
  (crate-source "cipher" "0.3.0"
                "1dyzsv0c84rgz98d5glnhsz4320wl24x3bq511vnyf0mxir21rby"))

(define rust-cipher-0.4.4
  (crate-source "cipher" "0.4.4"
                "1b9x9agg67xq5nq879z66ni4l08m6m3hqcshk37d4is4ysd3ngvp"))

(define rust-citationberg-0.5.0
  (crate-source "citationberg" "0.5.0"
                "1yj4z1iivzw5jmcwlb32zig54qindllbb0000wsh5d5gpq1mwng4"))

(define rust-claims-0.7.1
  (crate-source "claims" "0.7.1"
                "1da6z2r4zz4fw4a69286s54jzr7g7sz3dspq0xiw6mk432z5p6dn"))

(define rust-clang-sys-0.26.4
  (crate-source "clang-sys" "0.26.4"
                "1r50dwy5hj5gq07dn0qf8222d07qv0970ymx0j8n9779yayc3w3f"))

(define rust-clang-sys-1.6.1
  (crate-source "clang-sys" "1.6.1"
                "0vxg1dchm5nhiqj1z3nm5dmddqbbm0q8ibjcd3my62188dsgr266"))

(define rust-clang-sys-1.8.1
  (crate-source "clang-sys" "1.8.1"
                "1x1r9yqss76z8xwpdanw313ss6fniwc1r7dzb5ycjn0ph53kj0hb"))

(define rust-clap-2.34.0
  (crate-source "clap" "2.34.0"
                "071q5d8jfwbazi6zhik9xwpacx5i6kb2vkzy060vhf0c3120aqd0"))

(define rust-clap-3.2.25
  (crate-source "clap" "3.2.25"
                "08vi402vfqmfj9f07c4gl6082qxgf4c9x98pbndcnwbgaszq38af"))

(define rust-clap-4.3.21
  (crate-source "clap" "4.3.21"
                "1zd3qgkjgdlq0kmmsw0n90z53s4dkx09l31ba4zvl17nq0ldyz62"))

(define rust-clap-4.3.24
  (crate-source "clap" "4.3.24"
                "11z4695sw0kp1s4rfnfdgg19n6xllcp297z5mmx0s344qy0hwsgv"))

(define rust-clap-4.5.31
  (crate-source "clap" "4.5.31"
                "0ryp6xjbdc9cbjjkafjl35j91pvv0ykislwqhr537bi9hkcv0yq2"))

(define rust-clap-4.5.35
  (crate-source "clap" "4.5.35"
                "0i1rnz7mwbhs5qf10r6vmrkplkzm3477khkwz189rha49f9qdanq"))

(define rust-clap-4.5.36
  (crate-source "clap" "4.5.36"
                "011f220kg4qjmjzdsv5078zywphlhkswq62pjjlqml50r3c63y9d"))

(define rust-clap-4.5.37
  (crate-source "clap" "4.5.37"
                "0wg01bc78ivsh76g1cixyhd2zw7ikvkaim40nd087lybar7hbjzc"))

(define rust-clap-4.5.38
  (crate-source "clap" "4.5.38"
                "0060d8gx692via31il45pdnc6pix8l2hynf28bgk1acbby0bk4zd"))

(define rust-clap-4.5.39
  (crate-source "clap" "4.5.39"
                "17raqwxkhhhm80iyblp1v83fvpddkg7rgqr2cjsmz3p6kczfcq7x"))

(define rust-clap-4.5.40
  (crate-source "clap" "4.5.40"
                "03widrb9d7a0bka6lsf9r9f65zhfbkdkhm8iryycx1c63mx8idj0"))

(define rust-clap-4.5.41
  (crate-source "clap" "4.5.41"
                "1ydd3a22bxkn2a7bajnw57cwjhawqciyhz2x3rqm8fi4h0pd74my"))

(define rust-clap-4.5.45
  (crate-source "clap" "4.5.45"
                "0663m85dd0aq1g3mkwz5b8pkjv4f5k2smlz7bagib4iqf15fgh0z"))

(define rust-clap-4.5.46
  (crate-source "clap" "4.5.46"
                "0mvfywfsyyps8wkz7nx75aj1jc715maxis8yr92fbli1kk7lypic"))

(define rust-clap-4.5.47
  (crate-source "clap" "4.5.47"
                "0c99f6m4a7d4ffgahid49h0ci2pv4ccdf417f76nl4wx5n801b3y"))

(define rust-clap-4.5.48
  (crate-source "clap" "4.5.48"
                "1bjz3d7bavy13ph2a6rm3c9y02ak70b195xakii7h6q2xarln4z2"))

(define rust-clap-builder-4.3.21
  (crate-source "clap_builder" "4.3.21"
                "1am4bavjhp26pv0ymsprmdqdw44sxdi8abi03ywaj0czbsmz3a88"))

(define rust-clap-builder-4.3.24
  (crate-source "clap_builder" "4.3.24"
                "03l6d45csywv8xazs44jwy1hk2jbvznn7mj8dw7p93bdq5mykljy"))

(define rust-clap-builder-4.5.31
  (crate-source "clap_builder" "4.5.31"
                "0qyqd6kfcs41x29a95n15744jyv2v07srvwi6z9g7q3jl35y12am"))

(define rust-clap-builder-4.5.35
  (crate-source "clap_builder" "4.5.35"
                "1nczcw6cc49ap99nn3v3n0vrv7j74zin34palq6ji586vnrdn514"))

(define rust-clap-builder-4.5.36
  (crate-source "clap_builder" "4.5.36"
                "198ks2kqmsjir40vliwy1lpyza2229d9sj3a65w3hxdn1yjbsb8k"))

(define rust-clap-builder-4.5.37
  (crate-source "clap_builder" "4.5.37"
                "18k293yx8g16si1k12lch3z6y4ay5ivd9jhz7dfjahw5mipldngg"))

(define rust-clap-builder-4.5.38
  (crate-source "clap_builder" "4.5.38"
                "0821n0ri2nf1xqj11q1fn78i2hhw6qs96qpan08zdb1z53zjd41p"))

(define rust-clap-builder-4.5.39
  (crate-source "clap_builder" "4.5.39"
                "0lggb5vscs21jliisvjjphcazzb1iw8347yp42wbwazpl6967k49"))

(define rust-clap-builder-4.5.40
  (crate-source "clap_builder" "4.5.40"
                "17pmcjwk6rbkizj4y5vlhrnr7b5n1ffjgh75pj66j34zrq46rip0"))

(define rust-clap-builder-4.5.41
  (crate-source "clap_builder" "4.5.41"
                "0g8w6da5y13kv93psl8c00c7f4q01753wmwx84wr2bv2x50snzkh"))

(define rust-clap-builder-4.5.44
  (crate-source "clap_builder" "4.5.44"
                "1a48x3c9q1l7r6wbgy71mq6kfsihpqzxsnbaaamcgwvp88hz9rxk"))

(define rust-clap-builder-4.5.46
  (crate-source "clap_builder" "4.5.46"
                "0hbxwmi3amm9ls0vx3gkz9z55ylj4lpbq0b0d1ghbyzwwsh57jzy"))

(define rust-clap-builder-4.5.47
  (crate-source "clap_builder" "4.5.47"
                "1mp1f0fz6wp9v87jc9372lg6r4514ja1l8cazf25hfz7a3vvpn9a"))

(define rust-clap-builder-4.5.48
  (crate-source "clap_builder" "4.5.48"
                "1jaxnr7ik25r4yxgz657vm8kz62f64qmwxhplmzxz9n0lfpn9fn2"))

(define rust-clap-cargo-0.15.2
  (crate-source "clap-cargo" "0.15.2"
                "1rai72khjaj16b5751b8kjipnxl54bj4pdg93kda9gzj9zlg0inm"))

(define rust-clap-complete-3.2.5
  (crate-source "clap_complete" "3.2.5"
                "1n3whjkznszrxif1hzvql7hav7agq85j456fmwjwwi9cjq52wyiz"))

(define rust-clap-complete-4.5.3
  (crate-source "clap_complete" "4.5.3"
                "0lk3k40h0jlj6vpj7cs6n9amlzkrk3byaawbf06mbkkzqbpdd8mm"))

(define rust-clap-complete-4.5.47
  (crate-source "clap_complete" "4.5.47"
                "1dkzjgmi0c4jgq4cwvmzbaki9mxanll6d0mw5gwd8ji6x9w56vy0"))

(define rust-clap-complete-4.5.48
  (crate-source "clap_complete" "4.5.48"
                "1jdlrmn80bv4k6fhmsjk9dsih790fnxal4nwr8j9wazhlvrrg35y"))

(define rust-clap-complete-4.5.50
  (crate-source "clap_complete" "4.5.50"
                "1qac34cwa0d8v2k3mf2s4byq9cx04r0qfa7gwrh9v26d7fm3n7f9"))

(define rust-clap-complete-4.5.52
  (crate-source "clap_complete" "4.5.52"
                "03whszycrd16j6gi9nmclhgsvwrdkvdvwkzw6s1q631dwhwlcm8s"))

(define rust-clap-complete-4.5.54
  (crate-source "clap_complete" "4.5.54"
                "0xsnd9p6v7zm49ipfl0mzqxizv0f0fbqid3j4r0avzh4vssb3mda"))

(define rust-clap-complete-4.5.55
  (crate-source "clap_complete" "4.5.55"
                "16i2qv263ndlmnms4vyzdqiqd7y4cqdqz3wbpv2p1bvd912dxax5"))

(define rust-clap-complete-4.5.57
  (crate-source "clap_complete" "4.5.57"
                "1bbixanlxdsb47qhk9fp1jl31vbk218rmnh1xsxzf2az7yyh35ad"))

(define rust-clap-complete-4.5.58
  (crate-source "clap_complete" "4.5.58"
                "0jmg0idg96cvx51l35ypia1np3q7sfj5wqxvi7kjs59fmlr0pgvm"))

(define rust-clap-complete-command-0.6.1
  (crate-source "clap_complete_command" "0.6.1"
                "0qhv99j7msqyw7j17hswqwpqbdvqawy8l7ip6rnnh5930n61k3ns"))

(define rust-clap-complete-fig-4.5.2
  (crate-source "clap_complete_fig" "4.5.2"
                "0sy88ybw33ba7qj02caxr9jv03wq1f8rdbrbqw81i5gkiwn1156l"))

(define rust-clap-complete-nushell-4.5.5
  (crate-source "clap_complete_nushell" "4.5.5"
                "12miqxh9g7q37w11bgv55b32s0hdf6avf0lhagzc5psp6icv3a66"))

(define rust-clap-complete-nushell-4.5.7
  (crate-source "clap_complete_nushell" "4.5.7"
                "0g9g5chkshkkdhqr8ry26cbs8ly0qq099zkf2yrpy6cd75dk7f6d"))

(define rust-clap-complete-nushell-4.5.8
  (crate-source "clap_complete_nushell" "4.5.8"
                "1kixnzc8rjqjhk189s1jjvg24v21d1ymj7a2knzna7k9jhb9a30a"))

(define rust-clap-derive-3.2.25
  (crate-source "clap_derive" "3.2.25"
                "025hh66cyjk5xhhq8s1qw5wkxvrm8hnv5xwwksax7dy8pnw72qxf"))

(define rust-clap-derive-4.3.12
  (crate-source "clap_derive" "4.3.12"
                "0l50j37jwbzfl29bd2a1ib7z1jpch4b9868himrgwpgwb1bvpaal"))

(define rust-clap-derive-4.5.32
  (crate-source "clap_derive" "4.5.32"
                "1mqcag8qapb5yhygg2hi153kzmbf7w5hqp3nl3fvl5cn4yp6l5q9"))

(define rust-clap-derive-4.5.40
  (crate-source "clap_derive" "4.5.40"
                "1kjp4928wy132inisss42750rzv0wasvbbf10w98agfcwix99iyj"))

(define rust-clap-derive-4.5.41
  (crate-source "clap_derive" "4.5.41"
                "14glxqpfjs7z6m37f3ycrhgdkpyqmgwbr4vk1y34rjjrd8w54kzg"))

(define rust-clap-derive-4.5.45
  (crate-source "clap_derive" "4.5.45"
                "1xir8wn5d10wpmnzmzjf2k1ib7j5mmzsm6v3yap6qlvx1axk3jql"))

(define rust-clap-derive-4.5.47
  (crate-source "clap_derive" "4.5.47"
                "174z9g13s85la2nmi8gv8ssjwz77im3rqg5isiinw6hg1fp7xzdv"))

(define rust-clap-lex-0.2.4
  (crate-source "clap_lex" "0.2.4"
                "1ib1a9v55ybnaws11l63az0jgz5xiy24jkdgsmyl7grcm3sz4l18"))

(define rust-clap-lex-0.5.0
  (crate-source "clap_lex" "0.5.0"
                "06vcvpvp65qggc5agbirzqk2di00gxg6vazzc3qlwzkw70qxm9id"))

(define rust-clap-lex-0.5.1
  (crate-source "clap_lex" "0.5.1"
                "0qgrlq509vr49wq91jh50f9pm5f8lxmv1rcbklxnsg4nprxcaz6d"))

(define rust-clap-lex-0.7.4
  (crate-source "clap_lex" "0.7.4"
                "19nwfls5db269js5n822vkc8dw0wjq2h1wf0hgr06ld2g52d2spl"))

(define rust-clap-lex-0.7.5
  (crate-source "clap_lex" "0.7.5"
                "0xb6pjza43irrl99axbhs12pxq4sr8x7xd36p703j57f5i3n2kxr"))

(define rust-clap-mangen-0.2.26
  (crate-source "clap_mangen" "0.2.26"
                "0fn1svjqm3znajji679nc2kfwm9lkyl73lzknf4rnkqlkgx44j3j"))

(define rust-clap-mangen-0.2.29
  (crate-source "clap_mangen" "0.2.29"
                "1qkr87xz9mgvh9mqqqqni7fw8dffzihmzwj7rfnxkw1h9g2w7d17"))

(define rust-clap-markdown-0.1.5
  (crate-source "clap-markdown" "0.1.5"
                "0f93ij22sxl1ik0sz25h23n9zc7b0x9pnschnj2lhvd0arwn38nj"))

(define rust-clap-stdin-0.5.1
  (crate-source "clap-stdin" "0.5.1"
                "0gw0hpi8d99yqs71c40jjfw4q7l94a25jnyssgkw3grkcs4zf7a7"))

(define rust-clap-verbosity-flag-2.2.3
  (crate-source "clap-verbosity-flag" "2.2.3"
                "1qp128iwwjymrqvpsb4kn8qpfsc4514dfkvqh8jzcmvm0ikpzirl"))

(define rust-clearscreen-2.0.1
  (crate-source "clearscreen" "2.0.1"
                "15799yi8azfkn86m1i1kgb7xxk7k321gf8zx5vk08rjq38pz5wvj"
                #:snippet '(delete-file-recursively "logo.png")))

(define rust-clearscreen-4.0.1
  (crate-source "clearscreen" "4.0.1"
                "1x72fipqn3xk44v13ycxf41m97rhh99gdfr4h9hf963vb91xqhcc"
                #:snippet '(delete-file-recursively "logo.png")))

(define rust-clipboard-0.5.0
  (crate-source "clipboard" "0.5.0"
                "1rxjfn811h09g6jpjjs2vx7z52wj6dxnflbwryfj6h03dij09a95"))

(define rust-clipboard-win-2.2.0
  (crate-source "clipboard-win" "2.2.0"
                "0svqk0lrw66abaxd6h7l4k4g2s5vd1dcipy34kzfan6mzvb97873"))

(define rust-clipboard-win-3.1.1
  (crate-source "clipboard-win" "3.1.1"
                "0hh3npqfa1lfn62fwvkmjlpfnizq343a994b898ffsvb100mxpwz"))

(define rust-clipboard-win-4.5.0
  (crate-source "clipboard-win" "4.5.0"
                "0qh3rypkf1lazniq4nr04hxsck0d55rigb5sjvpvgnap4dyc54bi"))

(define rust-clipboard-win-5.4.0
  (crate-source "clipboard-win" "5.4.0"
                "14n87fc0vzbd0wdhqzvcs1lqgafsncplzcanhpik93xhhalfgvqm"))

(define rust-clipboard-win-5.4.1
  (crate-source "clipboard-win" "5.4.1"
                "1m44gqy11rq1ww7jls86ppif98v6kv2wkwk8p17is86zsdq3gq5x"))

(define rust-clircle-0.4.0
  (crate-source "clircle" "0.4.0"
                "0056l1bf2zmd7xs13xp8gxw9j9dngw4im0m8ijyifkrmsnz7rs68"))

(define rust-close-fds-0.3.2
  (crate-source "close_fds" "0.3.2"
                "1vd2i1gkvfcgdlzgrkgivhx3ky0zs98g8q3mwmwrxmg97pridi1v"))

(define rust-cloud-file-0.2.0
  (crate-source "cloud-file" "0.2.0"
                "1xvqmizskv88c9mgb78y2aclkbmq1bhlvsnk20xkx6x2wnlwyd51"))

(define rust-cloudabi-0.0.3
  (crate-source "cloudabi" "0.0.3"
                "0kxcg83jlihy0phnd2g8c2c303px3l2p3pkjz357ll6llnd5pz6x"))

(define rust-cloudabi-0.1.0
  (crate-source "cloudabi" "0.1.0"
                "0rv4yf5jlldfkynzrw687s00f4x12ypw7axv71vawhy6h4i52i23"))

(define rust-clru-0.6.2
  (crate-source "clru" "0.6.2"
                "0ngyycxpxif84wpjjn0ixywylk95h5iv8fqycg2zsr3f0rpggl6b"))

(define rust-cmac-0.7.2
  (crate-source "cmac" "0.7.2"
                "1an1vcala24grlyhvk71ikxk2kmgcbal9kgrzzpjcl9z7i74ahw5"
                #:snippet '(delete-file-recursively "tests")))

(define rust-cmake-0.1.54
  (crate-source "cmake" "0.1.54"
                "1w41ma28yzad9x757s9sfq1wigjs9j902hbzc0nbxpc9vvws7jp7"))

(define rust-cobs-0.2.3
  (crate-source "cobs" "0.2.3"
                "05gd16mws4yd63h8jr3p08in8y8w21rpjp5jb55hzl9bgalh5fk7"))

(define rust-cocoa-0.20.2
  (crate-source "cocoa" "0.20.2"
                "1y0wd1lyiz8cgbsf0fwyw06gb1akg6rvg5jr3wah8mvdqdpyhj8c"))

(define rust-cocoa-0.25.0
  (crate-source "cocoa" "0.25.0"
                "0g1pl5hq28arqmvsswf2ib7smj445miwa58qa7wrfvksz54h857n"))

(define rust-cocoa-foundation-0.1.2
  (crate-source "cocoa-foundation" "0.1.2"
                "1xwk1khdyqw3dwsl15vr8p86shdcn544fr60ass8biz4nb5k8qlc"))

(define rust-codemap-0.1.3
  (crate-source "codemap" "0.1.3"
                "091azkslwkcijj3lp9ymb084y9a0wm4fkil7m613ja68r2snkrxr"))

(define rust-codepage-0.1.2
  (crate-source "codepage" "0.1.2"
                "1d0qr4wqc4yrab7halsa3r6akb2i2bk2cqr04vl8m0n23c38vxj8"))

(define rust-codespan-reporting-0.11.1
  (crate-source "codespan-reporting" "0.11.1"
                "0vkfay0aqk73d33kh79k1kqxx06ka22894xhqi89crnc6c6jff1m"))

(define rust-codespan-reporting-0.12.0
  (crate-source "codespan-reporting" "0.12.0"
                "108g41xqzhr8fx8hlpy5qzmqq8ylldbj37wndkaqm34yy1d2wvgy"))

(define rust-codex-0.1.1
  (crate-source "codex" "0.1.1"
                "0dj0hqw3wk5p3k77zi1sczds00d6mdwhwdb4w7jh1drqxsh2fkbj"))

(define rust-codspeed-2.10.0
  (crate-source "codspeed" "2.10.0"
                "1x9anwlfzlfby794d5fcvp214bj8bp29br8pkksxwb7834djja6j"))

(define rust-codspeed-criterion-compat-2.10.0
  (crate-source "codspeed-criterion-compat" "2.10.0"
                "1jj7n8q4f80sgyp66fax09p76bhbwvlgsndd21l0ahw1xdckmdji"))

(define rust-codspeed-criterion-compat-walltime-2.10.0
  (crate-source "codspeed-criterion-compat-walltime" "2.10.0"
                "0jpl1hwyqcay8cffqd66swaw0mllq8avbpxxk09cgrp8cgw1z649"))

(define rust-coitrees-0.2.1
  (crate-source "coitrees" "0.2.1"
                "1l2ybr8n02vm08wq9mrix7r07bgwm85i6fyachlm8d626w9w9d3f"))

(define rust-color-eyre-0.6.3
  (crate-source "color-eyre" "0.6.3"
                "1m9shifr9sdw0drszzyhvaq5jysrsiki44bl7m1gfdzj8rg6y52m"
                #:snippet '(for-each delete-file-recursively '("pictures" "scripts"))))

(define rust-color-eyre-0.6.5
  (crate-source "color-eyre" "0.6.5"
                "0vgjy8q6c4fin6inh7ik66rkkyjwar3ai8z33vbacckqnkphp4p5"
                #:snippet '(for-each delete-file-recursively '("pictures" "scripts"))))

(define rust-color-print-0.3.7
  (crate-source "color-print" "0.3.7"
                "1x5nrpwwl3n8qawdyywiawv4j6yrd6mxjiz04db7sy8334bm9a9s"))

(define rust-color-print-proc-macro-0.3.7
  (crate-source "color-print-proc-macro" "0.3.7"
                "08la26krj5n9rl2c69hk2j711d4yrrza9bjrbbj0fh75xfsqc8b9"))

(define rust-color-quant-1.1.0
  (crate-source "color_quant" "1.1.0"
                "12q1n427h2bbmmm1mnglr57jaz2dj9apk0plcxw7nwqiai7qjyrx"))

(define rust-color-spantrace-0.2.1
  (crate-source "color-spantrace" "0.2.1"
                "1hkjgaqixrishwiq3lxxy1d4c3mvlv6avcj3ifwy50p3lyrf2syd"
                #:snippet '(delete-file-recursively "pictures")))

(define rust-color-spantrace-0.3.0
  (crate-source "color-spantrace" "0.3.0"
                "09xl077fs44yvqajmr52mjbghvkfwg6fnd3jpialnd8kvylqxf5q"))

(define rust-colorchoice-1.0.0
  (crate-source "colorchoice" "1.0.0"
                "1ix7w85kwvyybwi2jdkl3yva2r2bvdcc3ka2grjfzfgrapqimgxc"))

(define rust-colorchoice-1.0.3
  (crate-source "colorchoice" "1.0.3"
                "1439m3r3jy3xqck8aa13q658visn71ki76qa93cy55wkmalwlqsv"))

(define rust-colorchoice-1.0.4
  (crate-source "colorchoice" "1.0.4"
                "0x8ymkz1xr77rcj1cfanhf416pc4v681gmkc9dzb3jqja7f62nxh"))

(define rust-colored-1.9.4
  (crate-source "colored" "1.9.4"
                "0mc302pm2x0vpmc3ni35w0666858pmqlqzbipyz42cw2j4f78pss"))

(define rust-colored-2.0.4
  (crate-source "colored" "2.0.4"
                "1xhnlnyv3am5xx0gw5bgrfh33d3p06x44v0yycn02f5w5x4fqx16"))

(define rust-colored-2.2.0
  (crate-source "colored" "2.2.0"
                "0g6s7j2qayjd7i3sivmwiawfdg8c8ldy0g2kl4vwk1yk16hjaxqi"))

(define rust-colored-3.0.0
  (crate-source "colored" "3.0.0"
                "0plizddhxc4vgkzdbzky5zggyaqfrmyim2d0n6sb7py9j3nf1q7x"))

(define rust-colorsys-0.6.7
  (crate-source "colorsys" "0.6.7"
                "1g8vwcv89n2dzi9bmbzqlj9cl9a89jz49668grbcncv4cjx1l9jl"))

(define rust-colorz-1.1.4
  (crate-source "colorz" "1.1.4"
                "0yq6wvrajh73b9hwjr03brc2znhr1x1nym6bd5ry68c8g72kgsvc"))

(define rust-combine-4.6.7
  (crate-source "combine" "4.6.7"
                "1z8rh8wp59gf8k23ar010phgs0wgf5i8cx4fg01gwcnzfn5k0nms"))

(define rust-comemo-0.4.0
  (crate-source "comemo" "0.4.0"
                "14bsiayib4lhz3jrbf1fqh2fpwsm6cii90mifym3jhvji901csfz"))

(define rust-comemo-macros-0.4.0
  (crate-source "comemo-macros" "0.4.0"
                "1nr8w81hkzg49s515v61shxb077iq6d6001pybxbvxdlz516x4y8"))

(define rust-comfy-table-7.1.4
  (crate-source "comfy-table" "7.1.4"
                "16hxb4pa404r5h7570p58h3yx684sqbshi79j1phn6gvqkzfnraa"))

(define rust-comfy-table-7.2.1
  (crate-source "comfy-table" "7.2.1"
                "0fvsxnnvps35dj2vdhy8wl0c17p96i0n67jmdjnzvcmlw2w7sfxh"))

(define rust-command-group-2.1.0
  (crate-source "command-group" "2.1.0"
                "05lr58pzkpw9rnl3ssi6pv72b9vvjc6z0208ng57djqf1xmxz02h"
                #:snippet '(delete-file-recursively "logo.png")))

(define rust-compact-str-0.7.1
  (crate-source "compact_str" "0.7.1"
                "0gvvfc2c6pg1rwr2w36ra4674w3lzwg97vq2v6k791w30169qszq"))

(define rust-compact-str-0.8.1
  (crate-source "compact_str" "0.8.1"
                "0cmgp61hw4fwaakhilwznfgncw2p4wkbvz6dw3i7ibbckh3c8y9v"))

(define rust-compose-spec-0.2.0
  (crate-source "compose_spec" "0.2.0"
                "17h8g1mrwk8yphqxfhdgyvn5hpagmm5p3p8p6yxh23hl3d6zdshi"))

(define rust-compose-spec-macros-0.1.0
  (crate-source "compose_spec_macros" "0.1.0"
                "0z4r9n38h5346ksfilkaxa76zzivdhvj9axq8yjm6wv9nhl1pf25"))

(define rust-comrak-0.26.0
  (crate-source "comrak" "0.26.0"
                "1a87w6c0xkfml7rg1v34rhlcknr811s11mi9xsjgazf58dwbcnir"))

(define rust-comrak-0.37.0
  (crate-source "comrak" "0.37.0"
                "1w9sb35gg7d8hb53klf75qdnbl8zz403dhgbggr0m8x97kkhakra"))

(define rust-concat-string-1.0.1
  (crate-source "concat-string" "1.0.1"
                "02c6hfxsvs1ff2j58f3qzr526w1yg8d2nf6yyjv81ixgbz5vwfbl"))

(define rust-concolor-control-0.0.7
  (crate-source "concolor-control" "0.0.7"
                "1nplakbdb9sbd3h62d9zkan0xm1w0c7cbl3rk0iqgn405yf1213i"))

(define rust-concolor-query-0.0.4
  (crate-source "concolor-query" "0.0.4"
                "1isbqpyiwblp0rglnaqzai5hav23095s82mwgi09v3xcck4rq5dd"))

(define rust-concurrent-queue-2.5.0
  (crate-source "concurrent-queue" "2.5.0"
                "0wrr3mzq2ijdkxwndhf79k952cp4zkz35ray8hvsxl96xrx1k82c"))

(define rust-condtype-1.3.0
  (crate-source "condtype" "1.3.0"
                "1by78npyhkc30jccc7kirvwip1fj0jhi2bwfmcw44dqz81xa1w5s"))

(define rust-config-0.14.1
  (crate-source "config" "0.14.1"
                "1pyzr665pq6xcq3f6j1670v5xxk465n2bbkzn9hzycradlcqymv8"))

(define rust-config-0.15.11
  (crate-source "config" "0.15.11"
                "107glr9a6qwh8l8hy83yrg1lla95633fh62qs29fffswwqhawnjr"))

(define rust-config-0.15.16
  (crate-source "config" "0.15.16"
                "0j8ws99c4a58pcnq4l2ckf85k9fcw8q8cmsm2pqsx6zrxkq3dw6f"))

(define rust-config-file-0.2.3
  (crate-source "config-file" "0.2.3"
                "1yys2088y6lnc959k1k78y0amjkp6a00pjybsk3x50872lnfflfz"))

(define rust-configparser-3.1.0
  (crate-source "configparser" "3.1.0"
                "16v47b7lknb35ragwhj9gzgwfpxs34vn2b97hhaky30ry1r34zp5"))

(define rust-conpty-0.5.1
  (crate-source "conpty" "0.5.1"
                "1nmkhiysnssvbi4kqaq8cybb0ffngbl64kfpk8s86ihdg940caxp"))

(define rust-console-0.13.0
  (crate-source "console" "0.13.0"
                "022ai0jbbawngrx396qppwgzk4pk3v2fdwckzamvz6h154jsn2m5"))

(define rust-console-0.15.11
  (crate-source "console" "0.15.11"
                "1n5gmsjk6isbnw6qss043377kln20lfwlmdk3vswpwpr21dwnk05"))

(define rust-console-0.16.1
  (crate-source "console" "0.16.1"
                "1x4x6vfi1s55nbr4i77b9r87s213h46lq396sij9fkmidqx78c5l"))

(define rust-console-api-0.8.1
  (crate-source "console-api" "0.8.1"
                "0mr8xkzcb7mrcm2ca0ah1sj21rhpk207klscnql884hdrdg76c40"))

(define rust-console-error-panic-hook-0.1.7
  (crate-source "console_error_panic_hook" "0.1.7"
                "1g5v8s0ndycc10mdn6igy914k645pgpcl8vjpz6nvxkhyirynsm0"
                #:snippet '(for-each delete-file (find-files "." "\\.png$"))))

(define rust-console-subscriber-0.4.1
  (crate-source "console-subscriber" "0.4.1"
                "00badn9pjwbfd985vc7ngl7ai6mc3q58c3q43i5izlscdafalfb5"))

(define rust-const-format-0.2.34
  (crate-source "const_format" "0.2.34"
                "1pb3vx4k0bl3cy45fmba36hzds1jhkr8y9k3j5nnvm4abjb9fvqj"))

(define rust-const-format-proc-macros-0.2.34
  (crate-source "const_format_proc_macros" "0.2.34"
                "0i3pxxcl4xvwq4mlfg3csb4j0n6v0mhj07p6yk0vlvdirznc4mqx"))

(define rust-const-oid-0.9.6
  (crate-source "const-oid" "0.9.6"
                "1y0jnqaq7p2wvspnx7qj76m7hjcqpz73qzvr9l2p9n2s51vr6if2"))

(define rust-const-random-0.1.18
  (crate-source "const-random" "0.1.18"
                "0n8kqz3y82ks8znvz1mxn3a9hadca3amzf33gmi6dc3lzs103q47"))

(define rust-const-random-macro-0.1.16
  (crate-source "const-random-macro" "0.1.16"
                "03iram4ijjjq9j5a7hbnmdngj8935wbsd0f5bm8yw2hblbr3kn7r"))

(define rust-const-str-0.3.2
  (crate-source "const-str" "0.3.2"
                "1wxay9fr0ydvand3vidzcrhaapvjq4mg8fhw8axh441afrr7f1r1"))

(define rust-const-str-proc-macro-0.3.2
  (crate-source "const-str-proc-macro" "0.3.2"
                "0np2206r7w3c7lg5rg06wg8bby7fmrc13cp163jl2c2x5vfhy7jy"))

(define rust-constant-time-eq-0.1.5
  (crate-source "constant_time_eq" "0.1.5"
                "1g3vp04qzmk6cpzrd19yci6a95m7ap6wy7wkwgiy2pjklklrfl14"))

(define rust-constant-time-eq-0.3.1
  (crate-source "constant_time_eq" "0.3.1"
                "19nwwczii762pwlsm7bpizgjg8hkg1kqi32b2g4rglijklsbhx3w"))

(define rust-container-of-0.5.1
  (crate-source "container_of" "0.5.1"
                "0as7g6gspvdbp4vl1a1834pzh481x9jp4clfgyl6c7vnhvmvpxc9"))

(define rust-content-inspector-0.2.4
  (crate-source "content_inspector" "0.2.4"
                "0f1gwv4axxw9wck4a4jxlkm7xjjakb3616isll2k0s4chmpadgdp"
                #:snippet '(delete-file-recursively "testdata")))

(define rust-convert-case-0.4.0
  (crate-source "convert_case" "0.4.0"
                "03jaf1wrsyqzcaah9jf8l1iznvdw5mlsca2qghhzr9w27sddaib2"))

(define rust-convert-case-0.6.0
  (crate-source "convert_case" "0.6.0"
                "1jn1pq6fp3rri88zyw6jlhwwgf6qiyc08d6gjv0qypgkl862n67c"))

(define rust-convert-case-0.7.1
  (crate-source "convert_case" "0.7.1"
                "1rzih8qbd3xh87wp76nkjvnrimn7vlzcwl2n88898ml59j6jnh5v"))

(define rust-convert-case-0.8.0
  (crate-source "convert_case" "0.8.0"
                "17zqy79xlr1n7nc0n1mlnw5qpp8l2nbxrk13jixrhlavrbna1ams"))

(define rust-cookie-0.16.2
  (crate-source "cookie" "0.16.2"
                "1yriqbf77iigrnp2gmf6m1r296bndv051dc1qc39w3bis1bwsng8"))

(define rust-cookie-0.17.0
  (crate-source "cookie" "0.17.0"
                "096c52jg9iq4lfcps2psncswv33fc30mmnaa2sbzzcfcw71kgyvy"))

(define rust-cookie-factory-0.3.2
  (crate-source "cookie-factory" "0.3.2"
                "0sqjmw85ckqhppff6gjwmvjpkii35441a51xx7cv0ih3jy2fjv9r"))

(define rust-cookie-factory-0.3.3
  (crate-source "cookie-factory" "0.3.3"
                "18mka6fk3843qq3jw1fdfvzyv05kx7kcmirfbs2vg2kbw9qzm1cq"))

(define rust-cookie-store-0.20.0
  (crate-source "cookie_store" "0.20.0"
                "1xkc7fl1jik9ki13j9pjgyw51d0qd613srz1lv1qb0blpjmn2x1q"))

(define rust-copy-dir-0.1.3
  (crate-source "copy_dir" "0.1.3"
                "18rckga6rf3nzlw8avyi6chdlid0kp7lhfjyy0pnw27g738isgal"))

(define rust-copyless-0.1.5
  (crate-source "copyless" "0.1.5"
                "0dmmxsq3m0i6g9s2kj96n777qhmm7vjgv4r7agc2v6w6bl7rdpx2"))

(define rust-copypasta-0.10.1
  (crate-source "copypasta" "0.10.1"
                "0n58sfbahs49ar0qivz1sqz1ixhh1ifgp5bzifjkvabwhqi59f6y"))

(define rust-core-affinity-0.8.3
  (crate-source "core_affinity" "0.8.3"
                "0hhkjybngi5n2ayjmbba2n2gh9fc8xbqgpzm2dp6q094nskv6d50"))

(define rust-core-foundation-0.10.0
  (crate-source "core-foundation" "0.10.0"
                "0qscay14s2rwkg8nd8ljhiaf149hj8sfy95d70zssy64r3jp2lmm"))

(define rust-core-foundation-0.10.1
  (crate-source "core-foundation" "0.10.1"
                "1xjns6dqf36rni2x9f47b65grxwdm20kwdg9lhmzdrrkwadcv9mj"))

(define rust-core-foundation-0.7.0
  (crate-source "core-foundation" "0.7.0"
                "0wbias8f0m5kyn2pcksi0h58fdslams6nmf16w78fgn42dx4rljp"))

(define rust-core-foundation-0.9.4
  (crate-source "core-foundation" "0.9.4"
                "13zvbbj07yk3b61b8fhwfzhy35535a583irf23vlcg59j7h9bqci"))

(define rust-core-foundation-sys-0.7.0
  (crate-source "core-foundation-sys" "0.7.0"
                "1b5qfnnmg49sawwfsb0c0wbj81bqi7h7lh68pmhbidf0jjs1m9xk"))

(define rust-core-foundation-sys-0.8.7
  (crate-source "core-foundation-sys" "0.8.7"
                "12w8j73lazxmr1z0h98hf3z623kl8ms7g07jch7n4p8f9nwlhdkp"))

(define rust-core-graphics-0.19.2
  (crate-source "core-graphics" "0.19.2"
                "08z9pgwfc0wb5v3ns7rnb2010q9g42b5vfwhp9fv4spawrs9725k"))

(define rust-core-graphics-0.23.2
  (crate-source "core-graphics" "0.23.2"
                "10dhv3gk4kmbzl14xxkrhhky4fdp8h6nzff6h0019qgr6nz84xy0"))

(define rust-core-graphics-0.24.0
  (crate-source "core-graphics" "0.24.0"
                "1w8n8gqqm8swkanaibilqya8ryldp9fvf80byjxsaprn493a75gs"))

(define rust-core-graphics-types-0.1.3
  (crate-source "core-graphics-types" "0.1.3"
                "1bxg8nxc8fk4kxnqyanhf36wq0zrjr552c58qy6733zn2ihhwfa5"))

(define rust-core-graphics-types-0.2.0
  (crate-source "core-graphics-types" "0.2.0"
                "1sqka1rz84lr3p69i1s6lggnpnznmrw4ngc5q76w9xhky80s2i1x"))

(define rust-core-maths-0.1.1
  (crate-source "core_maths" "0.1.1"
                "0c0dv11ixxpc9bsx5xasvl98mb1dlprzcm6qq6ls3nsygw0mwx3p"))

(define rust-core-text-20.1.0
  (crate-source "core-text" "20.1.0"
                "1mffma8w0ic11ydv6zclamw4dslzmsych1fwz14msih8bh5pkln9"))

(define rust-core-video-sys-0.1.4
  (crate-source "core-video-sys" "0.1.4"
                "0a1qbn50jrb5hxrfshyb7y0f3pbf4ily6i6nciv7bn8ac4isvv1l"))

(define rust-core2-0.4.0
  (crate-source "core2" "0.4.0"
                "01f5xv0kf3ds3xm7byg78hycbanb8zlpvsfv4j47y46n3bpsg6xl"))

(define rust-coreaudio-rs-0.11.3
  (crate-source "coreaudio-rs" "0.11.3"
                "1kmssby4rqhv2iq1a8zmaav5p3bl40qs0wah9zv65ikr5lbpf41j"))

(define rust-coreaudio-sys-0.2.16
  (crate-source "coreaudio-sys" "0.2.16"
                "0asnnypbsqzj2lxx4irnmyi9b32sl0vkxb61mj3p5mvp1fm5gs1c"))

(define rust-corosensei-0.1.4
  (crate-source "corosensei" "0.1.4"
                "13ksv3m4w94xv59h8h0y5ixlh94j8kn5k9yj878cpacfqlr8h4l0"))

(define rust-cpal-0.15.3
  (crate-source "cpal" "0.15.3"
                "0yd7d51kcf8ml0bfkjrac12zgfjzk21wa97maxg0fhzpr03sngc7"))

(define rust-cpp-demangle-0.4.4
  (crate-source "cpp_demangle" "0.4.4"
                "0z8c656jiwphnw1brkb0whm4kgh39h1msvgig2wc44yi58s8vrcn"))

(define rust-cpufeatures-0.2.17
  (crate-source "cpufeatures" "0.2.17"
                "10023dnnaghhdl70xcds12fsx2b966sxbxjq5sxs49mvxqw5ivar"))

(define rust-cranelift-bforest-0.116.1
  (crate-source "cranelift-bforest" "0.116.1"
                "1i13rpcpzi2b26nf1fajf2jcvmpzjgv8rmlav3m3djw6rsh08pg1"))

(define rust-cranelift-bforest-0.82.3
  (crate-source "cranelift-bforest" "0.82.3"
                "0x8gi908lmnzkniwizgjjbhjvzcbp4k4fyyk325fgj0ncshs5yiq"))

(define rust-cranelift-bitset-0.116.1
  (crate-source "cranelift-bitset" "0.116.1"
                "0d4zvspdxpw8jl1q2zm8n1jrxg6kqmkphjr4rrcp49nflxlkjvkw"))

(define rust-cranelift-codegen-0.116.1
  (crate-source "cranelift-codegen" "0.116.1"
                "0kihpw7gaj9wx4rw2x9lkn0x3nkcy93vn5pm3hvmh9dl9hn068ic"))

(define rust-cranelift-codegen-0.82.3
  (crate-source "cranelift-codegen" "0.82.3"
                "0fzkf4b8xn9n2fm1f9n7zhhqyqnw2w12nzfhmxh3i8iv593r5w96"))

(define rust-cranelift-codegen-meta-0.116.1
  (crate-source "cranelift-codegen-meta" "0.116.1"
                "1f70xrardmbpvfij1djkavllzrlxy9wi6jkzh9bzq9qvqrqvq169"))

(define rust-cranelift-codegen-meta-0.82.3
  (crate-source "cranelift-codegen-meta" "0.82.3"
                "090s1g28wsw28sx9bb3nb8irh716ldz7snvangcx72cvkswdschg"))

(define rust-cranelift-codegen-shared-0.116.1
  (crate-source "cranelift-codegen-shared" "0.116.1"
                "1src5xxbh4va2g0f1n5lgcfyrqhsk20589ccx1668bspjxa0y620"))

(define rust-cranelift-codegen-shared-0.82.3
  (crate-source "cranelift-codegen-shared" "0.82.3"
                "1g2hl52ksmxm147nm3n2i348y9vyn5vagdrkjcnv9hcc52ghvz81"))

(define rust-cranelift-control-0.116.1
  (crate-source "cranelift-control" "0.116.1"
                "1vymgmnqqnmirvkyqraah03hgnjr2wbzqaj7cfali2mxs3335l96"))

(define rust-cranelift-entity-0.116.1
  (crate-source "cranelift-entity" "0.116.1"
                "08xk00xzncxycxm0z1g6daysq2g0qs503pbryvxp8m1732b0sbab"))

(define rust-cranelift-entity-0.82.3
  (crate-source "cranelift-entity" "0.82.3"
                "1kw5j62fcyhs9nf7md8m6jlx1q2k0k39vbqzc027c5l35cbzwfwy"))

(define rust-cranelift-frontend-0.116.1
  (crate-source "cranelift-frontend" "0.116.1"
                "0msd90p3p5hs1bl687hf21k2h1zv60vrw32b2y9p6419czhl3b2g"))

(define rust-cranelift-frontend-0.82.3
  (crate-source "cranelift-frontend" "0.82.3"
                "1kkdw9n4sh053nwi18c4r2l55l36c3vdvygilx5hxkl05piy61m0"))

(define rust-cranelift-isle-0.116.1
  (crate-source "cranelift-isle" "0.116.1"
                "0kclhdj14l7av24k2h2hlrm4vkza5pyfzabj196h8w2hdrbhv8hw"))

(define rust-cranelift-native-0.116.1
  (crate-source "cranelift-native" "0.116.1"
                "1ivna3w02yxa1hs2kkcv6xj8gs8g6mgcrwbpj6x4qb0z7wpyipmq"))

(define rust-crates-io-0.40.10
  (crate-source "crates-io" "0.40.10"
                "1rhds7nh2si7zkg28yx67ipn9czymky05vaqx1dkzmdvy93bj5cw"))

(define rust-crates-io-0.40.9
  (crate-source "crates-io" "0.40.9"
                "0553wkxfgki90cb7qkzyb5f2z4lvqzwc5za8fzw5qj5wh7xbjrv9"))

(define rust-crc-1.8.1
  (crate-source "crc" "1.8.1"
                "1sqal6gm6lbj7f45iv3rw2s9w3pvvha8v970y51s7k7mwy6m8qyn"))

(define rust-crc-3.3.0
  (crate-source "crc" "3.3.0"
                "0xg6yg57lbyzf69y8znq5gjb333w1fnlis2gnjg38blwffrx644p"))

(define rust-crc-catalog-2.4.0
  (crate-source "crc-catalog" "2.4.0"
                "1xg7sz82w3nxp1jfn425fvn1clvbzb3zgblmxsyqpys0dckp9lqr"))

(define rust-crc32fast-1.4.2
  (crate-source "crc32fast" "1.4.2"
                "1czp7vif73b8xslr3c9yxysmh9ws2r8824qda7j47ffs9pcnjxx9"))

(define rust-crc32fast-1.5.0
  (crate-source "crc32fast" "1.5.0"
                "04d51liy8rbssra92p0qnwjw8i9rm9c4m3bwy19wjamz1k4w30cl"))

(define rust-criterion-0.3.6
  (crate-source "criterion" "0.3.6"
                "13yd64ah93gkbdv7qq4cr6rhgl9979jjcjk3gkhnav1b7glns7dh"))

(define rust-criterion-0.5.1
  (crate-source "criterion" "0.5.1"
                "0bv9ipygam3z8kk6k771gh9zi0j0lb9ir0xi1pc075ljg80jvcgj"))

(define rust-criterion-plot-0.4.5
  (crate-source "criterion-plot" "0.4.5"
                "0xhq0jz1603585h7xvm3s4x9irmifjliklszbzs4cda00y1cqwr6"))

(define rust-criterion-plot-0.5.0
  (crate-source "criterion-plot" "0.5.0"
                "1c866xkjqqhzg4cjvg01f8w6xc1j3j7s58rdksl52skq89iq4l3b"))

(define rust-critical-section-1.2.0
  (crate-source "critical-section" "1.2.0"
                "02ylhcykxjc40xrfhk1lwc21jqgz4dbwv3jr49ymw733c51yl3kr"))

(define rust-crossbeam-0.8.4
  (crate-source "crossbeam" "0.8.4"
                "1a5c7yacnk723x0hfycdbl91ks2nxhwbwy46b8y5vyy0gxzcsdqi"))

(define rust-crossbeam-channel-0.5.14
  (crate-source "crossbeam-channel" "0.5.14"
                "0wa41qybq5w8s70anb472myh4fid4aw6v65vws6wn528w9l6vfh6"))

(define rust-crossbeam-channel-0.5.15
  (crate-source "crossbeam-channel" "0.5.15"
                "1cicd9ins0fkpfgvz9vhz3m9rpkh6n8d3437c3wnfsdkd3wgif42"))

(define rust-crossbeam-channel-0.5.8
  (crate-source "crossbeam-channel" "0.5.8"
                "004jz4wxp9k26z657i7rsh9s7586dklx2c5aqf1n3w1dgzvjng53"))

(define rust-crossbeam-deque-0.8.3
  (crate-source "crossbeam-deque" "0.8.3"
                "1vqczbcild7nczh5z116w8w46z991kpjyw7qxkf24c14apwdcvyf"))

(define rust-crossbeam-deque-0.8.5
  (crate-source "crossbeam-deque" "0.8.5"
                "03bp38ljx4wj6vvy4fbhx41q8f585zyqix6pncz1mkz93z08qgv1"))

(define rust-crossbeam-deque-0.8.6
  (crate-source "crossbeam-deque" "0.8.6"
                "0l9f1saqp1gn5qy0rxvkmz4m6n7fc0b3dbm6q1r5pmgpnyvi3lcx"))

(define rust-crossbeam-epoch-0.9.15
  (crate-source "crossbeam-epoch" "0.9.15"
                "1ixwc3cq816wb8rlh3ix4jnybqbyyq4l61nwlx0mfm3ck0s148df"))

(define rust-crossbeam-epoch-0.9.18
  (crate-source "crossbeam-epoch" "0.9.18"
                "03j2np8llwf376m3fxqx859mgp9f83hj1w34153c7a9c7i5ar0jv"))

(define rust-crossbeam-queue-0.3.12
  (crate-source "crossbeam-queue" "0.3.12"
                "059igaxckccj6ndmg45d5yf7cm4ps46c18m21afq3pwiiz1bnn0g"))

(define rust-crossbeam-utils-0.8.16
  (crate-source "crossbeam-utils" "0.8.16"
                "153j0gikblz7n7qdvdi8pslhi008s1yp9cmny6vw07ad7pbb48js"))

(define rust-crossbeam-utils-0.8.20
  (crate-source "crossbeam-utils" "0.8.20"
                "100fksq5mm1n7zj242cclkw6yf7a4a8ix3lvpfkhxvdhbda9kv12"))

(define rust-crossbeam-utils-0.8.21
  (crate-source "crossbeam-utils" "0.8.21"
                "0a3aa2bmc8q35fb67432w16wvi54sfmb69rk9h5bhd18vw0c99fh"))

(define rust-crossfont-0.8.0
  (crate-source "crossfont" "0.8.0"
                "1g490ap42qm1258222azy70mmd6b16dk8lza0142bjgk438jhkn4"))

(define rust-crossterm-0.25.0
  (crate-source "crossterm" "0.25.0"
                "0rsbkkhdf61aipc06b7vpl4cw3wnxz0miizp0ms3a5rcpq7nqkp6"
                #:snippet '(delete-file-recursively "docs")))

(define rust-crossterm-0.27.0
  (crate-source "crossterm" "0.27.0"
                "1pr413ki440xgddlmkrc4j1bfx1h8rpmll87zn8ykja1bm2gwxpl"
                #:snippet '(delete-file-recursively "docs")))

(define rust-crossterm-0.28.1
  (crate-source "crossterm" "0.28.1"
                "1im9vs6fvkql0sr378dfr4wdm1rrkrvr22v4i8byz05k1dd9b7c2"
                #:snippet '(delete-file-recursively "docs")))

(define rust-crossterm-0.29.0
  (crate-source "crossterm" "0.29.0"
                "0yzqxxd90k7d2ac26xq1awsznsaq0qika2nv1ik3p0vzqvjg5ffq"
                #:snippet '(delete-file-recursively "docs")))

(define rust-crossterm-winapi-0.9.1
  (crate-source "crossterm_winapi" "0.9.1"
                "0axbfb2ykbwbpf1hmxwpawwfs8wvmkcka5m561l7yp36ldi7rpdc"))

(define rust-crosstermion-0.14.0
  (crate-source "crosstermion" "0.14.0"
                "1i7caxqlz174zj6pll85vp4avhhzzy00vfdivry3v5hq3phgzf1v"))

(define rust-crunchy-0.2.3
  (crate-source "crunchy" "0.2.3"
                "0aa9k4izp962qlsn5ndgw2zq62mizcpnkns8bxscgz3gqr35knj3"))

(define rust-crunchy-0.2.4
  (crate-source "crunchy" "0.2.4"
                "1mbp5navim2qr3x48lyvadqblcxc1dm0lqr0swrkkwy2qblvw3s6"))

(define rust-crypto-bigint-0.5.5
  (crate-source "crypto-bigint" "0.5.5"
                "0xmbdff3g6ii5sbxjxc31xfkv9lrmyril4arh3dzckd4gjsjzj8d"))

(define rust-crypto-common-0.1.6
  (crate-source "crypto-common" "0.1.6"
                "1cvby95a6xg7kxdz5ln3rl9xh66nz66w46mm3g56ri1z5x815yqv"))

(define rust-crypto-mac-0.11.1
  (crate-source "crypto-mac" "0.11.1"
                "05672ncc54h66vph42s0a42ljl69bwnqjh0x4xgj2v1395psildi"))

(define rust-crypto-mac-0.8.0
  (crate-source "crypto-mac" "0.8.0"
                "1axfs4zmy74rn9666p92j7nmcv11zdp2d51yrppc2dv26cqa715m"))

(define rust-cryptovec-0.6.1
  (crate-source "cryptovec" "0.6.1"
                "1pqb2g1n3sx0d2cjiy06amcr2wlf9izwb4jj68nk5cmvlq9zmiyc"))

(define rust-css-colors-1.0.1
  (crate-source "css-colors" "1.0.1"
                "0dljfdw4p54drjy9a5m6h5qnvz8lkdllxfkln0vk9wh8azybphi2"))

(define rust-csscolorparser-0.6.2
  (crate-source "csscolorparser" "0.6.2"
                "1gxh11hajx96mf5sd0az6mfsxdryfqvcfcphny3yfbfscqq7sapb"))

(define rust-csscolorparser-0.7.2
  (crate-source "csscolorparser" "0.7.2"
                "05izklll0zgin58119aw1akz47h7gn6lqzxj2yi3mvzvw6n6mnjz"))

(define rust-cssparser-0.27.2
  (crate-source "cssparser" "0.27.2"
                "02nbm690rmkaz1ca0383qq7mc1g066w3s85f17pdihnda79njjvm"))

(define rust-cssparser-0.31.2
  (crate-source "cssparser" "0.31.2"
                "1gnmn2wjvhvkj98ygcd5jdwi0wxsigvd7j0yq0zfgfsz7vwz8gav"))

(define rust-cssparser-0.33.0
  (crate-source "cssparser" "0.33.0"
                "07i8k47fmym7kzs95qfhg6zrh4yyf2vl4460rmdyvyx06vck9scv"))

(define rust-cssparser-0.34.0
  (crate-source "cssparser" "0.34.0"
                "1qx3hha392szcl812l6hp0d4029gg8x62cl4nf0byqgdv0f6vimp"))

(define rust-cssparser-0.35.0
  (crate-source "cssparser" "0.35.0"
                "1am2mj4rddlbmi08drk7gv9m8vw47zgicld48kwp451sfgfix42f"))

(define rust-cssparser-color-0.1.0
  (crate-source "cssparser-color" "0.1.0"
                "17qcjsrph1ywcdsx1ipqgmzaas4dbbir5djjmzbqjnfqc6d0jv2m"))

(define rust-cssparser-macros-0.6.1
  (crate-source "cssparser-macros" "0.6.1"
                "0cfkzj60avrnskdmaf7f8zw6pp3di4ylplk455zrzaf19ax8id8k"))

(define rust-cstr-0.2.12
  (crate-source "cstr" "0.2.12"
                "0dj6ll9ry27kn4k0vvhlvbhn9dyyr9haxnd06bxaqnmfr01kjlk8"))

(define rust-cstr-core-0.2.6
  (crate-source "cstr_core" "0.2.6"
                "0mh9lyzc2iyz2wzzhj1jhxyy3384wg19s8db1ka35a6w9wp7966x"))

(define rust-csv-1.3.1
  (crate-source "csv" "1.3.1"
                "1bzxgbbhy27flcyafxbj7f1hbn7b8wac04ijfgj34ry9m61lip5c"))

(define rust-csv-core-0.1.12
  (crate-source "csv-core" "0.1.12"
                "0gfrjjlfagarhyclxrqv6b14iaxgvgc8kmwwdvw08racvaqg60kx"))

(define rust-csv-index-0.1.6
  (crate-source "csv-index" "0.1.6"
                "01048y84y0bakqm0x4y1svjv6lzc753b9q598xp7xgcqrdgi6x7j"))

(define rust-ct-codecs-1.1.3
  (crate-source "ct-codecs" "1.1.3"
                "191f2id5zqv5hjm0nsblbwq1n276ba55w0bgi6b2c674x66bl5mr"))

(define rust-ct-codecs-1.1.6
  (crate-source "ct-codecs" "1.1.6"
                "1a1al3jf0vx3zmzg3s2x6kzq1l6ghkw153zkz5hhsh2y3afmh44v"))

(define rust-ctor-0.1.22
  (crate-source "ctor" "0.1.22"
                "1712zdlrmcfkfw38dkj67xg72fd0p9slyqqi64c6n94zgi7vwxzq"))

(define rust-ctor-0.1.26
  (crate-source "ctor" "0.1.26"
                "15m0wqhv12p25xkxz5dxvg23r7a6bkh7p8zi1cdhgswjhdl028vd"))

(define rust-ctor-0.2.9
  (crate-source "ctor" "0.2.9"
                "00b5vprqi4a2cr29xhqijg800b4dwkhrr5wj2kf3s7vnambpi8ij"))

(define rust-ctr-0.8.0
  (crate-source "ctr" "0.8.0"
                "1sk1aykwhkny92cnvl6s75dx3fyvfzw5xkd6xz3y7w5anhgvk6q4"
                #:snippet '(delete-file-recursively "tests")))

(define rust-ctr-0.9.2
  (crate-source "ctr" "0.9.2"
                "0d88b73waamgpfjdml78icxz45d95q7vi2aqa604b0visqdfws83"
                #:snippet '(delete-file-recursively "tests")))

(define rust-ctrlc-3.2.0
  (crate-source "ctrlc" "3.2.0"
                "1l82dkflbai4dslq6djy055fmpsbhvry4qlcl70v583j5809nz1p"))

(define rust-ctrlc-3.4.6
  (crate-source "ctrlc" "3.4.6"
                "0735llsx3zwqvqbwfikz3j4aa03c075ii04f8zi5mza8yccm8yv9"))

(define rust-ctrlc-3.4.7
  (crate-source "ctrlc" "3.4.7"
                "0wvf4w2wbpdnhp828jqw435x5ly4k7k1y1vzxxbdddsrlj03gya6"))

(define rust-ctrlc-3.5.0
  (crate-source "ctrlc" "3.5.0"
                "1wy41p6lskx853xph4sb5lahwfbqvax2rs064f74kwdj2c55s748"))

(define rust-cty-0.2.2
  (crate-source "cty" "0.2.2"
                "0d8z0pbr87wgzqqb2jk5pvj0afzc6d3rb772ach6fijhg6yglrdk"))

(define rust-culpa-1.0.2
  (crate-source "culpa" "1.0.2"
                "1lqzzl88vs3h6ljpj6gi8wgkwyqm9gp6dmsnbbzv873v67lvzq2s"))

(define rust-culpa-macros-1.0.2
  (crate-source "culpa-macros" "1.0.2"
                "0ya207axxrjixibdnqfk0h9f96b2b1xmxdw9rwfwglv6f1qy2d0j"))

(define rust-curl-0.4.47
  (crate-source "curl" "0.4.47"
                "0rcjdrl35xs8l5v3wv6q5z37hjw3r5bvmbb09pqmhaxyl49lvyyr"))

(define rust-curl-sys-0.4.80+curl-8.12.1
  (crate-source "curl-sys" "0.4.80+curl-8.12.1"
                "0d7ppx4kq77hc5nyff6jydmfabpgd0i3ppjvn8x0q833mhpdzxsm"
                #:snippet '(delete-file-recursively "curl")))

(define rust-curl-sys-0.4.83+curl-8.15.0
  (crate-source "curl-sys" "0.4.83+curl-8.15.0"
                "10xl373idbap4rck2ym19dy3ysnlg544cak3h86b2z820krxlc2q"
                #:snippet '(delete-file-recursively "curl")))

(define rust-cursor-icon-1.1.0
  (crate-source "cursor-icon" "1.1.0"
                "14brf4vd6az9hnszwzqj7xyfaymqx9806d4i7xmwlaja3wjsr9ln"))

(define rust-cursor-icon-1.2.0
  (crate-source "cursor-icon" "1.2.0"
                "0bvkw7ak1mqwcpkgd9lh7n00hcvlh87jfl7188f231nz6zfy2ypj"))

(define rust-curve25519-dalek-3.2.1
  (crate-source "curve25519-dalek" "3.2.1"
                "1h0vcl8p4syvci9zxkn3h80h06xv1fyqgcrfwrv0lnbzjr9d1ych"
                #:snippet '(delete-file-recursively "docs")))

(define rust-curve25519-dalek-4.1.3
  (crate-source "curve25519-dalek" "4.1.3"
                "1gmjb9dsknrr8lypmhkyjd67p1arb8mbfamlwxm7vph38my8pywp"))

(define rust-curve25519-dalek-derive-0.1.1
  (crate-source "curve25519-dalek-derive" "0.1.1"
                "1cry71xxrr0mcy5my3fb502cwfxy6822k4pm19cwrilrg7hq4s7l"))

(define rust-custom-derive-0.1.7
  (crate-source "custom_derive" "0.1.7"
                "1f81bavw1wnykwh21hh4yyzigs6zl6f6pkk9p3car8kq95yfb2pg"))

(define rust-cvss-2.0.0
  (crate-source "cvss" "2.0.0"
                "03q1nh4jy0cvgckji1jr1kz3j7gf2zg74240j8qi1qxhk7vs5iky"))

(define rust-cvt-0.1.2
  (crate-source "cvt" "0.1.2"
                "1wdwg2rbjl86bcrymscl34pw31kyv1ada19jazpkjbdzgzvrpbnj"))

(define rust-cxx-1.0.130
  (crate-source "cxx" "1.0.130"
                "0hmrs458jq5fr5i3ydv27pvyyg553r6q6cln55azzajqpah45h13"))

(define rust-cxx-1.0.156
  (crate-source "cxx" "1.0.156"
                "0krizdz7f45s8d3rrvj3aq7w1v1bzc27py52wln6vpgkqhpj0fma"
                #:snippet '(delete-file-recursively "book")))

(define rust-cxx-1.0.186
  (crate-source "cxx" "1.0.186"
                "1zm6j158khka4ya6lrcl2gxacz8r5ncl66qshxi5rp7mybklz72f"
                #:snippet '(delete-file-recursively "book")))

(define rust-cxx-build-1.0.130
  (crate-source "cxx-build" "1.0.130"
                "19nf1g54g2siv3bh638cqgrsib5j9x7nq7wvi98mgb7xs241rp25"))

(define rust-cxx-build-1.0.156
  (crate-source "cxx-build" "1.0.156"
                "1q43i005bfm16r98p1np0airbfbsff7ava3n7xw2aqsgyd3dyjv4"))

(define rust-cxx-build-1.0.186
  (crate-source "cxx-build" "1.0.186"
                "1p96ic1nh4a2hb46kzsz0dzqlq624zy7d1i0vlyn703xsc4jkkxm"))

(define rust-cxxbridge-cmd-1.0.156
  (crate-source "cxxbridge-cmd" "1.0.156"
                "0fbf9ahj3s2xyg304kdr1d58in19djg16xlbl2sd0kmppjzfz34f"))

(define rust-cxxbridge-cmd-1.0.186
  (crate-source "cxxbridge-cmd" "1.0.186"
                "1zffz85s2kdrqiy0fxvc15zsyl6xdnlxm3r0hdzx5n5zsgimwzq7"))

(define rust-cxxbridge-flags-1.0.130
  (crate-source "cxxbridge-flags" "1.0.130"
                "0bxmbgsdsv7hb7myc2702g4n6fl0i76jrg2mrj92b7i81g9xfzma"))

(define rust-cxxbridge-flags-1.0.156
  (crate-source "cxxbridge-flags" "1.0.156"
                "0vjr4g0r5hmbbmahdszqr71k10q6394j3xhqss4p5wp2cbzkwkk0"))

(define rust-cxxbridge-flags-1.0.186
  (crate-source "cxxbridge-flags" "1.0.186"
                "0r185y7pd6kydm7q60jin6lvraxxzkgszpqmx7rbj9aic548f45h"))

(define rust-cxxbridge-macro-1.0.130
  (crate-source "cxxbridge-macro" "1.0.130"
                "13fpx8c1bj4w1lpfscin4rdh09lsfkyaac26q18vhiny49fld30b"))

(define rust-cxxbridge-macro-1.0.156
  (crate-source "cxxbridge-macro" "1.0.156"
                "18c9nk2q6b9yyk0zmpz34a2fgy30mzwjz3zhvgg1b70xa02kl30k"))

(define rust-cxxbridge-macro-1.0.186
  (crate-source "cxxbridge-macro" "1.0.186"
                "14488wmi1zl2835hpcw417g4nw0937b62sgs5i3drf7r3rl9ds76"))

(define rust-cypheraddr-0.4.0
  (crate-source "cypheraddr" "0.4.0"
                "1n9hlw1jg3qn6gi6b4kpnjkw3aqjbnvp352ihc9r9faamp958p5s"))

(define rust-cyphergraphy-0.3.0
  (crate-source "cyphergraphy" "0.3.0"
                "1111vaq7adf68abf1nnch8vflh05fy7gv0xbgasxmp2xxz41cz5n"))

(define rust-cyphernet-0.5.2
  (crate-source "cyphernet" "0.5.2"
                "1qb25qyp6lz2h8rh8radxj6bw1y7d4984sf65j01syjai1lr755c"))

(define rust-dactyl-0.9.1
  (crate-source "dactyl" "0.9.1"
                "1r5a8mb5pqs7cgmsciqzxj2f7slrkaf2yly2frg387dkf65jqkca"))

(define rust-daemonize-0.5.0
  (crate-source "daemonize" "0.5.0"
                "0vhikx85f85r46xghsb4avsv6ww8mz9lipqvsia7m21wrfmgv2xb"))

(define rust-darling-0.10.2
  (crate-source "darling" "0.10.2"
                "0n7qsp6854wm3y1q1lvylhv15zvc87ibbac1nyfmcdbyv1snww0d"))

(define rust-darling-0.13.4
  (crate-source "darling" "0.13.4"
                "0g25pad4mhq7315mw9n4wpg8j3mwyhwvr541kgdl0aar1j2ra7d0"))

(define rust-darling-0.14.4
  (crate-source "darling" "0.14.4"
                "0l1qrn805bsxa0iy7x8bmdwr8c10hlw0yiqs8ckv7lbz86rhqxbv"))

(define rust-darling-0.20.11
  (crate-source "darling" "0.20.11"
                "1vmlphlrlw4f50z16p4bc9p5qwdni1ba95qmxfrrmzs6dh8lczzw"))

(define rust-darling-0.21.3
  (crate-source "darling" "0.21.3"
                "1h281ah78pz05450r71h3gwm2n24hy8yngbz58g426l4j1q37pww"))

(define rust-darling-0.9.0
  (crate-source "darling" "0.9.0"
                "1mnksf0i47pb7sxvi1iqfwmqy9iny0x8w56ilybpb431b46cpyzw"))

(define rust-darling-core-0.10.2
  (crate-source "darling_core" "0.10.2"
                "16sija1jv0l754x4aa6b6fy01d1kf8m0r4id3flqipm45np61jgh"))

(define rust-darling-core-0.13.4
  (crate-source "darling_core" "0.13.4"
                "046n83f9jpszlngpjxkqi39ayzxf5a35q673c69jr1dn0ylnb7c5"))

(define rust-darling-core-0.14.4
  (crate-source "darling_core" "0.14.4"
                "1w4b2ndxmkwghwq84yphk8x15jnpivm08w596g12ry5pwsk1r70h"))

(define rust-darling-core-0.20.11
  (crate-source "darling_core" "0.20.11"
                "0bj1af6xl4ablnqbgn827m43b8fiicgv180749f5cphqdmcvj00d"))

(define rust-darling-core-0.21.3
  (crate-source "darling_core" "0.21.3"
                "193ya45qgac0a4siwghk0bl8im8h89p3cald7kw8ag3yrmg1jiqj"))

(define rust-darling-core-0.9.0
  (crate-source "darling_core" "0.9.0"
                "0b201dx4m58l5ac7gmbjvbf4z2xipnk5d4pqa7mz7gy3f21h3z3a"))

(define rust-darling-macro-0.10.2
  (crate-source "darling_macro" "0.10.2"
                "0wlv31cxkrjijz5gv13hvk55c9lmd781aj12c8n84sa9mksa5dfr"))

(define rust-darling-macro-0.13.4
  (crate-source "darling_macro" "0.13.4"
                "0d8q8ibmsb1yzby6vwgh2wx892jqqfv9clwhpm19rprvz1wjd5ww"))

(define rust-darling-macro-0.14.4
  (crate-source "darling_macro" "0.14.4"
                "13mlyd5w275c815k0ijf6g4c446hs8b3m2h4an5isqgpr7dv9am4"))

(define rust-darling-macro-0.20.11
  (crate-source "darling_macro" "0.20.11"
                "1bbfbc2px6sj1pqqq97bgqn6c8xdnb2fmz66f7f40nrqrcybjd7w"))

(define rust-darling-macro-0.21.3
  (crate-source "darling_macro" "0.21.3"
                "10ac85n4lnx3rmf5rw8lijl2c0sbl6ghcpgfmzh0s26ihbghi0yk"))

(define rust-darling-macro-0.9.0
  (crate-source "darling_macro" "0.9.0"
                "1lcq9418w0vmvncg4a3n9k64zjvqz0048aviqi0rmlpiqv0xmn66"))

(define rust-dary-heap-0.3.7
  (crate-source "dary_heap" "0.3.7"
                "0a5p4qfayfmkda4rb31zyc5sia11n0q0cq6scznm9x5r32fcvlh4"))

(define rust-dashmap-4.0.2
  (crate-source "dashmap" "4.0.2"
                "1773x18k5m2zw1iyibs8l3wl1p1aijdbrc0w844xys06inr46yp7"))

(define rust-dashmap-5.5.3
  (crate-source "dashmap" "5.5.3"
                "0miqnlxi501vfbv6mw5jbmzgnj0wjrch3p4abvpd59s9v30lg1wp"))

(define rust-dashmap-6.1.0
  (crate-source "dashmap" "6.1.0"
                "1kvnw859xvrqyd1lk89na6797yvl5bri4wi9j0viz2a4j54wqhah"))

(define rust-dasp-sample-0.11.0
  (crate-source "dasp_sample" "0.11.0"
                "0zzw35akm3qs2rixbmlijk6h0l4g9ry6g74qc59zv1q8vs1f31qc"))

(define rust-data-encoding-2.8.0
  (crate-source "data-encoding" "2.8.0"
                "0470yf5ly1ibzmwygyjqg9ii9njbsha3xr5qj5dxyf2psbgpapsp"))

(define rust-data-encoding-2.9.0
  (crate-source "data-encoding" "2.9.0"
                "0xm46371aw613ghc12ay4vsnn49hpcmcwlijnqy8lbp2bpd308ra"))

(define rust-data-encoding-macro-0.1.18
  (crate-source "data-encoding-macro" "0.1.18"
                "0gb7xr2a7gg62bfmkg32kvcwhnmx24v6i1d1488z00h1xab6rkj7"))

(define rust-data-encoding-macro-internal-0.1.16
  (crate-source "data-encoding-macro-internal" "0.1.16"
                "0xhrbiib486iggdzlcmpi13xvd7dqdd4zacdi940b6d6vbp2n5ld"))

(define rust-data-url-0.1.1
  (crate-source "data-url" "0.1.1"
                "14z15yiyklp5dv0k0q6pd83irrn0y8hj9y3fj17akkrbf37byc1s"))

(define rust-data-url-0.3.1
  (crate-source "data-url" "0.3.1"
                "0ahclz72myi350cs1xcsxdh1v0iljpfj4ghcy2fy46mpfhf7laaw"))

(define rust-datatest-stable-0.3.2
  (crate-source "datatest-stable" "0.3.2"
                "109r4iaib8w01wn3vn4x035fzr2y7d78dhl0k5rs6c9h80ybpsqr"))

(define rust-dateparser-0.1.8
  (crate-source "dateparser" "0.1.8"
                "1i00pd2spxsz1d89n0ssf79q1w8ijx0p0rybx5d9z8lpzqhjmqf9"))

(define rust-datetime-0.5.2
  (crate-source "datetime" "0.5.2"
                "1g2lhc2ay46njrnnyc0729ibyzkpv3r3d4g0h3gzwmrygykzghs4"))

(define rust-dbl-0.3.2
  (crate-source "dbl" "0.3.2"
                "1ng33ncai537xlxfy9r4v24parm9mf7zwiwkixq7d0qmj6kka9xx"))

(define rust-dbus-0.9.7
  (crate-source "dbus" "0.9.7"
                "06vdv4aarjs4w6byg9nqajr67c8qvlhk3153ic2i65pvp63ikchv"))

(define rust-dbus-crossroads-0.5.2
  (crate-source "dbus-crossroads" "0.5.2"
                "1q3dyywazr3hppm052fa8q2366q66ml789r42jjlnm47f51q6k1s"))

(define rust-dbus-tokio-0.7.6
  (crate-source "dbus-tokio" "0.7.6"
                "04xd3z2dnjv4d45kj3wqnwbnwllrp1zsg8v3q0qp2rxwb7a8hxh0"))

(define rust-deadpool-0.10.0
  (crate-source "deadpool" "0.10.0"
                "145lq79dlc4jn7jvlcf4lb105bs3z3jy6g7d15zv7iy1g04i117v"))

(define rust-deadpool-runtime-0.1.4
  (crate-source "deadpool-runtime" "0.1.4"
                "0arbchl5j887hcfvjy4gq38d32055s5cf7pkpmwn0lfw3ss6ca89"))

(define rust-debounced-0.2.0
  (crate-source "debounced" "0.2.0"
                "1irwf11jxagabw98233dbznzap72r8sgdvjk3rsijg0nnpcmqzhh"))

(define rust-debug-ignore-1.0.5
  (crate-source "debug-ignore" "1.0.5"
                "08gwdny6124ggy4hyli92hdyiqc5j2z9lqhbw81k0mgljcfyvrzz"))

(define rust-debug-unreachable-0.1.1
  (crate-source "debug_unreachable" "0.1.1"
                "1cx4vh1d0bxdpb8l9lyiv03az4527lykxy39s4a958swf2n2w0ws"))

(define rust-debugid-0.8.0
  (crate-source "debugid" "0.8.0"
                "13f15dfvn07fa7087pmacixqqv0lmj4hv93biw4ldr48ypk55xdy"))

(define rust-deelevate-0.2.0
  (crate-source "deelevate" "0.2.0"
                "0kj2kcqv47l3iacpamgzbn742yf9d09h0xgwbadxs1l9qkw9fwqw"))

(define rust-defer-drop-1.3.0
  (crate-source "defer-drop" "1.3.0"
                "0mswjjksrr6fvsgxvp64mzwkjkzjmpwjfaw4n76jhsvalsgyq4zn"))

(define rust-defer-heavy-0.1.0
  (crate-source "defer-heavy" "0.1.0"
                "1qd00mmhj43ic47mphdmjwc561jbxfvh6mdmrxwmj3kc7qg6jlsh"))

(define rust-deflate64-0.1.9
  (crate-source "deflate64" "0.1.9"
                "06scix17pa7wzzfsnhkycpcc6s04shs49cdaxx2k1sl0226jnsfs"))

(define rust-defmt-decoder-0.3.11
  (crate-source "defmt-decoder" "0.3.11"
                "1kyfqw49a8ivdl9g7lan4mnlajjdfwgljq5s01vva0r237r59fhn"))

(define rust-defmt-json-schema-0.1.0
  (crate-source "defmt-json-schema" "0.1.0"
                "0smd5mxzk3b6j82ykb0fzk34f6xl1fcchsyqhprir9jpiqi4vc4n"))

(define rust-defmt-parser-0.3.4
  (crate-source "defmt-parser" "0.3.4"
                "03zpg0i6vlalw7m976z66n70s041rvwii8qn3grxgs1hwgpmyjpz"))

(define rust-deku-0.16.0
  (crate-source "deku" "0.16.0"
                "10a4x39p348s8hb7292sy9i33zd5h0wrpr9qzkispcq5gb68g6w1"))

(define rust-deku-derive-0.16.0
  (crate-source "deku_derive" "0.16.0"
                "17h0wn3xipgdrg312dd2z0aaks6pfv3xfjm7aaiib4i3f8js2b2f"))

(define rust-delharc-0.6.1
  (crate-source "delharc" "0.6.1"
                "18g5haj6bj92azif4jifhdy9vrv6blg3wyvpmxslh2gm2wkbm4qw"))

(define rust-deltae-0.3.2
  (crate-source "deltae" "0.3.2"
                "1d3hw9hpvicl9x0x34jr2ybjk5g5ym1lhbyz6zj31110gq8zaaap"
                #:snippet '(delete-file-recursively "docs")))

(define rust-der-0.7.10
  (crate-source "der" "0.7.10"
                "1jyxacyxdx6mxbkfw99jz59dzvcd9k17rq01a7xvn1dr6wl87hg7"
                #:snippet '(delete-file-recursively "tests")))

(define rust-der-0.7.9
  (crate-source "der" "0.7.9"
                "1h4vzjfa1lczxdf8avfj9qlwh1qianqlxdy1g5rn762qnvkzhnzm"
                #:snippet '(delete-file-recursively "tests")))

(define rust-deranged-0.4.0
  (crate-source "deranged" "0.4.0"
                "13h6skwk411wzhf1l9l7d3yz5y6vg9d7s3dwhhb4a942r88nm7lw"))

(define rust-deranged-0.4.1
  (crate-source "deranged" "0.4.1"
                "0n7hswnz5jz1rjy6zr8sc9awbszkmv1345hphccawj40w1larkr8"))

(define rust-deranged-0.5.3
  (crate-source "deranged" "0.5.3"
                "1k473y8lzjac956dm3s1cs2rz364py9zd52y9fkbanws8b6vqc6n"))

(define rust-derivative-2.2.0
  (crate-source "derivative" "2.2.0"
                "02vpb81wisk2zh1d5f44szzxamzinqgq2k8ydrfjj2wwkrgdvhzw"))

(define rust-derive-arbitrary-1.3.0
  (crate-source "derive_arbitrary" "1.3.0"
                "1dy8pmv7d8diqhbh1b88v8q5g0inwkgb465877jqimbjqjgfpkgk"))

(define rust-derive-arbitrary-1.4.1
  (crate-source "derive_arbitrary" "1.4.1"
                "000839h4mbgs65x1f8540kbjk2ifw68c4d8r5b9f7q0jv4d2qm1h"))

(define rust-derive-builder-0.11.2
  (crate-source "derive_builder" "0.11.2"
                "18q4jx8zvg2pkx2d1nmlklx8m48bwrghyg8rdg1irdwkw5xxyynh"))

(define rust-derive-builder-0.20.2
  (crate-source "derive_builder" "0.20.2"
                "0is9z7v3kznziqsxa5jqji3ja6ay9wzravppzhcaczwbx84znzah"))

(define rust-derive-builder-core-0.11.2
  (crate-source "derive_builder_core" "0.11.2"
                "1i5gmf5lglbg7agj1khc6k9swf1clfs5fg6w0icw1w91m77x948z"))

(define rust-derive-builder-core-0.20.2
  (crate-source "derive_builder_core" "0.20.2"
                "1s640r6q46c2iiz25sgvxw3lk6b6v5y8hwylng7kas2d09xwynrd"))

(define rust-derive-builder-macro-0.11.2
  (crate-source "derive_builder_macro" "0.11.2"
                "0s6xfgsybd9wbk39hbgqjcn7d1l36a33q6v7d0x5y17d5fvi80wg"))

(define rust-derive-builder-macro-0.20.2
  (crate-source "derive_builder_macro" "0.20.2"
                "0g1zznpqrmvjlp2w7p0jzsjvpmw5rvdag0rfyypjhnadpzib0qxb"))

(define rust-derive-more-0.99.19
  (crate-source "derive_more" "0.99.19"
                "17y6g78dg31fsv7z4p455bzxs670spg476ww2ibg3mj3vww9m8ix"))

(define rust-derive-more-0.99.20
  (crate-source "derive_more" "0.99.20"
                "0zvz94kbc5d4r817wni1l7xk8f289nhf73vqk677p5rxlij4pnvf"))

(define rust-derive-more-2.0.1
  (crate-source "derive_more" "2.0.1"
                "0y3n97cc7rsvgnj211p92y1ppzh6jzvq5kvk6340ghkhfp7l4ch9"))

(define rust-derive-more-impl-2.0.1
  (crate-source "derive_more-impl" "2.0.1"
                "1wqxcb7d5lzvpplz9szp4rwy1r23f5wmixz0zd2vcjscqknji9mx"))

(define rust-derive-new-0.5.9
  (crate-source "derive-new" "0.5.9"
                "0d9m5kcj1rdmdjqfgj7rxxhdzx0as7p4rp1mjx5j6w5dl2f3461l"))

(define rust-derive-new-0.6.0
  (crate-source "derive-new" "0.6.0"
                "1b8jv6jx0b8jgkz9kmz0ciqmnf74xkk0mmvkb5z1c87932kdwl6i"))

(define rust-derive-new-0.7.0
  (crate-source "derive-new" "0.7.0"
                "1k5y4lrj182n86zjb3q0rkajgfqazamjzinsi7prw616yi88vp1c"))

(define rust-derive-where-1.2.7
  (crate-source "derive-where" "1.2.7"
                "09cx25r1zb8xrsgd9cj9j5zmg30n7lnvcmspqymyn9d887673mk2"))

(define rust-derive-where-1.4.0
  (crate-source "derive-where" "1.4.0"
                "00kr3npm8n8fgpbqd5glmfghq84lk2ik92d2vi0sqg5xsj92cgz7"))

(define rust-derive-where-1.6.0
  (crate-source "derive-where" "1.6.0"
                "13sa6vk8fqljf163s3sc5agirbq0c2n6jj1p82ds2p6ifzniv57g"))

(define rust-des-0.8.1
  (crate-source "des" "0.8.1"
                "07kshslxanmg0g6007scvglfhg6mli2a8qzhx4kxx4z9ik781pgz"
                #:snippet '(delete-file-recursively "tests")))

(define rust-descape-2.0.3
  (crate-source "descape" "2.0.3"
                "0c56nbjgnba9s4lmfq2b89mrifkzwfc48hh7n5iwk06z12wi64bw"))

(define rust-detect-newline-style-0.1.2
  (crate-source "detect-newline-style" "0.1.2"
                "0j9pcjk2ab21f36fqybz69whd1c4xy60hy7qd5v59aqm6rfg490i"))

(define rust-deunicode-1.6.1 rust-deunicode-1)

(define rust-deunicode-1.6.2 rust-deunicode-1)

(define rust-devicons-0.6.12
  (crate-source "devicons" "0.6.12"
                "0jwh0g72rfkpbsm16rxb47y3ylmr47wwx3cmbbflzkrhygi4f3l3"))

(define rust-dhcp4r-0.2.3
  (crate-source "dhcp4r" "0.2.3"
                "0f5v6a9bqx5z3gnrvh5066p4fzksg4k3gldx9d8rm2q76msckaxg"))

(define rust-dhcproto-0.9.0
  (crate-source "dhcproto" "0.9.0"
                "1hsbl77lxvxa94ihn2vna1sx2icrkch427w24a883xymhm9h9vnw"))

(define rust-dhcproto-macros-0.1.0
  (crate-source "dhcproto-macros" "0.1.0"
                "0w61gvnm17frkm6mk8q4xkwd1iji946lj5f175w585h4hvxkx6d7"))

(define rust-dialoguer-0.10.4
  (crate-source "dialoguer" "0.10.4"
                "11rgzrhi677w9gf1r3ip2x361svdkjkr2m5dsfca9fcljacg5ijr"))

(define rust-dialoguer-0.11.0
  (crate-source "dialoguer" "0.11.0"
                "1pl0744wwr97kp8qnaybzgrfwk66qakzq0i1qrxl03vpbn0cx2v5"))

(define rust-diesel-2.2.9
  (crate-source "diesel" "2.2.9"
                "0lsid456rv1bj7g1aai4sh33082yfxzb8qk12886jfmsj039blrl"))

(define rust-diesel-derives-2.2.4
  (crate-source "diesel_derives" "2.2.4"
                "0mddq3ha8v8yz6vp2w0pinfrsnc0078p7zw784xsdgkh9cjmhfd9"))

(define rust-diesel-migrations-2.2.0
  (crate-source "diesel_migrations" "2.2.0"
                "1xn12ny9m1ci74iqpvhcfyhapr6wj56k3wxz07q32hmd9dqcwwwa"))

(define rust-diesel-table-macro-syntax-0.2.0
  (crate-source "diesel_table_macro_syntax" "0.2.0"
                "09gvkyljhchbxfkxlkkrdcqcmcxwsim9sfljqilbq4x485b77710"))

(define rust-diff-0.1.13
  (crate-source "diff" "0.1.13"
                "1j0nzjxci2zqx63hdcihkp0a4dkdmzxd7my4m7zk6cjyfy34j9an"))

(define rust-difference-2.0.0
  (crate-source "difference" "2.0.0"
                "1621wx4k8h452p6xzmzzvm7mz87kxh4yqz0kzxfjj9xmjxlbyk2j"
                #:snippet '(delete-file-recursively "assets")))

(define rust-difflib-0.4.0
  (crate-source "difflib" "0.4.0"
                "1s7byq4d7jgf2hcp2lcqxi2piqwl8xqlharfbi8kf90n8csy7131"))

(define rust-diffs-0.5.1
  (crate-source "diffs" "0.5.1"
                "01f8bp77kgahgvr3s2igmnmsxynjklq830lmp2wp2jyph6bnq4gz"))

(define rust-digest-0.10.7
  (crate-source "digest" "0.10.7"
                "14p2n6ih29x81akj097lvz7wi9b6b9hvls0lwrv7b6xwyy0s5ncy"))

(define rust-digest-0.8.1
  (crate-source "digest" "0.8.1"
                "1madjl27f3kj5ql7kwgvb9c8b7yb7bv7yfgx7rqzj4i3fp4cil7k"))

(define rust-digest-0.9.0
  (crate-source "digest" "0.9.0"
                "0rmhvk33rgvd6ll71z8sng91a52rw14p0drjn1da0mqa138n1pfk"))

(define rust-directories-3.0.2
  (crate-source "directories" "3.0.2"
                "1mz1khca62zd48kphdy5hn4f8r55ywkyn23pjdbkj4h32zzh15p6"))

(define rust-directories-4.0.1
  (crate-source "directories" "4.0.1"
                "045jbj5y2f1fmjs9rfcw95y0vjydb2rqqhz1sdnqhdmxv96ms77m"))

(define rust-directories-5.0.0
  (crate-source "directories" "5.0.0"
                "1acv0z7vi7b8z0qn2xd8zirw8axva9351p1vwjfhi3n117l3pgkl"))

(define rust-directories-5.0.1
  (crate-source "directories" "5.0.1"
                "0dba6xzk79s1clqzxh2qlgzk3lmvvks1lzzjhhi3hd70hhxifjcs"))

(define rust-directories-6.0.0
  (crate-source "directories" "6.0.0"
                "0zgy2w088v8w865c11dmc3dih899fgrhvrfp7g83h6v6ai60kx8n"))

(define rust-directories-next-2.0.0
  (crate-source "directories-next" "2.0.0"
                "1g1vq8d8mv0vp0l317gh9y46ipqg2fxjnbc7lnjhwqbsv4qf37ik"))

(define rust-dirs-3.0.2
  (crate-source "dirs" "3.0.2"
                "028kqy0vrbfgrk1yc1flq2fqh8snyg17qlygawm0r79w211s1fih"))

(define rust-dirs-4.0.0
  (crate-source "dirs" "4.0.0"
                "0n8020zl4f0frfnzvgb9agvk4a14i1kjz4daqnxkgslndwmaffna"))

(define rust-dirs-5.0.1
  (crate-source "dirs" "5.0.1"
                "0992xk5vx75b2x91nw9ssb51mpl8x73j9rxmpi96cryn0ffmmi24"))

(define rust-dirs-6.0.0
  (crate-source "dirs" "6.0.0"
                "0knfikii29761g22pwfrb8d0nqpbgw77sni9h2224haisyaams63"))

(define rust-dirs-next-2.0.0
  (crate-source "dirs-next" "2.0.0"
                "1q9kr151h9681wwp6is18750ssghz6j9j7qm7qi1ngcwy7mzi35r"))

(define rust-dirs-sys-0.3.7
  (crate-source "dirs-sys" "0.3.7"
                "19md1cnkazham8a6kh22v12d8hh3raqahfk6yb043vrjr68is78v"))

(define rust-dirs-sys-0.4.1
  (crate-source "dirs-sys" "0.4.1"
                "071jy0pvaad9lsa6mzawxrh7cmr7hsmsdxwzm7jzldfkrfjha3sj"))

(define rust-dirs-sys-0.5.0
  (crate-source "dirs-sys" "0.5.0"
                "1aqzpgq6ampza6v012gm2dppx9k35cdycbj54808ksbys9k366p0"))

(define rust-dirs-sys-next-0.1.2
  (crate-source "dirs-sys-next" "0.1.2"
                "0kavhavdxv4phzj4l0psvh55hszwnr0rcz8sxbvx20pyqi2a3gaf"))

(define rust-discard-1.0.4
  (crate-source "discard" "1.0.4"
                "1h67ni5bxvg95s91wgicily4ix7lcw7cq0a5gy9njrybaibhyb91"))

(define rust-dispatch-0.2.0
  (crate-source "dispatch" "0.2.0"
                "0fwjr9b7582ic5689zxj8lf7zl94iklhlns3yivrnv8c9fxr635x"))

(define rust-dispatch2-0.3.0
  (crate-source "dispatch2" "0.3.0"
                "1v1ak9w0s8z1g13x4mj2y5im9wmck0i2vf8f8wc9l1n6lqi9z849"))

(define rust-display-error-chain-0.2.2
  (crate-source "display-error-chain" "0.2.2"
                "1xbcilzyfc8n60vjkmsf8v53kw855xw68jh69hpza6dwhrp19hhb"))

(define rust-displaydoc-0.2.5
  (crate-source "displaydoc" "0.2.5"
                "1q0alair462j21iiqwrr21iabkfnb13d6x5w95lkdg21q2xrqdlp"))

(define rust-dissimilar-1.0.10
  (crate-source "dissimilar" "1.0.10"
                "08b94x25x3ba6vg79i53wspxyagqr43crg9dw2zn2dpgl3dgyxc9"))

(define rust-divan-0.1.21
  (crate-source "divan" "0.1.21"
                "0cw9i6yrr78axsjpd7pb2vfzdpxm19bs7d1j1s5y13wbqxz4a1d4"))

(define rust-divan-macros-0.1.21
  (crate-source "divan-macros" "0.1.21"
                "08rkmilvqmdmgqb5msnk70psipx7bcz1fh5641j5sm2n160bqmlm"))

(define rust-dlib-0.5.2
  (crate-source "dlib" "0.5.2"
                "04m4zzybx804394dnqs1blz241xcy480bdwf3w9p4k6c3l46031k"))

(define rust-dlv-list-0.5.2
  (crate-source "dlv-list" "0.5.2"
                "0pqvrinxzdz7bpy4a3p450h8krns3bd0mc3w0qqvm03l2kskj824"))

(define rust-dns-lookup-2.0.4
  (crate-source "dns-lookup" "2.0.4"
                "1z74n2zij2gahycabm0gkmkyx574h76gwk7sz93yqpr3qa3n0xp5"))

(define rust-dns-parser-0.8.0
  (crate-source "dns-parser" "0.8.0"
                "1spv0psvxc31qg9xnqf0gmjclyiim7vp23r2b1gzf1ix8zlkply4"))

(define rust-doc-comment-0.3.3
  (crate-source "doc-comment" "0.3.3"
                "043sprsf3wl926zmck1bm7gw0jq50mb76lkpk49vasfr6ax1p97y"))

(define rust-dockworker-0.6.0
  (crate-source "dockworker" "0.6.0"
                "1bwlmrykkyv96xf1858j6irwglln78ra7jii8pf5l9nn4qg34aa7"))

(define rust-docopt-1.1.1
  (crate-source "docopt" "1.1.1"
                "07x5g52n6fzilxhk5220caznkvdnzzvahlzrzkmgj8y88sc12gvz"))

(define rust-doctest-file-1.0.0
  (crate-source "doctest-file" "1.0.0"
                "0qkmnrsx2kszm58wxyry63bs35msj9chdb6jlh54a8cdwaiizj5a"))

(define rust-document-features-0.2.11
  (crate-source "document-features" "0.2.11"
                "0pdhpbz687fk2rkgz45yy3gvbhlxliwb7g1lj3jbx1f1qr89n94m"))

(define rust-documented-0.9.1
  (crate-source "documented" "0.9.1"
                "1wy67lk299zn0csr5fkmxyz0vkas0ycqk26ys99lbg4m14pv6vdw"))

(define rust-documented-macros-0.9.1
  (crate-source "documented-macros" "0.9.1"
                "1yp5hy7r88x1bjcyclqyrkpiddm42xmpr3s1znjzjn4vj8svp553"))

(define rust-dot-writer-0.1.4
  (crate-source "dot-writer" "0.1.4"
                "1w3sqyy2sg2i29didlv183h2zl7sgqf27wmcb6spqnzrsc4abxx2"))

(define rust-dotenvy-0.15.7
  (crate-source "dotenvy" "0.15.7"
                "16s3n973n5aqym02692i1npb079n5mb0fwql42ikmwn8wnrrbbqs"))

(define rust-downcast-0.11.0
  (crate-source "downcast" "0.11.0"
                "1wa78ahlc57wmqyq2ncr80l7plrkgz57xsg7kfzgpcnqac8gld8l"))

(define rust-downcast-rs-1.2.1
  (crate-source "downcast-rs" "1.2.1"
                "1lmrq383d1yszp7mg5i7i56b17x2lnn3kb91jwsq0zykvg2jbcvm"))

(define rust-dpc-pariter-0.4.0
  (crate-source "dpc-pariter" "0.4.0"
                "0s546d43yyiw55jz3yw4nyxgzmnc4f0gamzkfi6m7kyw2xlf2anl"))

(define rust-dpi-0.1.1
  (crate-source "dpi" "0.1.1"
                "0lzz48gpgbwdrw0s8vib0589ij9jizv1vzsphm4xd9kw58lhwp7j"))

(define rust-dpi-0.1.2
  (crate-source "dpi" "0.1.2"
                "0xhsvzgjvdch2fwmfc9vkb708b0q59b6imypyjlgbiigyb74rcfq"))

(define rust-drm-0.14.1
  (crate-source "drm" "0.14.1"
                "0vvmj9n0wslrbw3rinpzlfyhwwgr02gqspy1al5gfh99dif8rg40"
                #:snippet '(delete-file-recursively "examples")))

(define rust-drm-ffi-0.9.0
  (crate-source "drm-ffi" "0.9.0"
                "12vff80hdpp81gj5lqw25xnkppwsxc4wklpn8nc556wsv5ci9r6q"))

(define rust-drm-fourcc-2.2.0
  (crate-source "drm-fourcc" "2.2.0"
                "1x76v9a0pkgym4n6cah4barnai9gsssm7gjzxskw2agwibdvrbqa"))

(define rust-drm-sys-0.8.0
  (crate-source "drm-sys" "0.8.0"
                "1345z72hd2rna4qxd2zcpbzvw0z7ywfndk6g2ngdci69vg46dyxs"))

(define rust-dsa-0.6.3
  (crate-source "dsa" "0.6.3"
                "12bn5d0nnni4mlla24m3bwi4mhy2nfmyak2qjl0pdbc4j1525g28"))

(define rust-dsl-auto-type-0.1.3
  (crate-source "dsl_auto_type" "0.1.3"
                "0nzzqmqyymlnffhms1kism5xhgsk73mq6r6pdpr8azsjlynfk6hk"))

(define rust-dtoa-1.0.10
  (crate-source "dtoa" "1.0.10"
                "016gid01rarcdv57h049d7nr9daxc2hc2gqzx0mji57krywd7bfn"))

(define rust-dtoa-short-0.3.5
  (crate-source "dtoa-short" "0.3.5"
                "11rwnkgql5jilsmwxpx6hjzkgyrbdmx1d71s0jyrjqm5nski25fd"))

(define rust-dtparse-2.0.1
  (crate-source "dtparse" "2.0.1"
                "1mqz4164mc4xyq73c22wf900v8cn4sy63nalrkr5mlr614y41yr3"))

(define rust-duct-0.13.7
  (crate-source "duct" "0.13.7"
                "174jk13rlvfgypha4f3l27mzzyc0ci7zginh5hjn6jr2s4c5gaz4"))

(define rust-dunce-1.0.5
  (crate-source "dunce" "1.0.5"
                "04y8wwv3vvcqaqmqzssi6k0ii9gs6fpz96j5w9nky2ccsl23axwj"))

(define rust-duplicate-1.0.0
  (crate-source "duplicate" "1.0.0"
                "1jrh28a3g9xw0w9v21y92d4k08lgr1ffg8mjhw2h67h6r5mfcy6y"))

(define rust-dwrote-0.11.3
  (crate-source "dwrote" "0.11.3"
                "1gi4027mm4mdqjkb54gj7f01vlg07jjsqffqv05mj0ffzj9g3qdz"))

(define rust-dyn-clonable-0.9.2
  (crate-source "dyn-clonable" "0.9.2"
                "01885xap4dmln3yspzyr0mmcwnm9mdhlp80ag0iig3nmpywznvm3"))

(define rust-dyn-clonable-impl-0.9.2
  (crate-source "dyn-clonable-impl" "0.9.2"
                "0fi1fy6r2zaq88n21yszlcdbm66iz3xi2dbgl8vrm5sq83ap31ky"))

(define rust-dyn-clone-1.0.19
  (crate-source "dyn-clone" "1.0.19"
                "01ahm5abl20480v48nxy4ffyx80cs6263q9zf0gnrxpvm6w8yyhw"))

(define rust-dyn-clone-1.0.20
  (crate-source "dyn-clone" "1.0.20"
                "0m956cxcg8v2n8kmz6xs5zl13k2fak3zkapzfzzp7pxih6hix26h"))

(define rust-easy-cast-0.5.3
  (crate-source "easy-cast" "0.5.3"
                "1mjhjxibj228bzs817rwaqdicxmln6dqqffabrjn5j1fd4v2g1bj"))

(define rust-eax-0.5.0
  (crate-source "eax" "0.5.0"
                "0a5cpzk577f1lw3wkk20iqvavnbdr5yzjrcglvbvk0ivj2yzlm4r"
                #:snippet '(delete-file-recursively "tests")))

(define rust-ec25519-0.1.0
  (crate-source "ec25519" "0.1.0"
                "09f8f60n3m79hzylpqf26z5x46ky3yp15569733ph4f05wx57zdx"))

(define rust-ecb-0.1.2
  (crate-source "ecb" "0.1.2"
                "1iw1i0mwkvg3599mlw24iibid6i6zv3a3jhghm2j3v0sbfbzm2qs"
                #:snippet '(delete-file-recursively "tests")))

(define rust-ecdsa-0.16.9
  (crate-source "ecdsa" "0.16.9"
                "1jhb0bcbkaz4001sdmfyv8ajrv8a1cg7z7aa5myrd4jjbhmz69zf"))

(define rust-ecies-ed25519-ng-0.5.3.554ca29
  (origin
    (method git-fetch)
    (uri (git-reference (url "https://github.com/Revertron/ecies-ed25519-ng")
                        (commit "554ca29a1bbd55f0c7e2f75cb3c7e0e3030afc15")))
    (file-name (git-file-name "rust-ecies-ed25519-ng" "0.5.3.554ca29"))
    (sha256 (base32 "04s0ycvnz1wbccf46a63w6zxiqm9yszw71q6fk1ssdc64qj7k5mh"))))

(define rust-ecow-0.2.5
  (crate-source "ecow" "0.2.5"
                "1i8d7ix1ggfmr1ncdn1c9ca8vjniaz813zwj1vldi7ymnlg4haxr"))

(define rust-ed25519-1.5.3
  (crate-source "ed25519" "1.5.3"
                "1rzydm5wd8szkddx3g55w4vm86y1ika8qp8qwckada5vf1fg7kwi"
                #:snippet '(delete-file-recursively "tests")))

(define rust-ed25519-2.2.3
  (crate-source "ed25519" "2.2.3"
                "0lydzdf26zbn82g7xfczcac9d7mzm3qgx934ijjrd5hjpjx32m8i"
                #:snippet '(delete-file-recursively "tests")))

(define rust-ed25519-compact-2.1.1
  (crate-source "ed25519-compact" "2.1.1"
                "1431kxw67xkk5y5kamfdjxnqbzqy5y4p032syi3wva5y8h7ldcz9"))

(define rust-ed25519-dalek-1.0.1
  (crate-source "ed25519-dalek" "1.0.1"
                "17bsriciv93nkm39z22w7mr0h2a3hnbmgf378v4c895gvkkblqn7"))

(define rust-ed25519-dalek-2.1.1
  (crate-source "ed25519-dalek" "2.1.1"
                "0w88cafwglg9hjizldbmlza0ns3hls81zk1bcih3m5m3h67algaa"
                #:snippet '(for-each delete-file-recursively '("docs" "tests"))))

(define rust-ed25519-dalek-2.2.0
  (crate-source "ed25519-dalek" "2.2.0"
                "1agcwij1z687hg26ngzwhnmpz29b2w56m8z1ap3pvrnfh709drvh"
                #:snippet '(delete-file-recursively "tests")))

(define rust-edit-0.1.5
  (crate-source "edit" "0.1.5"
                "02dan6bg9pcj42ny48g8fq9f76w30c826n4gihy1d1s7fq78cr7k"))

(define rust-edit-distance-2.1.3
  (crate-source "edit-distance" "2.1.3"
                "15vkx4haqr6ldgpmamb38h30v57clpx6klfzanhhk303ggl9gx73"))

(define rust-editdistancek-1.0.2
  (crate-source "editdistancek" "1.0.2"
                "04r6lfq9sfz3wqhqm6fzfcqbj8w16y8bh0x6kzkgkimislixy0iy"))

(define rust-either-1.13.0
  (crate-source "either" "1.13.0"
                "1w2c1mybrd7vljyxk77y9f4w9dyjrmp3yp82mk7bcm8848fazcb0"))

(define rust-either-1.15.0
  (crate-source "either" "1.15.0"
                "069p1fknsmzn9llaizh77kip0pqmcwpdsykv2x30xpjyija5gis8"))

(define rust-either-1.9.0
  (crate-source "either" "1.9.0"
                "01qy3anr7jal5lpc20791vxrw0nl6vksb5j7x56q2fycgcyy8sm2"))

(define rust-elasticlunr-rs-3.0.2
  (crate-source "elasticlunr-rs" "3.0.2"
                "0wb5kypd4pi3672zwq38vq1rynwwvs16dvjg47x6sr80lmikis21"))

(define rust-elliptic-curve-0.13.8
  (crate-source "elliptic-curve" "0.13.8"
                "0ixx4brgnzi61z29r3g1606nh2za88hzyz8c5r3p6ydzhqq09rmm"
                #:snippet '(delete-file-recursively "tests")))

(define rust-emacs-0.11.0
  (crate-source "emacs" "0.11.0"
                "1xn6nr8m3p24irlmv3i7c9n95yb10qc6aikwqv99qlmhbky5x0z7"))

(define rust-emacs-0.18.0
  (crate-source "emacs" "0.18.0"
                "0r860i73b2680i2fhdl2l1wwvvmf2zksncpckgkksdcx310ak5v7"))

(define rust-emacs-0.19.0
  (crate-source "emacs" "0.19.0"
                "133ip19m1nq8krf04klh1jpm833wpw6kdy2jksgylbw4xcp3yxws"))

(define rust-emacs-macros-0.11.0
  (crate-source "emacs-macros" "0.11.0"
                "0390y8vafxdi334hhgrzvcqjq3n5ckcmvilqcfp8vajjq8irrly6"))

(define rust-emacs-macros-0.17.0
  (crate-source "emacs-macros" "0.17.0"
                "0qg1dcn5acbirq617qq2fgg9adswif2dnr292s3qnq62wzgnyrb9"))

(define rust-emacs-module-0.10.0
  (crate-source "emacs_module" "0.10.0"
                "1gf9lz735xbkyir53dyv362drfx3nin5an5cx39kd8q8kjjwix5g"))

(define rust-emacs-module-0.18.0
  (crate-source "emacs_module" "0.18.0"
                "1ypjyyv2ca3vza4sia91ckxamgfk63yd8frkvg3d4ph4fk4pn1mk"))

(define rust-email-address-0.2.9
  (crate-source "email_address" "0.2.9"
                "0jf4v3npa524c7npy7w3jl0a6gng26f51a4bgzs3jqna12dz2yg0"))

(define rust-embed-resource-2.5.1
  (crate-source "embed-resource" "2.5.1"
                "0yb3kbw3xpghiwf69769jpng725kwa2cxm27qj5s7dm0cfgnz2xn"))

(define rust-embed-resource-3.0.2
  (crate-source "embed-resource" "3.0.2"
                "1qsicl30n4lzr3dki0fvqimrab1x8vq17a1wnmasc0qciq6nxg3z"))

(define rust-embedded-io-0.4.0
  (crate-source "embedded-io" "0.4.0"
                "1v9wrc5nsgaaady7i3ya394sik5251j0iq5rls7mrx7fv696h6pg"))

(define rust-embedded-io-0.6.1
  (crate-source "embedded-io" "0.6.1"
                "0v901xykajh3zffn6x4cnn4fhgfw3c8qpjwbsk6gai3gaccg3l7d"))

(define rust-eml-parser-0.1.4
  (crate-source "eml-parser" "0.1.4"
                "0bhgx1i952g2v7w0icnqkylvhfiyb2am2mbw89v8zq0jz0aqvnx7"))

(define rust-emojis-0.6.4
  (crate-source "emojis" "0.6.4"
                "1x02bixi1f8kjk8g5qsrwpwznpya65yh5gy8p8wja7qq3zgz3qcr"))

(define rust-ena-0.14.3
  (crate-source "ena" "0.14.3"
                "1m9a5hqk6qn5sqnrc40b55yr97drkfdzd0jj863ksqff8gfqn91x"))

(define rust-enable-ansi-support-0.2.1
  (crate-source "enable-ansi-support" "0.2.1"
                "0q5wv5b9inh7kzc2464ch51ffk920f9yb0q9xvvlp9cs5apg6kxa"))

(define rust-encode-unicode-0.3.6
  (crate-source "encode_unicode" "0.3.6"
                "07w3vzrhxh9lpjgsg2y5bwzfar2aq35mdznvcp3zjl0ssj7d4mx3"))

(define rust-encode-unicode-1.0.0
  (crate-source "encode_unicode" "1.0.0"
                "1h5j7j7byi289by63s3w4a8b3g6l5ccdrws7a67nn07vdxj77ail"))

(define rust-encoding-0.2.33
  (crate-source "encoding" "0.2.33"
                "1v1ndmkarh9z3n5hk53da4z56hgk9wa5kcsm7cnx345raqw983bb"))

(define rust-encoding-index-japanese-1.20141219.5
  (crate-source "encoding-index-japanese" "1.20141219.5"
                "148c1lmd640p1d7fzk0nv7892mbyavvwddgqvcsm78798bzv5s04"))

(define rust-encoding-index-korean-1.20141219.5
  (crate-source "encoding-index-korean" "1.20141219.5"
                "10cxabp5ppygbq4y6y680856zl9zjvq7ahpiw8zj3fmwwsw3zhsd"))

(define rust-encoding-index-simpchinese-1.20141219.5
  (crate-source "encoding-index-simpchinese" "1.20141219.5"
                "1xria2i7mc5dqdrpqxasdbxv1qx46jjbm53if3y1i4cvj2a72ynq"))

(define rust-encoding-index-singlebyte-1.20141219.5
  (crate-source "encoding-index-singlebyte" "1.20141219.5"
                "0jp85bz2pprzvg9m95w4q0vibh67b6w3bx35lafay95jzyndal9k"))

(define rust-encoding-index-tests-0.1.4
  (crate-source "encoding_index_tests" "0.1.4"
                "0s85y091gl17ixass49bzaivng7w8p82p6nyvz2r3my9w4mxhim2"))

(define rust-encoding-index-tradchinese-1.20141219.5
  (crate-source "encoding-index-tradchinese" "1.20141219.5"
                "060ci4iz6xfvzk38syfbjvs7pix5hch3mvxkksswmqwcd3aj03px"))

(define rust-encoding-rs-0.8.33
  (crate-source "encoding_rs" "0.8.33"
                "1qa5k4a0ipdrxq4xg9amms9r9pnnfn7nfh2i9m3mw0ka563b6s3j"))

(define rust-encoding-rs-0.8.35
  (crate-source "encoding_rs" "0.8.35"
                "1wv64xdrr9v37rqqdjsyb8l8wzlcbab80ryxhrszvnj59wy0y0vm"))

(define rust-encoding-rs-io-0.1.7
  (crate-source "encoding_rs_io" "0.1.7"
                "10ra4l688cdadd8h1lsbahld1zbywnnqv68366mbhamn3xjwbhqw"))

(define rust-endi-1.1.0
  (crate-source "endi" "1.1.0"
                "1gxp388g2zzbncp3rdn60wxkr49xbhhx94nl9p4a6c41w4ma7n53"))

(define rust-endian-type-0.1.2
  (crate-source "endian-type" "0.1.2"
                "0bbh88zaig1jfqrm7w3gx0pz81kw2jakk3055vbgapw3dmk08ky3"))

(define rust-enquote-1.1.0
  (crate-source "enquote" "1.1.0"
                "0clrjghlfkkb7sndabs5wch0fz2nif6nj4b117s8kqxx3nqnrhq6"))

(define rust-entities-1.0.1
  (crate-source "entities" "1.0.1"
                "1jnpr0zvj97wm9pnh7fnl74rzaar39hhg65p03cm08bqqgj0lcmm"))

(define rust-enum-as-inner-0.5.1
  (crate-source "enum-as-inner" "0.5.1"
                "05m1frlkgrikja714zxb97i00rhj36zlifiwiby7ymkx0jx0nwn9"))

(define rust-enum-as-inner-0.6.1
  (crate-source "enum-as-inner" "0.6.1"
                "1g3cywc65d9w974l2xy86ij13njss3qjc7b0kfbzbws9qrjs5rm1"))

(define rust-enum-dispatch-0.3.13
  (crate-source "enum_dispatch" "0.3.13"
                "1kby2jz173ggg7wk41vjsskmkdyx7749ll8lhqhv6mb5qqmww65a"))

(define rust-enum-iterator-0.7.0
  (crate-source "enum-iterator" "0.7.0"
                "1rldnx6avpz39i1bwb65d4gs303p40syyfc4zqwlx7mpxp2wbsjf"))

(define rust-enum-iterator-derive-0.7.0
  (crate-source "enum-iterator-derive" "0.7.0"
                "0ndihb41kmi6pxc2bs097abxliw2pgnnw412lhdqfymjc1vw6d61"))

(define rust-enum-map-0.6.6
  (crate-source "enum-map" "0.6.6"
                "166yh44izfz871f4avl10cp0l161shhsawfcl1q6slv3siamqn93"))

(define rust-enum-map-1.1.1
  (crate-source "enum-map" "1.1.1"
                "1n99fg6hwxjfb9fzil2zrvc7radj9yqnyjn8vrc110hnc6xag4z8"))

(define rust-enum-map-2.7.3
  (crate-source "enum-map" "2.7.3"
                "1sgjgl4mmz93jdkfdsmapc3dmaq8gddagw9s0fd501w2vyzz6rk8"))

(define rust-enum-map-derive-0.17.0
  (crate-source "enum-map-derive" "0.17.0"
                "1sv4mb343rsz4lc3rh7cyn0pdhf7fk18k1dgq8kfn5i5x7gwz0pj"))

(define rust-enum-map-derive-0.4.6
  (crate-source "enum-map-derive" "0.4.6"
                "0mg43p1x90cz604zddk9qzss077v2id04qmmbpa1i7jc637m1i75"))

(define rust-enum-map-derive-0.6.0
  (crate-source "enum-map-derive" "0.6.0"
                "1ah7b71llknvwj031zsrqpxam5566hbc2i6vq7v4zqzn1ap8w9w4"))

(define rust-enum-ordinalize-4.3.0
  (crate-source "enum-ordinalize" "4.3.0"
                "1max64z9giii61qcwl56rndd7pakaylkaij5zqbbbvjl9vxdr87y"))

(define rust-enum-ordinalize-derive-4.3.1
  (crate-source "enum-ordinalize-derive" "4.3.1"
                "1zy53fabazimwv5cl0366k834ybixzl84lxj9mfavbnlfn532a0d"))

(define rust-enum-primitive-0.1.1
  (crate-source "enum_primitive" "0.1.1"
                "100ip2p3w1rq0clca2ai5shhvpxfipnsjncj0f9ralad5w4m2idy"))

(define rust-enum-primitive-derive-0.2.2
  (crate-source "enum-primitive-derive" "0.2.2"
                "03ibjjx8dc4akpq8ck24qda5ix4jybz9jagfxykd0s6vxb2vjxf3"))

(define rust-enumber-0.3.1
  (crate-source "enumber" "0.3.1"
                "1gxxqms1zpadyy1whq0ms1vwymz19mq197gfq53dhxnx14cig50f"))

(define rust-enumflags2-0.6.4
  (crate-source "enumflags2" "0.6.4"
                "182xd6cxxmadx1axnz6x73d12pzgwkc712zq2lxd4z1k48lxij43"))

(define rust-enumflags2-0.7.11
  (crate-source "enumflags2" "0.7.11"
                "0iwi60d54lgby0f29b5isikxraf0wvnqdmlddx68a62kbx34nbxs"))

(define rust-enumflags2-0.7.12
  (crate-source "enumflags2" "0.7.12"
                "1vzcskg4dca2jiflsfx1p9yw1fvgzcakcs7cpip0agl51ilgf9qh"))

(define rust-enumflags2-derive-0.6.4
  (crate-source "enumflags2_derive" "0.6.4"
                "1kkcwi4n76bi1c16ms00dyk4d393gdf29kpr4k9zsn5z7m7fjvll"))

(define rust-enumflags2-derive-0.7.11
  (crate-source "enumflags2_derive" "0.7.11"
                "0yfdjyrf9b4mi1r589azkyirjhzmdw29nqq0mdjnsyldlmjayk7w"))

(define rust-enumflags2-derive-0.7.12
  (crate-source "enumflags2_derive" "0.7.12"
                "09rqffacafl1b83ir55hrah9gza0x7pzjn6lr6jm76fzix6qmiv7"))

(define rust-enumset-1.0.11
  (crate-source "enumset" "1.0.11"
                "0x70xzxs2sj0yn4sh99bwk8s51aswvyvf1ldm6kziwa89nrcv6a7"))

(define rust-enumset-derive-0.6.0
  (crate-source "enumset_derive" "0.6.0"
                "0k1hhph1kjkw1p726fk9f4nxz3kgngm1grxypk7rr68xvkxs70za"))

(define rust-env-filter-0.1.3
  (crate-source "env_filter" "0.1.3"
                "1l4p6f845cylripc3zkxa0lklk8rn2q86fqm522p6l2cknjhavhq"))

(define rust-env-home-0.1.0
  (crate-source "env_home" "0.1.0"
                "1zn08mk95rjh97831rky1n944k024qrwjhbcgb0xv9zhrh94xy67"))

(define rust-env-logger-0.10.2
  (crate-source "env_logger" "0.10.2"
                "1005v71kay9kbz1d5907l0y7vh9qn2fqsp2yfgb8bjvin6m0bm2c"))

(define rust-env-logger-0.11.8
  (crate-source "env_logger" "0.11.8"
                "17q6zbjam4wq75fa3m4gvvmv3rj3ch25abwbm84b28a0j3q67j0k"))

(define rust-env-logger-0.6.2
  (crate-source "env_logger" "0.6.2"
                "1lx2s5nk96xx4i3m4zc4ghqgi8kb07dsnyiv8jk2clhax42dxz5a"))

(define rust-env-logger-0.8.4
  (crate-source "env_logger" "0.8.4"
                "1qzw8g11dbdfi7ixm44ldykwcqsxqkh8vx5cgpd88zmclgz8g4d1"))

(define rust-env-logger-0.9.3
  (crate-source "env_logger" "0.9.3"
                "1rq0kqpa8my6i1qcyhfqrn1g9xr5fbkwwbd42nqvlzn9qibncbm1"))

(define rust-env-proxy-0.4.1
  (crate-source "env_proxy" "0.4.1"
                "1qabqhgybx1jzh6dmpx7kssciw312i8aa6al7fj0d12k32z1jl1s"))

(define rust-environment-0.1.1
  (crate-source "environment" "0.1.1"
                "1vh32mcxf3z8xaibwv751zj14d08nh7iwk1vqdj90rkq17i18jqz"))

(define rust-envmnt-0.8.4
  (crate-source "envmnt" "0.8.4"
                "0na01z737h9yn5y8qj3qjzipirnprgyprbv199632qbw53y2ilx2"
                #:snippet '(delete-file-recursively "docs")))

(define rust-envy-0.4.2
  (crate-source "envy" "0.4.2"
                "0rar459p7pl19v6pbx98q3hi2hxfl8q1ndxxw5d4zd9cgway0irz"))

(define rust-equator-0.4.2
  (crate-source "equator" "0.4.2"
                "1z760z5r0haxjyakbqxvswrz9mq7c29arrivgq8y1zldhc9v44a7"))

(define rust-equator-macro-0.4.2
  (crate-source "equator-macro" "0.4.2"
                "1cqzx3cqn9rxln3a607xr54wippzff56zs5chqdf3z2bnks3rwj4"))

(define rust-equivalent-1.0.1
  (crate-source "equivalent" "1.0.1"
                "1malmx5f4lkfvqasz319lq6gb3ddg19yzf9s8cykfsgzdmyq0hsl"))

(define rust-equivalent-1.0.2
  (crate-source "equivalent" "1.0.2"
                "03swzqznragy8n0x31lqc78g2af054jwivp7lkrbrc0khz74lyl7"))

(define rust-erased-serde-0.4.6
  (crate-source "erased-serde" "0.4.6"
                "1dx5hj16hsl143czwl2g7ymdi1y84lsjyyii2zprzjqzyn3xh170"))

(define rust-errno-0.2.8
  (crate-source "errno" "0.2.8"
                "18cnqgk8r6lq1n5cfy3bryiyz9zkqr10dxj49sa3fkzfamih8fgn"))

(define rust-errno-0.3.10
  (crate-source "errno" "0.3.10"
                "0pgblicz1kjz9wa9m0sghkhh2zw1fhq1mxzj7ndjm746kg5m5n1k"))

(define rust-errno-0.3.11
  (crate-source "errno" "0.3.11"
                "0kjrrcaa5nvickysw7z3vm5p0l7l457idf1ff3z6ang8qwnx8vcp"))

(define rust-errno-0.3.12
  (crate-source "errno" "0.3.12"
                "066ss2qln9z5q4816d3wcvq2bzasn7dajfkhcfqflfsy6pwlx8ff"))

(define rust-errno-0.3.13
  (crate-source "errno" "0.3.13"
                "1bd5g3srn66zr3bspac0150bvpg1s7zi6zwhwhlayivciz12m3kp"))

(define rust-errno-0.3.14
  (crate-source "errno" "0.3.14"
                "1szgccmh8vgryqyadg8xd58mnwwicf39zmin3bsn63df2wbbgjir"))

(define rust-errno-0.3.2
  (crate-source "errno" "0.3.2"
                "0kqg5zahxlyz1pzd8m7kgr7xcligr5n7dlkk2riz27lnlxlzcc3b"))

(define rust-errno-dragonfly-0.1.2
  (crate-source "errno-dragonfly" "0.1.2"
                "1grrmcm6q8512hkq5yzch3yv8wafflc2apbmsaabiyk44yqz2s5a"))

(define rust-error-chain-0.12.4
  (crate-source "error-chain" "0.12.4"
                "1z6y5isg0il93jp287sv7pn10i4wrkik2cpyk376wl61rawhcbrd"))

(define rust-error-code-2.3.1
  (crate-source "error-code" "2.3.1"
                "08baxlf8qz01lgjsdbfhs193r9y1nlc566s5xvzyf4dzwy8qkwb4"))

(define rust-error-code-3.2.0
  (crate-source "error-code" "3.2.0"
                "0nqpbhi501z3ydaxg4kjyb68xcw025cj22prwabiky0xsljl8ix0"))

(define rust-error-code-3.3.1
  (crate-source "error-code" "3.3.1"
                "0bx9hw3pahzqym8jvb0ln4qsabnysgn98mikyh2afhk9rif31nd5"))

(define rust-error-code-3.3.2
  (crate-source "error-code" "3.3.2"
                "0nacxm9xr3s1rwd6fabk3qm89fyglahmbi4m512y0hr8ym6dz8ny"))

(define rust-error-stack-0.5.0
  (crate-source "error-stack" "0.5.0"
                "1lf5zy1fjjqdwjkc445sw80hpmxi63ymcxgjh3q6642x2hck6hgy"))

(define rust-esaxx-rs-0.1.10
  (crate-source "esaxx-rs" "0.1.10"
                "1rm6vm5yr7s3n5ly7k9x9j6ra5p2l2ld151gnaya8x03qcwf05yq"))

(define rust-escape8259-0.5.3
  (crate-source "escape8259" "0.5.3"
                "1in2zrdak794x8v11lkrrgyqxjjqbijvg0yfn2paay0rb9xxv4jn"))

(define rust-escargot-0.5.14
  (crate-source "escargot" "0.5.14"
                "0482wkzl69ilwagbpcx3rwv9f7ynqyiqpkm07z1kyjc00xsm3ww3"))

(define rust-escargot-0.5.15
  (crate-source "escargot" "0.5.15"
                "1kyhkrxdbxlw839h3l838ck5ir96i9v2p9x6kh650yy95fisxhqi"))

(define rust-esp-idf-part-0.5.0
  (crate-source "esp-idf-part" "0.5.0"
                "0i4q8hanzphglwzj408bixkzwb9z6gapq226gc46f01p69n0pxar"
                #:snippet '(delete-file-recursively "tests")))

(define rust-etcetera-0.10.0
  (crate-source "etcetera" "0.10.0"
                "1rka6bskn93pdhx32xaagr147q95z5bnz7ym5xr85jw00wyv3ir6"))

(define rust-etcetera-0.8.0
  (crate-source "etcetera" "0.8.0"
                "0hxrsn75dirbjhwgkdkh0pnpqrnq17ypyhjpjaypgax1hd91nv8k"))

(define rust-etherparse-0.13.0
  (crate-source "etherparse" "0.13.0"
                "146rcbnhlpcbl6c6csfhvz0227wbiwhk13md6acq8211b7m94wl2"))

(define rust-ethtool-0.2.8
  (crate-source "ethtool" "0.2.8"
                "19diyw4y70ss6crmzqf6ddrahyv4a8hmzh920qjc0kip6iff2yal"))

(define rust-euclid-0.22.11
  (crate-source "euclid" "0.22.11"
                "0j4yb01x9dn5hbbbigd3mwdplv4m29k5drmhmc95lj3yfi5xp75d"))

(define rust-evdev-0.12.2
  (crate-source "evdev" "0.12.2"
                "19qh6r1z4v8ja6qqigjbg9vckbhlycc6wkqgzfz9fcln7almaq5b"))

(define rust-evdev-rs-0.6.1
  (crate-source "evdev-rs" "0.6.1"
                "1fkdw23kjjn8an3xjpi2g74yhdfkv89ngsrkjd2cxz5n1xwxa4lq"))

(define rust-evdev-sys-0.2.5
  (crate-source "evdev-sys" "0.2.5"
                "0vgax74wjsm22nrx6ikh8m7bj3nggf83s961i5qd85bvahmx9shl"
                #:snippet '(delete-file-recursively "libevdev")))

(define rust-event-listener-2.5.3
  (crate-source "event-listener" "2.5.3"
                "1q4w3pndc518crld6zsqvvpy9lkzwahp2zgza9kbzmmqh9gif1h2"))

(define rust-event-listener-3.1.0
  (crate-source "event-listener" "3.1.0"
                "1hihkg6ihvb6p9yi7nq11di8mhd5y0iqv81ij6h0rf0fvsy7ff6r"))

(define rust-event-listener-4.0.3
  (crate-source "event-listener" "4.0.3"
                "0vk4smw1vf871vi76af1zn7w69jg3zmpjddpby2qq91bkg21bck7"))

(define rust-event-listener-5.4.0
  (crate-source "event-listener" "5.4.0"
                "1bii2gn3vaa33s0gr2zph7cagiq0ppcfxcxabs24ri9z9kgar4il"))

(define rust-event-listener-5.4.1
  (crate-source "event-listener" "5.4.1"
                "1asnp3agbr8shcl001yd935m167ammyi8hnvl0q1ycajryn6cfz1"))

(define rust-event-listener-strategy-0.5.4
  (crate-source "event-listener-strategy" "0.5.4"
                "14rv18av8s7n8yixg38bxp5vg2qs394rl1w052by5npzmbgz7scb"))

(define rust-executable-path-1.0.0
  (crate-source "executable-path" "1.0.0"
                "0y0lhhrx9f9q1f81am3b20f8znixjcvqfg7kx220pjg3i5nmmg1y"))

(define rust-execute-0.2.13
  (crate-source "execute" "0.2.13"
                "046s3a33v4klqs4az69i4byvz1xpqb9visarnvm6mrvcx67610is"))

(define rust-execute-command-macro-0.1.9
  (crate-source "execute-command-macro" "0.1.9"
                "1g88vmsm3nf0xdq266m0dwdg8k564svyrwsgvh8yjr3mahywbplh"))

(define rust-execute-command-macro-impl-0.1.10
  (crate-source "execute-command-macro-impl" "0.1.10"
                "0jygv1jhfkqglascjk7yq3m2kdgm1yd3y9kikjmhbl0s0imd936f"))

(define rust-execute-command-tokens-0.1.7
  (crate-source "execute-command-tokens" "0.1.7"
                "1bdwzw0q983xx4d0wmmyzwhcn0qpl8xcl826cx27z5xynqg35p39"))

(define rust-exitcode-1.1.2
  (crate-source "exitcode" "1.1.2"
                "14x1pgwx86x1gfc5zqgj04akr9pzg14w75d9cblc49vhnij3g1fy"))

(define rust-expect-test-1.5.1
  (crate-source "expect-test" "1.5.1"
                "1c5c081ykm4k5rlsam9jw56w4wgs2h7r4aj78zxlis1i8kzl7bv3"))

(define rust-expectrl-0.7.1
  (crate-source "expectrl" "0.7.1"
                "0ykiqkh2r3ky1244r4j0x49hqawdrfdh1mai6fdapz2kb6989rzd"))

(define rust-exr-1.73.0
  (crate-source "exr" "1.73.0"
                "1q47yq78q9k210r6jy1wwrilxwwxqavik9l3l426rd17k7srfcgq"))

(define rust-eyra-0.20.0
  (crate-source "eyra" "0.20.0"
                "1mdl76qji4f5yjyg1z6k385pgfcywrbd1lpi8m0f1sz8las79w0y"))

(define rust-eyra-0.22.0
  (crate-source "eyra" "0.22.0"
                "15xsh4ff6g3vk4yq1jh643myhhiwszqx3vdbabj2z34jq9w8j1vd"))

(define rust-eyre-0.6.12
  (crate-source "eyre" "0.6.12"
                "1v1a3vb9gs5zkwp4jzkcfnpg0gvyp4ifydzx37f4qy14kzcibnbw"))

(define rust-faccess-0.2.4
  (crate-source "faccess" "0.2.4"
                "1m967rz9qknzbqh7l8hga91s32y3p041mbk8w81skmh2b116dbjr"))

(define rust-failure-0.1.8
  (crate-source "failure" "0.1.8"
                "11jg1wmbkijrs6bk9fqnbrm9zf0850whnqpgnxyswbn0dk8rnbnk"))

(define rust-failure-derive-0.1.8
  (crate-source "failure_derive" "0.1.8"
                "1936adqqk080439kx2bjf1bds7h89sg6wcif4jw0syndcv3s6kda"))

(define rust-fake-simd-0.1.2
  (crate-source "fake-simd" "0.1.2"
                "1vfylvk4va2ivqx85603lyqqp0zk52cgbs4n5nfbbbqx577qm2p8"))

(define rust-fallible-collections-0.5.1
  (crate-source "fallible_collections" "0.5.1"
                "0x6zmjkmr5ps2riylrrdvnds85qd3i31jmcjvghs76s19p8qaglb"))

(define rust-fallible-iterator-0.2.0
  (crate-source "fallible-iterator" "0.2.0"
                "1xq759lsr8gqss7hva42azn3whgrbrs2sd9xpn92c5ickxm1fhs4"))

(define rust-fallible-iterator-0.3.0
  (crate-source "fallible-iterator" "0.3.0"
                "0ja6l56yka5vn4y4pk6hn88z0bpny7a8k1919aqjzp0j1yhy9k1a"))

(define rust-fallible-streaming-iterator-0.1.9
  (crate-source "fallible-streaming-iterator" "0.1.9"
                "0nj6j26p71bjy8h42x6jahx1hn0ng6mc2miwpgwnp8vnwqf4jq3k"))

(define rust-fancy-regex-0.11.0
  (crate-source "fancy-regex" "0.11.0"
                "18j0mmzfycibhxhhhfja00dxd1vf8x5c28lbry224574h037qpxr"))

(define rust-fancy-regex-0.13.0
  (crate-source "fancy-regex" "0.13.0"
                "1wjbqjsdj8fkq6z2i9llq25iaqzd9f208vxnwg8mdbr2ba1lc7jk"))

(define rust-fancy-regex-0.14.0
  (crate-source "fancy-regex" "0.14.0"
                "162j2qx2ikgl79grq12mawyflwkirnjzrvxh11a1xbmwjidcn93f"))

(define rust-fast-glob-0.3.3
  (crate-source "fast-glob" "0.3.3"
                "1akg873y2j7pxgdwa5n8qh6y2adgsn14am0jk45kji1czbpz9z1s"))

(define rust-fast-image-resize-5.1.3
  (crate-source "fast_image_resize" "5.1.3"
                "1v2jzqbifid9ds1cd31i0qlhv7gwi70rxkwymrfrjl2zyy1cfip1"))

(define rust-fast-image-resize-5.1.4
  (crate-source "fast_image_resize" "5.1.4"
                "1kc9gqf92lc81k2kg213lqjblcwcibca6xfnb3c65wfqa8ranwnk"))

(define rust-fast-srgb8-1.0.0
  (crate-source "fast-srgb8" "1.0.0"
                "18g6xwwh4gnkyx1352hnvwagpv0n4y98yp2llm8vyvwxh487abnx"))

(define rust-faster-hex-0.10.0
  (crate-source "faster-hex" "0.10.0"
                "0wzvv4a1czxfxmh99cza2y0jps97hm3k1j6r6cs816qp5wnsw8vj"))

(define rust-faster-hex-0.9.0
  (crate-source "faster-hex" "0.9.0"
                "10wi4vqbdpkamw4qvra1ijp4as2j7j1zc66g4rdr6h0xv8gb38m2"))

(define rust-fastrand-1.9.0
  (crate-source "fastrand" "1.9.0"
                "1gh12m56265ihdbzh46bhh0jf74i197wm51jg1cw75q7ggi96475"))

(define rust-fastrand-2.1.1
  (crate-source "fastrand" "2.1.1"
                "19nyzdq3ha4g173364y2wijmd6jlyms8qx40daqkxsnl458jmh78"))

(define rust-fastrand-2.3.0
  (crate-source "fastrand" "2.3.0"
                "1ghiahsw1jd68df895cy5h3gzwk30hndidn3b682zmshpgmrx41p"))

(define rust-fat-macho-0.4.9
  (crate-source "fat-macho" "0.4.9"
                "0idkn366wipv2l757yqfgzgibqc6jvm89gdk9kpgmvf6lv54b72c"
                #:snippet '(delete-file-recursively "tests")))

(define rust-fax-0.2.6
  (crate-source "fax" "0.2.6"
                "1ax0jmvsszxd03hj6ga1kyl7gaqcfw0akg2wf0q6gk9pizaffpgh"))

(define rust-fax-derive-0.2.0
  (crate-source "fax_derive" "0.2.0"
                "0zap434zz4xvi5rnysmwzzivig593b4ng15vwzwl7js2nw7s3b50"))

(define rust-fd-lock-3.0.13
  (crate-source "fd-lock" "3.0.13"
                "1df1jdncda67g65hrnmd2zsl7q5hdn8cm84chdalxndsx7akw0zg"))

(define rust-fd-lock-4.0.4
  (crate-source "fd-lock" "4.0.4"
                "0y5a22zaqns06slndm64gjdx983i6b4l4ks895rxznnn4bv2zs8c"))

(define rust-fdeflate-0.3.7
  (crate-source "fdeflate" "0.3.7"
                "130ga18vyxbb5idbgi07njymdaavvk6j08yh1dfarm294ssm6s0y"
                #:snippet '(delete-file-recursively "tests")))

(define rust-feature-probe-0.1.1
  (crate-source "feature-probe" "0.1.1"
                "1nhif9zpr2f17gagf0qb0v914wc3jr9sfjzvnpi7b7pcs73ksnl3"))

(define rust-fern-0.6.2
  (crate-source "fern" "0.6.2"
                "1vpinainw32498p0zydmxc24yd3r6479pmhdfb429mfbji3c3w6r"
                #:snippet '(delete-file-recursively "examples")))

(define rust-fern-0.7.1
  (crate-source "fern" "0.7.1"
                "0a9v59vcq2fgd6bwgbfl7q6b0zzgxn85y6g384z728wvf1gih5j3"
                #:snippet '(delete-file-recursively "examples")))

(define rust-fetch-data-0.2.0
  (crate-source "fetch-data" "0.2.0"
                "1wjjb3l37kdvvq3pv93cxlxxr2pc7kx0ssk2gqispw5bhggnlyda"))

(define rust-ff-0.13.1
  (crate-source "ff" "0.13.1"
                "14v3bc6q24gbcjnxjfbq2dddgf4as2z2gd4mj35gjlrncpxhpdf0"))

(define rust-fiat-crypto-0.2.9
  (crate-source "fiat-crypto" "0.2.9"
                "07c1vknddv3ak7w89n85ik0g34nzzpms6yb845vrjnv9m4csbpi8"))

(define rust-field-offset-0.3.6
  (crate-source "field-offset" "0.3.6"
                "0zq5sssaa2ckmcmxxbly8qgz3sxpb8g1lwv90sdh1z74qif2gqiq"))

(define rust-fiemap-0.1.3
  (crate-source "fiemap" "0.1.3"
                "0nl7ib7nahj7qb5wxzrx8di2bj7jddp5dyqms5fblhikify8gqwj"))

(define rust-figment-0.10.19
  (crate-source "figment" "0.10.19"
                "1ww9sxdzjj2i80w7bq0kllnymjyrfb1cdx2h70ap5wqcdga1rc4c"))

(define rust-figment-file-provider-adapter-0.1.1
  (crate-source "figment_file_provider_adapter" "0.1.1"
                "0553h2yf5nlyhnhkywscmzaq8wih1njhi51h50vzlzkrjq41wgim"))

(define rust-file-id-0.2.2
  (crate-source "file-id" "0.2.2"
                "0dmylm34z6g8cg3b60sc6bk9v5wv9930vyx9wgcdpjpgpfwh9jbb"))

(define rust-file-owner-0.1.2
  (crate-source "file-owner" "0.1.2"
                "0366kwvmgin65dljd856kl7nj98asvzj8m3d93fqmbzzzk53xw1n"))

(define rust-filedescriptor-0.8.2
  (crate-source "filedescriptor" "0.8.2"
                "0vplyh0cw35kzq7smmp2ablq0zsknk5rkvvrywqsqfrchmjxk6bi"))

(define rust-filedescriptor-0.8.3
  (crate-source "filedescriptor" "0.8.3"
                "0bb8qqa9h9sj2mzf09yqxn260qkcqvmhmyrmdjvyxcn94knmh1z4"))

(define rust-filesize-0.2.0
  (crate-source "filesize" "0.2.0"
                "0hvx4dfnara3a2dnhb9ci5bmm1m8s44h9l61s5djwkjx87i43mqj"))

(define rust-filetime-0.1.15
  (crate-source "filetime" "0.1.15"
                "03xishfxzpr4nfz4g3r218d6b6g94rxsqw9pw96m6wa8wgrm6iki"))

(define rust-filetime-0.2.25
  (crate-source "filetime" "0.2.25"
                "11l5zr86n5sr6g6k6sqldswk0jzklm0q95rzikxcns0yk0p55h1m"))

(define rust-filetime-0.2.26
  (crate-source "filetime" "0.2.26"
                "1vb3vz83saxr084wjf2032hspx7wfc5ggggnhc15i9kg3g6ha1dw"))

(define rust-filetime-creation-0.2.0
  (crate-source "filetime_creation" "0.2.0"
                "0prsxylygvwwmycf5jdnwd258ck53iv8800cbgg5kraham3msny2"))

(define rust-filetreelist-0.5.2
  (crate-source "filetreelist" "0.5.2"
                "0j7s58lxg28kq39156n17qbf1iygkn7bxgi2xs99wz706syymyg8"))

(define rust-find-crate-0.6.3
  (crate-source "find-crate" "0.6.3"
                "1ljpkh11gj7940xwz47xjhsvfbl93c2q0ql7l2v0w77amjx8paar"))

(define rust-find-msvc-tools-0.1.0
  (crate-source "find-msvc-tools" "0.1.0"
                "0l4nnivhdigxd87drmahqq99qvz7479ad65syq1njwm2m3xy8y71"))

(define rust-find-msvc-tools-0.1.1
  (crate-source "find-msvc-tools" "0.1.1"
                "0b8rhghgjssjw9q8a3gg7f9kl8zhy9d7nqsc4s4nc52dyqq9knbz"))

(define rust-find-msvc-tools-0.1.2
  (crate-source "find-msvc-tools" "0.1.2"
                "0nbrhvk4m04hviiwbqp2jwcv9j2k70x0q2kcvfk51iygvaqp7v8w"))

(define rust-findshlibs-0.10.2
  (crate-source "findshlibs" "0.10.2"
                "0r3zy2r12rxzwqgz53830bk38r6b7rl8kq2br9n81q7ps2ffbfa0"))

(define rust-finl-unicode-1.3.0
  (crate-source "finl_unicode" "1.3.0"
                "0qy1rwjxkqbl6g8ngm2p33y83r4mbfk3l22075yv6vlh4nsp1jcl"))

(define rust-fixed-decimal-0.5.6
  (crate-source "fixed_decimal" "0.5.6"
                "1f1xqn2fxns768isv4h5vm1yi0j8npbfl825dvpcv7la26qvxsqg"))

(define rust-fixedbitset-0.2.0
  (crate-source "fixedbitset" "0.2.0"
                "0kg03p777wc0dajd9pvlcnsyrwa8dhqwf0sd9r4dw0p82rs39arp"))

(define rust-fixedbitset-0.4.2
  (crate-source "fixedbitset" "0.4.2"
                "101v41amgv5n9h4hcghvrbfk5vrncx1jwm35rn5szv4rk55i7rqc"))

(define rust-fixedbitset-0.5.7
  (crate-source "fixedbitset" "0.5.7"
                "16fd3v9d2cms2vddf9xhlm56sz4j0zgrk3d2h6v1l7hx760lwrqx"))

(define rust-flate2-1.1.0
  (crate-source "flate2" "1.1.0"
                "1p1qpmkkxky6y3869g2facflp0lmvgsbxq4bhkwpm69na9dazyhi"
                #:snippet '(for-each delete-file-recursively '("examples" "tests"))))

(define rust-flate2-1.1.1
  (crate-source "flate2" "1.1.1"
                "1kpycx57dqpkr3vp53b4nq75p9mflh0smxy8hkys4v4ndvkr5vbw"
                #:snippet '(for-each delete-file-recursively '("examples" "tests"))))

(define rust-flate2-1.1.2
  (crate-source "flate2" "1.1.2"
                "07abz7v50lkdr5fjw8zaw2v8gm2vbppc0f7nqm8x3v3gb6wpsgaa"
                #:snippet '(for-each delete-file-recursively '("examples" "tests"))))

(define rust-float-cmp-0.10.0
  (crate-source "float-cmp" "0.10.0"
                "1n760i3nxd2x0zc7fkxkg3vhvdyfbvzngna006cl9s9jacaz775h"))

(define rust-float-cmp-0.9.0
  (crate-source "float-cmp" "0.9.0"
                "1i799ksbq7fj9rm9m82g1yqgm6xi3jnrmylddmqknmksajylpplq"))

(define rust-float-eq-1.0.1
  (crate-source "float_eq" "1.0.1"
                "0lxqxkvdy5zh3qsksavpcazg57cbyjy9p54m16x13bfq8lqhxa18"))

(define rust-fluent-0.16.1
  (crate-source "fluent" "0.16.1"
                "0njmdpwz52yjzyp55iik9k6vrixqiy7190d98pk0rgdy0x3n6x5v"))

(define rust-fluent-bundle-0.15.3
  (crate-source "fluent-bundle" "0.15.3"
                "14zl0cjn361is69pb1zry4k2zzh5nzsfv0iz05wccl00x0ga5q3z"))

(define rust-fluent-langneg-0.13.0
  (crate-source "fluent-langneg" "0.13.0"
                "152yxplc11vmxkslvmaqak9x86xnavnhdqyhrh38ym37jscd0jic"))

(define rust-fluent-syntax-0.11.1
  (crate-source "fluent-syntax" "0.11.1"
                "0gd3cdvsx9ymbb8hijcsc9wyf8h1pbcbpsafg4ldba56ji30qlra"))

(define rust-fluent-uri-0.1.4
  (crate-source "fluent-uri" "0.1.4"
                "03ah2qajw5l1zbc81kh1n8g7n24mfxbg6vqyv9ixipg1vglh9iqp"))

(define rust-fluent-uri-0.3.2
  (crate-source "fluent-uri" "0.3.2"
                "1rgp9mv03bifxysy863kf38aagvyl76ca6gd3dcx6iyzjrfvc60r"))

(define rust-flume-0.10.14
  (crate-source "flume" "0.10.14"
                "0xvm1wpzkjvf99jxy9jp3dxw5nipa9blg7j0ngvxj0rl3i2b8mqn"))

(define rust-flume-0.11.1
  (crate-source "flume" "0.11.1"
                "15ch0slxa8sqsi6c73a0ky6vdnh48q8cxjf7rksa3243m394s3ns"))

(define rust-fnv-1.0.7
  (crate-source "fnv" "1.0.7"
                "1hc2mcqha06aibcaza94vbi81j6pr9a1bbxrxjfhc91zin8yr7iz"))

(define rust-foldhash-0.1.5
  (crate-source "foldhash" "0.1.5"
                "1wisr1xlc2bj7hk4rgkcjkz3j2x4dhd1h9lwk7mj8p71qpdgbi6r"))

(define rust-foldhash-0.2.0
  (crate-source "foldhash" "0.2.0"
                "1nvgylb099s11xpfm1kn2wcsql080nqmnhj1l25bp3r2b35j9kkp"))

(define rust-fontconfig-parser-0.5.8
  (crate-source "fontconfig-parser" "0.5.8"
                "0ijnbzg31sl6v49g7q2l7sl76hjj8z0hvlsz77cdvm029vi77ixv"))

(define rust-fontdb-0.21.0
  (crate-source "fontdb" "0.21.0"
                "1ywa563zfmq501pfhp7qn1zlfj9kyxkmg92prlwf8swn1p19zgip"
                #:snippet '(delete-file-recursively "tests")))

(define rust-foreign-types-0.3.2
  (crate-source "foreign-types" "0.3.2"
                "1cgk0vyd7r45cj769jym4a6s7vwshvd0z4bqrb92q1fwibmkkwzn"))

(define rust-foreign-types-0.5.0
  (crate-source "foreign-types" "0.5.0"
                "0rfr2zfxnx9rz3292z5nyk8qs2iirznn5ff3rd4vgdwza6mdjdyp"))

(define rust-foreign-types-macros-0.2.3
  (crate-source "foreign-types-macros" "0.2.3"
                "0hjpii8ny6l7h7jpns2cp9589016l8mlrpaigcnayjn9bdc6qp0s"))

(define rust-foreign-types-shared-0.1.1
  (crate-source "foreign-types-shared" "0.1.1"
                "0jxgzd04ra4imjv8jgkmdq59kj8fsz6w4zxsbmlai34h26225c00"))

(define rust-foreign-types-shared-0.3.1
  (crate-source "foreign-types-shared" "0.3.1"
                "0nykdvv41a3d4py61bylmlwjhhvdm0b3bcj9vxhqgxaxnp5ik6ma"))

(define rust-forgejo-api-0.4.1
  (crate-source "forgejo-api" "0.4.1"
                "0bc1m39g3i8mm70cf35hcqh898xpm2y4x819y8qfaaw6pnhg90pn"))

(define rust-forgejo-api-0.5.0
  (crate-source "forgejo-api" "0.5.0"
                "114lhnafx34xak9dx7hcsvgdr48bnzn7k32al276h00jl9wjd3ag"))

(define rust-forgejo-api-0.7.0
  (crate-source "forgejo-api" "0.7.0"
                "1l97d6wc9yd7hajzh3kflwc1cyayavhyqd26g3pkdmhm80m71jz2"))

(define rust-fork-0.2.0
  (crate-source "fork" "0.2.0"
                "0vb5557b03ghb1xmn86w9v57k5zjy2g475769xzw58845qq8pp05"))

(define rust-form-urlencoded-1.2.1
  (crate-source "form_urlencoded" "1.2.1"
                "0milh8x7nl4f450s3ddhg57a3flcv6yq8hlkyk6fyr3mcb128dp1"))

(define rust-form-urlencoded-1.2.2
  (crate-source "form_urlencoded" "1.2.2"
                "1kqzb2qn608rxl3dws04zahcklpplkd5r1vpabwga5l50d2v4k6b"))

(define rust-fps-ticker-1.0.0
  (crate-source "fps_ticker" "1.0.0"
                "06cj5c5rk5grm2ajh4sabcppxr1h57gxfqacvi5psxb9zw2lj5py"))

(define rust-fraction-0.15.3
  (crate-source "fraction" "0.15.3"
                "1rss1021dpzr12xl3dyqyj36ji4vv4f83yy9im039cx1y0zqw58g"))

(define rust-fraction-0.13.1
  (crate-source "fraction" "0.13.1"
                "0y6nh9qyfidm6hsp85wf1kv7l7nc9anzvj214bnln6ylz0fsw9rh"))

(define rust-fragile-2.0.1
  (crate-source "fragile" "2.0.1"
                "06g69s9w3hmdnjp5b60ph15v367278mgxy1shijrllarc2pnrp98"))

(define rust-freetype-rs-0.36.0
  (crate-source "freetype-rs" "0.36.0"
                "05rbdl18ks2rb0kz7giwpcv7k1hfg19hbp406l9h95m0dkixwhjl"
                #:snippet '(delete-file-recursively "examples")))

(define rust-freetype-sys-0.20.1
  (crate-source "freetype-sys" "0.20.1"
                "0d5iiv95ap3lwy7b0hxbc8caa9ng1fg3wlwrvb7rld39jrdxqzhf"
                #:snippet '(delete-file-recursively "freetype2")))

(define rust-from-variants-0.6.0
  (crate-source "from_variants" "0.6.0"
                "1qx4jmwljwmcdfc998ndf7iz8wyg7lmlc3vl3fy812f9lfqiw6i2"))

(define rust-from-variants-impl-0.6.0
  (crate-source "from_variants_impl" "0.6.0"
                "06i4bjjxbq6c4hlx2ly04s64d1972zkskshc2v4xx7n8lfghf23y"))

(define rust-fs-at-0.2.1
  (crate-source "fs_at" "0.2.1"
                "0dn0hi9inmppk3mypvnaimjcdrxr0f3pi8d2p8jxn9gajjb6rbql"))

(define rust-fs-err-2.11.0
  (crate-source "fs-err" "2.11.0"
                "0hdajzh5sjvvdjg0n15j91mv8ydvb7ff6m909frvdmg1bw81z948"))

(define rust-fs-err-3.1.0
  (crate-source "fs-err" "3.1.0"
                "1al2sj8src02wwww70vv2gypsrs6wyzx6zlpk82h84m2qajbv28z"))

(define rust-fs-extra-1.3.0
  (crate-source "fs_extra" "1.3.0"
                "075i25z70j2mz9r7i9p9r521y8xdj81q7skslyb7zhqnnw33fw22"))

(define rust-fs-utils-1.1.4
  (crate-source "fs-utils" "1.1.4"
                "14r5wl14mz227v0lpy89lvjzfnxgdxigvrrmm6c4r52w03fakivg"))

(define rust-fs2-0.4.3
  (crate-source "fs2" "0.4.3"
                "04v2hwk7035c088f19mfl5b1lz84gnvv2hv6m935n0hmirszqr4m"))

(define rust-fs4-0.12.0
  (crate-source "fs4" "0.12.0"
                "08gm0p6x133cav6yrcc3qhcr2qr1917yhj0bdx3psm0q8il31762"))

(define rust-fs4-0.8.4
  (crate-source "fs4" "0.8.4"
                "1y0m3pk9iq16b61p3g643234bfy0kdbyjymxczklafy2fsn81qgp"))

(define rust-fsevent-sys-4.1.0
  (crate-source "fsevent-sys" "4.1.0"
                "1liz67v8b0gcs8r31vxkvm2jzgl9p14i78yfqx81c8sdv817mvkn"))

(define rust-fsio-0.1.3
  (crate-source "fsio" "0.1.3"
                "1qs9i8blmbaxjl3csibn5sf0w27qf8dz2sgamvqz8fgpamr0izf1"
                #:snippet '(delete-file-recursively "docs")))

(define rust-fsio-0.4.1
  (crate-source "fsio" "0.4.1"
                "1ci5ixcc4z7ik2kxwal8a9gjgfv7d1wb91lrnyrb81baxcb4z57l"))

(define rust-fuchsia-cprng-0.1.1
  (crate-source "fuchsia-cprng" "0.1.1"
                "1fnkqrbz7ixxzsb04bsz9p0zzazanma8znfdqjvh39n14vapfvx0"))

(define rust-fuchsia-zircon-0.3.3
  (crate-source "fuchsia-zircon" "0.3.3"
                "10jxc5ks1x06gpd0xg51kcjrxr35nj6qhx2zlc5n7bmskv3675rf"))

(define rust-fuchsia-zircon-sys-0.3.3
  (crate-source "fuchsia-zircon-sys" "0.3.3"
                "19zp2085qsyq2bh1gvcxq1lb8w6v6jj9kbdkhpdjrl95fypakjix"))

(define rust-funty-2.0.0
  (crate-source "funty" "2.0.0"
                "177w048bm0046qlzvp33ag3ghqkqw4ncpzcm5lq36gxf2lla7mg6"))

(define rust-futf-0.1.5
  (crate-source "futf" "0.1.5"
                "0hvqk2r7v4fnc34hvc3vkri89gn52d5m9ihygmwn75l1hhp0whnz"))

(define rust-future-queue-0.4.0
  (crate-source "future-queue" "0.4.0"
                "1jx8cnrmjlsxiibfhd6cg8xf7pr4f7rxnisw3vx8z07lxskz9ka7"))

(define rust-futures-0.1.31
  (crate-source "futures" "0.1.31"
                "0y46qbmhi37dqkch8dlfq5aninqpzqgrr98awkb3rn4fxww1lirs"))

(define rust-futures-0.3.31
  (crate-source "futures" "0.3.31"
                "0xh8ddbkm9jy8kc5gbvjp9a4b6rqqxvc8471yb2qaz5wm2qhgg35"))

(define rust-futures-channel-0.3.28
  (crate-source "futures-channel" "0.3.28"
                "1wmm9wm5zjigxz61qkscmxp7c30zp08dy63spjz5pch9gva1hmcm"))

(define rust-futures-channel-0.3.31
  (crate-source "futures-channel" "0.3.31"
                "040vpqpqlbk099razq8lyn74m0f161zd0rp36hciqrwcg2zibzrd"))

(define rust-futures-core-0.3.28
  (crate-source "futures-core" "0.3.28"
                "137fdxy5amg9zkpa1kqnj7bnha6b94fmddz59w973x96gqxmijjb"))

(define rust-futures-core-0.3.31
  (crate-source "futures-core" "0.3.31"
                "0gk6yrxgi5ihfanm2y431jadrll00n5ifhnpx090c2f2q1cr1wh5"))

(define rust-futures-executor-0.3.28
  (crate-source "futures-executor" "0.3.28"
                "1q468di96knnla72xdvswic1ir2qkrf5czsdigc5n4l86a1fxv6c"))

(define rust-futures-executor-0.3.31
  (crate-source "futures-executor" "0.3.31"
                "17vcci6mdfzx4gbk0wx64chr2f13wwwpvyf3xd5fb1gmjzcx2a0y"))

(define rust-futures-intrusive-0.5.0
  (crate-source "futures-intrusive" "0.5.0"
                "0vwm08d1pli6bdaj0i7xhk3476qlx4pll6i0w03gzdnh7lh0r4qx"))

(define rust-futures-io-0.3.28
  (crate-source "futures-io" "0.3.28"
                "0r4rhkdhq1my4fahlhz59barqa511bylq813w3w4gvbidq4p9zsg"))

(define rust-futures-io-0.3.31
  (crate-source "futures-io" "0.3.31"
                "1ikmw1yfbgvsychmsihdkwa8a1knank2d9a8dk01mbjar9w1np4y"))

(define rust-futures-lite-1.13.0
  (crate-source "futures-lite" "1.13.0"
                "1kkbqhaib68nzmys2dc8j9fl2bwzf2s91jfk13lb2q3nwhfdbaa9"))

(define rust-futures-lite-2.6.0
  (crate-source "futures-lite" "2.6.0"
                "0cmmgszlmkwsac9pyw5rfjakmshgx4wmzmlyn6mmjs0jav4axvgm"))

(define rust-futures-lite-2.6.1
  (crate-source "futures-lite" "2.6.1"
                "1ba4dg26sc168vf60b1a23dv1d8rcf3v3ykz2psb7q70kxh113pp"))

(define rust-futures-macro-0.3.28
  (crate-source "futures-macro" "0.3.28"
                "0wpfsqxwqk5k569xl0jzz4zxy85x695mndf7y9jn66q6jid59jl9"))

(define rust-futures-macro-0.3.31
  (crate-source "futures-macro" "0.3.31"
                "0l1n7kqzwwmgiznn0ywdc5i24z72zvh9q1dwps54mimppi7f6bhn"))

(define rust-futures-sink-0.3.31
  (crate-source "futures-sink" "0.3.31"
                "1xyly6naq6aqm52d5rh236snm08kw8zadydwqz8bip70s6vzlxg5"))

(define rust-futures-task-0.3.28
  (crate-source "futures-task" "0.3.28"
                "0ravgihyarbplj32zp60asirfnaalw2wfsa0afhnl3kcpqrd3lvn"))

(define rust-futures-task-0.3.31
  (crate-source "futures-task" "0.3.31"
                "124rv4n90f5xwfsm9qw6y99755y021cmi5dhzh253s920z77s3zr"))

(define rust-futures-timer-3.0.3
  (crate-source "futures-timer" "3.0.3"
                "094vw8k37djpbwv74bwf2qb7n6v6ghif4myss6smd6hgyajb127j"))

(define rust-futures-util-0.3.28
  (crate-source "futures-util" "0.3.28"
                "0cwmls9369w6q6hwlbm10q0plr6hmg8w28fpqvv4rmbjnx01xc16"))

(define rust-futures-util-0.3.31
  (crate-source "futures-util" "0.3.31"
                "10aa1ar8bgkgbr4wzxlidkqkcxf77gffyj8j7768h831pcaq784z"))

(define rust-fuzzy-matcher-0.3.7
  (crate-source "fuzzy-matcher" "0.3.7"
                "153csv8rsk2vxagb68kpmiknvdd3bzqj03x805khckck28rllqal"))

(define rust-fxhash-0.2.1
  (crate-source "fxhash" "0.2.1"
                "037mb9ichariqi45xm6mz0b11pa92gj38ba0409z3iz239sns6y3"))

(define rust-gbm-0.18.0
  (crate-source "gbm" "0.18.0"
                "0skyaj51xlazaa24jdkxxi2g6pnw834k3yqlf2ly999wincjx1ff"))

(define rust-gbm-sys-0.4.0
  (crate-source "gbm-sys" "0.4.0"
                "0vzp28ip4w74p05ygs4p9m7sspggn2zvcykbpyv8ypbqrhm5yfn1"))

(define rust-gdk-0.12.1
  (crate-source "gdk" "0.12.1"
                "12bmk9vfpk7f03fx22cq9ps00xylsxcpmp8c8r95r1n05xvyirgv"))

(define rust-gdk-pixbuf-0.18.0
  (crate-source "gdk-pixbuf" "0.18.0"
                "1wy68zaxd1914qayjz4jn6k0i2759sx7k26hcmmma7d8fgnw5jdv"))

(define rust-gdk-pixbuf-0.18.5
  (crate-source "gdk-pixbuf" "0.18.5"
                "1v7svvl0g7zybndmis5inaqqgi1mvcc6s1n8rkb31f5zn3qzbqah"))

(define rust-gdk-pixbuf-0.19.8
  (crate-source "gdk-pixbuf" "0.19.8"
                "16c6kznkh3vi82843ays8awdm37fwjd1fblv6g3h64824shsnkk2"))

(define rust-gdk-pixbuf-0.20.10
  (crate-source "gdk-pixbuf" "0.20.10"
                "0371cfhxldrn2pf8zdjyx2b3xkhbfm96k988spp4nkq89j4l5lig"))

(define rust-gdk-pixbuf-0.8.0
  (crate-source "gdk-pixbuf" "0.8.0"
                "1mxxca0fkcw2rsd3kl3nvlb8ys4cgxqx4n5isjbv0adk8q624j72"))

(define rust-gdk-pixbuf-sys-0.10.0
  (crate-source "gdk-pixbuf-sys" "0.10.0"
                "13qyxazj9bmw3llvkh6br8v7sypnbin2nxis366ppsa3gy54dziv"))

(define rust-gdk-pixbuf-sys-0.18.0
  (crate-source "gdk-pixbuf-sys" "0.18.0"
                "1xya543c4ffd2n7aiwwrdxsyc9casdbasafi6ixcknafckm3k61z"))

(define rust-gdk-pixbuf-sys-0.19.8
  (crate-source "gdk-pixbuf-sys" "0.19.8"
                "0y93g24mdgskvyhva46xv3qyb1cvj5xpi0yqnh7cb31wz2j0byjf"))

(define rust-gdk-pixbuf-sys-0.20.10
  (crate-source "gdk-pixbuf-sys" "0.20.10"
                "15hb2f5mmyg5amaha6lx6spaygw2b7ga4hwmgqhvv269h2sz6d2v"))

(define rust-gdk-pixbuf-sys-0.9.1
  (crate-source "gdk-pixbuf-sys" "0.9.1"
                "1c2andpjb10y7bahh6nxnksh9m3g5qh4mgq9znx634cy1831p6fq"))

(define rust-gdk-sys-0.10.0
  (crate-source "gdk-sys" "0.10.0"
                "0s8d2jymffbv2kpwrx53krf7mpy3vdfhbb5i2n02dz80qp7m75ha"))

(define rust-gdk-sys-0.9.1
  (crate-source "gdk-sys" "0.9.1"
                "1fwgr1b3n0khlkhdq9145m6baz9y8207ya30d30g1gyij6g6gpva"))

(define rust-gdk4-0.7.3
  (crate-source "gdk4" "0.7.3"
                "1xiacc63p73apr033gjrb9dsk0y4yxnsljwfxbwfry41snd03nvy"))

(define rust-gdk4-0.9.6
  (crate-source "gdk4" "0.9.6"
                "0q1dld01fgj7qxj644by0fc242mcn36w3bagn4z1mkdfq7cwjl28"))

(define rust-gdk4-sys-0.7.2
  (crate-source "gdk4-sys" "0.7.2"
                "1w7yvir565sjrrw828lss07749hfpfsr19jdjzwivkx36brl7ayv"))

(define rust-gdk4-sys-0.9.6
  (crate-source "gdk4-sys" "0.9.6"
                "0fj722lp86fpa1b1i3s2anavdmcpybd0b47mkhknzd72k1bvjvkg"))

(define rust-gdk4-wayland-0.7.2
  (crate-source "gdk4-wayland" "0.7.2"
                "04zkspjs1r6l4gj241p9xm2zmp91phm1khakw5jvsm8yy4pi1f8d"))

(define rust-gdk4-wayland-sys-0.7.2
  (crate-source "gdk4-wayland-sys" "0.7.2"
                "092nbn4gk82kbdvji2qnqy181l4pf5i8961bb8nj3q3a4nz5k0fl"))

(define rust-gdk4-win32-0.7.2
  (crate-source "gdk4-win32" "0.7.2"
                "0mv04mipl57v1lj94j2rkrk9qm75jvdlnp7qm6nl2kpn8466arpy"))

(define rust-gdk4-win32-sys-0.7.2
  (crate-source "gdk4-win32-sys" "0.7.2"
                "0v9rkv1i1sbzckigdr5zjy0xzz6qf9iccikvg3qxjfnd8rsihp2b"))

(define rust-gdk4-x11-0.7.2
  (crate-source "gdk4-x11" "0.7.2"
                "0l54c1m0gsdm07drvy171a0i97ic2kygmzf3fjg4da0yxbwbpj98"))

(define rust-gdk4-x11-sys-0.7.2
  (crate-source "gdk4-x11-sys" "0.7.2"
                "09gill32x6qy4s55xjckqvqrfxw4jfjrlcpmd4iijn076w4igpm3"))

(define rust-generational-arena-0.2.9
  (crate-source "generational-arena" "0.2.9"
                "1rwnfyprjwqafkwdz2irkds5a41jcjb3bsma3djknx4fy2pr8zl7"))

(define rust-generator-0.7.5
  (crate-source "generator" "0.7.5"
                "0knp7cllkf90l4wmfbhmb3c0ha9xn9afrzmphain1d12zy26bhaw"))

(define rust-generator-0.8.4
  (crate-source "generator" "0.8.4"
                "1p9qqk9nzarjdcl5fr4iylvsv446g0svlpk63lxis4ysrqad2syc"))

(define rust-generator-0.8.5
  (crate-source "generator" "0.8.5"
                "09v88ic14jypd6sl1g3bqcp2a07r2h4lgxzidisgzy5pdjkp116i"))

(define rust-generator-0.8.7
  (crate-source "generator" "0.8.7"
                "18p5rjx7vbcz6jw6c6qdzpl2wkhr5mfcqdc60g0skqp372jq6lb0"))

(define rust-generic-array-0.12.4
  (crate-source "generic-array" "0.12.4"
                "1gfpay78vijl9vrwl1k9v7fbvbhkhcmnrk4kfg9l6x24y4s9zpzz"))

(define rust-generic-array-0.13.3
  (crate-source "generic-array" "0.13.3"
                "02g3zhqc086zmsb6kcmjs2fiprz8gq12g0xbm9g23215ydxfd5zp"))

(define rust-generic-array-0.14.7
  (crate-source "generic-array" "0.14.7"
                "16lyyrzrljfq424c3n8kfwkqihlimmsg5nhshbbp48np3yjrqr45"))

(define rust-generic-array-1.2.0
  (crate-source "generic-array" "1.2.0"
                "00qpb5rxfiv5mvdb2rvqmw40h612ysvwl36car8r66ypr55l9j78"))

(define rust-genetlink-0.2.5
  (crate-source "genetlink" "0.2.5"
                "17q7zlmp95fx28zjkzh69g6d5q6rm2a3dkj7yy5jk8gsq5v0128z"))

(define rust-getch-0.3.1
  (crate-source "getch" "0.3.1"
                "1a0v3ck9w65zcgg055laiyyvkb6lxaph1h3zvw6pgq99bcnhx68k"))

(define rust-gethostname-0.2.3
  (crate-source "gethostname" "0.2.3"
                "0bl6wf7zclzmjriipzh98rr84xv2ilj664z8ffxh0vn46m7d7sy1"))

(define rust-gethostname-0.3.0
  (crate-source "gethostname" "0.3.0"
                "0xy1wbx5k2bzi4cbaqj9wqgqsbn4f8pm6nsm1d86mibk66xd8rdv"))

(define rust-gethostname-0.4.3
  (crate-source "gethostname" "0.4.3"
                "063qqhznyckwx9n4z4xrmdv10s0fi6kbr17r6bi1yjifki2y0xh1"))

(define rust-gethostname-0.5.0
  (crate-source "gethostname" "0.5.0"
                "0c0y7mc8bpsjkvvykqcic9cynskvm82iz4fn4335pmhqd2m5adnw"))

(define rust-gethostname-1.0.1
  (crate-source "gethostname" "1.0.1"
                "17l5hgpfkjl3wbs2vbnwk2bwm8b8cvvkcrhf7r8n7pmvgbjk2wgd"))

(define rust-gethostname-1.0.2
  (crate-source "gethostname" "1.0.2"
                "0mdfkmfr41xx1i0nlwgzncmnj2a5f18kn6ydp7j1qc1q83dpy9gw"))

(define rust-getopts-0.2.21
  (crate-source "getopts" "0.2.21"
                "1mgb3qvivi26gs6ihqqhh8iyhp3vgxri6vwyrwg28w0xqzavznql"))

(define rust-getopts-0.2.23
  (crate-source "getopts" "0.2.23"
                "1ha8a3l3w68yrw3qjfzj0pak0rppf1yghign03iri1llxdisx9nb"))

(define rust-getopts-0.2.24
  (crate-source "getopts" "0.2.24"
                "1pylvsmq7fillnxmd6g58r7igdrlby412q37ws41z39va2ngpr6g"))

(define rust-getrandom-0.1.16
  (crate-source "getrandom" "0.1.16"
                "1kjzmz60qx9mn615ks1akjbf36n3lkv27zfwbcam0fzmj56wphwg"))

(define rust-getrandom-0.2.10
  (crate-source "getrandom" "0.2.10"
                "09zlimhhskzf7cmgcszix05wyz2i6fcpvh711cv1klsxl6r3chdy"))

(define rust-getrandom-0.2.15
  (crate-source "getrandom" "0.2.15"
                "1mzlnrb3dgyd1fb84gvw10pyr8wdqdl4ry4sr64i1s8an66pqmn4"))

(define rust-getrandom-0.2.16
  (crate-source "getrandom" "0.2.16"
                "14l5aaia20cc6cc08xdlhrzmfcylmrnprwnna20lqf746pqzjprk"))

(define rust-getrandom-0.3.1
  (crate-source "getrandom" "0.3.1"
                "1y154yzby383p63ndw6zpfm0fz3vf6c0zdwc7df6vkl150wrr923"))

(define rust-getrandom-0.3.2
  (crate-source "getrandom" "0.3.2"
                "1w2mlixa1989v7czr68iji7h67yra2pbg3s480wsqjza1r2sizkk"))

(define rust-getrandom-0.3.3
  (crate-source "getrandom" "0.3.3"
                "1x6jl875zp6b2b6qp9ghc84b0l76bvng2lvm8zfcmwjl7rb5w516"))

(define rust-getset-0.0.9
  (crate-source "getset" "0.0.9"
                "0aaldwfs2690rjqg2ygan27l2qa614w2p6zj7k99n36pv2vzbcsv"))

(define rust-getset-0.1.5
  (crate-source "getset" "0.1.5"
                "1zpap947rb3rw9xlp7v37hs4zsykwdrdlcxp9qh8gpric4jnyn7k"))

(define rust-gettext-rs-0.7.2
  (crate-source "gettext-rs" "0.7.2"
                "12ikrzvx35aybip55ib9zmnjf8is4mhy2pfmgv50lhq8vkvr4km4"))

(define rust-gettext-sys-0.22.5
  (crate-source "gettext-sys" "0.22.5"
                "0qgnkr3gaf6pndai26ysmhyr8kwn8psmhifhxhmg2ic9bczpfidv"
                #:snippet '(delete-file "gettext-0.22.5.tar.xz")))

(define rust-gh-emoji-1.0.8
  (crate-source "gh-emoji" "1.0.8"
                "0dwc7kr1dsb4bn8n87x7363c2c6vc4pi9fy6s74nk0r8vxhxz2pb"))

(define rust-ghash-0.5.1
  (crate-source "ghash" "0.5.1"
                "1wbg4vdgzwhkpkclz1g6bs4r5x984w5gnlsj4q5wnafb5hva9n7h"))

(define rust-gif-0.13.1
  (crate-source "gif" "0.13.1"
                "1whrkvdg26gp1r7f95c6800y6ijqw5y0z8rgj6xihpi136dxdciz"))

(define rust-gif-0.13.3
  (crate-source "gif" "0.13.3"
                "06z6gll24q7psbz9fb86jbcbmgwnxkym8jsp0fbq5qikbqilgq2a"))

(define rust-gimli-0.26.2
  (crate-source "gimli" "0.26.2"
                "0pafbk64rznibgnvfidhm1pqxd14a5s9m50yvsgnbv38b8n0w0r2"
                #:snippet '(delete-file-recursively "fixtures")))

(define rust-gimli-0.28.1
  (crate-source "gimli" "0.28.1"
                "0lv23wc8rxvmjia3mcxc6hj9vkqnv1bqq0h8nzjcgf71mrxx6wa2"))

(define rust-gimli-0.29.0
  (crate-source "gimli" "0.29.0"
                "1zgzprnjaawmg6zyic4f2q2hc39kdhn116qnkqpgvsasgc3x9v20"))

(define rust-gimli-0.31.1
  (crate-source "gimli" "0.31.1"
                "0gvqc0ramx8szv76jhfd4dms0zyamvlg4whhiz11j34hh3dqxqh7"))

(define rust-gimli-0.32.3
  (crate-source "gimli" "0.32.3"
                "1iqk5xznimn5bfa8jy4h7pa1dv3c624hzgd2dkz8mpgkiswvjag6"))

(define rust-gio-0.18.2
  (crate-source "gio" "0.18.2"
                "1caryyr1sciqd25np4v4701nkb7h59gqzpwaiqjrp6g5x222y1ap"))

(define rust-gio-0.18.4
  (crate-source "gio" "0.18.4"
                "0wsc6mnx057s4ailacg99dwgna38dbqli5x7a6y9rdw75x9qzz6l"))

(define rust-gio-0.19.8
  (crate-source "gio" "0.19.8"
                "1znz5ngfvv3gbndf6lzz3hs27hlb8ysls4axlfccrzvkscbz2jac"))

(define rust-gio-0.20.11
  (crate-source "gio" "0.20.11"
                "0iz77wp566viggmhkbgpvw7740zcn83pgf0d28awp52pky1c79fj"))

(define rust-gio-0.20.12
  (crate-source "gio" "0.20.12"
                "0cdq5116cwdgs0xkdp1v146yhcqilxlpgvkncc7xbf5nwxvf49wf"))

(define rust-gio-0.8.1
  (crate-source "gio" "0.8.1"
                "19cnla2ya0mi6wwaabd5mxbq2kzq46dg6jq2z19rpqyc2na0zl8c"))

(define rust-gio-sys-0.10.1
  (crate-source "gio-sys" "0.10.1"
                "1b2f6jvghm2djymj3qrgbqfci2f95gyhdin2pgv2qpcg5xszn92y"))

(define rust-gio-sys-0.18.1
 (crate-source "gio-sys" "0.18.1"
                "1lip8z35iy9d184x2qwjxlbxi64q9cpayy7v1p5y9xdsa3w6smip"))

(define rust-gio-sys-0.19.8
  (crate-source "gio-sys" "0.19.8"
                "1vylsskpipfwl7mvffp1s0227d0k5amyhd32dfnp3mhl8yx47mrc"))

(define rust-gio-sys-0.20.10
  (crate-source "gio-sys" "0.20.10"
                "10vc6gqhz5crnrh040rv6r5nm09njky2r9d9ms29xj3gwnkr67jj"))

(define rust-gio-sys-0.9.1
  (crate-source "gio-sys" "0.9.1"
                "049rafihxp3maxg3fwj6062ni9dcfilvhfq6ibnfgsmr89925bag"))

(define rust-git-ref-format-0.3.1
  (crate-source "git-ref-format" "0.3.1"
                "18ax2jngmx58kh8rxdwm28rjq57ml0wvh6ghsq9sdaa9wpbf0a3l"))

(define rust-git-ref-format-core-0.3.1
  (crate-source "git-ref-format-core" "0.3.1"
                "091l2h5k8janc5qsx7s9gnwmmci6flg7ivyknqnf7sam59kvkbmv"))

(define rust-git-ref-format-macro-0.3.1
  (crate-source "git-ref-format-macro" "0.3.1"
                "193cwdi7bw38m1xiibsddma9j036li5p9zsd6bv03hnc78ssav1v"))

(define rust-git-version-0.3.9
  (crate-source "git-version" "0.3.9"
                "06ddi3px6l2ip0srn8512bsh8wrx4rzi65piya0vrz5h7nm6im8s"))

(define rust-git-version-macro-0.3.9
  (crate-source "git-version-macro" "0.3.9"
                "1h1s08fgh9bkwnc2hmjxcldv69hlxpq7a09cqdxsd5hb235hq0ak"))

(define rust-git2-0.13.25
  (crate-source "git2" "0.13.25"
                "1mppxyjzi69m879mwpin4d9jljanwaijlx3f5w3fdh143g62k4pj"))

(define rust-git2-0.18.3
  (crate-source "git2" "0.18.3"
                "0w7gcq6v9kdlh0vcv27xrk09c1bhkarqhnp52pvnnximzrxnlbi3"))

(define rust-git2-0.19.0
  (crate-source "git2" "0.19.0"
                "091pv7866z1qjq800ys0wjv8n73wrv7fqdrddxcnq36w8lzbf0xr"))

(define rust-git2-0.20.0
  (crate-source "git2" "0.20.0"
                "1zwav0r76njd9chqxh7wj4r4zfn08nzsisrg05liyd6cjf4piniz"))

(define rust-git2-0.20.1
  (crate-source "git2" "0.20.1"
                "1fgf67h78yrw2gm1n8ghgr0jwsbkvmjfhnbng9zrm2n68jxbh82j"))

(define rust-git2-0.20.2
  (crate-source "git2" "0.20.2"
                "0451zzmvblvlrj6y6pgdsxrqh42hi789n3k9lp0hslmi6fhhgsrd"))

(define rust-git2-curl-0.20.0
  (crate-source "git2-curl" "0.20.0"
                "17q7p4xdmvzn8jy75cdpl6bncy70z1v864wv0ch2690wg9919zv8"))

(define rust-git2-hooks-0.4.0
  (crate-source "git2-hooks" "0.4.0"
                "06c60gya0kg5v5h8nzap1gl3yvk36q3y99qdii0mbssnx5ghz4rg"))

(define rust-gitoxide-core-0.46.0
  (crate-source "gitoxide-core" "0.46.0"
                "1dygrdaizkwjm4ly7v5v7mj1wamhs51j3mpzga9gdq6hym6afn9p"))

(define rust-gix-0.64.0
  (crate-source "gix" "0.64.0"
                "0kv3w87h8gbwg16k7nqzalb417v8yzh7fq0nh28350nckz91916p"))

(define rust-gix-0.66.0
  (crate-source "gix" "0.66.0"
                "1swmi6l1ph9rqyzqd4v04apxima9zi8c9r9prd2z0111mv8vhj4h"))

(define rust-gix-0.67.0
  (crate-source "gix" "0.67.0"
                "0zziyg31w7knv6cdizhm3fgxi8xg5ay64a5wpzix6s63va6ygly7"))

(define rust-gix-0.69.1
  (crate-source "gix" "0.69.1"
                "0gimmnxcrnz0icw089pf93gg93059xgghdis8gahbx6wxkdfn3ld"))

(define rust-gix-0.70.0
  (crate-source "gix" "0.70.0"
                "0s3b5407lqx9nf81xfrmka6l269551kkwm9blmpabwq5cxii8vvk"))

(define rust-gix-0.71.0
  (crate-source "gix" "0.71.0"
                "0dblc4j5bhs973y04zp2sah6xvjizqn6h4pq2agkrz0pd3n727m6"))

(define rust-gix-0.72.1
  (crate-source "gix" "0.72.1"
                "1s6wh2i1wg7apfg5v70w4mf2n2wcmv1b1s1bciqiyn3q7n6pw8q1"))

(define rust-gix-0.73.0
  (crate-source "gix" "0.73.0"
                "0dgnfpp5by749kgnjv5qks055cj9b9c0bhnbn2305p4vhz62jk2i"))

(define rust-gix-actor-0.20.0
  (crate-source "gix-actor" "0.20.0"
                "1ww4ixdc5ssm94fz091mkr5sg63glj11qscmixisikhh287zm3l4"))

(define rust-gix-actor-0.31.5
  (crate-source "gix-actor" "0.31.5"
                "1wm3i9g69hkfhwiw1c4z9fr1hkfxsfxyzdh07b637f1lgqsm9r50"))

(define rust-gix-actor-0.32.0
  (crate-source "gix-actor" "0.32.0"
                "0rc662dyhxipvz3c181h7rif3y627lb0ky83s1nadi25rl9f66gw"))

(define rust-gix-actor-0.33.2
  (crate-source "gix-actor" "0.33.2"
                "1cp47vxcd7f7nf225spdhncqqsrcjcf5qc68zkqnbq1jccd8l090"))

(define rust-gix-actor-0.34.0
  (crate-source "gix-actor" "0.34.0"
                "1rr05a5nfwba8nl5zg75ng7j762nn658vfl25ywa9b1881ywhf7l"))

(define rust-gix-actor-0.35.1
  (crate-source "gix-actor" "0.35.1"
                "1zz1r1swaa3fqmy0zk8gvh0fsb0rmg5b1qp55nyzdwri9xp0wc3b"))

(define rust-gix-actor-0.35.4
  (crate-source "gix-actor" "0.35.4"
                "0rq9506midd6wzwk42kn9q92cja1z3gwwczs5nqm2az3xzwxqdid"))

(define rust-gix-archive-0.20.0
  (crate-source "gix-archive" "0.20.0"
                "0vhsspf73j5mkhv0vkyk0ndscipamr6lcv8dl9kqv5dqv8caghwv"))

(define rust-gix-attributes-0.22.5
  (crate-source "gix-attributes" "0.22.5"
                "048k2c9kgh6bv7d1d2iyca8cmvb9mw090jjnab9kv5sam8jvzk7b"))

(define rust-gix-attributes-0.23.1
  (crate-source "gix-attributes" "0.23.1"
                "1p6a6ai3pk8c7xn48vbw7gvjh7rc5m13cbcsd7zfvh4l462vzyfx"))

(define rust-gix-attributes-0.24.0
  (crate-source "gix-attributes" "0.24.0"
                "0f6vdp77d5z98bv3w6i71zlaqcgf8bch4qfa3rj5zvv2yq5h0lgi"))

(define rust-gix-attributes-0.25.0
  (crate-source "gix-attributes" "0.25.0"
                "0zyxr1257fzfvi79109z2dmakm40cznmrdpqjq8a22j3w0jmiqp4"))

(define rust-gix-attributes-0.26.0
  (crate-source "gix-attributes" "0.26.0"
                "13lb0i0dkwpf8pmd9ipjigvjdqrfm3s383cqnrdv5pc0q8x6pqp7"))

(define rust-gix-attributes-0.27.0
  (crate-source "gix-attributes" "0.27.0"
                "0f7n6mqwvc9k6s3w7l6jsx84l90zrdczd5gikaasa23d46422i25"))

(define rust-gix-bitmap-0.2.14
  (crate-source "gix-bitmap" "0.2.14"
                "0h3msc00gi2vr2k4q41ddb68qprbvkih824glq6na0lmqrjrgnxi"))

(define rust-gix-blame-0.1.0
  (crate-source "gix-blame" "0.1.0"
                "1in781xvvc12cvlrhcqnj723sfjmjqxiblf2hz8bd76f26hmlpdj"))

(define rust-gix-chunk-0.4.11
  (crate-source "gix-chunk" "0.4.11"
                "0vxxq4q5pn5cz2xhghcjpp8z83r8xxy74gsffvf9k1lmcj3is7qb"))

(define rust-gix-command-0.3.11
  (crate-source "gix-command" "0.3.11"
                "0lzyg587s4rcrlvi42ml744ardqy6l5vh7hrx3bkyib47a7nnzbd"))

(define rust-gix-command-0.4.1
  (crate-source "gix-command" "0.4.1"
                "1wcdm6f8v28y2rv5lmz7kh4lnkdzplc92nh2c9gb8papss20nhfb"))

(define rust-gix-command-0.5.0
  (crate-source "gix-command" "0.5.0"
                "1y6yfasb60507mxmpq2bwqpldwzccclzwmz1z2bscwvphjaqjdy0"))

(define rust-gix-command-0.6.0
  (crate-source "gix-command" "0.6.0"
                "0a7vm42s486330qmasf95pa3gagj1q1y3q78c5068cxsnhzpzx6j"))

(define rust-gix-command-0.6.2
  (crate-source "gix-command" "0.6.2"
                "15mibs0na7rpnk41lj3bz4sijrhc998pmchjcgl2ldcalifbccbb"))

(define rust-gix-commitgraph-0.24.3
  (crate-source "gix-commitgraph" "0.24.3"
                "0y7wc0y0xb0kh3c22pj3ns04zdqglqb22gj71kn3cn2ngzv0cfqk"))

(define rust-gix-commitgraph-0.25.1
  (crate-source "gix-commitgraph" "0.25.1"
                "11cdlkbkv80imbdkiy8k09gb1c48k6qadpmxvavb53w6ly8nbnm8"))

(define rust-gix-commitgraph-0.26.0
  (crate-source "gix-commitgraph" "0.26.0"
                "0xs85svhri8b40paa3zjjxfqzl6g3ganxnxg1nhjcq51v318wfp2"))

(define rust-gix-commitgraph-0.27.0
  (crate-source "gix-commitgraph" "0.27.0"
                "1zi819j9ska52pv82rz2q5d36csqq6vkqpwpvd852l57nx4vwg04"))

(define rust-gix-commitgraph-0.28.0
  (crate-source "gix-commitgraph" "0.28.0"
                "0lbr29n2s1bfcar9q4rx5l3bw85mnf2r8zxxwcgp6v5adkym0l70"))

(define rust-gix-commitgraph-0.29.0
  (crate-source "gix-commitgraph" "0.29.0"
                "01hdk0r2f91na13i1937fhllgf3c6f80is730xdkmx2jx4hk3ckb"))

(define rust-gix-config-0.22.0
  (crate-source "gix-config" "0.22.0"
                "1haakps85dh5sw8h84a3vakkb279kzf7521x1mh79pxnvl72l98x"))

(define rust-gix-config-0.38.0
  (crate-source "gix-config" "0.38.0"
                "0n768fy37w3wy8jsjk34draw8fcljq44yrf8qay9xw0v7p83zx98"))

(define rust-gix-config-0.40.0
  (crate-source "gix-config" "0.40.0"
                "1c1bnijxf1j4qfqriw9kzf1g40i2yys324fyj4j5b8vcgr49grvq"))

(define rust-gix-config-0.41.0
  (crate-source "gix-config" "0.41.0"
                "0pj4mijnx46s2lq1sw78w82nq0brvvhfh1vjspllp6bv3jzx3v8b"))

(define rust-gix-config-0.42.0
  (crate-source "gix-config" "0.42.0"
                "06bgc1isbk831763ii4f1yahf7r3id301ksri4acp68zr83b8jb6"))

(define rust-gix-config-0.43.0
  (crate-source "gix-config" "0.43.0"
                "1sfry54k4f35ar6y0d7n52ccwyq9r192kkdkw1lx9m8l43yiwz1p"))

(define rust-gix-config-0.44.0
  (crate-source "gix-config" "0.44.0"
                "1ylcdqqkm7ipx4m19x69mnfc3ljmyvvsnj8v4r04jq26yw5q6vww"))

(define rust-gix-config-0.45.1
  (crate-source "gix-config" "0.45.1"
                "10fqy308ywf32q8mwj5f4k4wyn0g074jwg29fzxrn15fazrwiws8"))

(define rust-gix-config-0.46.0
  (crate-source "gix-config" "0.46.0"
                "1h1nn76xdsmmyzf68q3h8xfm8rasa8wbc2iwzjnd8pv9bf68kysx"))

(define rust-gix-config-value-0.12.5
  (crate-source "gix-config-value" "0.12.5"
                "15rqyj523ckas16sn0jbqpgzln4h1fcpdsnwj4lw0hbl8d0lz1vf"))

(define rust-gix-config-value-0.14.12
  (crate-source "gix-config-value" "0.14.12"
                "1dj4g52s18ab01pnw55rd0qdf7frdxryzawccy21h56gqi2cihld"))

(define rust-gix-config-value-0.15.0
  (crate-source "gix-config-value" "0.15.0"
                "1dw56b84kbcbnk3fpic4y3czjp17a7jspvmzapyxzqns87i657a3"))

(define rust-gix-config-value-0.15.1
  (crate-source "gix-config-value" "0.15.1"
                "02akr1nnljjyqszimafjwp9lssa7xjgn95pwq7367qk7xc1jf0cz"))

(define rust-gix-credentials-0.24.5
  (crate-source "gix-credentials" "0.24.5"
                "0y6f5g8ny3rh80vw12qxzzvisw6588yll71hmvqq51wn0p9r3qwc"))

(define rust-gix-credentials-0.25.1
  (crate-source "gix-credentials" "0.25.1"
                "0wdfnq6y3za7h1xqj32af84zdzwg0r2irxrf0gkydiszd2w7ps1b"))

(define rust-gix-credentials-0.26.0
  (crate-source "gix-credentials" "0.26.0"
                "02iy6pnvmd04jma303fszv922kpyhd5ggk5l38armhl5nxb0r9c2"))

(define rust-gix-credentials-0.27.0
  (crate-source "gix-credentials" "0.27.0"
                "0icymf6v01y2r07bmwaw3vb1mx59m2x54lcb732bj2v9w6g0z5fg"))

(define rust-gix-credentials-0.28.0
  (crate-source "gix-credentials" "0.28.0"
                "0qn0a4wqvc5bq4air5y49806jyrzgh9j23c6dd9qjmznm8426ci5"))

(define rust-gix-date-0.10.1
  (crate-source "gix-date" "0.10.1"
                "06iwdhvyhyrppchkwimrk3b09sb5qjr23ff52pxbj50y3wzmk61s"))

(define rust-gix-date-0.10.2
  (crate-source "gix-date" "0.10.2"
                "1jxq14p5swl2s4lza3hhxfn4wkgk3xgcc3rbqzqf6h8pn991v78k"))

(define rust-gix-date-0.10.5
  (crate-source "gix-date" "0.10.5"
                "077yp7411k8i5mhxh8cj6mwdq2a3wv1p9cljmwq76a7vpa86nswr"))

(define rust-gix-date-0.5.1
  (crate-source "gix-date" "0.5.1"
                "00jrc86398553z2mdljx9vh8skqgdydhsrr11ak3148fcx2l25mw"))

(define rust-gix-date-0.8.7
  (crate-source "gix-date" "0.8.7"
                "1h5qygjxr8p2g2vdrzpvx0plnyy9gsym29wjxc5fx48ly8qnkvcy"))

(define rust-gix-date-0.9.4
  (crate-source "gix-date" "0.9.4"
                "1r0pc9ra4r7qxwsyd0jvxh3vsnm3jvkgkr19qbxi2dbxxic018ys"))

(define rust-gix-diff-0.44.1
  (crate-source "gix-diff" "0.44.1"
                "17bci0q2d4bhm6g1gx6ibzbxjj6ykxy6303x8q4rgd85lg4db5hr"))

(define rust-gix-diff-0.46.0
  (crate-source "gix-diff" "0.46.0"
                "0v1k7ld1b4xinq3dhvia4g9dck7b5x22h68wifrzh07z1zcazjcj"))

(define rust-gix-diff-0.47.0
  (crate-source "gix-diff" "0.47.0"
                "03i6v91k0bwyzzyjl2jp9rsx780v149hs4wydzdi7wasq780z1f9"))

(define rust-gix-diff-0.49.0
  (crate-source "gix-diff" "0.49.0"
                "1xh4mac785zyvnyap5awpj2iw1kw6aqgz5hg19d21jnbxik2bsd8"))

(define rust-gix-diff-0.50.0
  (crate-source "gix-diff" "0.50.0"
                "0kbwn5js7qwnqxxva52hrhxrkwhvxfr6a86rvblz9k8arbsbgbv2"))

(define rust-gix-diff-0.51.0
  (crate-source "gix-diff" "0.51.0"
                "1y5301nl772hhhwif1rsrz6w6dp4zcg4si1z4d75xj5gsbd7bjd2"))

(define rust-gix-diff-0.52.1
  (crate-source "gix-diff" "0.52.1"
                "0lpj2z4xdg7d3xlgd4sdf8zdxhk0z17wiw39m61dllp3bzll76sy"))

(define rust-gix-diff-0.53.0
  (crate-source "gix-diff" "0.53.0"
                "1a6yphk8dx612x0zp8y8g2a1qs9yk1xdd4hcycbs6i0d0594i1fy"))

(define rust-gix-dir-0.11.0
  (crate-source "gix-dir" "0.11.0"
                "14qjgsbb1dkyyha3l68z7n84j1c7jjycqrw3ia746d5xyjygz8pv"))

(define rust-gix-dir-0.13.0
  (crate-source "gix-dir" "0.13.0"
                "0fvrsk2yyk3acqnjvf9bnvjhv9whaabwhkl6xmvq4pc1sdxljyaq"))

(define rust-gix-dir-0.14.1
  (crate-source "gix-dir" "0.14.1"
                "11rfg4bc0wdwhq0zjcz4f6vla7k7s44j49zy1wni85w9bgff5rh1"))

(define rust-gix-dir-0.15.0
  (crate-source "gix-dir" "0.15.0"
                "1mrd88qjx11djz2nlcays6n9xcm1ycfjl7dsy4nr151z6x7lxlys"))

(define rust-gix-dir-0.6.0
  (crate-source "gix-dir" "0.6.0"
                "0r8kif2i622saw068c63jh08q64a5qixnggxgdsjvp80m9wmd5qc"))

(define rust-gix-dir-0.9.0
  (crate-source "gix-dir" "0.9.0"
                "18rlpbjy14ljv1sq839skfn2x8f121gaspwjsjb3kbvvy6dw5xmv"))

(define rust-gix-discover-0.33.0
  (crate-source "gix-discover" "0.33.0"
                "03j52f646q05igg4bln6svjxhxiz944khinj7sx33jy3rqqjfrk7"))

(define rust-gix-discover-0.35.0
  (crate-source "gix-discover" "0.35.0"
                "1ljnv5c2q1xpwpw45qhli0hydl7ba52dfpw1dv16ndv7jmmkcxq5"))

(define rust-gix-discover-0.36.0
  (crate-source "gix-discover" "0.36.0"
                "1xkijvasm2c9a1pwjjc05xq8ydy5fc4f255hvw4syl4g8lgy68n5"))

(define rust-gix-discover-0.37.0
  (crate-source "gix-discover" "0.37.0"
                "1vxnzwaz7g5dzn3dscy4qry3yb7r0748zldlxjdllsi69vx6vgw3"))

(define rust-gix-discover-0.38.0
  (crate-source "gix-discover" "0.38.0"
                "1n35pfcr4didkxswigy4lvwkqrhyvbgjk82sb87lw1h4vx5l3hnh"))

(define rust-gix-discover-0.39.0
  (crate-source "gix-discover" "0.39.0"
                "10b7x3pnj3far7zqm728dfm1spqy6k9iiphm75m50m5q951qmyzp"))

(define rust-gix-discover-0.40.1
  (crate-source "gix-discover" "0.40.1"
                "0z43w47zryb30iglinjlrrnyzrnim4xxnmkcj61n192fbgif7kyw"))

(define rust-gix-discover-0.41.0
  (crate-source "gix-discover" "0.41.0"
                "02nkg5fypjs6gwyv70wvyhk7bypqv3c67fr8x19wz8m13k4q1cgz"))

(define rust-gix-features-0.29.0
  (crate-source "gix-features" "0.29.0"
                "15zqli3vrygaap9n5258gi1zcs3nj1qvc11j5pi3mk01qzsv0sfg"))

(define rust-gix-features-0.38.2
  (crate-source "gix-features" "0.38.2"
                "0sfw6zs3qgmlqjkj4cvyfz6q6dgdlw1d16c7yckwgyg5kyn4aw5c"))

(define rust-gix-features-0.39.1
  (crate-source "gix-features" "0.39.1"
                "07yqby9y0icx2l7kwbvxfg6z8b7gfznknwd4vd0a68p0y9rxd1bx"))

(define rust-gix-features-0.40.0
  (crate-source "gix-features" "0.40.0"
                "0m6mf6f341shzs5b1iy79klkw00x84kba34z5i4bshldia1x9zcb"))

(define rust-gix-features-0.41.1
  (crate-source "gix-features" "0.41.1"
                "0d4n21h3z63wfqj2gg69vqqiddwwxgyvs8py412x2n4l45860v81"))

(define rust-gix-features-0.42.1
  (crate-source "gix-features" "0.42.1"
                "1v9zj90l4ax9gv1afh5bjas5ly2wdijzjk6xhkdxjkzcysd3kx2n"))

(define rust-gix-features-0.43.1
  (crate-source "gix-features" "0.43.1"
                "0nqyz111qv8fxrhkj3d5vsjcvqlf2rf6lrhsmbmvrg4akg6l65fd"))

(define rust-gix-filter-0.11.3
  (crate-source "gix-filter" "0.11.3"
                "06m6ph3b67696ckq5gfn9pwm77sh507km7sfzx6my9r8v8w7fm76"))

(define rust-gix-filter-0.14.0
  (crate-source "gix-filter" "0.14.0"
                "1sk50qqkhvbql3slagm6y9sgc6zdbiqsx4w9xmq5fj54b4izhdvb"))

(define rust-gix-filter-0.16.0
  (crate-source "gix-filter" "0.16.0"
                "02krfk40gvz53wardlmzngkc27cf7zb5dzn742x41y37avpcs3ix"))

(define rust-gix-filter-0.17.0
  (crate-source "gix-filter" "0.17.0"
                "1frbjkmwrafbp7psbnh9rp9szlakcn44b1jmqc7fsqxwgp6kdk5x"))

(define rust-gix-filter-0.18.0
  (crate-source "gix-filter" "0.18.0"
                "12dbr5x98fm8p9457d627919ng0nk0dxm0pwh8mrpk65znzjnayb"))

(define rust-gix-filter-0.19.1
  (crate-source "gix-filter" "0.19.1"
                "0v6aamn8g0zkpj2kwg0mn8s4bgs70aq33i5ppccgay0psvq2237r"))

(define rust-gix-filter-0.20.0
  (crate-source "gix-filter" "0.20.0"
                "1zyribibp2bkdmwp2xsl0pdfh25f1n70i6i7ch7v2ykyjaip2rda"))

(define rust-gix-fs-0.1.1
  (crate-source "gix-fs" "0.1.1"
                "1abqc9hh5s7r65p8rily3xy764z4lbwnglhbj44xq7v95y1s2dwv"))

(define rust-gix-fs-0.11.3
  (crate-source "gix-fs" "0.11.3"
                "0xc5z6w5s9rr0rgf7aab8hqg0dnblhk543crw30d19pykhjfdgzj"))

(define rust-gix-fs-0.12.1
  (crate-source "gix-fs" "0.12.1"
                "1f8xifs0wkq7lhy3c8091kq2lx15qkynjb6fwnbiyqjsa2n4yg9v"))

(define rust-gix-fs-0.13.0
  (crate-source "gix-fs" "0.13.0"
                "0g86cb2i18c7jnj8i9691a3h07nz7hvinig7ryvzyi6zpykpybhq"))

(define rust-gix-fs-0.14.0
  (crate-source "gix-fs" "0.14.0"
                "04f1z60r0lyi8cp1p09nr9l27wa3jk4fbrakq35ahpyw41hqh7lm"))

(define rust-gix-fs-0.15.0
  (crate-source "gix-fs" "0.15.0"
                "1iwpnmz9b1y965rs4vdf7x3ch08l4dvizy2mxb9j9vxl95qn7837"))

(define rust-gix-fs-0.16.1
  (crate-source "gix-fs" "0.16.1"
                "0163s0craj3g6hazq4xrcfxgha5y64rb11qg1qisfkq6glq90kcs"))

(define rust-gix-fsck-0.10.0
  (crate-source "gix-fsck" "0.10.0"
                "06wfc1ha88m00lv2ka53np09gp68i2gnf2gal1nm0jmr5vql1i2l"))

(define rust-gix-glob-0.16.5
  (crate-source "gix-glob" "0.16.5"
                "04gijbam0k2vdshm2m0d6hg0hvv7ky4dgr9p4y2l02hapi5qp43l"))

(define rust-gix-glob-0.17.1
  (crate-source "gix-glob" "0.17.1"
                "0d9lrxas6zjia91j3m4z8rnazz1s02j9kgw4fib82d8aximrmxma"))

(define rust-gix-glob-0.18.0
  (crate-source "gix-glob" "0.18.0"
                "0kii7bpz1vcdykb0x1k9zmhn22hynwyk4n5acfrzjy0az94p572f"))

(define rust-gix-glob-0.19.0
  (crate-source "gix-glob" "0.19.0"
                "0c506lb1lfa33ajrfxw3cljbgfcmqv85z7h9l9ryfwrlq2cj95r0"))

(define rust-gix-glob-0.20.0
  (crate-source "gix-glob" "0.20.0"
                "0zckksxxmp8zkd69l5wp5nysq6jj6dbnxw0cq40qsfz8cqvb09i9"))

(define rust-gix-glob-0.21.0
  (crate-source "gix-glob" "0.21.0"
                "1m9cnqyl4clvlxmk7gspwhkpzq97ksr6nkr51isplgl2cs1xnixr"))

(define rust-gix-glob-0.7.0
  (crate-source "gix-glob" "0.7.0"
                "0am74xk2ch4j73ghi41lqyjd54r5hl558m27ncj3zb198lh9hz60"
                #:snippet '(delete-file-recursively "tests")))

(define rust-gix-hash-0.11.4
  (crate-source "gix-hash" "0.11.4"
                "0bq986grpsfc6ddav5dlb8zvz1aky264dnnnmax2h1lsmpr2yhjb"))

(define rust-gix-hash-0.14.2
  (crate-source "gix-hash" "0.14.2"
                "0pjdlxbqxd9lbkccryfw2ghifiq3gz9h8ylliw0va8b16vvpsggr"))

(define rust-gix-hash-0.15.1
  (crate-source "gix-hash" "0.15.1"
                "1kp4yjlkp8g4qg0r2zs0jmz19r076f2y91cjsikhxvclf70wqphb"))

(define rust-gix-hash-0.16.0
  (crate-source "gix-hash" "0.16.0"
                "1y79zcwja9b1bqlr27awndla5wcmzd7a8rnh7qdq5ca9hv25w778"))

(define rust-gix-hash-0.17.0
  (crate-source "gix-hash" "0.17.0"
                "0jfbdmb1a22x4arhccrbpcwjkkcmfphs3nif6h1qp5b341r7jkl3"))

(define rust-gix-hash-0.18.0
  (crate-source "gix-hash" "0.18.0"
                "1n45szq2xh9vyasfmn27y0jbzkpc88sg0bifnjk54a365ib00jcd"))

(define rust-gix-hash-0.19.0
  (crate-source "gix-hash" "0.19.0"
                "03nmd7dlz393abhnj9yyfjafka99xsaymnb4fqm1lwvag5wss7r5"))

(define rust-gix-hashtable-0.5.2
  (crate-source "gix-hashtable" "0.5.2"
                "0hp2m2rvbv0vav5lkq7d7bvx74qrb6w3hnj1rq3aq69wdzhq1pvx"))

(define rust-gix-hashtable-0.6.0
  (crate-source "gix-hashtable" "0.6.0"
                "1zhqgncv6jh3x7a7a2w3qbayghmiwv230mdw6gvqw1ricqjmpxhf"))

(define rust-gix-hashtable-0.7.0
  (crate-source "gix-hashtable" "0.7.0"
                "1l8jq85fkfw4inmpd6w2pk1dq67krsqmmp100lpd1k1a6yy3148q"))

(define rust-gix-hashtable-0.8.0
  (crate-source "gix-hashtable" "0.8.0"
                "0jh15sf0r6cg3gava752xi694zidzx8xf7nw3zf8d49af3c6cq7h"))

(define rust-gix-hashtable-0.8.1
  (crate-source "gix-hashtable" "0.8.1"
                "0129yss47n6s826lva7iziwi4r0f2cr4xxigc7r48hcb60ycpddm"))

(define rust-gix-hashtable-0.9.0
  (crate-source "gix-hashtable" "0.9.0"
                "01sgvkg2q4h1cjfnnbwdgsdxdk1i15380h9zsmakw5cn92sh0ly3"))

(define rust-gix-ignore-0.11.4
  (crate-source "gix-ignore" "0.11.4"
                "1dzs6zlwj8105ynmbiszf319i7x3a3lpav0gda8gaq44b6bcsiz4"))

(define rust-gix-ignore-0.12.1
  (crate-source "gix-ignore" "0.12.1"
                "12mv0lgq8aviy6fc4mdxr7r0ra0l1kb729wf8fkhmbx4s8jgpcdn"))

(define rust-gix-ignore-0.13.0
  (crate-source "gix-ignore" "0.13.0"
                "0vyz5jfqd72b4pygwqrssr96jvfzi32hm7y4lz05b65zh35rsljg"))

(define rust-gix-ignore-0.14.0
  (crate-source "gix-ignore" "0.14.0"
                "060x4vqgpjnz0kikarcdq27mfkcjh4znlxbz8p8i0fj91wwch9ws"))

(define rust-gix-ignore-0.15.0
  (crate-source "gix-ignore" "0.15.0"
                "1j3i3f8pfr4cwccvvv8pww1xdpnzi1kw0qysqymi02v6jqy8qddf"))

(define rust-gix-ignore-0.16.0
  (crate-source "gix-ignore" "0.16.0"
                "0w187k97lwj93n5lbivgmzfbs26b82nxc8qvawgrhb3fykfnykan"))

(define rust-gix-index-0.33.1
  (crate-source "gix-index" "0.33.1"
                "0gyply1fzp0wbl8jvckiw4hyv34na7lq0jd4z14bp15xapml96ls"))

(define rust-gix-index-0.35.0
  (crate-source "gix-index" "0.35.0"
                "0bfjbrwmg8w9hi5bv47db272ks8b3385zrkn45l1fh248hr21m0c"))

(define rust-gix-index-0.36.0
  (crate-source "gix-index" "0.36.0"
                "0agycrg9hywdn89sj8hxbhx1c2aszbsp64h4hpc3z8qyr84r0q97"))

(define rust-gix-index-0.37.0
  (crate-source "gix-index" "0.37.0"
                "0yi5g8vfk5vgb5la031s86a6j7i83f90sm51zz468ssm43yla1i7"))

(define rust-gix-index-0.38.0
  (crate-source "gix-index" "0.38.0"
                "1n45vkbmkpc4m570rdanyqz62a68mihsrqpz1wqnk4w74qv2xldc"))

(define rust-gix-index-0.39.0
  (crate-source "gix-index" "0.39.0"
                "19q8p7649hssm0yj3kxllgb8rkm1xqfxb00abnm56d0mskifqnw5"))

(define rust-gix-index-0.40.0
  (crate-source "gix-index" "0.40.0"
                "1d9hyn8jhcmy0lrcp8h2vns716d82b3r1jsk65x2di6plyp0bmg6"))

(define rust-gix-index-0.40.1
  (crate-source "gix-index" "0.40.1"
                "0rdn0dj58vc6bla0wqi7gc005l5r595kkliss9sq5jsrzng933mk"))

(define rust-gix-index-0.41.0
  (crate-source "gix-index" "0.41.0"
                "09by4r717ry2k3z7469diwqlghvfjgr2ds4w3lvi3kp47kg9zwra"))

(define rust-gix-lock-14.0.0
  (crate-source "gix-lock" "14.0.0"
                "17g1sknpvjqaq2s29c693mbmkp8sign0174qfi3n3x7ijzi7zg73"))

(define rust-gix-lock-15.0.1
  (crate-source "gix-lock" "15.0.1"
                "0h6r088yv5fk0d14zihssfh1zfhdyc8cpnpbygcn7nsjlilaplqw"))

(define rust-gix-lock-16.0.0
  (crate-source "gix-lock" "16.0.0"
                "0hn696w506zwqfl9pjhijaqkshzr5lb4v0j1hjb40sgzf1982fcp"))

(define rust-gix-lock-17.0.0
  (crate-source "gix-lock" "17.0.0"
                "0vqjl0l5f6nbkkw6lwy3rvghnjwfpjgzri8vajshslil3kqvhiyz"))

(define rust-gix-lock-17.1.0
  (crate-source "gix-lock" "17.1.0"
                "15m7wxli4kfljsz1asndrh5g40j9j8j1l3y9dlv5dwjr8q1qn3sp"))

(define rust-gix-lock-18.0.0
  (crate-source "gix-lock" "18.0.0"
                "1va2mpl51g447aa9pc5ky4d4f14rjxdyp17146k6hminj3d73ymr"))

(define rust-gix-lock-5.0.1
  (crate-source "gix-lock" "5.0.1"
                "0m8m26w2c0pqf835zihhhi8r78yfqynm0wa6gi5af3vk0mzkss9c"))

(define rust-gix-macros-0.1.5
  (crate-source "gix-macros" "0.1.5"
                "05ycxnh5sxjsn4lvay309n6knr8ksvkb6zx6f50ik24zc4iyk74r"))

(define rust-gix-mailmap-0.26.0
  (crate-source "gix-mailmap" "0.26.0"
                "14xkdb8mk8xrkncyv5rpng7igy1cknx7xhswg32k0kb8sa3d107z"))

(define rust-gix-merge-0.4.0
  (crate-source "gix-merge" "0.4.0"
                "1d4jd5mxpvdk269aci7fybdgj9325iavz1iwzjpka2clsczchzvb"
                #:snippet '(delete-file-recursively "tests")))

(define rust-gix-negotiate-0.13.2
  (crate-source "gix-negotiate" "0.13.2"
                "0962x6gxhwp55kyr26hzsswv8qbg9h1f16x8kd8n7fq7cgxpkj4y"))

(define rust-gix-negotiate-0.16.0
  (crate-source "gix-negotiate" "0.16.0"
                "1gfhnzjv0q2gj27xwgdx576q8kw5zx0diiirm6g39hrq30lhcj21"))

(define rust-gix-negotiate-0.17.0
  (crate-source "gix-negotiate" "0.17.0"
                "1a7igqmnnk183fyibbrwdfyjpqqz4vl5p79vr3lqcls02q586zyj"))

(define rust-gix-negotiate-0.18.0
  (crate-source "gix-negotiate" "0.18.0"
                "107gh0yn4z1lnzljlr538gg5bs9k9mzjncam1g9h7qxvywgaza56"))

(define rust-gix-negotiate-0.19.0
  (crate-source "gix-negotiate" "0.19.0"
                "0yz5hm15534ah4gz9660qgw1lf7lgwri8q43lkppv2m6ynn15nfs"))

(define rust-gix-object-0.29.2
  (crate-source "gix-object" "0.29.2"
                "1kk1wbcv6p71ysvyibf6y6s03c50ry1j22zp6zfwr2nh1xibv5id"))

(define rust-gix-object-0.42.3
  (crate-source "gix-object" "0.42.3"
                "11p3pynmriglj5j3hzh5qa4sz7pnhzxxzr1w85xzmhp7ni32zni5"))

(define rust-gix-object-0.44.0
  (crate-source "gix-object" "0.44.0"
                "1ylfp09gzy42b1m1day41m49b3d8dggw480qfd07dppi6hc80nrg"))

(define rust-gix-object-0.45.0
  (crate-source "gix-object" "0.45.0"
                "06pwqvxwr3appcw3k63hj6jfg0a4j921g2xfv59qaa9xfpkvcxra"))

(define rust-gix-object-0.46.1
  (crate-source "gix-object" "0.46.1"
                "1skp7mbqbcdqkv9n4nxk87ilzd4jxfs7k10864zh7vw3040mhbg4"))

(define rust-gix-object-0.47.0
  (crate-source "gix-object" "0.47.0"
                "0s7xwm1nmx2zp10qnrlxh8vmw5nakjkvfzrl4bzg0i220jhb7i6x"))

(define rust-gix-object-0.48.0
  (crate-source "gix-object" "0.48.0"
                "16nn1ndxij6sv5fsidsaja192lzdc81ix9wy1j93bhgzwvdgqhs9"))

(define rust-gix-object-0.49.1
  (crate-source "gix-object" "0.49.1"
                "1fqgqhj2ax4haln4bxllxn017yk9f737i0kzna5x8mf580vclmyr"))

(define rust-gix-object-0.50.2
  (crate-source "gix-object" "0.50.2"
                "01vw75jmgxfyic011vifxfx7i7a208ak7qdp9yymzdk7mc4f376n"))

(define rust-gix-odb-0.61.1
  (crate-source "gix-odb" "0.61.1"
                "16xwwfjna9m3rd66hkhcyps6s3bq23r5wpbxpfixi4qxakz89lr0"))

(define rust-gix-odb-0.63.0
  (crate-source "gix-odb" "0.63.0"
                "0iy7grqs5h73nz781zfmiyisr9pm4yjxsamby1ady5qwf1l805d3"))

(define rust-gix-odb-0.64.0
  (crate-source "gix-odb" "0.64.0"
                "0q8gwv4mdm8jqmfr73q0z009fvvh151wjkqvc20gkcpiyynnmf0b"))

(define rust-gix-odb-0.66.0
  (crate-source "gix-odb" "0.66.0"
                "1bnhjzg360vy6f303ps7halnnkx3x81dwy4l8q2f4bipng70wy6b"))

(define rust-gix-odb-0.67.0
  (crate-source "gix-odb" "0.67.0"
                "06ww8mc10iydvqxdin0miny89g9z8i7zmsccc1rrbl4wyrylb4ry"))

(define rust-gix-odb-0.68.0
  (crate-source "gix-odb" "0.68.0"
                "0476ynwy0aydycadpfac15xrqa56dq33y44kfmmyp0n9vi06sc2h"))

(define rust-gix-odb-0.69.1
  (crate-source "gix-odb" "0.69.1"
                "1jg00pj6sv90nnnqhllz0fvyq0wr8612z52hswdzrg7x0lwp13w6"))

(define rust-gix-odb-0.70.0
  (crate-source "gix-odb" "0.70.0"
                "1b09n472wxdgbj15nnmidk3b6q1mjq3xb6vz9yxz17fs1zqpm7cw"))

(define rust-gix-pack-0.51.1
  (crate-source "gix-pack" "0.51.1"
                "0aajbwpmfzg7a33yzqfjz2rmczvbask13hd19gwmvrgz3x4r819y"))

(define rust-gix-pack-0.53.0
  (crate-source "gix-pack" "0.53.0"
                "0m6rdxkq21ni71vk8k7qbsjxr7mgkjpdijh3wkhf28gf5qsal8rj"))

(define rust-gix-pack-0.54.0
  (crate-source "gix-pack" "0.54.0"
                "0sq240glmpvp0x1bpsngrlk82iz2d3dkk0a0f8v29fjmm1cnwgin"))

(define rust-gix-pack-0.56.0
  (crate-source "gix-pack" "0.56.0"
                "0v2n5k0q08zdiqwbmn1rgwd0fcm943lcibwpmgkwlady564r4n21"))

(define rust-gix-pack-0.57.0
  (crate-source "gix-pack" "0.57.0"
                "05d57xpzk35i2cclnb9iclvm1gvrc20mzcvz04bmcwyvndss84zw"))

(define rust-gix-pack-0.58.0
  (crate-source "gix-pack" "0.58.0"
                "1jvrjzrx5q8596cdyp0arnagnxl7zqnd62549ii2cg1r17xzyrcv"))

(define rust-gix-pack-0.59.1
  (crate-source "gix-pack" "0.59.1"
                "0l54502vck1sq87my8giwaqbdsygpaffnxss18m9yi68d5fwajcx"))

(define rust-gix-pack-0.60.0
  (crate-source "gix-pack" "0.60.0"
                "069h4rdyyg2r0divkqnvp3wfdxvs7wwp4crykjsap9gwkgw1smyq"))

(define rust-gix-packetline-0.17.6
  (crate-source "gix-packetline" "0.17.6"
                "0jay9kgy8fgc809xcipgwhz430a4pyywhcb7c0n25yp2bx6yyhwc"))

(define rust-gix-packetline-0.18.4
  (crate-source "gix-packetline" "0.18.4"
                "011sdpf03fp066v9q8zcjjz63vwavaqbl1nw84j3bmgl1jkl8f0j"))

(define rust-gix-packetline-0.19.0
  (crate-source "gix-packetline" "0.19.0"
                "075rvqjlyclbrvhxnn0866fl7zfqii9basjryx78x13wqr5h7p4d"))

(define rust-gix-packetline-0.19.1
  (crate-source "gix-packetline" "0.19.1"
                "14qzpxdk760rwiyiagchipriwc3nqdf0bxsn22hzx8j9cb9zp4i5"))

(define rust-gix-packetline-blocking-0.17.5
  (crate-source "gix-packetline-blocking" "0.17.5"
                "0h3bansw7mp2p6id78lwlrs4pm8nclmql07qbxpxv657p822705r"))

(define rust-gix-packetline-blocking-0.18.3
  (crate-source "gix-packetline-blocking" "0.18.3"
                "1crknbdkdnh5aanpyda3dhm2c0a3hacl1h5shxay9iq5w6i3xkqy"))

(define rust-gix-packetline-blocking-0.19.0
  (crate-source "gix-packetline-blocking" "0.19.0"
                "1wz2d08hdpwkiy8l83pdp18nn31i02ijfvd66z7xcims53q80j64"))

(define rust-gix-packetline-blocking-0.19.1
  (crate-source "gix-packetline-blocking" "0.19.1"
                "152cpn4jlhw12r0mxnxh253rlax74ghp48ckiz699m4c69pp0kpw"))

(define rust-gix-path-0.10.15
  (crate-source "gix-path" "0.10.15"
                "11xylymhw8maxv5z81w5hrxry10sibw4vw516pzmaakb5y76c47r"))

(define rust-gix-path-0.10.17
  (crate-source "gix-path" "0.10.17"
                "0hp62rngav8dilp68yv5yb0700hmc7mcali2ymi38b70hzld54f0"))

(define rust-gix-path-0.10.18
  (crate-source "gix-path" "0.10.18"
                "0nxn7zzl272mb250ryg3p3kpykgx4xmg4wdfjymxy47gqkz6azsn"))

(define rust-gix-path-0.10.20
  (crate-source "gix-path" "0.10.20"
                "1zdjwwb5fyk934mnd9h91jx0zg7pn8vx1g7pfvdbsyy6lhs71lq6"))

(define rust-gix-path-0.8.4
  (crate-source "gix-path" "0.8.4"
                "0z5733b3z2wbnz1x0y2aq3gpanrhrlrqr4v4gjlqwl68ps69qq0q"))

(define rust-gix-pathspec-0.10.0
  (crate-source "gix-pathspec" "0.10.0"
                "057ch8xlbadf2rnjvlzs5ciqagwn4m8h492b0x4xcrlh7hn45y7y"))

(define rust-gix-pathspec-0.11.0
  (crate-source "gix-pathspec" "0.11.0"
                "17cp238mc8nfy14j4lp87hl5x4xfk7ls6gfbr8qcixzqwm81q1nf"))

(define rust-gix-pathspec-0.12.0
  (crate-source "gix-pathspec" "0.12.0"
                "1fkzv9ql1n98kxjlvbw82wh91xs55c0sk46w80v1zgf927bfmvfs"))

(define rust-gix-pathspec-0.7.7
  (crate-source "gix-pathspec" "0.7.7"
                "1ssw9k2kq7hcn5xn9nby4gvq2628clxapf331r6l3d1jjlivy8sx"))

(define rust-gix-pathspec-0.8.1
  (crate-source "gix-pathspec" "0.8.1"
                "07mqfl6232285yzsmym2vr7vndwh3ivx9p7xgv7nzsd4wkxjsisc"))

(define rust-gix-pathspec-0.9.0
  (crate-source "gix-pathspec" "0.9.0"
                "0v7q0b55fn0raaj52cg75bi5yc8pijkzl1lq05crv3n0hskd6c34"))

(define rust-gix-prompt-0.10.0
  (crate-source "gix-prompt" "0.10.0"
                "1w6m975b2pj5ghxhxykrnz4s33xjx6sfwmww5hn2zlwz4gvcpygv"))

(define rust-gix-prompt-0.8.9
  (crate-source "gix-prompt" "0.8.9"
                "1505js24g8dziljc7jl5frmk0af1847v106fqsxmz75wqjpj4y3s"))

(define rust-gix-prompt-0.9.1
  (crate-source "gix-prompt" "0.9.1"
                "0v9v9icnryhcx2z256kmm8aa1p31ipghjx80kac2nlg1b1ciiwkr"))

(define rust-gix-protocol-0.45.3
  (crate-source "gix-protocol" "0.45.3"
                "0gjf2s9ssch79jfyv7bpa8pxwgdqks6940x04bpfzd81dw0a2hyc"))

(define rust-gix-protocol-0.46.1
  (crate-source "gix-protocol" "0.46.1"
                "1jmq10azisdp4k1i18hif4cdxchrm4ppwacc8k9k39fyl18pwzks"))

(define rust-gix-protocol-0.47.0
  (crate-source "gix-protocol" "0.47.0"
                "0rxhmvm3c1lamghxr9gp3bnb0mcw069rai6cx5f07mzynvl44in8"))

(define rust-gix-protocol-0.48.0
  (crate-source "gix-protocol" "0.48.0"
                "145sln6g810vab9jhwiz3r1bwr61jh1i1qj168hpvdn6mxhvsqbc"))

(define rust-gix-protocol-0.49.0
  (crate-source "gix-protocol" "0.49.0"
                "0njv6mkq6zsf3gw233p8w1r37bw7jcdvw0121v1hp2323npdsy2n"))

(define rust-gix-protocol-0.50.1
  (crate-source "gix-protocol" "0.50.1"
                "00c4wfhb9kcf2blmv26a1lrfqirfvjb235az1gbgh504pdw7vhgm"))

(define rust-gix-protocol-0.51.0
  (crate-source "gix-protocol" "0.51.0"
                "08p9cj9j7m7166wr7nlk6jgl8mla6s2ra4dqwp0zgz3zqh3vid0j"))

(define rust-gix-quote-0.4.15
  (crate-source "gix-quote" "0.4.15"
                "1ik6l3s0hjb2p4dlgdarb59v7n9jvgvak4ij786mrj5hrpy5g4z4"))

(define rust-gix-quote-0.5.0
  (crate-source "gix-quote" "0.5.0"
                "0sgr4zw14nnp6l74fc3x3k0a2ik17si41rd59arf6kgq1damq00v"))

(define rust-gix-quote-0.6.0
  (crate-source "gix-quote" "0.6.0"
                "1zc3gnc8z7z94sijyn0k9ij5adx230599x1vzsxfhqynnismldsa"))

(define rust-gix-ref-0.29.1
  (crate-source "gix-ref" "0.29.1"
                "1m8mgrqxx8cvkvkzpkaacg8qjwgw6215f9mmw5l475a9kng9h0qy"))

(define rust-gix-ref-0.45.0
  (crate-source "gix-ref" "0.45.0"
                "1dcv2pgaz6gzhx38zw0qh9pkmhqhf4hqq2gf7waia9snlnh9cvk3"))

(define rust-gix-ref-0.47.0
  (crate-source "gix-ref" "0.47.0"
                "1r8i8fj7xgmp88qk1w7xc2gs7l8sb8y0azx5algskaprxc3883df"))

(define rust-gix-ref-0.48.0
  (crate-source "gix-ref" "0.48.0"
                "18mfzrnp1308g5c454xwa85dz3c0913fyhp66n6dmnd23zkqawx4"))

(define rust-gix-ref-0.49.1
  (crate-source "gix-ref" "0.49.1"
                "0ps8xzgkblsn3k5961ykyjkpm55hmxwi340lf4dhz7c3divn26x9"))

(define rust-gix-ref-0.50.0
  (crate-source "gix-ref" "0.50.0"
                "03723r9s3m3grmjzcasxp7jcz0z5xs90spg9aj2ryhikz72z9ba7"))

(define rust-gix-ref-0.51.0
  (crate-source "gix-ref" "0.51.0"
                "12sx86q6gz5y7nrbziqvkfkkxavvcfyp87wn34njvs3wdgmzgqdj"))

(define rust-gix-ref-0.52.1
  (crate-source "gix-ref" "0.52.1"
                "0qkp1abj7222jjscvmcnal476b1ypjdb0mpnb7bq95h2axb9idyi"))

(define rust-gix-ref-0.53.1
  (crate-source "gix-ref" "0.53.1"
                "0n0pfaa2y239cr356mv0ryhwqi358z77pc8k2jjz8hls0xwgarmr"))

(define rust-gix-refspec-0.23.1
  (crate-source "gix-refspec" "0.23.1"
                "0drzc1p43mc8nprxlqm4s86y9b23rsz89dvqghgmsmb25v6zhs38"))

(define rust-gix-refspec-0.25.0
  (crate-source "gix-refspec" "0.25.0"
                "19lwhs10cg9rg040k20w8i27ay475isggngxbxhngfj14gw0bc7b"))

(define rust-gix-refspec-0.26.0
  (crate-source "gix-refspec" "0.26.0"
                "0hn4mbnvcammpwrqcawpysbqv1h2np5yzs1vfyzrl3fq165068h0"))

(define rust-gix-refspec-0.27.0
  (crate-source "gix-refspec" "0.27.0"
                "0ga1dh6500q88wb0ymd2s2iqraw1j6gjqdgb1bmwfs3qfjxmdh00"))

(define rust-gix-refspec-0.28.0
  (crate-source "gix-refspec" "0.28.0"
                "140aif2nciz9j9a0h9lqsg8cb1pkzhbza9bsgy7gc4pnv0l04rar"))

(define rust-gix-refspec-0.29.0
  (crate-source "gix-refspec" "0.29.0"
                "1z1g4mfchpxi6xbilx0h7b0pj9v6k5f0r54djglacr123sr8g18x"))

(define rust-gix-refspec-0.30.1
  (crate-source "gix-refspec" "0.30.1"
                "0hivk37yd3vaxp45pbmkcf0iqkpr7mrf7180k1wqx3mp7m7d2pj4"))

(define rust-gix-refspec-0.31.0
  (crate-source "gix-refspec" "0.31.0"
                "0m80rqn7f6ay6z05ngs1yh2v9jpspxh5wshmwwk8h41imvhwlabx"))

(define rust-gix-revision-0.27.2
  (crate-source "gix-revision" "0.27.2"
                "1kmd96vky58yf46g5nxqny8gl3df4447vb6x6w2lp30iq91kxc81"))

(define rust-gix-revision-0.29.0
  (crate-source "gix-revision" "0.29.0"
                "07jyh9vabfbkva99m0mavsk68v3asp1k0c48as9dn35c36r22ims"))

(define rust-gix-revision-0.30.0
  (crate-source "gix-revision" "0.30.0"
                "1wam9d627191a4qdfjjj8lryk44z0qg7apaamxi3bkpyi10fps2f"))

(define rust-gix-revision-0.31.1
  (crate-source "gix-revision" "0.31.1"
                "1rzrr2i5vb8yc0cffprgkyawxrigsxfp11c4rv96hnj0fk2dvqb1"))

(define rust-gix-revision-0.32.0
  (crate-source "gix-revision" "0.32.0"
                "0lvb7rrjjdr9h21ign5g0za2jg00nimzqvkcdvbacpd5rjy8pqiz"))

(define rust-gix-revision-0.33.0
  (crate-source "gix-revision" "0.33.0"
                "1sf3c0c824kj3lbngpygxbj8z5630xinabznml605wwd2m7alb1l"))

(define rust-gix-revision-0.34.1
  (crate-source "gix-revision" "0.34.1"
                "0pf6wdqnn555mwq3nr0s08a98hqpiz5qiq43agi2khyirgjvil3q"))

(define rust-gix-revision-0.35.0
  (crate-source "gix-revision" "0.35.0"
                "0gdxdicadvsa9qgw8cww7nvn5s86j8i46rqx2sw0nxigfjqz4lgn"))

(define rust-gix-revwalk-0.13.2
  (crate-source "gix-revwalk" "0.13.2"
                "1q23mcf4ji5q8qi3g86vxxgz4x4ykgqja0kyacgi9bvimg50q0qv"))

(define rust-gix-revwalk-0.15.0
  (crate-source "gix-revwalk" "0.15.0"
                "111r3ddls12y1s884x01gai8ykdsn4qmnggghbk4w24k9da747ml"))

(define rust-gix-revwalk-0.16.0
  (crate-source "gix-revwalk" "0.16.0"
                "1cirkpxgz52mvib9lw1vb0jp9a09pxv8afh637zkd3d9dm4skjg6"))

(define rust-gix-revwalk-0.17.0
  (crate-source "gix-revwalk" "0.17.0"
                "178hrfgigvr3a1m8mn1949ms75wb10s7rwyqczqghmpl6by2c02i"))

(define rust-gix-revwalk-0.18.0
  (crate-source "gix-revwalk" "0.18.0"
                "0iv2c479w9lkjwbngdwyial6km8dllgah8wvp7r9w7jv4c6biv6l"))

(define rust-gix-revwalk-0.19.0
  (crate-source "gix-revwalk" "0.19.0"
                "03fy5glq01v95cqz908crag9ykqamzj0c08k6n6sphfdwpbw7ird"))

(define rust-gix-revwalk-0.20.1
  (crate-source "gix-revwalk" "0.20.1"
                "0a1bnmjng8x5zxpfalqdgm0y4crwgg5d2wdqvrfh1gr56avmdiqv"))

(define rust-gix-revwalk-0.21.0
  (crate-source "gix-revwalk" "0.21.0"
                "0b60dvccv98w6195220540zj8bynlw7vs8b7mycy0acpf28lzrq6"))

(define rust-gix-sec-0.10.12
  (crate-source "gix-sec" "0.10.12"
                "122qvp6sll7hkrpjiwf9ga1dni0gwf6j3zzm6cq2zvz97pqv1bj7"))

(define rust-gix-sec-0.11.0
  (crate-source "gix-sec" "0.11.0"
                "1z88qf20pf05q5rb0dwawghwicji94wkj0wpdc0cr7kmik3vpnnh"))

(define rust-gix-sec-0.12.0
  (crate-source "gix-sec" "0.12.0"
                "0b6zdzgldi2y3jgxzhixmxp1dqvv6z9nxg2pdjsk6rn6swz0bxq9"))

(define rust-gix-sec-0.8.4
  (crate-source "gix-sec" "0.8.4"
                "1iz9rcyx7lpb4gxg5gyv93ygp0n321c5xmrcjkmqm2annkbcn5cn"))

(define rust-gix-shallow-0.1.0
  (crate-source "gix-shallow" "0.1.0"
                "00ffhjy3lzqr6zqj0h3c635634383z0g0wgndz5r4x7888r6gll8"))

(define rust-gix-shallow-0.2.0
  (crate-source "gix-shallow" "0.2.0"
                "0rjhwcjjixfy4fbzciyz5mikkvq38rwfyny86ckya0z324q58wmb"))

(define rust-gix-shallow-0.3.0
  (crate-source "gix-shallow" "0.3.0"
                "1hrlyhivjan43743dv18w8hvpvc6w3z9550wl9sjbmg1rym9h1fc"))

(define rust-gix-shallow-0.4.0
  (crate-source "gix-shallow" "0.4.0"
                "1m4s24l3i5jhx3wfprla9d960kqbk73yb29daa7y1vfn6ip6z6kb"))

(define rust-gix-shallow-0.5.0
  (crate-source "gix-shallow" "0.5.0"
                "1ds4i77bfzqd1yzgv2351440zyvkrrh0g7pi232y8fi40d8p8dnr"))

(define rust-gix-status-0.16.0
  (crate-source "gix-status" "0.16.0"
                "1pv4fprwb2iiz737p7kljzxwdzxkdjzsv1mg1rmvs23n4w77fr8n"))

(define rust-gix-status-0.18.0
  (crate-source "gix-status" "0.18.0"
                "1xdjd37n0a8pamz7rp1nn6fd6yzgcdxfxci4dv2805l9nl76snk0"))

(define rust-gix-status-0.19.1
  (crate-source "gix-status" "0.19.1"
                "17r82wdsvrk5973p9mp02z6x45k0rg5ylivxvybm7yjw8719j807"))

(define rust-gix-status-0.20.0
  (crate-source "gix-status" "0.20.0"
                "0d9dvh4aymg4bzaginci2dmlnhqqncpv8cnwig5cxvjfngwzyjia"))

(define rust-gix-submodule-0.12.0
  (crate-source "gix-submodule" "0.12.0"
                "1zalx2cc2b2sj8kqncbkbpnj1np9lxr09j4yscwmx000m9lhybhg"))

(define rust-gix-submodule-0.15.0
  (crate-source "gix-submodule" "0.15.0"
                "0yj9y2b7425a3bc2wp2sy7z50zialdv230pwh32kdkbk31i9kl1y"))

(define rust-gix-submodule-0.16.0
  (crate-source "gix-submodule" "0.16.0"
                "0p4zx1v997wjglya5xrfxdixf586scigbj43dqmbwvnb1y65yid2"))

(define rust-gix-submodule-0.17.0
  (crate-source "gix-submodule" "0.17.0"
                "1b532y2c7qg8axqc2nkw2mdiq8mg9hxq87mfj2aa1j3askl2z5vl"))

(define rust-gix-submodule-0.18.0
  (crate-source "gix-submodule" "0.18.0"
                "18ldqsxh0i11n6vi54bakis3b7yw9rnh2j4mbqv5ql2r4063kivq"))

(define rust-gix-submodule-0.19.1
  (crate-source "gix-submodule" "0.19.1"
                "1hdhr0k3yyg89fy21i1ipcmixqq6zdi2zvcipihwql540lplflaz"))

(define rust-gix-submodule-0.20.0
  (crate-source "gix-submodule" "0.20.0"
                "0gidhmwfyc6v98q3zvfdfz81aalwpv7h5bsski6s3iyb8gfwaz35"))

(define rust-gix-tempfile-14.0.2
  (crate-source "gix-tempfile" "14.0.2"
                "1ah8y51fbx6x9m71ihxk3zhny0c05kka5kd0a2hida4zjqkljsq4"))

(define rust-gix-tempfile-15.0.0
  (crate-source "gix-tempfile" "15.0.0"
                "10nvk82g7fhljg5y63dxpd8p7296wrfzxyssk957misc17pqdsrg"))

(define rust-gix-tempfile-16.0.0
  (crate-source "gix-tempfile" "16.0.0"
                "00c5czgzzi3c8yxv24vh1rmkgf06vgb1ypf5521lmwjyjhiz8n15"))

(define rust-gix-tempfile-17.0.0
  (crate-source "gix-tempfile" "17.0.0"
                "16q9gwc2lxrqqpxhvgpw438kgmnj2q7anzww1iax79drpcwy8v9x"))

(define rust-gix-tempfile-17.1.0
  (crate-source "gix-tempfile" "17.1.0"
                "1akh8jnmfm0x6mg3yfhp67d0avkinwldkc52cyx2sfj5130fhl67"))

(define rust-gix-tempfile-18.0.0
  (crate-source "gix-tempfile" "18.0.0"
                "0mys8wfjsha2axh425ywnwjapsvxi4y6dvwvbsh5zpyypi0h0v36"))

(define rust-gix-tempfile-5.0.3
  (crate-source "gix-tempfile" "5.0.3"
                "10pivlx6a5yph3jnsj6h0p3ap3fpxz52ahhjhrjyhwafycr0s6np"))

(define rust-gix-trace-0.1.12
  (crate-source "gix-trace" "0.1.12"
                "1xv54v5y91vxjx351wl3yk66fwk7ybkna2knbxlnj34j6qh6lfbw"))

(define rust-gix-trace-0.1.13
  (crate-source "gix-trace" "0.1.13"
                "0n76pqvz7zijlg8ybchhim6p80vnknms0b28bdlklx5in1aazk72"))

(define rust-gix-transport-0.42.3
  (crate-source "gix-transport" "0.42.3"
                "07s8lsq97r0hgg9znd2f0jaj49prm2bss9mjjxfs2h8vn35cq7a2"))

(define rust-gix-transport-0.43.1
  (crate-source "gix-transport" "0.43.1"
                "02r0fwai9pq6f2n1nn588pjc71rxh9zi9169w01nq8xpaw9s989r"))

(define rust-gix-transport-0.44.0
  (crate-source "gix-transport" "0.44.0"
                "1bybr8741kilfvmwg71p80xfadjvsy2mm39ilb7i71vsa0gdj16x"))

(define rust-gix-transport-0.45.0
  (crate-source "gix-transport" "0.45.0"
                "1nb4p7jwy80g51afzc64ya1faxxcpgnimbk2p2sv2xwl90c7860i"))

(define rust-gix-transport-0.46.0
  (crate-source "gix-transport" "0.46.0"
                "0zkq877jmmraqz9pflncn386fnr1yakq894x71w85jmzf0l8rxmk"))

(define rust-gix-transport-0.47.0
  (crate-source "gix-transport" "0.47.0"
                "0nbgah7w3qpwfar37j8ihqy5vrmg5s4bj4lzhwbmrdnl4sx25zpd"))

(define rust-gix-transport-0.48.0
  (crate-source "gix-transport" "0.48.0"
                "18qwfh8v1ccxsqr624xzni58d54l498wxyg1ahydb2gwg40wrxqj"))

(define rust-gix-traverse-0.39.2
  (crate-source "gix-traverse" "0.39.2"
                "0h48bbjym6fkdhy9q7p8v6rn9ksvkwxp8fs1415cyw8ya66a36g4"))

(define rust-gix-traverse-0.41.0
  (crate-source "gix-traverse" "0.41.0"
                "105prigk1pq7g3i3hvwh4y9rhc35ycl84cg9f9ag6kafz6da6383"))

(define rust-gix-traverse-0.42.0
  (crate-source "gix-traverse" "0.42.0"
                "1pqqx02bab9101iqry4f8nsbwd3azg1a0sjfna9bm9jgrh9in3zj"))

(define rust-gix-traverse-0.43.1
  (crate-source "gix-traverse" "0.43.1"
                "12dcmf67m409ss0jjpgr9vjh048cs6hbplkiz69kxqhrhrj7vm3f"))

(define rust-gix-traverse-0.44.0
  (crate-source "gix-traverse" "0.44.0"
                "1d311l7wlgpv41hvp1ni3r9hhwxn4x27xyiy5brnwn4n73jp1v1b"))

(define rust-gix-traverse-0.45.0
  (crate-source "gix-traverse" "0.45.0"
                "113kzy27skxddmg9qgp84wbjwzshgcpi153604h1pdmxz14v1h1n"))

(define rust-gix-traverse-0.46.1
  (crate-source "gix-traverse" "0.46.1"
                "0mqipqxn22lkwwx3pbklsx0n984qypvvn3qk048xi96rys2l229r"))

(define rust-gix-traverse-0.46.2
  (crate-source "gix-traverse" "0.46.2"
                "0cyc0b0qi98g22q5xi849vsswsy2g95m01lwj5p3vjjsz1r82r5q"))

(define rust-gix-traverse-0.47.0
  (crate-source "gix-traverse" "0.47.0"
                "1mdx05k5llll8infjqj3z48gwylw8rmnzy0mv05bm4np14jwikf7"))

(define rust-gix-url-0.27.5
  (crate-source "gix-url" "0.27.5"
                "128c2rp6780qspiqasmyd6a3gdmcmq6kl19axllf28pvhig0qa7x"))

(define rust-gix-url-0.28.2
  (crate-source "gix-url" "0.28.2"
                "1ncj6k4lk3qb0i27ida7ngi9z06qpmrbva6v0da3zgd67drzp5nh"))

(define rust-gix-url-0.29.0
  (crate-source "gix-url" "0.29.0"
                "04qb2p68886axrbx5gdjlhqwg55j0pn7zn25c08qzpakidv8q899"))

(define rust-gix-url-0.30.0
  (crate-source "gix-url" "0.30.0"
                "0q78bb1ark0chavckh5gspsip7d0gbfv02yqfx4vipgijczy5ps8"))

(define rust-gix-url-0.31.0
  (crate-source "gix-url" "0.31.0"
                "1f44xh1r5sy5vj0is5lnca22gdm9aa78irikn9f8nwd50h5sv8a2"))

(define rust-gix-url-0.32.0
  (crate-source "gix-url" "0.32.0"
                "0vyi2kjr6awl5ph54kcgn2cqg1p7i36nfi6lzy3x4ji5cv9ajxhv"))

(define rust-gix-utils-0.1.14
  (crate-source "gix-utils" "0.1.14"
                "0pykxyp0cm2x8lj4ryj1pflksf9k7iyrshf8g321d2dc0d7g427z"))

(define rust-gix-utils-0.2.0
  (crate-source "gix-utils" "0.2.0"
                "1l7qfjkw6lig08d7hb5jvjn5q9fv1717n2zygkapyglhrwj8g7qq"))

(define rust-gix-utils-0.3.0
  (crate-source "gix-utils" "0.3.0"
                "1ici87gkggdh9bklvz16zrq0sknq4r9lbd4ffail3brc2wmsylak"))

(define rust-gix-validate-0.10.0
  (crate-source "gix-validate" "0.10.0"
                "0kbdwvnbkcw6sq0mx6zis6nsc61v95n79v98ilw52nyymh6f1fbp"))

(define rust-gix-validate-0.7.7
  (crate-source "gix-validate" "0.7.7"
                "0h4hr3rpgwc7ixyynjp53s9il3sb0gq8ad332k8drwyfn8vkg6xs"))

(define rust-gix-validate-0.8.5
  (crate-source "gix-validate" "0.8.5"
                "1kqad8a2wdz69ma7hspi21pazgpkrc5hg4iw37gsvca99b9pvhl2"))

(define rust-gix-validate-0.9.4
  (crate-source "gix-validate" "0.9.4"
                "11204daz5qlk9kqnmiq4syv0n21phkiy3xkwxmwnrnh964jz3d9l"))

(define rust-gix-worktree-0.34.1
  (crate-source "gix-worktree" "0.34.1"
                "19p7xzrl1i4jw2fhzywqkfd00dn58k9nksll0qi7548bprp35xr6"))

(define rust-gix-worktree-0.37.0
  (crate-source "gix-worktree" "0.37.0"
                "177j311n46ysiyb52x68rwf02lp7gnavy4p9l17zwl1ma9dmwd0d"))

(define rust-gix-worktree-0.38.0
  (crate-source "gix-worktree" "0.38.0"
                "04za9fgw10w0r0sfqnjd6sji3xcwiwgr9asy1ma25yl8a7hvnvbm"))

(define rust-gix-worktree-0.39.0
  (crate-source "gix-worktree" "0.39.0"
                "0n49fywzh1f4gmv7gwd4d5jnq7ahiabsdv6wda3scmxagqpm2wv6"))

(define rust-gix-worktree-0.40.0
  (crate-source "gix-worktree" "0.40.0"
                "0yvjvjs1y6q7qikphvpj85v92sya3pac1b9hxm7jgakr9fy0sxpp"))

(define rust-gix-worktree-0.41.0
  (crate-source "gix-worktree" "0.41.0"
                "1sa7r2l7cj53s2npffw78sv4d1qaswypfzcp1hq6i0ljimpr3wal"))

(define rust-gix-worktree-0.42.0
  (crate-source "gix-worktree" "0.42.0"
                "1cbp70w7vhss4sxrgd1m3vi8fyfx7iqd5inv0vprph96j6n2bxjm"))

(define rust-gix-worktree-state-0.17.0
  (crate-source "gix-worktree-state" "0.17.0"
                "1w2vaz776v13hrnzhnsihmcbhb6883b33gc3cq475yasmncy3xc6"))

(define rust-gix-worktree-state-0.18.0
  (crate-source "gix-worktree-state" "0.18.0"
                "12z1h01r20xbpy56ashd3161mv243f4a62l4cqs5nwy2iv9v83j9"))

(define rust-gix-worktree-state-0.20.0
  (crate-source "gix-worktree-state" "0.20.0"
                "1y3v41sk3ay8sglzrrxi3mibvygp000v55qih0jjph5crcbrpfh6"))

(define rust-gix-worktree-stream-0.20.0
  (crate-source "gix-worktree-stream" "0.20.0"
                "0qzai5a0hxcsv2b0jbd5ja89ps5xk0m3grgqwvhaqdpi4fdxcqh0"))

(define rust-gl-generator-0.14.0
  (crate-source "gl_generator" "0.14.0"
                "0k8j1hmfnff312gy7x1aqjzcm8zxid7ij7dlb8prljib7b1dz58s"))

(define rust-glam-0.30.4
  (crate-source "glam" "0.30.4"
                "04bbd6iv3wvan0x9hx4bmqb9y1rkkazqajzs9ib3c9xpasz9vaah"))

(define rust-glam-0.30.5
  (crate-source "glam" "0.30.5"
                "11cd6rdpbqx1jg4hc6bp781ii3gchvamwpnasq0cxgb3csqamlgj"))

(define rust-glib-0.18.2
  (crate-source "glib" "0.18.2"
                "06dxrhispzz20n33b5k2gg723p27rprc87xbxb2ng06f07xnlc8w"))

(define rust-glib-0.18.5
  (crate-source "glib" "0.18.5"
                "1r8fw0627nmn19bgk3xpmcfngx3wkn7mcpq5a8ma3risx3valg93"))

(define rust-glib-0.19.9
  (crate-source "glib" "0.19.9"
                "0i2ak1scmzfmfxbm2dr146jl4y9mafxf1ald05jr8iimy5wh4r9r"))

(define rust-glib-0.20.10
  (crate-source "glib" "0.20.10"
                "1hljzcng4v2zrhagznkxg9fjmjj3lfjh7n7amk6j6archjaw80f5"))

(define rust-glib-0.20.12
  (crate-source "glib" "0.20.12"
                "10ynn8aiabbzrsgdswmqpr47sapfkbfn5rfxsy26swflabivdi7z"))

(define rust-glib-0.9.3
  (crate-source "glib" "0.9.3"
                "1h3100mf7kdfxibjz5na0sqzbd2mcsyd8pzivn3666w414x5gys0"))

(define rust-glib-macros-0.18.2
  (crate-source "glib-macros" "0.18.2"
                "0iqyzyarmgvm2jlfglrpqkpmsialy7y571jic0ix8dmi48w91npq"))

(define rust-glib-macros-0.18.5
  (crate-source "glib-macros" "0.18.5"
                "1p5cla53fcp195zp0hkqpmnn7iwmkdswhy7xh34002bw8y7j5c0b"))

(define rust-glib-macros-0.19.9
  (crate-source "glib-macros" "0.19.9"
                "1mzsh8jkg8vldvgvr9gsaidvn2myn5cbdn8a6m8rgbhlg8kv0aa4"))

(define rust-glib-macros-0.20.10
  (crate-source "glib-macros" "0.20.10"
                "1jbvxmzsqcinsa1xhzmfb2i61c9x8yxday2dnyrw91wqwafdrrpb"))

(define rust-glib-macros-0.20.12
  (crate-source "glib-macros" "0.20.12"
                "0ibi9vbpbw9vvl9ax60kxq07d7a21k0jj5lva8zmliq95zv4l278"))

(define rust-glib-sys-0.10.1
  (crate-source "glib-sys" "0.10.1"
                "1hfdwilf3dnrv3pnfbwmp3h2afgwvfsapcgjfg8276kflsbvksf7"))

(define rust-glib-sys-0.18.1
  (crate-source "glib-sys" "0.18.1"
                "164qhsfmlzd5mhyxs8123jzbdfldwxbikfpq5cysj3lddbmy4g06"))

(define rust-glib-sys-0.19.8
  (crate-source "glib-sys" "0.19.8"
                "19f4q8x77vd7c1d9ikw492yskq5kpd7k04qb8xnh1c427a6w2baw"))

(define rust-glib-sys-0.20.10
  (crate-source "glib-sys" "0.20.10"
                "05f29ky5dnvy8vp5rdld5f8r2lgr5w7dxqr7p27km016s4g9xdwa"))

(define rust-glib-sys-0.9.1
  (crate-source "glib-sys" "0.9.1"
                "1qhnwfqqcp63mx4q9744rfkq78g6ky2j8ppsxxgw0ipl08w6z1cm"))

(define rust-glob-0.2.11
  (crate-source "glob" "0.2.11"
                "1ysvi72slkw784fcsymgj4308c3y03gwjjzqxp80xdjnkbh8vqcb"))

(define rust-glob-0.3.1
  (crate-source "glob" "0.3.1"
                "16zca52nglanv23q5qrwd5jinw3d3as5ylya6y1pbx47vkxvrynj"))

(define rust-glob-0.3.2
  (crate-source "glob" "0.3.2"
                "1cm2w34b5w45fxr522h5b0fv1bxchfswcj560m3pnjbia7asvld8"))

(define rust-glob-0.3.3
  (crate-source "glob" "0.3.3"
                "106jpd3syfzjfj2k70mwm0v436qbx96wig98m4q8x071yrq35hhc"))

(define rust-glob-match-0.2.1
  (crate-source "glob-match" "0.2.1"
                "178bjn684dd50px9n8lwa72fn94566d9wmcp86m9h8a17d8ck1cr"))

(define rust-globset-0.4.16
  (crate-source "globset" "0.4.16"
                "1xa9ivqs74imf1q288spxh49g6iw2mn3x9snibdgapazzj6h58al"))

(define rust-globwalk-0.8.1
  (crate-source "globwalk" "0.8.1"
                "1k6xwkydr7igvwjn3xkwjywk4213lcs53f576ilqz1h84jaazqwk"))

(define rust-globwalk-0.9.1
  (crate-source "globwalk" "0.9.1"
                "0mz7bsa66p2rrgnz3l94ac4kbklh7mq8j30iizyxjy4qyvmn1xqb"))

(define rust-gloo-timers-0.3.0
  (crate-source "gloo-timers" "0.3.0"
                "1519157n7xppkk6pdw5w52vy1llzn5iljkqd7q1h5609jv7l7cdv"))

(define rust-glow-0.16.0
  (crate-source "glow" "0.16.0"
                "022z12nlyfpy36fvp2szq792xix1xbgkznpmicf1c404sxhfmrf5"))

(define rust-glutin-0.32.2
  (crate-source "glutin" "0.32.2"
                "0j28jyhkwp4hjwji01n8m8nxyx8za643xvmhvs926qnf1j5jnr03"))

(define rust-glutin-egl-sys-0.7.1
  (crate-source "glutin_egl_sys" "0.7.1"
                "1lh2rj77yvdqjx913nrf7xs5h3ialkkldfn3ppz29x4mc6x80ijc"))

(define rust-glutin-glx-sys-0.6.1
  (crate-source "glutin_glx_sys" "0.6.1"
                "118ifprw3y4jwrr25x862gh9hwd7fniwpywr4ihqpa25h29v4ywa"))

(define rust-glutin-wgl-sys-0.6.1
  (crate-source "glutin_wgl_sys" "0.6.1"
                "0gng2810jb5x133lmy17qifjx6s90lnprm86afg7mfls505y0kic"))

(define rust-gobject-sys-0.10.0
  (crate-source "gobject-sys" "0.10.0"
                "1311d3zbdhl1g7ibj1iy1650513yrhxbjxgfhazn52ii1jv368cm"))

(define rust-gobject-sys-0.18.0
  (crate-source "gobject-sys" "0.18.0"
                "0i6fhp3m6vs3wkzyc22rk2cqj68qvgddxmpaai34l72da5xi4l08"))

(define rust-gobject-sys-0.19.8
  (crate-source "gobject-sys" "0.19.8"
                "17lb7dfbpcg8zchwlfbc08kckwf0a7d9n5ly3pyic13f5ljpws9f"))

(define rust-gobject-sys-0.20.10
  (crate-source "gobject-sys" "0.20.10"
                "1niyqv22b2c38ks33i4isas4v83d3w7jx3xzzly9x63kpfacm6pc"))

(define rust-gobject-sys-0.9.1
  (crate-source "gobject-sys" "0.9.1"
                "1nakflbp3gjaas4fw7sn3p1p32khyfpcq1h06z7yqd10yq2ail9i"))

(define rust-goblin-0.6.1
  (crate-source "goblin" "0.6.1"
                "0s7zs27b192virbp88y2fgq8p6nb8blkn7byqyl4cv7bm3j4ssqd"))

(define rust-goblin-0.7.1
  (crate-source "goblin" "0.7.1"
                "0d11fk9bdxzf228xpr8v6d6a01dib00khjg5bldk9kf2d51inz7j"))

(define rust-goblin-0.9.3
  (crate-source "goblin" "0.9.3"
                "0ifpcsp0hpp7lx10yqln9ybmfkky7gig9idlhc2j7sx7456sd86s"))

(define rust-governor-0.6.3
  (crate-source "governor" "0.6.3"
                "0yw66yb1rfc7np23n9af9sb8kbhv3jnhvg3an1rsydbbxr1gb9v8"
                #:snippet '(delete-file-recursively "doc")))

(define rust-graphene-rs-0.18.1
  (crate-source "graphene-rs" "0.18.1"
                "00f4q1ra4haap5i7lazwhkdgnb49fs8adk2nm6ki6mjhl76jh8iv"))

(define rust-graphene-rs-0.20.10
  (crate-source "graphene-rs" "0.20.10"
                "16six67j0j57ynv7frxiwnsf7dslhyy67ppirad1q98lgnnxz1kb"))

(define rust-graphene-sys-0.18.1
  (crate-source "graphene-sys" "0.18.1"
                "0n8zlg7z26lwpnvlqp1hjlgrs671skqwagdpm7r8i1zwx3748hfc"))

(define rust-graphene-sys-0.20.10
  (crate-source "graphene-sys" "0.20.10"
                "1sk1736b4vay2hj9qz56c0pvqa3v0mkdch3yg7hiapidpa2kln6z"))

(define rust-grass-0.13.4
  (crate-source "grass" "0.13.4"
                "151wydlb0543hcyv00g06wiakrp6p5xnr3g4703qzxky8cb859pp"))

(define rust-grass-compiler-0.13.4
  (crate-source "grass_compiler" "0.13.4"
                "1xngzb4h04fkdhyagrwcqanrl7arghj3v5sl84cfab12y3vkv7id"))

(define rust-greetd-ipc-0.10.3
  (crate-source "greetd_ipc" "0.10.3"
                "0y1095qwmiwpm94z3vgbklbnlp088pi6vd1isp8l584izidjw3bh"))

(define rust-grep-0.2.13
  (crate-source "grep" "0.2.13"
                "1i864d726laskszdpm3fdjdcmfipf9d5y4rjhaby6bzg348g0ydx"))

(define rust-grep-0.3.2
  (crate-source "grep" "0.3.2"
                "1rij3x9xi70xmiqgjcp3iib4hrvmgcn22cvgm14xfa2ffd4yg2ih"))

(define rust-grep-cli-0.1.11
  (crate-source "grep-cli" "0.1.11"
                "1h6fwhkbxxvp4wpsh9vr6lms5lpwwcblryi697w7kwh61s7jiwa7"))

(define rust-grep-matcher-0.1.7
  (crate-source "grep-matcher" "0.1.7"
                "00mcjar5b6y1pwf0gjdywzgh1fnp6jl612n9qznwyfm420d198s7"))

(define rust-grep-pcre2-0.1.8
  (crate-source "grep-pcre2" "0.1.8"
                "0imybpfr9vpwna4qw30gxp9bs9j52f5152p06ak6x1sz0dp8cvyq"))

(define rust-grep-printer-0.1.7
  (crate-source "grep-printer" "0.1.7"
                "0sjsv4v7a88arckc880ns3mbn2y2p38mpnkh9dviznj9ixbm2ig1"))

(define rust-grep-printer-0.2.2
  (crate-source "grep-printer" "0.2.2"
                "1zhkw3agp0sy2y37chvg0rylj1xk6kvyr0mbhd6sm4d8wh5124n1"))

(define rust-grep-regex-0.1.13
  (crate-source "grep-regex" "0.1.13"
                "0zhzz5hffx30ff7xi3p4m3amg5c4rr2i7a6kdfifg5ijgry19pcy"))

(define rust-grep-searcher-0.1.14
  (crate-source "grep-searcher" "0.1.14"
                "1nici6klcniqph272i323rlrwl60vqqk4kb0sq3s3q627x5w3dmr"))

(define rust-group-0.13.0
  (crate-source "group" "0.13.0"
                "0qqs2p5vqnv3zvq9mfjkmw3qlvgqb0c3cm6p33srkh7pc9sfzygh"))

(define rust-gsk4-0.7.3
  (crate-source "gsk4" "0.7.3"
                "0zhzs2dkgiinhgc11akpn2harq3x5n1iq21dnc4h689g3lsqx58d"))

(define rust-gsk4-0.9.6
  (crate-source "gsk4" "0.9.6"
                "0mgqq5m6cm4q7ajjgw92z13z2ikpvh6zx2gwzdjrz30wjcpygxb1"))

(define rust-gsk4-sys-0.7.3
  (crate-source "gsk4-sys" "0.7.3"
                "0mbdlm9qi1hql48rr29vsj9vlqwc7gxg67wg1q19z67azwz9xg8j"))

(define rust-gsk4-sys-0.9.6
  (crate-source "gsk4-sys" "0.9.6"
                "1p1n4jhhxyvj7hb0cqhzvazrck0qw81sz36ydfj8avzsapg5jl3m"))

(define rust-gst-plugin-gtk4-0.11.4
  (crate-source "gst-plugin-gtk4" "0.11.4"
                "1hmky9p16hnhrwm5i63ynlwfl1bpc9fp3as5ibrni1qlfq0vhwdj"))

(define rust-gst-plugin-version-helper-0.8.2
  (crate-source "gst-plugin-version-helper" "0.8.2"
                "0alv0v7jfg7ryybb3qnbdwx3nqzkdl305il1xk92y9b02r7qfpjf"))

(define rust-gstreamer-0.21.3
  (crate-source "gstreamer" "0.21.3"
                "0mchpvvll5i4ck8zr7aarrz6p975n0dcyy92wksg8ycf9hzp15fy"))

(define rust-gstreamer-audio-0.21.3
  (crate-source "gstreamer-audio" "0.21.3"
                "0b91wjhhq0harwl7kyfv5l0kwp6w1vklpnpynakv92f8x6jci5vs"))

(define rust-gstreamer-audio-sys-0.21.1
  (crate-source "gstreamer-audio-sys" "0.21.1"
                "1lamp4s9cl0hhpbfcwdprn36fll6qq4xihrqbf2pfwqpifp99gbq"))

(define rust-gstreamer-base-0.21.2
  (crate-source "gstreamer-base" "0.21.2"
                "0zqnld0w2jqkz1m5xna3a3nnrpvrchpcrrzdgwim54540ilhn5fb"))

(define rust-gstreamer-base-sys-0.21.1
  (crate-source "gstreamer-base-sys" "0.21.1"
                "0r2bb4rmkpxs1l2jy61rn2srqzsp1f8q0k5j55di3zkqj0gp1jpl"))

(define rust-gstreamer-gl-0.21.2
  (crate-source "gstreamer-gl" "0.21.2"
                "140vnlxnkq12c8qqgc5i2y9wdz8c8dga25d99021cg16508vkkry"))

(define rust-gstreamer-gl-egl-0.21.2
  (crate-source "gstreamer-gl-egl" "0.21.2"
                "10nwlmyw1z4jccyrbqijx6iny2c64164jaz05dgnvi5378ianwx1"))

(define rust-gstreamer-gl-egl-sys-0.21.2
  (crate-source "gstreamer-gl-egl-sys" "0.21.2"
                "0m79m0lbk2s89cm4vc6ckwkgs9khmh2ri7x3gfgmz2hwy2v8hg7f"))

(define rust-gstreamer-gl-sys-0.21.2
  (crate-source "gstreamer-gl-sys" "0.21.2"
                "1kgi8rrlw2qx1p6q9ybk52wxpjn5wscx84lqfg4ng9lr7hdrg06m"))

(define rust-gstreamer-gl-wayland-0.21.1
  (crate-source "gstreamer-gl-wayland" "0.21.1"
                "1zz7as2qlf021dvpy1qs4rbahf94p6jb1msmfsgx08nhyai7dhpy"))

(define rust-gstreamer-gl-wayland-sys-0.21.1
  (crate-source "gstreamer-gl-wayland-sys" "0.21.1"
                "188j8i47zn93gph6ngmpjqbvm44jy0wzybr5052s6lxadzqqcywi"))

(define rust-gstreamer-gl-x11-0.21.1
  (crate-source "gstreamer-gl-x11" "0.21.1"
                "0zgn5aih3bcz3ci3xkdpc2jzxrxiz1hdpkwq121w5ln96ag1n0np"))

(define rust-gstreamer-gl-x11-sys-0.21.1
  (crate-source "gstreamer-gl-x11-sys" "0.21.1"
                "1p5wdrn3h55jx15963z1wnk7whwplpjfymy5yjsmkqdrqw1yz6n4"))

(define rust-gstreamer-pbutils-0.21.3
  (crate-source "gstreamer-pbutils" "0.21.3"
                "0idr354x9j77q8qrb99r6m6hrpa0z8j97jncqim5m08vhgbij9sb"))

(define rust-gstreamer-pbutils-sys-0.21.0
  (crate-source "gstreamer-pbutils-sys" "0.21.0"
                "0scx3w02wkyvmq76ia2jr6zhkf24zivn9vyphrcwmj2b8piydakg"))

(define rust-gstreamer-sys-0.21.2
  (crate-source "gstreamer-sys" "0.21.2"
                "1i1vrqs9ys5y0ljl4nxh1x25dnwlcyh9hiybh4dysviy5dwdlk2n"))

(define rust-gstreamer-video-0.21.2
  (crate-source "gstreamer-video" "0.21.2"
                "1r8mhzb1bq4dnj08f4szgarxd2fvqbakwv400fp9hyiv3m6jlnz8"))

(define rust-gstreamer-video-sys-0.21.2
  (crate-source "gstreamer-video-sys" "0.21.2"
                "1vhn7fiibwc2q5h8cjlg44imh8i0xss6nns83r859c76k26k20h3"))

(define rust-gtk-0.8.1
  (crate-source "gtk" "0.8.1"
                "13ygzblfv40l2kp70rnjymz7vk2g2wdjs04lhmk9q8wh0bbyiqc7"))

(define rust-gtk-sys-0.10.0
  (crate-source "gtk-sys" "0.10.0"
                "0mq4i161kk6dwiz19ayxgm9fhx7n3r5lm9lbjiyk0qs811pxmb49"))

(define rust-gtk-sys-0.9.2
  (crate-source "gtk-sys" "0.9.2"
                "1hd4w49iaivzjkbxi0bhabqp1ifkzg9g47n822sh12xlqxhgdpjk"))

(define rust-gtk4-0.7.3
  (crate-source "gtk4" "0.7.3"
                "0hh8nzglmz94v1m1h6vy8z12m6fr7ia467ry0md5fa4p7sm53sss"))

(define rust-gtk4-0.9.6
  (crate-source "gtk4" "0.9.6"
                "078911sc8wvnihlz3kq80chl0miz9z2g7rnds17rjc7ha484j75g"))

(define rust-gtk4-0.9.7
  (crate-source "gtk4" "0.9.7"
                "1mi6lcwm25jz7lznrb9glaabgyk40hnvkg4fzaxlf762080xsx7j"))

(define rust-gtk4-macros-0.7.2
  (crate-source "gtk4-macros" "0.7.2"
                "0bw3cchiycf7dw1bw4p8946gv38azxy05a5w0ndgcmxnz6fc8znm"))

(define rust-gtk4-macros-0.9.5
  (crate-source "gtk4-macros" "0.9.5"
                "169rqfxfczivcpz7019slsrpkx8crqjka43ymxmikp838xn7il8f"))

(define rust-gtk4-sys-0.7.3
  (crate-source "gtk4-sys" "0.7.3"
                "1f2ylskyqkjdik9fij2m46pra4jagnif5xyalbxfk3334fmc9n2l"))

(define rust-gtk4-sys-0.9.6
  (crate-source "gtk4-sys" "0.9.6"
                "1mh3xjkjb99y97z234cvyar08vcr7zblg1nrw48c6xsdwl0kpq21"))

(define rust-guess-host-triple-0.1.4
  (crate-source "guess_host_triple" "0.1.4"
                "062fs1jkg157240cdn6dw11zm1c02llm9kp95knq6b4s6iijgmjx"))

(define rust-guppy-0.17.19
  (crate-source "guppy" "0.17.19"
                "00xvglkibbnl8zdl3mnc152jzmi6r1p2gsh49qrj07a6j57bppw1"))

(define rust-guppy-workspace-hack-0.1.0
  (crate-source "guppy-workspace-hack" "0.1.0"
                "106pg6ifjq92rz5xbbv0aw4xchl1fkikpjry72p0nxczv620cqlj"))

(define rust-gzp-0.11.3
  (crate-source "gzp" "0.11.3"
                "1bvvz969c9kpyp7h6ry9mzhk7lb4hj4hpd810n0i26jjk4c5vip7"))

(define rust-h2-0.3.26
  (crate-source "h2" "0.3.26"
                "1s7msnfv7xprzs6xzfj5sg6p8bjcdpcqcmjjbkd345cyi1x55zl1"))

(define rust-h2-0.3.27
  (crate-source "h2" "0.3.27"
                "0b92141hilij015av6i5ziw9xfx4py3lbjy17yc35z5ih01sbv0b"))

(define rust-h2-0.4.10
  (crate-source "h2" "0.4.10"
                "19f0va87lhzrc0lmwkgcz1z0haf6glajb4icp0b7n50vdmkilhm9"))

(define rust-h2-0.4.11
  (crate-source "h2" "0.4.11"
                "118771sqbsa6cn48y9waxq24jx80f5xy8af0lq5ixq7ifsi51nhp"))

(define rust-h2-0.4.12
  (crate-source "h2" "0.4.12"
                "11hk5mpid8757z6n3v18jwb62ikffrgzjlrgpzqvkqdlzjfbdh7k"))

(define rust-h2-0.4.8
  (crate-source "h2" "0.4.8"
                "1hp3lijg1br982kzgglb5ks2ibg68a76z3rl052r8c5vyi7jj5sh"))

(define rust-h2-0.4.9
  (crate-source "h2" "0.4.9"
                "0cv6x0lndzxx59g49bhpl04zivfk0km9rzk9xs6iylrh80a9s93m"))

(define rust-half-1.8.3
  (crate-source "half" "1.8.3"
                "00q4ki8ycdswapw6xn1q89vr7rzi1c8m99igps0lx1i1gzhyshqv"))

(define rust-half-2.5.0
  (crate-source "half" "2.5.0"
                "1ldv2i761fjqxl4rn033nasjrdnw5ysnc1xalsfkfl5skc9zzckx"))

(define rust-half-2.6.0
  (crate-source "half" "2.6.0"
                "1j83v0xaqvrw50ppn0g33zig0zsbdi7xiqbzgn7sd5al57nrd4a5"))

(define rust-handlebars-4.5.0
  (crate-source "handlebars" "4.5.0"
                "09dj4rk6r9ngy1ki34mppcqq4pcnlhjd02yhnf724qpkkympp9ps"))

(define rust-handlebars-6.3.2
  (crate-source "handlebars" "6.3.2"
                "1630l9p8ag2f6lzj3gzwfjzvav46897kkv68j08wp1rjx9d2v7km"))

(define rust-hash32-0.2.1
  (crate-source "hash32" "0.2.1"
                "0rrbv5pc5b1vax6j6hk7zvlrpw0h6aybshxy9vbpgsrgfrc5zhxh"))

(define rust-hash32-0.3.1
  (crate-source "hash32" "0.3.1"
                "01h68z8qi5gl9lnr17nz10lay8wjiidyjdyd60kqx8ibj090pmj7"))

(define rust-hashbag-0.1.12
  (crate-source "hashbag" "0.1.12"
                "14wx5rx11h0nb1rmlwmiw7719q2qf941x7ipcdg8yahb0sr99x4q"))

(define rust-hashbrown-0.1.8
  (crate-source "hashbrown" "0.1.8"
                "1np350nrzysy021ndn2135q5vpzrp5nli78ywz114d1vcnv2kbiv"))

(define rust-hashbrown-0.11.2
  (crate-source "hashbrown" "0.11.2"
                "0vkjsf5nzs7qcia5ya79j9sq2p1caz4crrncr1675wwyj3ag0pmb"))

(define rust-hashbrown-0.12.3
  (crate-source "hashbrown" "0.12.3"
                "1268ka4750pyg2pbgsr43f0289l5zah4arir2k4igx5a8c6fg7la"))

(define rust-hashbrown-0.13.2
  (crate-source "hashbrown" "0.13.2"
                "03ji3n19j4b6mf2wlla81vsixcmlivglp6hgk79d1pcxfcrw38s3"))

(define rust-hashbrown-0.14.0
  (crate-source "hashbrown" "0.14.0"
                "0yj3nf0w30pf30w503kgaw4sbjnh62l5cbmc7dd0mnczzywh2qic"))

(define rust-hashbrown-0.14.5
  (crate-source "hashbrown" "0.14.5"
                "1wa1vy1xs3mp11bn3z9dv0jricgr6a2j0zkf1g19yz3vw4il89z5"))

(define rust-hashbrown-0.15.2
  (crate-source "hashbrown" "0.15.2"
                "12dj0yfn59p3kh3679ac0w1fagvzf4z2zp87a13gbbqbzw0185dz"))

(define rust-hashbrown-0.15.3
  (crate-source "hashbrown" "0.15.3"
                "1cwfw1yzkvsqkhmkg5igdvgsl8a0wyi716cn83k2j8h09ma6rcl4"))

(define rust-hashbrown-0.15.4
  (crate-source "hashbrown" "0.15.4"
                "1mg045sm1nm00cwjm7ndi80hcmmv1v3z7gnapxyhd9qxc62sqwar"))

(define rust-hashbrown-0.15.5
  (crate-source "hashbrown" "0.15.5"
                "189qaczmjxnikm9db748xyhiw04kpmhm9xj9k9hg0sgx7pjwyacj"))

(define rust-hashbrown-0.16.0
  (crate-source "hashbrown" "0.16.0"
                "13blh9j2yv77a6ni236ixiwdzbc1sh2bc4bdpaz7y859yv2bs6al"))

(define rust-hashlink-0.10.0
  (crate-source "hashlink" "0.10.0"
                "1h8lzvnl9qxi3zyagivzz2p1hp6shgddfmccyf6jv7s1cdicz0kk"))

(define rust-hashlink-0.8.4
  (crate-source "hashlink" "0.8.4"
                "1xy8agkyp0llbqk9fcffc1xblayrrywlyrm2a7v93x8zygm4y2g8"))

(define rust-hashlink-0.9.1
  (crate-source "hashlink" "0.9.1"
                "1byq4nyrflm5s6wdx5qwp96l1qbp2d0nljvrr5yqrsfy51qzz93b"))

(define rust-hayagriva-0.8.1
  ;; NB: The CBOR files contained here are human-editable with free tools, so
  ;; they should be ok to distribute.
  (crate-source "hayagriva" "0.8.1"
                "1n7frakfqq5igpwwqwnqd4dw57i87vac45zrlhlvmz5p9dahfjcm"))

(define rust-hdrhistogram-7.5.4
  (crate-source "hdrhistogram" "7.5.4"
                "07ai0r66l1n53f2757gv07za1l5g1bprb7zz4v75kpbky6c92p3n"
                #:snippet '(delete-file-recursively "tests")))

(define rust-headers-0.3.9
  (crate-source "headers" "0.3.9"
                "0w62gnwh2p1lml0zqdkrx9dp438881nhz32zrzdy61qa0a9kns06"))

(define rust-headers-0.4.0
  (crate-source "headers" "0.4.0"
                "1abari69kjl2yv2dg06g2x17qgd1a20xp7aqmmg2vfhcppk0c89j"))

(define rust-headers-core-0.2.0
  (crate-source "headers-core" "0.2.0"
                "0ab469xfpd411mc3dhmjhmzrhqikzyj8a17jn5bkj9zfpy0n9xp7"))

(define rust-headers-core-0.3.0
  (crate-source "headers-core" "0.3.0"
                "1r1w80i2bhmyh8s5mjr2dz6baqlrm6cak6yvzm4jq96lacjs5d2l"))

(define rust-heapless-0.7.17
  (crate-source "heapless" "0.7.17"
                "0kwn2wzk9fnsqnwp6rqjqhvh6hfq4rh225xwqjm72b5n1ry4bind"))

(define rust-heapless-0.8.0
  (crate-source "heapless" "0.8.0"
                "1b9zpdjv4qkl2511s2c80fz16fx9in4m9qkhbaa8j73032v9xyqb"))

(define rust-heck-0.3.3
  (crate-source "heck" "0.3.3"
                "0b0kkr790p66lvzn9nsmfjvydrbmh9z5gb664jchwgw64vxiwqkd"))

(define rust-heck-0.4.1
  (crate-source "heck" "0.4.1"
                "1a7mqsnycv5z4z5vnv1k34548jzmc0ajic7c1j8jsaspnhw5ql4m"))

(define rust-heck-0.5.0
  (crate-source "heck" "0.5.0"
                "1sjmpsdl8czyh9ywl3qcsfsq9a307dg4ni2vnlwgnzzqhc4y0113"))

(define rust-hecs-0.10.5
  (crate-source "hecs" "0.10.5"
                "0vbp7jk6fi2m9x3hdyanix9nd5nmi9zi71d90v9b95wdxrswdjz1"))

(define rust-hecs-macros-0.10.0
  (crate-source "hecs-macros" "0.10.0"
                "0w69bxb8pxd3m9zymsf0l0psfl30ny70kljwc2160cnw29dw4bq5"))

(define rust-hermit-abi-0.1.19
  (crate-source "hermit-abi" "0.1.19"
                "0cxcm8093nf5fyn114w8vxbrbcyvv91d4015rdnlgfll7cs6gd32"))

(define rust-hermit-abi-0.3.2
  (crate-source "hermit-abi" "0.3.2"
                "12v66gy77sqrgmjlx01w9p054nvz4mnhbd6xaazkxnddrp448ca4"))

(define rust-hermit-abi-0.3.9
  (crate-source "hermit-abi" "0.3.9"
                "092hxjbjnq5fmz66grd9plxd0sh6ssg5fhgwwwqbrzgzkjwdycfj"))

(define rust-hermit-abi-0.4.0
  (crate-source "hermit-abi" "0.4.0"
                "1k1zwllx6nfq417hy38x4akw1ivlv68ymvnzyxs76ffgsqcskxpv"))

(define rust-hermit-abi-0.5.0
  (crate-source "hermit-abi" "0.5.0"
                "0zp3khi7bl6x8gazm9i9dgjf4h47mj10v4j18i3823y3bkz81mzv"))

(define rust-hermit-abi-0.5.1
  (crate-source "hermit-abi" "0.5.1"
                "026bh0y8gpfd62gjm7gx6nyf6bgdyxdn0jc67i1ysl37hm3cwm7i"))

(define rust-hermit-abi-0.5.2
  (crate-source "hermit-abi" "0.5.2"
                "1744vaqkczpwncfy960j2hxrbjl1q01csm84jpd9dajbdr2yy3zw"))

(define rust-hex-0.4.3
  (crate-source "hex" "0.4.3"
                "0w1a4davm1lgzpamwnba907aysmlrnygbqmfis2mqjx5m552a93z"))

(define rust-hex-literal-0.4.1
  (crate-source "hex-literal" "0.4.1"
                "0iny5inkixsdr41pm2vkqh3fl66752z5j5c0cdxw16yl9ryjdqkg"))

(define rust-hex-slice-0.1.4
  (crate-source "hex-slice" "0.1.4"
                "0k3ck81m9lz5wz3c7qqj3j3m5xa5pr599n41gbq58i91w04a74al"))

(define rust-hickory-client-0.24.4
  (crate-source "hickory-client" "0.24.4"
                "1pm6ypps966d994bm8b3l5wbj3c7xsv23k47vzqcc7wdrnjpjr8m"))

(define rust-hickory-proto-0.24.4
  (crate-source "hickory-proto" "0.24.4"
                "0j3j26brvg83a4xvdlvxxgkycdmaklbd2f6cag76cvyfr5kj0rcj"
                #:snippet '(delete-file-recursively "tests")))

(define rust-hickory-proto-0.25.2
  (crate-source "hickory-proto" "0.25.2"
                "00k5dk572p0bsrnvsqv1yi8mwfpgwyj7q8bgiacri083q1bgx9pq"
                #:snippet '(delete-file-recursively "tests")))

(define rust-hickory-resolver-0.24.4
  (crate-source "hickory-resolver" "0.24.4"
                "0bnjdykfip04lr2ja97aw2s9vrnyxn46cbxb8cbi23jjrahigcfb"))

(define rust-hickory-resolver-0.25.2
  (crate-source "hickory-resolver" "0.25.2"
                "0ypqxipl0dmggv442qn1xxh11lsjr65219wmmg949yqbkflsjqnw"))

(define rust-hickory-server-0.24.4
  (crate-source "hickory-server" "0.24.4"
                "05llmq6ncass4m3bwhmqpj4s3j7fl7k8kg6gihzqa1g3yjpph009"))

(define rust-hifijson-0.2.2
  (crate-source "hifijson" "0.2.2"
                "0hjwcn3hj7bmwqw084lbbvmwx7lnkfyid6b74wd0c30pwcyann4r"))

(define rust-histo-1.0.0
  (crate-source "histo" "1.0.0"
                "0v6znd33clam2b37rhn2pldd39l61605s1ivxzpjwdygi8f6mad6"))

(define rust-hkdf-0.12.4
  (crate-source "hkdf" "0.12.4"
                "1xxxzcarz151p1b858yn5skmhyrvn8fs4ivx5km3i1kjmnr8wpvv"
                #:snippet '(delete-file-recursively "tests")))

(define rust-hmac-0.11.0
  (crate-source "hmac" "0.11.0"
                "16z61aibdg4di40sqi4ks2s4rz6r29w4sx4gvblfph3yxch26aia"
                #:snippet '(delete-file-recursively "tests")))

(define rust-hmac-0.12.1
  (crate-source "hmac" "0.12.1"
                "0pmbr069sfg76z7wsssfk5ddcqd9ncp79fyz6zcm6yn115yc6jbc"
                #:snippet '(delete-file-recursively "tests")))

(define rust-home-0.5.11
  (crate-source "home" "0.5.11"
                "1kxb4k87a9sayr8jipr7nq9wpgmjk4hk4047hmf9kc24692k75aq"))

(define rust-home-0.5.9
  (crate-source "home" "0.5.9"
                "19grxyg35rqfd802pcc9ys1q3lafzlcjcv2pl2s5q8xpyr5kblg3"))

(define rust-homedir-0.3.4
  (crate-source "homedir" "0.3.4"
                "18kb7myfvzzixv02k066477k11zzbaj2yddarjbrcx65r1dvvnsv"))

(define rust-hostname-0.3.1
  (crate-source "hostname" "0.3.1"
                "0rz8yf70cvzl3nry71m4bz9w6x4j9kdz3qng6pnwhk2h20z1qwrw"))

(define rust-hostname-0.4.1
  (crate-source "hostname" "0.4.1"
                "0rbxryl68bwv8hkjdjd8f37kdb10fncgsqrqksv64qy7s4y20vx5"))

(define rust-html-escape-0.2.13
  (crate-source "html-escape" "0.2.13"
                "0xml3hswv0205fbm5iq7dqiwjkr6d245xkfppwi7wqjdfr4x86kd"))

(define rust-html5ever-0.25.2
  (crate-source "html5ever" "0.25.2"
                "0j1iw2yy3fglr5rxqfrghmhksxzpcfp8ps753qazqkaxisq3zhg5"))

(define rust-html5ever-0.27.0
  (crate-source "html5ever" "0.27.0"
                "1m24sbpk572f5qhhkj4kkxvsd64rn968s0vxwvqlds76w2pp2dy1"))

(define rust-html5ever-0.35.0
  (crate-source "html5ever" "0.35.0"
                "1m4yajw7slxqn0x3zdh3i9qlhb03vgdf2pq3la3l8rjbyz15inam"))

(define rust-htmlescape-0.3.1
  (crate-source "htmlescape" "0.3.1"
                "0qria8paf19qy5sgzzk3iiii9fp2j7spbhqf0zjxwrg7v9c500p9"))

(define rust-hts-sys-2.2.0
  (crate-source "hts-sys" "2.2.0"
                "1cmvdwssd6xjk6w1iigaj5rl9ibx4zaaskfb2ji2mlhw28f7z3g3"
                #:snippet
                '(begin
                   (for-each delete-file-recursively
                             '("htslib/htscodecs/tests" "htslib/test"))
                   (substitute* "Cargo.toml"
                     ;; Do not use zlib-ng; just use zlib.
                     (("\"zlib-ng\",") "")
                     (("\"static\",") "")
                     ;; No static libraries please in curl-sys.
                     (("\"static-curl\",") "")
                     (("\"static-ssl\",") "\"ssl\",")
                     ;; No static lzma.
                     (("\\[\"static\"\\]") "[]")))))

(define rust-http-0.2.12
  (crate-source "http" "0.2.12"
                "1w81s4bcbmcj9bjp7mllm8jlz6b31wzvirz8bgpzbqkpwmbvn730"))

(define rust-http-1.3.1
  (crate-source "http" "1.3.1"
                "0r95i5h7dr1xadp1ac9453w0s62s27hzkam356nyx2d9mqqmva7l"))

(define rust-http-auth-0.1.10
  (crate-source "http-auth" "0.1.10"
                "08l8z75cpda5y25cnd5fzgsahb35xn29qlgl9j12dy9f8sls83qm"))

(define rust-http-body-0.4.6
  (crate-source "http-body" "0.4.6"
                "1lmyjfk6bqk6k9gkn1dxq770sb78pqbqshga241hr5p995bb5skw"))

(define rust-http-body-1.0.1
  (crate-source "http-body" "1.0.1"
                "111ir5k2b9ihz5nr9cz7cwm7fnydca7dx4hc7vr16scfzghxrzhy"))

(define rust-http-body-util-0.1.3
  (crate-source "http-body-util" "0.1.3"
                "0jm6jv4gxsnlsi1kzdyffjrj8cfr3zninnxpw73mvkxy4qzdj8dh"))

(define rust-http-content-range-0.2.1
  (crate-source "http-content-range" "0.2.1"
                "1jmzikxrqmjhkyw69b894sapr1bkyc7y7ca3vl5xg5hlkw58xaml"))

(define rust-http-range-header-0.4.2
  (crate-source "http-range-header" "0.4.2"
                "171mszmmq0lzpj9brig4wz1sz8hh3h6dgmaxs69q2db8ibma4wci"))

(define rust-httparse-1.10.1
  (crate-source "httparse" "1.10.1"
                "11ycd554bw2dkgw0q61xsa7a4jn1wb1xbfacmf3dbwsikvkkvgvd"))

(define rust-httpdate-1.0.3
  (crate-source "httpdate" "1.0.3"
                "1aa9rd2sac0zhjqh24c9xvir96g188zldkx0hr6dnnlx5904cfyz"))

(define rust-httpmock-0.7.0
  (crate-source "httpmock" "0.7.0"
                "0nzxw15pn453siq4v9xfjgggvw2jry50y7qsxhnlf409xs39bv08"))

(define rust-human-date-parser-0.2.0
  (crate-source "human-date-parser" "0.2.0"
                "0qqwf20wz13ww06i1xqr9hfbg4c598f34n442q90qxzyxd6wy5hi"))

(define rust-human-format-1.1.0
  (crate-source "human_format" "1.1.0"
                "1f45y1vib7z96ij5xv623rxpdm3ap4bj11j84jqjg7a5iir1yfsw"))

(define rust-human-panic-1.2.3
  (crate-source "human-panic" "1.2.3"
                "0w7kbj2y3rrnax2bj4s3aashsm5lrg55wh58a44v7fr0k741dw64"))

(define rust-human-sort-0.2.2
  (crate-source "human-sort" "0.2.2"
                "06b2rj4hd4h4rf3mp5qk89ymlxh5iv3cpmrfgramwvay634hj2hl"))

(define rust-humansize-2.1.3
  (crate-source "humansize" "2.1.3"
                "1msxd1akb3dydsa8qs461sds9krwnn31szvqgaq93p4x0ad1rdbc"))

(define rust-humantime-1.3.0
  (crate-source "humantime" "1.3.0"
                "0krwgbf35pd46xvkqg14j070vircsndabahahlv3rwhflpy4q06z"))

(define rust-humantime-2.2.0
  (crate-source "humantime" "2.2.0"
                "17rz8jhh1mcv4b03wnknhv1shwq2v9vhkhlfg884pprsig62l4cv"))

(define rust-humantime-2.3.0
  (crate-source "humantime" "2.3.0"
                "092lpipp32ayz4kyyn4k3vz59j9blng36wprm5by0g2ykqr14nqk"))

(define rust-humantime-serde-1.1.1
  (crate-source "humantime-serde" "1.1.1"
                "0310ri4zb33qbwqv0n51xysfjpnwc6rgxscl5i09jgcjlmgdp8sp"))

(define rust-hyper-0.14.32
  (crate-source "hyper" "0.14.32"
                "1rvcb0smz8q1i0y6p7rwxr02x5sclfg2hhxf3g0774zczn0cgps1"))

(define rust-hyper-1.6.0
  (crate-source "hyper" "1.6.0"
                "103ggny2k31z0iq2gzwk2vbx601wx6xkpjpxn40hr3p3b0b5fayc"))

(define rust-hyper-1.7.0
  (crate-source "hyper" "1.7.0"
                "07n59pxzlq621z611cbpvh7p4h9h15v0r7m5wgxygpx02d5aafpb"))

(define rust-hyper-proxy2-0.1.0
  (crate-source "hyper-proxy2" "0.1.0"
                "1brzvj6v5rfzq17x6wbg41vcqhpwli87phhlf0f4mg5h7yrbfhwh"))

(define rust-hyper-rustls-0.24.2
  (crate-source "hyper-rustls" "0.24.2"
                "1475j4a2nczz4aajzzsq3hpwg1zacmzbqg393a14j80ff8izsgpc"))

(define rust-hyper-rustls-0.26.0
  (crate-source "hyper-rustls" "0.26.0"
                "0b4m1jvs147hxi8677n2dxxib663s7c31xmfni7b5qkanihsggm0"))

(define rust-hyper-rustls-0.27.5
  (crate-source "hyper-rustls" "0.27.5"
                "1cjr3yf3x5mr3194llsfibacl6j7n2dknii2dwjha4ysyf1ia69d"))

(define rust-hyper-rustls-0.27.6
  (crate-source "hyper-rustls" "0.27.6"
                "0va008pmz5h062wnh2h08d3r3iizvqnw68k5ji8frp0vw6aib803"))
(define rust-hyper-rustls-0.27.7
  (crate-source "hyper-rustls" "0.27.7"
                "0n6g8998szbzhnvcs1b7ibn745grxiqmlpg53xz206v826v3xjg3"))

(define rust-hyper-timeout-0.5.2
  (crate-source "hyper-timeout" "0.5.2"
                "1c431l5ckr698248yd6bnsmizjy2m1da02cbpmsnmkpvpxkdb41b"))

(define rust-hyper-tls-0.5.0
  (crate-source "hyper-tls" "0.5.0"
                "01crgy13102iagakf6q4mb75dprzr7ps1gj0l5hxm1cvm7gks66n"))

(define rust-hyper-tls-0.6.0
  (crate-source "hyper-tls" "0.6.0"
                "1q36x2yps6hhvxq5r7mc8ph9zz6xlb573gx0x3yskb0fi736y83h"))

(define rust-hyper-util-0.1.11
  (crate-source "hyper-util" "0.1.11"
                "1wj3svb1r6yv6kgk5fsz6wwajmngc4zxcw4wxpwlmpbgl8rvqys9"))

(define rust-hyper-util-0.1.12
  (crate-source "hyper-util" "0.1.12"
                "040pyd26pssbgvrwr35xjcghv77j7ir1ci0q8wy1v78d1saix7yg"))

(define rust-hyper-util-0.1.13
  (crate-source "hyper-util" "0.1.13"
                "1s06md3mq6v6w2zqq0qfag2hw8drsvmxpiqd4mwcl7njnfv97hmi"))

(define rust-hyper-util-0.1.14
  (crate-source "hyper-util" "0.1.14"
                "1nqvf5azmv8p7hs5ghjlbgfya7xaafq377vppdazxbq8zzdxybyw"))

(define rust-hyper-util-0.1.15
  (crate-source "hyper-util" "0.1.15"
                "1pyi2h8idwyadljs95gpihjvkfkmcxi5vn7s882vy0kg9jyxarkz"))

(define rust-hyper-util-0.1.16
  (crate-source "hyper-util" "0.1.16"
                "0pmw8gqkqjnsdrxdy5wd5q8z1gh7caxqk2an7b4s53byghkhb6wd"))

(define rust-hyper-util-0.1.17
  (crate-source "hyper-util" "0.1.17"
                "1a5fcnz0alrg4lx9xf6ja66ihaab58jnm5msnky804wg39cras9w"))

(define rust-hyperlocal-0.8.0
  (crate-source "hyperlocal" "0.8.0"
                "136978rsp0wr6x28cxivxhbq2np66l4jrq3n9xwckrrd5dxxzbqg"))

(define rust-hypher-0.1.5 rust-hypher-0.1)

(define rust-hyprland-0.3.13
  (crate-source "hyprland" "0.3.13"
                "1s6xi0k93nxi4xgw09k14xxvdygwqc5nr4fk9n4m3m35q0f8yypq"))

(define rust-hyprland-macros-0.3.4
  (crate-source "hyprland-macros" "0.3.4"
                "1sz4yhwi7d1s2wfpk9ql43dgmiq0j3j8skmrijpi55lp5vax6hf9"))

(define rust-i18n-config-0.4.7
  (crate-source "i18n-config" "0.4.7"
                "1j89hnd1d2zqfb1s9jgcd5vzf5a9iisfc18v324mpgn06540g24f"))

(define rust-i18n-embed-0.14.1
  (crate-source "i18n-embed" "0.14.1"
                "17251vazz8mybcgic9wffk56acv5i5zsg3x9kvdvjnsgfsams84l"))

(define rust-i18n-embed-fl-0.8.0
  (crate-source "i18n-embed-fl" "0.8.0"
                "1lq374kywpg9aqf556k9kz695arzkkwd3z06w4ak94lyyj0sfhc2"))

(define rust-i18n-embed-impl-0.8.4
  (crate-source "i18n-embed-impl" "0.8.4"
                "1hmnimlv310cirg8nx77nf8q1si4hq1yarkg5kyfc7rxabhc0b0g"))

(define rust-iana-time-zone-0.1.60
  (crate-source "iana-time-zone" "0.1.60"
                "0hdid5xz3jznm04lysjm3vi93h3c523w0hcc3xba47jl3ddbpzz7"))

(define rust-iana-time-zone-0.1.63
  (crate-source "iana-time-zone" "0.1.63"
                "1n171f5lbc7bryzmp1h30zw86zbvl5480aq02z92lcdwvvjikjdh"))

(define rust-iana-time-zone-0.1.64
  (crate-source "iana-time-zone" "0.1.64"
                "1yz980fmhaq9bdkasz35z63az37ci6kzzfhya83kgdqba61pzr9k"))

(define rust-iana-time-zone-haiku-0.1.2
  (crate-source "iana-time-zone-haiku" "0.1.2"
                "17r6jmj31chn7xs9698r122mapq85mfnv98bb4pg6spm0si2f67k"))

(define rust-ical-0.11.0
  (crate-source "ical" "0.11.0"
                "1xkrs9a48qzbzf6mbrnsvj9i51h2z44l7h7236d75dx88dssnz4v"))

(define rust-icalendar-0.16.13
  (crate-source "icalendar" "0.16.13"
                "1r620adkzpyc5vjcgvxxb7pb8zyvmz2gncl6qb5if432hgnblidb"))

(define rust-icu-calendar-1.5.2
  (crate-source "icu_calendar" "1.5.2"
                "1zx8fli5aif3w7zwx5gxijxbsjsphkwr339h99izfdlsgw9v4rbj"))

(define rust-icu-calendar-data-1.5.1
  (crate-source "icu_calendar_data" "1.5.1"
                "1qg9zag3scrrksdc7bqf2lcjcmdlwx2g9d080qcn48c5gvkrj142"))

(define rust-icu-collections-1.5.0
  (crate-source "icu_collections" "1.5.0"
                "09j5kskirl59mvqc8kabhy7005yyy7dp88jw9f6f3gkf419a8byv"))

(define rust-icu-collections-2.0.0
  (crate-source "icu_collections" "2.0.0"
                "0izfgypv1hsxlz1h8fc2aak641iyvkak16aaz5b4aqg3s3sp4010"))

(define rust-icu-datetime-1.5.1
  (crate-source "icu_datetime" "1.5.1"
                "100pdcq2y452py0bxa7vzy1sfibm11z2xxbpgvbkzpq8bswfy5fi"
                #:snippet '(delete-file-recursively "tests")))

(define rust-icu-datetime-data-1.5.1
  (crate-source "icu_datetime_data" "1.5.1"
                "0zahlvry7ajl92cdnnjp1rx0vzi7vf1p0sljg8dsng0jfr0g1xdy"))

(define rust-icu-decimal-1.5.0
  (crate-source "icu_decimal" "1.5.0"
                "1sxhx2jw01csjbn37mqv28pbn673wj28ipqybvc4h17chs7xk3zv"))

(define rust-icu-decimal-data-1.5.1
  (crate-source "icu_decimal_data" "1.5.1"
                "1qhah965n48j2izqx8bcm1pn8dkxxjb584cwga1nvkswgzcmvjb7"))

(define rust-icu-locale-core-2.0.0
  (crate-source "icu_locale_core" "2.0.0"
                "02phv7vwhyx6vmaqgwkh2p4kc2kciykv2px6g4h8glxfrh02gphc"))

(define rust-icu-locid-1.5.0
  (crate-source "icu_locid" "1.5.0"
                "0dznvd1c5b02iilqm044q4hvar0sqibq1z46prqwjzwif61vpb0k"))

(define rust-icu-locid-transform-1.5.0
  (crate-source "icu_locid_transform" "1.5.0"
                "0kmmi1kmj9yph6mdgkc7v3wz6995v7ly3n80vbg0zr78bp1iml81"))

(define rust-icu-locid-transform-data-1.5.0
  (crate-source "icu_locid_transform_data" "1.5.0"
                "0vkgjixm0wzp2n3v5mw4j89ly05bg3lx96jpdggbwlpqi0rzzj7x"))

(define rust-icu-locid-transform-data-1.5.1
  (crate-source "icu_locid_transform_data" "1.5.1"
                "07gignya9gzynnyds88bmra4blq9jxzgrcss43vzk2q9h7byc5bm"))

(define rust-icu-normalizer-1.5.0
  (crate-source "icu_normalizer" "1.5.0"
                "0kx8qryp8ma8fw1vijbgbnf7zz9f2j4d14rw36fmjs7cl86kxkhr"))

(define rust-icu-normalizer-2.0.0
  (crate-source "icu_normalizer" "2.0.0"
                "0ybrnfnxx4sf09gsrxri8p48qifn54il6n3dq2xxgx4dw7l80s23"))

(define rust-icu-normalizer-data-1.5.0
  (crate-source "icu_normalizer_data" "1.5.0"
                "05lmk0zf0q7nzjnj5kbmsigj3qgr0rwicnn5pqi9n7krmbvzpjpq"))

(define rust-icu-normalizer-data-1.5.1
  (crate-source "icu_normalizer_data" "1.5.1"
                "1dqcm86spcqcs4jnra81yqq3g5bpw6bvf5iz621spj5x52137s65"))

(define rust-icu-normalizer-data-2.0.0
  (crate-source "icu_normalizer_data" "2.0.0"
                "1lvjpzxndyhhjyzd1f6vi961gvzhj244nribfpdqxjdgjdl0s880"))

(define rust-icu-plurals-1.5.0
  (crate-source "icu_plurals" "1.5.0"
                "055iwgyxmdwm197wvb8d9x16lrhirn4c39dh072xbnr5q3kp0nms"))

(define rust-icu-plurals-data-1.5.1
  (crate-source "icu_plurals_data" "1.5.1"
                "04nipymyc2xjvszh12vvzs06scw02wczi9bpda3nlzfb70r410x4"))

(define rust-icu-properties-1.5.1
  (crate-source "icu_properties" "1.5.1"
                "1xgf584rx10xc1p7zjr78k0n4zn3g23rrg6v2ln31ingcq3h5mlk"))

(define rust-icu-properties-2.0.0
  (crate-source "icu_properties" "2.0.0"
                "16hk94ng18d1r8989yr4inswacfr2psad8hbr1cjzj21fa6clj95"))

(define rust-icu-properties-2.0.1
  (crate-source "icu_properties" "2.0.1"
                "0az349pjg8f18lrjbdmxcpg676a7iz2ibc09d2wfz57b3sf62v01"))

(define rust-icu-properties-data-1.5.0
  (crate-source "icu_properties_data" "1.5.0"
                "0scms7pd5a7yxx9hfl167f5qdf44as6r3bd8myhlngnxqgxyza37"))

(define rust-icu-properties-data-1.5.1
  (crate-source "icu_properties_data" "1.5.1"
                "1qm5vf17nyiwb87s3g9x9fsj32gkv4a7q7d2sblawx9vfncqgyw5"))

(define rust-icu-properties-data-2.0.0
  (crate-source "icu_properties_data" "2.0.0"
                "016yaw60mm4m21zr7chq9c00dv1vj1rf2jajv7vzhs3vwikfi5w1"))

(define rust-icu-properties-data-2.0.1
  (crate-source "icu_properties_data" "2.0.1"
                "0cnn3fkq6k88w7p86w7hsd1254s4sl783rpz4p6hlccq74a5k119"))

(define rust-icu-provider-1.5.0
  (crate-source "icu_provider" "1.5.0"
                "1nb8vvgw8dv2inqklvk05fs0qxzkw8xrg2n9vgid6y7gm3423m3f"))

(define rust-icu-provider-2.0.0
  (crate-source "icu_provider" "2.0.0"
                "1bz5v02gxv1i06yhdhs2kbwxkw3ny9r2vvj9j288fhazgfi0vj03"))

(define rust-icu-provider-adapters-1.5.0
  (crate-source "icu_provider_adapters" "1.5.0"
                "1g60fydak0i4rxf8vfrr31mpck846k9ynix4fh1qx2il13ylscnn"
                #:snippet '(delete-file-recursively "tests")))

(define rust-icu-provider-blob-1.5.0
  (crate-source "icu_provider_blob" "1.5.0"
                "0lfgvia5v76gkpfgbga5ga6z1b5465v821f2hs0xfmaz6v8rhjy2"
                #:snippet '(delete-file-recursively "tests")))

(define rust-icu-provider-macros-1.5.0
  (crate-source "icu_provider_macros" "1.5.0"
                "1mjs0w7fcm2lcqmbakhninzrjwqs485lkps4hz0cv3k36y9rxj0y"))

(define rust-icu-segmenter-1.5.0
  (crate-source "icu_segmenter" "1.5.0"
                "1pmharib9s1hn5650d4lyd48145n1n14pja2gcnzqvrl29b745x7"))

(define rust-icu-segmenter-data-1.5.1
  (crate-source "icu_segmenter_data" "1.5.1"
                "1sx1rac5xxamqxr8x3cj91lkvlc4jblrrr4lqly3chcr2xsjgrd1"))

(define rust-icu-timezone-1.5.0
  (crate-source "icu_timezone" "1.5.0"
                "15nyqq5k231w51g3bmpbrsdnv1gfiam5s8w7qwha0farb1mbm4da"))

(define rust-icu-timezone-data-1.5.1
  (crate-source "icu_timezone_data" "1.5.1"
                "0s1f4g00mwmzv0r0hp0w0pk99qly9cmm78n2bc1ays522fvggp0s"))

(define rust-id-arena-2.2.1
  (crate-source "id-arena" "2.2.1"
                "01ch8jhpgnih8sawqs44fqsqpc7bzwgy0xpi6j0f4j0i5mkvr8i5"))

(define rust-idea-0.5.1
  (crate-source "idea" "0.5.1"
                "0xv4hd9mgrwgzfl7cc5nlwyahm9yni5z9dwb3c1z5mqr8h05fm87"
                #:snippet '(delete-file-recursively "tests")))

(define rust-ident-case-1.0.1
  (crate-source "ident_case" "1.0.1"
                "0fac21q6pwns8gh1hz3nbq15j8fi441ncl6w4vlnd1cmc55kiq5r"))

(define rust-idna-0.2.3
  (crate-source "idna" "0.2.3"
                "1y7ca2w5qp9msgl57n03zqp78gq1bk2crqzg6kv7a542mdphm2j1"))

(define rust-idna-0.3.0
  (crate-source "idna" "0.3.0"
                "1rh9f9jls0jy3g8rh2bfpjhvvhh4q80348jc4jr2s844133xykg1"))

(define rust-idna-1.0.3
  (crate-source "idna" "1.0.3"
                "0zlajvm2k3wy0ay8plr07w22hxkkmrxkffa6ah57ac6nci984vv8"))

(define rust-idna-1.1.0
  (crate-source "idna" "1.1.0"
                "1pp4n7hppm480zcx411dsv9wfibai00wbpgnjj4qj0xa7kr7a21v"))

(define rust-idna-adapter-1.0.0
  (crate-source "idna_adapter" "1.0.0"
                "0q4msgbfisi669k8b62wzr5l2laz3i5n44xbyn0m60ksjdflzpyg"))

(define rust-idna-adapter-1.1.0
  (crate-source "idna_adapter" "1.1.0"
                "1y7h6x9cms3gic9iqcfznx1kdslnzbf5yjchq88rvj41mjq5k4i7"))

(define rust-idna-adapter-1.2.0
  (crate-source "idna_adapter" "1.2.0"
                "0wggnkiivaj5lw0g0384ql2d7zk4ppkn3b1ry4n0ncjpr7qivjns"))

(define rust-idna-adapter-1.2.1
  (crate-source "idna_adapter" "1.2.1"
                "0i0339pxig6mv786nkqcxnwqa87v4m94b2653f6k3aj0jmhfkjis"))

(define rust-idna-mapping-1.0.0
  (crate-source "idna_mapping" "1.0.0"
                "0f0gyhdxrbh50ljcgaq0nm6a1sdmhq5rfpmlvdvrla34pk2jqhnm"))

(define rust-ieee754-0.2.6
  (crate-source "ieee754" "0.2.6"
                "1771d2kvw1wga65yrg9m7maky0fzsaq9hvhkv91n6gmxmjfdl1wh"))

(define rust-if-addrs-0.10.2
  (crate-source "if-addrs" "0.10.2"
                "02lgvpz14a7qx3s704i18xf3dp6ywniwij8mzy0kfr0sslch1fya"))

(define rust-if-addrs-0.12.0
  (crate-source "if-addrs" "0.12.0"
                "1j0c5xzzbfcb3k97zb050cygikfxv47mpj1hlyyyr249qglk6amv"))

(define rust-if-chain-1.0.2
  (crate-source "if_chain" "1.0.2"
                "1vdrylnqw8vd99j20jgalh1vp1vh7dwnkdzsmlx4yjsvfsmf2mnb"))

(define rust-ignore-0.4.23
  (crate-source "ignore" "0.4.23"
                "0jysggjfmlxbg60vhhiz4pb8jfb7cnq5swdsvxknbs7x18wgv2bd"))

(define rust-ignore-files-1.3.1
  (crate-source "ignore-files" "1.3.1"
                "0fq0h5igv3338v4rpnz68gpni2gkjkvyxawcjf958cwdd82p6kas"))

(define rust-ignore-files-3.0.3
  (crate-source "ignore-files" "3.0.3"
                "1sj8ddmsf51gb5cnv7zbd0g029sn4cz8vfznqyq88b63g4lma80a"))

(define rust-ignore-files-3.0.4
  (crate-source "ignore-files" "3.0.4"
                "03gn51p62wb5vfg65qbr6y6zk3hin3v2g006vxdxc2x00yz7hkc3"))

(define rust-ihex-3.0.0
  (crate-source "ihex" "3.0.0"
                "1wlzfyy5fsqgpki5vdapw0jjczqdm6813fgd3661wf5vfi3phnin"))

(define rust-im-rc-15.1.0
  (crate-source "im-rc" "15.1.0"
                "1zp5vdjj4b4lg8jnrz0wmdln2cdd9gn24a4psdvwd050bykma6dg"))

(define rust-image-0.24.9
  (crate-source "image" "0.24.9"
                "17gnr6ifnpzvhjf6dwbl9hki8x6bji5mwcqp0048x1jm5yfi742n"
                #:snippet '(for-each delete-file-recursively '("examples" "tests"))))

(define rust-image-0.25.6
  (crate-source "image" "0.25.6"
                "06i522bq4qlwylwnlmcn0sgqg72swwan544aldbhi0drwr66cdfv"))

(define rust-image-0.25.7
  (crate-source "image" "0.25.7"
                "01cnnnx692nqkllgvgian8jm847d29gkd5bgvwj8wxs3c7hkqshw"))

(define rust-image-webp-0.1.3
  (crate-source "image-webp" "0.1.3"
                "0179iqgvh6ybbq0r5d3ms11kka8jihhpliydkksj1vz2ps6gp6pp"))

(define rust-image-webp-0.2.1
  (crate-source "image-webp" "0.2.1"
                "0zwg4gpnp69dpn8pdhgjy14mawwi3md02mp1162al6s64bl02zdp"))

(define rust-image-webp-0.2.3
  (crate-source "image-webp" "0.2.3"
                "02vg5rfmkvslypd0xij6ynzaah3m32hfylicwr14n2rhlpkhz5zn"))

(define rust-image-webp-0.2.4
  (crate-source "image-webp" "0.2.4"
                "1hz814csyi9283vinzlkix6qpnd6hs3fkw7xl6z2zgm4w7rrypjj"))

(define rust-imagesize-0.13.0
  (crate-source "imagesize" "0.13.0"
                "11f26ac9zvbr7sjnsv2z9jd3ryaz40pg8xch4ij1q1rg5zbjgkgd"))

(define rust-imara-diff-0.1.8
  (crate-source "imara-diff" "0.1.8"
                "1lmk5dpha2fhahrnsrgavxn1qz6ydp1w8jz8fpvlb28p89ylplqp"))

(define rust-imara-diff-0.2.0
  (crate-source "imara-diff" "0.2.0"
                "0p2wmak4pbqfa93fihply18kq8q0nxg6zl0dhampipv6yxid809g"))

(define rust-imgref-1.11.0
  (crate-source "imgref" "1.11.0"
                "0254wzkakm31fdix6diqng0fkggknibh0b1iv570ap0djwykl9nh"))

(define rust-impl-more-0.1.9
  (crate-source "impl-more" "0.1.9"
                "1llwkdr56n340md0nh0h9m8gkbdymax45fdc92hwg1h0zyhak9g8"))

(define rust-implib-0.3.5
  (crate-source "implib" "0.3.5"
                "0qjyhapssradwljryq8v31kbyg6s6fqdg50cyyq3by0mc84zg2sr"))

(define rust-indent-write-2.2.0
  (crate-source "indent_write" "2.2.0"
                "1hqjp80argdskrhd66g9sh542yxy8qi77j6rc69qd0l7l52rdzhc"))

(define rust-indenter-0.3.3
  (crate-source "indenter" "0.3.3"
                "10y6i6y4ls7xsfsc1r3p5j2hhbxhaqnk5zzk8aj52b14v05ba8yf"))

(define rust-indenter-0.3.4
  (crate-source "indenter" "0.3.4"
                "1maq7yl2px9y40f68c2g2gjsq93rabphzp5shinj8nsldplfckcn"))

(define rust-indexmap-1.9.3
  (crate-source "indexmap" "1.9.3"
                "16dxmy7yvk51wvnih3a3im6fp5lmx0wx76i03n06wyak6cwhw1xx"))

(define rust-indexmap-2.0.1
  (crate-source "indexmap" "2.0.1"
                "0kmi1dhkmnv937mbkzlqq9kmkdr8k5030vfk19bi8jcxy4x7q8md"))

(define rust-indexmap-2.10.0
  (crate-source "indexmap" "2.10.0"
                "0qd6g26gxzl6dbf132w48fa8rr95glly3jhbk90i29726d9xhk7y"))

(define rust-indexmap-2.11.0
  (crate-source "indexmap" "2.11.0"
                "1sb3nmhisf9pdwfcxzqlvx97xifcvlh5g0rqj9j7i7qg8f01jj7j"))

(define rust-indexmap-2.11.1
  (crate-source "indexmap" "2.11.1"
                "088rg9dlk7lmw5hz5yi4jdyif25mrsj7lgwd5ska93y6mr180si0"))

(define rust-indexmap-2.11.3
  (crate-source "indexmap" "2.11.3"
                "1hqs931f1sd3r92zj77ji9bs75f20amnj0s3aqas9zqkym29h4cj"))

(define rust-indexmap-2.11.4
  (crate-source "indexmap" "2.11.4"
                "1rc8bgcjzfcskz1zipjjm7s3m1jskzhnhr9jxmsafhdk1xv863sb"))

(define rust-indexmap-2.7.1
  (crate-source "indexmap" "2.7.1"
                "0lmnm1zbr5gq3wic3d8a76gpvampridzwckfl97ckd5m08mrk74c"))

(define rust-indexmap-2.9.0
  (crate-source "indexmap" "2.9.0"
                "07m15a571yywmvqyb7ms717q9n42b46badbpsmx215jrg7dhv9yf"))

(define rust-indicatif-0.15.0
  (crate-source "indicatif" "0.15.0"
                "1r4n50mclyi4c7b9c9mlma1rhchjamw71r3z8vgqcmp24mhvbakv"))

(define rust-indicatif-0.16.2
  (crate-source "indicatif" "0.16.2"
                "06xyjs0kzqiqkjn60n1miwm2l87sa9p2lmzz0ymq18y72z37s81d"))

(define rust-indicatif-0.17.11
  (crate-source "indicatif" "0.17.11"
                "0db2b2r79r9x8x4lysq1ci9xm13c0xg0sqn3z960yh2bk2430fqq"))

(define rust-indicatif-0.17.4
  (crate-source "indicatif" "0.17.4"
                "1y0f3lnr2dnacmpgh8q8dws26pvy6xyyvhxn35jlwigg6xzk2ifv"))

(define rust-indicatif-0.18.0
  (crate-source "indicatif" "0.18.0"
                "1kg1wi3x9x15f22q99spfzcg5fzlmhcc5i6aqjxyssyh8vcld9kh"))

(define rust-indoc-0.3.6
  (crate-source "indoc" "0.3.6"
                "1n2fd2wm1h005hd7pjgx4gv5ymyq4sxqn8z0ssw6xchgqs5ilx27"))

(define rust-indoc-1.0.9
  (crate-source "indoc" "1.0.9"
                "01l3b4ami6sck57yrn8n2z44jifph2m3jiivkws7w2njbvfrk9xz"))

(define rust-indoc-2.0.5
  (crate-source "indoc" "2.0.5"
                "1dgjk49rkmx4kjy07k4b90qb5vl89smgb5rcw02n0q0x9ligaj5j"))

(define rust-indoc-2.0.6
  (crate-source "indoc" "2.0.6"
                "1gbn2pkx5sgbd9lp05d2bkqpbfgazi0z3nvharh5ajah11d29izl"))

(define rust-indoc-impl-0.3.6
  (crate-source "indoc-impl" "0.3.6"
                "1w58yg249kmzsn75kcj34qaxqh839l1hsaj3bzggy3q03wb6s16f"))

(define rust-infer-0.16.0
  (crate-source "infer" "0.16.0"
                "0iqq2llvl88nmpnxpaar2zrkmrah6bsy7q3cc6w9a0ikw9f0w5dw"))

(define rust-inferno-0.11.21
  (crate-source "inferno" "0.11.21"
                "126v1njhhx1shw0ammn3ngxxp20dmlb78p1xd9brks2zszhjja93"))

(define rust-inflate-0.4.5
  (crate-source "inflate" "0.4.5"
                "1zxjdn8iwa0ssxrnjmywm3r1v284wryvzrf8vkc7nyf5ijbjknqw"
                #:snippet '(delete-file-recursively "tests")))

(define rust-inflections-1.1.1
  (crate-source "inflections" "1.1.1"
                "0yl3gas612q25c72lwf04405i87yxr02vgv3ckcnz2fyvhpmhmx2"))

(define rust-inflector-0.11.4
  (crate-source "Inflector" "0.11.4"
                "1lqmcni21ifzyq41fhz6k1j2b23cmsx469s4g4sf01l78miqqhzy"))

(define rust-inkwell-0.2.0
  (crate-source "inkwell" "0.2.0"
                "1hb9wdh6hfr8shdnx3hda2mrb2vgglqyc94f2ysggf509x5cnkrz"))

(define rust-inkwell-internals-0.8.0
  (crate-source "inkwell_internals" "0.8.0"
                "1sgqira2mpbjdcah551rhl27a47hzgc19yifa08h90nnd38fg1di"))

(define rust-inlinable-string-0.1.15
  (crate-source "inlinable_string" "0.1.15"
                "1ysjci8yfvxgf51z0ny2nnwhxrclhmb3vbngin8v4bznhr3ybyn8"))

(define rust-inotify-0.10.2
  (crate-source "inotify" "0.10.2"
                "1k2m6a95827yspax1icmwiz4szr7c01w3dnn2b2bil4hfvcnilgx"))

(define rust-inotify-0.11.0
  (crate-source "inotify" "0.11.0"
                "1wq8m657rl085cg59p38sc5y62xy9yhhpvxbkd7n1awi4zzwqzgk"))

(define rust-inotify-0.9.6
  (crate-source "inotify" "0.9.6"
                "1zxb04c4qccp8wnr3v04l503qpxzxzzzph61amlqbsslq4z9s1pq"))

(define rust-inotify-sys-0.1.5
  (crate-source "inotify-sys" "0.1.5"
                "1syhjgvkram88my04kv03s0zwa66mdwa5v7ddja3pzwvx2sh4p70"))

(define rust-inout-0.1.4
  (crate-source "inout" "0.1.4"
                "008xfl1jn9rxsq19phnhbimccf4p64880jmnpg59wqi07kk117w7"))

(define rust-input-0.9.1
  (crate-source "input" "0.9.1"
                "1abmv1djhynihipjppgsmw6nbp6pcgzk8rzi4v6wmyci9990kp7v"))

(define rust-input-sys-1.18.0
  (crate-source "input-sys" "1.18.0"
                "1c4y24wf0jixi52js4f7cjspbgi0bzzaqfhn8m91qcq03i6mnkxx"))

(define rust-inquire-0.7.5
  (crate-source "inquire" "0.7.5"
                "0an1bjs0yklk990d7ni65znmirs99p01wld221affw5g64qgkp8g"))

(define rust-insta-1.42.2
  (crate-source "insta" "1.42.2"
                "111hrdc3bxwp146kz2ffwdq0qypdjk8a2yzwr8mivlb7maxrl9ah"))

(define rust-insta-1.43.1
  (crate-source "insta" "1.43.1"
                "0wa3iz1bafg3jwaqkjcs63spyalcm6chnhyxassm9065f3m38j8m"))

(define rust-insta-1.43.2
  (crate-source "insta" "1.43.2"
                "181m173v4f7s2f3j0lq462s0r6yg60y7fgxmnm1hy06yxd3vdza6"))

(define rust-insta-cmd-0.6.0
  (crate-source "insta-cmd" "0.6.0"
                "1rix5nmswns1p5p5f7pj5l9wvm69awzby0fbkkacwp4j4ylyzvpz"))

(define rust-instability-0.3.7
  (crate-source "instability" "0.3.7"
                "07f7k0cs1l8cdwxm46vy457bk880hgg6p83nfi777yqwv7bgxy8b"))

(define rust-instability-0.3.9
  (crate-source "instability" "0.3.9"
                "16pbfp50y1f6qqifk77zymz7r20f96v2gjc84vb8frwk1f080pa3"))

(define rust-instant-0.1.13
  (crate-source "instant" "0.1.13"
                "08h27kzvb5jw74mh0ajv0nv9ggwvgqm8ynjsn2sa9jsks4cjh970"))

(define rust-interception-sys-0.1.3
  (crate-source "interception-sys" "0.1.3"
                "1lgwbml7gzq5a5rriy708w68gx6yiw9cdg7xy2c5vsrrck7pbs5b"
                #:snippet '(for-each delete-file (find-files "." "\\.(dll|lib)$"))))

(define rust-interim-0.2.1
  (crate-source "interim" "0.2.1"
                "0njv1x7v55mr90ib278y8il8jm6hhmzghnr2sdiqcijzm2cr1km9"))

(define rust-interpolate-name-0.2.4
  (crate-source "interpolate_name" "0.2.4"
                "0q7s5mrfkx4p56dl8q9zq71y1ysdj4shh6f28qf9gly35l21jj63"))

(define rust-interprocess-2.2.3
  (crate-source "interprocess" "2.2.3"
                "0bgcvxjgqqm9m1iwfppc4id98d4imkk8x6l5hww9j8i3pl2v8hfr"))

(define rust-intl-memoizer-0.5.3
  (crate-source "intl-memoizer" "0.5.3"
                "0gqn5wwhzacvj0z25r5r3l2pajg9c8i1ivh7g8g8dszm8pis439i"))

(define rust-intl-pluralrules-7.0.2
  (crate-source "intl_pluralrules" "7.0.2"
                "0wprd3h6h8nfj62d8xk71h178q7zfn3srxm787w4sawsqavsg3h7"))

(define rust-inventory-0.3.20
  (crate-source "inventory" "0.3.20"
                "10ybwdx175d7xpvzpz0g2cczn0yvqykkwf75974z55sq5k6xf25b"))

(define rust-io-close-0.3.7
  (crate-source "io-close" "0.3.7"
                "1g4hldfn436rkrx3jlm4az1y5gdmkcixdlhkwy64yx06gx2czbcw"))

(define rust-io-kit-sys-0.4.1
  (crate-source "io-kit-sys" "0.4.1"
                "0ysy5k3wf54yangy25hkj10xx332cj2hb937xasg6riziv7yczk1"))

(define rust-io-lifetimes-1.0.11
  (crate-source "io-lifetimes" "1.0.11"
                "1hph5lz4wd3drnn6saakwxr497liznpfnv70via6s0v8x6pbkrza"))

(define rust-io-reactor-0.5.2
  (crate-source "io-reactor" "0.5.2"
                "0ybcgk7d8843f9klm3ifgx546qfipg0738bbr0gac10gccz8rmvp"))

(define rust-io-uring-0.7.10
  (crate-source "io-uring" "0.7.10"
                "0yvjyygwdcqjcgw8zp254hvjbm7as1c075dl50spdshas3aa4vq4"))

(define rust-io-uring-0.7.8
  (crate-source "io-uring" "0.7.8"
                "04whnj5a4pml44jhsmmf4p87bpgr7swkcijx4yjcng8900pj0vmq"))

(define rust-io-uring-0.7.9
  (crate-source "io-uring" "0.7.9"
                "1i60fxfbxypfgfmq883kwxgldxcjlnnwjazgjiys3893fvrqfdfr"))

(define rust-ioctl-rs-0.1.6
  (crate-source "ioctl-rs" "0.1.6"
                "0zdrgqxblrwm4ym8pwrr7a4dwjzxrvr1k0qjx6rk1vjwi480b5zp"))

(define rust-iovec-0.1.4
  (crate-source "iovec" "0.1.4"
                "0ph73qygwx8i0mblrf110cj59l00gkmsgrpzz1rm85syz5pymcxj"))

(define rust-ipconfig-0.3.2
  (crate-source "ipconfig" "0.3.2"
                "0zwr0x3jnqmjdqqbzhb0nid011qyhcyfdfqv32cdw85pjqpvk3dm"))

(define rust-ipnet-2.11.0
  (crate-source "ipnet" "2.11.0"
                "0c5i9sfi2asai28m8xp48k5gvwkqrg5ffpi767py6mzsrswv17s6"))

(define rust-ipnetwork-0.20.0
  (crate-source "ipnetwork" "0.20.0"
                "03hhmxyimz0800z44wl3z1ak8iw91xcnk7sgx5p5jinmx50naimz"))

(define rust-ipnetwork-0.21.1
  (crate-source "ipnetwork" "0.21.1"
                "0qqpyjfma524v734nw4xri38xwj524z8q3k29bji6kfmmyyhldyg"))

(define rust-iptables-0.5.3
  (crate-source "iptables" "0.5.3"
                "07z25iq9aaqldfq8j7wngs4lxqljqpf2h6b3y5x4rpv71mkjjzsq"))

(define rust-iri-string-0.7.8
  (crate-source "iri-string" "0.7.8"
                "1cl0wfq97wq4s1p4dl0ix5cfgsc5fn7l22ljgw9ab9x1qglypifv"))

(define rust-irx-config-3.5.0
  (crate-source "irx-config" "3.5.0"
                "0slp6f0ysnfwi1fyhd5bh2jy0h1a5ck4kgfppzzmsqy5z63a0rk2"))

(define rust-is-ci-1.2.0
  (crate-source "is_ci" "1.2.0"
                "0ifwvxmrsj4r29agfzr71bjq6y1bihkx38fbzafq5vl0jn1wjmbn"))

(define rust-is-debug-1.1.0
  (crate-source "is_debug" "1.1.0"
                "01yl28nv69wsqiyyhfbgx52yskpjyw5z4xq137c33ja3wb96dqhz"))

(define rust-is-docker-0.2.0
  (crate-source "is-docker" "0.2.0"
                "1cyibrv6817cqcpf391m327ss40xlbik8wxcv5h9pj9byhksx2wj"))

(define rust-is-executable-1.0.1
  (crate-source "is_executable" "1.0.1"
                "1j59iqaxcgax0qll30rarpcr7y3dpkl38iv4mlkfcxbvsv3cv6ps"))

(define rust-is-executable-1.0.4
  (crate-source "is_executable" "1.0.4"
                "1qlafm7f0zq0kzvbd4fhcfci4g9gxp6g3yqxjqsjj1zrssxbb8fl"
                #:snippet '(delete-file-recursively "tests")))

(define rust-is-executable-1.0.5
  (crate-source "is_executable" "1.0.5"
                "1i78ss45h94nwabbn6ki64a91djlli8zdwwbh56jj9kvhssbiaxs"))

(define rust-is-macro-0.2.2
  (crate-source "is-macro" "0.2.2"
                "083v9iqim0cf7rlj09gmks82qv6fx77z3i6595x4fxwv2ag0fzca"))

(define rust-is-terminal-0.4.16
  (crate-source "is-terminal" "0.4.16"
                "1acm63whnpwiw1padm9bpqz04sz8msymrmyxc55mvlq8hqqpykg0"))

(define rust-is-terminal-0.4.9
  (crate-source "is-terminal" "0.4.9"
                "12xgvc7nsrp3pn8hcxajfhbli2l5wnh3679y2fmky88nhj4qj26b"))

(define rust-is-terminal-polyfill-1.70.1
  (crate-source "is_terminal_polyfill" "1.70.1"
                "1kwfgglh91z33kl0w5i338mfpa3zs0hidq5j4ny4rmjwrikchhvr"))

(define rust-is-wsl-0.4.0
  (crate-source "is-wsl" "0.4.0"
                "19bs5pq221d4bknnwiqqkqrnsx2in0fsk8fylxm1747iim4hjdhp"))

(define rust-isal-rs-0.5.3+496255c
  (crate-source "isal-rs" "0.5.3+496255c"
                "1vd9zi1b6mhsd3zpdq38n54w25x7m8845sf0p8c4rvxpkm7p7isf"))

(define rust-isal-sys-0.5.3+496255c
  (crate-source "isal-sys" "0.5.3+496255c"
                "16x2wm1k6pkg2v163x81iplhn9xmjycdkk8w42xflq4sjlwr5z5f"
                #:snippet
                '(begin
                   (delete-file-recursively "isa-l")
                   (delete-file "build.rs")
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main() {~@
                        println!(\"cargo:rustc-link-lib=isal\");~@
                        }~%"))))))

(define rust-isatty-0.1.9
  (crate-source "isatty" "0.1.9"
                "1p2yx591kwz2b3f2bv7al9hamhn052zmz9jd969rdv4kzj0q46p3"))

(define rust-iso7816-tlv-0.4.4
  (crate-source "iso7816-tlv" "0.4.4"
                "0immb18gxx8sycf4k3ks2qxhz8sl8ra5s9wa4a8dccd84j6x4q3n"))

(define rust-iso8601-0.6.2
  (crate-source "iso8601" "0.6.2"
                "13f9a6izrm87dd66qcagw65lw714072a8y8hyjk23ar4z37pghf5"))

(define rust-iso8601-0.6.3
  (crate-source "iso8601" "0.6.3"
                "0inz8m51brcn17az8yln1c9yxkk0wdkjy4n638m48hzi9062y271"))

(define rust-itertools-0.10.5
  (crate-source "itertools" "0.10.5"
                "0ww45h7nxx5kj6z2y6chlskxd1igvs4j507anr6dzg99x1h25zdh"))

(define rust-itertools-0.11.0
  (crate-source "itertools" "0.11.0"
                "0mzyqcc59azx9g5cg6fs8k529gvh4463smmka6jvzs3cd2jp7hdi"))

(define rust-itertools-0.12.1
  (crate-source "itertools" "0.12.1"
                "0s95jbb3ndj1lvfxyq5wanc0fm0r6hg6q4ngb92qlfdxvci10ads"))

(define rust-itertools-0.13.0
  (crate-source "itertools" "0.13.0"
                "11hiy3qzl643zcigknclh446qb9zlg4dpdzfkjaa9q9fqpgyfgj1"))

(define rust-itertools-0.14.0
  (crate-source "itertools" "0.14.0"
                "118j6l1vs2mx65dqhwyssbrxpawa90886m3mzafdvyip41w2q69b"))

(define rust-itertools-0.9.0
  (crate-source "itertools" "0.9.0"
                "0jyml7ygr7kijkcjdl3fk5f34y5h5jsavclim7l13zjiavw1hkr8"))

(define rust-itertools-num-0.1.3
  (crate-source "itertools-num" "0.1.3"
                "1rr7ig9nkpampcas23s91x7yac6qdnwssq3nap522xbgkqps4wm8"))

(define rust-itoa-0.4.8
  (crate-source "itoa" "0.4.8"
                "1m1dairwyx8kfxi7ab3b5jc71z1vigh9w4shnhiajji9avzr26dp"))

(define rust-itoa-1.0.11
  (crate-source "itoa" "1.0.11"
                "0nv9cqjwzr3q58qz84dcz63ggc54yhf1yqar1m858m1kfd4g3wa9"))

(define rust-itoa-1.0.14
  (crate-source "itoa" "1.0.14"
                "0x26kr9m062mafaxgcf2p6h2x7cmixm0zw95aipzn2hr3d5jlnnp"))

(define rust-itoa-1.0.15
  (crate-source "itoa" "1.0.15"
                "0b4fj9kz54dr3wam0vprjwgygvycyw8r0qwg7vp19ly8b2w16psa"))

(define rust-itoa-1.0.9
  (crate-source "itoa" "1.0.9"
                "0f6cpb4yqzhkrhhg6kqsw3wnmmhdnnffi6r2xzy248gzi2v0l5dg"))

(define rust-itoap-1.0.1
  (crate-source "itoap" "1.0.1"
                "1f48gsd18kbvskwbnwszhqjpk1l4rdmahh7kaz86b432cj9g8a4h"
                #:snippet '(delete-file "bench.png")))

(define rust-ivf-0.1.3
  (crate-source "ivf" "0.1.3"
                "1jjy911flpfpflnxw5fqsx6a3ghaq5wi2q18nx9cawpf81qnabsm"))

(define rust-jack-0.11.4
  (crate-source "jack" "0.11.4"
                "1kd6p6bfmxyclkkq9pkrqyynf0mj53ias4binx7kbyxfqaiihnhf"))

(define rust-jack-sys-0.5.1
  (crate-source "jack-sys" "0.5.1"
                "1aw6zishflmd5v9dz5yvpc5f9jsfm9pjjhzvdmbjp8lmkdhvf4v0"))

(define rust-jaq-core-2.1.1
  (crate-source "jaq-core" "2.1.1"
                "12q4pnssz0r9n7f14658dn7v3l5lhziqqqbgdixh1a1drmw1y1ai"))

(define rust-jaq-core-2.2.0
  (crate-source "jaq-core" "2.2.0"
                "0ak7k7wvs1pf6yryaglr5hrjddzps0j310m5kpplms92dhnwygac"))

(define rust-jaq-json-1.1.1
  (crate-source "jaq-json" "1.1.1"
                "01rir05smgwi806wj1r5mma7yc19jkngkgr30qkil27bpgn41fjq"))

(define rust-jaq-json-1.1.2
  (crate-source "jaq-json" "1.1.2"
                "101hkx4is3krq7rlimfnj3ddfgzyqlp3j8g0hy651h52v9f0i3w4"))

(define rust-jaq-std-2.1.0
  (crate-source "jaq-std" "2.1.0"
                "18gv6fx5b4ch4vhwsjlhkcs1fw4bhza21cnhqpmmaxzjz765wdfz"))

(define rust-jaq-std-2.1.1
  (crate-source "jaq-std" "2.1.1"
                "0ilsnghw2db5k01qdri2nyggynwwn8hqx9cdmljgwdvq5cp1hq4k"))

(define rust-javascriptcore-rs-sys-0.2.0
  (crate-source "javascriptcore-rs-sys" "0.2.0"
                "1619vl48yw3wrjw4c3dfm3s5s9sizgxp5s7s1ahpbkcdl2lasiiz"))

(define rust-jemalloc-sys-0.5.4+5.3.0-patched
  (crate-source "jemalloc-sys" "0.5.4+5.3.0-patched"
                "1wpbpwhfs6wd484cdfpl0zdf441ann9wj0fypy67i8ffw531jv5c"
                #:snippet '(delete-file-recursively "jemalloc")))

(define rust-jemallocator-0.5.4
  (crate-source "jemallocator" "0.5.4"
                "1g6k9ly6wxj53bp8lz9lg9nj4s662k6612jydw71aqwfkx53gpm0"))

(define rust-jieba-rs-0.6.8
  (crate-source "jieba-rs" "0.6.8"
                "0glq71kil7hic6m3227w177nblwnqhxc6b9sdry8vb6kghsc3w4k"))

(define rust-jiff-0.2.10
  (crate-source "jiff" "0.2.10"
                "1xmfywmhfwn25wi6k8db27b4a36qi2l5zv75p87s3ijd44c441js"))

(define rust-jiff-0.2.13
  (crate-source "jiff" "0.2.13"
                "01h87izvrn3nnbkvygc1286vv63ln4bayxhi0838jqrh1mk0087h"))

(define rust-jiff-0.2.14
  (crate-source "jiff" "0.2.14"
                "14yaijl4hw0mnqiavpsrkna521ajk03wh0ixz7s7jg7k0w8xz551"))

(define rust-jiff-0.2.15
  (crate-source "jiff" "0.2.15"
                "0jby6kbs2ra33ji0rx4swcp66jzmcvgszc5v4izwfsgbn6w967xy"))

(define rust-jiff-0.2.5
  (crate-source "jiff" "0.2.5"
                "0q3jpq3scznmviiajldyf5xby38zgyvkxbrmgb9hf78r6416f0n1"))

(define rust-jiff-0.2.6
  (crate-source "jiff" "0.2.6"
                "1274a8wj5czbysk1nzgrk18prsq6c5cvviv28cb3ga5ybid18cqz"))

(define rust-jiff-0.2.8
  (crate-source "jiff" "0.2.8"
                "18140w5m8bblbxaj3wpa55kj0808g6lr6a6wsi65xx8hj748gbg5"))

(define rust-jiff-0.2.9
  (crate-source "jiff" "0.2.9"
                "1l11404xxp75qcgfk9jf6bag53fvhl5za8dhw4agxrib2kvk1v2r"))

(define rust-jiff-static-0.2.10
  (crate-source "jiff-static" "0.2.10"
                "0m2jyhksclxpb54b5lg1d1i2qzzyx90y2c3h7bzmlckyv4r7k6qr"))

(define rust-jiff-static-0.2.13
  (crate-source "jiff-static" "0.2.13"
                "0j0cyq3rk34d1hfk30h5r524zbw825fw8gviqqlqc66pvmc0ghzk"))

(define rust-jiff-static-0.2.14
  (crate-source "jiff-static" "0.2.14"
                "0hk4dx1dlmb5ffg1n9asywv3d4zp9gizm5wlfldwgiijxnvisvkc"))

(define rust-jiff-static-0.2.15
  (crate-source "jiff-static" "0.2.15"
                "1d4l4pvlhz3w487gyhnzvagpbparspv4c8f35qk6g5w9zx8k8d03"))

(define rust-jiff-static-0.2.5
  (crate-source "jiff-static" "0.2.5"
                "0k1v30mhbgh4zj2r9d7lfqlh5b20b5573cx0a4gip7rlkldf7pac"))

(define rust-jiff-static-0.2.6
  (crate-source "jiff-static" "0.2.6"
                "06aw1z5qj4hkcm0s37phcfr27wrdxshrfpb3ldbn35f61v217kj3"))

(define rust-jiff-static-0.2.8
  (crate-source "jiff-static" "0.2.8"
                "019n229ygcx5fviby0s2j07mqwm6lhyc80izfkqfcbvy9avdaxnh"))

(define rust-jiff-static-0.2.9
  (crate-source "jiff-static" "0.2.9"
                "1ghxrhwgf9i01gi62hwsfaqy3pcsdliw7c3nn0vrvmbzf96q6ssj"))

(define rust-jiff-tzdb-0.1.4
  (crate-source "jiff-tzdb" "0.1.4"
                "09350bna4vxdn2fv7gd08ay41llkflmfyvpx5d6l088axc2kfa61"
                #:snippet '(delete-file "concatenated-zoneinfo.dat")))

(define rust-jiff-tzdb-platform-0.1.3
  (crate-source "jiff-tzdb-platform" "0.1.3"
                "1s1ja692wyhbv7f60mc0x90h7kn1pv65xkqi2y4imarbmilmlnl7"))

(define rust-jiter-0.7.1
  (crate-source "jiter" "0.7.1"
                "09fx71x93sh15028pd0f78flv9j4fkvg6lgi22y5gbv83c99mxh7"))

(define rust-jni-0.19.0
  (crate-source "jni" "0.19.0"
                "1v0pn0i1wb8zp4wns4l8hz9689hqsplv7iba7hylaznvwg11ipy6"))

(define rust-jni-0.21.1
  (crate-source "jni" "0.21.1"
                "15wczfkr2r45slsljby12ymf2hij8wi5b104ghck9byjnwmsm1qs"))

(define rust-jni-sys-0.3.0
  (crate-source "jni-sys" "0.3.0"
                "0c01zb9ygvwg9wdx2fii2d39myzprnpqqhy7yizxvjqp5p04pbwf"))

(define rust-jobserver-0.1.32
  (crate-source "jobserver" "0.1.32"
                "1l2k50qmj84x9mn39ivjz76alqmx72jhm12rw33zx9xnpv5xpla8"))

(define rust-jobserver-0.1.33
  (crate-source "jobserver" "0.1.33"
                "12jkn3cxvfs7jsb6knmh9y2b41lwmrk3vdqywkmssx61jzq65wiq"))

(define rust-jobserver-0.1.34
  (crate-source "jobserver" "0.1.34"
                "0cwx0fllqzdycqn4d6nb277qx5qwnmjdxdl0lxkkwssx77j3vyws"))

(define rust-jpeg-decoder-0.3.1
  (crate-source "jpeg-decoder" "0.3.1"
                "1c1k53svpdyfhibkmm0ir5w0v3qmcmca8xr8vnnmizwf6pdagm7m"
                #:snippet '(delete-file-recursively "benches")))

(define rust-jpeg-decoder-0.3.2
  (crate-source "jpeg-decoder" "0.3.2"
                "01qs04rnackrc5xhw7w9difn2h3pqsdbhgdz7nqn9gklicfhz080"
                #:snippet '(delete-file-recursively "benches")))

(define rust-js-sys-0.3.61
  (crate-source "js-sys" "0.3.61"
                "0c075apyc5fxp2sbgr87qcvq53pcjxmp05l47lzlhpn5a0hxwpa4"))

(define rust-js-sys-0.3.70
  (crate-source "js-sys" "0.3.70"
                "0yp3rz7vrn9mmqdpkds426r1p9vs6i8mkxx8ryqdfadr0s2q0s0q"))

(define rust-js-sys-0.3.77
  (crate-source "js-sys" "0.3.77"
                "13x2qcky5l22z4xgivi59xhjjx4kxir1zg7gcj0f1ijzd4yg7yhw"))

(define rust-js-sys-0.3.78
  (crate-source "js-sys" "0.3.78"
                "0f17vdkpbarm0qlbqb0p1fyiy4l9bs62zvw3fv0ywb29g0shc2qc"))

(define rust-js-sys-0.3.80
  (crate-source "js-sys" "0.3.80"
                "0bkhnbna0a9sqhhswfar0mzi8mpy2dygv4zbzfdbm97bqnz16bw5"))

(define rust-js-sys-0.3.81
  (crate-source "js-sys" "0.3.81"
                "01ckbf16iwh7qj92fax9zh8vf2y9sk60cli6999cn7a1jxx96j7c"))

(define rust-json-0.12.4
  (crate-source "json" "0.12.4"
                "1z9vybaaz3xq95bwmyln2ijmn4dmzj61xlz06jsc9dnzmxg2i3h7"))

(define rust-json-comments-0.2.2
  (crate-source "json_comments" "0.2.2"
                "0191qf6gaw3bcv13kx9xzwbfx71jv7gly5ds2l77bacvwpagxfwx"))

(define rust-json5-0.4.1
  (crate-source "json5" "0.4.1"
                "1h9hni897zmn3vcixfbwwkj2gkz27h7z9dah8bk1qv37mwhxpc4n"))

(define rust-jsonrpc-core-18.0.0
  (crate-source "jsonrpc-core" "18.0.0"
                "1sv5m6bxyscdqg8cfzlsm8f3vks3972zc9w475l4h19dxxmggxql"))

(define rust-jsonschema-0.30.0
  (crate-source "jsonschema" "0.30.0"
                "07gsyqmq02y62smmnagxzabdmahhz7f08c8lsbqzn4d6cl1nmd7i"))

(define rust-jsonschema-0.17.1
  (crate-source "jsonschema" "0.17.1"
                "0y397fmb7qkah166lq5q39p9hizj9sls09xnvwc936pwgr7iy1ra"))

(define rust-junction-1.2.0
  (crate-source "junction" "1.2.0"
                "05mxqwzxgb2aqgbq9b5lgbqq0r0nds6yx68zzhyxlhx26zbxzfvj"))

(define rust-jwalk-0.8.1
  (crate-source "jwalk" "0.8.1"
                "0miw0074ch6p89dbvrlnz9wp13rh7542cf1ak8bx4v1mcrsq8d97"))

(define rust-k8s-openapi-0.22.0
  (crate-source "k8s-openapi" "0.22.0"
                "0mcpy1gdxpqayc74i3x3ayh3q5bl6dzvsgmw91jq1r9sjkxill0r"))

(define rust-kamadak-exif-0.6.1
  (crate-source "kamadak-exif" "0.6.1"
                "0ds8k36qi88mzhx5fm1cgbxg0s4kmyiibmqpl5asvvvlfc6dhc0i"
                #:snippet '(delete-file-recursively "tests")))

(define rust-kanata-interception-0.3.0
  (crate-source "kanata-interception" "0.3.0"
                "01mn1dskhm124x0nxfcw5cyb07h0i256x9bfj23aq6adjsdpprg2"))

(define rust-kanata-keyberon-0.180.2
  (crate-source "kanata-keyberon" "0.180.2"
                "1b0axjmzq79pi5xbj82c38mvvwwylar42jwiwzz3n8v0bjln6czj"
                #:snippet '(delete-file-recursively "images")))

(define rust-kanata-keyberon-macros-0.2.0
  (crate-source "kanata-keyberon-macros" "0.2.0"
                "0lj7ldiazmszh0k01h7mjzhjg59bdakvx2pnpc9mq2ir0czzixkk"))

(define rust-kanata-parser-0.180.2
  (crate-source "kanata-parser" "0.180.2"
                "0dfbjc2gy5jc3wqy0fnn6c9wpqxcwwwkkv6lf4lgnp6sfkvqsi18"))

(define rust-kanata-tcp-protocol-0.180.2
  (crate-source "kanata-tcp-protocol" "0.180.2"
                "1x74ncvffz3cani6l84jl8rqr26d445hz3h88h75k7aa59jc8fax"))

(define rust-karabiner-driverkit-0.1.5
  (crate-source "karabiner-driverkit" "0.1.5"
                "0pqnh9n3a8wxqzdj7d30f99g322da8zpnixsq5gfs9n1klccj380"
                #:snippet '(delete-file-recursively "c_src")))

(define rust-keccak-0.1.5
  (crate-source "keccak" "0.1.5"
                "0m06swsyd58hvb1z17q6picdwywprf1yf1s6l491zi8r26dazhpc"))

(define rust-kernel32-sys-0.2.2
  (crate-source "kernel32-sys" "0.2.2"
                "1389av0601a9yz8dvx5zha9vmkd6ik7ax0idpb032d28555n41vm"))

(define rust-keyframe-1.1.1
  (crate-source "keyframe" "1.1.1"
                "1afr5ffns3k79xaqnw6rw3qn8sngwly6gxfnjn8d060mk3vqnw30"))

(define rust-keyring-2.3.3
  (crate-source "keyring" "2.3.3"
                "184mshdrgghlvmlz0n7w1167yx0sa3zb82n31jk4lwcx07q8fcrn"))

(define rust-khronos-api-3.1.0
  (crate-source "khronos_api" "3.1.0"
                "1p0xj5mlbagqyvvnv8wmv3cr7l9y1m153888pxqwg3vk3mg5inz2"))

(define rust-kmeans-colors-0.6.0
  (crate-source "kmeans_colors" "0.6.0"
                "153qfhvvm7iisqjy6sy7mfw8x9h8y9y25z7ygvfkbp0i53xmz8g9"))

(define rust-knuffel-3.2.0
  (crate-source "knuffel" "3.2.0"
                "04vl2xmdn280rcigv96v06a00v7gbxqggr0w9cqi2407qvfydgh4"
                #:snippet '(delete-file-recursively "images")))

(define rust-knuffel-derive-3.2.0
  (crate-source "knuffel-derive" "3.2.0"
                "0g98909l5wb1d1hcz61q53kvsmjadry2w3l47lg9dywwqib7z5wi"))

(define rust-kqueue-1.0.8
  (crate-source "kqueue" "1.0.8"
                "033x2knkbv8d3jy6i9r32jcgsq6zm3g97zh5la43amkv3g5g2ivl"))

(define rust-kqueue-1.1.1
  (crate-source "kqueue" "1.1.1"
                "0sjrsnza8zxr1zfpv6sa0zapd54kx9wlijrz9apqvs6wsw303hza"))

(define rust-kqueue-sys-1.0.4
  (crate-source "kqueue-sys" "1.0.4"
                "12w3wi90y4kwis4k9g6fp0kqjdmc6l00j16g8mgbhac7vbzjb5pd"))

(define rust-kstring-2.0.2
  (crate-source "kstring" "2.0.2"
                "1lfvqlqkg2x23nglznb7ah6fk3vv3y5i759h5l2151ami98gk2sm"))

(define rust-kuchiki-0.8.1
  (crate-source "kuchiki" "0.8.1"
                "0n53hhscgf4az58nnbjfxlqz9k96h3f02c7fhbzpqdriw33fka0y"))

(define rust-kurbo-0.11.2
  (crate-source "kurbo" "0.11.2"
                "034byrnsl7c19k63hjfsfc3wl07k4qq3qvdrrkcp0qgaxwrx6xqh"))

(define rust-kv-log-macro-1.0.7
  (crate-source "kv-log-macro" "1.0.7"
                "0zwp4bxkkp87rl7xy2dain77z977rvcry1gmr5bssdbn541v7s0d"))

(define rust-lab-0.11.0
  (crate-source "lab" "0.11.0"
                "13ymsn5cwl5i9pmp5mfmbap7q688dcp9a17q82crkvb784yifdmz"))

(define rust-lalrpop-0.20.2
  (crate-source "lalrpop" "0.20.2"
                "1jn1qg7gs9kka6sy2sbxx8wp6z8lm892ksr414b9yaansrx0gjsm"))

(define rust-lalrpop-0.22.1
  (crate-source "lalrpop" "0.22.1"
                "0nqx5ad8dvgn8wcg87bi8snpf2zgi8wnpd41y7wan5i0wins4ivh"))

(define rust-lalrpop-0.22.2
  (crate-source "lalrpop" "0.22.2"
                "00cm28s3m7hcscqf7d14icyzlyqslpsqaldk1z8w24g4ikabnkms"))

(define rust-lalrpop-util-0.20.2
  (crate-source "lalrpop-util" "0.20.2"
                "0lr5r12bh9gjjlmnjrbblf4bfcwnad4gz1hqjvp34yzb22ln0x2h"))

(define rust-lalrpop-util-0.22.1
  (crate-source "lalrpop-util" "0.22.1"
                "0fy9s948nlxrrhlz2rcgli8r9swvz9fp5nrqqdidb2sbwczmpl78"))

(define rust-lalrpop-util-0.22.2
  (crate-source "lalrpop-util" "0.22.2"
                "0cw7nizh1a0p8m4msxbs2i98lls6fs86js2f4vysxwc4zzlsbfmm"))

(define rust-language-tags-0.3.2
  (crate-source "language-tags" "0.3.2"
                "124k6w9nx33q4xs8rpa9f7klshrsa0x4f7qngdwq890lpdj5jd6l"))

(define rust-lapack-0.19.0
  (crate-source "lapack" "0.19.0"
                "01yja6fdjr7h6nqhlamn0mhqv5533735030av2gnmrzp9mmnlrxd"))

(define rust-lapack-sys-0.14.0
  (crate-source "lapack-sys" "0.14.0"
                "1gmw6fn4zc6106pzpi88p79b260h7hav40b7sfisf45lbz45czs4"
                #:snippet
                '(begin
                   (delete-file-recursively "lapack")
                   (substitute* "bin/wrapper.h"
                     (("\"../lapack/LAPACKE/include/lapack.h\"")
                      "<lapack.h>")))))

(define rust-lasso-0.7.3
  (crate-source "lasso" "0.7.3"
                "1yz92fy2zv6wslfwwf3j7lw1wxja8d91rrcwgfzv751l1ajys53f"))

(define rust-layout-rs-0.1.2
  (crate-source "layout-rs" "0.1.2"
                "1a6y7wjc0p7cbawdwa9gkvb1c518697nchbk5aj9r0vc7a5b5pl4"))

(define rust-lazy-bytes-cast-5.0.1
  (crate-source "lazy-bytes-cast" "5.0.1"
                "0sr0dy1jfg7bjwm9js4hk0ngl0cmgparq2idv1m1bkc9y2cp898h"))

(define rust-lazy-init-0.5.1
  (crate-source "lazy-init" "0.5.1"
                "0vvhq8xdmajy2ai8p7zxja68a95n7m65xhdgjapxq4mc4qv9ch4z"))

(define rust-lazy-static-0.2.11
  (crate-source "lazy_static" "0.2.11"
                "0wxy8vak7jsx6r8gx475pjqpx11p2bfq4wvw6idmqi31mp3k7w3n"))

(define rust-lazy-static-1.4.0
  (crate-source "lazy_static" "1.4.0"
                "0in6ikhw8mgl33wjv6q6xfrb5b9jr16q8ygjy803fay4zcisvaz2"))

(define rust-lazy-static-1.5.0
  (crate-source "lazy_static" "1.5.0"
                "1zk6dqqni0193xg6iijh7i3i44sryglwgvx20spdvwk3r6sbrlmv"))

(define rust-lazycell-1.3.0
  (crate-source "lazycell" "1.3.0"
                "0m8gw7dn30i0zjjpjdyf6pc16c34nl71lpv461mix50x3p70h3c3"))

(define rust-lddtree-0.3.7
  (crate-source "lddtree" "0.3.7"
                "13mh4y1bwlqpa8jpsgj3zb0mxi4syzdqh8x13xwapdnp9g4rlxz0"
                #:snippet '(delete-file-recursively "tests")))

(define rust-leb128-0.2.5
  (crate-source "leb128" "0.2.5"
                "0rxxjdn76sjbrb08s4bi7m4x47zg68f71jzgx8ww7j0cnivjckl8"))

(define rust-leb128fmt-0.1.0
  (crate-source "leb128fmt" "0.1.0"
                "1chxm1484a0bly6anh6bd7a99sn355ymlagnwj3yajafnpldkv89"))

(define rust-lebe-0.5.2
  (crate-source "lebe" "0.5.2"
                "1j2l6chx19qpa5gqcw434j83gyskq3g2cnffrbl3842ymlmpq203"))

(define rust-levenshtein-1.0.5
  (crate-source "levenshtein" "1.0.5"
                "0q0pa1prsjk3v28xkas5rf30ng9816mxpr2n3xls65dmgawss4yv"))

(define rust-lexical-5.2.2
  (crate-source "lexical" "5.2.2"
                "1r8lsi523h53kbb99xgv31jabwhcp4rzqd4hfazfhcjffh5aj17l"
                #:snippet '(delete-file-recursively "assets")))

(define rust-lexical-core-0.7.6
  (crate-source "lexical-core" "0.7.6"
                "1zjzab1fnaw4kj6ixyrskp4dyz761gdcab07m4bkvlk1l4mcc1v6"))

(define rust-lexical-parse-float-0.8.5
  (crate-source "lexical-parse-float" "0.8.5"
                "0py0gp8hlzcrlvjqmqlpl2v1as65iiqxq2xsabxvhc01pmg3lfv8"))

(define rust-lexical-parse-integer-0.8.6
  (crate-source "lexical-parse-integer" "0.8.6"
                "1sayji3mpvb2xsjq56qcq3whfz8px9a6fxk5v7v15hyhbr4982bd"))

(define rust-lexical-sort-0.3.1
  (crate-source "lexical-sort" "0.3.1"
                "0yi0jzlvjaszwl5a49r0a0gcq404rdk5ls2c9npis8qyc68lb7n0"))

(define rust-lexical-util-0.8.5
  (crate-source "lexical-util" "0.8.5"
                "1z73qkv7yxhsbc4aiginn1dqmsj8jarkrdlyxc88g2gz2vzvjmaj"))

(define rust-lexiclean-0.0.1
  (crate-source "lexiclean" "0.0.1"
                "1rb8sky7gi75a7xxn6xrfkrbqkp465npm54p5s89ysqhgc0ja4j4"))

(define rust-lexopt-0.3.1
  (crate-source "lexopt" "0.3.1"
                "19zklfskhlvrfbl55h9g2a1v61k2jxcj4d1fqkk6nbxyzjhy584z"))

(define rust-libadwaita-0.5.3
  (crate-source "libadwaita" "0.5.3"
                "174pzn9dwsk8ikvrhx13vkh0zrpvb3rhg9yd2q5d2zjh0q6fgrrg"))

(define rust-libadwaita-0.7.2
  (crate-source "libadwaita" "0.7.2"
                "14c1qy6mq5l9wlwsr2x9ijbvis283msfglxgp9kvzahnkk93a0ah"))

(define rust-libadwaita-sys-0.5.3
  (crate-source "libadwaita-sys" "0.5.3"
                "16n6xsy6jhbj0jbpz8yvql6c9b89a99v9vhdz5s37mg1inisl42y"))

(define rust-libadwaita-sys-0.7.2
  (crate-source "libadwaita-sys" "0.7.2"
                "1nqjr514hhdc4aldlsc4y3vkpnkq9q73g2jl7ypqnmf2b209i036"))

(define rust-libbzip3-sys-0.4.0+1.4.0
  ;; TODO: Package bzip3.
  (crate-source "libbzip3-sys" "0.4.0+1.4.0"
                "0ssx5f16hhif8zjdd1fz4cwn1cwjslwcbssm2xs7l93g2i6hw6h0"))

(define rust-libc-0.2.147
  (crate-source "libc" "0.2.147"
                "1cwv2lkzk3p5lby79fm42qhsh29lvbqwayhjjkq1s746xaq8yrml"))

(define rust-libc-0.2.148
  (crate-source "libc" "0.2.148"
                "16rn9l8s5sj9n2jb2pw13ghqwa5nvjggkh9q3lp6vs1jfghp3p4w"))

(define rust-libc-0.2.170
  (crate-source "libc" "0.2.170"
                "0a38q3avb6r6azxb7yfbjly5sbr8926z6c4sryyp33rgrf03cnw7"))

(define rust-libc-0.2.171
  (crate-source "libc" "0.2.171"
                "1mipla3dy3l59pfa9xy4iw2vdgn8n30dzf4vdnasjflxdqhkg6f1"))

(define rust-libc-0.2.172
  (crate-source "libc" "0.2.172"
                "1ykz4skj7gac14znljm5clbnrhini38jkq3d60jggx3y5w2ayl6p"))

(define rust-libc-0.2.174
  (crate-source "libc" "0.2.174"
                "0xl7pqvw7g2874dy3kjady2fjr4rhj5lxsnxkkhr5689jcr6jw8i"))

(define rust-libc-0.2.175
  (crate-source "libc" "0.2.175"
                "0hw5sb3gjr0ivah7s3fmavlpvspjpd4mr009abmam2sr7r4sx0ka"))

(define rust-libc-0.2.176
  (crate-source "libc" "0.2.176"
                "0x7ivn80h7nz2l46vra7bxx36s6r8d0lkax14dx97skjsss2kyaq"))

(define rust-libc-stdhandle-0.1.0
  (crate-source "libc-stdhandle" "0.1.0"
                "0lbmg42kncl3yjgzx48d2sa3p78w4fmhs9c21dg4r4r8virj9b3d"))

(define rust-libcramjam-0.7.0
  (crate-source "libcramjam" "0.7.0"
                "0a5wkdlx4i9ri7pqzsfj3sk2z57h872ivhpmnr08s6dgk97c0iqd"))

(define rust-libdav1d-sys-0.6.0
  (crate-source "libdav1d-sys" "0.6.0"
                "0wh5jgdm33ld6djxsc7cmwd1ifqys145zlbsf8516n625lscrj8j"
                #:snippet '(delete-file-recursively "vendor")))

(define rust-libdbus-sys-0.2.5
  (crate-source "libdbus-sys" "0.2.5"
                "0wjw93q6ckrn8qdrxzdi02f0ma9g7nnlpgkrkcll1mjhnw95a206"
                #:snippet '(delete-file-recursively "vendor")))

(define rust-libdeflate-sys-1.19.3
  (crate-source "libdeflate-sys" "1.19.3"
                "0xzi0c9dl9jdzs758fvxmb9wmsqjm0a4b3ygxz2fhvncr1vam76c"
                #:snippet
                '(begin
                   ;; Prefer the system libdeflate.
                   (substitute* "Cargo.toml"
                     (("^dynamic") "default = [\"dynamic\"]\ndynamic"))
                   ;; Accept any version of libdeflate.
                   (substitute* "build.rs"
                     ((".*exactly_version.*") ""))
                   (delete-file-recursively "libdeflate"))))

(define rust-libdeflater-1.19.3
  (crate-source "libdeflater" "1.19.3"
                "00xxwh9x43fh7jb4cvi3mr0cyv2csjz7q45cnbi24pqysddrhni6"
                #:snippet '(delete-file-recursively "tests")))

(define rust-libdisplay-info-0.2.2
  (crate-source "libdisplay-info" "0.2.2"
                "0avs90mwzbfkwc09xlvvdy0szrbi670y61c1w0l75hqd7blwy422"))

(define rust-libdisplay-info-derive-0.1.0
  (crate-source "libdisplay-info-derive" "0.1.0"
                "1vpwss66rmhdd0f85c3nwjshddmiarf4iya5v13aacmp6q8d677a"))

(define rust-libdisplay-info-sys-0.2.2
  (crate-source "libdisplay-info-sys" "0.2.2"
                "0v34vjczpj8hzxnx1nj5cxwf326m91gn7bi3l3zkfg72xij94kvz"))

(define rust-libflate-2.1.0
  (crate-source "libflate" "2.1.0"
                "07mj9z89vbhq837q58m4v2nblgsmrn6vrp8w1j8g0kpa2kfdzna5"
                #:snippet '(delete-file-recursively "data")))

(define rust-libflate-lz77-2.1.0
  (crate-source "libflate_lz77" "2.1.0"
                "0gc6h98jwigscasz8vw1vv65b3rismqcbndb8hf6yf4z6qxxgq76"))

(define rust-libfuzzer-sys-0.4.10
  (crate-source "libfuzzer-sys" "0.4.10"
                "0124z86582vyzl8gbadqscjgf9i94jcpa9mxcpsyxjvh3w71jdsh"
                #:snippet '(delete-file-recursively "libfuzzer")))

(define rust-libfuzzer-sys-0.4.9
  (crate-source "libfuzzer-sys" "0.4.9"
                "0xfwg8shqvysl2bma2lyfcswbbdljajphflp795diwhc80nzay6g"
                #:snippet '(delete-file-recursively "libfuzzer")))

(define rust-libgit2-sys-0.12.26+1.3.0
  (crate-source "libgit2-sys" "0.12.26+1.3.0"
                "153l8nvz9p8vyd5840xi6fwblvhpn3c33jwdwsznyq4f4jcwiq8r"
                #:snippet '(delete-file-recursively "libgit2")))

(define rust-libgit2-sys-0.16.2+1.7.2
  (crate-source "libgit2-sys" "0.16.2+1.7.2"
                "1s149nkximw3n42925rp0jax1sf1zmf8gpcim2g9sp7fnkc2chgf"
                #:snippet '(delete-file-recursively "libgit2")))

(define rust-libgit2-sys-0.17.0+1.8.1
  (crate-source "libgit2-sys" "0.17.0+1.8.1"
                "093jxfl2i9vxdlgf7vk9d040sjwy0nq4fid640y7qix6m0k26iqh"
                #:snippet '(delete-file-recursively "libgit2")))

(define rust-libgit2-sys-0.18.1+1.9.0
  (crate-source "libgit2-sys" "0.18.1+1.9.0"
                "03i98nb84aa99bn7sxja11pllq6fghsaw4d3qwjxikgzhh7v5p71"
                #:snippet '(delete-file-recursively "libgit2")))

(define rust-libgit2-sys-0.18.2+1.9.1
  (crate-source "libgit2-sys" "0.18.2+1.9.1"
                "08n223x2pkf4gj6yrjmh3z6q236qj6nifwww78xcblrbvw1zwhhw"
                #:snippet '(delete-file-recursively "libgit2")))

(define rust-libloading-0.5.2
  (crate-source "libloading" "0.5.2"
                "0lyply8rcqc8agajzxs7bq6ivba9dnn1i68kgb9z2flnfjh13cgj"
                #:snippet '(delete-file-recursively "tests")))

(define rust-libloading-0.7.4
  (crate-source "libloading" "0.7.4"
                "17wbccnjvhjd9ibh019xcd8kjvqws8lqgq86lqkpbgig7gyq0wxn"
                #:snippet '(delete-file-recursively "tests")))

(define rust-libloading-0.8.6
  (crate-source "libloading" "0.8.6"
                "0d2ccr88f8kv3x7va2ccjxalcjnhrci4j2kwxp7lfmbkpjs4wbzw"
                #:snippet '(delete-file-recursively "tests")))

(define rust-libloading-0.8.7
  (crate-source "libloading" "0.8.7"
                "036zkka8hsdpli0j75wzd5svjkqdgypx6wdljm7wbsmgszq3syba"
                #:snippet '(delete-file-recursively "tests")))

(define rust-libloading-0.8.8
  (crate-source "libloading" "0.8.8"
                "0rw6q94psj3d6k0bi9nymqhyrz78lbdblryphhaszsw9p9ikj0q7"
                #:snippet '(delete-file-recursively "tests")))

(define rust-libloading-0.8.9
  (crate-source "libloading" "0.8.9"
                "0mfwxwjwi2cf0plxcd685yxzavlslz7xirss3b9cbrzyk4hv1i6p"
                #:snippet '(delete-file-recursively "tests")))

(define rust-libm-0.1.4
  (crate-source "libm" "0.1.4"
                "16pc0gx4gkg0q2s1ssq8268brn14j8344623vwhadmivc4lsmivz"))

(define rust-libm-0.2.11
  (crate-source "libm" "0.2.11"
                "1yjgk18rk71rjbqcw9l1zaqna89p9s603k7n327nqs8dn88vwmc3"))

(define rust-libm-0.2.15
  (crate-source "libm" "0.2.15"
                "1plpzf0p829viazdj57yw5dhmlr8ywf3apayxc2f2bq5a6mvryzr"))

(define rust-libm-0.2.7
  (crate-source "libm" "0.2.7"
                "1x1z4gayv1v0dzb78bkwb5nm932nk0w1sqa7gh4y2687pcdjn0gp"))

(define rust-libmacchina-8.1.0
  (crate-source "libmacchina" "8.1.0"
                "00di3k393j1xicqxmw3hx7q2b4xamj5hhv5nnwafm4ir14qf6z90"))

(define rust-libmdns-0.9.1
  (crate-source "libmdns" "0.9.1"
                "0qjq77vbmpspq943z686kfrb1jjz6vicws8v8cri848vw6cld1a8"))

(define rust-libmimalloc-sys-0.1.24
  (crate-source "libmimalloc-sys" "0.1.24"
                "0s8ab4nc33qgk9jybpv0zxcb75jgwwjb7fsab1rkyjgdyr0gq1bp"
                #:snippet
                '(begin
                   (delete-file-recursively "c_src")
                   (delete-file "build.rs")
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main() {~@
                        println!(\"cargo:rustc-link-lib=mimalloc\");~@
                        }~%"))))))

(define rust-libmimalloc-sys-0.1.41
  (crate-source "libmimalloc-sys" "0.1.41"
                "0yykxwnppvcwk8yfvniij65a945lqw7ykiakqyylvhaa7b5dl83b"
                #:snippet
                '(begin
                   (delete-file-recursively "c_src")
                   (delete-file "build.rs")
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main() {~@
                        println!(\"cargo:rustc-link-lib=mimalloc\");~@
                        }~%"))))))

(define rust-libmimalloc-sys-0.1.42
  (crate-source "libmimalloc-sys" "0.1.42"
                "1x60mbzyv4an8gjwp6h8k7f24w3bn36p2dg79v6sn7bn4yn6z7gc"
                #:snippet
                '(begin
                   (delete-file-recursively "c_src")
                   (delete-file "build.rs")
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main() {~@
                        println!(\"cargo:rustc-link-lib=mimalloc\");~@
                        }~%"))))))

(define rust-libmimalloc-sys-0.1.43
  (crate-source "libmimalloc-sys" "0.1.43"
                "0kbdw7d5vp7v70n8l6w3mvm6dbd3l50zdqnvh4biq9fyx5kwv25z"
                #:snippet
                '(begin
                   (delete-file-recursively "c_src")
                   (delete-file "build.rs")
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main() {~@
                        println!(\"cargo:rustc-link-lib=mimalloc\");~@
                        }~%"))))))

(define rust-libmimalloc-sys-0.1.44
  (crate-source "libmimalloc-sys" "0.1.44"
                "0w4q14nfpvwa8q4ygz72syd1cv3rj4nmhz1mqxmzr7gj43n4yzv6"
                #:snippet
                '(begin
                   (delete-file-recursively "c_src")
                   (delete-file "build.rs")
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main() {~@
                        println!(\"cargo:rustc-link-lib=mimalloc\");~@
                        }~%"))))))

(define rust-libnghttp2-sys-0.1.11+1.64.0
  (crate-source "libnghttp2-sys" "0.1.11+1.64.0"
                "1i0klzhn5s5y2v0am948qrk2wj7sfzakknhrf7xcyrviibj28v0v"
                #:snippet
                '(begin
                   (delete-file-recursively "nghttp2")
                   (delete-file "build.rs")
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main() {~@
                        println!(\"cargo:rustc-link-lib=nghttp2\");~@
                        }~%"))))))

(define rust-libpijul-1.0.0-beta.10
  (crate-source "libpijul" "1.0.0-beta.10"
                "0yjii3hh8s7cvp5aihngh49266iz9h5bwlfrcyd1dglinyv0f213"))

(define rust-libproc-0.14.10
  (crate-source "libproc" "0.14.10"
                "1yzbhywjzvhi4353pni6k8jbpildd1qp66d1banvrbg5dfshk2p7"))

(define rust-libpulse-binding-2.30.1
  (crate-source "libpulse-binding" "2.30.1"
                "15x3srbvqmwmhk20q9binq3cwakj59pgwnp61xl77qqnkq2b77lh"))

(define rust-libpulse-simple-binding-2.29.0
  (crate-source "libpulse-simple-binding" "2.29.0"
                "1vygcjjfy4d1ysbdwa71xhmlfdxbdzrsl96c4d5kx3hw73qbxgmp"))

(define rust-libpulse-simple-sys-1.22.0
  (crate-source "libpulse-simple-sys" "1.22.0"
                "1g7fb0ksbli4323a5bzn3j3j856arhnfixdb2q6jgb9pzs46in9v"))

(define rust-libpulse-sys-1.23.0
  (crate-source "libpulse-sys" "1.23.0"
                "0qayl38603vlzrfb9n2p2mqn0jfpxg9225nc57w8ks92if272hyp"))

(define rust-libradicl-0.5.1
  (crate-source "libradicl" "0.5.1"
                "0hgh395sys55z3xayg7pwibz0ymf83zj3bm111fvrq1jjh77zij4"))

(define rust-libredox-0.1.10
  (crate-source "libredox" "0.1.10"
                "1jswil4ai90s4rh91fg8580x8nikni1zl3wnch4h01nvidqpwvs1"))

(define rust-libredox-0.1.3
  (crate-source "libredox" "0.1.3"
                "139602gzgs0k91zb7dvgj1qh4ynb8g1lbxsswdim18hcb6ykgzy0"))

(define rust-libredox-0.1.4
  (crate-source "libredox" "0.1.4"
                "0f06ikfym363zrqy9llp4asgcbakz0aiq0ds0rkljdg52088100m"))

(define rust-libredox-0.1.9
  (crate-source "libredox" "0.1.9"
                "1qqczzfqcc3sw3bl7la6qv7i9hy1s7sxhxmdvpxkfgdd3c9904ir"))

(define rust-librespot-audio-0.6.0
  (crate-source "librespot-audio" "0.6.0"
                "1f5sd0a68v16irrzk9saj151295mxndxvfv1dj9l4c2mwxpmc1vy"))

(define rust-librespot-connect-0.6.0
  (crate-source "librespot-connect" "0.6.0"
                "0q7s2visnzxk5ddgsigh3l8k2j27g91zbkr5li8y17366ib3mkk7"))

(define rust-librespot-core-0.6.0
  (crate-source "librespot-core" "0.6.0"
                "1rs4bv62kx3r00d69cyv6bx74k8c8b5200iqz5crm56gxw1n7iy4"))

(define rust-librespot-discovery-0.6.0
  (crate-source "librespot-discovery" "0.6.0"
                "1a7yjx1xq9lalvrb5k3z2lqwnd58wg005483nswn4mbpik955acl"))

(define rust-librespot-metadata-0.6.0
  (crate-source "librespot-metadata" "0.6.0"
                "0pxyz32fklj5cmsl3603w6x4nsdab7lxrsv2w39is6s0db2dbxkw"))

(define rust-librespot-oauth-0.6.0
  (crate-source "librespot-oauth" "0.6.0"
                "0kyiz3a7nn634sq0ap56737myp1sm06a9qlkwi1b9hx6k9vsvp00"))

(define rust-librespot-playback-0.6.0
  (crate-source "librespot-playback" "0.6.0"
                "0r6zf72kyhlzlq7nsv7c56kmwqgj6kq4v3bz81sr532cl1vgglaf"))

(define rust-librespot-protocol-0.6.0
  (crate-source "librespot-protocol" "0.6.0"
                "0ih5ignz7srpl3zabq22p7vzn9x6hfjarrjlfhas5cx1nm92z040"))

(define rust-libseat-0.2.3
  (crate-source "libseat" "0.2.3"
                "0350b89h2xk5rdqqla162a2pak7yzbpfckqwg68cd42ppmdj8fn2"))

(define rust-libseat-sys-0.1.9
  (crate-source "libseat-sys" "0.1.9"
                "0997n2s413ggzi4amy0jbnfl8jvgpjnkxzycjs56ks2p0pjj2ihk"))

(define rust-libsensors-sys-0.2.0
  (crate-source "libsensors-sys" "0.2.0"
                "0cii4jbj19d95q6nyczfi3d3s2j3jg4qf7qc8qfxx5jb2v3hqwx1"))

(define rust-libsodium-sys-0.2.7
  (crate-source "libsodium-sys" "0.2.7"
                "1zcjka23grayr8kjrgbada6vwagp0kkni9m45v0gpbanrn3r6xvb"
                #:snippet '(for-each delete-file-recursively (list "libsodium" "mingw" "msvc"))))

(define rust-libspa-0.7.2
  (crate-source "libspa" "0.7.2"
                "1i58ayw50brjh1a7gzfs9ijwl42lqmq2k4b7c2w1ijnx41q62d04"))

(define rust-libspa-0.8.0
  (crate-source "libspa" "0.8.0"
                "044qs48yl0llp2dmrgwxj9y1pgfy09i6fhq661zqqb9a3fwa9wv5"))

(define rust-libspa-0.8.0.93138d0 rust-pipewire-0.8.0.93138d0)

(define rust-libspa-0.8.0.fd3d8f7 rust-pipewire-0.8.0.fd3d8f7)

(define rust-libspa-sys-0.7.2
  (crate-source "libspa-sys" "0.7.2"
                "0ifkg4k7azfqbz4i1lk31d77nhlc2w36s133yf78a3z7yfihrrxk"))

(define rust-libspa-sys-0.8.0
  (crate-source "libspa-sys" "0.8.0"
                "07yh4i5grzbxkchg6dnxlwbdw2wm5jnd7ffbhl77jr0388b9f3dz"))

(define rust-libspa-sys-0.8.0.93138d0 rust-pipewire-0.8.0.93138d0)

(define rust-libspa-sys-0.8.0.fd3d8f7 rust-pipewire-0.8.0.fd3d8f7)

(define rust-libsqlite3-sys-0.27.0
  (crate-source "libsqlite3-sys" "0.27.0"
                "05pp60ncrmyjlxxjj187808jkvpxm06w5lvvdwwvxd2qrmnj4kng"
                #:snippet
                '(for-each delete-file
                           (append
                            (find-files "sqlcipher" "\\.(c|h)$")
                            (find-files "sqlite3" "\\.(c|h)$")))))

(define rust-libsqlite3-sys-0.28.0
  (crate-source "libsqlite3-sys" "0.28.0"
                "0gzwfw0n2wqgaihcgj65wzd3lclfxyy62gixq8sv6z04fi15h40c"
                #:snippet
                '(for-each delete-file
                           (append
                            (find-files "sqlcipher" "\\.(c|h)$")
                            (find-files "sqlite3" "\\.(c|h)$")))))

(define rust-libsqlite3-sys-0.30.1
  (crate-source "libsqlite3-sys" "0.30.1"
                "0jcikvgbj84xc7ikdmpc8m4y5lyqgrb9aqblphwk67kv95xgp69f"
                #:snippet
                '(for-each delete-file
                           (append
                            (find-files "sqlcipher" "\\.(c|h)$")
                            (find-files "sqlite3" "\\.(c|h)$")))))

(define rust-libsqlite3-sys-0.32.0
  (crate-source "libsqlite3-sys" "0.32.0"
                "1rvgxy0hy7p4jib8hw1l5bcg2806v190rwlndrndf2q6nh5jgf7v"
                #:snippet
                '(for-each delete-file
                           (append
                            (find-files "sqlcipher" "\\.(c|h)$")
                            (find-files "sqlite3" "\\.(c|h)$")))))

(define rust-libssh2-sys-0.2.23
  (crate-source "libssh2-sys" "0.2.23"
                "1jplndqhlsygjmsni1ydb4zbw0j5jjr47bmqnjkwif5qnipa755h"
                #:snippet '(delete-file-recursively "libssh2")))

(define rust-libssh2-sys-0.3.1
  (crate-source "libssh2-sys" "0.3.1"
                "1f8i31h3666rl6sq7v64ajdq03hmylkh6c1vaf9828aaml2ly3i2"
                #:snippet '(delete-file-recursively "libssh2")))

(define rust-libtest-mimic-0.8.1
  (crate-source "libtest-mimic" "0.8.1"
                "0cwxnbv3wl00xsz7q0sf5xzksnqadqw8995a6g7a9pcyy4p9d5sj"))

(define rust-libudev-0.3.0
  (crate-source "libudev" "0.3.0"
                "1q1my5alvdwyi8i9pc9gn2mcx5rhbsssmz5cjnxzfpd65laj9cvq"))

(define rust-libudev-sys-0.1.4
  (crate-source "libudev-sys" "0.1.4"
                "09236fdzlx9l0dlrsc6xx21v5x8flpfm3d5rjq9jr5ivlas6k11w"))

(define rust-libusb1-sys-0.7.0
  (crate-source "libusb1-sys" "0.7.0"
                "03yfx469d1ldpw2h21hy322f5a0h1ahlgy4s6yjipzy4gbg0l1fs"
                #:snippet '(delete-file-recursively "libusb")))

(define rust-libwebp-sys-0.9.6
  (crate-source "libwebp-sys" "0.9.6"
                "0cv7hxzh9p66q5c4ay30bvffh0y66abwmr2nliscwrbigkgk1kal"
                #:snippet
                '(begin
                   (delete-file-recursively "tests")
                   ;; Remove bundled C sources.
                   (delete-file-recursively "vendor")
                   ;; Remove the original build.rs which builds bundled code.
                   (delete-file "build.rs")
                   ;; Replace it with a simple linker hint to use system libwebp.
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main() {~@
                        println!(\"cargo:rustc-link-lib=webp\");~@
                        }~%"))))))

(define rust-libz-ng-sys-1.1.22
  ;; TODO: Find a way to unbundle libz-ng.
  (crate-source "libz-ng-sys" "1.1.22"
                "096qkwzy596zf88nfppr2vbw9fbjfr81k2ws4zf6wyrw58n8q4d7"))

(define rust-libz-rs-sys-0.5.0
  (crate-source "libz-rs-sys" "0.5.0"
                "0nlc7cdcrh8mxqb08yw5i7ghgpcs1ixq4kk4sx19dzk0sydwm2b4"))

(define rust-libz-rs-sys-0.5.1
  (crate-source "libz-rs-sys" "0.5.1"
                "08a2grn3bp05696pd27s6kmq1icnbzffizl0nihic8m26y2phahp"))

(define rust-libz-sys-1.1.22
  (crate-source "libz-sys" "1.1.22"
                "07b5wxh0ska996kc0g2hanjhmb4di7ksm6ndljhr4pi0vykyfw4b"
                #:snippet '(for-each delete-file-recursively '("src/zlib" "src/zlib-ng"))))

(define rust-lifeguard-0.6.1
  (crate-source "lifeguard" "0.6.1"
                "09iwwy7888i1kfvbrgwpl4xv3pwsz1fbzx54djs3gnvmszdr9gl9"))

(define rust-lightningcss-1.0.0-alpha.66
  (crate-source "lightningcss" "1.0.0-alpha.66"
                "0m4f0m7ljddyz4shfkjg5k9gm28am922y8r7npj38rg6gnhzywws"
                #:snippet '(delete-file-recursively "website")))

(define rust-lightningcss-1.0.0-alpha.67
  (crate-source "lightningcss" "1.0.0-alpha.67"
                "0j91k4p56jlz810scp454x79213z7z658xzdp1bd7vh5297bm3vr"
                #:snippet '(delete-file-recursively "website")))

(define rust-lightningcss-derive-1.0.0-alpha.43
  (crate-source "lightningcss-derive" "1.0.0-alpha.43"
                "0lh2hxppjs53hjdlzkgzn0zxa9f3jkq9wws1xp56g4r7s522ghc4"))

(define rust-lindera-0.14.0
  (crate-source "lindera" "0.14.0"
                "0rbrl1qxyc9wfmnbb173vhxazn9cdkibd9i5j32hxb0wj88x1pad"))

(define rust-lindera-cc-cedict-builder-0.13.5
  (crate-source "lindera-cc-cedict-builder" "0.13.5"
                "03qbpiinsdcs0kh4s31r24n2ylipv1lzkah26bpr53vm3flr2i2q"))

(define rust-lindera-core-0.13.5
  (crate-source "lindera-core" "0.13.5"
                "1v43zwwaaw92rg8f25b88m0sx2pqsc1mw3ym0al229i8ycdyw9n7"))

(define rust-lindera-decompress-0.13.5
  (crate-source "lindera-decompress" "0.13.5"
                "0i37kmr72cr1s99nw5z32j9w85rzpq7gyr7frmb142iim67g77bg"))

(define rust-lindera-dictionary-0.13.5
  (crate-source "lindera-dictionary" "0.13.5"
                "19p44a78q2vwg17mx9jl9fc1wrac7yivnnkwj8krzzs28rjmnlha"))

(define rust-lindera-ipadic-0.13.5
  (crate-source "lindera-ipadic" "0.13.5"
                "1lmczx3w9n8x8dcgz92383hk6gljkqxxlcccvffp1h3xzhbf15s7"
                #:snippet '(delete-file-recursively "resources")))

(define rust-lindera-ipadic-builder-0.13.5
  (crate-source "lindera-ipadic-builder" "0.13.5"
                "147xh57wzpf290wxkhknxbq20qdn0il87yhklddhz0g80zxyrlwv"))

(define rust-lindera-ko-dic-builder-0.13.5
  (crate-source "lindera-ko-dic-builder" "0.13.5"
                "10rsmdpjyqnpg4nj8nnw3wy9pwqkis6hgk7f2cafjsgac1jz51pw"))

(define rust-lindera-unidic-builder-0.13.5
  (crate-source "lindera-unidic-builder" "0.13.5"
                "10kv4qmjsni9mldyjds9rykxz5pgnq1wdqlyjiyyrzrqgn5vph85"))

(define rust-line-col-0.2.1
  (crate-source "line-col" "0.2.1"
                "1nvqjxn2kkxkph3qdlkc5xq1m2ygla4l0saga778sp2vp3vcsscy"))

(define rust-line-numbers-0.3.1
  (crate-source "line-numbers" "0.3.1"
                "1p9fdrf3283zvpn8c008pbywx4p44di6ij3v7jjis36gyyqh5jr0"))

(define rust-line-span-0.1.5
  (crate-source "line-span" "0.1.5"
                "12j07vn0vnchlvlrh3cc3320jaz07sggcj42l6f0j0365wxi5z19"))

(define rust-linear-map-1.2.0
  (crate-source "linear-map" "1.2.0"
                "1vh3sczl4xb5asdlpafdf3y4g9bp63fgs8y2a2sjgmcsn7v21bmz"))

(define rust-linecount-0.1.0
  (crate-source "linecount" "0.1.0"
                "1n2733qcmvxl3fra3924yj8sf21vr6f6g6dg6wnhbkxr8fia9mfm"))

(define rust-link-cplusplus-1.0.10
  (crate-source "link-cplusplus" "1.0.10"
                "04h26gpg1k6g9qh61hbhvs1h35p2hyqhannny3ggjs7r0yh6svsa"))

(define rust-link-cplusplus-1.0.12
  (crate-source "link-cplusplus" "1.0.12"
                "10lcgfp9pnxpihp21s86xnq57vpr97m2k419d8rvkl57m8qcfy3z"))

(define rust-link-cplusplus-1.0.9
  (crate-source "link-cplusplus" "1.0.9"
                "1jc56xf51f3fn5hvppsj9c2pa0fxm7k79xs90a5s58qvgrphq94x"))

(define rust-linked-hash-map-0.5.6
  (crate-source "linked-hash-map" "0.5.6"
                "03vpgw7x507g524nx5i1jf5dl8k3kv0fzg8v3ip6qqwbpkqww5q7"))

(define rust-linkify-0.7.0
  (crate-source "linkify" "0.7.0"
                "1y99xcmy6793m6x3z2qcyiw3dfrdvza3n659lmlv7kpzq7yjin04"))

(define rust-linux-keyutils-0.2.4
  (crate-source "linux-keyutils" "0.2.4"
                "13nipvk2mzk76y7yfsqwnwsqk21x6xy8fkmqz5is99fqbzn4j7kn"))

(define rust-linux-raw-sys-0.11.0
  (crate-source "linux-raw-sys" "0.11.0"
                "0fghx0nn8nvbz5yzgizfcwd6ap2pislp68j8c1bwyr6sacxkq7fz"))

(define rust-linux-raw-sys-0.3.8
  (crate-source "linux-raw-sys" "0.3.8"
                "068mbigb3frrxvbi5g61lx25kksy98f2qgkvc4xg8zxznwp98lzg"))

(define rust-linux-raw-sys-0.4.14
  (crate-source "linux-raw-sys" "0.4.14"
                "12gsjgbhhjwywpqcrizv80vrp7p7grsz5laqq773i33wphjsxcvq"))

(define rust-linux-raw-sys-0.4.15
  (crate-source "linux-raw-sys" "0.4.15"
                "1aq7r2g7786hyxhv40spzf2nhag5xbw2axxc1k8z5k1dsgdm4v6j"))

(define rust-linux-raw-sys-0.4.5
  ;; TODO: Check bundled sources.
  (crate-source "linux-raw-sys" "0.4.5"
                "00w52pb2cb4b2880ksyzagmzbyjdmp9ac0w3qfvjv3453fnzvg2p"))

(define rust-linux-raw-sys-0.6.5
  (crate-source "linux-raw-sys" "0.6.5"
                "1mv3c1zz51ydcj768zavm8g06gz5jb1p7yigmmif7hz5whdmnf1a"))

(define rust-linux-raw-sys-0.9.2
  (crate-source "linux-raw-sys" "0.9.2"
                "1s89d3ykla46h6i3z42972gnlm9xbdlyj1kmgdbxr1zhva1wdfbd"))

(define rust-linux-raw-sys-0.9.3
  (crate-source "linux-raw-sys" "0.9.3"
                "04zl7j4k1kgbn7rrl3i7yszaglgxp0c8dbwx8f1cabnjjwhb2zgy"))

(define rust-linux-raw-sys-0.9.4
  (crate-source "linux-raw-sys" "0.9.4"
                "04kyjdrq79lz9ibrf7czk6cv9d3jl597pb9738vzbsbzy1j5i56d"))

(define rust-lipsum-0.9.1
  (crate-source "lipsum" "0.9.1"
                "0r40mf2cwh4fp9pdfcc1n8hjxw05w7galjvb1z23r5pq38jn0s33"))

(define rust-litemap-0.7.3
  (crate-source "litemap" "0.7.3"
                "0157lf44c3s2piqiwpppnynzzpv1rxyddl2z9l089hpwsjwb0g34"))

(define rust-litemap-0.7.5
  (crate-source "litemap" "0.7.5"
                "0mi8ykav0s974ps79p438x04snh0cdb7lc864b42jws5375i9yr3"))

(define rust-litemap-0.8.0
  (crate-source "litemap" "0.8.0"
                "0mlrlskwwhirxk3wsz9psh6nxcy491n0dh8zl02qgj0jzpssw7i4"))

(define rust-litrs-0.4.1
  (crate-source "litrs" "0.4.1"
                "19cssch9gc0x2snd9089nvwzz79zx6nzsi3icffpx25p4hck1kml"))

(define rust-litrs-0.4.2
  (crate-source "litrs" "0.4.2"
                "1v8bxsrkm0w2k9nmbp8hsspy9i1lawajywqdw4hx87rjzqv41rgm"))

(define rust-lld-rs-140.0.0
  (crate-source "lld_rs" "140.0.0"
                "13xzaf4dh5ljy9rrlsgy3dvs1va0cfxbwqawbz2pkbcjjqp6pbxs"
                #:snippet
                '(begin
                   (substitute* "build.rs"
                     (("println\\!\\(\"cargo:rustc-link-lib=static=lldWasm\"\\);" all)
                      (string-append all
                                     ;; add all input library paths to the search path
                                     "let key = \"LIBRARY_PATH\";"
                                     "match env::var_os(key) {"
                                     "Some(paths) => {"
                                     "for path in env::split_paths(&paths) {"
                                     "println!(\"cargo:rustc-link-search=native={}\", path.display());}}"
                                     "None => println!(\"{key} not defined in environment.\")}"
                                     ;; add zlib link flag which is omitted
                                     "println!(\"cargo:rustc-link-lib=dylib=z\");"))
                     ;; the `-std=c++17' flag is overwritten by `llvm-config --cxxflags`
                     ;; which breaks the build as `build.rs' uses c++17 features, so
                     ;; re-add it.
                     (("\\.join\\(\" \"\\)" all)
                      (string-append all "+\"-std=c++17\""))
                     ;; llvm-14 doesn't have static outputs for whatever reason, so we
                     ;; need to force use of dynamic libs.
                     (("\".a\"")
                      "\".so\"")
                     ;; compensate for the longer suffix.
                     (("\\.len\\(\\) \\- 2")
                      ".len() - 3")
                     ;; force dynamic linking for llvm libraries.
                     (("static=\\{\\}")
                      "dylib={}")
                     (("\"--link-static\"")
                      "\"--link-shared\"")))))

(define rust-local-channel-0.1.5
  (crate-source "local-channel" "0.1.5"
                "1j1ywn459kl4fdmjfyljm379k40qwwscd7mqp25lppxqd5gcijxn"))

(define rust-llvm-sys-140.1.3
  (crate-source "llvm-sys" "140.1.3"
                "030srgq5vgshgpmb0w05gpf3yjbh73vcrgg327n320kwhplpip73"
                #:snippet
                '(begin
                   (substitute* "Cargo.toml.orig"
                     (("\\[features\\]" all)
                      (string-append all "\n default=[\"force-dynamic\"]")))
                   (copy-file "Cargo.toml.orig" "Cargo.toml"))))

(define rust-local-ip-address-0.5.7
  (crate-source "local-ip-address" "0.5.7"
                "0pfiy5pj70y0ycdf3scmxpgmj7jyi99054rk4rfzpb75kkmd8bk1"))

(define rust-local-waker-0.1.4
  (crate-source "local-waker" "0.1.4"
                "11vlcm8q6dhdf0srkgjnwca48dn9zcz820fq20hv82ffcxy3v1sd"))

(define rust-locale-0.2.2
  (crate-source "locale" "0.2.2"
                "1z87wc7z6889x1pqlrwjw8f1crshzi15q5m102lqs8y0m69f9nsz"))

(define rust-locale-config-0.3.0
  (crate-source "locale_config" "0.3.0"
                "0d399alr1i7h7yji4vydbdbzd8hp0xaykr7h4rn3yj7l2rdw7lh8"))

(define rust-localtime-1.3.1
  (crate-source "localtime" "1.3.1"
                "11vdz49sqqw8ydgg6vnc892z4vpym08vyx7v54r6xfmq1fg00sh1"))

(define rust-lock-api-0.4.12
  (crate-source "lock_api" "0.4.12"
                "05qvxa6g27yyva25a5ghsg85apdxkvr77yhkyhapj6r8vnf8pbq7"))

(define rust-lock-api-0.4.13
  (crate-source "lock_api" "0.4.13"
                "0rd73p4299mjwl4hhlfj9qr88v3r0kc8s1nszkfmnq2ky43nb4wn"))

(define rust-lockfree-object-pool-0.1.6
  (crate-source "lockfree-object-pool" "0.1.6"
                "0bjm2g1g1avab86r02jb65iyd7hdi35khn1y81z4nba0511fyx4k"))

(define rust-log-0.4.20
  (crate-source "log" "0.4.20"
                "13rf7wphnwd61vazpxr7fiycin6cb1g8fmvgqg18i464p0y1drmm"))

(define rust-log-0.4.26
  (crate-source "log" "0.4.26"
                "17mvchkvhnm2zxyfagh2g9p861f0qx2g1sg2v14sww9nvjry5g9h"))

(define rust-log-0.4.27
  (crate-source "log" "0.4.27"
                "150x589dqil307rv0rwj0jsgz5bjbwvl83gyl61jf873a7rjvp0k"))

(define rust-log-0.4.28
  (crate-source "log" "0.4.28"
                "0cklpzrpxafbaq1nyxarhnmcw9z3xcjrad3ch55mmr58xw2ha21l"))

(define rust-log-reroute-0.1.8
  (crate-source "log-reroute" "0.1.8"
                "00mw91qd2ibaawl7x1pxc1kryki0ixyirnlx64qx78d9g6k3n6kl"))

(define rust-logos-0.12.1
  (crate-source "logos" "0.12.1"
                "1w82qm3hck5cr6ax3j3yzrpf4zzbffahz126ahyqwyn6h8b072xz"))

(define rust-logos-0.15.1
  (crate-source "logos" "0.15.1"
                "0m41xcjn6yh3v18618v9f04v7vkmf3zn07y5c68xkhjfkf4jyizz"))

(define rust-logos-codegen-0.15.1
  (crate-source "logos-codegen" "0.15.1"
                "0p04jfvaaiw2rj4kzk1s4hlmwhbwvgn3xi5jl0kmph5hj0mklahr"))

(define rust-logos-derive-0.12.1
  (crate-source "logos-derive" "0.12.1"
                "0v295x78vcskab88hshl530w9d1vn61cmlaic4d6dydsila4kn51"))

(define rust-logos-derive-0.15.1
  (crate-source "logos-derive" "0.15.1"
                "0w5l4qm67b551pnx3dksbyia9mm339a53z4fsd13mvympjbrcpb0"))

(define rust-loom-0.5.6
  (crate-source "loom" "0.5.6"
                "1x9hmiv00ir79kypcg0jdw0j0fkd3ymq6rpv6pli6q5qifrfql7z"))

(define rust-loom-0.7.2
  (crate-source "loom" "0.7.2"
                "1jpszf9qxv8ydpsm2h9vcyvxvyxcfkhmmfbylzd4gfbc0k40v7j1"))

(define rust-loop9-0.1.5
  (crate-source "loop9" "0.1.5"
                "0qphc1c0cbbx43pwm6isnwzwbg6nsxjh7jah04n1sg5h4p0qgbhg"))

(define rust-lopdf-0.32.0
  (crate-source "lopdf" "0.32.0"
                "0aw7diz39z3mk22k0mp7jk7qiaaagfvggzly1baqg2jf4vpf8xg7"
                #:snippet '(delete-file-recursively "assets")))

(define rust-loupe-0.1.3
  (crate-source "loupe" "0.1.3"
                "0kb3bc62xh20i405afr8bf65m6gznbp0fhxrfrg5pqaglkgp4slv"))

(define rust-loupe-derive-0.1.3
  (crate-source "loupe-derive" "0.1.3"
                "0ljrdhp4yk87xnbdq53f679yzm7yghanxq4s5sgjfs3i6f4gryy0"))

(define rust-lru-0.10.1
  (crate-source "lru" "0.10.1"
                "0w5n2sgh66ac8ihqv6688mlm7zb3ks18jlbzpbhwgw3x8jp8z3ki"))

(define rust-lru-0.12.4
  (crate-source "lru" "0.12.4"
                "017rzh4kyl3j79sj0qc35wallblsqbnkzxpn6i3xkrv02y4kkvip"))

(define rust-lru-0.12.5
  (crate-source "lru" "0.12.5"
                "0f1a7cgqxbyhrmgaqqa11m3azwhcc36w0v5r4izgbhadl3sg8k13"))

(define rust-lru-0.7.8
  (crate-source "lru" "0.7.8"
                "0yp4ai5rpr2czxklzxxx98p6l2aqv4g1906j3dr4b0vfgfxbx6g9"))

(define rust-lru-cache-0.1.2
  (crate-source "lru-cache" "0.1.2"
                "071viv6g2p3akwqmfb3c8vsycs5n7kr17b70l7la071jv0d4zqii"))

(define rust-lru-slab-0.1.2
  (crate-source "lru-slab" "0.1.2"
                "0m2139k466qj3bnpk66bwivgcx3z88qkxvlzk70vd65jq373jaqi"))

(define rust-lscolors-0.13.0
  (crate-source "lscolors" "0.13.0"
                "1wnxs5d004fx71apvh9124xqky0qjjmpibag24km7bvvss2xrpn2"))

(define rust-lscolors-0.16.0
  (crate-source "lscolors" "0.16.0"
                "0yqfym0lfdkj0f7lpvd0a0brljs7cpknyh14c202frcpqfg202xb"))

(define rust-lscolors-0.17.0
  (crate-source "lscolors" "0.17.0"
                "08z5jslgigvnpc1gj2i8r9pi8yn4m0pf8dzf3rk9grdidbzlyc2k"))

(define rust-lscolors-0.19.0
  (crate-source "lscolors" "0.19.0"
                "115bffi44w028n26vqxyknwn9pzp0axz8839846cg0dl32yg7xam"))

(define rust-lsp-server-0.7.8
  (crate-source "lsp-server" "0.7.8"
                "1yanavncgsx0i0rj65q12ddfcwpvzx5x8wgiq4g9fzz1fgfc8qll"))

(define rust-lsp-textdocument-0.4.2
  (crate-source "lsp-textdocument" "0.4.2"
                "0b7dxa7x3v5z58nj5rsscv1kpifasbdp5x0v0wycsgafbxclsmid"))

(define rust-lsp-types-0.95.1
  (crate-source "lsp-types" "0.95.1"
                "0ra36fd4yr7lf5igfrdvwjx9g87z3a99mrjgzk9nq04viqxd6d4f"))

(define rust-lsp-types-0.97.0
  (crate-source "lsp-types" "0.97.0"
                "0wb0yr2cdhlndjkcfyabr17ib0nvqa4v3zl5qm3aq13wl583adak"))

(define rust-lv2-0.6.0
  (crate-source "lv2" "0.6.0"
                "1xh4hjfh2w5rhzbk0g9845k25f6fxrv7xqpkr09p0x57b200qc41"))

(define rust-lv2-atom-2.0.0
  (crate-source "lv2-atom" "2.0.0"
                "0wd9rgsn8sag8wyhjccmnn82gx4w1yyiav52nyvk579l21xlw6wm"))

(define rust-lv2-core-3.0.0
  (crate-source "lv2-core" "3.0.0"
                "1pj9l15zwqwj2h83f3xfpwxsj70vvhkw52gyzkljafvrbx1h00fm"))

(define rust-lv2-core-derive-2.1.1
  (crate-source "lv2-core-derive" "2.1.1"
                "12w3l41jzargrcywz13hbmaazfw4ix2sljl3601h6jfbdrw8zybv"))

(define rust-lv2-midi-1.2.0
  (crate-source "lv2-midi" "1.2.0"
                "0x0glbrfri1glgcrmvc6i1jfv6azhpqvp4ibk5cihsq3s2yfc8xd"))

(define rust-lv2-sys-2.0.0
  (crate-source "lv2-sys" "2.0.0"
                "0c4f59mrjyy0z0wf033wp648df0sc6zirrcd6kndqj9nvvkzkl4x"))

(define rust-lv2-units-0.1.3
  (crate-source "lv2-units" "0.1.3"
                "0fdamp3hxdr36hqi1j6y01rz1x17if1ibzr7rr4nrabidw74gf82"))

(define rust-lv2-urid-2.1.0
  (crate-source "lv2-urid" "2.1.0"
                "0s2fcb0nyn54ml6azkbhnnxghy898x1q5vs5qgdznrhy9m20624c"))

(define rust-lz4-1.28.1
  (crate-source "lz4" "1.28.1"
                "1x2svvs3gkn3krv61nd7ms4vmikibsnfl31mk0z480qdhqz542x2"))

(define rust-lz4-flex-0.10.0
  (crate-source "lz4_flex" "0.10.0"
                "10sgbj93sagbl0ngzqvnlkldzbfz5vnzr7fry8sgssy299cp534b"))

(define rust-lz4-flex-0.11.3
  (crate-source "lz4_flex" "0.11.3"
                "1xg3h3y0ghnq3widdssd36s02pvy29c0afbwgq6mh3ibmri12xkm"))

(define rust-lz4-sys-1.11.1+lz4-1.10.0
  (crate-source "lz4-sys" "1.11.1+lz4-1.10.0"
                "1rhqnhwq05fmlc2q39ipsq0vpi0xf6w6p22j6q5x637dqvbc1n3b"
                #:snippet
                '(begin
                   (delete-file-recursively "liblz4")
                   (delete-file "build.rs")
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main() {~@
                        println!(\"cargo:rustc-link-lib=lz4\");~@
                        }~%"))))))

(define rust-lzma-rs-0.2.0
  (crate-source "lzma-rs" "0.2.0"
                "1cbz5pdsjkjn71ba1bgp467zxpvm1bnmw23jmpjbryhd8nqfra5b"))

(define rust-lzma-rust-0.1.7
  (crate-source "lzma-rust" "0.1.7"
                "0q96pl4wfd4hsdpkidrgv5bkk40f4xwzzsbisr2a2xbxpnxv5ajv"))

(define rust-lzma-rust2-0.2.0
  (crate-source "lzma-rust2" "0.2.0"
                "06h542wsgdnk3m63rggfzn1yghsamvjmvhsvqld681wv5lg167an"))

(define rust-lzma-sys-0.1.20
  (crate-source "lzma-sys" "0.1.20"
                "09sxp20waxyglgn3cjz8qjkspb3ryz2fwx4rigkwvrk46ymh9njz"
                #:snippet
                '(begin
                   (delete-file-recursively "xz-5.2")
                   (substitute* "build.rs"
                     (("(want_static = ).*" _ prefix)
                      (string-append prefix "false;\n"))))))

(define rust-mac-0.1.1
  (crate-source "mac" "0.1.1"
                "194vc7vrshqff72rl56f9xgb0cazyl4jda7qsv31m5l6xx7hq7n4"))

(define rust-mac-address-1.1.8
  (crate-source "mac_address" "1.1.8"
                "00r3n18mxglq1dzshnm0vxk1fgsp3c2hd08w6hfcqdp8ymmv5bn0"))

(define rust-mac-notification-sys-0.6.4
  (crate-source "mac-notification-sys" "0.6.4"
                "1vv7qfz14p01kb52ly5n2ka74fjy64zgjav28lmmklbi82rxz58b"))

(define rust-mach-0.3.2
  (crate-source "mach" "0.3.2"
                "1yksa8lwzqh150gr4417rls1wk20asy9vhp8kq5g9n7z58xyh8xq"))

(define rust-mach-o-sys-0.1.1
  (crate-source "mach_o_sys" "0.1.1"
                "09l8p7nmzq37450x2h6nb7dzg1sk6dk36a5rkcrcy81zm21lb19y"))

(define rust-mach2-0.4.2
  (crate-source "mach2" "0.4.2"
                "02gpyq89rcrqdbz4hgp5bpjas21dllxfc70jgw8vj0iaxg6mbf8r"))

(define rust-mach2-0.4.3
  (crate-source "mach2" "0.4.3"
                "0i6vcnbq5v51whgyidzhf7cbxqrmj2nkw8z0m2ib02rc60mjhh6n"))

(define rust-macro-rules-attribute-0.2.0
  (crate-source "macro_rules_attribute" "0.2.0"
                "04waa4qm28adwnxsxhx9135ki68mwkikr6m5pi5xhcy0gcgjg0la"))

(define rust-macro-rules-attribute-proc-macro-0.2.0
  (crate-source "macro_rules_attribute-proc_macro" "0.2.0"
                "0s45j4zm0a5d041g3vcbanvr76p331dfjb7gw9qdmh0w8mnqbpdq"))

(define rust-maildir-0.6.4
  (crate-source "maildir" "0.6.4"
                "0dqlkvhcrigs4y5vg0jf8ccgwns8jj85cjp6vsgj3f1sfkk6m6l7"))

(define rust-mailparse-0.14.1
  (crate-source "mailparse" "0.14.1"
                "0mwpkxi41ak8pabknmvj49dpsbs0q6w7w12f0lp49avcjaa6a29d"
                #:snippet '(delete-file-recursively "tests")))

(define rust-mailparse-0.15.0
  (crate-source "mailparse" "0.15.0"
                "0zkwbrzgr7pp1wyywjgvlxayr1p3nnkn2yxgi97746j1h1ckv81x"))

(define rust-mailparse-0.16.1
  (crate-source "mailparse" "0.16.1"
                "0bzdd3yliadzjnirb4g0jwz7j3k2yds02fzb2ib1m0ybvnbrm0b0"
                #:snippet '(delete-file-recursively "tests")))

(define rust-malloc-buf-0.0.6
  (crate-source "malloc_buf" "0.0.6"
                "1jqr77j89pwszv51fmnknzvd53i1nkmcr8rjrvcxhm4dx1zr1fv2"))

(define rust-maplit-1.0.2
  (crate-source "maplit" "1.0.2"
                "07b5kjnhrrmfhgqm9wprjw8adx6i225lqp49gasgqg74lahnabiy"))

(define rust-marked-yaml-0.7.2
  (crate-source "marked-yaml" "0.7.2"
                "13ddwcjd55ysa5ccw22zawy16p72abmdyx3ssrc40vqlmfkjbszj"))

(define rust-markup5ever-0.10.1
  (crate-source "markup5ever" "0.10.1"
                "1zf8iq2czd6kz99fjs3pgf5c17lfz75ds31khkfiqbc50gxl0kx2"))

(define rust-markup5ever-0.11.0
  (crate-source "markup5ever" "0.11.0"
                "05mhzsp6lfxla1fgd0ac283b405s6kyj27wj5r6d7wq42jxjj9ks"))

(define rust-markup5ever-0.12.1
  (crate-source "markup5ever" "0.12.1"
                "0idy4vjihg2dw73j2vkb5kdghvga3bwnw0qx8jwci4m6xfxkmkhn"))

(define rust-markup5ever-0.35.0
  (crate-source "markup5ever" "0.35.0"
                "1hy1xh07jkm13j7vdnsphynl3z7hfmh99csjjvqzhl26jfffc7ri"))

(define rust-markup5ever-rcdom-0.3.0
  (crate-source "markup5ever_rcdom" "0.3.0"
                "065yb6zn9sfn7kqk5wwc48czsls5z3hzgrddk58fxgq16ymj3apd"))

(define rust-match-cfg-0.1.0
  (crate-source "match_cfg" "0.1.0"
                "1r5j3zqc3qr8ybcx95bk8q57mkizmgmffj5lmicd4i8d9riyigpz"))

(define rust-match-token-0.35.0
  (crate-source "match_token" "0.35.0"
                "1ksqd8li4kdd463cb2qf10d7d4j1m416y62xbzf47k0g6qzzv15c"))

(define rust-matchers-0.1.0
  (crate-source "matchers" "0.1.0"
                "0n2mbk7lg2vf962c8xwzdq96yrc9i0p8dbmm4wa1nnkcp1dhfqw2"))

(define rust-matchers-0.2.0
  (crate-source "matchers" "0.2.0"
                "1sasssspdj2vwcwmbq3ra18d3qniapkimfcbr47zmx6750m5llni"))

(define rust-matches-0.1.10
  (crate-source "matches" "0.1.10"
                "1994402fq4viys7pjhzisj4wcw894l53g798kkm2y74laxk0jci5"))

(define rust-matchit-0.7.3
  (crate-source "matchit" "0.7.3"
                "156bgdmmlv4crib31qhgg49nsjk88dxkdqp80ha2pk2rk6n6ax0f"))

(define rust-matchit-0.8.4
  (crate-source "matchit" "0.8.4"
                "1hzl48fwq1cn5dvshfly6vzkzqhfihya65zpj7nz7lfx82mgzqa7"))

(define rust-material-colors-0.4.2
  (crate-source "material-colors" "0.4.2"
                "0hbrwk94i2wxih894pg30pxys7l5l0vxcc7pp1mw4nrd6vdgy9p2"))

(define rust-matrixmultiply-0.2.4
  (crate-source "matrixmultiply" "0.2.4"
                "1hc4vp19x823xgkm374wsxnzmqbjhmyaj5nr0lhm9k9i02x0cs4i"))

(define rust-matrixmultiply-0.3.9
  (crate-source "matrixmultiply" "0.3.9"
                "06msav241ybxvsqfwm4hfmb1pbws71v0inhmyk0i0vg9wc8vk04k"))

(define rust-maybe-async-0.2.10
  (crate-source "maybe-async" "0.2.10"
                "04fvg2ywb2p9dzf7i35xqfibxc05k1pirv36jswxcqg3qw82ryaw"))

(define rust-maybe-rayon-0.1.1
  (crate-source "maybe-rayon" "0.1.1"
                "06cmvhj4n36459g327ng5dnj8d58qs472pv5ahlhm7ynxl6g78cf"))

(define rust-md-5-0.10.6
  (crate-source "md-5" "0.10.6"
                "1kvq5rnpm4fzwmyv5nmnxygdhhb2369888a06gdc9pxyrzh7x7nq"
                #:snippet '(delete-file-recursively "tests")))

(define rust-md5-0.7.0
  (crate-source "md5" "0.7.0"
                "0wcps37hrhz59fkhf8di1ppdnqld6l1w5sdy7jp7p51z0i4c8329"))

(define rust-md5-0.8.0
  (crate-source "md5" "0.8.0"
                "1q6jfsa5w3993dzymxkv9jxpp7vyhgga6z35g6c0c8rk50w0i5mf"))

(define rust-mdbook-0.4.52
  (crate-source "mdbook" "0.4.52"
                "1v1h40c6kdygqwlm9xdn5ly7viygjzc9mkqrb5yay5jrhp989hlk"
                #:snippet '(delete-file-recursively "src/front-end/fonts")))

(define rust-memchr-2.5.0
  (crate-source "memchr" "2.5.0"
                "0vanfk5mzs1g1syqnj03q8n0syggnhn55dq535h2wxr7rwpfbzrd"))

(define rust-memchr-2.6.3
  (crate-source "memchr" "2.6.3"
                "0p6kn2awqf47m3brk0nmajarhwhylg9969il8dm9bq87yxp2s8wg"))

(define rust-memchr-2.7.4
  (crate-source "memchr" "2.7.4"
                "18z32bhxrax0fnjikv475z7ii718hq457qwmaryixfxsl2qrmjkq"
                #:snippet '(delete-file-recursively "src/tests")))

(define rust-memchr-2.7.5
  (crate-source "memchr" "2.7.5"
                "1h2bh2jajkizz04fh047lpid5wgw2cr9igpkdhl3ibzscpd858ij"))

(define rust-memchr-2.7.6
  (crate-source "memchr" "2.7.6"
                "0wy29kf6pb4fbhfksjbs05jy2f32r2f3r1ga6qkmpz31k79h0azm"))

(define rust-memfd-0.6.4
  (crate-source "memfd" "0.6.4"
                "0r5cm3wzyr1x7768h3hns77b494qbz0g05cb9wgpjvrcsm5gmkxj"))

(define rust-memmap2-0.3.1
  (crate-source "memmap2" "0.3.1"
                "0mz3fdcq443m3a1phrhp2yvd1h9vrvbhinzmi23ik031zzmw5dh0"))

(define rust-memmap2-0.5.10
  (crate-source "memmap2" "0.5.10"
                "09xk415fxyl4a9pgby4im1v2gqlb5lixpm99dczkk30718na9yl3"))

(define rust-memmap2-0.8.0
  (crate-source "memmap2" "0.8.0"
                "1vf3djv9s917fbvw5vclllpl22g12iph6cz11gn57ndhxwya19a3"))

(define rust-memmap2-0.9.4
  (crate-source "memmap2" "0.9.4"
                "08hkmvri44j6h14lyq4yw5ipsp91a9jacgiww4bs9jm8whi18xgy"))

(define rust-memmap2-0.9.5
  (crate-source "memmap2" "0.9.5"
                "0krpvvkpg4i3l05cv3q2xk24a1vj5c86gbrli2wzhj1qkpnpwgzx"))

(define rust-memmap2-0.9.8
  (crate-source "memmap2" "0.9.8"
                "1dqxjs89krh8cin0k7ksqc9myw9yni9kn8d8cllwq4fn1isrhfl4"))

(define rust-memmem-0.1.1
  (crate-source "memmem" "0.1.1"
                "05ccifqgxdfxk6yls41ljabcccsz3jz6549l1h3cwi17kr494jm6"))

(define rust-memo-map-0.3.3
  (crate-source "memo-map" "0.3.3"
                "0yw5ria5j6kd8z0p2yiji1vc2x53zg348dv1fgip822n0x813l9q"))

(define rust-memoffset-0.6.5
  (crate-source "memoffset" "0.6.5"
                "1kkrzll58a3ayn5zdyy9i1f1v3mx0xgl29x0chq614zazba638ss"))

(define rust-memoffset-0.7.1
  (crate-source "memoffset" "0.7.1"
                "1x2zv8hv9c9bvgmhsjvr9bymqwyxvgbca12cm8xkhpyy5k1r7s2x"))

(define rust-memoffset-0.8.0
  (crate-source "memoffset" "0.8.0"
                "1qcdic88dhgw76pafgndpz04pig8il4advq978mxdxdwrydp276n"))

(define rust-memoffset-0.9.0
  (crate-source "memoffset" "0.9.0"
                "0v20ihhdzkfw1jx00a7zjpk2dcp5qjq6lz302nyqamd9c4f4nqss"))

(define rust-memoffset-0.9.1
  (crate-source "memoffset" "0.9.1"
                "12i17wh9a9plx869g7j4whf62xw68k5zd4k0k5nh6ys5mszid028"))

(define rust-memsec-0.7.0
  (crate-source "memsec" "0.7.0"
                "14ml8sva5apz4vg5pxlsfmf4y8a9prqzhrcw6vyb5ai3pgbbk5y7"))

(define rust-merge-0.1.0
  (crate-source "merge" "0.1.0"
                "1sck0vhi9lk8a6mgky0rgn842fj7yspywidwbd963nmimf9yzfqh"))

(define rust-merge-derive-0.1.0
  (crate-source "merge_derive" "0.1.0"
                "01wxhi7mqmp34l540wcfb24hb13vcbps4wlynas66bnsfra0g790"))

(define rust-metal-0.18.0
  (crate-source "metal" "0.18.0"
                "1h1dx42sdkn4jl1rnjisv687vd5jyck982wxq3r9xgmx8bpa1671"
                #:snippet '(delete-file-recursively "examples")))

(define rust-metrohash-1.0.7
  (crate-source "metrohash" "1.0.7"
                "1ifqmkg81nprxm9dv52k6wbcv48aibd830f3ph7zcff3mnzi2h58"))

(define rust-miette-5.10.0
  (crate-source "miette" "5.10.0"
                "0vl5qvl3bgha6nnkdl7kiha6v4ypd6d51wyc4q1bvdpamr75ifsr"))

(define rust-miette-7.5.0
  (crate-source "miette" "7.5.0"
                "114lv0nx46lxc5pncz6iyrzcfhn5g9a5janzc8cgsdvvz1jm358s"))

(define rust-miette-7.6.0
  (crate-source "miette" "7.6.0"
                "1dwjnnpcff4jzpf5ns1m19di2p0n5j31zmjv5dskrih7i3nfz62z"))

(define rust-miette-derive-5.10.0
  (crate-source "miette-derive" "5.10.0"
                "0p33msrngkxlp5ajm8nijamii9vcwwpy8gfh4m53qnmrc0avrrs9"))

(define rust-miette-derive-7.5.0
  (crate-source "miette-derive" "7.5.0"
                "0irig3c79184h54zasn06yiz25znqrpvx8r72byr5gj9md2byidz"))

(define rust-miette-derive-7.6.0
  (crate-source "miette-derive" "7.6.0"
                "12w13a67n2cc37nzidvv0v0vrvf4rsflzxz6slhbn3cm9rqjjnyv"))

(define rust-migrations-internals-2.2.0
  (crate-source "migrations_internals" "2.2.0"
                "1zw0lf2lw3wlmyb0kv68cnr3ya2n80svpavf0jcqfbz8a6c060gx"))

(define rust-migrations-macros-2.2.0
  (crate-source "migrations_macros" "2.2.0"
                "1z9p2ag0fnnh0m7z8qfncwyjc0pgschca7vzlixb6v0pfb663cgz"))

(define rust-mimalloc-0.1.28
  (crate-source "mimalloc" "0.1.28"
                "09ir1qlg5rwb74l8zh5nyxgwmnpwf7r5zxj26bdnnbqc74qs3pxh"))

(define rust-mimalloc-0.1.45
  (crate-source "mimalloc" "0.1.45"
                "0pwjnz8s4qmbz4qxncxqqx3i64zczwvd758ir9hh1r9z1641zjq3"))

(define rust-mimalloc-0.1.46
  (crate-source "mimalloc" "0.1.46"
                "1bzjb3jg8m3v15gshfr7146mn64mls3slgwwgqm85d5v6bs44ncr"))

(define rust-mimalloc-0.1.47
  (crate-source "mimalloc" "0.1.47"
                "0h5wyqdywhgrpbbgknv9iwazf885fvv20vzhcibsz58y22z1qydi"))

(define rust-mimalloc-0.1.48
  (crate-source "mimalloc" "0.1.48"
                "1n2gi51zb6jhr2d154sisymgxhyr58cvkjwb53pz8x2cnsj6dvp1"))

(define rust-mimalloc-rust-0.1.5
  (crate-source "mimalloc-rust" "0.1.5"
                "0wj4hkspsmlgm6lh5bf2blzalgjcc356bpkh4k5vcnyvvzwhvhxw"))

(define rust-mimalloc-rust-sys-1.7.3-source
  (crate-source "mimalloc-rust-sys" "1.7.3-source"
                "0rh6lx32yb4vnaavwzif3gzcl1c326x0sxv01172z61b4qqqgp1s"
                #:snippet
                '(begin
                   (delete-file-recursively "mimalloc")
                   ;; Inspired by Debian's patch for bzip2-sys.
                   (substitute* "Cargo.toml.orig"
                     (("cc = .*") "pkg-config = \"0.3\"\n"))
                   (copy-file "Cargo.toml.orig" "Cargo.toml")
                   (delete-file "build.rs")
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main() {~@
                        println!(\"cargo:rustc-link-lib=mimalloc\");~@
                        }~%"))))))

(define rust-mime2ext-0.1.54
  (crate-source "mime2ext" "0.1.54"
                "1z11pfc74s3sjlibqizx51l03yacw8ywsiiq4g2l5347f1hg7xnb"))

(define rust-minidom-0.12.0
  (crate-source "minidom" "0.12.0"
                "06nbqscsv2clc4mvdzzl1syn89plsqvmxn2lqxjfrxbllqar2m7y"))

(define rust-mio-0.6.23
  (crate-source "mio" "0.6.23"
                "1i2c1vl8lr45apkh8xbh9k56ihfsmqff5l7s2fya7whvp7sndzaa"))

(define rust-mio-1.0.2
  (crate-source "mio" "1.0.2"
                "1v1cnnn44awxbcfm4zlavwgkvbyg7gp5zzjm8mqf1apkrwflvq40"))

(define rust-mio-extras-2.0.6
  (crate-source "mio-extras" "2.0.6"
                "069gfhlv0wlwfx1k2sriwfws490kjp490rv2qivyfb01j3i3yh2j"))

(define rust-miow-0.2.2
  (crate-source "miow" "0.2.2"
                "0kcl8rnv0bhiarcdakik670w8fnxzlxhi1ys7152sck68510in7b"))

(define rust-mockito-1.7.0
  (crate-source "mockito" "1.7.0"
                "0j5acfmm1ki098rwn63g0swn2f69ljf00x03givybdyr33jf0q3p"
                #:snippet '(delete-file-recursively "tests/files")))

(define rust-moka-0.12.10
  (crate-source "moka" "0.12.10"
                "09h9ww66vxrkizma99n7ib2fm91crkw4msp650j2i94lr911ccm9"))

(define rust-moxcms-0.7.5
  (crate-source "moxcms" "0.7.5"
                "026df3qpxn430dlngpj3gjip0m9280g3asvbia5dpsjsjfl2zlyx"))

(define rust-multibase-0.9.1
  (crate-source "multibase" "0.9.1"
                "014l697md16829k41hzmfx4in9jzhn774q5292bsq10z7kn3jdcv"))

(define rust-mime-0.3.17
  (crate-source "mime" "0.3.17"
                "16hkibgvb9klh0w0jk5crr5xv90l3wlf77ggymzjmvl1818vnxv8"))

(define rust-mime-guess-2.0.5
  (crate-source "mime_guess" "2.0.5"
                "03jmg3yx6j39mg0kayf7w4a886dl3j15y8zs119zw01ccy74zi7p"))

(define rust-minicov-0.3.7
  (crate-source "minicov" "0.3.7"
                "0jsvi62lklfyvdmsiizipkqcfpsc7h4c4illgxlf28iwrkqyjzzj"))

(define rust-minify-html-0.16.4
  (crate-source "minify-html" "0.16.4"
                "1mdv9k2mw6way5kya4nmkr1c03m5b8d9wqipk8zal7ddg007dpzh"))

(define rust-minify-html-common-0.0.2
  (crate-source "minify-html-common" "0.0.2"
                "100p85mrv95rdrism2mbkpm9q0y25gf0kmybq08dxigxvx06nyk9"))

(define rust-minify-js-0.6.0
  (crate-source "minify-js" "0.6.0"
                "0bm9vlhn1dji43m0ck09b9ipc3i3wjmnql1y24j61mlbxr35bymi"))

(define rust-minijinja-2.10.2
  (crate-source "minijinja" "2.10.2"
                "0m9i8ch5my3skk1hd6qm6qscf5f00zv57j5yv8758x12wjsfhwnx"))

(define rust-minijinja-2.9.0
  (crate-source "minijinja" "2.9.0"
                "0m77dg3fp7xdbfpjl8rjpqgrbam40z6pfyrhk9vj44d9zinjlr4q"))

(define rust-minijinja-embed-2.10.2
  (crate-source "minijinja-embed" "2.10.2"
                "0rrcawzidx6gxinafkw803xcv89ajkirk5ah4adznp1cq8sfpq2x"))

(define rust-minimal-lexical-0.2.1
  (crate-source "minimal-lexical" "0.2.1"
                "16ppc5g84aijpri4jzv14rvcnslvlpphbszc7zzp6vfkddf4qdb8"))

(define rust-miniz-oxide-0.6.2
  (crate-source "miniz_oxide" "0.6.2"
                "1yp8z6yll5ypz1ldmgnv7zi0r78kbvmqmn2mii77jzmk5069axdj"))

(define rust-miniz-oxide-0.7.4
  (crate-source "miniz_oxide" "0.7.4"
                "024wv14aa75cvik7005s5y2nfc8zfidddbd7g55g7sjgnzfl18mq"))

(define rust-miniz-oxide-0.8.5
  (crate-source "miniz_oxide" "0.8.5"
                "1r9whkc61xri7m1cn4rjrjlhr32ab29nvfxcbg0ri5mmpgg08glf"))

(define rust-miniz-oxide-0.8.7
  (crate-source "miniz_oxide" "0.8.7"
                "0c4lj692adnzw0h9j8l24d7imds3icpgdkk3b03zlhxf90zcww7z"))

(define rust-miniz-oxide-0.8.8
  (crate-source "miniz_oxide" "0.8.8"
                "0al9iy33flfgxawj789w2c8xxwg1n2r5vv6m6p5hl2fvd2vlgriv"))

(define rust-miniz-oxide-0.8.9
  (crate-source "miniz_oxide" "0.8.9"
                "05k3pdg8bjjzayq3rf0qhpirq9k37pxnasfn4arbs17phqn6m9qz"))

(define rust-mint-0.5.9
  (crate-source "mint" "0.5.9"
                "1zw5glv8z2d99c82jy2za97hh9p6377xmf4rbwz7jynsdfxfngg5"))

(define rust-minus-5.6.1
  (crate-source "minus" "5.6.1"
                "0ck8d1wgqzyry38piab6bp3xm520skk50dx7cqsr8dra1m9d0fq9"
                #:snippet '(delete-file "demo.png")))

(define rust-mio-0.8.11
  (crate-source "mio" "0.8.11"
                "034byyl0ardml5yliy1hmvx8arkmn9rv479pid794sm07ia519m4"))

(define rust-mio-1.0.3
  (crate-source "mio" "1.0.3"
                "1gah0h4ia3avxbwym0b6bi6lr6rpysmj9zvw6zis5yq0z0xq91i8"))

(define rust-mio-1.0.4
  (crate-source "mio" "1.0.4"
                "073n3kam3nz8j8had35fd2nn7j6a33pi3y5w3kq608cari2d9gkq"))

(define rust-miow-0.6.0
  (crate-source "miow" "0.6.0"
                "0i307jyhxnhgzj148cdb9zq59rhlhr1b65g142g9z9r01d1pd7rm"))

(define rust-mockall-0.11.4
  (crate-source "mockall" "0.11.4"
                "15kww0a3wv300wkksc6zj0kz1jwk0hyly48daxs2vvpj300lk12c"))

(define rust-mockall-0.13.1
  (crate-source "mockall" "0.13.1"
                "1lir70dd9cnsjlf20gi3i51ha9n7mlrkx74bx5gfszlcdk6bz9ir"))

(define rust-mockall-derive-0.11.4
  (crate-source "mockall_derive" "0.11.4"
                "1fvc9kwjcc9ia6ng7z9z02b4qkl9dvsx9m4z51xz9i0mj1k7bki2"))

(define rust-mockall-derive-0.13.1
  (crate-source "mockall_derive" "0.13.1"
                "1608qajqrz23xbvv81alc6wm4l24as1bsqg4shdh3sggq8231ji5"))

(define rust-monostate-0.1.14
  (crate-source "monostate" "0.1.14"
                "1vpv8d9j8i7wachlcrpbwsy1rvzimpncgv8gwpil4mn7s3lipzma"))

(define rust-monostate-impl-0.1.14
  (crate-source "monostate-impl" "0.1.14"
                "1db3jrnbriivny6cahvhcc9af7w38q846mg1r4r4y82y5l4s80n4"))

(define rust-more-asserts-0.2.2
  (crate-source "more-asserts" "0.2.2"
                "12b3fypg2sr4qmxy6wiyx6k9sdg573f5ij98cdmbrg00whnyqhvq"))

(define rust-mozim-0.2.5
  (crate-source "mozim" "0.2.5"
                "1bg8xkvcpxr06a1kp1dp78gjkgsgq5a9i1qzcf43az007l8k83k1"
                #:snippet '(delete-file-recursively "doc")))

(define rust-mptcp-pm-0.1.3
  (crate-source "mptcp-pm" "0.1.3"
                "1ma2r9ffp2164s1msfjw78553xccy4i9lvrkbsvhgr6wcgyaibry"))

(define rust-mukti-metadata-0.3.0
  (crate-source "mukti-metadata" "0.3.0"
                "1s71fs2v0phf4hhgac4j6cc38hpq968l1cfbyw7phf8fipg6dyi6"))

(define rust-muldiv-0.2.1
  (crate-source "muldiv" "0.2.1"
                "014jlry2l2ph56mp8knw65637hh49q7fmrraim2bx9vz0a638684"))

(define rust-muldiv-1.0.1
  (crate-source "muldiv" "1.0.1"
                "1c6ljsp41n8ijsx7zicwfm135drgyhcms12668ivvsbm1r98frwm"))

(define rust-multi-stash-0.2.0
  (crate-source "multi-stash" "0.2.0"
                "03s12sf633n02mcqcv2yxdx545lwc127hsic3n774khznv29lnk8"))

(define rust-multimap-0.10.0
  (crate-source "multimap" "0.10.0"
                "00vs2frqdhrr8iqx4y3fbq73ax5l12837fvbjrpi729d85alrz6y"))

(define rust-multimap-0.10.1
  (crate-source "multimap" "0.10.1"
                "1150lf0hjfjj4ksb8s3y0hl7a2nqzqlbh0is7vdym2iyjfrfr1qx"))

(define rust-multimap-0.8.3
  (crate-source "multimap" "0.8.3"
                "0sicyz4n500vdhgcxn4g8jz97cp1ijir1rnbgph3pmx9ckz4dkp5"))

(define rust-multimap-0.9.1
  (crate-source "multimap" "0.9.1"
                "04fmnxhpawndrr5x730s3gb4h77ldbrnlww86a8vsb9mkf5x79g1"))

(define rust-multipart-0.18.0
  (crate-source "multipart" "0.18.0"
                "10libwfbazqcyxcpgpcdf1a66jnzghwlmxlxnffg4rrqhqrwdph0"))

(define rust-multipart-rs-0.1.13
  (crate-source "multipart-rs" "0.1.13"
                "1wj5jgbrq7svcqdahxi17j8vws8nsz6a5y9f6ir51ajjgq7f1jk4"))

(define rust-munge-0.4.3
  (crate-source "munge" "0.4.3"
                "1myrsmqpwk3vlbvbs6jjni1iqsybri8dz7r7djj6vw4cr41142d0"))

(define rust-munge-macro-0.4.3
  (crate-source "munge_macro" "0.4.3"
                "0n94vxw2fygwqsqh4kvxwa8xsh09hli1mikhz6r215s7j77rjivk"))

(define rust-mutate-once-0.1.1
  (crate-source "mutate_once" "0.1.1"
                "0ys9mpjhwsj5md10ykmkin0wv7bz8dvc292hqczs9l5l4cd6ikqn"))

(define rust-nalgebra-0.19.0
  (crate-source "nalgebra" "0.19.0"
                "0i87k57nav221lnr9z7ljlwxh8073qsx33bajdm146y00q805fqa"
                #:snippet '(delete-file-recursively "tests")))

(define rust-nalgebra-0.27.1
  (crate-source "nalgebra" "0.27.1"
                "0811vksy1ls1m41gqkh1i5cy3h3g3b615kwxd8gjwkrg03jgybs6"
                #:snippet '(delete-file-recursively "tests")))

(define rust-nalgebra-0.29.0
  (crate-source "nalgebra" "0.29.0"
                "1zri11vkrbk3ayvy8xqcdvvyjvfcbl5a18x8z82racnn11zfn1nm"))

(define rust-nalgebra-0.32.6
  (crate-source "nalgebra" "0.32.6"
                "1r033ciacblmkif5njlhprfp0k59spjv54cqsyggb1is0bg1fp3v"))

(define rust-nalgebra-macros-0.1.0
  (crate-source "nalgebra-macros" "0.1.0"
                "063jvvvlwmzzxfr4wyiil2cn1yqj3arvghwsr2nk4ilv2jwc1z01"))

(define rust-nalgebra-macros-0.2.2
  (crate-source "nalgebra-macros" "0.2.2"
                "1z6v9phhr1hwzyyblf792128lxfv1hy1sxl4cvikihcgmxr56ji5"))

(define rust-named-pipe-0.4.1
  (crate-source "named_pipe" "0.4.1"
                "0azby10wzmsrf66m1bysbil0sjfybnvhsa8py093xz4irqy4975d"))

(define rust-nanoid-0.4.0
  (crate-source "nanoid" "0.4.0"
                "1n5l5xig5zcinh41bb4p0rzam8gxic02qpngnylb3d8pq3g01yiz"))

(define rust-nanorand-0.7.0
  (crate-source "nanorand" "0.7.0"
                "1hr60b8zlfy7mxjcwx2wfmhpkx7vfr3v9x12shmv1c10b0y32lba"))

(define rust-napi-2.12.3
  (crate-source "napi" "2.12.3"
                "17ylj8c9s498wm9r44mxv68ma6kinixscp3rw6hla2j1bf31mc0x"))

(define rust-napi-build-1.2.1
  (crate-source "napi-build" "1.2.1"
                "0asjspv0gkg6lgdz3d6gl9ab7hbc9z2hc51m637j6x3jfa8l3m7b"
                #:snippet '(delete-file-recursively "src/libs")))

(define rust-napi-derive-2.14.3
  (crate-source "napi-derive" "2.14.3"
                "1bz2ibcpxaqrsmbdsmzs5dz86nj83pq7g0qvhlh9rwm3yr1hf31b"))

(define rust-napi-derive-backend-1.0.56
  (crate-source "napi-derive-backend" "1.0.56"
                "1cqv3fjcx8dcwxga601a6lvswby6bdmc6f7yvxbspz8m0vcf8sa8"))

(define rust-napi-sys-2.4.0
  (crate-source "napi-sys" "2.4.0"
                "18sfjqbvf2lj602rbhavvks1zkhhlaa5a0y1zqql6wrsxkl04y22"))

(define rust-nasm-rs-0.2.5
  (crate-source "nasm-rs" "0.2.5"
                "0lfs2xfbpl1j7zq6qfg2wmi4djbl36qsygjb2spisjsz0v89hkgy"))

(define rust-native-tls-0.2.12
  (crate-source "native-tls" "0.2.12"
                "0rkl65z70n7sy4d5w0qa99klg1hr43wx6kcprk4d2n9xr2r4wqd8"))

(define rust-native-tls-0.2.14
  (crate-source "native-tls" "0.2.14"
                "03hga800x8bzkp8h7frnm7yp545dwwawgmaq673vx7byk1139pl7"))

(define rust-native-tls-0.2.4
  (crate-source "native-tls" "0.2.4"
                "0bcbqm8gpm5azlp8rwznhrfk3ngikr041fj883lmw3g9dz08h39b"
                #:snippet '(delete-file-recursively "test")))

(define rust-native-windows-gui-1.0.13
  (crate-source "native-windows-gui" "1.0.13"
                "0m44lbslzvs04i4rgcphld23qlwf9zzlzmspgimyp3gnd6k06w2g"
                #:snippet '(delete-file-recursively "test_rc")))

(define rust-natord-1.0.9
  (crate-source "natord" "1.0.9"
                "0z75spwag3ch20841pvfwhh3892i2z2sli4pzp1jgizbipdrd39h"))

(define rust-natord-plus-plus-2.0.0
  (crate-source "natord-plus-plus" "2.0.0"
                "0r6sqgf0gqdldzh68dsvjqwnay3dcw9m359gxkj16bbqg45ary36"))

(define rust-nb-connect-1.2.0
  (crate-source "nb-connect" "1.2.0"
                "05fadcyxaz0r8pbh498y24xay3b2gb73ih3fj7hwylggqq6m9fxi"))

(define rust-ndarray-0.14.0
  (crate-source "ndarray" "0.14.0"
                "011wqzmrd9gpfcfvy1xfbskqfiahn96pmi2d0r9x34d682amq3bc"))

(define rust-ndarray-0.15.6
  (crate-source "ndarray" "0.15.6"
                "0cpsm28hyk8qfjs4g9649dprv3hm53z12qqwyyjqbi3yjr72vcdd"))

(define rust-ndarray-0.16.1
  (crate-source "ndarray" "0.16.1"
                "0ha8sg5ad501pgkxw0wczh8myc2ma3gyxgcny4mq8rckrqnxfbl8"))

(define rust-ndarray-npy-0.9.1
  (crate-source "ndarray-npy" "0.9.1"
                "1abqnbfn9xhq5q5nli78yip3wh2zq4gi7dnrm50r3i38qj43fcbv"
                #:snippet '(delete-file-recursively "resources")))

(define rust-ndarray-rand-0.15.0
  (crate-source "ndarray-rand" "0.15.0"
                "1n15dm0x9g6s69vwqgsfj2pdw5wlha6bv9pfrn6p356idzdv74zh"))

(define rust-ndk-0.8.0
  (crate-source "ndk" "0.8.0"
                "1dx5yyqh32bi161mipg4br4i33syjidw81qrq0w7mc8hf0ds6xi0"))

(define rust-ndk-0.9.0
  (crate-source "ndk" "0.9.0"
                "1m32zpmi5w1pf3j47k6k5fw395dc7aj8d0mdpsv53lqkprxjxx63"))

(define rust-ndk-context-0.1.1
  (crate-source "ndk-context" "0.1.1"
                "12sai3dqsblsvfd1l1zab0z6xsnlha3xsfl7kagdnmj3an3jvc17"))

(define rust-ndk-sys-0.5.0+25.2.9519653
  (crate-source "ndk-sys" "0.5.0+25.2.9519653"
                "14bnxww0f17xl8pyn6j5kpkl98snjl9lin8i7qv4zzb0vmlnf6cc"))

(define rust-ndk-sys-0.6.0+11769913
  (crate-source "ndk-sys" "0.6.0+11769913"
                "0wx8r6pji20if4xs04g73gxl98nmjrfc73z0v6w1ypv6a4qdlv7f"))

(define rust-neli-0.6.5
  (crate-source "neli" "0.6.5"
                "1agcbm9vz1vsf9m7iv016s6y2k0qi3y1sc2zldz538kdrq6jl1lk"))

(define rust-neli-proc-macros-0.1.4
  (crate-source "neli-proc-macros" "0.1.4"
                "1zk17wvdzd73is6r6j84qwsgrjcdvxp9xhcn59dlbydnzfvk900c"))

(define rust-neli-wifi-0.6.0
  (crate-source "neli-wifi" "0.6.0"
                "0w5wkg8vzdzmsy8g1g5f65fxdnxxb9nbvfi47m9fjqnsrwr2584y"))

(define rust-nested-0.1.1
  (crate-source "nested" "0.1.1"
                "17lwhdw0z8c4g00yfdasxh4zc5dq1ccylmbb0n1zw1wgcc7l4aya"))

(define rust-net2-0.2.39
  (crate-source "net2" "0.2.39"
                "1b1lxvs192xsvqnszcz7dn4dw3fsvzxnc23qvq39scx26s068fxi"))

(define rust-netlink-packet-core-0.7.0
  (crate-source "netlink-packet-core" "0.7.0"
                "197dh9c5570135kv5q770n2ih5prhsql58cd71xxcya4f2plywkj"))

(define rust-netlink-packet-generic-0.3.3
  (crate-source "netlink-packet-generic" "0.3.3"
                "12rhb95ayx63zb55mz8z5vc5lghk92s8bxmpiimlrj1isf5fpmqw"))

(define rust-netlink-packet-route-0.18.1
  (crate-source "netlink-packet-route" "0.18.1"
                "1i9z89dy3p2jbs5rclvc6yxly81a4qwynrhyqv3hazj5pgxcv76x"))

(define rust-netlink-packet-route-0.19.0
  (crate-source "netlink-packet-route" "0.19.0"
                "1x4bjc97nq6ckvn25l8qysybf324jbinqx6s11vqrvmlfz6p3hbl"))

(define rust-netlink-packet-route-0.21.0
  (crate-source "netlink-packet-route" "0.21.0"
                "0vh6qc3c87ihbwb7s1z7ik1i5f2fa1yhk3w52j96jrggpza2acs8"))

(define rust-netlink-packet-route-0.22.0
  (crate-source "netlink-packet-route" "0.22.0"
                "1r6qvhippjj1xhdgpbxva03ybhyz61f9l7ywapsss545na3pj3pw"))

(define rust-netlink-packet-utils-0.5.2
  (crate-source "netlink-packet-utils" "0.5.2"
                "0d3xfh9zg0x2hbmh8iws0dvhj69prpx54khfvmfakm8sqw48mphf"))

(define rust-netlink-proto-0.11.5
  (crate-source "netlink-proto" "0.11.5"
                "0q0xqhp0n8307zakbfsa4w2n5dfrw80ym7nq20j6339g5h0jwibj"))

(define rust-netlink-sys-0.8.7
  (crate-source "netlink-sys" "0.8.7"
                "08xxx1ng7fxkvqrh81y95hni5mnir1ksg5xzj4vbj32rf2m07j8n"))

(define rust-netservices-0.8.1
  (crate-source "netservices" "0.8.1"
                "1nm0lkylmn44ny1lbmz6lygyv23d02k74y6qjpyssmi1ykryx0ym"))

(define rust-nettle-7.4.0
  (crate-source "nettle" "7.4.0"
                "0dk9rlpz4c0kf2c7298vllpnwr3lh10kkgdbslglmlz5ji5gzrj4"))

(define rust-nettle-sys-2.3.1
  (crate-source "nettle-sys" "2.3.1"
                "0d4c91v0i4rz5q1rcr3v0kscljcsbhy9s88skgai1lv4c10gb8v1"))

(define rust-new-debug-unreachable-1.0.6
  (crate-source "new_debug_unreachable" "1.0.6"
                "11phpf1mjxq6khk91yzcbd3ympm78m3ivl7xg6lg2c0lf66fy3k5"))

(define rust-newline-converter-0.3.0
  (crate-source "newline-converter" "0.3.0"
                "0zyw2hyjl89rj1zmp9n8fq69pbfp9zl1cbal73agxjxixjbv1dj7"))

(define rust-newtype-derive-0.1.6
  (crate-source "newtype_derive" "0.1.6"
                "1v3170xscs65gjx5vl1zjnqp86wngbzw3n2q74ibfnqqkx6x535c"))

(define rust-newtype-uuid-1.2.2
  (crate-source "newtype-uuid" "1.2.2"
                "10f0ylm2pz3v44wipjb5pxliq3zm33cwy4qmzvhxx3wgg8y31fm8"))

(define rust-nextest-filtering-0.15.0
  (crate-source "nextest-filtering" "0.15.0"
                "1l3l5g57i1dw8n3819r933bvfvz70appjddbmkha2pizh9qzxj4m"))

(define rust-nextest-metadata-0.12.2
  (crate-source "nextest-metadata" "0.12.2"
                "1lshdgsdwc438hi08lcq8xcqnqzsn3nx1vnxv237p4d0jy5nk9nr"))

(define rust-nextest-runner-0.80.0
  (crate-source "nextest-runner" "0.80.0"
                "1spr8rxwmxmcv9a648qwzidn6byara8pvc5b5c47phhwqw7bj2vl"))

(define rust-nextest-workspace-hack-0.1.0
  (crate-source "nextest-workspace-hack" "0.1.0"
                "1cxjiwja0idhd8as3drl2wgk5y7f84k2rrk67pbxk7kkk1m881nr"))

(define rust-nftables-0.5.0
  (crate-source "nftables" "0.5.0"
                "0mz6i4cm5qbc11zdrzghixvml2j9vxdf97nqjbkqc471kvg8hwkc"))

(define rust-nias-0.5.0
  (crate-source "nias" "0.5.0"
                "1w0jrshjqr1dxd01kg12f1ic067hvwwqc9jxbf0m063gr11089db"))

(define rust-nibble-vec-0.1.0
  (crate-source "nibble_vec" "0.1.0"
                "0hsdp3s724s30hkqz74ky6sqnadhp2xwcj1n1hzy4vzkz4yxi9bp"))

(define rust-nispor-1.2.24
  (crate-source "nispor" "1.2.24"
                "06icslv9x24b4xkl5i3i3x2g3wcrmkycsdkzvj9yd3wpmaa88kyy"))

(define rust-nitrokey-0.9.0
  (crate-source "nitrokey" "0.9.0"
                "0c8mj52ckvy1311vwxs6jpw16b2bihp5cc811isb96j9slcjvsyx"))

(define rust-nitrokey-sys-3.7.0
  (crate-source "nitrokey-sys" "3.7.0"
                "0hcns32sclzf8xbdjg5iqndbn1b66l3j61zbgbl8ds8n6nind16q"
                #:snippet '(delete-file-recursively "libnitrokey-v3.7")))

(define rust-nitrokey-test-0.5.0
  (crate-source "nitrokey-test" "0.5.0"
                "197j9r2s4ydzbqfydza6v31mgcsgd29jpidz4psqawjdm49f8lg6"))

(define rust-nitrokey-test-state-0.1.0
  (crate-source "nitrokey-test-state" "0.1.0"
                "00w817kl8q3sna924pzl72ybqljny14rdv1ixlj288fmsqp776x5"))

(define rust-nix-0.19.1
  (crate-source "nix" "0.19.1"
                "1wk1pmaf9pv84sc4jf19gm1as2yq3ydwcx0n5nc1bpsgzq6bmk5j"))

(define rust-nix-0.22.3
  (crate-source "nix" "0.22.3"
                "1bsgc8vjq07a1wg9vz819bva3dvn58an4r87h80dxrfqkqanz4g4"
                #:snippet '(delete-file-recursively "test")))

(define rust-nix-0.23.2
  (crate-source "nix" "0.23.2"
                "0p5kxhm5d8lry0szqbsllpcb5i3z7lg1dkglw0ni2l011b090dwg"
                #:snippet '(delete-file-recursively "test")))

(define rust-nix-0.24.3
  (crate-source "nix" "0.24.3"
                "0sc0yzdl51b49bqd9l9cmimp1sw1hxb8iyv4d35ww6d7m5rfjlps"
                #:snippet '(delete-file-recursively "test")))

(define rust-nix-0.25.1
  (crate-source "nix" "0.25.1"
                "1r4vyp5g1lxzpig31bkrhxdf2bggb4nvk405x5gngzfvwxqgyipk"
                #:snippet '(delete-file-recursively "test")))

(define rust-nix-0.26.4
  (crate-source "nix" "0.26.4"
                "06xgl4ybb8pvjrbmc3xggbgk3kbs1j0c4c0nzdfrmpbgrkrym2sr"
                #:snippet '(delete-file-recursively "test")))

(define rust-nix-0.27.1
  (crate-source "nix" "0.27.1"
                "0ly0kkmij5f0sqz35lx9czlbk6zpihb7yh1bsy4irzwfd2f4xc1f"
                #:snippet '(delete-file-recursively "test")))

(define rust-nix-0.28.0
  (crate-source "nix" "0.28.0"
                "1r0rylax4ycx3iqakwjvaa178jrrwiiwghcw95ndzy72zk25c8db"
                #:snippet '(delete-file-recursively "test")))

(define rust-nix-0.29.0
  (crate-source "nix" "0.29.0"
                "0ikvn7s9r2lrfdm3mx1h7nbfjvcc6s9vxdzw7j5xfkd2qdnp9qki"
                #:snippet '(delete-file-recursively "test")))

(define rust-nix-0.30.1
  (crate-source "nix" "0.30.1"
                "1dixahq9hk191g0c2ydc0h1ppxj0xw536y6rl63vlnp06lx3ylkl"))

(define rust-no-color-0.1.0
  (crate-source "no_color" "0.1.0"
                "1m0x6i5q1jav2jgajkg4irrkfgv4lakpdvmj310b90wswdyy9xdx"))

(define rust-no-panic-0.1.26
  (crate-source "no-panic" "0.1.26"
                "0sp4d9815f8vdz31qm10wpnswi0x89hlbzw78nxy0p2g88kd39ki"))

(define rust-no-std-compat-0.4.1
  (crate-source "no-std-compat" "0.4.1"
                "132vrf710zsdp40yp1z3kgc2ss8pi0z4gmihsz3y7hl4dpd56f5r"))

(define rust-no-std-net-0.6.0
  (crate-source "no-std-net" "0.6.0"
                "0ravflgyh0q2142gjdz9iav5yqci3ga7gbnk4mmfcnqkrq54lya3"))

(define rust-nodrop-0.1.14
  (crate-source "nodrop" "0.1.14"
                "1fz1v9r8ijacf0hlq0pdv5l9mz8vgqg1snmhvpjmi9aci1b4mvvj"))

(define rust-noise-framework-0.4.0
  (crate-source "noise-framework" "0.4.0"
                "1bhvnpcbgza5hc6gf97n2czac2494fr5n3jxfmcdr6fm2gkrczmm"))

(define rust-nom-4.2.3
  (crate-source "nom" "4.2.3"
                "1mkvby8b4m61p4g1px0pwr58yfkphyp1jcfbp4qfp7l6iqdaklia"))

(define rust-nom-5.1.3
  (crate-source "nom" "5.1.3"
                "0jyxc4d3pih60pp8hvzpg5ajh16s273cpnsdpzp04qv7g8w9m588"))

(define rust-nom-7.1.3
  (crate-source "nom" "7.1.3"
                "0jha9901wxam390jcf5pfa0qqfrgh8li787jx2ip0yk5b8y9hwyj"))

(define rust-nom-8.0.0
  (crate-source "nom" "8.0.0"
                "01cl5xng9d0gxf26h39m0l8lprgpa00fcc75ps1yzgbib1vn35yz"))

(define rust-nom-bibtex-0.5.0
  (crate-source "nom-bibtex" "0.5.0"
                "0dyy0ifxpl2y0wg18pddm6hanmxvxq2vmpdsbk9inw0nbvv9nqvp"))

(define rust-nom-derive-0.10.1
  (crate-source "nom-derive" "0.10.1"
                "19vrbhkn47zw5sks17ril7aryzx0ixg62n0ddrxbil48igb47y8z"))

(define rust-nom-derive-impl-0.10.1
  (crate-source "nom-derive-impl" "0.10.1"
                "0qnsvvg74wdpzhbkxrndr7x6vb1kvhrd60hfwz1kw3abm29rl2yd"))

(define rust-nom-language-0.1.0
  (crate-source "nom-language" "0.1.0"
                "0abbzawam1nh75igvyn1vh5pgxgzm0wqj2y9jbpxmzhv8mdvrqid"))

(define rust-nom-locate-4.2.0
  (crate-source "nom_locate" "4.2.0"
                "1wx87c2pm84h63rb4rsjrqzgx574x1zy93av1jk3swdhag086g0y"))

(define rust-nom-regex-0.2.0
  (crate-source "nom-regex" "0.2.0"
                "060wr7yxvm0377rckph8ksn1m1jy8yi55v842sv704sc3irwgrbj"))

(define rust-nom-tracable-0.9.1
  (crate-source "nom-tracable" "0.9.1"
                "1552z4bvp16r2z9r6cpb6rajgh4590g6pga0lmn83jav9vnd6fba"))

(define rust-nom-tracable-macros-0.9.1
  (crate-source "nom-tracable-macros" "0.9.1"
                "0jnzbfg6nyghi5g5irkihjjy36pirbv87q1w7ccswk952r9qzin9"))

(define rust-nonempty-0.9.0
  (crate-source "nonempty" "0.9.0"
                "1rh8yv3l0531696d27x3fmv4ykbbi4xqwbmxs7ysr2d5l3ffypcr"))

(define rust-nonzero-ext-0.3.0
  (crate-source "nonzero_ext" "0.3.0"
                "08fghyinb07xwhbj7vwvlhg45g5cvhvld2min25njidir12rdgrq"))

(define rust-noop-proc-macro-0.3.0
  (crate-source "noop_proc_macro" "0.3.0"
                "1j2v1c6ric4w9v12h34jghzmngcwmn0hll1ywly4h6lcm4rbnxh6"))

(define rust-normalize-line-endings-0.3.0
  (crate-source "normalize-line-endings" "0.3.0"
                "1gp52dfn2glz26a352zra8h04351icf0fkqzw1shkwrgh1vpz031"))

(define rust-normalize-path-0.2.1
  (crate-source "normalize-path" "0.2.1"
                "0bc919zp34b9312wqd92jbi3xa9gxljxh8nfw7v6sk7znb98shzm"))

(define rust-normpath-0.2.0
  (crate-source "normpath" "0.2.0"
                "0y354v327vbcxap7ag88j81y4zifzhnjgxvn7l1s2d9cjb4si79a"))

(define rust-normpath-1.3.0
  (crate-source "normpath" "1.3.0"
                "1vfplrj3miplk0qc7b6psvf6vrmhr2whvqvlvk09lm5iqibik4f8"))

(define rust-normpath-1.4.0
  (crate-source "normpath" "1.4.0"
                "1i69fj97w9x3gd9vqnk6b4q0wmhb8c6yacb9f8ivazgxf69kcy61"))

(define rust-normpath-1.5.0
  (crate-source "normpath" "1.5.0"
                "16z68q809749ky2vl72f3lqnhf3vjclvcc3y2z5v8m2nj0msn8xz"))

(define rust-notify-5.2.0
  (crate-source "notify" "5.2.0"
                "11l4r19z69mplx124hsvm2flf740ykykwkzs7vz46njmrbhn77vj"))

(define rust-notify-6.1.1
  (crate-source "notify" "6.1.1"
                "0bad98r0ilkhhq2jg3zs11zcqasgbvxia8224wpasm74n65vs1b2"))

(define rust-notify-7.0.0
  (crate-source "notify" "7.0.0"
                "02a0a1n0raxqslwhfprwmz7w34v54r42006q0m8bmy89jz1v8cy5"))

(define rust-notify-8.0.0
  (crate-source "notify" "8.0.0"
                "0hz9ab68gsbkidms6kgl4v7capfqjyl40vpfdarcfsnnnc1q9vig"))

(define rust-notify-8.1.0
  (crate-source "notify" "8.1.0"
                "15wcrmvk1rz0jfahjrl36zblz6drlxk2j91gqgw9w3pssffgaqri"))

(define rust-notify-8.2.0
  (crate-source "notify" "8.2.0"
                "1hrb83451vm5cpjw83nz5skgwjg5ara28zq8nxsqbzsif690fgad"))

(define rust-notify-debouncer-full-0.3.2
  (crate-source "notify-debouncer-full" "0.3.2"
                "0kpkz1y4v2kwh5rgkhx4h2fxnwqka30lsrcy2vzwk2cpfdkd2zzv"))

(define rust-notify-debouncer-full-0.5.0
  (crate-source "notify-debouncer-full" "0.5.0"
                "1ldqk50zzfayq7l4adzv2c2mj6lkgkgki0r5r18l619qfld8pn6j"))

(define rust-notify-debouncer-mini-0.6.0
  (crate-source "notify-debouncer-mini" "0.6.0"
                "1f6cdmqxfmzcxwjfs0xbh9k73sl37387q27r4wbrlk8qc91fp2d6"))

(define rust-notify-rust-4.11.7
  (crate-source "notify-rust" "4.11.7"
                "0024xqbn29z1k6cfbi8w7c1p73hscqwkpbwlwwa2bam5cn328hk4"))

(define rust-notify-types-1.0.1
  (crate-source "notify-types" "1.0.1"
                "0x5idrpxzf70ng88rz28dqmgx1jyddf0vxx1x3csw09fw6skqpaq"))

(define rust-notify-types-2.0.0
  (crate-source "notify-types" "2.0.0"
                "0pcjm3wnvb7pvzw6mn89csv64ip0xhx857kr8jic5vddi6ljc22y"))

(define rust-notmuch-0.8.0
  (crate-source "notmuch" "0.8.0"
                "0i6xc7lv10m2sq6vlpjr5wxmlxihvd0v4f5if75r2kwz8ji12pg2"))

(define rust-nt-time-0.11.1
  (crate-source "nt-time" "0.11.1"
                "0m17ghcqjrlf63cgv4hpxmiywkqjs9yj5cdc9cs63cc9q0413rzm"))

(define rust-ntapi-0.3.7
  (crate-source "ntapi" "0.3.7"
                "03wqq2x5jv5xrsirbxmm460gcfmpakjpq8yqmc5lzfrgznkp91y2"))

(define rust-ntapi-0.4.1
  (crate-source "ntapi" "0.4.1"
                "1r38zhbwdvkis2mzs6671cm1p6djgsl49i7bwxzrvhwicdf8k8z8"))

(define rust-ntest-0.7.5
  (crate-source "ntest" "0.7.5"
                "0i4xsvx52hmcnga2xbjl74hdylz4jy8bc2swcichlvw1di4lwm2w"))

(define rust-ntest-0.9.3
  (crate-source "ntest" "0.9.3"
                "0c03bhsbksx89cvk3maji8mzayvjvdz7pvp5fbv3gad73l53y67v"))

(define rust-ntest-proc-macro-helper-0.7.5
  (crate-source "ntest_proc_macro_helper" "0.7.5"
                "0lkyfx97aynbm7cnhzyc9cr0rpq1xzng1hwmzizbf1a6855y6llg"))

(define rust-ntest-test-cases-0.7.5
  (crate-source "ntest_test_cases" "0.7.5"
                "1ghal2rb03cnj7ciqgdq0dvifdf8qp2hnmi9z1ip1j5b02s1xa4r"))

(define rust-ntest-test-cases-0.9.3
  (crate-source "ntest_test_cases" "0.9.3"
                "14xa1qalxn5lyl0x5ykf2l9sli0xgxp9kgpbd19jwnc8lkrd7l0n"))

(define rust-ntest-timeout-0.7.5
  (crate-source "ntest_timeout" "0.7.5"
                "08v3r6hggh43qabl887pkz88k6lg6hrc62mppxyabb0pw44v03di"))

(define rust-ntest-timeout-0.9.3
  (crate-source "ntest_timeout" "0.9.3"
                "1db3dzgcj2xyr4vcji90mwh8937nvj0mwbrk9adzg5qc34pwkizw"))

(define rust-nu-ansi-term-0.46.0
  (crate-source "nu-ansi-term" "0.46.0"
                "115sywxh53p190lyw97alm14nc004qj5jm5lvdj608z84rbida3p"))

(define rust-nu-ansi-term-0.49.0
  (crate-source "nu-ansi-term" "0.49.0"
                "0s2svfnircd9jp06wk55qcbb9v5cadkfcjfg99vm21qdjg0x6wy0"))

(define rust-nu-ansi-term-0.50.1
  (crate-source "nu-ansi-term" "0.50.1"
                "16a3isvbxx8pa3lk71h3cq2fsx2d17zzq42j4mhpxy81gl2qx8nl"))

(define rust-nu-cli-0.103.0
  (crate-source "nu-cli" "0.103.0"
                "0w7fqf138sz6rk5krsi3w5qm6z05ii2c8khh9gayiqfp01vv1ggz"))

(define rust-nu-cmd-base-0.103.0
  (crate-source "nu-cmd-base" "0.103.0"
                "1ln1hr86kzy0d4r16r5acg1yhrn126yz0variivq6cxzwzgzmqxm"))

(define rust-nu-cmd-extra-0.103.0
  (crate-source "nu-cmd-extra" "0.103.0"
                "1sdlyagcsdgs832l3d6xrlhnfhl6j24vzi2zh9i3lip45ssacsz2"))

(define rust-nu-cmd-lang-0.103.0
  (crate-source "nu-cmd-lang" "0.103.0"
                "0382aqmzyr3rx8qhvfzrkcvj6yz5v277q7928si49jc9y7ir7slc"))

(define rust-nu-cmd-plugin-0.103.0
  (crate-source "nu-cmd-plugin" "0.103.0"
                "1jhjqnz7yvm8idhl4hj3yb923rvs3rn2587k59ky9sb5xyl9nq31"))

(define rust-nu-color-config-0.103.0
  (crate-source "nu-color-config" "0.103.0"
                "08k3vjg8wylv0bhznmagswshqgwfbba0iqxw78wwfyy33s8khdmx"))

(define rust-nu-command-0.103.0
  (crate-source "nu-command" "0.103.0"
                "1v6ph836j86v2vibhrg2mjq4nfqm4mbxrvgg6sghn4zs63y18mcy"))

(define rust-nu-derive-value-0.103.0
  (crate-source "nu-derive-value" "0.103.0"
                "098dr378j106pyvcyp654w43lm1igsi62yl2k8m5b4k86sc527wg"))

(define rust-nu-engine-0.103.0
  (crate-source "nu-engine" "0.103.0"
                "0mma3n2abhqylfxa1qhi6a0ck43i9zq11dy546fjbr0q9jxibdqc"))

(define rust-nu-explore-0.103.0
  (crate-source "nu-explore" "0.103.0"
                "03hw24njj4h9zmc63d7wbg6bvnrc5g4lasssvq5kpa79p2av60wx"))

(define rust-nu-glob-0.103.0
  (crate-source "nu-glob" "0.103.0"
                "016p18zd8fd9169j7j7i0m55zrdv4hlgcqf5xqwm9mryb5vaakwh"))

(define rust-nu-json-0.103.0
  (crate-source "nu-json" "0.103.0"
                "1affhp2fxa51kyb6v9kyvp7sijhl40dfk8ynlvkp6fbbnj2li8qd"))

(define rust-nu-lsp-0.103.0
  (crate-source "nu-lsp" "0.103.0"
                "0b42ihc2mcpfv4hmkh0hghr089awx50b5zzwm1a2c7ikykilz9c0"))

(define rust-nu-parser-0.103.0
  (crate-source "nu-parser" "0.103.0"
                "148zq4wm7a28sqp362af0g7m92m7qijx0zn6rd5m7li3q5v6vb6s"))

(define rust-nu-path-0.103.0
  (crate-source "nu-path" "0.103.0"
                "1ymj4n4fg46alnyymi8k3zvf3pn7zjj1zx5lzfc1ybg44rgsbqz6"))

(define rust-nu-plugin-0.103.0
  (crate-source "nu-plugin" "0.103.0"
                "0gf0wjqb26ngd08x1m8p2b8zlpxs06sla1z5zpnl2x071hlz4daz"))

(define rust-nu-plugin-core-0.103.0
  (crate-source "nu-plugin-core" "0.103.0"
                "1zj52rnazk49w0nwd9ip28xgmf1hynh6b3kfq999ijpnqz8z39jv"))

(define rust-nu-plugin-engine-0.103.0
  (crate-source "nu-plugin-engine" "0.103.0"
                "17vwr08s2n4ssdv1bjs57b7ljrlgc1jp1gcp7hak8i2ycvzrrizd"))

(define rust-nu-plugin-protocol-0.103.0
  (crate-source "nu-plugin-protocol" "0.103.0"
                "11qznn3c0ia5gd99jv3ds3wfiykaw6kyvcwz6hdmc2crlmjz3gl8"))

(define rust-nu-plugin-test-support-0.103.0
  (crate-source "nu-plugin-test-support" "0.103.0"
                "0wx7sih6sdl50q2vsky9cmx91v41vp6xmlx971rv3h9zzgpi7fci"))

(define rust-nu-pretty-hex-0.100.0
  (crate-source "nu-pretty-hex" "0.100.0"
                "1alqadx31r5gf914blrjaw93iqisnjx0p94z08mc7qbnpdjdncz7"))

(define rust-nu-pretty-hex-0.103.0
  (crate-source "nu-pretty-hex" "0.103.0"
                "1mp9q6j2gx7d3r19rkgh5vmni21kwcqvb0xkskxvz0dqyi40a2m1"))

(define rust-nu-protocol-0.103.0
  (crate-source "nu-protocol" "0.103.0"
                "1dvabyfqdqlpi2ixir85nahbg7y95zv768yljh4qw7hp1n3badfa"))

(define rust-nu-std-0.103.0
  (crate-source "nu-std" "0.103.0"
                "1h12bca1s7srswazfi6i82zzfphbp4ihnkrk4bdzqjl2jfn3amj6"))

(define rust-nu-system-0.103.0
  (crate-source "nu-system" "0.103.0"
                "09rvsapdiyv61h3sk2a6ybi3w6j90cg9x1xl6syp9lmcb4f9pfvh"))

(define rust-nu-table-0.103.0
  (crate-source "nu-table" "0.103.0"
                "18v1ax4ynm1jl6xk95agrdqs0i9jxx5qvdwhfnbgpx5wn5w5mcy9"))

(define rust-nu-term-grid-0.103.0
  (crate-source "nu-term-grid" "0.103.0"
                "133fnxqj7icwvqiblkyf46548wjb3014jq72wmdschwkjmgffszi"))

(define rust-nu-test-support-0.103.0
  (crate-source "nu-test-support" "0.103.0"
                "1hdl24c37qxk7b1563g9w05bpxriwnjbk9fym8pcr2gvf9g3xgcm"))

(define rust-nu-utils-0.103.0
  (crate-source "nu-utils" "0.103.0"
                "14s74lc376hw38f97chlnxn306vfadn2hl020abm7xwl7id3809g"))

(define rust-nucleo-0.5.0
  (crate-source "nucleo" "0.5.0"
                "1m1rq0cp02hk31z7jsn2inqcpy9a1j8gfvxcqm32c74jji6ayqjj"))

(define rust-nucleo-matcher-0.3.1
  (crate-source "nucleo-matcher" "0.3.1"
                "11dc5kfin1n561qdcg0x9aflvw876a8vldmqjhs5l6ixfcwgacxz"))

(define rust-num-0.2.1
  (crate-source "num" "0.2.1"
                "0dhcvhprvvx1iaaq7sxlgxw5awmj8dibni8vhizi59zyz4q60lxq"))

(define rust-num-0.4.3
  (crate-source "num" "0.4.3"
                "08yb2fc1psig7pkzaplm495yp7c30m4pykpkwmi5bxrgid705g9m"))

(define rust-num-bigint-0.2.6
  (crate-source "num-bigint" "0.2.6"
                "015k3wixdi4w698sappvy43pf8bvkw0f88xplmdgc3zfk2cpy309"))

(define rust-num-bigint-0.4.6
  (crate-source "num-bigint" "0.4.6"
                "1f903zd33i6hkjpsgwhqwi2wffnvkxbn6rv4mkgcjcqi7xr4zr55"))

(define rust-num-bigint-dig-0.8.4
  (crate-source "num-bigint-dig" "0.8.4"
                "0lb12df24wgxxbspz4gw1sf1kdqwvpdcpwq4fdlwg4gj41c1k16w"))

(define rust-num-cmp-0.1.0
  (crate-source "num-cmp" "0.1.0"
                "1alavi36shn32b3cwbmkncj1wal3y3cwzkm21bxy5yil5hp5ncv3"))

(define rust-num-complex-0.2.4
  (crate-source "num-complex" "0.2.4"
                "15dwaksw729r3v14sgzc9723s3fnfixiir8jzwx7b7kim48r9cdn"))

(define rust-num-complex-0.3.1
  (crate-source "num-complex" "0.3.1"
                "1igjwm5kk2df9mxmpb260q6p40xfnkrq4smymgdqg2sm1hn66zbl"))

(define rust-num-complex-0.4.6
  (crate-source "num-complex" "0.4.6"
                "15cla16mnw12xzf5g041nxbjjm9m85hdgadd5dl5d0b30w9qmy3k"))

(define rust-num-conv-0.1.0
  (crate-source "num-conv" "0.1.0"
                "1ndiyg82q73783jq18isi71a7mjh56wxrk52rlvyx0mi5z9ibmai"))

(define rust-num-cpus-1.16.0
  (crate-source "num_cpus" "1.16.0"
                "0hra6ihpnh06dvfvz9ipscys0xfqa9ca9hzp384d5m02ssvgqqa1"))

(define rust-num-cpus-1.17.0
  (crate-source "num_cpus" "1.17.0"
                "0fxjazlng4z8cgbmsvbzv411wrg7x3hyxdq8nxixgzjswyylppwi"))

(define rust-num-derive-0.2.5
  (crate-source "num-derive" "0.2.5"
                "1wnv7776fh4i40r3zfxcxcmm0dh029skx7gp4sjknz2kqm2hpzga"))

(define rust-num-derive-0.3.3
  (crate-source "num-derive" "0.3.3"
                "0gbl94ckzqjdzy4j8b1p55mz01g6n1l9bckllqvaj0wfz7zm6sl7"))

(define rust-num-derive-0.4.2
  (crate-source "num-derive" "0.4.2"
                "00p2am9ma8jgd2v6xpsz621wc7wbn1yqi71g15gc3h67m7qmafgd"))

(define rust-num-enum-0.6.1
  (crate-source "num_enum" "0.6.1"
                "18bna04g6zq978z2b4ygz0f8pbva37id4xnpgwh8l41w1m1mn0bs"))

(define rust-num-enum-0.7.3
  (crate-source "num_enum" "0.7.3"
                "0yai0vafhy85mvhknzfqd7lm04hzaln7i5c599rhy8mj831kyqaf"))

(define rust-num-enum-0.7.4
  (crate-source "num_enum" "0.7.4"
                "0ykvfah4ddfi2dwjcksc33j34i84kb7plycxwr6dijp69kjb8wx9"))

(define rust-num-enum-derive-0.6.1
  (crate-source "num_enum_derive" "0.6.1"
                "19k57c0wg56vzzj2w77jsi8nls1b8xh8pvpzjnrgf8d9cnvpsrln"))

(define rust-num-enum-derive-0.7.3
  (crate-source "num_enum_derive" "0.7.3"
                "0mksna1jj87ydh146gn6jcqkvvs920c3dgh0p4f3xk184kpl865g"))

(define rust-num-enum-derive-0.7.4
  (crate-source "num_enum_derive" "0.7.4"
                "03gr5218x4rs52kx4srx3jn1c6vmx3drd506vl0axax88v47is3p"))

(define rust-num-format-0.4.4
  (crate-source "num-format" "0.4.4"
                "1hvjmib117jspyixfr76f900mhz5zfn71dnyqg9iywb339vxjlm6"))

(define rust-num-integer-0.1.46
  (crate-source "num-integer" "0.1.46"
                "13w5g54a9184cqlbsq80rnxw4jj4s0d8wv75jsq5r2lms8gncsbr"))

(define rust-num-iter-0.1.45
  (crate-source "num-iter" "0.1.45"
                "1gzm7vc5g9qsjjl3bqk9rz1h6raxhygbrcpbfl04swlh0i506a8l"))

(define rust-num-modular-0.6.1
  (crate-source "num-modular" "0.6.1"
                "0zv4miws3q1i93a0bd9wgc4njrr5j5786kr99hzxi9vgycdjdfqp"))

(define rust-num-order-1.2.0
  (crate-source "num-order" "1.2.0"
                "1dhvdncf91ljxh9sawnfxcbiqj1gnag08lyias0cy3y4jxmmjysk"))

(define rust-num-rational-0.2.4
  (crate-source "num-rational" "0.2.4"
                "1vsaz96chxcgpqd5a0dq8hb3b4sj6dnlhwmpbkf4mx6vnls0202w"))

(define rust-num-rational-0.4.2
  (crate-source "num-rational" "0.4.2"
                "093qndy02817vpgcqjnj139im3jl7vkq4h68kykdqqh577d18ggq"))

(define rust-num-threads-0.1.7
  (crate-source "num_threads" "0.1.7"
                "1ngajbmhrgyhzrlc4d5ga9ych1vrfcvfsiqz6zv0h2dpr2wrhwsw"))

(define rust-num-traits-0.1.43
  (crate-source "num-traits" "0.1.43"
                "0c9whknf2dm74a3cqirafy6gj83a76gl56g4v3g19k6lkwz13rcj"))

(define rust-num-traits-0.2.16
  (crate-source "num-traits" "0.2.16"
                "1hp6x4gayrib34y14gpcfx60hbqsmh7i8whjrbzy5rrvfayhl2zk"))

(define rust-num-traits-0.2.19
  (crate-source "num-traits" "0.2.19"
                "0h984rhdkkqd4ny9cif7y2azl3xdfb7768hb9irhpsch4q3gq787"))

(define rust-number-prefix-0.3.0
  (crate-source "number_prefix" "0.3.0"
                "0slm4mqmpgs6hvz22ycny9lvyvl9ivs80a1lncslp7lszz02zc0p"))

(define rust-number-prefix-0.4.0
  (crate-source "number_prefix" "0.4.0"
                "1wvh13wvlajqxkb1filsfzbrnq0vrmrw298v2j3sy82z1rm282w3"))

(define rust-numerals-0.1.4
  (crate-source "numerals" "0.1.4"
                "0cdx6yf5zcx2nvmzavr4qk9m35ha6i2rhy5fjxgx2wm7fq9y4nz2"))

(define rust-numpy-0.21.0
  (crate-source "numpy" "0.21.0"
                "1x1p5x7lwfc5nsccwj98sln5vx3g3n8sbgm5fmfmy5rpr8rhf5zc"))

(define rust-numpy-0.22.1
  (crate-source "numpy" "0.22.1"
                "0kpii5mvz4ag29qw4zrqzfmi3m2kmbg882kcxn2ls6m91ny2kfgd"))

(define rust-numtoa-0.1.0
  (crate-source "numtoa" "0.1.0"
                "1vs9rhggqbql1p26x8nkha1j06wawwgb2jp5fs88b5gi7prvvy5q"))

(define rust-nuon-0.103.0
  (crate-source "nuon" "0.103.0"
                "05nd4acyfc7nbl2bnvlcra60h8n28b8bqaa24j645rk9hz6ckb0r"))

(define rust-nvml-wrapper-0.10.0
  (crate-source "nvml-wrapper" "0.10.0"
                "1i9q80f8x5zy646wvz1wkaydryvnfnwsi8jy72hh92fll45gz6qc"))

(define rust-nvml-wrapper-sys-0.8.0
  (crate-source "nvml-wrapper-sys" "0.8.0"
                "1dgw2dpldxik2gn3m3aqgl2qj1da5sz6wav5g571ly18dwalb3b9"))

(define rust-oauth2-4.4.2
  (crate-source "oauth2" "4.4.2"
                "0zwkmwxwygl4fwghgyanixzqgn7yvkwwwacdghz7x124v36l3263"))

(define rust-objc-0.2.7
  (crate-source "objc" "0.2.7"
                "1cbpf6kz8a244nn1qzl3xyhmp05gsg4n313c9m3567625d3innwi"))

(define rust-objc-exception-0.1.2
  (crate-source "objc_exception" "0.1.2"
                "191cmdmlypp6piw67y4m8y5swlxf5w0ss8n1lk5xd2l1ans0z5xd"))

(define rust-objc-foundation-0.1.1
  (crate-source "objc-foundation" "0.1.1"
                "1y9bwb3m5fdq7w7i4bnds067dhm4qxv4m1mbg9y61j9nkrjipp8s"))

(define rust-objc-id-0.1.1
  (crate-source "objc_id" "0.1.1"
                "0fq71hnp2sdblaighjc82yrac3adfmqzhpr11irhvdfp9gdlsbf9"))

(define rust-objc-sys-0.3.5
  (crate-source "objc-sys" "0.3.5"
                "0423gry7s3rmz8s3pzzm1zy5mdjif75g6dbzc2lf2z0c77fipffd"))

(define rust-objc2-0.5.2
  (crate-source "objc2" "0.5.2"
                "015qa2d3vh7c1j2736h5wjrznri7x5ic35vl916c22gzxva8b9s6"))

(define rust-objc2-0.6.0
  (crate-source "objc2" "0.6.0"
                "0nbf78zvqz3wnp23iianpwbds563fiz8b6bsnxizikyrj18zcc9m"))

(define rust-objc2-0.6.1
  (crate-source "objc2" "0.6.1"
                "0l85a8r77i8i183fqyx55kqm2nh9rzg2z3z59kjb4fj92iz5kil8"))

(define rust-objc2-0.6.2
  (crate-source "objc2" "0.6.2"
                "1g3qa1vxp6nlh4wllll921z299d3s1is31m1ccasd8pklxxka7sn"))

(define rust-objc2-app-kit-0.2.2
  (crate-source "objc2-app-kit" "0.2.2"
                "1zqyi5l1bm26j1bgmac9783ah36m5kcrxlqp5carglnpwgcrms74"))

(define rust-objc2-app-kit-0.3.0
  (crate-source "objc2-app-kit" "0.3.0"
                "1yyh2j44kjhj2n55y2vlj82lzsbfpngvivv9w1x2z3hpawrgj1jr"))

(define rust-objc2-app-kit-0.3.1
  (crate-source "objc2-app-kit" "0.3.1"
                "1k4vz0s63rpp1yyhx96mh9nndn1zzv2cwxzpvw6rnigcidb9zwp6"))

(define rust-objc2-cloud-kit-0.2.2
  (crate-source "objc2-cloud-kit" "0.2.2"
                "02dhjvmcq8c2bwj31jx423jygif1scs9f0lmlab0ayhw75b3ppbl"))

(define rust-objc2-contacts-0.2.2
  (crate-source "objc2-contacts" "0.2.2"
                "12a8m927xrrxa54xhqhqnkkl1a6l07pyrpnqfk9jz09kkh755zx5"))

(define rust-objc2-core-data-0.2.2
  (crate-source "objc2-core-data" "0.2.2"
                "1vvk8zjylfjjj04dzawydmqqz5ajvdkhf22cnb07ihbiw14vyzv1"))

(define rust-objc2-core-foundation-0.3.0
  (crate-source "objc2-core-foundation" "0.3.0"
                "09frj2bc6w6dnpjfix1skq8g91kx7w788bqwiaa2c7a74l7zdsns"))

(define rust-objc2-core-foundation-0.3.1
  (crate-source "objc2-core-foundation" "0.3.1"
                "0rn19d70mwxyv74kx7aqm5in6x320vavq9v0vrm81vbg9a4w440w"))

(define rust-objc2-core-graphics-0.3.0
  (crate-source "objc2-core-graphics" "0.3.0"
                "00nynwlbppcdp7q51cmq4x04sns0lqhhla8kcmmkarcbc81adp7q"))

(define rust-objc2-core-graphics-0.3.1
  (crate-source "objc2-core-graphics" "0.3.1"
                "197mf2a4yvvigkd9hsp8abbpip7rn3mmc55psv1ba89hq5l6r74q"))

(define rust-objc2-core-image-0.2.2
  (crate-source "objc2-core-image" "0.2.2"
                "102csfb82zi2sbzliwsfd589ckz0gysf7y6434c9zj97lmihj9jm"))

(define rust-objc2-core-location-0.2.2
  (crate-source "objc2-core-location" "0.2.2"
                "10apgsrigqryvi4rcc0f6yfjflvrl83f4bi5hkr48ck89vizw300"))

(define rust-objc2-encode-4.1.0
  (crate-source "objc2-encode" "4.1.0"
                "0cqckp4cpf68mxyc2zgnazj8klv0z395nsgbafa61cjgsyyan9gg"))

(define rust-objc2-foundation-0.2.2
  (crate-source "objc2-foundation" "0.2.2"
                "1a6mi77jsig7950vmx9ydvsxaighzdiglk5d229k569pvajkirhf"))

(define rust-objc2-foundation-0.3.0
  (crate-source "objc2-foundation" "0.3.0"
                "166rz3jh3b2nrwvldllp3ihq4lb4j5pkjnyv2naw70jb074wc89s"))

(define rust-objc2-foundation-0.3.1
  (crate-source "objc2-foundation" "0.3.1"
                "0g5hl47dxzabs7wndcg6kz3q137v9hwfay1jd2da1q9gglj3224h"))

(define rust-objc2-io-surface-0.3.0
  (crate-source "objc2-io-surface" "0.3.0"
                "068wb02jcnna2qgir250nfvrsic4kz1rx7ks39p0h416wf3qn6hn"))

(define rust-objc2-io-surface-0.3.1
  (crate-source "objc2-io-surface" "0.3.1"
                "0g0c89swz8hgfrh0j1iqhcz3ig7cyhavn3p9gi2s77sjjanfk0kj"))

(define rust-objc2-link-presentation-0.2.2
  (crate-source "objc2-link-presentation" "0.2.2"
                "160k4qh00yrx57dabn3hzas4r98kmk9bc0qsy1jvwday3irax8d1"))

(define rust-objc2-metal-0.2.2
  (crate-source "objc2-metal" "0.2.2"
                "1mmdga66qpxrcfq3gxxhysfx3zg1hpx4z886liv3j0pnfq9bl36x"))

(define rust-objc2-quartz-core-0.2.2
  (crate-source "objc2-quartz-core" "0.2.2"
                "0ynw8819c36l11rim8n0yzk0fskbzrgaqayscyqi8swhzxxywaz4"))

(define rust-objc2-symbols-0.2.2
  (crate-source "objc2-symbols" "0.2.2"
                "1p04hjkxan18g2b7h9n2n8xxsvazapv2h6mfmmdk06zc7pz4ws0a"))

(define rust-objc2-ui-kit-0.2.2
  (crate-source "objc2-ui-kit" "0.2.2"
                "0vrb5r8z658l8c19bx78qks8c5hg956544yirf8npk90idwldfxq"))

(define rust-objc2-uniform-type-identifiers-0.2.2
  (crate-source "objc2-uniform-type-identifiers" "0.2.2"
                "1ziv4wkbxcaw015ypg0q49ycl7m14l3x56mpq2k1rznv92bmzyj4"))

(define rust-objc2-user-notifications-0.2.2
  (crate-source "objc2-user-notifications" "0.2.2"
                "1cscv2w3vxzaslz101ddv0z9ycrrs4ayikk4my4qd3im8bvcpkvn"))

(define rust-object-0.28.4
  (crate-source "object" "0.28.4"
                "0964501nlfh806mik3f9v6n05mx74qa0w7byvn0sqpwm5lprhb74"))

(define rust-object-0.32.2
  (crate-source "object" "0.32.2"
                "0hc4cjwyngiy6k51hlzrlsxgv5z25vv7c2cp0ky1lckfic0259m6"))

(define rust-object-0.35.0
  (crate-source "object" "0.35.0"
                "0pnv84mx3f3p847hfnsp4znivnwkc1x53maq459a92w42fw7mv5q"))

(define rust-object-0.36.4
  (crate-source "object" "0.36.4"
                "02h7k38dwi8rndc3y81n6yjxijbss99p2jm9c0b6ak5c45c1lkq8"))

(define rust-object-0.36.7
  (crate-source "object" "0.36.7"
                "11vv97djn9nc5n6w1gc6bd96d2qk2c8cg1kw5km9bsi3v4a8x532"))

(define rust-object-0.37.3
  (crate-source "object" "0.37.3"
                "1zikiy9xhk6lfx1dn2gn2pxbnfpmlkn0byd7ib1n720x0cgj0xpz"))

(define rust-object-store-0.11.2
  (crate-source "object_store" "0.11.2"
                "1bywk2bbjpvr26581rs44csl6myiw0cr6gqnw76zlmhsjrlcpz1w"))

(define rust-oboe-0.6.1
  (crate-source "oboe" "0.6.1"
                "1yv7x06mwk61nsy3ckcmqwgg9q0n3j4y4zncz3sl6pcyskmipdp8"))

(define rust-oboe-sys-0.6.1
  (crate-source "oboe-sys" "0.6.1"
                "17g7yb4kk6bakc4rhv1izfcqjgqhpkasgq6gf20nc79b9adb12vc"
                #:snippet '(for-each delete-file-recursively '("oboe" "oboe-ext"))))

(define rust-ocb3-0.1.0
  (crate-source "ocb3" "0.1.0"
                "1nyyj0rx870iv24ad8j2x55qlabald1pazkpslyq8727dhky15n1"))

(define rust-oem-cp-2.0.0
  (crate-source "oem_cp" "2.0.0"
                "1r0fn4bnmc78pkqhvzjjbm1zidgzvryspdz6hsdb1nml5a83h09k"))

(define rust-omnipath-0.1.6
  (crate-source "omnipath" "0.1.6"
                "0xd5a4xwsfmhzk59v6wz65f59rk16d7gvkg90w1qhb0jg08b7bc0"))

(define rust-once-cell-1.18.0
  (crate-source "once_cell" "1.18.0"
                "0vapcd5ambwck95wyz3ymlim35jirgnqn9a0qmi19msymv95v2yx"))

(define rust-once-cell-1.20.2
  (crate-source "once_cell" "1.20.2"
                "0xb7rw1aqr7pa4z3b00y7786gyf8awx2gca3md73afy76dzgwq8j"))

(define rust-once-cell-1.20.3
  (crate-source "once_cell" "1.20.3"
                "0bp6rgrsri1vfdcahsimk08zdiilv14ppgcnpbiw8hqyp2j64m4l"))

(define rust-once-cell-1.21.3
  (crate-source "once_cell" "1.21.3"
                "0b9x77lb9f1j6nqgf5aka4s2qj0nly176bpbrv6f9iakk5ff3xa2"))

(define rust-once-cell-polyfill-1.70.1
  (crate-source "once_cell_polyfill" "1.70.1"
                "1bg0w99srq8h4mkl68l1mza2n2f2hvrg0n8vfa3izjr5nism32d4"))

(define rust-onenote-parser-0.3.1.29c0853
  (origin
    (method git-fetch)
    (uri (git-reference (url "https://github.com/Cisco-Talos/onenote.rs.git")
                        (commit "29c08532252b917543ff268284f926f30876bb79")))
    (file-name (git-file-name "rust-onenote-parser" "0.3.1.29c0853"))
    (sha256 (base32 "1sfx3jgj1vgwh00kckl5hfbmdpp4wmvszsxwzi1k10nihryvkdy5"))
    (modules '((guix build utils)))
    (snippet '(delete-file-recursively "tests"))))

(define rust-onig-6.4.0
  (crate-source "onig" "6.4.0"
                "0kyaz2fwa5dkr04rvk5ga2yv5jkqn1ymblvpdlf1gn9afb432jwc"))

(define rust-onig-6.5.1
  (crate-source "onig" "6.5.1"
                "1w63vbzamn2v9jpnlj3wkglapqss0fcvhhd8pqafzkis8iirqsrk"))

(define rust-onig-sys-69.8.1
  (crate-source "onig_sys" "69.8.1"
                "1rw6y2qkb765gzylmrydbbd90hdzhnqyvs2y65z4riwwgqyrx0kv"
                #:snippet '(delete-file-recursively "oniguruma")))

(define rust-onig-sys-69.9.1
  (crate-source "onig_sys" "69.9.1"
                "1p17cxzqnpqzpzamh7aqwpagxlnbhzs6myxw4dgz2v9xxxp6ry67"
                #:snippet '(delete-file-recursively "oniguruma")))

(define rust-oo7-0.2.2
  (crate-source "oo7" "0.2.2"
                "13cpaq7f51gqcspd4097vjr7r2cjpxpn6c02x67dsdizk0xaiv5c"))

(define rust-oorandom-11.1.5
  (crate-source "oorandom" "11.1.5"
                "07mlf13z453fq01qff38big1lh83j8l6aaglf63ksqzzqxc0yyfn"))

(define rust-opaque-debug-0.2.3
  (crate-source "opaque-debug" "0.2.3"
                "172j6bs8ndclqxa2m64qc0y1772rr73g4l9fg2svscgicnbfff98"))

(define rust-opaque-debug-0.3.1
  (crate-source "opaque-debug" "0.3.1"
                "10b3w0kydz5jf1ydyli5nv10gdfp97xh79bgz327d273bs46b3f0"))

(define rust-open-2.0.3
  (crate-source "open" "2.0.3"
                "0rss5ffgw1r6rsqma6h33bkgrxm8d469w4c768pag3d0jr0027jb"))

(define rust-open-3.2.0
  (crate-source "open" "3.2.0"
                "1y3nhpgh444ns5vqy87jbcnlryqmw62akyi85k2a0m3akq1w0y10"))

(define rust-open-4.2.0
  (crate-source "open" "4.2.0"
                "04kw6xmsjm0wdqidjndhpzl79iksyrhwyik32z28wjjygq63q21s"))

(define rust-open-5.3.2
  (crate-source "open" "5.3.2"
                "15ggfx1p8rl7w4rr1n5qj1wxy1kk7757lsjpyc947a9fwri3aj72"))

(define rust-opener-0.6.1
  (crate-source "opener" "0.6.1"
                "1217aqxl8srr7v0bcc174yf5apaxjmzg0j62x8kb772g2yvdqqkc"))

(define rust-opener-0.7.2
  (crate-source "opener" "0.7.2"
                "10bn0m6pfv9mvv9lky0l48fb6vflx9pkg8sir1aa73gh9mg2x0fh"))

(define rust-opener-0.8.2
  (crate-source "opener" "0.8.2"
                "08r2jr56zha702lpsb5scrlplla71frj18s7qx7452ydz029f6vp"))

(define rust-opener-0.8.3
  (crate-source "opener" "0.8.3"
                "0isfar4r3h25kf1z35mz8r1sdh8gilm3a51akp4007mr5ab2946b"))

(define rust-openpgp-card-0.5.1
  (crate-source "openpgp-card" "0.5.1"
                "0vfcl9bb1ia7v2bj934957rc35jb8dqz9pcx50mvhs685c1zknan"))

(define rust-openpgp-cert-d-0.3.4
  (crate-source "openpgp-cert-d" "0.3.4"
                "1mpj8gsbf9m37bjiy6ggmcl305f3c497j1lsib52446znsq4gpfk"))

(define rust-openssh-keys-0.6.4
  (crate-source "openssh-keys" "0.6.4"
                "0gkcflf89fdzpmy066larfwk8fxc77ldq3b2r4bsrclq52l31f5b"))

(define rust-openssl-0.10.57
  (crate-source "openssl" "0.10.57"
                "0z0f8g84y0lvnbc60586ibjpf8r1q1dv672vfqan5d5bk7imxhms"
                #:snippet '(delete-file-recursively "test")))

(define rust-openssl-0.10.68
  (crate-source "openssl" "0.10.68"
                "1xbiz2bmba2fibg70s462yk2fndp3f9vz11c7iw0ilh2y54bqx31"
                #:snippet '(delete-file-recursively "test")))

(define rust-openssl-0.10.72
  (crate-source "openssl" "0.10.72"
                "1np54pm6hw512rmfjv3kc54h8yvf51mdlm8a8cc33xx1b1yympzy"
                #:snippet '(delete-file-recursively "test")))

(define rust-openssl-0.10.73
  (crate-source "openssl" "0.10.73"
                "1y7b3kanpgb92wiqhwbyqfsdr1xdjzssxqywl4cixay88r6p61c5"
                #:snippet '(delete-file-recursively "test")))

(define rust-openssl-macros-0.1.0
  (crate-source "openssl-macros" "0.1.0"
                "0v3kgnzbadrf9c06q4cqmbjas53av73n5w7wwz3n0nb6257y80dm"))

(define rust-openssl-macros-0.1.1
  (crate-source "openssl-macros" "0.1.1"
                "173xxvfc63rr5ybwqwylsir0vq6xsj4kxiv4hmg4c3vscdmncj59"))

(define rust-openssl-probe-0.1.6
  (crate-source "openssl-probe" "0.1.6"
                "0bl52x55laalqb707k009h8kfawliwp992rlsvkzy49n47p2fpnh"))

(define rust-openssl-src-300.4.2+3.4.1 #f)

(define rust-openssl-src-300.5.0+3.5.0 #f)

(define rust-openssl-sys-0.9.107
  (crate-source "openssl-sys" "0.9.107"
                "01yydv8yaagdnapvair8b6rggf225lwb854h99s9qx44rnd9g242"
                #:snippet
                #~(begin
                    ;; Remove dependency on boringssl and vendor openssl source.
                    (substitute* "Cargo.toml.orig"
                      (("vendored = .*") "vendored = []\n")
                      ((".*bssl.*") "")
                      ((".*openssl-src.*") "")
                      ;; Allow any version of bindgen.
                      (("(bindgen = \\{ version =) \".*\"," _ bindgen)
                       (string-append bindgen "\"*\",")))
                    (copy-file "Cargo.toml.orig" "Cargo.toml"))))

(define rust-openssl-sys-0.9.109
  (crate-source "openssl-sys" "0.9.109"
                "0wc54dshsac1xicq6b5wz01p358zcbf542f2s6vph3b38wp6w2ch"
                #:snippet
                #~(begin
                    ;; Remove dependency on boringssl and vendor openssl source.
                    (substitute* "Cargo.toml.orig"
                      (("vendored = .*") "vendored = []\n")
                      ((".*bssl.*") "")
                      ((".*openssl-src.*") "")
                      ;; Allow any version of bindgen.
                      (("(bindgen = \\{ version =) \".*\"," _ bindgen)
                       (string-append bindgen "\"*\",")))
                    (copy-file "Cargo.toml.orig" "Cargo.toml"))))

(define rust-optfield-0.3.0
  (crate-source "optfield" "0.3.0"
                "079a5n7lhqsj444fkgk580vra5v2nhym7czwnky9iip9rljz0ngs"))

(define rust-option-ext-0.2.0
  (crate-source "option-ext" "0.2.0"
                "0zbf7cx8ib99frnlanpyikm1bx8qn8x602sw1n7bg6p9x94lyx04"))

(define rust-option-operations-0.5.0
  (crate-source "option-operations" "0.5.0"
                "1l13n9487gk6063zzjdwzqbig78n8mh6dxsbiq9nbaxfn5xx49kw"))

(define rust-orbclient-0.3.48
  (crate-source "orbclient" "0.3.48"
                "0hzxjsvvsl5i9d3aqzc6kdcsch1i6flij5dkignhhkz2qb72c2xs"
                #:snippet '(delete-file-recursively "res")))

(define rust-ordered-float-1.1.1
  (crate-source "ordered-float" "1.1.1"
                "1drp3v9jsm1mn0wqp4p5y19j3bmz3w5rw4yxzi39zlld4wssy19k"))

(define rust-ordered-float-2.10.1
  (crate-source "ordered-float" "2.10.1"
                "075i108hr95pr7hy4fgxivib5pky3b6b22rywya5qyd2wmkrvwb8"))

(define rust-ordered-float-3.9.2
  (crate-source "ordered-float" "3.9.2"
                "1p3jkxlz89ndm4lmwr2n5kdnckhm5pcmqqkihkag259dff8c7qgi"))

(define rust-ordered-float-4.6.0
  (crate-source "ordered-float" "4.6.0"
                "0ldrcgilsiijd141vw51fbkziqmh5fpllil3ydhirjm67wdixdvv"))

(define rust-ordered-float-5.0.0
  (crate-source "ordered-float" "5.0.0"
                "009z1k7w729ls2sfg4zknn9v63sk1zghnq54p2lwcjjkdvszkhg2"))

(define rust-ordered-multimap-0.7.3
  (crate-source "ordered-multimap" "0.7.3"
                "0ygg08g2h381r3zbclba4zx4amm25zd2hsqqmlxljc00mvf3q829"))

(define rust-ordered-stream-0.2.0
  (crate-source "ordered-stream" "0.2.0"
                "0l0xxp697q7wiix1gnfn66xsss7fdhfivl2k7bvpjs4i3lgb18ls"))

(define rust-origin-0.24.0
  (crate-source "origin" "0.24.0"
                "13q8mffk9wmrs38v5phy2f8mjya2c1ghrd67icnpqrw27dvbivbd"))

(define rust-origin-0.26.1
  (crate-source "origin" "0.26.1"
                "18c8pddd01c12plxb7v98dd04cal6s0kdmchl43xczmkgmpzh8my"))

(define rust-orion-0.17.10
  (crate-source "orion" "0.17.10"
                "1r411g57482vcb94q12bvjk9rbirwhyx6b2n9i0czdf6kcv5vjfc"))

(define rust-orion-0.17.9
  (crate-source "orion" "0.17.9"
                "02sk0qv7r8nffqv56im11x4balxjb7zrxw867zab6pvwk9s0nbmz"))

(define rust-os-display-0.1.4
  (crate-source "os_display" "0.1.4"
                "13x07viih4f7l5cicbqw9xv378h0a096vphdclcbjvq2g4dxfpxd"))

(define rust-os-info-3.10.0
  (crate-source "os_info" "3.10.0"
                "1dd6hpdrrmir059dxcbnci96209snb4f597bc1l2hqa7q99lwq1a"))

(define rust-os-info-3.11.0
  (crate-source "os_info" "3.11.0"
                "1sdi8hbz4iawlrmn2adr9p0ql948lnj2xcjgqgaw4gd15hz8dz21"))

(define rust-os-pipe-1.2.1
  (crate-source "os_pipe" "1.2.1"
                "10nrh0i507560rsiy4c79fajdmqgbr6dha2pbl9mncrlaq52pzaz"))

(define rust-os-pipe-1.2.2
  (crate-source "os_pipe" "1.2.2"
                "090jhg0i1pj6l9w9zm0p0r7lv89kpwkj8vqij1iaskmic13mycyv"))

(define rust-os-release-0.1.0
  (crate-source "os-release" "0.1.0"
                "09zckcazggm33cbhjgdajm8rg41x9y75yf13rhcyqlqvyzi9mwl2"))

(define rust-os-str-bytes-6.6.1
  (crate-source "os_str_bytes" "6.6.1"
                "1885z1x4sm86v5p41ggrl49m58rbzzhd1kj72x46yy53p62msdg2"))

(define rust-os-str-bytes-7.1.0
  (crate-source "os_str_bytes" "7.1.0"
                "0qkbwv5fnbivv8zvnmg20irgfsr8v61vpabpik4b826hdnw2svn8"))

(define rust-ouroboros-0.18.5
  (crate-source "ouroboros" "0.18.5"
                "0ndy8hvp24gs7yxw9wj81hs5rb36wxmpw4i38ylrfjy4p46ha3qy"))

(define rust-ouroboros-macro-0.18.5
  (crate-source "ouroboros_macro" "0.18.5"
                "1l343ss6hlh0abbwjk6zah6mdlyhh1v1imflv3v86c6lsfyjhw1w"))

(define rust-output-vt100-0.1.3
  (crate-source "output_vt100" "0.1.3"
                "0rpvpiq7gkyvvwyp9sk0zxhbk99ldlrv5q3ycr03wkmbxgx270k2"))

(define rust-outref-0.1.0
  (crate-source "outref" "0.1.0"
                "1x61h7dl1cc6cj2f3zsalr8d98v0cw6497sykwxf74wjmqljh8kz"))

(define rust-outref-0.5.2
  (crate-source "outref" "0.5.2"
                "03pzw9aj4qskqhh0fkagy2mkgfwgj5a1m67ajlba5hw80h68100s"))

(define rust-overload-0.1.1
  (crate-source "overload" "0.1.1"
                "0fdgbaqwknillagy1xq7xfgv60qdbk010diwl7s1p0qx7hb16n5i"
                #:snippet '(delete-file "logo.png")))

(define rust-owned-ttf-parser-0.15.2
  (crate-source "owned_ttf_parser" "0.15.2"
                "1frgpf2a5j17ylk1lmrj2lpyhf6izq7x8b1xlbv6ybb3n7zazrh5"
                #:snippet '(delete-file-recursively "fonts")))

(define rust-owning-ref-0.4.1
  (crate-source "owning_ref" "0.4.1"
                "1kjj9m28wjv452jw49p1mp3d8ql058x78v4bz00avr7rvsnmpxbg"))

(define rust-owo-colors-3.5.0
  (crate-source "owo-colors" "3.5.0"
                "0vyvry6ba1xmpd45hpi6savd8mbx09jpmvnnwkf6z62pk6s4zc61"))

(define rust-owo-colors-4.2.0
  (crate-source "owo-colors" "4.2.0"
                "0r7mxiyxg8zbyjqm8y2n2amykl2i51y6agvjrw036ba2p5dqcdhh"))

(define rust-owo-colors-4.2.1
  (crate-source "owo-colors" "4.2.1"
                "1v0cvfas9v8fkk4w8jk3m51igj57sjp1cry3aqv7z88z40bm7696"))

(define rust-p256-0.13.2
  (crate-source "p256" "0.13.2"
                "0jyd3c3k239ybs59ixpnl7dqkmm072fr1js8kh7ldx58bzc3m1n9"
                #:snippet '(for-each delete-file-recursively '("src/test_vectors" "tests"))))

(define rust-p384-0.13.1
  (crate-source "p384" "0.13.1"
                "1dnnp133mbpp72mfss3fhm8wx3yp3p3abdhlix27v92j19kz2hpy"
                #:snippet '(delete-file-recursively "src/test_vectors")))

(define rust-p521-0.13.3
  (crate-source "p521" "0.13.3"
                "1cl5y2aypa1vxg181a0na3abndz1981pfdp2zkyml88z3wbf5j8g"
                #:snippet '(delete-file-recursively "src/test_vectors")))

(define rust-packed-simd-0.3.9
  (crate-source "packed_simd" "0.3.9"
                "0bck71ngyhx9icy7q3xzgmjxkylysxm6hgif5rqp2xc71jphi7qz"))

(define rust-packed-simd-2-0.3.8
  (crate-source "packed_simd_2" "0.3.8"
                "10p2bm0p57shg3arlpfwm6z0bbnlkyr4g0dlkmpwvz6qaba4r4d1"))

(define rust-pad-0.1.6
  (crate-source "pad" "0.1.6"
                "1qy0r26r52gp034fv252mmg0s599a55h9nr4xswy04hvky49pbfj"))

(define rust-pager-0.16.1
  (crate-source "pager" "0.16.1"
                "10188qgnsz988g30fvl4whkljh2zl4gpbp6kc48bpywpbhd23695"))

(define rust-palette-0.7.6
  (crate-source "palette" "0.7.6"
                "1rmn02mv6cb112504qyg7pyfa83c08hxpk5sw7jc5v659hc73gsc"))

(define rust-palette-derive-0.7.6
  (crate-source "palette_derive" "0.7.6"
                "0c0xhpk1nqyq4jr2m8xnka7w47vqzc7m2vq9ih8wxyjv02phs0zm"))

(define rust-pam-sys-0.5.6
  (crate-source "pam-sys" "0.5.6"
                "0d14501d5vybjnzxfjf96321xa5wa36x1xvf02h02zq938qmhj6d"))

(define rust-pandoc-0.8.11
  (crate-source "pandoc" "0.8.11"
                "0p56085drzwb143ximsvr5bni7k0cl9ld7dz7lh92hkslz8m6ga6"))

(define rust-pango-0.18.0
  (crate-source "pango" "0.18.0"
                "1l82j97pg7hyq2qnflg3lzyza307y8jvi5h02q3340qxhd5yba86"))

(define rust-pango-0.18.3
  (crate-source "pango" "0.18.3"
                "1r5ygq7036sv7w32kp8yxr6vgggd54iaavh3yckanmq4xg0px8kw"))

(define rust-pango-0.19.8
  (crate-source "pango" "0.19.8"
                "1kffxkk7730csly86fkgja50k1184zj9lz49sv7qb0059233439z"))

(define rust-pango-0.20.10
  (crate-source "pango" "0.20.10"
                "0f1902vsnhb6jprra2c0mhsr6k28d00pync2jc6qz17jc70kg3fq"))

(define rust-pango-0.20.12
  (crate-source "pango" "0.20.12"
                "0p5bj7k8sd2pgm7v907i9bip53ys46hahprs0jbr6rfzyq8v6xk5"))

(define rust-pango-0.8.0
  (crate-source "pango" "0.8.0"
                "0xq50950il3228grzs4xvc5s6phxcl5l50grz6syvs0vixr6p70y"))

(define rust-pango-sys-0.10.0
  (crate-source "pango-sys" "0.10.0"
                "1468xzyk2wanxb2b32fsmwk6bnafdaicxl5b4301dlb2ic66bli4"))

(define rust-pango-sys-0.18.0
  (crate-source "pango-sys" "0.18.0"
                "1iaxalcaaj59cl9n10svh4g50v8jrc1a36kd7n9yahx8j7ikfrs3"))

(define rust-pango-sys-0.19.8
  (crate-source "pango-sys" "0.19.8"
                "182bcd6255v5yvnskbhxnb6kwak240z7sn54si2b5h46l17xl0zz"))

(define rust-pango-sys-0.20.10
  (crate-source "pango-sys" "0.20.10"
                "1yj3n87whqx6gw3vip08zbckqxfg7l5jqc2wamaf76y07xkhjs8q"))

(define rust-pango-sys-0.9.1
  (crate-source "pango-sys" "0.9.1"
                "0zdynikh6jrfa31fpmbrwnz46a57idci73zzkf0z0g3vj223vfc6"))

(define rust-pangocairo-0.19.8
  (crate-source "pangocairo" "0.19.8"
                "1n8wrqy260zpfiifb2n10mbsv3kbrvxm1z7pv8b4w77c08yb9j74"))

(define rust-pangocairo-0.20.10
  (crate-source "pangocairo" "0.20.10"
                "0kyqga9i164d1kzq2lma68wln2iplh1zjx485nn696fva720v2aq"))

(define rust-pangocairo-sys-0.19.8
  (crate-source "pangocairo-sys" "0.19.8"
                "1myq3p8qrd63nlacd4sba66c17lfqgvzv8mpyn2rg1rqhi4h86ar"))

(define rust-pangocairo-sys-0.20.10
  (crate-source "pangocairo-sys" "0.20.10"
                "0hjwdfifvpbbrss814jsahkgqr5fxg8s5g774wlkxacaz01jk5dr"))

(define rust-papergrid-0.13.0
  (crate-source "papergrid" "0.13.0"
                "1ggfj9xvqqazcc2avmi9va40crdmlxjxlgjzi4yf25ziy7ggic6j"))

(define rust-papergrid-0.17.0
  (crate-source "papergrid" "0.17.0"
                "1wg3k8kgv0rgxqszf5c6dv6347mm58qb5kii0q4g9n2iif614y39"))

(define rust-parcel-selectors-0.28.2
  (crate-source "parcel_selectors" "0.28.2"
                "15m1hvl6rj8k0nkb28dlvgkkkmiviaklyhdpq4z6pjr6mpqh7zal"))

(define rust-parcel-sourcemap-2.1.1
  (crate-source "parcel_sourcemap" "2.1.1"
                "1fsvw1mlqc5x4psj90jxrdbivq8sqvxi5zz3q2vv4s4047bp8ns8"))

(define rust-paris-1.5.15
  (crate-source "paris" "1.5.15"
                "0bd6kb8kzbvy1mfk8jfpp0cjzi717q7hdjwjy98phg294cvspv4g"))

(define rust-paris-log-1.0.2
  (crate-source "paris-log" "1.0.2"
                "1syp82r34s9kc7m3qm5iy1lhamwzkmw34sxf5x39dvpy01nbwkbp"))

(define rust-parking-2.2.1
  (crate-source "parking" "2.2.1"
                "1fnfgmzkfpjd69v4j9x737b1k8pnn054bvzcn5dm3pkgq595d3gk"))

(define rust-parking-lot-0.11.1
  (crate-source "parking_lot" "0.11.1"
                "1sqmgaia8zfd5fbnqw2w13ijh7crk3lf9vw4cb52vwlx0an48xvd"))

(define rust-parking-lot-0.11.2
  (crate-source "parking_lot" "0.11.2"
                "16gzf41bxmm10x82bla8d6wfppy9ym3fxsmdjyvn61m66s0bf5vx"))

(define rust-parking-lot-0.12.3
  (crate-source "parking_lot" "0.12.3"
                "09ws9g6245iiq8z975h8ycf818a66q3c6zv4b5h8skpm7hc1igzi"))

(define rust-parking-lot-0.12.4
  (crate-source "parking_lot" "0.12.4"
                "04sab1c7304jg8k0d5b2pxbj1fvgzcf69l3n2mfpkdb96vs8pmbh"))

(define rust-parking-lot-core-0.8.0
  (crate-source "parking_lot_core" "0.8.0"
                "16yazfg3sq9mz6cfdkhgbv8yvc1kkasyhys4y7r3g16hgmralqf3"))

(define rust-parking-lot-core-0.8.6
  (crate-source "parking_lot_core" "0.8.6"
                "1p2nfcbr0b9lm9rglgm28k6mwyjwgm4knipsmqbgqaxdy3kcz8k0"))

(define rust-parking-lot-core-0.9.10
  (crate-source "parking_lot_core" "0.9.10"
                "1y3cf9ld9ijf7i4igwzffcn0xl16dxyn4c5bwgjck1dkgabiyh0y"))

(define rust-parking-lot-core-0.9.11
  (crate-source "parking_lot_core" "0.9.11"
                "19g4d6m5k4ggacinqprnn8xvdaszc3y5smsmbz1adcdmaqm8v0xw"))

(define rust-parse-datetime-0.6.0
  (crate-source "parse_datetime" "0.6.0"
                "1l4n9gsflf1fksfrvyhqyf8bdwxrk0s705l7x86g4jnxwds08wm8"
                #:snippet '(delete-file-recursively "tests")))

(define rust-parse-display-0.9.1
  (crate-source "parse-display" "0.9.1"
                "12jkqych9y44v1dqrhwjax4lylnqhsncd0hjj53293n9cli1qjli"))

(define rust-parse-display-derive-0.9.1
  (crate-source "parse-display-derive" "0.9.1"
                "10aj6h3hf2maf1zayg0p2m2ag9vriqrncwlivw9gsklp9h581rra"))

(define rust-parse-int-0.6.0
  (crate-source "parse_int" "0.6.0"
                "0niwamixgrpzb1k5l035aa40450dnrx69rvvzyy0hb3aj5wmns9d"))

(define rust-parse-js-0.20.1
  (crate-source "parse-js" "0.20.1"
                "03kijcsvms2aqkxc8yf0gy387n6zlw0y9yfrgr230nfb5pivahi7"))

(define rust-parse-zoneinfo-0.3.1
  (crate-source "parse-zoneinfo" "0.3.1"
                "093cs8slbd6kyfi6h12isz0mnaayf5ha8szri1xrbqj4inqhaahz"))

(define rust-partial-ref-0.3.3
  (crate-source "partial_ref" "0.3.3"
                "034i78nmzp2bdapvpz8fgh14932aj0s70l5s1kj5d5j7n74qnwhg"))

(define rust-partial-ref-derive-0.3.3
  (crate-source "partial_ref_derive" "0.3.3"
                "10bm0pxwjph40z9pqwdzkkkbyz9n1h6lx69f6jjvb65qnln1s3ih"))

(define rust-partition-identity-0.3.0
  (crate-source "partition-identity" "0.3.0"
                "08cymccnyf2b0pwc7x4wj889k1knqmrb8500idsjslybpvwjbacz"))

(define rust-pasetors-0.6.8
  (crate-source "pasetors" "0.6.8"
                "1fkdp4lya95nrkrds3i14288g4a85gxrlg8lnz8hs8zjcryd8dkb"))

(define rust-pasetors-0.7.2
  (crate-source "pasetors" "0.7.2"
                "0adpf4fd6bgkznrb4fzmbiahvh0c6s6i2pring2wkrx64px48jf5"))

(define rust-pasetors-0.7.4
  (crate-source "pasetors" "0.7.4"
                "0lb52kwpjhgzcxv0ikb9fcp84dw9jabpk8dy6pwqlwc03ik8qyfb"))

(define rust-password-hash-0.2.3
  (crate-source "password-hash" "0.2.3"
                "1rr4kd52ld978a2xhcvlc54p1d92yhxl9kvbajba7ia6rs5b5q3p"))

(define rust-password-hash-0.4.2
  (crate-source "password-hash" "0.4.2"
                "003p2hssyrcaxyq9fs8x2wx5di8ny9byaakskrf352pfm963fxkn"))

(define rust-password-hash-0.5.0
  (crate-source "password-hash" "0.5.0"
                "0ri1mim11zk0a9s40zdi288dfqvmdiryc7lw8vl46b59ifa08vrl"))

(define rust-paste-0.1.18
  (crate-source "paste" "0.1.18"
                "10587zrlmzhq66yhd0z36fzglf32m1nlhi9bxxm6dgl0gp3j1jj5"))

(define rust-paste-1.0.15
  (crate-source "paste" "1.0.15"
                "02pxffpdqkapy292harq6asfjvadgp1s005fip9ljfsn9fvxgh2p"))

(define rust-paste-impl-0.1.18
  (crate-source "paste-impl" "0.1.18"
                "1dlqzk05cx74522s4iyhyzzhszig4n401pp6r1qg6zmr02r7snnr"))

(define rust-path-abs-0.5.1
  (crate-source "path_abs" "0.5.1"
                "1hrkjrk9w2mhgzwl84gl2lmajs7yagdrcpxnjf51vh1a6kv05vq5"))

(define rust-path-absolutize-3.1.1
  (crate-source "path-absolutize" "3.1.1"
                "1xc36c5lz187wy452qph3lrr41x8ffgxk1clj2s9b8czwwgkibz4"))

(define rust-path-clean-0.1.0
  (crate-source "path-clean" "0.1.0"
                "1pcgqxw0mgg3ha5hi5xkjhyjf488bw5rw1g3qlr9awbq4szh3fpc"))

(define rust-path-clean-1.0.1
  (crate-source "path-clean" "1.0.1"
                "1vzwcrlz39rd94l89rppvkbsn7dvng449f1bnkyk3ayp43y9ld8p"))

(define rust-path-dedot-3.1.1
  (crate-source "path-dedot" "3.1.1"
                "15wkx8q3vra34fslzlg1lkq7liyxwqrpbxiz44a28wa7w3bhmfh7"))

(define rust-path-slash-0.1.5
  (crate-source "path-slash" "0.1.5"
                "15mni0f28mblwd97192c4zyyi6054yljmiqrdb6bx97ga69hk2j9"))

(define rust-path-slash-0.2.1
  (crate-source "path-slash" "0.2.1"
                "0hjgljv4vy97qqw9gxnwzqhhpysjss2yhdphfccy3c388afhk48y"))

(define rust-pathdiff-0.2.1
  (crate-source "pathdiff" "0.2.1"
                "1pa4dcmb7lwir4himg1mnl97a05b2z0svczg62l8940pbim12dc8"))

(define rust-pathdiff-0.2.3
  (crate-source "pathdiff" "0.2.3"
                "1lrqp4ip05df8dzldq6gb2c1sq2gs54gly8lcnv3rhav1qhwx56z"))

(define rust-pathsearch-0.2.0
  (crate-source "pathsearch" "0.2.0"
                "01bvjyighsrb5bgk8vcwlr1mjv6pqxk4n2qrkhbigaw2wp2kp66s"))

(define rust-patricia-tree-0.8.0
  (crate-source "patricia_tree" "0.8.0"
                "0s5fya6rvgg2gxxp5mbv0xdq8jqikps1sc6snk23zrgzkd9z9wii"))

(define rust-paw-1.0.0
  (crate-source "paw" "1.0.0"
                "1sc481y42rb08hmww525m4539ppl8k0w14kwxp13vg2dasdzrh09"))

(define rust-paw-attributes-1.0.2
  (crate-source "paw-attributes" "1.0.2"
                "0fda1v7y5pfmg8d2v7m0pyvif6c44qjz914jjn718pdyclrmhd8g"))

(define rust-paw-raw-1.0.0
  (crate-source "paw-raw" "1.0.0"
                "1wk76ipp34gjh42vivmgdkb2rgr26gwhn34gk7z5l378ixk5j2vz"))

(define rust-pay-respects-parser-0.3.9
  (crate-source "pay-respects-parser" "0.3.9"
                "0aww6qm0agzcx3g69bblkkhxz2gnbmvyk335jla0bayvhx6pczqy"))

(define rust-pay-respects-utils-0.1.8
  (crate-source "pay-respects-utils" "0.1.8"
                "191z4xjsdzn8k821071z4m8zl0jnd2r7j8ki2wpzdkdjha2g5mvn"))

(define rust-pbkdf2-0.11.0
  (crate-source "pbkdf2" "0.11.0"
                "05q9wqjvfrs4dvw03yn3bvcs4zghz0a7ycfa53pz2k2fqhp6k843"
                #:snippet '(delete-file-recursively "tests")))

(define rust-pbkdf2-0.12.2
  (crate-source "pbkdf2" "0.12.2"
                "1wms79jh4flpy1zi8xdp4h8ccxv4d85adc6zjagknvppc5vnmvgq"))

(define rust-pbkdf2-0.8.0
  (crate-source "pbkdf2" "0.8.0"
                "1ykgicvyjm41701mzqhrfmiz5sm5y0zwfg6csaapaqaf49a54pyr"
                #:snippet '(delete-file-recursively "tests")))

(define rust-pbkdf2-0.9.0
  (crate-source "pbkdf2" "0.9.0"
                "0fa7j0gdgghk64qlhzdv32yg52p0cfaz5ifhk7i4pfm1wsy98n7h"
                #:snippet '(delete-file-recursively "tests")))

(define rust-pcap-sys-0.1.3
  (crate-source "pcap-sys" "0.1.3"
                "18w9c8i5zjnazay86219w4789482ya617z35cw433h12wbgjaiqi"
                #:snippet '(for-each delete-file-recursively '("Lib" "libpcap"))))

(define rust-pciid-parser-0.6.3
  (crate-source "pciid-parser" "0.6.3"
                "0d6pliaab6xs2m1vnialy1biaw3nv6g9s0ykdrracx2a5h1p4jdd"))

(define rust-pcre2-0.2.9
  (crate-source "pcre2" "0.2.9"
                "0avb52h3ppw6kp9ynlw988c745am9j7px2cdsm0la10qmi1mrr9v"))

(define rust-pcre2-0.2.9.85b7afb rust-pcre2-utf32-0.2)

(define rust-pcre2-sys-0.2.9
  (crate-source "pcre2-sys" "0.2.9"
                "12hxy54i8amdnmcm034hqdc7iryd20n8aqg1hw5w540vzcc5s3sm"
                #:snippet '(delete-file-recursively "upstream")))

(define rust-pcre2-sys-0.2.9.85b7afb rust-pcre2-utf32-0.2)

(define rust-pcsc-2.9.0
  (crate-source "pcsc" "2.9.0"
                "1nj3i6h2qfhdpprrgr6hysdq6aar2wd55ls99j9nazlnz3n37n3x"))

(define rust-pcsc-sys-1.3.0
  (crate-source "pcsc-sys" "1.3.0"
                "00vhv7mmivqhn8373vdpp8h53sms3mn396p3m695abjxw4bz0kp1"))

(define rust-pdb-0.8.0
  (crate-source "pdb" "0.8.0"
                "0qs8lxx3ly029c77ip2mhlf0s9fmcbzlmaq0khkydar354whl142"))

(define rust-pdf-writer-0.12.1
  (crate-source "pdf-writer" "0.12.1"
                "1mdi4s5f164dbhczi5mvjz2z4q2s73qhdvwqyf9nzq3d45ykrw2x"
                #:snippet '(delete-file-recursively "examples")))

(define rust-pear-0.2.9
  (crate-source "pear" "0.2.9"
                "0rxlyizzaqq6lswgyfdxjxd3dyb1jfml9gwfpbx5g1j8rq0amvmx"))

(define rust-pear-codegen-0.2.9
  (crate-source "pear_codegen" "0.2.9"
                "0izijffdd2xs762497mk0xr7xwmyw62dzdqjz12v70n0bnc5pasb"))

(define rust-peeking-take-while-0.1.2
  (crate-source "peeking_take_while" "0.1.2"
                "16bhqr6rdyrp12zv381cxaaqqd0pwysvm1q8h2ygihvypvfprc8r"))

(define rust-peg-0.8.5
  (crate-source "peg" "0.8.5"
                "0xr42lbgihlwbyfcmjggqv1s3a38agp4j077cd8yqdhv235cya4r"))

(define rust-peg-macros-0.8.5
  (crate-source "peg-macros" "0.8.5"
                "0w8g53c75hz6iy3khc1ja2qm0idpnygjc0xsa9fmpyh2q82ap632"))

(define rust-peg-runtime-0.8.5
  (crate-source "peg-runtime" "0.8.5"
                "1jkgfy5jkjqwqy7h11nn3bmm1qfyn9kq2wnxnlspp4ldhsdwlb8k"))

(define rust-pem-3.0.5
  (crate-source "pem" "3.0.5"
                "1wwfk8sbyi9l18fvvn6z9p2gy7v7q7wimbhvrvixxj8a8zl3ibrq"))

(define rust-pem-rfc7468-0.7.0
  (crate-source "pem-rfc7468" "0.7.0"
                "04l4852scl4zdva31c1z6jafbak0ni5pi0j38ml108zwzjdrrcw8"
                #:snippet '(delete-file-recursively "tests")))

(define rust-pep440-rs-0.6.6
  (crate-source "pep440_rs" "0.6.6"
                "11g35bi0gagfchjcdvpaj1i82gxkrc391fcpm2f0cblw2yissvj6"))

(define rust-pep508-rs-0.6.1
  (crate-source "pep508_rs" "0.6.1"
                "0y99l63mi09vkl7b7qzdw5gn8kpq6kj26c9k280cik4rk947g21z"))

(define rust-percent-encoding-2.3.1
  (crate-source "percent-encoding" "2.3.1"
                "0gi8wgx0dcy8rnv1kywdv98lwcx67hz0a0zwpib5v2i08r88y573"))

(define rust-percent-encoding-2.3.2
  (crate-source "percent-encoding" "2.3.2"
                "083jv1ai930azvawz2khv7w73xh8mnylk7i578cifndjn5y64kwv"))

(define rust-pest-2.8.0
  (crate-source "pest" "2.8.0"
                "1dp741bxqiracvvwl66mfvlr29byvvph28n4c6ip136m652vg38r"))

(define rust-pest-2.8.1
  (crate-source "pest" "2.8.1"
                "08s342r6vv6ml5in4jk7pb97wgpf0frcnrvg0sqshn23sdb5zc0x"))

(define rust-pest-2.8.2
  (crate-source "pest" "2.8.2"
                "1a6g94pr4npsg0s6ljddwp4g127ks0nygzhxcpwfmyik6yis7q11"))

(define rust-pest-derive-2.8.0
  (crate-source "pest_derive" "2.8.0"
                "1icp5i01mgpbgwbkrcy4d0ykbxmns4wyz8j1jg6dr1wysz7xj9fp"))

(define rust-pest-derive-2.8.1
  (crate-source "pest_derive" "2.8.1"
                "1g20ma4y29axbjhi3z64ihhpqzmiix71qjn7bs224yd7isg6s1dv"))

(define rust-pest-derive-2.8.2
  (crate-source "pest_derive" "2.8.2"
                "0qy6nv84q1m6m2rzw1qjfbxlcizz7h557rkk16yivjqafxpp0n5w"))

(define rust-pest-generator-2.8.0
  (crate-source "pest_generator" "2.8.0"
                "0hgqngsxfr8y5p47bgjvd038j55ix1x4dpzr6amndaz8ddr02zfv"))

(define rust-pest-generator-2.8.1
  (crate-source "pest_generator" "2.8.1"
                "0rj9a20g4bjb4sl3zyzpxqg8mbn8c1kxp0nw08rfp0gp73k09r47"))

(define rust-pest-generator-2.8.2
  (crate-source "pest_generator" "2.8.2"
                "0bws5i6g3v1sldvy66p7qbzmz5mqbiflcqilaywgf1zy3n0kckvd"))

(define rust-pest-meta-2.8.0
  (crate-source "pest_meta" "2.8.0"
                "182w5fyiqm7zbn0p8313xc5wc73rnn59ycm5zk8hcja9f0j877vz"))

(define rust-pest-meta-2.8.1
  (crate-source "pest_meta" "2.8.1"
                "1mf01iln7shbnyxpdfnpf59gzn83nndqjkwiw3yh6n8g2wgi1lgd"))

(define rust-pest-meta-2.8.2
  (crate-source "pest_meta" "2.8.2"
                "0844iv71ibf414yid0bvhi9i0zfi0jrmyh6mvjjx1jws102rp4a2"))

(define rust-petgraph-0.5.1
  (crate-source "petgraph" "0.5.1"
                "1dzxda6z17sfxly11m8ja3iargh73pw0s1sdgjyp0qp5dm51cza6"))

(define rust-petgraph-0.6.5
  (crate-source "petgraph" "0.6.5"
                "1ns7mbxidnn2pqahbbjccxkrqkrll2i5rbxx43ns6rh6fn3cridl"
                #:snippet '(for-each delete-file-recursively '("assets"))))

(define rust-petgraph-0.7.1
  (crate-source "petgraph" "0.7.1"
                "0wkpppwrfv1h197asz1p4yfb4li5b1kw0nqllil67n6vj1qb6win"
                #:snippet '(delete-file-recursively "assets")))

(define rust-phf-0.10.1
  (crate-source "phf" "0.10.1"
                "0naj8n5nasv5hj5ldlva3cl6y3sv7zp3kfgqylhbrg55v3mg3fzs"))

(define rust-phf-0.11.3
  (crate-source "phf" "0.11.3"
                "0y6hxp1d48rx2434wgi5g8j1pr8s5jja29ha2b65435fh057imhz"))

(define rust-phf-0.8.0
  (crate-source "phf" "0.8.0"
                "04pyv8bzqvw69rd5dynd5nb85py1hf7wa4ixyhrvdz1l5qin3yrx"))

(define rust-phf-codegen-0.10.0
  (crate-source "phf_codegen" "0.10.0"
                "1k8kdad9wk2d5972k6jmjki2xpdy2ky4zd19rv7ybm2dpjlc7cag"))

(define rust-phf-codegen-0.11.3
  (crate-source "phf_codegen" "0.11.3"
                "0si1n6zr93kzjs3wah04ikw8z6npsr39jw4dam8yi9czg2609y5f"))

(define rust-phf-codegen-0.8.0
  (crate-source "phf_codegen" "0.8.0"
                "05d8w7aqqjb6039pfm6404gk5dlwrrf97kiy1n21212vb1hyxzyb"))

(define rust-phf-generator-0.10.0
  (crate-source "phf_generator" "0.10.0"
                "1mlq6hlajsvlsx6rhw49g9ricsm017lrxmgmmbk85sxm7f4qaljx"))

(define rust-phf-generator-0.11.3
  (crate-source "phf_generator" "0.11.3"
                "0gc4np7s91ynrgw73s2i7iakhb4lzdv1gcyx7yhlc0n214a2701w"))

(define rust-phf-generator-0.8.0
  (crate-source "phf_generator" "0.8.0"
                "09i5338d1kixq6a60fcayz6awgxjlxcfw9ic5f02abbgr067ydhp"))

(define rust-phf-macros-0.11.3
  (crate-source "phf_macros" "0.11.3"
                "05kjfbyb439344rhmlzzw0f9bwk9fp95mmw56zs7yfn1552c0jpq"))

(define rust-phf-macros-0.8.0
  (crate-source "phf_macros" "0.8.0"
                "170qm6yqn6b9mjlwb2xmm3iad9d5nzwgfawfwy7zr7s2zwcdwvvz"))

(define rust-phf-shared-0.10.0
  (crate-source "phf_shared" "0.10.0"
                "15n02nc8yqpd8hbxngblar2g53p3nllc93d8s8ih3p5cf7bnlydn"))

(define rust-phf-shared-0.11.3
  (crate-source "phf_shared" "0.11.3"
                "1rallyvh28jqd9i916gk5gk2igdmzlgvv5q0l3xbf3m6y8pbrsk7"))

(define rust-phf-shared-0.8.0
  (crate-source "phf_shared" "0.8.0"
                "1xssnqrrcn0nr9ayqrnm8xm37ac4xvwcx8pax7jxss7yxawzh360"))

(define rust-pico-args-0.5.0
  (crate-source "pico-args" "0.5.0"
                "05d30pvxd6zlnkg2i3ilr5a70v3f3z2in18m67z25vinmykngqav"))

(define rust-pid1-0.1.4
  (crate-source "pid1" "0.1.4"
                "1ss4jvsq6ygfjyswwqlzd16nh9gxhf95apzvbqfrj8pf1irz8svr"))

(define rust-pijul-config-0.0.1
  (crate-source "pijul-config" "0.0.1"
                "19xdcqyw2lh4r8rs5dfjzqf27ajfld2gimw07bhvnjixz0gplylq"))

(define rust-pijul-identity-0.0.1
  (crate-source "pijul-identity" "0.0.1"
                "1hpsw5k3s47nf8hxmi7pb6552ld3xz688hqldz57n9v7d7k660m1"))

(define rust-pijul-interaction-0.0.1
  (crate-source "pijul-interaction" "0.0.1"
                "0vv1z0flsgvf09y18sv2jbk9sn0rllka15wdskpv9ziydp9xr1h7"))

(define rust-pijul-macros-0.5.0
  (crate-source "pijul-macros" "0.5.0"
                "0r3hivav1mzpzdpk2rj8flhl4vgp2r85gdi0kw5x8r8bv0y79cy2"))

(define rust-pijul-remote-1.0.0-beta.6
  (crate-source "pijul-remote" "1.0.0-beta.6"
                "1pxk6bv3q9ppqivj6rdy2lgv444zxxkqs8jgb1zq3412j6gpsa6d"))

(define rust-pijul-repository-0.0.1
  (crate-source "pijul-repository" "0.0.1"
                "06vl6id0amw1pxfhj6m34mvr171czikph8g21xg34k7c263maiq4"))

(define rust-pikchr-0.1.4
  (crate-source "pikchr" "0.1.4"
                "09hqsykj71b02yzjmy7nxc73ir8arkg56l7i4gc78286m4v06s0k"))

(define rust-pin-project-1.1.10
  (crate-source "pin-project" "1.1.10"
                "12kadbnfm1f43cyadw9gsbyln1cy7vj764wz5c8wxaiza3filzv7"))

(define rust-pin-project-internal-1.1.10
  (crate-source "pin-project-internal" "1.1.10"
                "0qgqzfl0f4lzaz7yl5llhbg97g68r15kljzihaw9wm64z17qx4bf"))

(define rust-pin-project-lite-0.2.13
  (crate-source "pin-project-lite" "0.2.13"
                "0n0bwr5qxlf0mhn2xkl36sy55118s9qmvx2yl5f3ixkb007lbywa"))

(define rust-pin-project-lite-0.2.14
  (crate-source "pin-project-lite" "0.2.14"
                "00nx3f04agwjlsmd3mc5rx5haibj2v8q9b52b0kwn63wcv4nz9mx"))

(define rust-pin-project-lite-0.2.16
  (crate-source "pin-project-lite" "0.2.16"
                "16wzc7z7dfkf9bmjin22f5282783f6mdksnr0nv0j5ym5f9gyg1v"))

(define rust-pin-utils-0.1.0
  (crate-source "pin-utils" "0.1.0"
                "117ir7vslsl2z1a7qzhws4pd01cg2d3338c47swjyvqv2n60v1wb"))

(define rust-pinned-vec-0.1.1
  (crate-source "pinned_vec" "0.1.1"
                "12il2y234k9r4i3v9yh7qpavrwdhi48b057z96hb0bv2j8nxi2i6"))

(define rust-pinot-0.1.5
  (crate-source "pinot" "0.1.5"
                "16l044217rshhmr89ml0ppi4s86v6mhq5kx3lhac8djhz0zh38vb"))

(define rust-piper-0.2.4
  (crate-source "piper" "0.2.4"
                "0rn0mjjm0cwagdkay77wgmz3sqf8fqmv9d9czm79mvr2yj8c9j4n"))

(define rust-pipewire-0.7.2
  (crate-source "pipewire" "0.7.2"
                "1sg9cbvhp0s07a337zwli0xm40f8wkvm06d72nsr1s35vp40kl52"))

(define rust-pipewire-0.8.0
  (crate-source "pipewire" "0.8.0"
                "1nldg1hz4v0qr26lzdxqpvrac4zbc3pb6436sl392425bjx4brh8"))

(define rust-pipewire-0.8.0.93138d0 rust-pipewire-0.8.0.93138d0)

(define rust-pipewire-0.8.0.fd3d8f7 rust-pipewire-0.8.0.fd3d8f7)

(define rust-pipewire-sys-0.8.0.93138d0 rust-pipewire-0.8.0.93138d0)

(define rust-pipewire-sys-0.7.2
  (crate-source "pipewire-sys" "0.7.2"
                "0r4z0farzflycgfp6x7z65h57np4l1qnpj4r8z5lcwkkgd70h349"))

(define rust-pipewire-sys-0.8.0
  (crate-source "pipewire-sys" "0.8.0"
                "04hiy3rl8v3j2dfzp04gr7r8l5azzqqsvqdzwa7sipdij27ii7l4"))

(define rust-pipewire-sys-0.8.0.fd3d8f7 rust-pipewire-0.8.0.fd3d8f7)

(define rust-pixglyph-0.5.1
  (crate-source "pixglyph" "0.5.1"
                "152lvambhwqhsg8c6imzaxc081hch4lcw12sdy3kvgrng29zlnni"
                #:snippet '(delete-file-recursively "tests")))

(define rust-pixman-0.2.1
  (crate-source "pixman" "0.2.1"
                "1pqybqb7rmd58yr9xvmd8iix30znw5w71cq2wnlc16n1jva1g8nf"))

(define rust-pixman-sys-0.1.0
  (crate-source "pixman-sys" "0.1.0"
                "1nja8kc7zs1w4lhllvsgssa0b07n4cgwb0zyvqapj7g8i4z4i851"))

(define rust-pkcs1-0.7.5
  (crate-source "pkcs1" "0.7.5"
                "0zz4mil3nchnxljdfs2k5ab1cjqn7kq5lqp62n9qfix01zqvkzy8"
                #:snippet '(delete-file-recursively "tests")))

(define rust-pkcs8-0.10.2
  (crate-source "pkcs8" "0.10.2"
                "1dx7w21gvn07azszgqd3ryjhyphsrjrmq5mmz1fbxkj5g0vv4l7r"
                #:snippet '(delete-file-recursively "tests")))

(define rust-pkg-config-0.3.27
  (crate-source "pkg-config" "0.3.27"
                "0r39ryh1magcq4cz5g9x88jllsnxnhcqr753islvyk4jp9h2h1r6"))

(define rust-pkg-config-0.3.31
  (crate-source "pkg-config" "0.3.31"
                "1wk6yp2phl91795ia0lwkr3wl4a9xkrympvhqq8cxk4d75hwhglm"))

(define rust-pkg-config-0.3.32
  (crate-source "pkg-config" "0.3.32"
                "0k4h3gnzs94sjb2ix6jyksacs52cf1fanpwsmlhjnwrdnp8dppby"))

(define rust-pktparse-0.7.1
  (crate-source "pktparse" "0.7.1"
                "06sy7lwnhwmkyqfdbi4cs11z55rihipxafbdspnd5zg76pnbgbm8"))

(define rust-plain-0.2.3
  (crate-source "plain" "0.2.3"
                "19n1xbxb4wa7w891268bzf6cbwq4qvdb86bik1z129qb0xnnnndl"))

(define rust-platform-info-2.0.5
  (crate-source "platform-info" "2.0.5"
                "06j5v6hg914lbdr732jfx5syx703l5qwy1qk6dm4zjyqznrswfbm"))

(define rust-platforms-3.5.0
  (crate-source "platforms" "3.5.0"
                "009pi8n0vca83az15gj0fzpjq0dwy1y96jw6kr7ml21p08q6fd6l"))

(define rust-pledge-0.4.2
  (crate-source "pledge" "0.4.2"
                "1rc1r71hk0qi4ldbwbhbj6l6d1l4n28dfgf6znvl76kxgd0rj995"))

(define rust-plist-1.7.1
  (crate-source "plist" "1.7.1"
                "07a5s06q03n7s9xsw39nsn0mabr0wc9w2hzf18zfb9h33jc6xhpa"
                #:snippet '(delete-file-recursively "tests")))

(define rust-plist-1.7.2
  (crate-source "plist" "1.7.2"
                "1vaq384l8hj6w04pr0gxzc6jm6wcbccqcpx1ys2cv16mw9628xrx"
                #:snippet '(delete-file-recursively "tests")))

(define rust-plist-1.7.4
  (crate-source "plist" "1.7.4"
                "1qg7zkvnm4r4n9s9hbwwjnwznvkg6v0f035hza4agib3w64vbxis"
                #:snippet '(delete-file-recursively "tests")))

(define rust-plist-1.8.0
  (crate-source "plist" "1.8.0"
                "01qyv51ljbvhjbg8mva5c802b3dzrr95y6nd23wjh52xbjhvw3kl"))

(define rust-plotters-0.3.7
  (crate-source "plotters" "0.3.7"
                "0ixpy9svpmr2rkzkxvvdpysjjky4gw104d73n7pi2jbs7m06zsss"))

(define rust-plotters-backend-0.3.7
  (crate-source "plotters-backend" "0.3.7"
                "0ahpliim4hrrf7d4ispc2hwr7rzkn6d6nf7lyyrid2lm28yf2hnz"))

(define rust-plotters-svg-0.3.7
  (crate-source "plotters-svg" "0.3.7"
                "0w56sxaa2crpasa1zj0bhxzihlapqfkncggavyngg0w86anf5fji"))

(define rust-pmutil-0.5.3
  (crate-source "pmutil" "0.5.3"
                "0170zgziivri4qsch682pga3qq3z4wpr4wngzr5f9jyc97ayb51q"))

(define rust-pnet-base-0.35.0
  (crate-source "pnet_base" "0.35.0"
                "1xxj1ym32zqmy7m7ciiisv513rk9qis3p6x4mgrnmwbx0va91hgz"))

(define rust-pnet-datalink-0.35.0
  (crate-source "pnet_datalink" "0.35.0"
                "1dx7a9j2n7r463w8dv0wn1vasqnkhrajs79f6cm10qz11gn717p7"))

(define rust-pnet-sys-0.35.0
  (crate-source "pnet_sys" "0.35.0"
                "0jqgl34w5jckvby74nh89hjc94m8m6pz7hjh21s0hsyvsk9l6ikx"))

(define rust-png-0.17.16
  (crate-source "png" "0.17.16"
                "09kmkms9fmkbkarw0lnf0scqvjwwg3r7riddag0i3q39r0pil5c2"))

(define rust-png-0.18.0
  (crate-source "png" "0.18.0"
                "187jf0m873qn5biix8z7gjdsyf8r6vj3yr495pa0jja6i39wxflp"))

(define rust-polling-2.8.0
  (crate-source "polling" "2.8.0"
                "1kixxfq1af1k7gkmmk9yv4j2krpp4fji2r8j4cz6p6d7ihz34bab"))

(define rust-polling-3.11.0
  (crate-source "polling" "3.11.0"
                "0622qfbxi3gb0ly2c99n3xawp878fkrd1sl83hjdhisx11cly3jx"))

(define rust-polling-3.7.4
  (crate-source "polling" "3.7.4"
                "0bs4nhwfwsvlzlhah2gbhj3aa9ynvchv2g350wapswh26a65c156"))

(define rust-polling-3.8.0
  (crate-source "polling" "3.8.0"
                "0l7xlnzans2hc4zjxgplfp7gs16h9z3vdklnlpb2s05dj51nhfmm"))

(define rust-polling-3.10.0
  (crate-source "polling" "3.10.0"
                "0afqlnm45081k06sngc052k9vmh33j2rqrmjgi7q1zjhcca1kgdm"))

(define rust-pollster-0.4.0
  (crate-source "pollster" "0.4.0"
                "1qqcn0h2bvmgm9rlhfrdk7lfaiw1ad86g9500bhx1rj1s0c9yfig"))

(define rust-poly1305-0.8.0
  (crate-source "poly1305" "0.8.0"
                "1grs77skh7d8vi61ji44i8gpzs3r9x7vay50i6cg8baxfa8bsnc1"
                #:snippet '(delete-file-recursively "src/fuzz")))

(define rust-polyval-0.6.2
  (crate-source "polyval" "0.6.2"
                "09gs56vm36ls6pyxgh06gw2875z2x77r8b2km8q28fql0q6yc7wx"))

(define rust-popol-3.0.1
  (crate-source "popol" "3.0.1"
                "0s352804klrca8zcyw849p0qsr07y2h6qr62yls67slbzwf4rb7a"))

(define rust-pori-0.0.0
  (crate-source "pori" "0.0.0"
                "01p9g4fn3kasnmwj8i4plzk6nnnk7ak2qsfcv9b9y4zcilrkv9m4"))

(define rust-portable-atomic-1.11.0
  (crate-source "portable-atomic" "1.11.0"
                "0glb2wngflvfmg789qbf6dbnwcf6ai212fs7n0lf1c66rd49n3im"))

(define rust-portable-atomic-1.11.1
  (crate-source "portable-atomic" "1.11.1"
                "10s4cx9y3jvw0idip09ar52s2kymq8rq9a668f793shn1ar6fhpq"))

(define rust-portable-atomic-util-0.2.4
  (crate-source "portable-atomic-util" "0.2.4"
                "01rmx1li07ixsx3sqg2bxqrkzk7b5n8pibwwf2589ms0s3cg18nq"))

(define rust-portable-pty-0.8.1
  (crate-source "portable-pty" "0.8.1"
                "1gmh9ij90qwxx8gzvs6dj2vlc1ackv8zhd4mzfly3nq3586fhvl0"))

(define rust-portaudio-rs-0.3.2
  (crate-source "portaudio-rs" "0.3.2"
                "0qnmc7amk0fzbcs985ixv0k4955f0fmpkhrl9ps9pk3cz7pvbdnd"))

(define rust-portaudio-sys-0.1.1
  (crate-source "portaudio-sys" "0.1.1"
                "1xdpywirpr1kqkbak7hnny62gmsc93qgc3ij3j2zskrvjpxa952i"))

(define rust-posix-regex-0.1.1
  (crate-source "posix-regex" "0.1.1"
                "0hfkrlrfj10i7a54p3fg4c8ww01dy0v7l3lvmwc9pfliprf6dy0d"))

(define rust-posix-regex-0.1.4
  (crate-source "posix-regex" "0.1.4"
                "1g7k5chhpaq8h29czmlj39ryrmwkakl1parr91s2zaya6h1mipv6"))

(define rust-postcard-1.1.1
  (crate-source "postcard" "1.1.1"
                "1a18yvgmjazphf6frnlnq4nnl6pp2l5qgi6qxnldpjbwyq0jc2hp"))

(define rust-potential-utf-0.1.2
  (crate-source "potential_utf" "0.1.2"
                "11dm6k3krx3drbvhgjw8z508giiv0m09wzl6ghza37176w4c79z5"))

(define rust-potential-utf-0.1.3
  (crate-source "potential_utf" "0.1.3"
                "12mhwvhpvvim6xqp6ifgkh1sniv9j2cmid6axn10fnjvpsnikpw4"))

(define rust-powerfmt-0.2.0
  (crate-source "powerfmt" "0.2.0"
                "14ckj2xdpkhv3h6l5sdmb9f1d57z8hbfpdldjc2vl5givq2y77j3"))

(define rust-ppv-lite86-0.2.17
  (crate-source "ppv-lite86" "0.2.17"
                "1pp6g52aw970adv3x2310n7glqnji96z0a9wiamzw89ibf0ayh2v"))

(define rust-ppv-lite86-0.2.21
  (crate-source "ppv-lite86" "0.2.21"
                "1abxx6qz5qnd43br1dd9b2savpihzjza8gb4fbzdql1gxp2f7sl5"))

(define rust-precomputed-hash-0.1.1
  (crate-source "precomputed-hash" "0.1.1"
                "075k9bfy39jhs53cb2fpb9klfakx2glxnf28zdw08ws6lgpq6lwj"))

(define rust-predicates-2.1.5
  (crate-source "predicates" "2.1.5"
                "1pdd8zgw8351bvk882zadyqghnxzkjlf946vkdb1hgkzqdihl8sr"))

(define rust-predicates-3.1.3
  (crate-source "predicates" "3.1.3"
                "0wrm57acvagx0xmh5xffx5xspsr2kbggm698x0vks132fpjrxld5"))

(define rust-predicates-core-1.0.9
  (crate-source "predicates-core" "1.0.9"
                "1yjz144yn3imq2r4mh7k9h0r8wv4yyjjj57bs0zwkscz24mlczkj"))

(define rust-predicates-tree-1.0.12
  (crate-source "predicates-tree" "1.0.12"
                "0p223d9y02ywwxs3yl68kziswz4da4vabz67jfhp7yqx71njvpbj"))

(define rust-pretty-0.12.4
  (crate-source "pretty" "0.12.4"
                "0xx4jqir4ipkv2c11lvnsxcpxf34kd616nmbfps7bg09f4xpg65c"))

(define rust-pretty-assertions-0.6.1
  (crate-source "pretty_assertions" "0.6.1"
                "09yl14gnmpygiqrdlsa64lcl4w6ydjl9m8jri6kgam0v9rjf309z"
                #:snippet '(delete-file-recursively "examples")))

(define rust-pretty-assertions-0.7.2
  (crate-source "pretty_assertions" "0.7.2"
                "12yris0ni87wvzhj23a5nnz7amskvahhnpikx5snhdyg09y0xaqw"
                #:snippet '(delete-file-recursively "examples")))

(define rust-pretty-assertions-1.4.1
  (crate-source "pretty_assertions" "1.4.1"
                "0v8iq35ca4rw3rza5is3wjxwsf88303ivys07anc5yviybi31q9s"
                #:snippet '(delete-file-recursively "examples")))

(define rust-pretty-bytes-0.2.2
  (crate-source "pretty-bytes" "0.2.2"
                "05jkd1f824b93jh0jwfskba9hd70crvjz2nl1hf2xgqx5kfnx780"))

(define rust-pretty-env-logger-0.5.0
  (crate-source "pretty_env_logger" "0.5.0"
                "076w9dnvcpx6d3mdbkqad8nwnsynb7c8haxmscyrz7g3vga28mw6"))

(define rust-pretty-hex-0.4.1
  (crate-source "pretty-hex" "0.4.1"
                "0m0j8pqmh6gq1mq7yzp12z0ix159fw0di5lhiwv2y1j0m3j3xj5v"
                #:snippet '(delete-file-recursively "tests")))

(define rust-prettyplease-0.1.25
  (crate-source "prettyplease" "0.1.25"
                "11lskniv8pf8y8bn4dc3nmjapfhnibxbm5gamp2ad9qna3lld1kc"))

(define rust-prettyplease-0.2.32
  (crate-source "prettyplease" "0.2.32"
                "1xmdmwhsvqc8l5ns029vzjida4k3lp5ynin0xra43qsiki0wakk6"))

(define rust-prettyplease-0.2.34
  (crate-source "prettyplease" "0.2.34"
                "0mfzvrniybmfwk1m1xmccrn20ggfs61qz02hgnc5zx311phvjdv8"))

(define rust-prettyplease-0.2.35
  (crate-source "prettyplease" "0.2.35"
                "16jklwmgqfzi6izcs6c4mqbmkzjv4zrbn9cx8wk9n1qycchi4706"))

(define rust-prettyplease-0.2.37
  (crate-source "prettyplease" "0.2.37"
                "0azn11i1kh0byabhsgab6kqs74zyrg69xkirzgqyhz6xmjnsi727"))

(define rust-prettytable-rs-0.10.0
  (crate-source "prettytable-rs" "0.10.0"
                "0nnryfnahfwy0yxhv4nsp1id25k00cybx3ih8xjsp9haa43mx8pf"))

(define rust-primal-check-0.3.4
  (crate-source "primal-check" "0.3.4"
                "025xnak4rhkwa4h970bjb3cvp2k853wviyr84n8gjfhy65dqj3fw"))

(define rust-primeorder-0.13.6
  (crate-source "primeorder" "0.13.6"
                "1rp16710mxksagcjnxqjjq9r9wf5vf72fs8wxffnvhb6i6hiqgim"))

(define rust-print-bytes-1.2.0
  (crate-source "print_bytes" "1.2.0"
                "0a1pc0bs0f3mgy86mazdy81k6iaj8s23ly49mcka4in2pj27lzxl"))

(define rust-print-positions-0.6.1
  (crate-source "print-positions" "0.6.1"
                "026jzdf63b37bb9ix3mpczln2pqylsiwkkxhikj05x9y1r3r7x8x"))

(define rust-printf-compat-0.1.1
  (crate-source "printf-compat" "0.1.1"
                "0164d89yw0fy8fydx5jk9m4ml4i81a0igrra41kksggyizr2l01v"))

(define rust-priority-queue-1.4.0
  (crate-source "priority-queue" "1.4.0"
                "11g7bzlzva56k19h1023d1pw6hrk86p4mmajawicjnz09wbakgd0"))

(define rust-priority-queue-2.3.1
  (crate-source "priority-queue" "2.3.1"
                "13ff7y3s9x6m9q0dazdnjz6v0b3j2iyxfjljm9cim6jql5gp027g"))

(define rust-proc-macro-crate-0.1.5
  (crate-source "proc-macro-crate" "0.1.5"
                "11cpihdk9ba68hzw95aa8zxn0i5g6kdrfd4l2cy3d5jvb72a6vhx"))

(define rust-proc-macro-crate-1.3.1
  (crate-source "proc-macro-crate" "1.3.1"
                "069r1k56bvgk0f58dm5swlssfcp79im230affwk6d9ck20g04k3z"))

(define rust-proc-macro-crate-2.0.0
  (crate-source "proc-macro-crate" "2.0.0"
                "1s23imns07vmacn2xjd5hv2h6rr94iqq3fd2frwa6i4h2nk6d0vy"))

(define rust-proc-macro-crate-3.3.0
  (crate-source "proc-macro-crate" "3.3.0"
                "0d9xlymplfi9yv3f5g4bp0d6qh70apnihvqcjllampx4f5lmikpd"))

(define rust-proc-macro-error-1.0.4
  (crate-source "proc-macro-error" "1.0.4"
                "1373bhxaf0pagd8zkyd03kkx6bchzf6g0dkwrwzsnal9z47lj9fs"))

(define rust-proc-macro-error-attr-1.0.4
  (crate-source "proc-macro-error-attr" "1.0.4"
                "0sgq6m5jfmasmwwy8x4mjygx5l7kp8s4j60bv25ckv2j1qc41gm1"))

(define rust-proc-macro-error-attr2-2.0.0
  (crate-source "proc-macro-error-attr2" "2.0.0"
                "1ifzi763l7swl258d8ar4wbpxj4c9c2im7zy89avm6xv6vgl5pln"))

(define rust-proc-macro-error2-2.0.1
  (crate-source "proc-macro-error2" "2.0.1"
                "00lq21vgh7mvyx51nwxwf822w2fpww1x0z8z0q47p8705g2hbv0i"))

(define rust-proc-macro-hack-0.5.20+deprecated
  (crate-source "proc-macro-hack" "0.5.20+deprecated"
                "0s402hmcs3k9nd6rlp07zkr1lz7yimkmcwcbgnly2zr44wamwdyw"))

(define rust-proc-macro2-0.4.30
  (crate-source "proc-macro2" "0.4.30"
                "0nd71fl24sys066jrha6j7i34nfkjv44yzw8yww9742wmc8j0gfg"))

(define rust-proc-macro2-1.0.32
  (crate-source "proc-macro2" "1.0.32"
                "0hqbxlvhiaybakl1gai3mgps1dxsmxricxsr2rfdrh222z0qql5s"))

(define rust-proc-macro2-1.0.66
  (crate-source "proc-macro2" "1.0.66"
                "1ngawak3lh5p63k5x2wk37qy65q1yylk1phwhbmb5pcv7zdk3yqq"))

(define rust-proc-macro2-1.0.67
  (crate-source "proc-macro2" "1.0.67"
                "0a0k7adv0yswsgzsqkd7r6ng8rpcdyqrhra5v5ii531y3agkshrx"))

(define rust-proc-macro2-1.0.89
  (crate-source "proc-macro2" "1.0.89"
                "0vlq56v41dsj69pnk7lil7fxvbfid50jnzdn3xnr31g05mkb0fgi"))

(define rust-proc-macro2-1.0.93
  (crate-source "proc-macro2" "1.0.93"
                "169dw9wch753if1mgyi2nfl1il77gslvh6y2q46qplprwml6m530"))

(define rust-proc-macro2-1.0.94
  (crate-source "proc-macro2" "1.0.94"
                "114wxb56gdj9vy44q0ll3l2x9niqzcbyqikydmlb5f3h5rsp26d3"))

(define rust-proc-macro2-1.0.95
  (crate-source "proc-macro2" "1.0.95"
                "0y7pwxv6sh4fgg6s715ygk1i7g3w02c0ljgcsfm046isibkfbcq2"))

(define rust-proc-macro2-1.0.101
  (crate-source "proc-macro2" "1.0.101"
                "1pijhychkpl7rcyf1h7mfk6gjfii1ywf5n0snmnqs5g4hvyl7bl9"))

(define rust-proc-macro2-diagnostics-0.10.1
  (crate-source "proc-macro2-diagnostics" "0.10.1"
                "1j48ipc80pykvhx6yhndfa774s58ax1h6sm6mlhf09ls76f6l1mg"))

(define rust-proc-maps-0.4.0
  (crate-source "proc-maps" "0.4.0"
                "1ys4dj9wzqj6z5k7i28qy7l2ff224znvjfyrzhjkl68flrd4rd1x"))

(define rust-proc-mounts-0.3.0
  (crate-source "proc-mounts" "0.3.0"
                "1wpw3z2qq8wm3da8d0253a1h95nma6lad41m9yzp1ayh6n22yr8d"))

(define rust-process-control-5.0.0
  (crate-source "process_control" "5.0.0"
                "0qhbyxrfz6ywmsm69pjl7vxzgj6plc6qn6lp0zgqlw17laslwb7k"))

(define rust-process-wrap-8.2.0
  (crate-source "process-wrap" "8.2.0"
                "0m1zn33axqn9djw52c9yl1x75vf3a3lzbpmlcnq2c4wdk34lspyk"
                #:snippet '(delete-file-recursively "logo.png")))

(define rust-process-wrap-8.2.1
  (crate-source "process-wrap" "8.2.1"
                "189vzjn8dan18cnb0qlk3b472a6imji8wqlzxj13mwi20hplzvx3"))

(define rust-procfs-0.17.0
  (crate-source "procfs" "0.17.0"
                "17swyjqinpb745f07dpdi7c8q37hxvhx9xmmsi2dhxaj2kc74nyc"))

(define rust-procfs-core-0.17.0
  (crate-source "procfs-core" "0.17.0"
                "1v0jdbyc1rq1x22m0wn7n4iq4h86gdls38wqfg06zc29hcnz1793"))

(define rust-prodash-28.0.0
  (crate-source "prodash" "28.0.0"
                "0y9d16s79168rc5k2djjb16vjcx27yargbfb6xz6m2mq4r6jcjkl"))

(define rust-prodash-29.0.1
  (crate-source "prodash" "29.0.1"
                "12xm50jzkqzdqdcidmzy4d6rj9r8x6mf8sidgrh7dfc0r4jcxrwy"))

(define rust-prodash-29.0.2
  (crate-source "prodash" "29.0.2"
                "1g6i2zk6q9dqgspxn14zp31w14rcph7983lap4ilr228yq4b2jzh"))

(define rust-prodash-30.0.1
  (crate-source "prodash" "30.0.1"
                "0fdi0wxgy3s9643dgyfkwgmm12g4a360djy56zbxkls9d1bgqvjs"))

(define rust-profiling-1.0.16
  (crate-source "profiling" "1.0.16"
                "0kcz2xzg4qx01r5az8cf9ffjasi2srj56sna32igddh0vi7cggdg"))

(define rust-profiling-1.0.17
  (crate-source "profiling" "1.0.17"
                "0wqp6i1bl7azy9270dp92srbbr55mgdh9qnk5b1y44lyarmlif1y"))

(define rust-profiling-procmacros-1.0.16
  (crate-source "profiling-procmacros" "1.0.16"
                "0c7y2k4mz5dp2ksj1h4zbxsxq4plmjzccscdaml3h1pizdh2wpx6"))

(define rust-profiling-procmacros-1.0.17
  (crate-source "profiling-procmacros" "1.0.17"
                "0nrxdh5r723raxbs136jmjx46p0c5qgai8jwz4j555mn0ad7ywaj"))

(define rust-progressing-3.0.2
  (crate-source "progressing" "3.0.2"
                "06sb1cxpkc8lx56s76c95cfljs0513nsnn35wd6w79sblwcxpdwp"))

(define rust-project-origins-1.2.0
  (crate-source "project-origins" "1.2.0"
                "0ya364x9dj72bcapvndmhv5qwyw4m3p1ddjw6hw85jk5y9bhv7k2"))

(define rust-project-origins-1.4.1
  (crate-source "project-origins" "1.4.1"
                "0b4wihd54zj017v71dg7gpmzivm4i18j5993zzcivs5ccdqj187i"))

(define rust-project-origins-1.4.2
  (crate-source "project-origins" "1.4.2"
                "09qmg243nqhpakn6x4j7piz092rq7i8lncllvw4vhb8h3la848z4"))

(define rust-prometheus-0.14.0
  (crate-source "prometheus" "0.14.0"
                "0fpl98whrg5j4bpb3qfswii4yfa58zws7rl7rnd0m58bimnk599w"))

(define rust-promptly-0.3.1
  (crate-source "promptly" "0.3.1"
                "1fl0548ww11gpja8hlsfc8jgxk00rdd48n1g6icgwadhlp2wdjws"))

(define rust-proptest-1.6.0
  (crate-source "proptest" "1.6.0"
                "0l4y4bb8hffv7cys7d59qwqdmvmqjfzz0x9vblc08209clqfkjhl"))

(define rust-proptest-1.7.0
  (crate-source "proptest" "1.7.0"
                "07s73fqbhdzmnhnd8ks6l5mlkwaz048a49kprwqml6dmvqcspkbg"))

(define rust-proptest-1.8.0
  (crate-source "proptest" "1.8.0"
                "1kp45ni5j31l0r27fqx68if0ldi5nj7jj1xl1dm6h46dpq3vxc1b"))

(define rust-proptest-derive-0.5.1
  (crate-source "proptest-derive" "0.5.1"
                "0jay6jwfvrwzz5bqpi4hxx3ax6kax06p0h29vgkxb0vl42nckqaf"))

(define rust-proptest-derive-0.6.0
  (crate-source "proptest-derive" "0.6.0"
                "0c6xy4m6ab7a6h24gxiicwlq12dgvbl2ns4v6l14hwv9bkvrjnh9"))

(define rust-proptest-state-machine-0.3.1
  (crate-source "proptest-state-machine" "0.3.1"
                "1m18ybyxk86qvkls5i4s71q1rcwfzd8pqj4nnh7p81wxw10d2hz9"))

(define rust-prost-0.12.6
  (crate-source "prost" "0.12.6"
                "0a8z87ir8yqjgl1kxbdj30a7pzsjs9ka85szll6i6xlb31f47cfy"))

(define rust-prost-0.13.5
  (crate-source "prost" "0.13.5"
                "1r8yi6zxxwv9gq5ia9p55nspgwmchs94sqpp64x33v5k3njgm5i7"))

(define rust-prost-build-0.13.5
  (crate-source "prost-build" "0.13.5"
                "1gw1mr0rmv15fc2yvn9jmxbqaj8qh80w5nn5x5s1932y8ijr8xmy"))

(define rust-prost-derive-0.12.6
  (crate-source "prost-derive" "0.12.6"
                "1waaq9d2f114bvvpw957s7vsx268licnfawr20b51ydb43dxrgc1"))

(define rust-prost-derive-0.13.5
  (crate-source "prost-derive" "0.13.5"
                "0kgc9gbzsa998xixblfi3kfydka64zqf6rmpm53b761cjxbxfmla"))

(define rust-prost-types-0.13.5
  (crate-source "prost-types" "0.13.5"
                "05mx699wyg7cjil3hz7h8lp4dhi7xhy1lq5kjv1s3cfx6szw3hjj"))

(define rust-protobuf-3.7.2
  (crate-source "protobuf" "3.7.2"
                "1x4riz4znnjsqpdxnhxj0aq8rfivmbv4hfqmd3gbbn77v96isnnn"))

(define rust-protobuf-codegen-3.7.2
  (crate-source "protobuf-codegen" "3.7.2"
                "1kjaakqk0595akxdhv68w23zw136hw0h0kxkyg9bn500bj17cfax"))

(define rust-protobuf-parse-3.7.2
  (crate-source "protobuf-parse" "3.7.2"
                "0wy9pnfrsk2iz2ghhvzdpp0riklrm6p8dvdfxr4d7wb04hgsmbml"))

(define rust-protobuf-support-3.7.2
  (crate-source "protobuf-support" "3.7.2"
                "1mnpn2q96bxm2vidh86m5p2x5z0z8rgfyixk1wlgjiqa3vrw4diy"))

(define rust-psl-2.1.56
  (crate-source "psl" "2.1.56"
                "00mmvh1w0mdqpzsa8ylggcxc1gvi7zc9hyswlsb7bk5ahqx42zjk"))

(define rust-psl-types-2.0.11
  (crate-source "psl-types" "2.0.11"
                "1b3cz1q07iy744a39smykra2j83nv8vmni6la37wnx3ax17jkjrk"))

(define rust-psm-0.1.25
  (crate-source "psm" "0.1.25"
                "125y7h40mkwb64j4v2v7s6f69ilk745kg60w1s2cq62cw8im93pm"
                #:snippet '(delete-file "src/arch/wasm32.o")))

(define rust-psm-0.1.26
  (crate-source "psm" "0.1.26"
                "07xfl8gpz8v9qz2zvnpcy9r3nppbhxlxkgqbxcdwsdl5xij4953f"
                #:snippet '(delete-file "src/arch/wasm32.o")))

(define rust-ptr-meta-0.1.4
  (crate-source "ptr_meta" "0.1.4"
                "1wd4wy0wxrcays4f1gy8gwcmxg7mskmivcv40p0hidh6xbvwqf07"))

(define rust-ptr-meta-0.3.0
  (crate-source "ptr_meta" "0.3.0"
                "147a6z4qz35gipj9k0d2yh4wygmibhaqsna59vs0d5izdpv7d7py"))

(define rust-ptr-meta-derive-0.1.4
  (crate-source "ptr_meta_derive" "0.1.4"
                "1b69cav9wn67cixshizii0q5mlbl0lihx706vcrzm259zkdlbf0n"))

(define rust-ptr-meta-derive-0.3.0
  (crate-source "ptr_meta_derive" "0.3.0"
                "1l9jznaz85cchixyp07v6sxcvjadsyq6lmhjbh98sk0v2pdlwhfa"))

(define rust-ptree-0.4.0
  (crate-source "ptree" "0.4.0"
                "0w72k0svlj8ihbf7m7ivay7rpv38xz9ad5a06isyqsiiddwq1pm0"
                #:snippet '(delete-file "data/showcase-1.png")))

(define rust-pty-process-0.4.0
  (crate-source "pty-process" "0.4.0"
                "1h2wx8mwhr3a90al1y22pf41hkqrq967crsp9bvhpja4w92vajc7"))

(define rust-ptyprocess-0.4.1
  (crate-source "ptyprocess" "0.4.1"
                "1kz8gzsr2pjpxcxy708338wcdqnxg3cpfbca8q8a44gvpvvsw1by"))

(define rust-pubgrub-0.3.0-alpha.1.b70cf70 rust-pubgrub-0.3.0.b70cf70)

(define rust-public-api-0.33.1
  (crate-source "public-api" "0.33.1"
                "1i1qg7z96l0y5cjna6kppsvkvlpqkvmsns75zjb61jbsh3pfwndp"))

(define rust-publicsuffix-2.3.0
  (crate-source "publicsuffix" "2.3.0"
                "1pzsm61x49zgdcb14b5fnmjj36k129gc37lpfrgk6q5bdi2flhkg"))

(define rust-puffin-0.16.0
  (crate-source "puffin" "0.16.0"
                "08ass1hfdcq86y7dywa1jylzq57la95rgpcmd6yx82hs9symlhkn"))

(define rust-puffin-0.19.1
  (crate-source "puffin" "0.19.1"
                "07vlkf4i88475a80fhckayzxr9v4pkc21kwvpjkc2bn00mxsx7gs"))

(define rust-puffin-http-0.13.0
  (crate-source "puffin_http" "0.13.0"
                "14w1ihjlv48mpbh114yvgixdqdnzzipnmsg158l3v49m1ihgrgqk"))

(define rust-pulldown-cmark-0.10.3
  (crate-source "pulldown-cmark" "0.10.3"
                "14rrzqnv6j64j75558m7gzw6lc9b24057v6415smx1z7cvm9p5vn"))

(define rust-pulldown-cmark-escape-0.10.1
  (crate-source "pulldown-cmark-escape" "0.10.1"
                "1lqx7c2f0bx0qq9kkyn18gsa2dl2sk8x5jp8gvdax75w73sqyd5x"))

(define rust-pulldown-cmark-0.12.2
  (crate-source "pulldown-cmark" "0.12.2"
                "055v2bzzrkrbrc1s6l9mbkvpdkhkid5j7vdkpcnc9k7b582s4szq"))

(define rust-pulldown-cmark-0.13.0
  (crate-source "pulldown-cmark" "0.13.0"
                "183fg8lrdhvydrx0dmg446n108rwrvnzci0sd1ig7lkbjqdbx2qy"))

(define rust-pulldown-cmark-0.9.6
  (crate-source "pulldown-cmark" "0.9.6"
                "0av876a31qvqhy7gzdg134zn4s10smlyi744mz9vrllkf906n82p"))

(define rust-pulldown-cmark-escape-0.11.0
  (crate-source "pulldown-cmark-escape" "0.11.0"
                "1bp13akkz52p43vh2ffpgv604l3xd9b67b4iykizidnsbpdqlz80"))

(define rust-pulldown-cmark-to-cmark-10.0.4
  (crate-source "pulldown-cmark-to-cmark" "10.0.4"
                "0gc366cmd5jxal9m95l17rvqsm4dn62lywc8v5gwq8vcjvhyd501"))

(define rust-pulley-interpreter-29.0.1
  (crate-source "pulley-interpreter" "29.0.1"
                "0pg7rvnqkj47vz91zyqh0b1rvkw8m14jy64qhdqa4jfzfn2mznb2"))

(define rust-pure-rust-locales-0.8.1
  (crate-source "pure-rust-locales" "0.8.1"
                "0fkkwggiq2053rmiah2h06dz6w3yhy9pa82g30vy3sbcmqcgv40i"))

(define rust-pwd-1.4.0
  (crate-source "pwd" "1.4.0"
                "18p4j95sqqcxn3fbm6gbi7klxp8n40xmcjqy9vz1ww5rg461rivj"))

(define rust-pxfm-0.1.20
  (crate-source "pxfm" "0.1.20"
                "1y4n7c9m7hkykh7c9dmfd4v34xqrk2k442jzjj36wvsg360hhybf"))

(define rust-py-literal-0.4.0
  (crate-source "py_literal" "0.4.0"
                "1qd8j3a9zlpq6rjaxabpc9sacw62dn1cr38p3y4x7fbdsjizfb8h"))

(define rust-py-spy-testdata-0.1.0
  (crate-source "py-spy-testdata" "0.1.0"
                "18d880f2rjdd42828srh97vpvlrs9pg23j28gphw9qd2w3bp128q"))

(define rust-pyo3-0.15.2
  (crate-source "pyo3" "0.15.2"
                "0ljp3m1frg8y50al4cbadwxfwwrgsvslmljclp4cf20y4ykm07fl"))

(define rust-pyo3-0.16.6
  (crate-source "pyo3" "0.16.6"
                "1jn6nhp9a8i5kfwds6fzy3hzb938qim8bajp8gfkkcn9892c8802"))

(define rust-pyo3-0.19.2
  (crate-source "pyo3" "0.19.2"
                "0f1aqx947mzsrk05zdm9w2702lj5k9s97y9w9lxwkp2avk7sd0g6"))

(define rust-pyo3-0.20.3
  (crate-source "pyo3" "0.20.3"
                "0cw2pgab0pq5gd98nhv18xgxvyigygspla6c8mfycmwisjbbpgak"))

(define rust-pyo3-0.21.2
  (crate-source "pyo3" "0.21.2"
                "1n39y183jfqbyvimscqr0ysxxa6804pn46ivw048ww91lnb0pq55"))

(define rust-pyo3-0.22.6
  (crate-source "pyo3" "0.22.6"
                "110qrq9yibfv40zzind9p2i87617lhzs379ix0m2065b2qk0c0pl"))

(define rust-pyo3-0.23.5
  (crate-source "pyo3" "0.23.5"
                "0wm8z6jgg18z2cgr99wc34mbkffhcnb50igmq5d1ff6ghpyvyy3p"
                #:snippet '(delete-file-recursively "branding")))

(define rust-pyo3-0.24.1
  (crate-source "pyo3" "0.24.1"
                "0abjav5868avfgb8fjfvz2dbpm916zmhm8xvxzfgns5hhq033nhp"
                #:snippet '(delete-file-recursively "branding")))

(define rust-pyo3-build-config-0.15.2
  (crate-source "pyo3-build-config" "0.15.2"
                "0414biclhrxv6y0bjm6s9fq9z6yah393ffkd8748pqdq83y3k4kp"))

(define rust-pyo3-build-config-0.16.6
  (crate-source "pyo3-build-config" "0.16.6"
                "1hndr61lxc2kw7wnnggps3h08lnsqapwa3lxc0l5ji2rg0wrv0cw"))

(define rust-pyo3-build-config-0.19.2
  (crate-source "pyo3-build-config" "0.19.2"
                "19bb7aqyvr4kmh8b2lnrmcv9251j8yxw7l7xyr77m3s3pk876v07"))

(define rust-pyo3-build-config-0.20.3
  (crate-source "pyo3-build-config" "0.20.3"
                "1ms83n1qa81989c6pakpznifalvxv5fiyyji23732lizvr2mgany"))

(define rust-pyo3-build-config-0.21.2
  (crate-source "pyo3-build-config" "0.21.2"
                "0l0vxvnqsbjrpdmysnwaljf4q3zcr1kb526qq23svzgs6mcdz0vq"))

(define rust-pyo3-build-config-0.22.6
  (crate-source "pyo3-build-config" "0.22.6"
                "0f4w8waba9cyzllq0dpxpw7qmgic05wdf4k20p8nsi7znmsmfjxi"))

(define rust-pyo3-build-config-0.23.5
  (crate-source "pyo3-build-config" "0.23.5"
                "1yqhw1k466k65rqvy2d4xz2shl0hzkry1xlxinciigzkdvlcpxll"))

(define rust-pyo3-build-config-0.24.1
  (crate-source "pyo3-build-config" "0.24.1"
                "1hazmb2hm7qad5xcy9gajwj3231hqjnndfcb145014ypkf46awg2"))

(define rust-pyo3-ffi-0.16.6
  (crate-source "pyo3-ffi" "0.16.6"
                "0pzpcmndak7y72p4rw494cki1bxh86diiqdzsw14gxammc1jg26a"))

(define rust-pyo3-ffi-0.19.2
  (crate-source "pyo3-ffi" "0.19.2"
                "1nc4696k03ydyiy1f69l3ywknavjzxzag2mscrh2bgkywx1fwg75"))

(define rust-pyo3-ffi-0.20.3
  (crate-source "pyo3-ffi" "0.20.3"
                "1yja1npmzh4i73jn5dv2rnw7idif8bns51bf3zpx821ys0qjbd32"))

(define rust-pyo3-ffi-0.21.2
  (crate-source "pyo3-ffi" "0.21.2"
                "00vlp4gmzn76gz250pz1r3fydd5r41nwl7dd9nmidfb0vi1migh1"))

(define rust-pyo3-ffi-0.22.6
  (crate-source "pyo3-ffi" "0.22.6"
                "0dl6zj806rkvs67q2mdgjbnzjhzm8glms46nqx8bpp1c9bqbrdcs"))

(define rust-pyo3-ffi-0.23.5
  (crate-source "pyo3-ffi" "0.23.5"
                "13fxvxijl59vilv39akdzwqd1l7fb6c70f53n27irfy0672b9wg9"))

(define rust-pyo3-ffi-0.24.1
  (crate-source "pyo3-ffi" "0.24.1"
                "1g5a5w7rsq2afb5zamz1jjx5834cnwlg4gig0qhb9xnvw4k0aa05"))

(define rust-pyo3-macros-0.15.2
  (crate-source "pyo3-macros" "0.15.2"
                "0fmrzl185i00c6kdvy7icmhhc99c51pyha46incqggk4qvl4gch0"))

(define rust-pyo3-macros-0.16.6
  (crate-source "pyo3-macros" "0.16.6"
                "089z1x8fm7078kyhpjcsaipd6745jf2rp86rpbkppbam5504k1sn"))

(define rust-pyo3-macros-0.19.2
  (crate-source "pyo3-macros" "0.19.2"
                "1lggr5pnpmdj0cznlhw7ykm1qka3wlymwzfxqql6a4vyb6clrsyz"))

(define rust-pyo3-macros-0.20.3
  (crate-source "pyo3-macros" "0.20.3"
                "0n61s98qb2qc1wlda3bz4r0wi0vsr9p4lj2yr5g0bf01z8hcf1bk"))

(define rust-pyo3-macros-0.21.2
  (crate-source "pyo3-macros" "0.21.2"
                "0g65z9yj2iffjrkrkzan9hwhhj7rrchh7lfv64dy30h6zill1cvp"))

(define rust-pyo3-macros-0.22.6
  (crate-source "pyo3-macros" "0.22.6"
                "0lylczfabgylnfldns6m36vsw98m9sini0wn1gcfda83g64lvlhg"))

(define rust-pyo3-macros-0.23.5
  (crate-source "pyo3-macros" "0.23.5"
                "1nm9i19aff7zn245v35qb0lbr3cxr19zdgcayq84fg7n509j1hpv"))

(define rust-pyo3-macros-0.24.1
  (crate-source "pyo3-macros" "0.24.1"
                "0i2zrpazxwgfbpmzvrcqhdgg3j97l5i0q455b4rzdlx4d9lfag2w"))

(define rust-pyo3-macros-backend-0.15.2
  (crate-source "pyo3-macros-backend" "0.15.2"
                "15bhc1xib9yz4l1sd2lk3nc7scbqsjfvgvlr3mj0xq0jqh92i32s"))

(define rust-pyo3-macros-backend-0.16.6
  (crate-source "pyo3-macros-backend" "0.16.6"
                "1f0y9vxmyq1cidk544pwg3my30f6xfqfgf42grw4gx4q5pl687v1"))

(define rust-pyo3-macros-backend-0.19.2
  (crate-source "pyo3-macros-backend" "0.19.2"
                "0dlm4pg29hjmlqx15gcy9cmnabvc8ycy60hcvjg8hm62flhw2zcl"))

(define rust-pyo3-macros-backend-0.20.3
  (crate-source "pyo3-macros-backend" "0.20.3"
                "11b1z7qnbdnd9hy74ds3xcjx3mjkz43mvpnan32ljccwpdl9nzkw"))

(define rust-pyo3-macros-backend-0.21.2
  (crate-source "pyo3-macros-backend" "0.21.2"
                "0p58yp8ajlc8bq56wghw1syrjszmadasasdfpsjy3d9dychhf9h8"))

(define rust-pyo3-macros-backend-0.22.6
  (crate-source "pyo3-macros-backend" "0.22.6"
                "1gmz3i0sr4fdg7qjvd8ylbkrgbbch9wv955kni903rd17fh13h1n"))

(define rust-pyo3-macros-backend-0.23.5
  (crate-source "pyo3-macros-backend" "0.23.5"
                "0a10yxj41kvjhh9vywzd2zj3h6iwm4bg3mlkw2frrnpks1m759pw"))

(define rust-pyo3-macros-backend-0.24.1
  (crate-source "pyo3-macros-backend" "0.24.1"
                "0m9q9zlwn5s0pa2819z3a13yn94pmq1q3swvx04fvc5z1jm6zkzl"))

(define rust-pyproject-toml-0.11.0
  (crate-source "pyproject-toml" "0.11.0"
                "0sm3ncm57hgcyladl55w59ycl39vq4crigjb9bya0n6b7c162w7g"))

(define rust-python-pkginfo-0.6.5
  (crate-source "python-pkginfo" "0.6.5"
                "006n1myk8lx98g9w4d250lmnmahs0ns9lqrgkg992pn41y45h7y2"
                #:snippet '(delete-file-recursively "tests")))

(define rust-python3-dll-a-0.2.13
  (crate-source "python3-dll-a" "0.2.13"
                "1vfl9qpx1mj4fdzijkx1yb8z91gilc725l1ajj9r9kw8l8kl5zj9"))

(define rust-qcheck-1.0.0
  (crate-source "qcheck" "1.0.0"
                "1w7knnk2zhhczmymh6hdddsld4vlvmm6lpn930nxclfs891bsfdl"))

(define rust-qcheck-macros-1.0.0
  (crate-source "qcheck-macros" "1.0.0"
                "1ddiynmcs0i49gxx6vky3jnsg7nvyj1ikw8q9f81a1k9bapvjhi7"))

(define rust-qcms-0.3.0
  (crate-source "qcms" "0.3.0"
                "1yihv9rsa0qc4mmhzp8f0xdfrnkw7q8l7kr4ivcyb9amszazrv7d"))

(define rust-qoi-0.4.1
  (crate-source "qoi" "0.4.1"
                "00c0wkb112annn2wl72ixyd78mf56p4lxkhlmsggx65l3v3n8vbz"
                #:snippet '(delete-file-recursively "doc")))

(define rust-qrcode-0.14.1
  (crate-source "qrcode" "0.14.1"
                "1v693x68yg90wfpas5v4bf6cfmnq9dq54qfgd3kb33j07r3851yn"
                #:snippet '(for-each delete-file (find-files "src" "^test_annex_i_.*"))))

(define rust-qrencode-0.14.0
  (crate-source "qrencode" "0.14.0"
                "1hgp0lchnp3zx79j3799nm445rvqg7x62x2x7926ky22lqhv23d6"
                #:snippet '(for-each delete-file (find-files "src" "^test_annex_i_.*"))))

(define rust-quick-error-1.2.3
  (crate-source "quick-error" "1.2.3"
                "1q6za3v78hsspisc197bg3g7rpc989qycy8ypr8ap8igv10ikl51"))

(define rust-quick-error-2.0.1
  (crate-source "quick-error" "2.0.1"
                "18z6r2rcjvvf8cn92xjhm2qc3jpd1ljvcbf12zv0k9p565gmb4x9"))

(define rust-quick-junit-0.5.1
  (crate-source "quick-junit" "0.5.1"
                "1mxczpzhcnj3gkd22q448339lln6i1md0fhhaxr325hs769sdl9y"))

(define rust-quick-xml-0.17.2
  (crate-source "quick-xml" "0.17.2"
                "1c5vqqbxr4bx7mw6qyv1yhm6g95ikd2rqfq54pf9y37kvh5l67py"))

(define rust-quick-xml-0.26.0
  (crate-source "quick-xml" "0.26.0"
                "1kckgj8qscpi23y62zrfmni73k6h78nvhs3z9myiwq9q7g3b2l3z"))

(define rust-quick-xml-0.28.2
  (crate-source "quick-xml" "0.28.2"
                "1lfr3512x0s0i9kbyglyzn0rq0i1bvd2mqqfi8gs685808rfgr8c"))

(define rust-quick-xml-0.30.0
  (crate-source "quick-xml" "0.30.0"
                "0mp9cqy06blsaka3r1n2p40ddmzhsf7bx37x22r5faw6hq753xpg"))

(define rust-quick-xml-0.31.0
  (crate-source "quick-xml" "0.31.0"
                "0cravqanylzh5cq2v6hzlfqgxcid5nrp2snnb3pf4m0and2a610h"))

(define rust-quick-xml-0.32.0
  (crate-source "quick-xml" "0.32.0"
                "1hk9x4fij5kq1mnn7gmxz1hpv8s9wnnj4gx4ly7hw3mn71c6wfhx"))

(define rust-quick-xml-0.36.2
  (crate-source "quick-xml" "0.36.2"
                "1zj3sjcjk6sn544wb2wvhr1km5f9cy664vzclygfsnph9mxrlr7p"))

(define rust-quick-xml-0.37.4
  (crate-source "quick-xml" "0.37.4"
                "0s8krrf4ci10kcxfzdja7h7dz5kcp1mgndhgf0wghkrjvs48rkm4"))

(define rust-quick-xml-0.37.5
  (crate-source "quick-xml" "0.37.5"
                "1yxpd7rc2qn6f4agfj47ps2z89vv7lvzxpzawqirix8bmyhrf7ik"))

(define rust-quick-xml-0.38.0
  (crate-source "quick-xml" "0.38.0"
                "06vvgd9arm1nrsd4d0ii6lhnp6m11bwy7drqa4k9hnjw9xkb09w9"))

(define rust-quick-xml-0.38.3
  (crate-source "quick-xml" "0.38.3"
                "12bvsbnnmlnq9xg9in3h3080ag3sisafgpcn7lqyzhkz93kk58j2"))

(define rust-quickcheck-0.6.2
  (crate-source "quickcheck" "0.6.2"
                "1dyazm2fcq0v9fscq1a7597zsvdl9f0j8c2bfj1jm2nlzz2sn6y0"))

(define rust-quickcheck-1.0.3
  (crate-source "quickcheck" "1.0.3"
                "1mjhkfqwrb8mdyxdqr4zzbj1rm5dfx25n9zcc25lb6fxwiw673sq"))

(define rust-quickxml-to-serde-0.6.0
  (crate-source "quickxml_to_serde" "0.6.0"
                "0zj07hy3av98lxy0czbp5vh3n2lfc86y262jmkv6yd8bl33hasr8"))

(define rust-quinn-0.11.7
  (crate-source "quinn" "0.11.7"
                "04ihd2jibw0carrx081pwdkh8n0l03n9zjvxi21yyylnyak1bgf3"))

(define rust-quinn-0.11.8
  (crate-source "quinn" "0.11.8"
                "1j02h87nfxww5mjcw4vjcnx8b70q0yinnc8xvjv82ryskii18qk2"))

(define rust-quinn-0.11.9
  (crate-source "quinn" "0.11.9"
                "086gzj666dr3slmlynkvxlndy28hahgl361d6bf93hk3i6ahmqmr"))

(define rust-quinn-proto-0.11.10
  (crate-source "quinn-proto" "0.11.10"
                "1k12m8y3k8dszv9ysb3hxp92cnhva6f670w176img6ywni77885q"))

(define rust-quinn-proto-0.11.12
  (crate-source "quinn-proto" "0.11.12"
                "0bj2yyrf1mrg2bcj19ipsspvrj5sq0di0pz5maw5pj31j4x89ps9"))

(define rust-quinn-proto-0.11.13
  (crate-source "quinn-proto" "0.11.13"
                "0cca3mgja9p4w66f6sl1kfhj8rdf4mwsg1jxzssh9g63n14np47i"))

(define rust-quinn-udp-0.5.11
  (crate-source "quinn-udp" "0.5.11"
                "1daa7c1hws33395zzw62i879dxr168fp8llaff87lx7cqrbhy7al"))

(define rust-quinn-udp-0.5.12
  (crate-source "quinn-udp" "0.5.12"
                "0hm89bv7mm383lv2c8z5r9512pdgp1q26lsmazicajgrj6cm4kpf"))

(define rust-quinn-udp-0.5.13
  (crate-source "quinn-udp" "0.5.13"
                "0w0ri3wv5g419i5dfv4qmjxh4ayc4hp77y2gy4p3axp2kqhb3szw"))

(define rust-quinn-udp-0.5.14
  (crate-source "quinn-udp" "0.5.14"
                "1gacawr17a2zkyri0r3m0lc9spzmxbq1by3ilyb8v2mdvjhcdpmd"))

(define rust-quitters-0.1.0
  (crate-source "quitters" "0.1.0"
                "055hdmm9b78jpdirsbfl7bi2zi1zjwiskjxma2r1a8adv3kwv378"))

(define rust-quote-0.6.13
  (crate-source "quote" "0.6.13"
                "1qgqq48jymp5h4y082aanf25hrw6bpb678xh3zw993qfhxmkpqkc"))

(define rust-quote-1.0.10
  (crate-source "quote" "1.0.10"
                "01ff7a76f871ggnby57iagw6499vci4bihcr11g6bqzjlp38rg1q"))

(define rust-quote-1.0.32
  (crate-source "quote" "1.0.32"
                "0rarx33n4sp7ihsiasrjip5qxh01f5sn80daxc6m885pryfb7wsh"))

(define rust-quote-1.0.33
  (crate-source "quote" "1.0.33"
                "1biw54hbbr12wdwjac55z1m2x2rylciw83qnjn564a3096jgqrsj"))

(define rust-quote-1.0.37
  (crate-source "quote" "1.0.37"
                "1brklraw2g34bxy9y4q1nbrccn7bv36ylihv12c9vlcii55x7fdm"))

(define rust-quote-1.0.38
  (crate-source "quote" "1.0.38"
                "1k0s75w61k6ch0rs263r4j69b7vj1wadqgb9dia4ylc9mymcqk8f"))

(define rust-quote-1.0.40
  (crate-source "quote" "1.0.40"
                "1394cxjg6nwld82pzp2d4fp6pmaz32gai1zh9z5hvh0dawww118q"))

(define rust-quote-1.0.41
  (crate-source "quote" "1.0.41"
                "1lg108nb57lwbqlnpsii89cchk6i8pkcvrv88xh1p7a9gdz7c9ff"))

(define rust-quoted-printable-0.5.1
  (crate-source "quoted_printable" "0.5.1"
                "0wvwq6w6rdsx1yxzr7ckspff0qk0q9252dzmxrd4c0kv97c9n334"))

(define rust-r-efi-5.2.0
  (crate-source "r-efi" "5.2.0"
                "1ig93jvpqyi87nc5kb6dri49p56q7r7qxrn8kfizmqkfj5nmyxkl"))

(define rust-r-efi-5.3.0
  (crate-source "r-efi" "5.3.0"
                "03sbfm3g7myvzyylff6qaxk4z6fy76yv860yy66jiswc2m6b7kb9"))

(define rust-r2d2-0.8.10
  (crate-source "r2d2" "0.8.10"
                "14qw32y4m564xb1f5ya8ii7dwqyknvk8bsx2r0lljlmn7zxqbpji"))

(define rust-rabbitmq-http-client-0.35.0
  (crate-source "rabbitmq_http_client" "0.35.0"
                "0g5xipi5chwd0z4zzlryz6513czmq9pzy9l9npzfm7xbj00fzwlb"))

(define rust-rabbitmq-http-client-0.44.0
  (crate-source "rabbitmq_http_client" "0.44.0"
                "0p5h4shaii9xmg934c6x93b2q0flhs6v4faysaji9mllm5ab56f4"))

(define rust-radicle-git-ext-0.8.1
  (crate-source "radicle-git-ext" "0.8.1"
                "0l363i5gwnd3pl0f415cf89lc2b64cxsw0n6l3ajlwficxpc4y2b"))

(define rust-radicle-std-ext-0.1.1
  (crate-source "radicle-std-ext" "0.1.1"
                "0am3pnfzhiq8s8yd3p536w4nh72cs9mv4iwwvj9cxdh68nhff42k"))

(define rust-radicle-surf-0.22.1
  (crate-source "radicle-surf" "0.22.1"
                "01dqxrrd5hsnyaj9f148vl94d3fgbkf6v0pp1chfgs10fpsszxhb"))

(define rust-radium-0.7.0
  (crate-source "radium" "0.7.0"
                "02cxfi3ky3c4yhyqx9axqwhyaca804ws46nn4gc1imbk94nzycyw"))

(define rust-radix-fmt-1.0.0
  (crate-source "radix_fmt" "1.0.0"
                "09jlq152iwn56215kghqby4pi8vamhg0nzcb9any5b5782cjl26f"))

(define rust-radix-heap-0.4.2
  (crate-source "radix-heap" "0.4.2"
                "00g6vq9dn5v5h8sj2298h6zk3pm3wq8qhacrnjak40b4yjfyrzsr"))

(define rust-radix-trie-0.2.1
  (crate-source "radix_trie" "0.2.1"
                "1zaq3im5ss03w91ij11cj97vvzc5y1f3064d9pi2ysnwziww2sf0"))

(define rust-railroad-0.2.0
  (crate-source "railroad" "0.2.0"
                "08y86hb3vp0i9k9dkz3lbfhrd8x051f6pcg3p87z65ss3shdmzh1"))

(define rust-rancor-0.1.0
  (crate-source "rancor" "0.1.0"
                "0iyr19x1aryadcyc2zwjbwmskkkjqfbvrjp4l37d3f9434bggxfa"))

(define rust-rand-0.3.23
  (crate-source "rand" "0.3.23"
                "0v679h38pjjqj5h4md7v2slsvj6686qgcn7p9fbw3h43iwnk1b34"))

(define rust-rand-0.4.6
  (crate-source "rand" "0.4.6"
                "14qjfv3gggzhnma20k0sc1jf8y6pplsaq7n1j9ls5c8kf2wl0a2m"))

(define rust-rand-0.6.5
  (crate-source "rand" "0.6.5"
                "1jl4449jcl4wgmzld6ffwqj5gwxrp8zvx8w573g1z368qg6xlwbd"))

(define rust-rand-0.7.3
  (crate-source "rand" "0.7.3"
                "00sdaimkbz491qgi6qxkv582yivl32m2jd401kzbn94vsiwicsva"))

(define rust-rand-0.8.5
  (crate-source "rand" "0.8.5"
                "013l6931nn7gkc23jz5mm3qdhf93jjf0fg64nz2lp4i51qd8vbrl"))

(define rust-rand-0.9.0
  (crate-source "rand" "0.9.0"
                "156dyvsfa6fjnv6nx5vzczay1scy5183dvjchd7bvs47xd5bjy9p"))

(define rust-rand-0.9.1
  (crate-source "rand" "0.9.1"
                "15yxfcxbgmwba5cv7mjg9bhc1r5c9483dfcdfspg62x4jk8dkgwz"))

(define rust-rand-0.9.2
  (crate-source "rand" "0.9.2"
                "1lah73ainvrgl7brcxx0pwhpnqa3sm3qaj672034jz8i0q7pgckd"))

(define rust-rand-chacha-0.1.1
  (crate-source "rand_chacha" "0.1.1"
                "1vxwyzs4fy1ffjc8l00fsyygpiss135irjf7nyxgq2v0lqf3lvam"))

(define rust-rand-chacha-0.2.2
  (crate-source "rand_chacha" "0.2.2"
                "00il36fkdbsmpr99p9ksmmp6dn1md7rmnwmz0rr77jbrca2yvj7l"))

(define rust-rand-chacha-0.3.1
  (crate-source "rand_chacha" "0.3.1"
                "123x2adin558xbhvqb8w4f6syjsdkmqff8cxwhmjacpsl1ihmhg6"))

(define rust-rand-chacha-0.9.0
  (crate-source "rand_chacha" "0.9.0"
                "1jr5ygix7r60pz0s1cv3ms1f6pd1i9pcdmnxzzhjc3zn3mgjn0nk"))

(define rust-rand-core-0.3.1
  (crate-source "rand_core" "0.3.1"
                "0jzdgszfa4bliigiy4hi66k7fs3gfwi2qxn8vik84ph77fwdwvvs"))

(define rust-rand-core-0.4.2
  (crate-source "rand_core" "0.4.2"
                "1p09ynysrq1vcdlmcqnapq4qakl2yd1ng3kxh3qscpx09k2a6cww"))

(define rust-rand-core-0.5.1
  (crate-source "rand_core" "0.5.1"
                "06bdvx08v3rkz451cm7z59xwwqn1rkfh6v9ay77b14f8dwlybgch"))

(define rust-rand-core-0.6.4
  (crate-source "rand_core" "0.6.4"
                "0b4j2v4cb5krak1pv6kakv4sz6xcwbrmy2zckc32hsigbrwy82zc"))

(define rust-rand-core-0.9.3
  (crate-source "rand_core" "0.9.3"
                "0f3xhf16yks5ic6kmgxcpv1ngdhp48mmfy4ag82i1wnwh8ws3ncr"))

(define rust-rand-distr-0.2.2
  (crate-source "rand_distr" "0.2.2"
                "1cpz577qid09lirjjhhn98yqdwsv0c01jf973pxpcr9svp5pm5wn"))

(define rust-rand-distr-0.4.3
  (crate-source "rand_distr" "0.4.3"
                "0cgfwg3z0pkqhrl0x90c77kx70r6g9z4m6fxq9v0h2ibr2dhpjrj"))

(define rust-rand-hc-0.1.0
  (crate-source "rand_hc" "0.1.0"
                "1i0vl8q5ddvvy0x8hf1zxny393miyzxkwqnw31ifg6p0gdy6fh3v"))

(define rust-rand-hc-0.2.0
  (crate-source "rand_hc" "0.2.0"
                "0g31sqwpmsirdlwr0svnacr4dbqyz339im4ssl9738cjgfpjjcfa"))

(define rust-rand-isaac-0.1.1
  (crate-source "rand_isaac" "0.1.1"
                "027flpjr4znx2csxk7gxb7vrf9c7y5mydmvg5az2afgisp4rgnfy"))

(define rust-rand-jitter-0.1.4
  (crate-source "rand_jitter" "0.1.4"
                "16z387y46bfz3csc42zxbjq89vcr1axqacncvv8qhyy93p4xarhi"))

(define rust-rand-os-0.1.3
  (crate-source "rand_os" "0.1.3"
                "0wahppm0s64gkr2vmhcgwc0lij37in1lgfxg5rbgqlz0l5vgcxbv"))

(define rust-rand-pcg-0.1.2
  (crate-source "rand_pcg" "0.1.2"
                "0i0bdla18a8x4jn1w0fxsbs3jg7ajllz6azmch1zw33r06dv1ydb"))

(define rust-rand-pcg-0.2.1
  (crate-source "rand_pcg" "0.2.1"
                "0ab4h6s6x3py833jk61lwadq83qd1c8bih2hgi6yps9rnv0x1aqn"))

(define rust-rand-pcg-0.3.1
  (crate-source "rand_pcg" "0.3.1"
                "0gn79wzs5b19iivybwa09wv4lhi4kbcqciasiqqynggnr8cd1jjr"))

(define rust-rand-pcg-0.9.0
  (crate-source "rand_pcg" "0.9.0"
                "1xr04g5zrzqi9n7vyzjznnyrmy55i8k34ripsb2gmdxazzvw72ml"))

(define rust-rand-regex-0.18.1
  (crate-source "rand_regex" "0.18.1"
                "0dyzikkp5bjfp8sv06ylbkf7kfyd0vir03j088g9lrqk5lw2xnq4"))

(define rust-rand-seeder-0.4.0
  (crate-source "rand_seeder" "0.4.0"
                "07l4fsxizhh3fxpvkvpapwamlfvqacnvp58c7valb5iwzkyjfaah"))

(define rust-rand-xorshift-0.1.1
  (crate-source "rand_xorshift" "0.1.1"
                "0p2x8nr00hricpi2m6ca5vysiha7ybnghz79yqhhx6sl4gkfkxyb"))

(define rust-rand-xorshift-0.3.0
  (crate-source "rand_xorshift" "0.3.0"
                "13vcag7gmqspzyabfl1gr9ykvxd2142q2agrj8dkyjmfqmgg4nyj"))

(define rust-rand-xorshift-0.4.0
  (crate-source "rand_xorshift" "0.4.0"
                "0njsn25pis742gb6b89cpq7jp48v9n23a9fvks10yczwks8n4fai"))

(define rust-rand-xoshiro-0.6.0
  (crate-source "rand_xoshiro" "0.6.0"
                "1ajsic84rzwz5qr0mzlay8vi17swqi684bqvwqyiim3flfrcv5vg"))

(define rust-ratatui-0.26.3
  (crate-source "ratatui" "0.26.3"
                "1vqyk46r6mbsv6qqvjf8rmc7lr8b0khqbfvgcifa3va6zml9wk7l"))

(define rust-ratatui-0.27.0
  (crate-source "ratatui" "0.27.0"
                "1lv8r99miibk6np2d2m0y6vf62jf5dr1x272ws6byalnnp2lcrfi"))

(define rust-ratatui-0.28.1
  (crate-source "ratatui" "0.28.1"
                "0zd5pgx1x2bda83mjw0z2skad6rlckcz9gaq1n4jy4n0wndpzvzx"))

(define rust-ratatui-0.29.0
  (crate-source "ratatui" "0.29.0"
                "0yqiccg1wmqqxpb2sz3q2v3nifmhsrfdsjgwhc2w40bqyg199gga"))

(define rust-raunch-1.0.1
  (crate-source "raunch" "1.0.1"
                "1ia9mfxpsvpdr0dzcr1kb6q911nmyw26x0jwbz2ikm6ci5kj4rqs"))

(define rust-rav1e-0.7.1
  (crate-source "rav1e" "0.7.1"
                "1sawva6nmj2fvynydbcirr3nb7wjyg0id2hz2771qnv6ly0cx1yd"))

(define rust-ravif-0.11.11
  (crate-source "ravif" "0.11.11"
                "1ij51acd3pkl3rr2ha3r3nc7pvg649m49bvyngpcv98fpnbgs4r4"))

(define rust-ravif-0.11.12
  (crate-source "ravif" "0.11.12"
                "1in0111cgrn06bkwxkr24fgwj9ammfj8x1ga3x0gj03mrwgz79fn"))

(define rust-ravif-0.11.20
  (crate-source "ravif" "0.11.20"
                "0sryi4qzwv3rjhy4h0z9fb0zw3k3i814j3ck2psvjsnivmpw49aq"))

(define rust-raw-window-handle-0.6.2
  (crate-source "raw-window-handle" "0.6.2"
                "0ff5c648hncwx7hm2a8fqgqlbvbl4xawb6v3xxv9wkpjyrr5arr0"))

(define rust-rawpointer-0.2.1
  (crate-source "rawpointer" "0.2.1"
                "1qy1qvj17yh957vhffnq6agq0brvylw27xgks171qrah75wmg8v0"))

(define rust-rayon-1.10.0
  (crate-source "rayon" "1.10.0"
                "1ylgnzwgllajalr4v00y4kj22klq2jbwllm70aha232iah0sc65l"))

(define rust-rayon-1.11.0
  (crate-source "rayon" "1.11.0"
                "13x5fxb7rn4j2yw0cr26n7782jkc7rjzmdkg42qxk3xz0p8033rn"))

(define rust-rayon-1.7.0
  (crate-source "rayon" "1.7.0"
                "0fzh8w5ds1qjhilll4rkpd3kimw70zi5605wprxcig1pdqczab8x"))

(define rust-rayon-cond-0.3.0
  (crate-source "rayon-cond" "0.3.0"
                "1ybxppq84p3q60h9rng9j3dm79f6970hn4wljyf31lpgan5m77q5"))

(define rust-rayon-core-1.11.0
  (crate-source "rayon-core" "1.11.0"
                "13dymrhhdilzpbfh3aylv6ariayqdfk614b3frvwixb6d6yrb3sb"))

(define rust-rayon-core-1.12.1
  (crate-source "rayon-core" "1.12.1"
                "1qpwim68ai5h0j7axa8ai8z0payaawv3id0lrgkqmapx7lx8fr8l"))

(define rust-rayon-core-1.13.0
  (crate-source "rayon-core" "1.13.0"
                "14dbr0sq83a6lf1rfjq5xdpk5r6zgzvmzs5j6110vlv2007qpq92"))

(define rust-rbase64-2.0.3
  (crate-source "rbase64" "2.0.3"
                "04wb2fprnv5vdj7y00grc9j0avpa6j53rnc6kihvgjx7abfky4sv"
                #:snippet '(delete-file-recursively "tests")))

(define rust-rcgen-0.13.2
  (crate-source "rcgen" "0.13.2"
                "18l0rz228pvnc44bjmvq8cchhh5d2rrkk98y9lqvan9243jnkrkm"))

(define rust-rctree-0.6.0
  (crate-source "rctree" "0.6.0"
                "1sd6vsa5p3j27v6f1v0l0afl3hn4an1jr3psky3024gcmdk7hgp0"))

(define rust-rdrand-0.4.0
  (crate-source "rdrand" "0.4.0"
                "1cjq0kwx1bk7jx3kzyciiish5gqsj7620dm43dc52sr8fzmm9037"))

(define rust-read-process-memory-0.1.6
  (crate-source "read-process-memory" "0.1.6"
                "1c1vmw8a2ywf7v96xfwbsyrnpk3fy8cc2a0rdxw8fs0b5wxni5w4"))

(define rust-realpath-ext-0.1.3
  (crate-source "realpath-ext" "0.1.3"
                "0bq9pmwag74q7k7d496zmsg6brqp1yqmz5pll122nlqd5a374bv9"))

(define rust-recursion-0.5.2
  (crate-source "recursion" "0.5.2"
                "0wc8x5340scj9fl281nydgwrpzbbdffj764pw6zppkcchlk58w4z"
                #:snippet '(delete-file-recursively "img_assets")))

(define rust-recvmsg-1.0.0
  (crate-source "recvmsg" "1.0.0"
                "0xa173gbg1cx8q7wyzi6c4kmcsz5rka68r4jb6kg14icskax9vfk"))

(define rust-redox-syscall-0.1.57
  (crate-source "redox_syscall" "0.1.57"
                "1kh59fpwy33w9nwd5iyc283yglq8pf2s41hnhvl48iax9mz0zk21"))

(define rust-redox-syscall-0.2.16
  (crate-source "redox_syscall" "0.2.16"
                "16jicm96kjyzm802cxdd1k9jmcph0db1a4lhslcnhjsvhp0mhnpv"))

(define rust-redox-syscall-0.4.1
  (crate-source "redox_syscall" "0.4.1"
                "1aiifyz5dnybfvkk4cdab9p2kmphag1yad6iknc7aszlxxldf8j7"))

(define rust-redox-syscall-0.5.10
  (crate-source "redox_syscall" "0.5.10"
                "1l9b638qx72312yzh8ykvda9b3lqd9gf6yqn66b23a331ck0r30b"))

(define rust-redox-syscall-0.5.11
  (crate-source "redox_syscall" "0.5.11"
                "18qijn18r10haiglv4261wb0yh1agqqlvs0nxfy8yjbpsb307wfj"))

(define rust-redox-syscall-0.5.12
  (crate-source "redox_syscall" "0.5.12"
                "1by5k76jr4kjy37287ifn56dzw6jh6nrwnrjm29j615ayafcm3wj"))

(define rust-redox-syscall-0.5.13
  (crate-source "redox_syscall" "0.5.13"
                "1mlzna9bcd7ss1973bmysr3hpjrys82b3bd7l03h4jkbxv8bf10d"))

(define rust-redox-syscall-0.5.17
  (crate-source "redox_syscall" "0.5.17"
                "0xrvpchkaxph3r5ww2i04v9nwg3843fp3prf8kqlh1gv01b4c1sl"))

(define rust-redox-syscall-0.5.3
  (crate-source "redox_syscall" "0.5.3"
                "1916m7abg9649gkif055pn5nsvqjhp70isy0v7gx1zgi01p8m41a"))

(define rust-redox-syscall-0.5.9
  (crate-source "redox_syscall" "0.5.9"
                "0bvykdad226m3nqzkbb95piglyfn7m2yxp4r10m9xr4q7qr6idc2"))

(define rust-redox-termios-0.1.3
  (crate-source "redox_termios" "0.1.3"
                "1jzifsj7fqyksz4325l3azfzpyv027kjabf93zcmass3p9q5c510"))

(define rust-redox-users-0.4.3
  (crate-source "redox_users" "0.4.3"
                "0asw3s4iy69knafkhvlbchy230qawc297vddjdwjs5nglwvxhcxh"))

(define rust-redox-users-0.4.6
  (crate-source "redox_users" "0.4.6"
                "0hya2cxx6hxmjfxzv9n8rjl5igpychav7zfi1f81pz6i4krry05s"))

(define rust-redox-users-0.5.0
  (crate-source "redox_users" "0.5.0"
                "0awxx66izdw6kz97r3zxrl5ms5f6dqi5l0f58mlsvlmx8wyrsvyx"))

(define rust-redox-users-0.5.2
  (crate-source "redox_users" "0.5.2"
                "1b17q7gf7w8b1vvl53bxna24xl983yn7bd00gfbii74bcg30irm4"))

(define rust-reedline-0.39.0
  (crate-source "reedline" "0.39.0"
                "0jn6vq2x0najy4yvfa1dgd3g1cv4mb8p0i769qv3manjf7p2hiyx"))

(define rust-ref-cast-1.0.24
  (crate-source "ref-cast" "1.0.24"
                "1kx57g118vs9sqi6d2dcxy6vp8jbx8n5hilmv1sacip9vc8y82ja"))

(define rust-ref-cast-impl-1.0.24
  (crate-source "ref-cast-impl" "1.0.24"
                "1ir7dm7hpqqdgg60hlspsc1ck6wli7wa3xcqrsxz7wdz45f24r8i"))

(define rust-referencing-0.30.0
  (crate-source "referencing" "0.30.0"
                "03j27y4zbghw2pi4ccfjc072zi89lzijypswx1bjlp4bfzxg9vy8"))

(define rust-reflink-0.1.3
  (crate-source "reflink" "0.1.3"
                "1glcyqvryv2zj6kjbfji0cldrkincqx3ds3wjwl4qnsnig15wn5w"))

(define rust-reflink-0.1.3.e8d93b4
  ;; TODO: Define standalone package if this is a workspace.
  (origin
    (method git-fetch)
    (uri (git-reference (url "https://github.com/nicokoch/reflink")
                        (commit "e8d93b465f5d9ad340cd052b64bbc77b8ee107e2")))
    (file-name (git-file-name "rust-reflink" "0.1.3.e8d93b4"))
    (sha256 (base32 "1hl5lp5r6r32hjpd0ngqan0hbawb867zyxl6pb5v13d9czz5v3nn"))))

(define rust-reflink-copy-0.1.26
  (crate-source "reflink-copy" "0.1.26"
                "15g89wsra90s4wcp2fvcj1brwfcx04pzklh0rhrl2lic1801vj3q"))

(define rust-regalloc-0.0.34
  (crate-source "regalloc" "0.0.34"
                "00ncljiasvlw66avy9756327pcvp3nph0xw3d3f0p65z7qfnni32"))

(define rust-regalloc2-0.11.2
  (crate-source "regalloc2" "0.11.2"
                "16k30jbh1ag5vb4l6p43jik1dzq8pymjbiwblkj189hl32ryc1nw"))

(define rust-regex-1.11.1
  (crate-source "regex" "1.11.1"
                "148i41mzbx8bmq32hsj1q4karkzzx5m60qza6gdw4pdc9qdyyi5m"
                #:snippet '(delete-file-recursively "tests")))

(define rust-regex-1.11.2
  (crate-source "regex" "1.11.2"
                "04k9rzxd11hcahpyihlswy6f1zqw7lspirv4imm4h0lcdl8gvmr3"
                #:snippet '(delete-file-recursively "tests")))

(define rust-regex-1.11.3
  (crate-source "regex" "1.11.3"
                "0b58ya98c4i5cjjiwhpcnjr61cv9g143qhdwhsryggj09098hllb"
                #:snippet '(delete-file-recursively "tests")))

(define rust-regex-1.5.6
  (crate-source "regex" "1.5.6"
                "1wczbykw6fas7359j9lhkkv13dplhiphzrf2ii6dmg5xjiyi4gyq"))

(define rust-regex-1.9.5
  (crate-source "regex" "1.9.5"
                "0iwcri16a10izcm1jskvl6w24kpy5fpd0rsgc95akd513qi62w39"
                #:snippet '(delete-file-recursively "tests")))

(define rust-regex-automata-0.1.10
  (crate-source "regex-automata" "0.1.10"
                "0ci1hvbzhrfby5fdpf4ganhf7kla58acad9i1ff1p34dzdrhs8vc"
                #:snippet '(delete-file-recursively "tests")))

(define rust-regex-automata-0.3.8
  (crate-source "regex-automata" "0.3.8"
                "1587iyw9x0r33b23vwn4s7cgzavqkkp6dv7qaqxnj82jjps03x62"))

(define rust-regex-automata-0.4.10
  (crate-source "regex-automata" "0.4.10"
                "1mllcfmgjcl6d52d5k09lwwq9wj5mwxccix4bhmw5spy1gx5i53b"
                #:snippet '(delete-file-recursively "tests")))

(define rust-regex-automata-0.4.11
  (crate-source "regex-automata" "0.4.11"
                "1bawj908pxixpggcnma3xazw53mwyz68lv9hn4yg63nlhv7bjgl3"
                #:snippet '(delete-file-recursively "tests")))

(define rust-regex-automata-0.4.9
  (crate-source "regex-automata" "0.4.9"
                "02092l8zfh3vkmk47yjc8d631zhhcd49ck2zr133prvd3z38v7l0"
                #:snippet '(delete-file-recursively "tests")))

(define rust-regex-cursor-0.1.5
  (crate-source "regex-cursor" "0.1.5"
                "07d64dfcg361mn7mxahxajj5hzl7dbp2m4yjhj1ax0prsa0wg5q4"))

(define rust-regex-lite-0.1.6
  (crate-source "regex-lite" "0.1.6"
                "0almvx3z75f611pdcd9mslh7zxg76zh3shifql4ndch6mn3rb92k"
                #:snippet '(delete-file-recursively "tests")))

(define rust-regex-syntax-0.6.29
  (crate-source "regex-syntax" "0.6.29"
                "1qgj49vm6y3zn1hi09x91jvgkl2b1fiaq402skj83280ggfwcqpi"))

(define rust-regex-syntax-0.7.5
  (crate-source "regex-syntax" "0.7.5"
                "1nhjmqdlakfi4yb8lh7vbbh71dsy90jjvrjvvnrih6larldgpdfv"))

(define rust-regex-syntax-0.8.5
  (crate-source "regex-syntax" "0.8.5"
                "0p41p3hj9ww7blnbwbj9h7rwxzxg0c1hvrdycgys8rxyhqqw859b"))

(define rust-regex-syntax-0.8.6
  (crate-source "regex-syntax" "0.8.6"
                "00chjpglclfskmc919fj5aq308ffbrmcn7kzbkz92k231xdsmx6a"))

(define rust-region-3.0.2
  (crate-source "region" "3.0.2"
                "19wrf7fg741jfnyz2314dv9m9hwssh816v27rpwsw2f07g8ypdp6"))

(define rust-relative-path-1.9.3
  (crate-source "relative-path" "1.9.3"
                "1limlh8fzwi21g0473fqzd6fln9iqkwvzp3816bxi31pkilz6fds"))

(define rust-relative-path-2.0.1
  (crate-source "relative-path" "2.0.1"
                "1c4jm6x0p88722c77xx53mb7zcs4xznp9d3whdsbmn1248qhm95w"))

(define rust-remoteprocess-0.5.0
  (crate-source "remoteprocess" "0.5.0"
                "087z016qvypalbqjgdxmri97vsrqg4k1kpmc8b5d5hdgqxq4f6g6"))

(define rust-remove-dir-all-0.5.3
  (crate-source "remove_dir_all" "0.5.3"
                "1rzqbsgkmr053bxxl04vmvsd1njyz0nxvly97aip6aa2cmb15k9s"))

(define rust-remove-dir-all-1.0.0
  (crate-source "remove_dir_all" "1.0.0"
                "026xl6wlkjxksm1n3dcccygssami56aa937h6vgnmxxcfnsc1340"))

(define rust-rend-0.4.2
  (crate-source "rend" "0.4.2"
                "0z4rrkycva0lcw0hxq479h4amxj9syn5vq4vb2qid5v2ylj3izki"))

(define rust-rend-0.5.2
  (crate-source "rend" "0.5.2"
                "05gjxzzsajl61sgif4h0lvagmbry5rm2xak6782j3lccy9mqlpm3"))

(define rust-renderdoc-0.11.0
  (crate-source "renderdoc" "0.11.0"
                "04hycbzwqmzw25qnk0lwps70jgxi43cgmkjdvwbyzc183vnajb97"))

(define rust-renderdoc-sys-1.1.0
  (crate-source "renderdoc-sys" "1.1.0"
                "0cj8zjs7k0gvchcx3jhpg8r9bbqy8b1hsgbz0flcq2ydn12hmcqr"))

(define rust-reqwest-0.11.27
  (crate-source "reqwest" "0.11.27"
                "0qjary4hpplpgdi62d2m0xvbn6lnzckwffm0rgkm2x51023m6ryx"))

(define rust-reqwest-0.12.15
  (crate-source "reqwest" "0.12.15"
                "1fvvrl3jmsnlm99ldl0ariklrlsmrky06qabp7dc92ylznk4d76i"
                #:snippet '(delete-file-recursively "tests")))

(define rust-reqwest-0.12.18
  (crate-source "reqwest" "0.12.18"
                "1dghn902xsww81qd757i64bd5arbr17kshwg64vmlkdyvfqgd3z9"
                #:snippet '(delete-file-recursively "tests")))

(define rust-reqwest-0.12.20
  (crate-source "reqwest" "0.12.20"
                "04qqxghqszjxk4pl4vxa5qlwinkfx0vvjkk10vv2n3hkv6blrgza"
                #:snippet '(delete-file-recursively "tests")))

(define rust-reqwest-0.12.22
  (crate-source "reqwest" "0.12.22"
                "0cbmfrcrk6wbg93apmji0fln1ca9322af2kc7dpa18vcgs9k3jfb"))

(define rust-reqwest-0.12.23
  (crate-source "reqwest" "0.12.23"
                "1svw1k0jx17cmlwhixwqfv3bgpjapciw7klkghnd9cljh16g6afl"))

(define rust-reqwest-middleware-0.4.1
  (crate-source "reqwest-middleware" "0.4.1"
                "1i64p3wqqgj8yrfr97afn5fy161lkgkk046hm91pm6mx2dargs34"))

(define rust-reqwest-retry-0.7.0
  (crate-source "reqwest-retry" "0.7.0"
                "0y0ihmfs4pailrcdwmbcjx30dfbs82rxk45pfjqvrgx6jm0kxir9"))

(define rust-resolv-conf-0.7.1
  (crate-source "resolv-conf" "0.7.1"
                "1i6lwdckplmq5hgp8a0fd2ibmiyqiwagfwa3ssrd8g9yc2a56ds8"))

(define rust-resolv-conf-0.7.4
  (crate-source "resolv-conf" "0.7.4"
                "1qw8kbpn52nbc1b624x53jjnsjifqk0hpqv5fygwicc4qram2clm"))

(define rust-resolve-path-0.1.0
  (crate-source "resolve-path" "0.1.0"
                "1dbvi31ffhwgiskhd2g6qnwb9236rgm9snz7y6vdm4mind0mw7ij"))

(define rust-resvg-0.43.0
  (crate-source "resvg" "0.43.0"
                "1l8inhd3a1353851qkpsl0lbjdrc17fx6fp230ff6z4wqmilacf7"
                #:snippet '(delete-file-recursively "examples")))

(define rust-retain-mut-0.1.9
  (crate-source "retain_mut" "0.1.9"
                "1c0jfxzba0mcw8q4z7dnc4ypwzxgqkdggngb55hazvwvg3az32a3"))

(define rust-retry-policies-0.4.0
  (crate-source "retry-policies" "0.4.0"
                "070alim5nhp3bvf58cyc729kq4dmgmr8rjqf2ny72a5bdhg4fxaq"))

(define rust-rfc2047-decoder-1.0.6
  (crate-source "rfc2047-decoder" "1.0.6"
                "0afyg0k2hagnirvv5rk2ysr3khz8ab5ifdap3dsnli9121fm8dmw"
                #:snippet '(delete-file "rustfmt.toml")))

(define rust-rfc6979-0.4.0
  (crate-source "rfc6979" "0.4.0"
                "1chw95jgcfrysyzsq6a10b1j5qb7bagkx8h0wda4lv25in02mpgq"))

(define rust-rgb-0.8.50
  (crate-source "rgb" "0.8.50"
                "02ii3nsciska0sj23ggxaz8gj64ksw8nbpfjcwxlh037chb7sfap"))

(define rust-rgb-0.8.52
  (crate-source "rgb" "0.8.52"
                "1km115a9lblf9pldvx51dmmg30y8ms4ka67hvas2ndcq556qhshc"))

(define rust-ring-0.17.14 rust-ring-0.17)

(define rust-ripemd-0.1.3
  (crate-source "ripemd" "0.1.3"
                "17xh5yl9wjjj2v18rh3m8ajlmdjg1yj13l6r9rj3mnbss4i444mx"
                #:snippet '(delete-file-recursively "tests")))

(define rust-rkyv-0.7.45
  (crate-source "rkyv" "0.7.45"
                "16vp6m4sq41smhvym8ijy4id1hr3vm4na7wy4bc63qdrhmiws24h"))

(define rust-rkyv-0.8.10
  (crate-source "rkyv" "0.8.10"
                "0rbvkcm1ia5rafajf9hlqcw882slm123jj6vzkif2lsmqxqp650y"))

(define rust-rkyv-derive-0.7.45
  (crate-source "rkyv_derive" "0.7.45"
                "1h1jwmyivx7g88d41gzcjrqnax98m9algjd49hx0laqab4kisgah"))

(define rust-rkyv-derive-0.8.10
  (crate-source "rkyv_derive" "0.8.10"
                "0ymv3al6d3qza3lpqdhp7v2lclkdxzl05f14s5swdxls32n40sr4"))

(define rust-rle-decode-fast-1.0.3
  (crate-source "rle-decode-fast" "1.0.3"
                "08kljzl29rpm12fiz0qj5pask49aiswdvcjigdcq73s224rgd0im"
                #:snippet '(delete-file-recursively "docs")))

(define rust-rlimit-0.9.1
  (crate-source "rlimit" "0.9.1"
                "13mb3ligflqb4h7m76pkyc8z5pswpc38fcl6qm1lvp2jls3rv8pq"))

(define rust-rmp-0.8.14
  (crate-source "rmp" "0.8.14"
                "1i1l6dhv7vws5vp0ikakj44fk597xi59g3j6ng1q55x3dz0xg3i2"))

(define rust-rmp-serde-0.15.5
  (crate-source "rmp-serde" "0.15.5"
                "178f4qlicldm9iy74q4wdqldk5i11p1ad30wzs9avx04mpwwygkj"))

(define rust-rmp-serde-1.3.0
  (crate-source "rmp-serde" "1.3.0"
                "1nylmh7w2vpa1bwrnx1jfp2l4yz6i5qrmpic5zll166gfyj9kraj"))

(define rust-rmpv-1.3.0
  (crate-source "rmpv" "1.3.0"
                "1adjigqyrzbv71s18qz3sa77zqggqip0p8j4rrrk5scyrlihfiaq"))

(define rust-roadmap-0.7.0
  (crate-source "roadmap" "0.7.0"
                "0xwzxsh3qk5mg7yfsjxjm6x68adp5c8szv7dznzf9irla5f7fjgc"))

(define rust-roaring-0.10.12
  (crate-source "roaring" "0.10.12"
                "0yqy3h6c13c73519bnwzmiv9nzxyl7sajl6n4q3lvnc4l77x5s0r"
                #:snippet '(delete-file-recursively "tests")))

(define rust-rodio-0.19.0
  (crate-source "rodio" "0.19.0"
                "1jvs8a6iq7h7s23acq1d76jk9zlc85snap58sgrkg3d3q4ksc1k0"))

(define rust-roff-0.2.2
  (crate-source "roff" "0.2.2"
                "1wyqz6m0pm4p6wzhwhahvcidfm7nwb38zl4q7ha940pn3w66dy48"))

(define rust-ron-0.8.1
  (crate-source "ron" "0.8.1"
                "154w53s895yxdfg7rn87c6f6x4yncc535x1x31zpcj7p0pzpw7xr"
                #:snippet '(delete-file-recursively "tests")))

(define rust-ron-0.9.0-alpha.1
  (crate-source "ron" "0.9.0-alpha.1"
                "0dh8fd4l54a36881b51275z3hbbjrmrj6rglr28sjzzz76js4i3n"
                #:snippet '(delete-file-recursively "tests")))

(define rust-ropey-1.6.1
  (crate-source "ropey" "1.6.1"
                "1dckf3likfi1my2ilqwhq2ifsm9iq8cayg6ws7fpa6nd1d11whck"))

(define rust-roxmltree-0.18.1
  (crate-source "roxmltree" "0.18.1"
                "00mkd2xyrxm8ap39sxpkhzdzfn2m98q3zicf6wd2f6yfa7il08w6"))

(define rust-roxmltree-0.20.0
  (crate-source "roxmltree" "0.20.0"
                "15vw91ps91wkmmgy62khf9zb63bdinvm80957dascbsw7dwvc83c"))

(define rust-rpassword-5.0.1
  (crate-source "rpassword" "5.0.1"
                "1yry1kmgjcb4qd5wak91203493x42ak3dz9hy1c0r9kyib7kdjgz"))

(define rust-rpassword-7.3.1
  (crate-source "rpassword" "7.3.1"
                "0gvy3lcpph9vv1rl0cjfn72ylvmgbw2vklmj6w0iv4cpr3ijniw0"))

(define rust-rpassword-7.4.0
  (crate-source "rpassword" "7.4.0"
                "0ffzfff51pl95a7px9gwlz243mn3vxyw7klcxhhng7049yvcim36"))

(define rust-rpds-1.1.0
  (crate-source "rpds" "1.1.0"
                "194hjbsicmgqi3dyllqrz09mmhh597m2j9l49lr16cyfscambqd0"))

(define rust-rpm-pkg-count-0.2.1
  (crate-source "rpm-pkg-count" "0.2.1"
                "1jzh63l7k30l37s5gd2m2hvh8slzjaxw54s4xpcjnaqb8xfsq0sa"))

(define rust-rsa-0.9.8
  (crate-source "rsa" "0.9.8"
                "06v9zl604jsqjajm647l9jjirn7k4lc8lmvys6hmqshpxp0qm4kq"
                #:snippet '(delete-file-recursively "tests")))

(define rust-rsconf-0.2.2
  (crate-source "rsconf" "0.2.2"
                "1p5w6qiskg43f5qhsh6slpc3klmhhyf76xvmzkkh215gy5czhamx"))

(define rust-rstest-0.19.0
  (crate-source "rstest" "0.19.0"
                "0c43nsxpm1b74jxc73xwg94is6bwqvfzkrr1xbqyx7j7l791clwx"))

(define rust-rstest-0.21.0
  (crate-source "rstest" "0.21.0"
                "10inb7hv1p42i981bf6705dakbgfbdgl2qaim4sf9mk9f2k5bzcs"))

(define rust-rstest-0.22.0
  (crate-source "rstest" "0.22.0"
                "0dlrn6y4z5xgsvf6ky3lrjwsxpvi13sizlkwnqs1gmmxc873yhkv"))

(define rust-rstest-0.23.0
  (crate-source "rstest" "0.23.0"
                "0d90hr3i2yajzgpzvsh6p2yjzmcb3nm8884xdbb5sswvwmdmhb0a"))

(define rust-rstest-0.25.0
  (crate-source "rstest" "0.25.0"
                "17bqpqcy029gmsqy2y3v4iyb9kj4i4mqkypqglzr2gk1z2995hvg"))

(define rust-rstest-macros-0.19.0
  (crate-source "rstest_macros" "0.19.0"
                "09ackagv8kc2v4xy0s7blyg4agij9bz9pbb31l5h4rqzrirdza84"))

(define rust-rstest-macros-0.21.0
  (crate-source "rstest_macros" "0.21.0"
                "13brqq64wp8gjaiq1d7g8jbzcwdwpwy0swpcilfx97d3b6pdyra1"))

(define rust-rstest-macros-0.22.0
  (crate-source "rstest_macros" "0.22.0"
                "0hiba8l3d20ajkifd3kz5rzzpxsy311ca4c4ll94pxqlglg73qf5"))

(define rust-rstest-macros-0.23.0
  (crate-source "rstest_macros" "0.23.0"
                "0nmdm7a4ysihnh0zz6w6gqrmw205zfp7xqkb2id3858vg20afpl2"))

(define rust-rstest-macros-0.25.0
  (crate-source "rstest_macros" "0.25.0"
                "0ikpla102bmmzzv0ivr5h9v9k3b24v93zljlvvlpnc4xfjcqs5hz"))

(define rust-rstest-reuse-0.7.0
  (crate-source "rstest_reuse" "0.7.0"
                "057y4v1rh9br58n2m3xqvm8xyx8k96jpgibgls3sah78f93gpa5k"))

(define rust-rt-format-0.3.1
  (crate-source "rt-format" "0.3.1"
                "1qjjwh9ny95xck1kp99gi6hfm9glrx54jx8npnj6yccxc7p7q225"))

(define rust-rtnetlink-0.14.1
  (crate-source "rtnetlink" "0.14.1"
                "19dbbky7bcrwk4jhgam148s8ykxc7pfrblxjvmciipyq8i9lg15n"))

(define rust-rtnetlink-0.16.0
  (crate-source "rtnetlink" "0.16.0"
                "0mqg85nfjghfilbv6vnp2cp7l445pr5njmzi8jpc1jd2b85qbd9w"))

(define rust-rtoolbox-0.0.2
  (crate-source "rtoolbox" "0.0.2"
                "03n9z8x353kylxhr9im8zawcisnmid3jiqrs8rbdn313cd7d4iy2"))

(define rust-rtoolbox-0.0.3
  (crate-source "rtoolbox" "0.0.3"
                "0vvz9p8wdzspwd7hk0cxyjr0i49cfqks5q02drym5glz4h5rgk57"))

(define rust-run-script-0.10.1
  (crate-source "run_script" "0.10.1"
                "1ix9kf1b3h5vmdadpv7rfxylmj8mphlbx0xgv6frhy4dqpyri7w2"
                #:snippet '(delete-file-recursively "docs")))

(define rust-rusb-0.9.4
  (crate-source "rusb" "0.9.4"
                "1905rijhabvylblh24379229hjmkfhxr80jc79aqd9v3bgq9z7xb"))

(define rust-rusoto-credential-0.48.0
  (crate-source "rusoto_credential" "0.48.0"
                "019dq3aq6hnfg4xvxdfsnrba08dwvciz0km4nr3n1basvc9nq2pf"))

(define rust-rusqlite-0.30.0
  (crate-source "rusqlite" "0.30.0"
                "0kgjk7h53wp8k2di2j83ivf50f0bk6rg7bq8j36ygxb42lb4d057"))

(define rust-rusqlite-0.31.0
  (crate-source "rusqlite" "0.31.0"
                "1bic69apqidimqf8gm80b98a832qzl9x6ns8myzah4yjg2ifnf5q"))

(define rust-rusqlite-0.32.1
  (crate-source "rusqlite" "0.32.1"
                "0vlx040bppl414pbjgbp7qr4jdxwszi9krx0m63zzf2f2whvflvp"))

(define rust-rusqlite-0.34.0
  (crate-source "rusqlite" "0.34.0"
                "0hy15qj9fzi4xhpslq32ajl6x07x2lzkrqn0qx8y1n48va349qrp"))

(define rust-rust-argon2-2.1.0
  (crate-source "rust-argon2" "2.1.0"
                "1s66kgbvnv5vaq4vlglx587bq93c662whrniz6ycpjb03m9li64x"))

(define rust-rust-decimal-1.37.1
  (crate-source "rust_decimal" "1.37.1"
                "0l3d6p96vqzand26al7r24aawljag37bxff6j2yr3hkallmxx9zs"))

(define rust-rust-decimal-1.37.2
  (crate-source "rust_decimal" "1.37.2"
                "0p83j981r1y8s8k6c11j26gyflfar93kqbf4344kx800am1ac0xj"))

(define rust-rust-embed-5.9.0
  (crate-source "rust-embed" "5.9.0"
                "08smnqn21vn5zaywfki391v2l9v30cs0446qzzhvjssxmimgxq9g"
                #:snippet '(delete-file-recursively "examples")))

(define rust-rust-embed-6.8.1
  (crate-source "rust-embed" "6.8.1"
                "0q96f3valahk4m4ir6c9vg45jhyalzn5iw90ijy4x33g4z1j8qm3"
                #:snippet '(delete-file-recursively "examples")))

(define rust-rust-embed-8.6.0
  (crate-source "rust-embed" "8.6.0"
                "0vxz02gqj9d65ym7ygbnwwrbgzh88zjqs2b1zk4vabb20i8vlfhb"
                #:snippet '(delete-file-recursively "examples")))

(define rust-rust-embed-8.7.0
  (crate-source "rust-embed" "8.7.0"
                "17f4pribh9nd97szi8zzc2a5xd5myxfjwi5vrvvrmfgwa3pc1yz5"
                #:snippet '(delete-file-recursively "examples")))

(define rust-rust-embed-8.7.2
  (crate-source "rust-embed" "8.7.2"
                "12hprnl569f1pg2sn960gfla913mk1mxdwpn2a6vl9iad2w0hn82"
                #:snippet '(delete-file-recursively "examples")))

(define rust-rust-embed-impl-5.9.0
  (crate-source "rust-embed-impl" "5.9.0"
                "0rlykmykrpllkf4900wv2jfdmq3m5qqrqhw4fdlbzxrfqi0irn9y"))

(define rust-rust-embed-impl-6.8.1
  (crate-source "rust-embed-impl" "6.8.1"
                "1b71mwb08nmwmzwhn15giiir1gii5aggp8j185l89hmjwn0lpfa9"))

(define rust-rust-embed-impl-8.6.0
  (crate-source "rust-embed-impl" "8.6.0"
                "1bms1vmb9z8rcwrjdk1rzmb3wi08mmh7jbz1m1d2r384prrqq68z"))

(define rust-rust-embed-impl-8.7.0
  (crate-source "rust-embed-impl" "8.7.0"
                "0bkh66kzmqv1i478d24nsv4nf89crhs732lblcy6dxp3lb4iix3b"))

(define rust-rust-embed-impl-8.7.2
  (crate-source "rust-embed-impl" "8.7.2"
                "171lshvdh122ypbf23gmhvrqnhbk0q9g27gaq6g82w9b76jg2rb0"))

(define rust-rust-embed-utils-5.1.0
  (crate-source "rust-embed-utils" "5.1.0"
                "0vc7492a6qlq7r899p7vyx5cwiqwkw2pf85mfw5anwr42ccj4l9a"))

(define rust-rust-embed-utils-7.8.1
  (crate-source "rust-embed-utils" "7.8.1"
                "0x7dvlmzzx0db3ra73g8h0fsfgy3c1xrzkhg22vkpp3hymmzyf4x"))

(define rust-rust-embed-utils-8.6.0
  (crate-source "rust-embed-utils" "8.6.0"
                "16mxa8asv0aq04fnbz1748rrzl8sc7frmjj2529c538cyk4wsbss"))

(define rust-rust-embed-utils-8.7.0
  (crate-source "rust-embed-utils" "8.7.0"
                "08cfp8x1nw1p272128hfwr9fvnlbg7dmafbbs1ji5q3z2jampm88"))

(define rust-rust-embed-utils-8.7.2
  (crate-source "rust-embed-utils" "8.7.2"
                "151m1966qk75y10msazdp0xj4fqw1khcry0z946bf84bcj0hrk7n"))

(define rust-rust-fuzzy-search-0.1.1
  (crate-source "rust-fuzzy-search" "0.1.1"
                "1chvl47hq42r219yxs6r1dp4l19acy5ay145hpc5drgzaiq6amx1"
                #:snippet '(delete-file-recursively "target")))

(define rust-rust-htslib-0.39.5
  (crate-source "rust-htslib" "0.39.5"
                "0lwf22acf6zn9znqd9irc9ny7msya8macvvsdgasqndz9lrzg7i3"))

(define rust-rust-htslib-0.40.2
  (crate-source "rust-htslib" "0.40.2"
                "1m08935ijvxaq6pjmzhkimc6isb3qm7h1sza1012vwk99vzdwwqb"))

(define rust-rust-i18n-3.1.5
  (crate-source "rust-i18n" "3.1.5"
                "0cn3gmqhabcnskv81krmcnp94yq4azhibp1s53pcav5gvwgmb8px"))

(define rust-rust-i18n-macro-3.1.5
  (crate-source "rust-i18n-macro" "3.1.5"
                "0rgra2j2ynrbd4z5072z5b72n3fllmbvpxi4pqzd4mk6ypbzgfi2"))

(define rust-rust-i18n-support-3.1.5
  (crate-source "rust-i18n-support" "3.1.5"
                "06cgfn6vad512vq4lf6l0ybascvi7db1wxvd0m902k5s5gsx83ll"))

(define rust-rust-ini-0.20.0
  (crate-source "rust-ini" "0.20.0"
                "12h608iy8kzxdrd0i5r20pzmgnw5fwg4rjwy5azq526bdch9h1iy"))

(define rust-rust-ini-0.21.1
  (crate-source "rust-ini" "0.21.1"
                "17s0fmfba05i9cnfwfy02wp7mf5wkapis5x1d68vgvmnw7q0wcaf"))

(define rust-rust-netrc-0.1.2
  (crate-source "rust-netrc" "0.1.2"
                "1sb3bl0aqisg7q6hxrz29s5c33m6q1qizdazz7dr57vnc9zhk63y"))

(define rust-rust-stemmers-1.2.0
  (crate-source "rust-stemmers" "1.2.0"
                "0m6acgdflrrcm17dj7lp7x4sfqqhga24qynv660qinwz04v20sp4"))

(define rust-rustc-demangle-0.1.24
  (crate-source "rustc-demangle" "0.1.24"
                "07zysaafgrkzy2rjgwqdj2a8qdpsm6zv6f5pgpk9x0lm40z9b6vi"))

(define rust-rustc-demangle-0.1.25
  (crate-source "rustc-demangle" "0.1.25"
                "0kxq6m0drr40434ch32j31dkg00iaf4zxmqg7sqxajhcz0wng7lq"))

(define rust-rustc-demangle-0.1.26
  (crate-source "rustc-demangle" "0.1.26"
                "1kja3nb0yhlm4j2p1hl8d7sjmn2g9fa1s4pj0qma5kj2lcndkxsn"))

(define rust-rustc-hash-1.1.0
  (crate-source "rustc-hash" "1.1.0"
                "1qkc5khrmv5pqi5l5ca9p5nl5hs742cagrndhbrlk3dhlrx3zm08"))

(define rust-rustc-hash-2.1.1
  (crate-source "rustc-hash" "2.1.1"
                "03gz5lvd9ghcwsal022cgkq67dmimcgdjghfb5yb5d352ga06xrm"))

(define rust-rustc-stable-hash-0.1.2
  (crate-source "rustc-stable-hash" "0.1.2"
                "026drx2ly2b8b1pp1c2v3p3ws6k0jaa5bbc5f4xwkibhj7r4453q"))

(define rust-rustc-version-0.1.7
  (crate-source "rustc_version" "0.1.7"
                "1160jjsqhqr25cvhr48hmpp8v61bjvjcnxzb0cyf4373lmp3gxf5"))

(define rust-rustc-version-0.2.3
  (crate-source "rustc_version" "0.2.3"
                "02h3x57lcr8l2pm0a645s9whdh33pn5cnrwvn5cb57vcrc53x3hk"))

(define rust-rustc-version-0.4.0
  (crate-source "rustc_version" "0.4.0"
                "0rpk9rcdk405xhbmgclsh4pai0svn49x35aggl4nhbkd4a2zb85z"))

(define rust-rustc-version-0.4.1
  (crate-source "rustc_version" "0.4.1"
                "14lvdsmr5si5qbqzrajgb6vfn69k0sfygrvfvr2mps26xwi3mjyg"))

(define rust-rustdct-0.7.1
  (crate-source "rustdct" "0.7.1"
                "0lcm1191xx8wizima5j3n25fs90x58v3q1kwg6cbzafn0m8maqcb"))

(define rust-rustdoc-json-0.8.9
  (crate-source "rustdoc-json" "0.8.9"
                "120k214xy255pfvizb6xlks0yv7psgjr3nh6l9xj9bgdfis747m2"))

(define rust-rustdoc-types-0.23.0
  (crate-source "rustdoc-types" "0.23.0"
                "095jdf44p0b02xq0n9vx1qpnrfgdqkd8nsbmp61r0s4x0q4nyr6c"))

(define rust-rustfft-6.2.0
  (crate-source "rustfft" "6.2.0"
                "11hx83yr2h2jszkba9qhq2d08q9i5rsashq62rfhqvahpihnb023"))

(define rust-rustfix-0.8.7
  (crate-source "rustfix" "0.8.7"
                "0nic48fb1a18nkr6n3d2rfr00i5gnad8xpmg4d7di56qk2qnkyl2"))

(define rust-rustfix-0.9.0
  (crate-source "rustfix" "0.9.0"
                "1a79gyag6w459qani0a1m6asadz6vxvgvmrw4l94zzvifiniarkz"))

(define rust-rusticata-macros-4.1.0
  (crate-source "rusticata-macros" "4.1.0"
                "0ch67lljmgl5pfrlb90bl5kkp2x6yby1qaxnpnd0p5g9xjkc9w7s"))

(define rust-rustix-0.37.23
  (crate-source "rustix" "0.37.23"
                "01mbsk0q93rh5ji6k27zq09r5fz88akl8kn6knj2fq8wz25p2sad"))

(define rust-rustix-0.37.28
  (crate-source "rustix" "0.37.28"
                "1dn131z1vj1ani37acirby3rwh6ksm3m1qdv8k554xxrg39nb4ai"
                #:snippet
                '(begin
                   (for-each delete-file (find-files "." "\\.a$"))
                   (delete-file "Cargo.toml")
                   (substitute* "Cargo.toml.orig"
                     ;; Depend unconditionally on the cc crate
                     (("(cc = .*), optional = true.*" _ cc)
                      (string-append cc " }\n"))
                     ;; Disable using the linux_raw backend
                     (("not\\(rustic_use_libc\\)") "miri"))
                   (substitute* "build.rs"
                     ;; Always use the 'feature = "cc"' path
                     (("not\\(feature = \"cc\"\\)") "feature = \"foobar\"")
                     (("#\\[cfg\\(feature = \"cc\"\\)\\]" all)
                      (string-append "//" all)))
                   (copy-file "Cargo.toml.orig" "Cargo.toml"))))

(define rust-rustix-0.38.44
  (crate-source "rustix" "0.38.44"
                "0m61v0h15lf5rrnbjhcb9306bgqrhskrqv7i1n0939dsw8dbrdgx"))

(define rust-rustix-0.38.8
  (crate-source "rustix" "0.38.8"
                "13r1lxa2p1id937y2yhb0nch59m97l50bnq4rs98676q46h4zv8r"))

(define rust-rustix-1.0.5
  (crate-source "rustix" "1.0.5"
                "1gsqrw9cp762ps9dl1d13n8mk5r0b6r2s002l1njxfylilwify6r"))

(define rust-rustix-1.0.7
  (crate-source "rustix" "1.0.7"
                "0rhjh16bnxi86nrn9qwcnw5632mvd5m1vdy61s4n9zz7mzb867n7"))

(define rust-rustix-1.0.8
  (crate-source "rustix" "1.0.8"
                "1j6ajqi61agdnh1avr4bplrsgydjw1n4mycdxw3v8g94pyx1y60i"))

(define rust-rustix-1.1.1
  (crate-source "rustix" "1.1.1"
                "07a4shlw0lssawnyvhg50cpljsdqd4ykhdlr4saf1jhhl64y68cn"))

(define rust-rustix-1.1.2
  (crate-source "rustix" "1.1.2"
                "0gpz343xfzx16x82s1x336n0kr49j02cvhgxdvaq86jmqnigh5fd"))

(define rust-rustix-dlmalloc-0.1.7
  (crate-source "rustix-dlmalloc" "0.1.7"
                "1pgjqzxydrp734zb76xba5akapqqk123rzvnlqvdyv2agggnim7w"))

(define rust-rustix-dlmalloc-0.2.2
  (crate-source "rustix-dlmalloc" "0.2.2"
                "0lqn1cz1yviwb69l0n5cx5lmlam2b6rqjgvy3785bar01ljq67q1"))

(define rust-rustix-futex-sync-0.2.3
  (crate-source "rustix-futex-sync" "0.2.3"
                "0ni5mc6bc02ghbzwccd9w7fi22vmyfjb1f5hkj9i4bl9j421pcn9"))

(define rust-rustix-futex-sync-0.4.0
  (crate-source "rustix-futex-sync" "0.4.0"
                "0r19a5q8nzw4dmrxh73x9pgh065gipwqf7g3fj50d8n1gcmd98jb"))

(define rust-rustix-openpty-0.1.1
  (crate-source "rustix-openpty" "0.1.1"
                "04pw7k8z1fk3642f9wr4jy1f2f6rly4pf2485jw4whn1kynklp52"))

(define rust-rustix-openpty-0.2.0
  (crate-source "rustix-openpty" "0.2.0"
                "14r3bvc2rcvvpdmhjivl6bqifdcl23f8bw9ncc58faw9b5y6rq8x"))

(define rust-rustls-0.21.12
  (crate-source "rustls" "0.21.12"
                "0gjdg2a9r81sdwkyw3n5yfbkrr6p9gyk3xr2kcsr3cs83x6s2miz"
                #:snippet
                '(begin
                   (delete-file-recursively "src/testdata")
                   (for-each delete-file (find-files "." "\\.bin")))))

(define rust-rustls-0.22.4
  (crate-source "rustls" "0.22.4"
                "0cl4q6w0x1cl5ldjsgbbiiqhkz6qg5vxl5dkn9wwsyxc44vzfkmz"
                #:snippet
                '(begin
                   (delete-file-recursively "src/testdata")
                   (for-each delete-file (find-files "." "\\.bin")))))

(define rust-rustls-0.23.25
  (crate-source "rustls" "0.23.25"
                "0g5idwxm04i71k3n66ml30zyfbgv6p85a7jky2i09v64i8cfjbl2"))

(define rust-rustls-0.23.26
  (crate-source "rustls" "0.23.26"
                "1w7ha5ivbhs91rvzh3ccgdgjizw6shagzs65xaaiai1sky3balfz"))

(define rust-rustls-0.23.27
  (crate-source "rustls" "0.23.27"
                "08d3nipyhmy4apksjyrb98s9k91wjyg1k7y0flx2671w135482bk"))

(define rust-rustls-0.23.28
  (crate-source "rustls" "0.23.28"
                "0hv6sk3r60vw11in2p8phpjd132684b4wg3zac456lzl1ghy6q3i"))

(define rust-rustls-0.23.29
  (crate-source "rustls" "0.23.29"
                "1lcvzvzqk8xx8jzg0x5v3mkqgwkwr7v6zdq8zw8rp6xj74h3i494"))

(define rust-rustls-0.23.31
  (crate-source "rustls" "0.23.31"
                "1k5ncablbb2h7hzllq3j3panqnks295v56xd488zrq1xy39cpsy0"))

(define rust-rustls-0.23.32
  (crate-source "rustls" "0.23.32"
                "0h2ddlnbjhs47hcmf3rbvr32sxj5kpf0m56rgk739l192rijag6d"))

(define rust-rustls-ffi-0.14.1
  (crate-source "rustls-ffi" "0.14.1"
                "1hky8z8xf9l813b1f54b0ir2knzv3kflh41i7q7xwlkyafza65n9"))

(define rust-rustls-native-certs-0.7.3
  (crate-source "rustls-native-certs" "0.7.3"
                "1r9ib5gwkfci2wbqnbh44nigvrfgxs4n1x89js82w97dxsab7gz5"))

(define rust-rustls-native-certs-0.8.1
  (crate-source "rustls-native-certs" "0.8.1"
                "1ls7laa3748mkn23fmi3g4mlwk131lx6chq2lyc8v2mmabfz5kvz"))

(define rust-rustls-pemfile-1.0.4
  (crate-source "rustls-pemfile" "1.0.4"
                "1324n5bcns0rnw6vywr5agff3rwfvzphi7rmbyzwnv6glkhclx0w"
                #:snippet '(delete-file-recursively "tests")))

(define rust-rustls-pemfile-2.2.0
  (crate-source "rustls-pemfile" "2.2.0"
                "0l3f3mrfkgdjrava7ibwzgwc4h3dljw3pdkbsi9rkwz3zvji9qyw"
                #:snippet '(delete-file-recursively "tests")))

(define rust-rustls-pki-types-1.11.0
  (crate-source "rustls-pki-types" "1.11.0"
                "0755isc0x5iymm3wsn59s0ad1pm9zidw7p34qfqlsjsac9jf4z4i"
                #:snippet '(delete-file-recursively "tests")))

(define rust-rustls-pki-types-1.12.0
  (crate-source "rustls-pki-types" "1.12.0"
                "0yawbdpix8jif6s8zj1p2hbyb7y3bj66fhx0y7hyf4qh4964m6i2"
                #:snippet '(delete-file-recursively "tests")))

(define rust-rustls-platform-verifier-0.3.4
  (crate-source "rustls-platform-verifier" "0.3.4"
                "145lhjn9w5khp0fn4lagaa8qqyjyhfqn2dg6llva6qyzvy5qgfxg"
                #:snippet '(delete-file-recursively "src/tests")))

(define rust-rustls-platform-verifier-android-0.1.1
  (crate-source "rustls-platform-verifier-android" "0.1.1"
                "13vq6sxsgz9547xm2zbdxiw8x7ad1g8n8ax6xvxsjqszk7q6awgq"
                #:snippet '(delete-file-recursively "maven")))

(define rust-rustls-webpki-0.101.7
  (crate-source "rustls-webpki" "0.101.7"
                "0rapfhpkqp75552i8r0y7f4vq7csb4k7gjjans0df73sxv8paqlb"
                #:snippet '(delete-file-recursively "tests")))

(define rust-rustls-webpki-0.102.8
  (crate-source "rustls-webpki" "0.102.8"
                "1sdy8ks86b7jpabpnb2px2s7f1sq8v0nqf6fnlvwzm4vfk41pjk4"
                #:snippet '(delete-file-recursively "tests")))

(define rust-rustls-webpki-0.103.1
  (crate-source "rustls-webpki" "0.103.1"
                "00rcdz0rb9ia2ivrq7412ry9qkvbh78pra2phl4p7kxck9vbiy7y"
                #:snippet '(delete-file-recursively "tests")))

(define rust-rustls-webpki-0.103.3
  (crate-source "rustls-webpki" "0.103.3"
                "0ddl9qxx94iyichk05r7l30d9dxfd35ybffhsxpsr9pppki2z9z4"))

(define rust-rustls-webpki-0.103.4
  (crate-source "rustls-webpki" "0.103.4"
                "1z4jmmgasjgk9glb160a66bshvgifa64mgfjrkqp7dy1w158h5qa"))

(define rust-rustls-webpki-0.103.6
  (crate-source "rustls-webpki" "0.103.6"
                "1szqh1g2cwx89f21mf6zkxc8rxg187y9jm5law8j6d4rrg1g6wl5"))

(define rust-rustsec-0.30.2
  (crate-source "rustsec" "0.30.2"
                "1j9fl7wx4zz7rq6i1p4jsvn7z704l3d082bizzg6pwdmpmfmr8vc"))

(define rust-rusttype-0.9.3
  (crate-source "rusttype" "0.9.3"
                "0rx9z8pdg9rxall8nlk9dwizgis1vk9kmdlmg55jad21l153gy1z"))

(define rust-rustup-toolchain-0.1.10
  (crate-source "rustup-toolchain" "0.1.10"
                "1q5712wbbpfhblx5zv87fpywpjq22gdss6j284k56a9z3h01amrd"))

(define rust-rustversion-1.0.14
  (crate-source "rustversion" "1.0.14"
                "1x1pz1yynk5xzzrazk2svmidj69jhz89dz5vrc28sixl20x1iz3z"))

(define rust-rustversion-1.0.18
  (crate-source "rustversion" "1.0.18"
                "0j2207vmgrcxwwwvknfn3lwv4i8djhjnxlvwdnz8bwijqqmrz08f"))

(define rust-rustversion-1.0.20
  (crate-source "rustversion" "1.0.20"
                "1lhwjb16dsm8brd18bn2bh0ryzc7qi29bi2jjsc6ny2zbwn3ivgd"))

(define rust-rustversion-1.0.21
  (crate-source "rustversion" "1.0.21"
                "07bb1xx05hhwpnl43sqrhsmxyk5sd5m5baadp19nxp69s9xij3ca"))

(define rust-rustversion-1.0.22
  (crate-source "rustversion" "1.0.22"
                "0vfl70jhv72scd9rfqgr2n11m5i9l1acnk684m2w83w0zbqdx75k"))

(define rust-rusty-fork-0.3.0
  (crate-source "rusty-fork" "0.3.0"
                "0kxwq5c480gg6q0j3bg4zzyfh2kwmc3v2ba94jw8ncjc8mpcqgfb"))

(define rust-rusty-hook-0.11.2
  (crate-source "rusty-hook" "0.11.2"
                "18r1mh2br76b6zwkvch01xi9rhj9fknmh7l5vnx1qzmyc6zfkkln"))

(define rust-rustybuzz-0.18.0
  (crate-source "rustybuzz" "0.18.0"
                "108igxavhzmln7h6qgfxk3bdghp8hvlc8lpbk13q6qcya76iqpf8"))

(define rust-rustyline-10.1.1
  (crate-source "rustyline" "10.1.1"
                "1vvsd68cch0lpcg6mcwfvfdd6r4cxbwis3bf9443phzkqcr3rs61"))

(define rust-rustyline-9.1.2
  (crate-source "rustyline" "9.1.2"
                "0f8069ljhiv9nf97y975wvv9yvx82w3lm9g50d5n298fkiw2cy6v"))

(define rust-ruzstd-0.6.0
  (crate-source "ruzstd" "0.6.0"
                "0yygqpar2x910lnii4k5p43aj4943hlnxpczmqhsfddmxrqa8x2i"))

(define rust-ruzstd-0.7.3
  (crate-source "ruzstd" "0.7.3"
                "17z83lw00pd3190yvdk2lksdi7my6wc0psgg07is6gf7pyb2kl7s"))

(define rust-ryu-0.2.8
  (crate-source "ryu" "0.2.8"
                "13wsi4408qxi9w44pdf5zfds4ym7np2070wkhg1g4j4dvi4rasmr"))

(define rust-ryu-1.0.15
  (crate-source "ryu" "1.0.15"
                "0hfphpn1xnpzxwj8qg916ga1lyc33lc03lnf1gb3wwpglj6wrm0s"))

(define rust-ryu-1.0.18
  (crate-source "ryu" "1.0.18"
                "17xx2s8j1lln7iackzd9p0sv546vjq71i779gphjq923vjh5pjzk"))

(define rust-ryu-1.0.19
  (crate-source "ryu" "1.0.19"
                "1pg6a0b80m32ahygsdkwzs3bfydk4snw695akz4rqxj4lv8a58bf"))

(define rust-ryu-1.0.20
  (crate-source "ryu" "1.0.20"
                "07s855l8sb333h6bpn24pka5sp7hjk2w667xy6a0khkf6sqv5lr8"))

(define rust-safe-arch-0.7.4
  (crate-source "safe_arch" "0.7.4"
                "08sk47n1kcm5w2di6bpgi2hsw8r2caz2230pwqvbdqfv5pl2vc4n"))

(define rust-safetensors-0.4.5
  (crate-source "safetensors" "0.4.5"
                "1djm1sz72da28xws6dlcpwgchvckc9m86v73djj30qba4c8hqmj4"))

(define rust-salsa20-0.10.2
  (crate-source "salsa20" "0.10.2"
                "04w211x17xzny53f83p8f7cj7k2hi8zck282q5aajwqzydd2z8lp"))

(define rust-same-file-1.0.6
  (crate-source "same-file" "1.0.6"
                "00h5j1w87dmhnvbv9l8bic3y7xxsnjmssvifw2ayvgx9mb1ivz4k"))

(define rust-sanakirja-1.4.3
  (crate-source "sanakirja" "1.4.3"
                "04bdmhs3y35w3ip265mm4yiyf2493vcizl04kwh248af0q6zgal1"))

(define rust-sanakirja-core-1.4.1
  (crate-source "sanakirja-core" "1.4.1"
                "1xcb4d9gaxp61na4dyz9iiq3n1shhiivqs0iv5xnxb1ymqsdnxl3"))

(define rust-sanitize-filename-0.6.0
  (crate-source "sanitize-filename" "0.6.0"
                "0kd37m2nd364vj09f330y853fmnw2yyy7hsmnxxnlwzbki7lz65w"))

(define rust-sapling-renderdag-0.1.0
  (crate-source "sapling-renderdag" "0.1.0"
                "0qbv8k698kiz8rpr63hn0m7g789pbmpmg7blql0hkgc7mffbizzd"))

(define rust-sapling-streampager-0.11.0
  (crate-source "sapling-streampager" "0.11.0"
                "14ns7a8lmsvnn3kpcs3vdfj7f9ya9pfzbwzgh902sghzhkh5l8k7"))

(define rust-scan-fmt-0.2.6
  (crate-source "scan_fmt" "0.2.6"
                "0j0jb1dsa8zjpnc875wy72190zlyngvl62mfv8pqwal8vfjv0lqb"))

(define rust-scc-2.3.3
  (crate-source "scc" "2.3.3"
                "1cc5ccsjhs0b31yfcdjw0dbkgm1fd7pg811zk4wam595min1y2ga"))

(define rust-scc-2.3.4
  (crate-source "scc" "2.3.4"
                "1x1739jz8bx9x46d26aagwin9psykka1siw9fn0lbwi8zdsxgci2"))

(define rust-scc-2.4.0
  (crate-source "scc" "2.4.0"
                "1k2nwz3bysf1s3r5g437vq9xfm9i4sadfzn5c0k8xx7ynx3g1rj6"))

(define rust-schannel-0.1.27
  (crate-source "schannel" "0.1.27"
                "0gbbhy28v72kd5iina0z2vcdl3vz63mk5idvkzn5r52z6jmfna8z"
                #:snippet '(delete-file-recursively "test")))

(define rust-schannel-0.1.28
  (crate-source "schannel" "0.1.28"
                "1qb6s5gyxfz2inz753a4z3mc1d266mwvz0c5w7ppd3h44swq27c9"
                #:snippet '(delete-file-recursively "test")))

(define rust-scheduled-thread-pool-0.2.7
  (crate-source "scheduled-thread-pool" "0.2.7"
                "068s77f9xcpvzl70nsxk8750dzzc6f9pixajhd979815cj0ndg1w"))

(define rust-schemars-0.8.22
  (crate-source "schemars" "0.8.22"
                "05an9nbi18ynyxv1rjmwbg6j08j0496hd64mjggh53mwp3hjmgrz"))

(define rust-schemars-0.9.0
  (crate-source "schemars" "0.9.0"
                "0pqncln5hqbzbl2r3yayyr4a82jjf93h2cfxrn0xamvx77wr3lac"))

(define rust-schemars-1.0.4
  (crate-source "schemars" "1.0.4"
                "1l7w773jfk6mz0v8wpahp60aslksjijlbm65ysi4y5mwj520rll2"))

(define rust-schemars-derive-0.8.22
  (crate-source "schemars_derive" "0.8.22"
                "0kakyzrp5801s4i043l4ilv96lzimnlh01pap958h66n99w6bqij"))

(define rust-schemars-derive-1.0.4
  (crate-source "schemars_derive" "1.0.4"
                "107sprdfa5kacifxq41qv5ccv7a78msxyr8ikz0qs4qxdlwj1l1k"))

(define rust-scm-record-0.8.0
  (crate-source "scm-record" "0.8.0"
                "15yjqrq78q24znz1sdykgbrmlb5x6yf4ay3i7rsn65paw77acqij"))

(define rust-scoped-threadpool-0.1.9
  (crate-source "scoped_threadpool" "0.1.9"
                "1a26d3lk40s9mrf4imhbik7caahmw2jryhhb6vqv6fplbbgzal8x"))

(define rust-scoped-tls-1.0.1
  (crate-source "scoped-tls" "1.0.1"
                "15524h04mafihcvfpgxd8f4bgc3k95aclz8grjkg9a0rxcvn9kz1"))

(define rust-scopeguard-0.3.3
  (crate-source "scopeguard" "0.3.3"
                "09sy9wbqp409pkwmqni40qmwa99ldqpl48pp95m1xw8sc19qy9cl"))

(define rust-scopeguard-1.2.0
  (crate-source "scopeguard" "1.2.0"
                "0jcz9sd47zlsgcnm1hdw0664krxwb5gczlif4qngj2aif8vky54l"))

(define rust-scopetime-0.1.2
  (crate-source "scopetime" "0.1.2"
                "1ngkasx6l072cvigx7q5r33i8acjardr4g8jnwdrcym4758f5vb6"))

(define rust-scratch-1.0.7
  (crate-source "scratch" "1.0.7"
                "0lk160986gamss2rxi746nj747xx7gg8r2hfskrr9fccqc8prkx3"))

(define rust-scratch-1.0.8
  (crate-source "scratch" "1.0.8"
                "0llahyff0cjjs9yklcgf7cxq6j29r3mlaym5dm9mkxg5hspq0qlz"))

(define rust-scratch-1.0.9
  (crate-source "scratch" "1.0.9"
                "1cj826qggwn482wbfnzij5g9p411qszai0dnfld4qzh93g2jx3yn"))

(define rust-scroll-0.11.0
  (crate-source "scroll" "0.11.0"
                "1nhrhpzf95pxbcjjy222blwf8rl3adws6vsqax0yzyxsa6snbi84"))

(define rust-scroll-0.12.0
  (crate-source "scroll" "0.12.0"
                "19mix9vm4k23jkknpgbi0ylmhpf2hnlpzzrfj9wqcj88lj55kf3a"))

(define rust-scroll-derive-0.11.1
  (crate-source "scroll_derive" "0.11.1"
                "1bi5ljnzksvqhic6j7i2a2ap41s78xr0gifkgjxdxlj63pw4kc8x"))

(define rust-scroll-derive-0.12.0
  (crate-source "scroll_derive" "0.12.0"
                "0cmr3hxk318s2ivv37cik2l1r0d8r0qhahnin5lpxbr5w3yw50bz"))

(define rust-scroll-derive-0.12.1
  (crate-source "scroll_derive" "0.12.1"
                "0gb89b1yr8a6jwp4rcm00xqry6ajvmfywsm7bf5f42a686yfm0qp"))

(define rust-scrypt-0.11.0
  (crate-source "scrypt" "0.11.0"
                "07zxfaqpns9jn0mnxm7wj3ksqsinyfpirkav1f7kc2bchs2s65h5"))

(define rust-sct-0.7.1
  (crate-source "sct" "0.7.1"
                "056lmi2xkzdg1dbai6ha3n57s18cbip4pnmpdhyljli3m99n216s"
                #:snippet '(delete-file-recursively "src/testdata")))

(define rust-sctk-adwaita-0.10.1
  (crate-source "sctk-adwaita" "0.10.1"
                "1v14vqp7k39jk7pgaibwc06qq9vcmi82k7zlv3qpfvq52w17y9xn"
                #:snippet '(delete-file "src/title/Cantarell-Regular.ttf")))

(define rust-sd-notify-0.4.5
  (crate-source "sd-notify" "0.4.5"
                "1x1bmz30x2i35j771rqyyan40473aqk0xjrh2dk9xdnqf7gylhxr"))

(define rust-sdd-3.0.8
  (crate-source "sdd" "3.0.8"
                "08gfrs3cz2ncbqz7v1xm5687hggl166pic1frfk7s0f7244hfkjq"))

(define rust-sdd-3.0.10
  (crate-source "sdd" "3.0.10"
                "1jj1brjjasx7r3lf6iyhhrpglx47vzr0z1qi1n0fcszjzv5wy3a9"))

(define rust-seahash-3.0.7
  (crate-source "seahash" "3.0.7"
                "0iqg12lxkn0ivsfa1gkylcwj5wmi6zl87mbizlrkg918s6hprxaq"))

(define rust-seahash-4.1.0
  (crate-source "seahash" "4.1.0"
                "0sxsb64np6bvnppjz5hg4rqpnkczhsl8w8kf2a5lr1c08xppn40w"
                #:snippet '(delete-file "logo.png")))

(define rust-search-provider-0.6.0
  (crate-source "search-provider" "0.6.0"
                "01jby7xq0dd9rafw5kgskpbxgppn4imzm71w1sdj8iy9ci4ghh9z"))

(define rust-sec1-0.7.3
  (crate-source "sec1" "0.7.3"
                "1p273j8c87pid6a1iyyc7vxbvifrw55wbxgr0dh3l8vnbxb7msfk"
                #:snippet '(delete-file-recursively "tests")))

(define rust-seccomp-sys-0.1.3
  (crate-source "seccomp-sys" "0.1.3"
                "1rkbiq1vd5d3cc4w5ql05pj1vbjqfi7zir02szwfm2khl15zgg75"))

(define rust-secrecy-0.10.3
  (crate-source "secrecy" "0.10.3"
                "0nmfsf9qm8921v2jliz08bj8zrryqar4gj3d6irqfc3kaj2az4g8"))

(define rust-secrecy-0.8.0
  (crate-source "secrecy" "0.8.0"
                "07p9h2bpkkg61f1fzzdqqbf74kwv1gg095r1cdmjzzbcl17cblcv"))

(define rust-secret-service-3.1.0
  (crate-source "secret-service" "3.1.0"
                "1yd8r1k0706052zrk9bwxy6h01gy5pyk4llk8hcnvw1pvwwls85m"))

(define rust-section-testing-0.0.5
  (crate-source "section_testing" "0.0.5"
                "18dfl8nacwcd1z2y4sgdx7751czzpl6282q6cd49ys5gd0xlkljz"))

(define rust-security-framework-0.4.4
  (crate-source "security-framework" "0.4.4"
                "0dg504y0ys27qksdcnn8qcyvrfhkjx8f5d6jvpm7i7yrsw18k034"))

(define rust-security-framework-2.11.1
  (crate-source "security-framework" "2.11.1"
                "00ldclwx78dm61v7wkach9lcx76awlrv0fdgjdwch4dmy12j4yw9"))

(define rust-security-framework-3.2.0
  (crate-source "security-framework" "3.2.0"
                "05mkrddi9i18h9p098d0iimqv1xxz0wd8mbgpbvh9jj67x0205r7"))

(define rust-security-framework-3.3.0
  (crate-source "security-framework" "3.3.0"
                "037f0h06p00gg7ycczx3jsz4ikxzll177gdqnhca72h2qn91vyw0"))

(define rust-security-framework-3.5.0
  (crate-source "security-framework" "3.5.0"
                "0jkgq7k2yvaqgsbnbmvi6czr231a0vsib6lwjckhhldpv518w6fc"))

(define rust-security-framework-sys-0.4.3
  (crate-source "security-framework-sys" "0.4.3"
                "01c4fkj78frz6kz9vri1am2pbkz52raxws5lbrli5xajjbci3gqp"))

(define rust-security-framework-sys-2.14.0
  (crate-source "security-framework-sys" "2.14.0"
                "0chwn01qrnvs59i5220bymd38iddy4krbnmfnhf4k451aqfj7ns9"))

(define rust-security-framework-sys-2.15.0
  (crate-source "security-framework-sys" "2.15.0"
                "1h6mijxnfrwvl1y4dzwn3m877j6dqp9qn3g37i954j5czazhq7yc"))

(define rust-selectors-0.22.0
  (crate-source "selectors" "0.22.0"
                "1zhjpvww238lh4nz7kdw4ywlpmjbmzvrm76w1jyacjxci4c0ycnz"))

(define rust-selectors-0.25.0
  (crate-source "selectors" "0.25.0"
                "01kvl1r7plzlb665r64p11djabhsrd88si2zh7vci3v3ydshbcsf"))

(define rust-self-cell-0.10.3
  (crate-source "self_cell" "0.10.3"
                "0pci3zh23b7dg6jmlxbn8k4plb7hcg5jprd1qiz0rp04p1ilskp1"))

(define rust-self-cell-1.1.0
  (crate-source "self_cell" "1.1.0"
                "1gmxk5bvnnimcif7v1jk8ai2azfvh9djki545nd86vsnphjgrzf2"))

(define rust-self-cell-1.2.0
  (crate-source "self_cell" "1.2.0"
                "0jg70srf4hzrw96x8iclgf6i8dfgm1x8ds2i7yzcgq0i8njraz8g"))

(define rust-self-replace-1.5.0
  (crate-source "self-replace" "1.5.0"
                "1drganasvf5b0x6c9g60jkfhzjc9in3r6cznjfw0lhmbbrdq3v03"))

(define rust-self-update-0.42.0
  (crate-source "self_update" "0.42.0"
                "0p3vwhj44z75z44722z4x4py3xmq351bnir9zclwrnp0xj3c0cnq"))

(define rust-sem-safe-0.2.1
  (crate-source "sem_safe" "0.2.1"
                "0is9ks9bq98vky2hlx078d8y7zkc29ahz30anbr6qncv5zlnb5my"))

(define rust-semver-0.1.20
  (crate-source "semver" "0.1.20"
                "1b10m0hxrr947gp41lj9vnmgl5ddwx3d41vnblsg06ppvkz11x6l"))

(define rust-semver-0.11.0
  (crate-source "semver" "0.11.0"
                "1dn6064fipjymnmjccyjhb70miyvqvp08gvw1wbg8vbg4c8ay0gk"))

(define rust-semver-0.9.0
  (crate-source "semver" "0.9.0"
                "00q4lkcj0rrgbhviv9sd4p6qmdsipkwkbra7rh11jrhq5kpvjzhx"))

(define rust-semver-1.0.19
  (crate-source "semver" "1.0.19"
                "1c2gg8k8sij0vbbd2dv30mx228xxqiw37apjiqdf0v8w419715xd"))

(define rust-semver-1.0.20
  (crate-source "semver" "1.0.20"
                "140hmbfa743hbmah1zjf07s8apavhvn04204qjigjiz5w6iscvw3"))

(define rust-semver-1.0.26
  (crate-source "semver" "1.0.26"
                "1l5q2vb8fjkby657kdyfpvv40x2i2xqq9bg57pxqakfj92fgmrjn"))

(define rust-semver-1.0.27
  (crate-source "semver" "1.0.27"
                "1qmi3akfrnqc2hfkdgcxhld5bv961wbk8my3ascv5068mc5fnryp"))

(define rust-semver-parser-0.10.3
  (crate-source "semver-parser" "0.10.3"
                "1hnyq04mmxwx40xi3a2a9yn8nmjav7zqp4waggf7ylm3aimj004r"))

(define rust-semver-parser-0.7.0
  (crate-source "semver-parser" "0.7.0"
                "18vhypw6zgccnrlm5ps1pwa0khz7ry927iznpr88b87cagr1v2iq"))

(define rust-sensors-0.2.2
  (crate-source "sensors" "0.2.2"
                "1l6nvmbbbbl9mdkd9806mmls68d6hvqb2nl0nm14chva5xwz8fks"))

(define rust-sentry-0.35.0
  (crate-source "sentry" "0.35.0"
                "07p5wvg3ba1nhrf1gbagx42ish9qy6825q61szm1v1ln3gsmhs81"))

(define rust-sentry-backtrace-0.35.0
  (crate-source "sentry-backtrace" "0.35.0"
                "0gvx1dxfw0rqzxkiapjvpkb0mwcla946rc5lfpqjw9cr9v114xz5"))

(define rust-sentry-contexts-0.35.0
  (crate-source "sentry-contexts" "0.35.0"
                "12d0mp5icb6vw7yrlsd1f0ij0i0x45pr2r3dm809w9xrqd77ba7b"))

(define rust-sentry-core-0.35.0
  (crate-source "sentry-core" "0.35.0"
                "14jdq200ykmhgw7h9av23gijf2rnf0kg6ixl48z1xbpvskfbdy7r"))

(define rust-sentry-debug-images-0.35.0
  (crate-source "sentry-debug-images" "0.35.0"
                "0z70jamb1cah9wcgxa3mwaic6hyagy9pc1zsvgpy9xfk6f8sd0l9"))

(define rust-sentry-panic-0.35.0
  (crate-source "sentry-panic" "0.35.0"
                "09bb3zyibv87y4c40x3zhl4pk8k6ipi5zvkia2v33s81dyp6safy"))

(define rust-sentry-tracing-0.35.0
  (crate-source "sentry-tracing" "0.35.0"
                "1w56b6qqfgyfsh1ic9mljn54kbccw7m2aip3w79iaygdaz3p6gr6"))

(define rust-sentry-types-0.35.0
  (crate-source "sentry-types" "0.35.0"
                "09x5f2d51kcf72vsgib0jvq25ji34xx9k3mrj9nnm2lli6ix67m7"))

(define rust-seq-io-0.3.4
  (crate-source "seq_io" "0.3.4"
                "1pkasxcf25p1cf2w99a4flhjjaicg4rs14w5g8fkrs0fafg5a0qk"))

(define rust-sequoia-autocrypt-0.25.1
  (crate-source "sequoia-autocrypt" "0.25.1"
                "0ns121ggmx690m8czhc7zbb7rwz0jjv3l5gw4igs6mn1hznc0kz2"
                #:snippet '(delete-file-recursively "tests")))

(define rust-sequoia-autocrypt-0.26.0
  (crate-source "sequoia-autocrypt" "0.26.0"
                "12wjwv5bsbwjqf25zdkfilnlmgw3jj8y3j1nbm83x6jqr3qmzmhi"
                #:snippet '(delete-file-recursively "tests")))

(define rust-sequoia-cert-store-0.6.2
  (crate-source "sequoia-cert-store" "0.6.2"
                "19drjzxihs1bgqb0klwf81nxxx9jqgifzi49v8gqw00d6ba9lcwy"
                #:snippet '(delete-file-recursively "tests")))

(define rust-sequoia-cert-store-0.7.0
  (crate-source "sequoia-cert-store" "0.7.0"
                "0x0bcznz72251j56qm29523fabivx48bqzlwa3p1m4z96znqg2fc"
                #:snippet '(delete-file-recursively "tests")))

(define rust-sequoia-directories-0.1.0
  (crate-source "sequoia-directories" "0.1.0"
                "1m9plvzm61571y1vzsp3jkba2mgbxgwckrbpmcbqdky5c24x87dh"))

(define rust-sequoia-gpg-agent-0.5.1
  (crate-source "sequoia-gpg-agent" "0.5.1"
                "1j3pawsnxj27ak5gfw6aa7crypbbvv5whwpm3cml7ay4yv6qp8hh"))

(define rust-sequoia-gpg-agent-0.6.1
  (crate-source "sequoia-gpg-agent" "0.6.1"
                "1qq93rywy7zigxjiffkywv2x2h6zsamrnbk5zxj7i1vdz8pnmarr"))

(define rust-sequoia-ipc-0.35.1
  (crate-source "sequoia-ipc" "0.35.1"
                "1qryibvxs7fgbfi55sxsmh6kpim41rz06sslfb0n2cr3jn5cpbl9"))

(define rust-sequoia-ipc-0.36.0
  (crate-source "sequoia-ipc" "0.36.0"
                "1aw0haz6bdr01khviydspqvm0fga3xvygp74876bnqkzsfxpj9f9"))

(define rust-sequoia-keystore-0.6.2
  (crate-source "sequoia-keystore" "0.6.2"
                "1qy3nk2r39m5mzvx58ij7a1r9hiw0fmgmjrad6j4nf8djids5lsx"))

(define rust-sequoia-keystore-0.7.0
  (crate-source "sequoia-keystore" "0.7.0"
                "0iv3aimmi8lp9rwhsn1nm19fp6fr343vpcwn646plxn09088245m"))

(define rust-sequoia-keystore-backend-0.6.0
  (crate-source "sequoia-keystore-backend" "0.6.0"
                "15nzpqgpnnbmpcdldzgzx5v0ifgm1njqhvzsh40cg3c02p7xyz40"))

(define rust-sequoia-keystore-backend-0.7.0
  (crate-source "sequoia-keystore-backend" "0.7.0"
                "007m2nicrp4f4aqylvsi54gbyp10g3r44sm2qx6w64k4kdzh8ny5"))

(define rust-sequoia-keystore-gpg-agent-0.4.1
  (crate-source "sequoia-keystore-gpg-agent" "0.4.1"
                "1qnpcydrw0l3i0i082cy9mghjjq3l25clxwfj6gcpf72d6hq0wkq"))

(define rust-sequoia-keystore-gpg-agent-0.5.0
  (crate-source "sequoia-keystore-gpg-agent" "0.5.0"
                "0ayz2hrvy2xyjdhp2h2ax52fqdh1p3r5p3bkypski2yqy7qh3nbs"))

(define rust-sequoia-keystore-openpgp-card-0.1.1
  (crate-source "sequoia-keystore-openpgp-card" "0.1.1"
                "0qna0si81iwr3axcg77lsri6d92xjz3w3y9s3yxf2rgrwkqpvpbp"))

(define rust-sequoia-keystore-openpgp-card-0.2.0
  (crate-source "sequoia-keystore-openpgp-card" "0.2.0"
                "1bwfjzbs4w1j46sw5w1sn2mqmfn67ps0nlq167hh0pxbhvn7slr5"))

(define rust-sequoia-keystore-softkeys-0.6.0
  (crate-source "sequoia-keystore-softkeys" "0.6.0"
                "1zyapjfadgmy5fnk1kwxr0dq7i4qmj4614r0g0z68dawpp8mdflr"))

(define rust-sequoia-keystore-softkeys-0.7.0
  (crate-source "sequoia-keystore-softkeys" "0.7.0"
                "0hl7xq9k7206ai1ry68czqc7gh7nq2cszvx157w8yzr7lplkcdvi"))

(define rust-sequoia-man-0.3.2
  (crate-source "sequoia-man" "0.3.2"
                "05yj22jdqng0lizvx80jvafkja7qh9abcmdc2g69q7bs3mqsj7wn"))

(define rust-sequoia-net-0.29.0
  (crate-source "sequoia-net" "0.29.0"
                "0xdraqrjlpjpzyn8sc8c8xfq13pr1gp6sd4c0n80x30i6kc60zjl"))

(define rust-sequoia-net-0.30.0
  (crate-source "sequoia-net" "0.30.0"
                "099rwgk1wxygx5parwvn5zliwd9my72slv1wrmck5xa1gv9zavlm"))

(define rust-sequoia-openpgp-1.22.0
  (crate-source "sequoia-openpgp" "1.22.0"
                "0ngg32kqcrg6lz1c0g2fkb76cm1ajifb9qcjvv77kw4gwkly8n78"))

(define rust-sequoia-openpgp-2.0.0
  (crate-source "sequoia-openpgp" "2.0.0"
                "1s616nrmway3qm2wmrdgxrr5hc0bx7rsgaccp6fqnh93s31myph1"))

(define rust-sequoia-policy-config-0.6.0
  (crate-source "sequoia-policy-config" "0.6.0"
                "0x42h22kng4dsbfr0a6zdf2j9bcq14r0yr6xdw6rrggj139lazbm"))

(define rust-sequoia-policy-config-0.7.0
  (crate-source "sequoia-policy-config" "0.7.0"
                "17alq9dyg9gd26zbc8bcgm0vgwnlghqp0npvh088fc768c05yzb1"))

(define rust-sequoia-policy-config-0.8.0
  (crate-source "sequoia-policy-config" "0.8.0"
                "15z5n4vf430siri4gljhsh7f6wqv8zck38z1jxm7p1b4imq6n0cf"))

(define rust-sequoia-wot-0.13.3
  (crate-source "sequoia-wot" "0.13.3"
                "08948jazk7c5a4sza5amq9ah8r0mg02lcrrpqhwgi3qjx6w550v0"))

(define rust-sequoia-wot-0.14.0
  (crate-source "sequoia-wot" "0.14.0"
                "1474vjbi2kfn5dxdk49rcd894vnqxg4pz3j0c9iphvza94rgl9bp"))

(define rust-serde-1.0.136
  (crate-source "serde" "1.0.136"
                "12a791cbdd3gi08536i4frrqsps0ak8gvhpijvgj9rg1055y4cff"))

(define rust-serde-1.0.183
  (crate-source "serde" "1.0.183"
                "0g7bv28rp0h0bdy116r63aldyvlfkpy318a54x9nv1vp4sh8vb1j"))

(define rust-serde-1.0.188
  (crate-source "serde" "1.0.188"
                "17jlqzfhimsk8w37ifjwnm86nwjzawlbgwmwc7nhwdwslv5hz7ng"))

(define rust-serde-1.0.215
  (crate-source "serde" "1.0.215"
                "13xqkw93cw9rnbkm0zy1apnilzq7l2xf1qw8m1nkga8i1fnw24v5"))

(define rust-serde-1.0.216
  (crate-source "serde" "1.0.216"
                "13ikqs0cvd220530x4rj1m9ab5wcflrwkw7cpvl9fnlkdq0q35qb"))

(define rust-serde-1.0.218
  (crate-source "serde" "1.0.218"
                "0q6z4bnrwagnms0bds4886711l6mc68s979i49zd3xnvkg8wkpz8"))

(define rust-serde-1.0.219
  (crate-source "serde" "1.0.219"
                "1dl6nyxnsi82a197sd752128a4avm6mxnscywas1jq30srp2q3jz"))

(define rust-serde-1.0.225
  (crate-source "serde" "1.0.225"
                "07dxpjh0g1mq3md9yvn7jbgssgcizcircf23f04xml1mwbg28v7x"))

(define rust-serde-1.0.226
  (crate-source "serde" "1.0.226"
                "1zcm2asp9fiphbp0k96whabw02kiiqgzxhbyz85vc92v088n9jhd"))

(define rust-serde-1.0.228
  (crate-source "serde" "1.0.228"
                "17mf4hhjxv5m90g42wmlbc61hdhlm6j9hwfkpcnd72rpgzm993ls"))

(define rust-serde-aux-4.7.0
  (crate-source "serde-aux" "4.7.0"
                "02lamg1xzz2xghbqwr872gi33s553szz16rsa2bba3z9iyr6fzr0"))

(define rust-serde-bencode-0.2.4
  (crate-source "serde_bencode" "0.2.4"
                "0gj17p1w5hyi69fngv55dai4nb4fmdij76gqwyb9if9qfixzq3d7"))

(define rust-serde-big-array-0.5.1
  (crate-source "serde-big-array" "0.5.1"
                "0zsb9s9rcca3408kg20c6xpx917c9vbbnap5gvrf0wvdqz17rz0i"))

(define rust-serde-bser-0.4.0
  (crate-source "serde_bser" "0.4.0"
                "05w7iyrm8xrinnqh6b7ydxnd707g5dc7avvcw5d5nbp42p64nsx5"))

(define rust-serde-bytes-0.11.10
  (crate-source "serde_bytes" "0.11.10"
                "03pg7rx69sv8506z8v0bg580ysrygw32sd47cp4s38z48cr13igk"))

(define rust-serde-bytes-0.11.17
  (crate-source "serde_bytes" "0.11.17"
                "15kds0mw19lvm8ydd1qbharh5rz96zir06yn2silqbfy3cigsdw4"))

(define rust-serde-cbor-0.11.2
  (crate-source "serde_cbor" "0.11.2"
                "1xf1bq7ixha30914pd5jl3yw9v1x6car7xgrpimvfvs5vszjxvrb"
                #:snippet '(delete-file-recursively "tests")))

(define rust-serde-core-1.0.225
  (crate-source "serde_core" "1.0.225"
                "10v3z58j5k6xhdxh90xgrv20wlnz5fnl67n04jdm47nbl3wmd4v5"))

(define rust-serde-core-1.0.226
  (crate-source "serde_core" "1.0.226"
                "1936x6cpqgyq57nm7qi416dsc4fiq3jv6d7vh74xmfgdk4wscaxs"))

(define rust-serde-core-1.0.228
  (crate-source "serde_core" "1.0.228"
                "1bb7id2xwx8izq50098s5j2sqrrvk31jbbrjqygyan6ask3qbls1"))

(define rust-serde-derive-1.0.136
  (crate-source "serde_derive" "1.0.136"
                "1yb28smlymba4qbj2bn4c4myvblypqvkxv9q33s0dlzwa9qpwn88"))

(define rust-serde-derive-1.0.183
  (crate-source "serde_derive" "1.0.183"
                "05j8kbch6wpjk6m7kcgnx3bwkcsg5pp2zfci3bkvxfdhc0nrgzma"))

(define rust-serde-derive-1.0.188
  (crate-source "serde_derive" "1.0.188"
                "1wjaclvsfxgqnnnykllvb5gffsxynk66x6h4c1ds6anq8b37mjjf"))

(define rust-serde-derive-1.0.215
  (crate-source "serde_derive" "1.0.215"
                "1h2nswy0rmzblil38h12wxsgni1ik63rk22wy19g48v9hrpqc7md"))

(define rust-serde-derive-1.0.216
  (crate-source "serde_derive" "1.0.216"
                "0pm5bm4354n40ir12bbs829arlqwjrw0wmzd4xk5r1kkpzdmky26"))

(define rust-serde-derive-1.0.218
  (crate-source "serde_derive" "1.0.218"
                "0azqd74xbpb1v5vf6w1fdbgmwp39ljjfj25cib5rgrzlj7hh75gh"))

(define rust-serde-derive-1.0.219
  (crate-source "serde_derive" "1.0.219"
                "001azhjmj7ya52pmfiw4ppxm16nd44y15j2pf5gkcwrcgz7pc0jv"))

(define rust-serde-derive-1.0.225
  (crate-source "serde_derive" "1.0.225"
                "05j5zj2jdba3jnm7kh3fpljmhngmsa8pp5x495lpc7wbyynkda8f"))

(define rust-serde-derive-1.0.226
  (crate-source "serde_derive" "1.0.226"
                "0cyvkilp34an3f90b0idw0jjsyq20h7v47gsp8qkfmrl5zi3mdcd"))

(define rust-serde-derive-1.0.228
  (crate-source "serde_derive" "1.0.228"
                "0y8xm7fvmr2kjcd029g9fijpndh8csv5m20g4bd76w8qschg4h6m"))

(define rust-serde-derive-internals-0.29.1
  (crate-source "serde_derive_internals" "0.29.1"
                "04g7macx819vbnxhi52cx0nhxi56xlhrybgwybyy7fb9m4h6mlhq"))

(define rust-serde-ignored-0.1.11
  (crate-source "serde_ignored" "0.1.11"
                "0xhdf1qd3c8blczag31b8f0v231nwpq1ywxk52bh0bp9h1yscvan"))

(define rust-serde-ignored-0.1.12
  (crate-source "serde_ignored" "0.1.12"
                "1zv50daj0yk1lh61im1f4zxks5biswlbb9sqssv3ad8ymifl85mm"))

(define rust-serde-json-1.0.105
  (crate-source "serde_json" "1.0.105"
                "0q73miix7rmja8vss57s42dm7a2xpngyx7gcpinksmi7mkhm2cb9"))

(define rust-serde-json-1.0.133
  (crate-source "serde_json" "1.0.133"
                "0xz3bswa527wln3fy0qb7y081nx3cp5yy1ggjhi6n5mrfcjfpz67"))

(define rust-serde-json-1.0.139
  (crate-source "serde_json" "1.0.139"
                "19kj3irpa22a7djz1jaf4wambzh7psiqa6zyafqnb76crhx6ry24"))

(define rust-serde-json-1.0.140
  (crate-source "serde_json" "1.0.140"
                "0wwkp4vc20r87081ihj3vpyz5qf7wqkqipq17v99nv6wjrp8n1i0"))

(define rust-serde-json-1.0.143
  (crate-source "serde_json" "1.0.143"
                "0njabwzldvj13ykrf1aaf4gh5cgl25kf9hzbpafbv3qh3ppsn0fl"))

(define rust-serde-json-1.0.145
  (crate-source "serde_json" "1.0.145"
                "1767y6kxjf7gwpbv8bkhgwc50nhg46mqwm9gy9n122f7v1k6yaj0"))

(define rust-serde-json-1.0.99
  (crate-source "serde_json" "1.0.99"
                "1qzal5a1wlfw587xqfwngly0nhrkzqi7d1rva27hp820q9qnh9j6"))

(define rust-serde-jsonrc-0.1.0
  (crate-source "serde_jsonrc" "0.1.0"
                "1a5q0ba7jhgfl63l16plgl7sx1k58l2caxf7z2j5l677rh5yk4dm"))

(define rust-serde-norway-0.9.42
  (crate-source "serde_norway" "0.9.42"
                "130nx1r3nwydglq1yrrcydavd6w5zj219zsimc7m1zdmi6ag4274"))

(define rust-serde-path-to-error-0.1.20
  (crate-source "serde_path_to_error" "0.1.20"
                "0mxls44p2ycmnxh03zpnlxxygq42w61ws7ir7r0ba6rp5s1gza8h"))

(define rust-serde-spanned-1.0.1
  (crate-source "serde_spanned" "1.0.1"
                "1rh9w15agr8hgljjm1l4p2ykihx38nmipsji3fzc8gx52d527297"))

(define rust-serde-spanned-1.0.2
  (crate-source "serde_spanned" "1.0.2"
                "1vh4kcnzhw0fbr1jhg41p8yybnp5gmpnh171fy25bgn2a8s7h5sl"))

(define rust-serde-untagged-0.1.8
  (crate-source "serde-untagged" "0.1.8"
                "1wzgb2qq79jg14kcnq859khv2f80hx2af1q9vypz3inbkdi6m0rl"))

(define rust-serde-path-to-error-0.1.17
  (crate-source "serde_path_to_error" "0.1.17"
                "0alb447z25dvczd6ll3vfjbf51pypn23mgs5hv8978vzjczv3yjr"))

(define rust-serde-plain-1.0.2
  (crate-source "serde_plain" "1.0.2"
                "0l4d4nbw00pz6n43icrc605bhgynfmlyq39sn8i10qasnrnzrqcw"))

(define rust-serde-regex-1.1.0
  (crate-source "serde_regex" "1.1.0"
                "1pxsnxb8c198szghk1hvzvhva36w2q5zs70hqkmdf5d89qd6y4x8"))

(define rust-serde-repr-0.1.20
  (crate-source "serde_repr" "0.1.20"
                "1755gss3f6lwvv23pk7fhnjdkjw7609rcgjlr8vjg6791blf6php"))

(define rust-serde-spanned-0.6.3
  (crate-source "serde_spanned" "0.6.3"
                "11p1l83r5g3k18pi88cqri2r9ai03pm8b4azj4j02ypx6scnqhln"))

(define rust-serde-spanned-0.6.8
  (crate-source "serde_spanned" "0.6.8"
                "1q89g70azwi4ybilz5jb8prfpa575165lmrffd49vmcf76qpqq47"))

(define rust-serde-spanned-0.6.9
  (crate-source "serde_spanned" "0.6.9"
                "18vmxq6qfrm110caszxrzibjhy2s54n1g5w1bshxq9kjmz7y0hdz"))

(define rust-serde-test-1.0.177
  (crate-source "serde_test" "1.0.177"
                "1vgisk4dgwdmz4prc2iwq8lddrp4pbqdbljk0h3b7dnafgjix43z"))

(define rust-serde-untagged-0.1.7
  (crate-source "serde-untagged" "0.1.7"
                "0vp6gvl4a40rzsxdn5js5bn5sqac7rqdbp8an55dnrnlswcrr799"))

(define rust-serde-urlencoded-0.7.1
  (crate-source "serde_urlencoded" "0.7.1"
                "1zgklbdaysj3230xivihs30qi5vkhigg323a9m62k8jwf4a1qjfk"))

(define rust-serde-value-0.7.0
  (crate-source "serde-value" "0.7.0"
                "0b18ngk7n4f9zmwsfdkhgsp31192smzyl5z143qmx1qi28sa78gk"))

(define rust-serde-wasm-bindgen-0.3.1
  (crate-source "serde-wasm-bindgen" "0.3.1"
                "1xzh1a3rhapirlca4rbvh6bycphxg5yjmdwjsqij3hl6wpl6b0v1"))

(define rust-serde-with-2.3.3
  (crate-source "serde_with" "2.3.3"
                "1gmgl2raxxnpvjgcvkrimgv55kbjywi5xbffc9rvb9s7q7973zq7"))

(define rust-serde-with-3.12.0
  (crate-source "serde_with" "3.12.0"
                "1ai9c3cbdgrsvmlc4qpg9z73y80yplk3k7zp45wp97xnzkrggdnn"))

(define rust-serde-with-3.14.0
  (crate-source "serde_with" "3.14.0"
                "1manlm83865xwlvgv8frc472x19b75pd89a54mpxpagg3zb5ri7j"))

(define rust-serde-with-macros-2.3.3
  (crate-source "serde_with_macros" "2.3.3"
                "0pv9wy1aihsaklg83qac4q0ksimbxcwwk56lwla35l8p3f46y6w8"))

(define rust-serde-with-macros-3.12.0
  (crate-source "serde_with_macros" "3.12.0"
                "13hznly0qq1rngsdh8gpnajab2knkrmvwwrbmii86g1s36jwl04d"))

(define rust-serde-with-macros-3.14.0
  (crate-source "serde_with_macros" "3.14.0"
                "03xk9ghj2s6n331r565mgh22w0749vnq50094nd0vkk5cmg9946y"))

(define rust-serde-yaml-0.8.26
  (crate-source "serde_yaml" "0.8.26"
                "06y7gxy312mink8nsnmci9cw0ykpgsdcxmayg0snmdbnnwrp92jp"))

(define rust-serde-yaml-0.9.34+deprecated
  (crate-source "serde_yaml" "0.9.34+deprecated"
                "0isba1fjyg3l6rxk156k600ilzr8fp7crv82rhal0rxz5qd1m2va"))

(define rust-serial-0.4.0
  (crate-source "serial" "0.4.0"
                "11iyvc1z123hn7zl6bk5xpf6xdlsb33qh6xa7g0pghqgayb7l8x1"))

(define rust-serial-core-0.4.0
  (crate-source "serial-core" "0.4.0"
                "10a5lvllz3ljva66bqakrn8cxb3pkaqyapqjw9x760al6jdj0iiz"))

(define rust-serial-test-0.5.1
  (crate-source "serial_test" "0.5.1"
                "0pchc7imdi9wv8xxnwkb9lzs6cg06ghs0gaajjb834y8837wpg70"))

(define rust-serial-test-1.0.0
  (crate-source "serial_test" "1.0.0"
                "04864v5ain4nan2k5l32sr3bxpg0sfxxs2iki3xxcq78g9s3132k"))

(define rust-serial-test-2.0.0
  (crate-source "serial_test" "2.0.0"
                "0b9v0csv9wxl1gcjq99plwimxbmhgr6kzbwqyb457qh3d22xsmhf"))

(define rust-serial-test-3.2.0
  (crate-source "serial_test" "3.2.0"
                "1a8zg87gi28952hzj363ykwd8p1ssrakl1gi3f4xdqa4y84q298v"))

(define rust-serial-test-derive-0.5.1
  (crate-source "serial_test_derive" "0.5.1"
                "1m8sd97xr8dn6p9by0xwfqm0rz8cbn1ghs5l1fv1xd6xzvgddb5j"))

(define rust-serial-test-derive-1.0.0
  (crate-source "serial_test_derive" "1.0.0"
                "0scscldvlz3an9v0spcizaqp5wa2y4w15bk4ink8jpgq2pgq76h7"))

(define rust-serial-test-derive-2.0.0
  (crate-source "serial_test_derive" "2.0.0"
                "13zvd5ds76hhjn3z0axc05n15lzpxpz77jcykic8q5knhlbjklci"))

(define rust-serial-test-derive-3.2.0
  (crate-source "serial_test_derive" "3.2.0"
                "1vwyz2k5kiy5jmba0fvp6ph8ia707801bz918n2ff7bm11d2csax"))

(define rust-serial-unix-0.4.0
  (crate-source "serial-unix" "0.4.0"
                "1dyaaca8g4q5qzc2l01yirzs6igmhc9agg4w8m5f4rnqr6jbqgzh"))

(define rust-serial-windows-0.4.0
  (crate-source "serial-windows" "0.4.0"
                "0ql1vjy57g2jf218bhmgr98i41faq0v5vzdx3g9payi6fsvx7ihm"))

(define rust-serialport-4.7.2
  (crate-source "serialport" "4.7.2"
                "0aqaig121lm034irzal6j9dyg7jpf4hczrjlmf5yzxka9ycbrc6d"))

(define rust-service-binding-3.0.0
  (crate-source "service-binding" "3.0.0"
                "1s6n10jcv9prdx33bajv7vdpli7507yv7vc367w3w6c3iffknri5"))

(define rust-servo-arc-0.1.1
  (crate-source "servo_arc" "0.1.1"
                "0cjljr9znwahry6p95xvd3p4pmy24wlf6gbfidnmglg002w3i0nr"))

(define rust-servo-arc-0.3.0
  (crate-source "servo_arc" "0.3.0"
                "0i0s9786np106yl6w29bfzwnj29rqak912skcdxcf04yjlddfdnh"))

(define rust-sevenz-rust2-0.13.1
  (crate-source "sevenz-rust2" "0.13.1"
                "0spc0iy0kvs7grg2hlmh8p4dqs84hwzj9gr8jxhank75spvx09yx"
                #:snippet '(for-each delete-file-recursively '("examples" "tests"))))

(define rust-sha-1-0.10.1
  (crate-source "sha-1" "0.10.1"
                "1700fs5aiiailpd5h0ax4sgs2ngys0mqf3p4j0ry6j2p2zd8l1gm"
                #:snippet '(delete-file-recursively "tests")))

(define rust-sha-1-0.8.2
  (crate-source "sha-1" "0.8.2"
                "1pv387q0r7llk2cqzyq0nivzvkgqgzsiygqzlv7b68z9xl5lvngp"))

(define rust-sha-1-0.9.8
  (crate-source "sha-1" "0.9.8"
                "19jibp8l9k5v4dnhj5kfhaczdfd997h22qz0hin6pw9wvc9ngkcr"))

(define rust-sha1-0.10.6
  (crate-source "sha1" "0.10.6"
                "1fnnxlfg08xhkmwf2ahv634as30l1i3xhlhkvxflmasi5nd85gz3"
                #:snippet '(delete-file-recursively "tests")))

(define rust-sha1-0.6.1
  (crate-source "sha1" "0.6.1"
                "0w1p0s9060cv1vlgfa5c93kjksmvzjjc8j780lns3jj5fk4hbnn1"))

(define rust-sha1-asm-0.5.3
  (crate-source "sha1-asm" "0.5.3"
                "0asqxlxf5li7ac9mi49qj890rzsfb5px5ynzmqq12z5nz2xcwsi8"))

(define rust-sha1-checked-0.10.0
  (crate-source "sha1-checked" "0.10.0"
                "08s4h1drgwxzfn1mk11rn0r9i0rbjra1m0l2c0fbngij1jn9kxc9"
                #:snippet '(delete-file-recursively "tests")))

(define rust-sha1-smol-1.0.1
  (crate-source "sha1_smol" "1.0.1"
                "0pbh2xjfnzgblws3hims0ib5bphv7r5rfdpizyh51vnzvnribymv"))

(define rust-sha1collisiondetection-0.3.4
  (crate-source "sha1collisiondetection" "0.3.4"
                "0jwnwrk1x78v9r16jnac2s4v1m2f5a19khwkx1vjh0d6whhn8q0z"
                #:snippet '(delete-file-recursively "test")))

(define rust-sha2-0.10.8
  (crate-source "sha2" "0.10.8"
                "1j1x78zk9il95w9iv46dh9wm73r6xrgj32y6lzzw7bxws9dbfgbr"
                #:snippet '(delete-file-recursively "tests")))

(define rust-sha2-0.10.9
  (crate-source "sha2" "0.10.9"
                "10xjj843v31ghsksd9sl9y12qfc48157j1xpb8v1ml39jy0psl57"
                #:snippet '(delete-file-recursively "tests")))

(define rust-sha2-0.9.9
  (crate-source "sha2" "0.9.9"
                "006q2f0ar26xcjxqz8zsncfgz86zqa5dkwlwv03rhx1rpzhs2n2d"
                #:snippet '(delete-file-recursively "tests")))

(define rust-sha256-1.6.0
  (crate-source "sha256" "1.6.0"
                "1xn86nlyq9mr4a8ybfw3z3275rnh58mb83phjfbp1sxxca2zr07q"))

(define rust-sha3-0.10.8
  (crate-source "sha3" "0.10.8"
                "0q5s3qlwnk8d5j34jya98j1v2p3009wdmnqdza3yydwgi8kjv1vm"
                #:snippet '(delete-file-recursively "tests")))

(define rust-shadow-rs-0.35.2
  (crate-source "shadow-rs" "0.35.2"
                "11w6677ymsb69fy716adf51zjbzwgkkyf5arz41563gcnf7k5cpi"))

(define rust-shadow-rs-0.38.1
  (crate-source "shadow-rs" "0.38.1"
                "12gd7bq9lmpc1qgmr1q3paglwkf0az39jhkc9bvlp7y2k33lrhbf"))

(define rust-shannon-0.2.0
  (crate-source "shannon" "0.2.0"
                "0qa52zs4y1i87ysr11g9p6shpdagl14bb340gfm6rd97jhfb99by"))

(define rust-sharded-slab-0.1.7
  (crate-source "sharded-slab" "0.1.7"
                "1xipjr4nqsgw34k7a2cgj9zaasl2ds6jwn89886kww93d32a637l"))

(define rust-shared-child-1.0.1
  (crate-source "shared_child" "1.0.1"
                "035679h89ppqcfkjzgz9bb2hdlkw5wjv598l310xz8frmqw97yh9"))

(define rust-shared-child-1.0.2
  (crate-source "shared_child" "1.0.2"
                "1p7998vrva9h2srb0y8ybpw3vwa2w6z5fs88dilf1fwi57apnaby"))

(define rust-shared-library-0.1.9
  (crate-source "shared_library" "0.1.9"
                "04fs37kdak051hm524a360978g58ayrcarjsbf54vqps5c7px7js"))

(define rust-shell-escape-0.1.5
  (crate-source "shell-escape" "0.1.5"
                "0kqq83dk0r1fqj4cfzddpxrni2hpz5i1y607g366c4m9iyhngfs5"))

(define rust-shell-quote-0.7.2
  (crate-source "shell-quote" "0.7.2"
                "1r6pxib3hs700ik79c707k88yr57ki9a27sj4mc3dqjsjwajcl7v"))

(define rust-shell-words-1.1.0
  (crate-source "shell-words" "1.1.0"
                "1plgwx8r0h5ismbbp6cp03740wmzgzhip85k5hxqrrkaddkql614"))

(define rust-shellexpand-3.1.0
  (crate-source "shellexpand" "3.1.0"
                "0jz1i14ziz8gbyj71212s7dqrw6q96f25i48zkmy66fcjhxzl0ys"))

(define rust-shellexpand-3.1.1
  (crate-source "shellexpand" "3.1.1"
                "1fwhid2r117rbis2f6lj8mpfjf0w6lq6nqfxjha86cb3vmjxy7wb"))

(define rust-shlex-1.2.0
  (crate-source "shlex" "1.2.0"
                "1033pj9dyb76nm5yv597nnvj3zpvr2aw9rm5wy0gah3dk99f1km7"))

(define rust-shlex-1.3.0
  (crate-source "shlex" "1.3.0"
                "0r1y6bv26c1scpxvhg2cabimrmwgbp4p3wy6syj9n0c4s3q2znhg"))

(define rust-signal-hook-0.1.17
  (crate-source "signal-hook" "0.1.17"
                "0a97migr3gcy4sfkgxdp29082s3f3lbf4was3dkpl13gq51d8cby"))

(define rust-signal-hook-0.3.17
  (crate-source "signal-hook" "0.3.17"
                "0098nsah04spqf3n8niirmfym4wsdgjl57c78kmzijlq8xymh8c6"))

(define rust-signal-hook-0.3.18
  (crate-source "signal-hook" "0.3.18"
                "1qnnbq4g2vixfmlv28i1whkr0hikrf1bsc4xjy2aasj2yina30fq"))

(define rust-signal-hook-mio-0.2.4
  (crate-source "signal-hook-mio" "0.2.4"
                "1k8pl9aafiadr4czsg8zal9b4jdk6kq5985p90i19jc5sh31mnrl"))

(define rust-signal-hook-registry-1.4.2
  (crate-source "signal-hook-registry" "1.4.2"
                "1cb5akgq8ajnd5spyn587srvs4n26ryq0p78nswffwhv46sf1sd9"))

(define rust-signal-hook-registry-1.4.3
  (crate-source "signal-hook-registry" "1.4.3"
                "0kxzh7qmvmzg93qwhxg3m0yvmk7izwd99xcyan1jzn52x2plx2y2"))

(define rust-signal-hook-registry-1.4.5
  (crate-source "signal-hook-registry" "1.4.5"
                "042lkqrpnlrgvrrcirgigxyp1zk70d8v0fsr5w7a18k3bw2vh0wj"))

(define rust-signal-hook-registry-1.4.6
  (crate-source "signal-hook-registry" "1.4.6"
                "12y2v1ms5z111fymaw1v8k93m5chnkp21h0jknrydkj8zydp395j"))

(define rust-signal-hook-tokio-0.3.1
  (crate-source "signal-hook-tokio" "0.3.1"
                "07nggsi80jv39xisdk2r7cik7hx2d2qa2sivvqkpxqxidzvl2ci1"))

(define rust-signals-receipts-0.2.4
  (crate-source "signals_receipts" "0.2.4"
                "0kx1ljf2kax7ic1ckzhd1cj3qdbqyfhd51z4bpnxv7va0w33gprs"))

(define rust-signature-1.6.4
  (crate-source "signature" "1.6.4"
                "0z3xg405pg827g6hfdprnszsdqkkbrsfx7f1dl04nv9g7cxks8vl"))

(define rust-signature-2.2.0
  (crate-source "signature" "2.2.0"
                "1pi9hd5vqfr3q3k49k37z06p7gs5si0in32qia4mmr1dancr6m3p"))

(define rust-sigpipe-0.1.3
  (crate-source "sigpipe" "0.1.3"
                "1rnqcgbx2mv3w11y6vf05a8a3y6jyqwmwa0hhafi6j6kw2rvz12m"))

(define rust-simba-0.5.1
  (crate-source "simba" "0.5.1"
                "0p1x1ndajy4j3dr9zbh79cz5k0hbj4p9bagd7cj00gc5aws0d0lf"))

(define rust-simba-0.6.0
  (crate-source "simba" "0.6.0"
                "0px0nncs3ki86pjcldz40mhvraywh7y9jypfcqqdcihs287q9dzh"))

(define rust-simba-0.8.1
  (crate-source "simba" "0.8.1"
                "1bnf7ainywmaz2z67ss1q0bjwccf80c50c50r6hlpay69z4hf586"))

(define rust-simd-abstraction-0.7.1
  (crate-source "simd-abstraction" "0.7.1"
                "11v9hy8qg0b4qypz2p75ijv41ln1rssk6qilz0gwbbfaayfb5bcw"))

(define rust-simd-adler32-0.3.7
  (crate-source "simd-adler32" "0.3.7"
                "1zkq40c3iajcnr5936gjp9jjh1lpzhy44p3dq3fiw75iwr1w2vfn"))

(define rust-simd-helpers-0.1.0
  (crate-source "simd_helpers" "0.1.0"
                "19idqicn9k4vhd04ifh2ff41wvna79zphdf2c81rlmpc7f3hz2cm"))

(define rust-simdutf8-0.1.4
  (crate-source "simdutf8" "0.1.4"
                "0fi6zvnldaw7g726wnm9vvpv4s89s5jsk7fgp3rg2l99amw64zzj"))

(define rust-simdutf8-0.1.5
  (crate-source "simdutf8" "0.1.5"
                "0vmpf7xaa0dnaikib5jlx6y4dxd3hxqz6l830qb079g7wcsgxag3"))

(define rust-similar-2.7.0
  (crate-source "similar" "2.7.0"
                "1aidids7ymfr96s70232s6962v5g9l4zwhkvcjp4c5hlb6b5vfxv"))

(define rust-simple-logger-4.3.3
  (crate-source "simple_logger" "4.3.3"
                "1w9ypyn5n7bxw6aylbkwz3hfsjpvkx0qm2xj11yx8l82r744czlf"))

(define rust-simplecss-0.2.2
  (crate-source "simplecss" "0.2.2"
                "0v0kid7b2602kcka2x2xs9wwfjf8lnvpgpl8x287qg4wra1ni73s"))

(define rust-simplelog-0.12.2
  (crate-source "simplelog" "0.12.2"
                "1h59cp84gwdmbxiljq6qmqq1x3lv9ikc1gb32f5ya7pgzbdpl98n"))

(define rust-siphasher-0.3.11
  (crate-source "siphasher" "0.3.11"
                "03axamhmwsrmh0psdw3gf7c0zc4fyl5yjxfifz9qfka6yhkqid9q"))

(define rust-siphasher-1.0.1
  (crate-source "siphasher" "1.0.1"
                "17f35782ma3fn6sh21c027kjmd227xyrx06ffi8gw4xzv9yry6an"))

(define rust-size-format-1.0.2
  (crate-source "size_format" "1.0.2"
                "0fxjl0rc0x7yc14x885dh7jjf2jrlhpwf66akp3dxii246mzdmbf"))

(define rust-sized-chunks-0.6.5
  (crate-source "sized-chunks" "0.6.5"
                "07ix5fsdnpf2xsb0k5rbiwlmsicm2237fcx7blirp9p7pljr5mhn"))

(define rust-skeptic-0.13.7
  (crate-source "skeptic" "0.13.7"
                "1a205720pnss0alxvbx0fcn3883cg3fbz5y1047hmjbnaq0kplhn"))

(define rust-slab-0.4.10
  (crate-source "slab" "0.4.10"
                "03f5a9gdp33mngya4qwq2555138pj74pl015scv57wsic5rikp04"))

(define rust-slab-0.4.11
  (crate-source "slab" "0.4.11"
                "12bm4s88rblq02jjbi1dw31984w61y2ldn13ifk5gsqgy97f8aks"))

(define rust-slab-0.4.9
  (crate-source "slab" "0.4.9"
                "0rxvsgir0qw5lkycrqgb1cxsvxzjv9bmx73bk5y42svnzfba94lg"))

(define rust-sled-0.34.7
  (crate-source "sled" "0.34.7"
                "0dcr2s7cylj5mb33ci3kpx7fz797jwvysnl5airrir9cgirv95kz"
                #:snippet '(delete-file-recursively "art")))

(define rust-slice-group-by-0.3.1
  (crate-source "slice-group-by" "0.3.1"
                "19vbyyxqvc25fv2dmhlxijlk5sa9j34yb6hyydb9vf89kh36fqc2"))

(define rust-slip-codec-0.4.0
  (crate-source "slip-codec" "0.4.0"
                "0nwnwywkgjvzyismfzvv9n204afz8xrqc76lgikrp39z8j9bd577"))

(define rust-slog-2.7.0
  (crate-source "slog" "2.7.0"
                "01ldk4yarx7x4y4rgsf4kmrcy3wrpcxdd53v2lkk355x9rnh8iw3"))

(define rust-slog-async-2.8.0
  (crate-source "slog-async" "2.8.0"
                "113b17aw7jx7mr68vwfq2yiv6mb4702hz6a0g587jb4ai67h7j3j"))

(define rust-slog-kvfilter-0.7.0
  (crate-source "slog-kvfilter" "0.7.0"
                "1q3mq6a9aq8iscy9xh97zifxrxa6y10d8p2gkxlxkvk9s7brx4xf"))

(define rust-slog-scope-4.4.0
  (crate-source "slog-scope" "4.4.0"
                "11n7nd0g3iab8ahcwnxzpmchi4ycgjsq5nj9jn3d4k17qfsa959g"))

(define rust-slog-stdlog-4.1.1
  (crate-source "slog-stdlog" "4.1.1"
                "0gpsf62ckblpc6a70dnhsz677c7s5cz4glpqsf8p5bmvwnnb41k7"))

(define rust-slog-term-2.9.1
  (crate-source "slog-term" "2.9.1"
                "1s0h8qhqnvy5a7m7gmnca2a2d5m5a4sz1hc26xfgxawqp7825q5n"))

(define rust-sloggers-2.2.0
  (crate-source "sloggers" "2.2.0"
                "16yvn2v8vs7bsg11vnma03ml6grrwfm3qqikwrdd8b5q70kjq1km"))

(define rust-slotmap-1.0.7
  (crate-source "slotmap" "1.0.7"
                "0amqb2fn9lcy1ri0risblkcp88dl0rnfmynw7lx0nqwza77lmzyv"))

(define rust-slug-0.1.6
  (crate-source "slug" "0.1.6"
                "0977cyp88xrwbpmqwzafkvv8vm9i0gdb5zjskb6f6pg45vvq0al8"))

(define rust-smallbitvec-2.6.0
  (crate-source "smallbitvec" "2.6.0"
                "0gqf2nj32gwrk5v4xx9v7b8c2vfanqm1h8i994d0wmhqs4yjc7fk"))

(define rust-smallvec-1.11.0
  (crate-source "smallvec" "1.11.0"
                "1y9g8jcsizjbsiilgplrnavy8pd3cliy40pqgrq9zpczwkp4zfv2"))

(define rust-smallvec-1.11.1
  (crate-source "smallvec" "1.11.1"
                "0nmx8aw3v4jglqdcjv4hhn10d6g52c4bhjlzwf952885is04lawl"))

(define rust-smallvec-1.13.2
  (crate-source "smallvec" "1.13.2"
                "0rsw5samawl3wsw6glrsb127rx6sh89a8wyikicw6dkdcjd1lpiw"))

(define rust-smallvec-1.14.0
  (crate-source "smallvec" "1.14.0"
                "1z8wpr53x6jisklqhkkvkgyi8s5cn69h2d2alhqfxahzxwiq7kvz"))

(define rust-smallvec-1.15.0
  (crate-source "smallvec" "1.15.0"
                "1sgfw8z729nlxk8k13dhs0a762wnaxmlx70a7xlf3wz989bjh5w9"))

(define rust-smallvec-1.15.1
  (crate-source "smallvec" "1.15.1"
                "00xxdxxpgyq5vjnpljvkmy99xij5rxgh913ii1v16kzynnivgcb7"))

(define rust-smart-default-0.7.1
  (crate-source "smart-default" "0.7.1"
                "1hgzs1250559bpayxmn46gzas5ycqn39wkf4srjgqh4461k1ic0f"))

(define rust-smartstring-1.0.1
  (crate-source "smartstring" "1.0.1"
                "0agf4x0jz79r30aqibyfjm1h9hrjdh0harcqcvb2vapv7rijrdrz"))

(define rust-smawk-0.3.1
  (crate-source "smawk" "0.3.1"
                "0hv0q1mw1r1brk7v3g4a80j162p7g1dri4bdidykrakzfqjd4ypn"))

(define rust-smawk-0.3.2
  (crate-source "smawk" "0.3.2"
                "0344z1la39incggwn6nl45k8cbw2x10mr5j0qz85cdz9np0qihxp"))

(define rust-smithay-0.6.0.ede2707 rust-smithay-0.6.0.ede2707)

(define rust-smithay-0.7.0.20d2dac rust-smithay-0.7.0.20d2dac)

(define rust-smithay-drm-extras-0.1.0.20d2dac rust-smithay-0.7.0.20d2dac)

(define rust-smithay-client-toolkit-0.15.4
  (crate-source "smithay-client-toolkit" "0.15.4"
                "18wxla80y6m4l3dwawi7bl1d9m9dfcg4sxxjcgjqq3psjxmg2a4a"))

(define rust-smithay-client-toolkit-0.19.2
  (crate-source "smithay-client-toolkit" "0.19.2"
                "05h05hg4dn3v6br5jbdbs5nalk076a64s7fn6i01nqzby2hxwmrl"))

(define rust-smithay-clipboard-0.7.2
  (crate-source "smithay-clipboard" "0.7.2"
                "0inqnr4x458nqz0wdcanbgn77dad3bj0m7pjj074lrv3qkp1d0nc"))

(define rust-smithay-drm-extras-0.1.0.ede2707 rust-smithay-0.6.0.ede2707)

(define rust-smol-str-0.1.24
  (crate-source "smol_str" "0.1.24"
                "1j891lgnflvnzgbs7fhwd6sxrrx47ii5mj0yy3f2f9mbrdbwimps"))

(define rust-smol-str-0.2.2
  (crate-source "smol_str" "0.2.2"
                "1bfylqf2vnqaglw58930vpxm2rfzji5gjp15a2c0kh8aj6v8ylyx"))

(define rust-smol-str-0.3.2
  (crate-source "smol_str" "0.3.2"
                "039mj6lc1vkljj17ndlzzkak8kvlmw8ppi6yjdxsh433snfbhxln"))

(define rust-snafu-0.8.5
  (crate-source "snafu" "0.8.5"
                "06ahz9g4f6475rspfcay8512x776wpxc205rizzc6a9abv492f12"))

(define rust-snafu-0.8.9
  (crate-source "snafu" "0.8.9"
                "18p1y5qxwjn5j902wqsdr75n17b29lxpdipa0p7a3wybxbsb713f"))

(define rust-snafu-derive-0.8.5
  (crate-source "snafu-derive" "0.8.5"
                "05zr38bcngn8ha4mfi7lr2pqqfysjhwhxvk9lz57xzkzjavwdhq3"))

(define rust-snafu-derive-0.8.9
  (crate-source "snafu-derive" "0.8.9"
                "0lg4s58jzx6w48ig4qp8jasrrs886pifqqd58k5b2jzlvd3pgjf1"))

(define rust-snap-1.1.1
  (crate-source "snap" "1.1.1"
                "0fxw80m831l76a5zxcwmz2aq7mcwc1pp345pnljl4cv1kbxnfsqv"))

(define rust-snapbox-0.4.17
  (crate-source "snapbox" "0.4.17"
                "114xa4v5icljaacmyj9i2jc8apq04ifihnx7xy4jikgvh1p1p0sb"))

(define rust-snapbox-0.6.21
  (crate-source "snapbox" "0.6.21"
                "0ss3nd9ky0fkq7idj7jzr22kvkhxz3ylrq9fmiq5sdg3h52zrp4n"))

(define rust-snapbox-macros-0.3.10
  (crate-source "snapbox-macros" "0.3.10"
                "1bv4lq1kw1vrd9lk7yk79a0z8q8nma2502ifysv1p913r99rymhn"))

(define rust-socket2-0.4.10
  (crate-source "socket2" "0.4.10"
                "03ack54dxhgfifzsj14k7qa3r5c9wqy3v6mqhlim99cc03y1cycz"))

(define rust-socket2-0.5.7
  (crate-source "socket2" "0.5.7"
                "070r941wbq76xpy039an4pyiy3rfj7mp7pvibf1rcri9njq5wc6f"))

(define rust-socket2-0.5.10
  (crate-source "socket2" "0.5.10"
                "0y067ki5q946w91xlz2sb175pnfazizva6fi3kfp639mxnmpc8z2"))

(define rust-socket2-0.5.9
  (crate-source "socket2" "0.5.9"
                "1vzds1wwwi0a51fn10r98j7cx3ir4shvhykpbk7md2h5h1ydapsg"))

(define rust-socket2-0.6.0
  (crate-source "socket2" "0.6.0"
                "01qqdzfnr0bvdwq6wl56c9c4m2cvbxn43dfpcv8gjx208sph8d93"))

(define rust-socks-0.3.4
  (crate-source "socks" "0.3.4"
                "12ymihhib0zybm6n4mrvh39hj1dm0ya8mqnqdly63079kayxphzh"))

(define rust-socks5-client-0.4.1
  (crate-source "socks5-client" "0.4.1"
                "0k1jpbpc0wqqqykz27706vj6rmsqqr66l01kss15vmmizbvdrizz"))

(define rust-soft-assert-0.1.1
  (crate-source "soft_assert" "0.1.1"
                "15vxbp1j0n908yffvm3xacbcdl212j7k95md85ai663jxb3pw2dm"))

(define rust-soup-sys-0.10.0
  (crate-source "soup-sys" "0.10.0"
                "0gm1b1dj6z3l94sklw6jvqblhik8i8gz2mgrf6xhnqv5hpqaviy3"))

(define rust-sourcemap-6.4.1
  (crate-source "sourcemap" "6.4.1"
                "145xyp6qaai2gd5p92bkg7yzrm4n5mqx1y111vsnqmywlxfgdjz4"))

(define rust-spdx-0.10.8
  (crate-source "spdx" "0.10.8"
                "14r1bl3gmx7cj91l5r1qr5wildjacmzflw9cahgzrqk7v9b97djq"))

(define rust-speedate-0.15.0
  (crate-source "speedate" "0.15.0"
                "0dzryiwvdbdcz74x0hidb5g1c2aal4n9cxsdlqwxxrq79vgplpls"))

(define rust-spin-0.9.8
  (crate-source "spin" "0.9.8"
                "0rvam5r0p3a6qhc18scqpvpgb3ckzyqxpgdfyjnghh8ja7byi039"))

(define rust-spinners-4.1.1
  (crate-source "spinners" "4.1.1"
                "10jgzdy6x79ipnfhavn46zbg4hlx98mcfr7p4f4j774b6mzr9vx0"))

(define rust-spinning-top-0.3.0
  (crate-source "spinning_top" "0.3.0"
                "001kjbiz1gg111rsqxc4pq9a1izx7wshkk38f69h1dbgf4fjsvfr"
                #:snippet '(delete-file-recursively "img")))

(define rust-spki-0.7.3
  (crate-source "spki" "0.7.3"
                "17fj8k5fmx4w9mp27l970clrh5qa7r5sjdvbsln987xhb34dc7nr"
                #:snippet '(delete-file-recursively "tests")))

(define rust-spm-precompiled-0.1.4
  (crate-source "spm_precompiled" "0.1.4"
                "09pkdk2abr8xf4pb9kq3rk80dgziq6vzfk7aywv3diik82f6jlaq"))

(define rust-spmc-0.3.0
  (crate-source "spmc" "0.3.0"
                "1rgcqgj6b3d0cshi7277akr2xk0cx11rkmviaahy7a3pla6l5a02"))

(define rust-sptr-0.3.2
  (crate-source "sptr" "0.3.2"
                "0shddkys046nnrng929mrnjjrh31mlxl95ky7dgxd6i4kclkk6rv"))

(define rust-sqlite-0.32.0
  (crate-source "sqlite" "0.32.0"
                "1rpqpkpxn2qdvghsnak2b73cn5ca37p6ri0ylyjdcmrq3481r003"))

(define rust-sqlite-0.36.2
  (crate-source "sqlite" "0.36.2"
                "1lxyqhxb1nkifbgfwki0yn4dvi6paxkqfnbyw1dsj3138rfzyhv8"))

(define rust-sqlite3-src-0.5.1
  (crate-source "sqlite3-src" "0.5.1"
                "0m74wrkpify3z0xvrw4i2yssn9m9sjwqa5ipk6aq6f7fl58mmjdz"
                #:snippet
                '(begin
                   ;; Only allow for linking to system sqlite3.
                   (delete-file-recursively "source")
                   (delete-file "build.rs")
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main (){~@
                             println!(\"cargo:rustc-link-lib=sqlite3\");~@
                             }~%"))))))

(define rust-sqlite3-src-0.6.1
  (crate-source "sqlite3-src" "0.6.1"
                "0jkvjhgrfsq5m2ps3hh792mamwv8v6kf2gdj3wldn9vwyxnllk8p"
                #:snippet
                '(begin
                   ;; Only allow for linking to system sqlite3.
                   (delete-file-recursively "source")
                   (delete-file "build.rs")
                   (with-output-to-file "build.rs"
                     (lambda _
                       (format #t "fn main (){~@
                             println!(\"cargo:rustc-link-lib=sqlite3\");~@
                             }~%"))))))

(define rust-sqlite3-sys-0.15.2
  (crate-source "sqlite3-sys" "0.15.2"
                "0fq6m21dnd5yqrzknsmnl2565nahdwa29s7x12xhxr1kjik2qxgj"))

(define rust-sqlite3-sys-0.17.0
  (crate-source "sqlite3-sys" "0.17.0"
                "1rdds3kzxbxwy3lpsvgy7g8nh609nzqpxv4jvj23ag0c16kss09r"))

(define rust-sqlx-0.8.6
  (crate-source "sqlx" "0.8.6"
                "1p4pgppy10ch4vj0fyay9q3g8y5xhzsayyfrrnfncacli69vivqz"
                #:snippet '(delete-file-recursively "tests")))

(define rust-sqlx-core-0.8.6
  (crate-source "sqlx-core" "0.8.6"
                "1ildwsjy7lwfxsvfh174jwhk0rjqvyw37h87q1lhyslbhfqrhrzf"))

(define rust-sqlx-macros-0.8.6
  (crate-source "sqlx-macros" "0.8.6"
                "0pbiwsv5ysv3qcx1g4p1pvsqlz0xp67k9g5xw3szpb6aijc55m52"))

(define rust-sqlx-macros-core-0.8.6
  (crate-source "sqlx-macros-core" "0.8.6"
                "16r1slvkzfdxjkc2v5i3yd5l4xzcwbcy35hzfihmmb14262c3a8r"))

(define rust-sqlx-mysql-0.8.6
  (crate-source "sqlx-mysql" "0.8.6"
                "09n5k60z9j1ilbdmggcla6s27np3zwxc3fnbzsw4wy6z7003y05a"))

(define rust-sqlx-postgres-0.8.6
  (crate-source "sqlx-postgres" "0.8.6"
                "0insvvaql0pz6nk64dbss4q4qzilj7zh2j0m9cc7rw1wlpazqn6v"))

(define rust-sqlx-sqlite-0.8.6
  (crate-source "sqlx-sqlite" "0.8.6"
                "1siy1jhqf5flpxyrmy7rw66j0y0v2l7zjmc00c0l86rc1gkjzlf2"))

(define rust-ssh-agent-lib-0.5.1
  (crate-source "ssh-agent-lib" "0.5.1"
                "1fmbwz0qxds2w7rsvcg9kqm9x3wwifll8wxcc3g43k2mh357rxps"
                #:snippet '(delete-file-recursively "tests")))

(define rust-ssh-cipher-0.2.0
  (crate-source "ssh-cipher" "0.2.0"
                "0kvq113x9fcy2jcxp00xk472zxm8d9zxxz2vyqx3rlzh88ki7b6a"))

(define rust-ssh-encoding-0.2.0
  (crate-source "ssh-encoding" "0.2.0"
                "05aavlhk68vm60vbw8lcgx1p5wry367ck8niij7af221xywl54pb"))

(define rust-ssh-key-0.6.7
  (crate-source "ssh-key" "0.6.7"
                "1hx8as8rvnk31ncqg7dlqgcw9bmngkznn3xamf6d010ggwlzb1iv"))

(define rust-ssh2-config-0.5.4
  (crate-source "ssh2-config" "0.5.4"
                "0sw698aaar25pici7xxdic36kxmnriy48zkd59h1yhgxfr191j17"))

(define rust-st-map-0.1.8
  (crate-source "st-map" "0.1.8"
                "1vdmg8sr3iynkblcd97pl4yslisdnn7lgm4dlpab0xph6lc8k7gh"))

(define rust-stability-0.2.1
  (crate-source "stability" "0.2.1"
                "1b7w6qknq0w5y7s358j62pzi9kbh6g73lal3jx9aydpikl0ff16r"))

(define rust-stable-deref-trait-1.2.0
  (crate-source "stable_deref_trait" "1.2.0"
                "1lxjr8q2n534b2lhkxd6l6wcddzjvnksi58zv11f9y0jjmr15wd8"))

(define rust-stacker-0.1.20
  (crate-source "stacker" "0.1.20"
                "1ab039lan55s3vrb5vcldfdfzfajk52vyy344q09rc5rzq0r47v0"))

(define rust-stacker-0.1.21
  (crate-source "stacker" "0.1.21"
                "0syhx9fiydyd901v464saqd6z8f31a6is24iv94hfxyv5pihgnyd"))

(define rust-starship-battery-0.10.1
  (crate-source "starship-battery" "0.10.1"
                "1k7xz7i9p543bi54h49hbibgal2yp5bw884m336skjx2hqjwj2yf"))

(define rust-starship-battery-0.9.1
  (crate-source "starship-battery" "0.9.1"
                "0gzldhpfhqh12wgrahfdl2vma3qmrv434c4xcj7qndclcxs1ayfs"))

(define rust-state-0.6.0
  (crate-source "state" "0.6.0"
                "1n3n2h324h1y5zhaajh6kplvzfvg1l6hsr8siggmf4yq8m24m31b"))

(define rust-static-assertions-1.1.0
  (crate-source "static_assertions" "1.1.0"
                "0gsl6xmw10gvn3zs1rv99laj5ig7ylffnh71f9l34js4nr4r7sx2"))

(define rust-static-map-macro-0.2.5
  (crate-source "static-map-macro" "0.2.5"
                "16c5xw3krap14d9mhb3z8vkh3x9k2kkf543q05dhixy9zfcdaqmq"))

(define rust-statistical-1.0.0
  (crate-source "statistical" "1.0.0"
                "0wm35q6p6jcq1r1darczv4if7qss460kd391nlw5x3hjpc17kma9"))

(define rust-statrs-0.13.0
  (crate-source "statrs" "0.13.0"
                "0r00b60zlsn6srb6m6bzbw3w5cyihcy4w2rfjav64x4viy5bad0y"))

(define rust-statrs-0.15.0
  (crate-source "statrs" "0.15.0"
                "01bggaq9n09ch20r4yq9s2c4y54367nd71asg22nl8bq9s7bpg85"))

(define rust-statrs-0.16.1
  (crate-source "statrs" "0.16.1"
                "08bp7n3rwk41r11ynwl5x7xdc9cv85zw4r7ww117mhfsp8nhcnmk"))

(define rust-statrs-0.17.1
  (crate-source "statrs" "0.17.1"
                "0mbjasnlh7xqa1rvq48xffqxnc53hgjlgqjd0ifa58068rza15zn"))

(define rust-status-line-0.2.0
  (crate-source "status-line" "0.2.0"
                "1hgb8smsil4bwlxc3ya4h25rl9q754sw83l5d9a0b0v0psdwj352"))

(define rust-std-prelude-0.2.12
  (crate-source "std_prelude" "0.2.12"
                "1ghcwnhnqn3rphyhlknmxpj5clzqva46z1vh25k5bpzzan2ff1w2"))

(define rust-stdweb-0.4.20
  (crate-source "stdweb" "0.4.20"
                "1md14n9rzxzdskz3hpgln8vxfwqsw2cswc0f5nslh4r82rmlj8nh"
                #:snippet '(delete-file-recursively "info")))

(define rust-stdweb-derive-0.5.3
  (crate-source "stdweb-derive" "0.5.3"
                "1vsh7g0gaxn4kxqq3knhymdn02p2pfxmnd2j0vplpj6c1yj60yn8"))

(define rust-stdweb-internal-macros-0.2.9
  (crate-source "stdweb-internal-macros" "0.2.9"
                "049fq8fl5ny9l5if2qv7kxwng7g6ns95h4fbm3zx360dmpv5zyjq"))

(define rust-stdweb-internal-runtime-0.1.5
  (crate-source "stdweb-internal-runtime" "0.1.5"
                "1h0nkppb4r8dbrbms2hw9n5xdcs392m0r5hj3b6lsx3h6fx02dr1"))

(define rust-stfu8-0.2.7
  (crate-source "stfu8" "0.2.7"
                "0y0rzzphh2mzfhjz0sxymnjn0s4ap21c74f469s9xycky24iw7z5"))

(define rust-str-buf-1.0.6
  (crate-source "str-buf" "1.0.6"
                "1l7q4nha7wpsr0970bfqm773vhmpwr9l6rr8r4gwgrh46wvdh24y"))

(define rust-str-indices-0.4.3
  (crate-source "str_indices" "0.4.3"
                "0p6kggjax1mx0niq22dsm5xq2jvg6l4nyrm8a6f0138yaav7qmg9"))

(define rust-str-indices-0.4.4
  (crate-source "str_indices" "0.4.4"
                "1rj7xrvv5m97qiqwqk1sqxazyggrw7h8kbb6vc438s08akn8k26h"))

(define rust-str-stack-0.1.0
  (crate-source "str_stack" "0.1.0"
                "1sxl8xd8kiaffsryqpfwcb02lnd3djfin7gf38ag5980908vd4ch"))

(define rust-streaming-iterator-0.1.9
  (crate-source "streaming-iterator" "0.1.9"
                "0845zdv8qb7zwqzglpqc0830i43xh3fb6vqms155wz85qfvk28ib"))

(define rust-streaming-stats-0.1.29
  (crate-source "streaming-stats" "0.1.29"
                "09cbpj40g1wcmkz9bpm9yrqyrvqizn6kmwac1jpb2prq6gr2phyv"))

(define rust-streaming-stats-0.2.3
  (crate-source "streaming-stats" "0.2.3"
                "0iz5dlq51w5hxjrv6a4hpf8rrj91kgvy0s9mhj0j12il9v771mmh"))

(define rust-strength-reduce-0.2.4
  (crate-source "strength_reduce" "0.2.4"
                "10jdq9dijjdkb20wg1dmwg447rnj37jbq0mwvbadvqi2gys5x2gy"))

(define rust-strict-num-0.1.1
  (crate-source "strict-num" "0.1.1"
                "0cb7l1vhb8zj90mzm8avlk815k40sql9515s865rqdrdfavvldv6"))

(define rust-string-cache-0.8.9
  (crate-source "string_cache" "0.8.9"
                "03z7km2kzlwiv2r2qifq5riv4g8phazwng9wnvs3py3lzainnxxz"))

(define rust-string-cache-codegen-0.5.4
  (crate-source "string_cache_codegen" "0.5.4"
                "181ir4d6y053s1kka2idpjx5g9d9jgll6fy517jhzzpi2n3r44f7"))

(define rust-string-interner-0.18.0
  (crate-source "string-interner" "0.18.0"
                "16vg2fphp673h9pqsk46lbdj91baz4p9rj2pmi62v7vs9m37achs"))

(define rust-stringprep-0.1.5
  (crate-source "stringprep" "0.1.5"
                "1cb3jis4h2b767csk272zw92lc6jzfzvh8d6m1cd86yqjb9z6kbv"))

(define rust-strip-ansi-escapes-0.2.1
  (crate-source "strip-ansi-escapes" "0.2.1"
                "0980min1s9f5g47rwlq8l9njks952a0jlz0v7yxrm5p7www813ra"))

(define rust-strsim-0.10.0
  (crate-source "strsim" "0.10.0"
                "08s69r4rcrahwnickvi0kq49z524ci50capybln83mg6b473qivk"))

(define rust-strsim-0.11.1
  (crate-source "strsim" "0.11.1"
                "0kzvqlw8hxqb7y598w1s0hxlnmi84sg5vsipp3yg5na5d1rvba3x"))

(define rust-strsim-0.7.0
  (crate-source "strsim" "0.7.0"
                "0l7mkwvdk4vgnml67b85mczk466074aj8yf25gjrjslj4l0khkxv"))

(define rust-strsim-0.8.0
  (crate-source "strsim" "0.8.0"
                "0sjsm7hrvjdifz661pjxq5w4hf190hx53fra8dfvamacvff139cf"))

(define rust-strsim-0.9.3
  (crate-source "strsim" "0.9.3"
                "0k497pv882qn3q977ckznm13vxx927g8s1swvcv68j3c1pccwik4"))

(define rust-struct-patch-0.8.7
  (crate-source "struct-patch" "0.8.7"
                "011z9z1709ksb051ckh2cclm4yi9y78a5ph2qfsy9vxrwifbbqfd"))

(define rust-struct-patch-derive-0.8.7
  (crate-source "struct-patch-derive" "0.8.7"
                "00vm883pbbach6mfh04df0aqx184796yc8glsxzga8gp9fhgx55c"))

(define rust-structmeta-0.3.0
  (crate-source "structmeta" "0.3.0"
                "0afk0s9paazsvyvsirxvbnqp3blhdck3fmfhdw7xf209skc7a59f"))

(define rust-structmeta-derive-0.3.0
  (crate-source "structmeta-derive" "0.3.0"
                "1z12r4v2d3272hxqxclnr1kn2kp07qsy5aswm4ynrzwhlmjhnahm"))

(define rust-structopt-0.3.26
  (crate-source "structopt" "0.3.26"
                "043sg3qxllann6q9i71d05qp3q13scmcvhxhd950ka2v8ij5qsqc"))

(define rust-structopt-derive-0.4.18
  (crate-source "structopt-derive" "0.4.18"
                "1q5gcigmvw0cinjxzpyrkflliq5r1ivljmrvfrl3phcwgwraxdfw"))

(define rust-strum-0.18.0
  (crate-source "strum" "0.18.0"
                "0asjskn1qhqqfiq673np0gvmnd1rsp506m38vk53gi7l93mq3gap"))

(define rust-strum-0.20.0
  (crate-source "strum" "0.20.0"
                "0p5cslmdnz261kiwmm4h7qsmv9bh83r0f9lq6f2z2mxsnl4wa63k"))

(define rust-strum-0.23.0
  (crate-source "strum" "0.23.0"
                "1fvhkg7di4psfw289v2flv19i28rcflq1g1z3n2rl76iqy8lpqfa"))

(define rust-strum-0.24.1
  (crate-source "strum" "0.24.1"
                "0gz6cjhlps5idwasznklxdh2zsas6mxf99vr0n27j876q12n0gh6"))

(define rust-strum-0.25.0
  (crate-source "strum" "0.25.0"
                "09g1q55ms8vax1z0mxlbva3vm8n2r1179kfvbccnkjcidzm58399"))

(define rust-strum-0.26.3
  (crate-source "strum" "0.26.3"
                "01lgl6jvrf4j28v5kmx9bp480ygf1nhvac8b4p7rcj9hxw50zv4g"))

(define rust-strum-0.27.1
  (crate-source "strum" "0.27.1"
                "0cic9r2sc2h17nnpjm2yfp7rsd35gkbcbqvrhl553jaiih4fykgn"))

(define rust-strum-0.27.2
  (crate-source "strum" "0.27.2"
                "1ksb9jssw4bg9kmv9nlgp2jqa4vnsa3y4q9zkppvl952q7vdc8xg"))

(define rust-strum-macros-0.18.0
  (crate-source "strum_macros" "0.18.0"
                "0k3pwbv0c8q00jnsjshzfc2d5r3y6ppgf9fz7pyknrgaz2immj47"))

(define rust-strum-macros-0.20.1
  (crate-source "strum_macros" "0.20.1"
                "0j9ikvxlqzr667ghc045qkpwprjgcfmzgagln7maw4jigawcd2zf"))

(define rust-strum-macros-0.23.1
  (crate-source "strum_macros" "0.23.1"
                "0f2sr3nnhbfpg92ralzqka0wb90nz8ks3sfdk5hylp61x5zdrc2v"))

(define rust-strum-macros-0.24.3
  (crate-source "strum_macros" "0.24.3"
                "0naxz2y38kwq5wgirmia64vvf6qhwy8j367rw966n62gsbh5nf0y"))

(define rust-strum-macros-0.25.3
  (crate-source "strum_macros" "0.25.3"
                "184y62g474zqb2f7n16x3ghvlyjbh50viw32p9w9l5lwmjlizp13"))

(define rust-strum-macros-0.26.4
  (crate-source "strum_macros" "0.26.4"
                "1gl1wmq24b8md527cpyd5bw9rkbqldd7k1h38kf5ajd2ln2ywssc"))

(define rust-strum-macros-0.27.1
  (crate-source "strum_macros" "0.27.1"
                "1s7x07nkrgjfvxrvcdjw6qanad4c55yjnd32bph9q3xgpid8qyn7"))

(define rust-strum-macros-0.27.2
  (crate-source "strum_macros" "0.27.2"
                "19xwikxma0yi70fxkcy1yxcv0ica8gf3jnh5gj936jza8lwcx5bn"))

(define rust-subplot-0.12.0
  (crate-source "subplot" "0.12.0"
                "008cibbnfh1g5mnhz01pxw8kqgrw05r2kzfh7d4sr695bg69ifay"))

(define rust-subplot-build-0.12.0
  (crate-source "subplot-build" "0.12.0"
                "1grmmy2gmpygxcx09dsrh9mm9pqpf5qvy44k6q062lj26jlk32q0"))

(define rust-subplotlib-0.12.0
  (crate-source "subplotlib" "0.12.0"
                "08k4j32mxymhl2fj91vmj8f6c7lwmxxnym4b09vjzxjf3n68kpvd"))

(define rust-subplotlib-derive-0.12.0
  (crate-source "subplotlib-derive" "0.12.0"
                "1yrfjxh8z7dbsh01p4lrq73nci1qql2z9lphlnr2xmydip09nz7k"))

(define rust-subsetter-0.2.1
  (crate-source "subsetter" "0.2.1"
                "15qlr65i97yq14nnv68vx8darrdibpw7ad8zq388vknwwf6rwlrm"))

(define rust-subst-0.3.8
  (crate-source "subst" "0.3.8"
                "16gcai8b798yh9j0pis5wgn6xbmzm1lq4s9lhg8w4qsg2kjqd6ha"))

(define rust-subtle-2.4.1
  (crate-source "subtle" "2.4.1"
                "00b6jzh9gzb0h9n25g06nqr90z3xzqppfhhb260s1hjhh4pg7pkb"))

(define rust-subtle-2.6.1
  (crate-source "subtle" "2.6.1"
                "14ijxaymghbl1p0wql9cib5zlwiina7kall6w7g89csprkgbvhhk"))

(define rust-supports-color-2.1.0
  (crate-source "supports-color" "2.1.0"
                "12csf7chawxinaapm9rh718nha9hggk6ra86fdaw9hxdagg8qffn"))

(define rust-supports-color-3.0.2
  (crate-source "supports-color" "3.0.2"
                "1mk7r2j6l7zmqk3pg7av0l6viq413lmk1vz4bjnf9lnq5liwfky6"))

(define rust-supports-hyperlinks-2.1.0
  (crate-source "supports-hyperlinks" "2.1.0"
                "0g93nh1db3f9lyd0ry35bqjrxkg6sbysn36x9hgd9m5h5rlk2hpq"))

(define rust-supports-hyperlinks-3.1.0
  (crate-source "supports-hyperlinks" "3.1.0"
                "12r8d8ckdx78rhdsavh08gg4210i3bmcn2prm7k2s5b37knl8kw0"))

(define rust-supports-unicode-2.1.0
  (crate-source "supports-unicode" "2.1.0"
                "0yp703pvpzpmaw9mpncvwf0iqis4xmhs569ii1g20jhqvngc2l7q"))

(define rust-supports-unicode-3.0.0
  (crate-source "supports-unicode" "3.0.0"
                "1qpc344453x3ai4k9iygxnbk6lr2nw5jflj8ns5q3dbcmwq1lh5p"))

(define rust-svd-parser-0.14.9
  (crate-source "svd-parser" "0.14.9"
                "048q40fx818fxw3zy470ka1lwsvdpy5qf3mjsfkn7qyy8rzf27x4"))

(define rust-svd-rs-0.14.12
  (crate-source "svd-rs" "0.14.12"
                "0n3lfymlvsc6kj4m8s0jld872bjsq5ymhrb9ywzn429gx3i8xf69"))

(define rust-svg-0.17.0
  (crate-source "svg" "0.17.0"
                "17kp090hniz0axnv75ripfr5d2xhcbnyhiml30yc4ngmyd0gn3kh"))

(define rust-svg-metadata-0.4.4
  (crate-source "svg_metadata" "0.4.4"
                "002j0na1kfz4pgi43hdcz5baygzk6irnjd5lrmbqqfjldwn3sbx4"
                #:snippet '(for-each delete-file-recursively '("fixtures" "tests"))))

(define rust-svg-metadata-0.5.1
  (crate-source "svg_metadata" "0.5.1"
                "0717wb4zkj5qdypg58zs7m1x45i49pdrhzga0k0wvqn2g2y7qk17"))

(define rust-svg2pdf-0.12.0
  ;; NB: The .icc color profiles within are an open-standard format.
  (crate-source "svg2pdf" "0.12.0"
                "0r03ql053ib4rjydaq2dw8ddxjdwbblkhr61z1zgn67kvkdcj52h"))

(define rust-svgtypes-0.15.3
  (crate-source "svgtypes" "0.15.3"
                "1z4a0b76ww6rf2c8zdapqh2a7r7kmmy7m957q5h5ics4zwgm9iv8"))

(define rust-swaybar-types-3.0.0
  (crate-source "swaybar-types" "3.0.0"
                "0hz1ialgs4l0ki5s9gqhsdlfy0v3fph7mkw2z1rdi29bjlsv9h52"))

(define rust-swayipc-2.7.2
  (crate-source "swayipc" "2.7.2"
                "03r15c2sijyrxmpsyjgma4gz7zmdl1g8akjnjkw6hrml91d5dilj"))

(define rust-swayipc-3.0.3
  (crate-source "swayipc" "3.0.3"
                "0dn733vl3fhqvs1ljfhwcgxgdn7z3xwyydba0r98ps4q5v5m131b"))

(define rust-swayipc-async-2.0.4
  (crate-source "swayipc-async" "2.0.4"
                "12fb0fym5n2dn80bgmbi9jfdjgpbwsdjcrmjli9l4ymk7rvxsa4r"))

(define rust-swayipc-types-1.4.1
  (crate-source "swayipc-types" "1.4.1"
                "1zza14kgbix24k9s4zxv026q9avp8p21rhllh7xprs130g3364jm"))

(define rust-swrite-0.1.0
  (crate-source "swrite" "0.1.0"
                "00vsicglylq4qq6dc497jdgzfnxi5mh7padwxijnvh1d1giyqgvz"))

(define rust-symphonia-0.5.4
  (crate-source "symphonia" "0.5.4"
                "1agmsnmzgsmvd70hq760nvkjrb52nnjmz5hgn1xp6x7fwwm98p41"))

(define rust-symphonia-bundle-mp3-0.5.4
  (crate-source "symphonia-bundle-mp3" "0.5.4"
                "1m062zkxq2cbwqxbm3qp4qvgpc9hm49g23vgdc4zpwghf2p2l760"))

(define rust-symphonia-codec-vorbis-0.5.4
  (crate-source "symphonia-codec-vorbis" "0.5.4"
                "0c4z98b8yg2kws3pknw7ipvvca911j3y5xq7n0r6f2kanigpd62s"))

(define rust-symphonia-core-0.5.4
  (crate-source "symphonia-core" "0.5.4"
                "1hr2w2a217vq4lpghszmsdwxr5ilh5d1ysfm3cixbirxkrvhd0vr"))

(define rust-symphonia-format-ogg-0.5.4
  (crate-source "symphonia-format-ogg" "0.5.4"
                "0cd9py2xgx211qvwl9sw8n5l5vgd55vwcmqizh0cyssii5bm18xd"))

(define rust-symphonia-metadata-0.5.4
  (crate-source "symphonia-metadata" "0.5.4"
                "0g02lhhyf6yyxm7bynx5b9fn2ha39y8fp6cfn72qj05186c2nqmw"))

(define rust-symphonia-utils-xiph-0.5.4
  (crate-source "symphonia-utils-xiph" "0.5.4"
                "1zhhs1p0h6wdcgcwfqpmqq07n8v2wvn50razvapr36d41xc74i28"))

(define rust-syn-0.15.44
  (crate-source "syn" "0.15.44"
                "1id5g6x6zihv3j7hwrw3m1jp636bg8dpi671r7zy3jvpkavb794w"))

(define rust-syn-1.0.109
  (crate-source "syn" "1.0.109"
                "0ds2if4600bd59wsv7jjgfkayfzy3hnazs394kz6zdkmna8l3dkj"))

(define rust-syn-1.0.84
  (crate-source "syn" "1.0.84"
                "0sz0q8g0bm2cgxlb6mdhrs8wb731m4ps6ql70rhrmsz5ivdfdcpc"))

(define rust-syn-2.0.100
  (crate-source "syn" "2.0.100"
                "18623wdkns03blpv65xsjn8fipl9p9hj98vlrnhin7nqran496mh"))

(define rust-syn-2.0.101
  (crate-source "syn" "2.0.101"
                "1brwsh7fn3bnbj50d2lpwy9akimzb3lghz0ai89j8fhvjkybgqlc"))

(define rust-syn-2.0.103
  (crate-source "syn" "2.0.103"
                "1s517h7zk5bwkfq0y08z7cig6cly3wxdmwhjyapxcvwx10q7wc74"))

(define rust-syn-2.0.104
  (crate-source "syn" "2.0.104"
                "0h2s8cxh5dsh9h41dxnlzpifqqn59cqgm0kljawws61ljq2zgdhp"))

(define rust-syn-2.0.106
  (crate-source "syn" "2.0.106"
                "19mddxp1ia00hfdzimygqmr1jqdvyl86k48427bkci4d08wc9rzd"))

(define rust-syn-2.0.28
  (crate-source "syn" "2.0.28"
                "0rqm6lnxdgg5blva19l2sjr7iwj2r5dwb3cx32r4iqzmndsijdh4"))

(define rust-syn-2.0.31
  (crate-source "syn" "2.0.31"
                "165k1hg1h641adw36xchm3gvccbr5bqinylifpbqp3fbbd0s53vi"))

(define rust-syn-2.0.37
  (crate-source "syn" "2.0.37"
                "1s42ypidkj410dfc4ir7bni0r114k822j395rf342rfd0lnfy0vk"))

(define rust-syn-2.0.87
  (crate-source "syn" "2.0.87"
                "0bd3mfcswvn4jkrp7ich5kk58kmpph8412yxd36nsfnh8vilrai5"))

(define rust-syn-2.0.98
  (crate-source "syn" "2.0.98"
                "1cfk0qqbl4fbr3dz61nw21d5amvl4rym6nxwnfsw43mf90d7y51n"))

(define rust-sync-ptr-0.1.4
  (crate-source "sync-ptr" "0.1.4"
                "0kva6631hw8k866kmhrfiiagmxms2nmlh3hfyljidla2qx65n49b"))

(define rust-sync-wrapper-0.1.2
  (crate-source "sync_wrapper" "0.1.2"
                "0q01lyj0gr9a93n10nxsn8lwbzq97jqd6b768x17c8f7v7gccir0"))

(define rust-sync-wrapper-1.0.2
  (crate-source "sync_wrapper" "1.0.2"
                "0qvjyasd6w18mjg5xlaq5jgy84jsjfsvmnn12c13gypxbv75dwhb"))

(define rust-synstructure-0.12.6
  (crate-source "synstructure" "0.12.6"
                "03r1lydbf3japnlpc4wka7y90pmz1i0danaj3f9a7b431akdlszk"))

(define rust-synstructure-0.13.1
  (crate-source "synstructure" "0.13.1"
                "0wc9f002ia2zqcbj0q2id5x6n7g1zjqba7qkg2mr0qvvmdk7dby8"))

(define rust-synstructure-0.13.2
  (crate-source "synstructure" "0.13.2"
                "1lh9lx3r3jb18f8sbj29am5hm9jymvbwh6jb1izsnnxgvgrp12kj"))

(define rust-syntect-5.2.0 rust-syntect-5)

(define rust-sys-info-0.9.1
  (crate-source "sys-info" "0.9.1"
                "0b759814ng0cj5a1iiqqjgrzfg9vqlpkbp6z3l76mycbp850sfhb"))

(define rust-sys-locale-0.3.2
  (crate-source "sys-locale" "0.3.2"
                "1i16hq9mkwpzqvixjfy1ph4i2q5klgagjg4hibz6k894l2crmawf"))

(define rust-syscallz-0.17.0
  (crate-source "syscallz" "0.17.0"
                "19zc9rdjxhddbmnkdhk2aixh8csqv4rhbll78smvy471qdvxhpx0"))

(define rust-sysctl-0.5.5
  (crate-source "sysctl" "0.5.5"
                "1sly34bm4w2vcfqgn7f6255pxwa2wa4vkzdrz2x0drgyy32xszgc"))

(define rust-sysctl-0.6.0
  (crate-source "sysctl" "0.6.0"
                "1z0x1lim5929fs60wjxnvd31z59d120p1v16d0mwcdxjxcnql681"))

(define rust-sysexits-0.9.0
  (crate-source "sysexits" "0.9.0"
                "1d7ilmqq6li2dq0gvih51zphkw9d10ny94bna1lg20zhyz8n13qr"))

(define rust-sysexits-0.9.1
  (crate-source "sysexits" "0.9.1"
                "11mw126w984y5lw9qd94q3hdqzf8780dm2b2l8j9mqya4a7vby7z"))

(define rust-sysinfo-0.21.2
  (crate-source "sysinfo" "0.21.2"
                "16j4wfiihlq261dh2k939br6q9zzaiwh719m7lb8idj16dpxd24g"))

(define rust-sysinfo-0.27.8
  (crate-source "sysinfo" "0.27.8"
                "0cqy39g76298pqfr8jv30j6cxl9bpnd7c2smfxl5s2na1w2yj0m9"))

(define rust-sysinfo-0.29.11
  (crate-source "sysinfo" "0.29.11"
                "0rp6911qqjppvvbh72j27znscrawfvplqlyrj9n0y1n24g27ywnd"))

(define rust-sysinfo-0.30.13
  (crate-source "sysinfo" "0.30.13"
                "1csbkx1hdlacgzw5ynjyfvgc1xg58w3h1rgh5gm2pysmxvd4snqa"))

(define rust-sysinfo-0.31.4
  (crate-source "sysinfo" "0.31.4"
                "1gm1d4pxwnx4gmh6kwawchv8v8djb7y0a3qvbsq09cwrhx7vwp9m"))

(define rust-sysinfo-0.33.1
  (crate-source "sysinfo" "0.33.1"
                "00bcbj9rk39n07ylclj9klggkshxyianv2lfkpqnc6x0iqj5ij2g"))

(define rust-syslog-6.1.1
  (crate-source "syslog" "6.1.1"
                "1lvs8ld2ps38yll29fryqwr45axm55vf46b5zvx24lbrbddykiyz"))

(define rust-syslog-7.0.0
  (crate-source "syslog" "7.0.0"
                "044z9mq055a9k6k8i8d725ixww2pqybz6pa5a78bfy9kl401b7q1"))

(define rust-system-configuration-0.5.1
  (crate-source "system-configuration" "0.5.1"
                "1rz0r30xn7fiyqay2dvzfy56cvaa3km74hnbz2d72p97bkf3lfms"))

(define rust-system-configuration-0.6.1
  (crate-source "system-configuration" "0.6.1"
                "0sxslml567zm0v8g732314vd2gk9sd3k4xj22xk6p64xir29v1rw"))

(define rust-system-configuration-sys-0.5.0
  (crate-source "system-configuration-sys" "0.5.0"
                "1jckxvdr37bay3i9v52izgy52dg690x5xfg3hd394sv2xf4b2px7"))

(define rust-system-configuration-sys-0.6.0
  (crate-source "system-configuration-sys" "0.6.0"
                "1i5sqrmgy58l4704hibjbl36hclddglh73fb3wx95jnmrq81n7cf"))

(define rust-system-deps-1.3.2
  (crate-source "system-deps" "1.3.2"
                "16v4ljmj8sj030mdcc1yk615vciqlyxi7csq6lxka6cs4qbwqghg"))

(define rust-system-deps-6.1.1
  (crate-source "system-deps" "6.1.1"
                "1wvipxn08j4sa9z32vvm7b2riv3l5b6wkzsaccyq4jwg9n5dxhih"))

(define rust-system-deps-6.2.2
  (crate-source "system-deps" "6.2.2"
                "0j93ryw031n3h8b0nfpj5xwh3ify636xmv8kxianvlyyipmkbrd3"
                #:snippet '(delete-file-recursively "src/tests")))

(define rust-system-deps-7.0.5
  (crate-source "system-deps" "7.0.5"
                "1nzzhqm3nnzacpcs5q1rlbzxrb1hq6xl5g8mqqnnv2ds1jm57gp4"))

(define rust-systemd-journal-logger-2.2.2
  (crate-source "systemd-journal-logger" "2.2.2"
                "1777sfc47l5wnr4y26zbsbhx72z1h265bz250lilnnna4i6k0rkj"))

(define rust-systemstat-0.2.3
  (crate-source "systemstat" "0.2.3"
                "1lhbbgywmhfdb2vjaxzl9g3dbbz2nyfyzqr8kacq6b1im4jfqjm2"))

(define rust-tabled-0.17.0
  (crate-source "tabled" "0.17.0"
                "02ji1rhr6kig7mhh3zlwbflqghadav6mj1g5gi13fccpyci94w66"))

(define rust-tabled-0.20.0
  (crate-source "tabled" "0.20.0"
                "0zawf8zg5frprmqjygldggn8zj4cyw0b7qbilw2hhdndzghjx6p3"))

(define rust-tabled-derive-0.11.0
  (crate-source "tabled_derive" "0.11.0"
                "0ii8l0l6fi8sbxrnir5mv91jbmzgbvqn5z8z4gwz3kx67jqx398f"))

(define rust-tabwriter-1.4.1
  (crate-source "tabwriter" "1.4.1"
                "0ch4823i90iw35an0g000f3ii8cs8dkv5gnbddzgyzf81qpizsgw"))

(define rust-tagptr-0.2.0
  (crate-source "tagptr" "0.2.0"
                "05r4mwvlsclx1ayj65hpzjv3dn4wpi8j4xm695vydccf9k7r683v"))

(define rust-take-mut-0.2.2
  (crate-source "take_mut" "0.2.2"
                "0q2d7w6nd5bl7bay5csq065sjg8fw0jcx6hl1983cpzf25fh0r7p"))

(define rust-tame-index-0.17.0
  (crate-source "tame-index" "0.17.0"
                "138k9c7ww7dhy923nfxd1ylx9yarnbhwd2gn4csr87jdd6iwllhn"))

(define rust-tame-index-0.18.1
  (crate-source "tame-index" "0.18.1"
                "0365pyq3qp7415z1xql03763krh63779gqdgxwc8l22dq5hrxkpz"))

(define rust-tango-bench-0.6.0
  (crate-source "tango-bench" "0.6.0"
                "0gj2jgfdmwhrdggqh3yp8h33n1jrz6f3drmzg3nny83gihsj4y15"))

(define rust-tap-1.0.1
  (crate-source "tap" "1.0.1"
                "0sc3gl4nldqpvyhqi3bbd0l9k7fngrcl4zs47n314nqqk4bpx4sm"))

(define rust-tar-0.4.44
  (crate-source "tar" "0.4.44"
                "0yk69a8j9xv51mdcy0853jai5zh1pd9yn456q4cpmj0js9w3i1hx"))

(define rust-target-2.1.0
  (crate-source "target" "2.1.0"
                "0idi9b9ndcvpjzm3y0gaz1ljc4711flkg0jsmnykbnxjfkvhb3qy"))

(define rust-target-lexicon-0.12.11
  (crate-source "target-lexicon" "0.12.11"
                "12nwfd1ylqysn1mqf967hc33wcvg0jyvq7hfhmiy5j2825mr23lx"))

(define rust-target-lexicon-0.12.16
  (crate-source "target-lexicon" "0.12.16"
                "1cg3bnx1gdkdr5hac1hzxy64fhw4g7dqkd0n3dxy5lfngpr1mi31"))

(define rust-target-lexicon-0.13.2
  (crate-source "target-lexicon" "0.13.2"
                "16m6smfz533im9dyxfhnzmpi4af75g2iii36ylc4gfmqvf6gf0p5"))

(define rust-target-spec-3.4.2
  (crate-source "target-spec" "3.4.2"
                "0lr5sp4hjiawwhdmv3hbg5ip566iqha31shwdpj0ci5svh7lshj9"))

(define rust-target-spec-miette-0.4.4
  (crate-source "target-spec-miette" "0.4.4"
                "0bkcp7ff3z8kccfays45pv8ya4g66lld6srgzn2gic7fwd04yxa1"))

(define rust-target-triple-0.1.4
  (crate-source "target-triple" "0.1.4"
                "140p6rjx7ychv0sryndziia1w14cfjflmhh7ccjj57ar3wvsmj8s"))

(define rust-tauri-winrt-notification-0.7.2
  (crate-source "tauri-winrt-notification" "0.7.2"
                "1fd9gcllx1rkp9h1ppq976bhqppnil5xsy36li1zx2g4gph6c7hb"))

(define rust-temp-dir-0.1.14
  (crate-source "temp-dir" "0.1.14"
                "0wiwpkkxln7ykj44029bmc86cariqq2mjwllrdjzf4jgygpfc7mw"))

(define rust-temp-dir-0.1.16
  (crate-source "temp-dir" "0.1.16"
                "0r09qwiiqm8pk6inaqmmp0h6zjg9py6m1dkcwqgghv21x5cnf5w3"))

(define rust-temp-env-0.3.6
  (crate-source "temp-env" "0.3.6"
                "0l7hpkd0nhiy4w70j9xbygl1vjr9ipcfxii164n40iwg0ralhdwn"))

(define rust-temp-testdir-0.2.3
  (crate-source "temp_testdir" "0.2.3"
                "1z5yv6d9944md5zg6g0sbahv7xjga2k232x40x4l20kq8af1w7wj"))

(define rust-tempfile-3.0.8
  (crate-source "tempfile" "3.0.8"
                "1vqk7aq2l04my2r3jiyyxirnf8f90nzcvjasvrajivb85s7p7i3x"))

(define rust-tempfile-3.15.0
  (crate-source "tempfile" "3.15.0"
                "016pmkbwn3shas44gcwq1kc9lajalb90qafhiip5fvv8h6f5b2ls"))

(define rust-tempfile-3.17.1
  (crate-source "tempfile" "3.17.1"
                "0c52ggq5vy5mzgk5ly36cgzs1cig3cv6r1jarijmzxgkn6na1r92"))

(define rust-tempfile-3.19.1
  (crate-source "tempfile" "3.19.1"
                "1grmcj8y6rcavndw2dm18ndzdimsq5f8lcrwyg627cdrcdvsqdvl"))

(define rust-tempfile-3.20.0
  (crate-source "tempfile" "3.20.0"
                "18fnp7mjckd9c9ldlb2zhp1hd4467y2hpvx9l50j97rlhlwlx9p8"))

(define rust-tempfile-3.21.0
  (crate-source "tempfile" "3.21.0"
                "07kx58ibjk3ydq1gcb7q637fs5zkxaa550lxckhgg9p3427izdhm"))

(define rust-tempfile-3.22.0
  (crate-source "tempfile" "3.22.0"
                "0lza9r7dzm4k9fghw24yql6iz59wq8xgs46a7i29ir6xz88lvyl4"))

(define rust-tempfile-3.23.0
  (crate-source "tempfile" "3.23.0"
                "05igl2gml6z6i2va1bv49f9f1wb3f752c2i63lvlb9s2vxxwfc9d"))

(define rust-temptree-0.2.0
  (crate-source "temptree" "0.2.0"
                "0c461j4xrjp1mz89fb3rmv8w36m1dm1nymv9ny60hh0v4pc99nlg"))

(define rust-tendril-0.4.3
  (crate-source "tendril" "0.4.3"
                "1c3vip59sqwxn148i714nmkrvjzbk7105vj0h92s6r64bw614jnj"))

(define rust-tera-1.20.0
  (crate-source "tera" "1.20.0"
                "1vnj9imw2h9szkd1izsrhwrc9jvazvdsp84x65wg2rg88ldqb7db"))

(define rust-term-0.7.0
  (crate-source "term" "0.7.0"
                "07xzxmg7dbhlirpyfq09v7cfb9gxn0077sqqvszgjvyrjnngi7f5"
                #:snippet '(delete-file-recursively "tests")))

(define rust-term-1.0.2
  (crate-source "term" "1.0.2"
                "0l6pj1p1r76g2q8dx5lii1qigid37x4yvqp8ynm7yqlc0n6lr64a"
                #:snippet '(delete-file-recursively "tests")))

(define rust-term-1.2.0
  (crate-source "term" "1.2.0"
                "1h1sif95rkq6n7kyn8f6m0rwl43kkr08kfsjjyp811p2v92fy491"
                #:snippet '(delete-file-recursively "tests")))

(define rust-term-grid-0.1.7
  (crate-source "term_grid" "0.1.7"
                "1kq2sy3b8329jrsrpcvijvyz4gbqjyvyy6c3n0wmmvda9y03w393"))

(define rust-term-size-0.3.2
  (crate-source "term_size" "0.3.2"
                "1n885cykajsppx86xl7d0dqkgmgsp8v914lvs12qzvd0dij2jh8y"))

(define rust-termbg-0.6.2
  (crate-source "termbg" "0.6.2"
                "1m1x1axm22izpibr3zrmmnr2fzj6fhq7df63zvbxzwxdl5vlbx4b"))

(define rust-termcolor-1.4.1
  (crate-source "termcolor" "1.4.1"
                "0mappjh3fj3p2nmrg4y7qv94rchwi9mzmgmfflr8p2awdj7lyy86"))

(define rust-terminal-colorsaurus-0.4.8
  (crate-source "terminal-colorsaurus" "0.4.8"
                "1s16dvq5ml490vj0fkhx27llzxyscn4v44fbxd9gpjx3fk0y9bxp"))

(define rust-terminal-prompt-0.2.3
  (crate-source "terminal-prompt" "0.2.3"
                "0xk3y4m2qysa8sxb46h79f9qxh8m6x0s6ipzbnysq4198yriha2p"))

(define rust-terminal-size-0.1.17
  (crate-source "terminal_size" "0.1.17"
                "1pq60ng1a7fjp597ifk1cqlz8fv9raz9xihddld1m1pfdia1lg33"))

(define rust-terminal-size-0.2.6
  (crate-source "terminal_size" "0.2.6"
                "0drj7gb77kay5r1cv53ysq3g9g4f8n0jkhld0kadi3lzkvqzcswf"))

(define rust-terminal-size-0.3.0
  (crate-source "terminal_size" "0.3.0"
                "1xqdzdjq77smg41z67vg3qwrcilf1zf5330gdrgm22lyghmvzgi1"))

(define rust-terminal-size-0.4.2
  (crate-source "terminal_size" "0.4.2"
                "1vdm5xhzn7sqcsr762vmnavkhid3hs8w8qjyh9iwrr1990f4iij5"))

(define rust-terminal-size-0.4.3
  (crate-source "terminal_size" "0.4.3"
                "1l7cicmz49c0cyskfp5a389rsai649xi7y032v73475ikjbwpf30"))

(define rust-terminal-trx-0.2.4
  (crate-source "terminal-trx" "0.2.4"
                "1ask2zj8q78rh50wylcc60z78g35c4nb4lsydm2h46zsmqrl4nwp"))

(define rust-terminfo-0.7.5
  (crate-source "terminfo" "0.7.5"
                "11d576lbjis5437rdz7k3k8glayxnhz6hqmlphnkbxm01pvswcfs"
                #:snippet '(delete-file-recursively "tests")))

(define rust-terminfo-0.8.0
  (crate-source "terminfo" "0.8.0"
                "13s9920jzmnr0jidik8yn6gvkic9n39sl28440mx4x8pd2kd6v36"
                #:snippet '(delete-file-recursively "tests")))

(define rust-terminfo-0.9.0
  (crate-source "terminfo" "0.9.0"
                "0qp6rrzkxcg08vjzsim2bw7mid3vi29mizrg70dzbycj0q7q3snl"
                #:snippet '(delete-file-recursively "tests")))

(define rust-termini-1.0.0
  (crate-source "termini" "1.0.0"
                "0n8dvbwkp2k673xqwivb01iqg5ir91zgpwhwngpcb2yrgpc43m1a"))

(define rust-termion-1.5.6
  (crate-source "termion" "1.5.6"
                "0zk023f0zkws358ll399cawvwdnd0wg8wad4g61kz766xbi8aw87"))

(define rust-termios-0.2.2
  (crate-source "termios" "0.2.2"
                "0fk8nl0rmk43jrh6hjz6c6d83ri7l6fikag6lh0ffz3di9cwznfm"))

(define rust-termios-0.3.3
  (crate-source "termios" "0.3.3"
                "0sxcs0g00538jqh5xbdqakkzijadr8nj7zmip0c7jz3k83vmn721"))

(define rust-termsize-0.1.9
  (crate-source "termsize" "0.1.9"
                "1zb80dcqngbvw1mgkdsagwx6hvcsxr1zpql5bf6n0wn14mfgy4bg"))

(define rust-termtree-0.5.1
  (crate-source "termtree" "0.5.1"
                "10s610ax6nb70yi7xfmwcb6d3wi9sj5isd0m63gy2pizr2zgwl4g"))

(define rust-termwiz-0.15.0
  (crate-source "termwiz" "0.15.0"
                "1ig83sldqnbmy95kb4ixhrjn63qym7dssxy3p2rsjj03rj96ivri"
                #:snippet '(for-each delete-file '("data/wezterm" "data/xterm-256color"))))

(define rust-termwiz-0.23.3
  (crate-source "termwiz" "0.23.3"
                "1xzq6l7rx285ax57dz8gdh44kp1790x0knvfynmimgfc89rb6xj6"))

(define rust-test-case-3.3.1
  (crate-source "test-case" "3.3.1"
                "1a380yzm6787737cw7s09jqmkn9035hghahradl2ikdg2gfm09gb"))

(define rust-test-case-core-3.3.1
  (crate-source "test-case-core" "3.3.1"
                "0krqi0gbi1yyycigyjlak63r8h1n0vms7mg3kckqwlfd87c7zjxd"))

(define rust-test-case-macros-3.3.1
  (crate-source "test-case-macros" "3.3.1"
                "1yvgky3qax73bic6m368q04xc955p4a91mddd6b5fk7d04mfg2aw"))

(define rust-test-log-0.2.17
  (crate-source "test-log" "0.2.17"
                "03ydax0mamyhn5n6x0l8d2kz76fly7jv3xmnwmk12611sa1n1x77"))

(define rust-test-log-0.2.18
  (crate-source "test-log" "0.2.18"
                "0yxywma018rfr4mb409b1yz4ppg8ir9rg87bd08vx81fb25bjcqy"))

(define rust-test-log-macros-0.2.17
  (crate-source "test-log-macros" "0.2.17"
                "0gsmmdsi8aqgxbr16n78sf1m6x5s29gfvlk0n7d0yg5mdly0r3c8"))

(define rust-test-log-macros-0.2.18
  (crate-source "test-log-macros" "0.2.18"
                "0djzwzwqnalwf00r81lv0yv71s4sqwmx7y7fn40pc3ck552kf6s5"))

(define rust-test-strategy-0.4.1
  (crate-source "test-strategy" "0.4.1"
                "04lbqbhgxbnr8q8gpv44xnsz572wm6nz8k47sy6yrlsw7wi2vswm"))

(define rust-testing-table-0.3.0
  (crate-source "testing_table" "0.3.0"
                "1k0l036hgxmvjzr8ngc57ngkhnza3p9xh6cyc5jlz8lmk7iam38g"))

(define rust-textwrap-0.11.0
  (crate-source "textwrap" "0.11.0"
                "0q5hky03ik3y50s9sz25r438bc4nwhqc6dqwynv4wylc807n29nk"))

(define rust-textwrap-0.14.2
  (crate-source "textwrap" "0.14.2"
                "106xjfzfpk3nj51fx9slf9kyir7xjwvpqm003v9ardgq5b8whrh0"))

(define rust-textwrap-0.15.2
  (crate-source "textwrap" "0.15.2"
                "0galmidi6gpn308b1kv3r4qbb48j2926lcj0idwhdhlylhjybcxp"))

(define rust-textwrap-0.16.0
  (crate-source "textwrap" "0.16.0"
                "0gbwkjf15l6p3x2rkr75fa4cpcs1ly4c8pmlfx5bl6zybcm24ai2"))

(define rust-textwrap-0.16.2
  (crate-source "textwrap" "0.16.2"
                "0mrhd8q0dnh5hwbwhiv89c6i41yzmhw4clwa592rrp24b9hlfdf1"))

(define rust-thin-slice-0.1.1
  (crate-source "thin-slice" "0.1.1"
                "0g4z51g3yarah89ijpakbwhrrknw6d7k3ry0m1zqcn3hbhiq3alf"))

(define rust-thin-vec-0.2.14
  (crate-source "thin-vec" "0.2.14"
                "0zbi73v2in3ay5h1q17gw8y09z5vmf3zqscx5xwsq5c4656pakql"))

(define rust-thiserror-1.0.39
  (crate-source "thiserror" "1.0.39"
                "171dbc6ln19hig5h2hcczcb6m9ldvw3ji24pca2nsm0hnmnh3ax5"))

(define rust-thiserror-1.0.44
  (crate-source "thiserror" "1.0.44"
                "143zzmardcq447va2pw09iq9rajvr48v340riljghf84iah40431"))

(define rust-thiserror-1.0.49
  (crate-source "thiserror" "1.0.49"
                "1m3yalbcx89bx397igc1ckypw8hiwq9jbzc56pgazrzdsz3fhxqi"))

(define rust-thiserror-1.0.69
  (crate-source "thiserror" "1.0.69"
                "0lizjay08agcr5hs9yfzzj6axs53a2rgx070a1dsi3jpkcrzbamn"))

(define rust-thiserror-2.0.12
  (crate-source "thiserror" "2.0.12"
                "024791nsc0np63g2pq30cjf9acj38z3jwx9apvvi8qsqmqnqlysn"))

(define rust-thiserror-2.0.16
  (crate-source "thiserror" "2.0.16"
                "1h30bqyjn5s9ypm668yd9849371rzwk185klwgjg503k2hadcrrl"))

(define rust-thiserror-2.0.17
  (crate-source "thiserror" "2.0.17"
                "1j2gixhm2c3s6g96vd0b01v0i0qz1101vfmw0032mdqj1z58fdgn"))

(define rust-thiserror-impl-1.0.39
  (crate-source "thiserror-impl" "1.0.39"
                "0zi6kaf9s9isyllcamczziwz7gdkp4jwmg0a571kh35gj0pd882l"))

(define rust-thiserror-impl-1.0.44
  (crate-source "thiserror-impl" "1.0.44"
                "15nwh6qfwxlwimgij1p6ajb377p4rlvvc6sx7amiz11h959rh089"))

(define rust-thiserror-impl-1.0.49
  (crate-source "thiserror-impl" "1.0.49"
                "1k643ga9sbqpcb873n299n3zjzc4srfakyv98xwqi4ly0412yw8h"))

(define rust-thiserror-impl-1.0.69
  (crate-source "thiserror-impl" "1.0.69"
                "1h84fmn2nai41cxbhk6pqf46bxqq1b344v8yz089w1chzi76rvjg"))

(define rust-thiserror-impl-2.0.12
  (crate-source "thiserror-impl" "2.0.12"
                "07bsn7shydaidvyyrm7jz29vp78vrxr9cr9044rfmn078lmz8z3z"))

(define rust-thiserror-impl-2.0.16
  (crate-source "thiserror-impl" "2.0.16"
                "0q3r1ipr1rhff6cgrcvc0njffw17rpcqz9hdc7p754cbqkhinpkc"))

(define rust-thiserror-impl-2.0.17
  (crate-source "thiserror-impl" "2.0.17"
                "04y92yjwg1a4piwk9nayzjfs07sps8c4vq9jnsfq9qvxrn75rw9z"))

(define rust-thousands-0.2.0
  (crate-source "thousands" "0.2.0"
                "0848gnkn7ah51lrx15z9zmn701ipn6gc4xbk4kfdlfahkypkpxiv"))

(define rust-thread-id-4.2.2
  (crate-source "thread-id" "4.2.2"
                "1sn0dkvi4kvxm9g50gs8yzw4p7655pyzgb1l3mzdn06ipmdz5s6g"))

(define rust-thread-local-1.1.8
  (crate-source "thread_local" "1.1.8"
                "173i5lyjh011gsimk21np9jn8al18rxsrkjli20a7b8ks2xgk7lb"))

(define rust-thread-local-1.1.9
  (crate-source "thread_local" "1.1.9"
                "1191jvl8d63agnq06pcnarivf63qzgpws5xa33hgc92gjjj4c0pn"))

(define rust-thread-priority-1.2.0
  (crate-source "thread-priority" "1.2.0"
                "1x9byxzv6n6qplkdfkw40xi6wfwisjklx8qkajn63bix0pbpbq6g"))

(define rust-threadpool-1.8.1
  (crate-source "threadpool" "1.8.1"
                "1amgfyzvynbm8pacniivzq9r0fh3chhs7kijic81j76l6c5ycl6h"))

(define rust-thrussh-0.33.5
  (crate-source "thrussh" "0.33.5"
                "1y1nhpg9l8jwjv59am9kmf9gib2j58c1qm36dvgq7pwsi8il0rcf"))

(define rust-thrussh-0.34.1
  (crate-source "thrussh" "0.34.1"
                "0k128v3dn6wv0ywmk8bjp2xx11niafp9q49xps5fwjy8dy0s4y3v"))

(define rust-thrussh-config-0.5.0
  (crate-source "thrussh-config" "0.5.0"
                "1ygm5vdy180gslgmbd0v722ap451pw8cwaba0g0dzx5gg3cl9h37"))

(define rust-thrussh-config-0.6.0
  (crate-source "thrussh-config" "0.6.0"
                "08k6r947q5xx73y29s24snwlzym5x7v2drdl5y794957i6mlm5zs"))

(define rust-thrussh-keys-0.21.0
  (crate-source "thrussh-keys" "0.21.0"
                "1azmv32m5jpxsflj4dq62i84mv2cavc354r8yy98vc9j54dcab57"))

(define rust-thrussh-keys-0.22.1
  (crate-source "thrussh-keys" "0.22.1"
                "1fsr8ank8vciyfdr27b56gnwasjrvadxkklv6w7dn22c7sqmjgf4"))

(define rust-thrussh-libsodium-0.2.3
  (crate-source "thrussh-libsodium" "0.2.3"
                "07lscwsfrh02hqnjw62wyknwmydiqwq4jz1gg1g25xygnh79qksz"))

(define rust-thrussh-libsodium-0.3.0
  (crate-source "thrussh-libsodium" "0.3.0"
                "18vf8zpvyhbcdkn3cl6rdc2s57676jj6j4m2ykszc3fyi2xh1vaq"))

(define rust-tiff-0.10.3
  (crate-source "tiff" "0.10.3"
                "0vrkdk9cdk07rh7iifcxpn6m8zv3wz695mizhr8rb3gfgzg0b5mg"
                #:snippet '(delete-file-recursively "tests")))

(define rust-tiff-0.9.1
  (crate-source "tiff" "0.9.1"
                "0ghyxlz566dzc3scvgmzys11dhq2ri77kb8sznjakijlxby104xs"
                #:snippet '(delete-file-recursively "tests")))

(define rust-tikv-jemalloc-sys-0.6.0+5.3.0-1-ge13ca993e8ccb9ba9847cc330696e02839f328f7
  (crate-source "tikv-jemalloc-sys"
                "0.6.0+5.3.0-1-ge13ca993e8ccb9ba9847cc330696e02839f328f7"
                "0baf5vjpg9ipa388md4yxim77rdblnk8r95mnp1akbqjcj860g6d"
                #:snippet '(delete-file-recursively "jemalloc")))

(define rust-tikv-jemallocator-0.6.0
  (crate-source "tikv-jemallocator" "0.6.0"
                "0r985npb7d9hrbs3mb0bkfbv0nvzjpgvzsbpyj21bn0qhpqmzv2c"
                #:snippet '(delete-file ".Cargo.toml.swp")))

(define rust-time-0.3.41
  (crate-source "time" "0.3.41"
                "0h0cpiyya8cjlrh00d2r72bmgg4lsdcncs76qpwy0rn2kghijxla"))

(define rust-time-0.3.42
  (crate-source "time" "0.3.42"
                "1qz8x3xsy90w6k5f8mspsjn82gnhh5yldvc6il2vi3lxkwvngacc"))

(define rust-time-0.3.43
  (crate-source "time" "0.3.43"
                "0c90pxn59zccwdyvh8pn9ql04c32ky9kqqli7mc2vrqhxkqydgc3"))

(define rust-time-0.3.44
  (crate-source "time" "0.3.44"
                "179awlwb36zly3nmz5h9awai1h4pbf1d83g2pmvlw4v1pgixkrwi"))

(define rust-time-core-0.1.4
  (crate-source "time-core" "0.1.4"
                "0z5h9fknvdvbs2k2s1chpi3ab3jvgkfhdnqwrvixjngm263s7sf9"))

(define rust-time-core-0.1.5
  (crate-source "time-core" "0.1.5"
                "0s0asjj3mz5ahpymx7wbqsns852aliays3cm9ck0f6w6h2rqn459"))

(define rust-time-core-0.1.6
  (crate-source "time-core" "0.1.6"
                "0sqwhg7n47gbffyr0zhipqcnskxgcgzz1ix8wirqs2rg3my8x1j0"))

(define rust-time-macros-0.2.22
  (crate-source "time-macros" "0.2.22"
                "0jcaxpw220han2bzbrdlpqhy1s5k9i8ri3lw6n5zv4zcja9p69im"))

(define rust-time-macros-0.2.23
  (crate-source "time-macros" "0.2.23"
                "0k1k1c4rrxpqsnp3j9ixv7xazhgi701xjjdkj3jlq9m78n97k0ki"))

(define rust-time-macros-0.2.24
  (crate-source "time-macros" "0.2.24"
                "1wzb6hnl35856f58cx259q7ijc4c7yis0qsnydvw5n8jbw9b1krh"))

(define rust-timeago-0.4.2
  (crate-source "timeago" "0.4.2"
                "1rnh92sh1l4jbjvz4g7xvcvmfh7nk5k7mm2w56pnm9z0kmc0wwd1"))

(define rust-timer-0.2.0
  (crate-source "timer" "0.2.0"
                "0srhqyp7fr91d1i43aqs7wc6yn1i3kdkh1pm05bicdw961v23m1i"))

(define rust-tint-1.0.1
  (crate-source "tint" "1.0.1"
                "15k89pp3ddxczfnv2j995yvy02mfpmjrmxmv7mkp8c2acrq4bwks"))

(define rust-tiny-bench-0.4.0
  (crate-source "tiny-bench" "0.4.0"
                "0g0xdw5acam15philby6jd6w5zbhn34bbnm0gvi82nhj42ss3c3j"))

(define rust-tiny-http-0.12.0
  (crate-source "tiny_http" "0.12.0"
                "10nw9kk2i2aq4l4csy0825qkq0l66f9mz2c1n57yg8hkckgib69q"))

(define rust-tiny-keccak-2.0.2
  (crate-source "tiny-keccak" "2.0.2"
                "0dq2x0hjffmixgyf6xv9wgsbcxkd65ld0wrfqmagji8a829kg79c"))

(define rust-tiny-skia-0.11.4
  (crate-source "tiny-skia" "0.11.4"
                "1aq9gd4qh4418g8v08qzakqqggx8hl66qcianl3k5bjdsja37lc3"
                #:snippet '(delete-file-recursively "tests")))

(define rust-tiny-skia-path-0.11.4
  (crate-source "tiny-skia-path" "0.11.4"
                "14ywbdfakvacl6rxxmzbnycplaxpc6i2linh2yqk0sp8qb07z7lw"))

(define rust-tinyfiledialogs-3.9.1
  ;; TODO: Unbundle libtinyfiledialogs.
  (crate-source "tinyfiledialogs" "3.9.1"
                "0s55kkzai2rn3fnvwia4lgmpp9d57zgrdb6pqqn6wmm68fya0pz2"))

(define rust-tinystr-0.7.6
  (crate-source "tinystr" "0.7.6"
                "0bxqaw7z8r2kzngxlzlgvld1r6jbnwyylyvyjbv1q71rvgaga5wi"))

(define rust-tinystr-0.8.1
  (crate-source "tinystr" "0.8.1"
                "12sc6h3hnn6x78iycm5v6wrs2xhxph0ydm43yyn7gdfw8l8nsksx"))

(define rust-tinytemplate-1.2.1
  (crate-source "tinytemplate" "1.2.1"
                "1g5n77cqkdh9hy75zdb01adxn45mkh9y40wdr7l68xpz35gnnkdy"))

(define rust-tinyvec-1.10.0
  (crate-source "tinyvec" "1.10.0"
                "1yhk0qdqyiaa4v2j9h8pzax5gxgwpz4da0lcphfil6g6pk1zv9dz"))

(define rust-tinyvec-1.8.0
  (crate-source "tinyvec" "1.8.0"
                "0f5rf6a2wzyv6w4jmfga9iw7rp9fp5gf4d604xgjsf3d9wgqhpj4"))

(define rust-tinyvec-1.9.0
  (crate-source "tinyvec" "1.9.0"
                "0w9w8qcifns9lzvlbfwa01y0skhr542anwa3rpn28rg82wgndcq9"))

(define rust-tinyvec-macros-0.1.1
  (crate-source "tinyvec_macros" "0.1.1"
                "081gag86208sc3y6sdkshgw3vysm5d34p431dzw0bshz66ncng0z"))

(define rust-titlecase-3.5.0
  (crate-source "titlecase" "3.5.0"
                "0dcl4rg82qrkzh7paivvc519cgf8w84m1svd5n9lawjz89yx1ym1"))

(define rust-tl-0.7.8.6e25b2e
  (origin
    (method git-fetch)
    (uri (git-reference (url "https://github.com/astral-sh/tl.git")
                        (commit "6e25b2ee2513d75385101a8ff9f591ef51f314ec")))
    (file-name (git-file-name "rust-tl" "0.7.8.6e25b2e"))
    (sha256 (base32 "0r6wwvw1apsfzdhzvw2vinjb3nwbyly2ycx09yfqc0wrwiav6khp"))))

(define rust-tls-parser-0.12.2
  (crate-source "tls-parser" "0.12.2"
                "0hf7k9fn1ziy6lmskkz6gcch4lfzv9mhzrr4yaqq8988qr4n5hr2"))

(define rust-tokenizers-0.19.1
  (crate-source "tokenizers" "0.19.1"
                "1zg6ffpllygijb5bh227m9p4lrhf0pjkysky68kddwrsvp8zl075"))

(define rust-tokio-1.44.1
  (crate-source "tokio" "1.44.1"
                "06n90q5hh1yd844s6nf4j3fwbrkm2bnq533kp3a488l4bdhxm0pk"))

(define rust-tokio-1.44.2
  (crate-source "tokio" "1.44.2"
                "0j4w3qvlcqzgbxlnap0czvspqj6x461vyk1sbqcf97g4rci8if76"))

(define rust-tokio-1.45.0
  (crate-source "tokio" "1.45.0"
                "0rg1i83awynp1xnhz4y1klmi1jq787pa8wgy4gxy1vgr9rlwl4r5"))

(define rust-tokio-1.45.1
  (crate-source "tokio" "1.45.1"
                "0yb7h0mr0m0gfwdl1jir2k37gcrwhcib2kiyx9f95npi7sim3vvm"))

(define rust-tokio-1.46.1
  (crate-source "tokio" "1.46.1"
                "05sxldy7kcgysnxyzz1h1l8j3d9mjyqfh7r48ni27gmg9lsa5hqc"))

(define rust-tokio-1.47.1
  (crate-source "tokio" "1.47.1"
                "0f2hp5v3payg6x04ijj67si1wsdhksskhmjs2k9p5f7bmpyrmr49"))

(define rust-tokio-macros-2.5.0
  (crate-source "tokio-macros" "2.5.0"
                "1f6az2xbvqp7am417b78d1za8axbvjvxnmkakz9vr8s52czx81kf"))

(define rust-tokio-native-tls-0.3.1
  (crate-source "tokio-native-tls" "0.3.1"
                "1wkfg6zn85zckmv4im7mv20ca6b1vmlib5xwz9p7g19wjfmpdbmv"
                #:snippet '(for-each delete-file-recursively '("examples" "tests"))))

(define rust-tokio-openssl-0.6.5
  (crate-source "tokio-openssl" "0.6.5"
                "1pga4xm5fcms6k1rqg4hsl8mmna7qiizhdlsgxbbffx4r94nipsr"
                #:snippet '(delete-file-recursively "tests")))

(define rust-tokio-rusqlite-0.5.0
  (crate-source "tokio-rusqlite" "0.5.0"
                "0f8xnb6c47306wp22i15hm76nflg7bry36z5h59lawn8s2c5qy6w"))

(define rust-tokio-rustls-0.24.1
  (crate-source "tokio-rustls" "0.24.1"
                "10bhibg57mqir7xjhb2xmf24xgfpx6fzpyw720a4ih8a737jg0y2"))

(define rust-tokio-rustls-0.25.0
  (crate-source "tokio-rustls" "0.25.0"
                "03w6d5aqqf084rmcmrsyq5grhydl53blaiqcl0i2yfnv187hqpkp"))

(define rust-tokio-rustls-0.26.2
  (crate-source "tokio-rustls" "0.26.2"
                "16wf007q3584j46wc4s0zc4szj6280g23hka6x6bgs50l4v7nwlf"))

(define rust-tokio-rustls-0.26.3
  (crate-source "tokio-rustls" "0.26.3"
                "1gc6fxpd185xclg5y8z1kana0axv4g6vnypmpvqj78ccj8skixh5"))

(define rust-tokio-socks-0.5.2
  (crate-source "tokio-socks" "0.5.2"
                "0gq40sgggz21wfpshiq8pryh062vp7m36rrz3c8c2wj60aw70iqd"))

(define rust-tokio-stream-0.1.17
  (crate-source "tokio-stream" "0.1.17"
                "0ix0770hfp4x5rh5bl7vsnr3d4iz4ms43i522xw70xaap9xqv9gc"))

(define rust-tokio-test-0.4.4
  (crate-source "tokio-test" "0.4.4"
                "1xzri2m3dg8nzdyznm77nymvil9cyh1gfdfrbnska51iqfmvls14"))

(define rust-tokio-tungstenite-0.24.0
  (crate-source "tokio-tungstenite" "0.24.0"
                "1nfw1i6yy120a14h1xagd4f31k3g1mz4rdxpvgh77jcd4i7ggigd"))

(define rust-tokio-tungstenite-0.26.2
  (crate-source "tokio-tungstenite" "0.26.2"
                "117hwxfwmpxazxks076w5i73xj7cgv5iqs7x2rnbzln60zvaz7bs"))

(define rust-tokio-util-0.6.10
  (crate-source "tokio-util" "0.6.10"
                "01v5zkcxjdd5zaniqxxfl6isvd7y5qfmljpqsdyrfrvd3bh3x51n"))

(define rust-tokio-util-0.7.14
  (crate-source "tokio-util" "0.7.14"
                "0d7hm1jrwpzryvni72fy5dg9blqs776wq5w38lwigk3g7swr15bb"
                #:snippet '(delete-file "src/sync/tests/mod.rs")))

(define rust-tokio-util-0.7.15
  (crate-source "tokio-util" "0.7.15"
                "1pypd9lm1fdnpw0779pqvc16qqrxjy63dgfm20ajhpbdmnlkk9b6"))

(define rust-tokio-util-0.7.16
  (crate-source "tokio-util" "0.7.16"
                "1r9wdrg1k5hna3m0kc8kcb8jdb6n52g7vnw93kw2xxw4cyc7qc0l"))

(define rust-toml-0.5.11
  (crate-source "toml" "0.5.11"
                "0d2266nx8b3n22c7k24x4428z6di8n83a9n466jm7a2hipfz1xzl"))

(define rust-toml-0.7.8
  (crate-source "toml" "0.7.8"
                "0mr2dpmzw4ndvzpnnli2dprcx61pdk62fq4mzw0b6zb27ffycyfx"))

(define rust-toml-0.8.20
  (crate-source "toml" "0.8.20"
                "0j012b37iz1mihksr6a928s6dzszxvblzg3l5wxp7azzsv6sb1yd"))

(define rust-toml-0.8.22
  (crate-source "toml" "0.8.22"
                "1bjsrmi9w88v4sik8bd6chp0d7bnb9dzzmxyb5xd3i083yfk5bh5"
                #:snippet '(delete-file-recursively "tests")))

(define rust-toml-0.8.23
  (crate-source "toml" "0.8.23"
                "0qnkrq4lm2sdhp3l6cb6f26i8zbnhqb7mhbmksd550wxdfcyn6yw"))

(define rust-toml-0.9.6
  (crate-source "toml" "0.9.6"
                "00g2y1r2znl73s03qfb2sgc977hhqajxyp617gax28yshprlqamf"))

(define rust-toml-0.9.7
  (crate-source "toml" "0.9.7"
                "187av4nsjc0cdfixpc24sqpxqwy5ijvdm7hd9yfsqx94pzcybr80"))

(define rust-toml-datetime-0.6.11
  (crate-source "toml_datetime" "0.6.11"
                "077ix2hb1dcya49hmi1avalwbixmrs75zgzb3b2i7g2gizwdmk92"))

(define rust-toml-datetime-0.6.3
  (crate-source "toml_datetime" "0.6.3"
                "0jsy7v8bdvmzsci6imj8fzgd255fmy5fzp6zsri14yrry7i77nkw"))

(define rust-toml-datetime-0.6.8
  (crate-source "toml_datetime" "0.6.8"
                "0hgv7v9g35d7y9r2afic58jvlwnf73vgd1mz2k8gihlgrf73bmqd"))

(define rust-toml-datetime-0.6.9
  (crate-source "toml_datetime" "0.6.9"
                "18q9ix4isxw2n4mij3v6x0pb4jw1hbxbc5wbprlbq91yjrddp99x"))

(define rust-toml-datetime-0.7.1
  (crate-source "toml_datetime" "0.7.1"
                "1gmn2rmd0g5pnq2vzbd7bnrylnjrl4ml9j42grpzq6qkgpnc15x1"))

(define rust-toml-datetime-0.7.2
  (crate-source "toml_datetime" "0.7.2"
                "1hgff8gdk9yx7dljkqfijmj0sc5ln4xhpj045divdhi7xifhiw9j"))

(define rust-toml-edit-0.19.15
  (crate-source "toml_edit" "0.19.15"
                "08bl7rp5g6jwmfpad9s8jpw8wjrciadpnbaswgywpr9hv9qbfnqv"
                #:snippet '(delete-file-recursively "tests")))

(define rust-toml-edit-0.20.7
  (crate-source "toml_edit" "0.20.7"
                "10bdyrl1yj5jxkiqfa2fyx9inlzlm7s8nf1jnysp4k6qwky2gx3h"
                #:snippet '(delete-file-recursively "tests")))

(define rust-toml-edit-0.22.24
  (crate-source "toml_edit" "0.22.24"
                "0x0lgp70x5cl9nla03xqs5vwwwlrwmd0djkdrp3h3lpdymgpkd0p"
                #:snippet '(delete-file-recursively "tests")))

(define rust-toml-edit-0.22.26
  (crate-source "toml_edit" "0.22.26"
                "0zhk135y6rn6jxik7knia3xblpc545fw6b4xcpkv7idj7n3nh01i"
                #:snippet '(delete-file-recursively "tests")))

(define rust-toml-edit-0.22.27
  (crate-source "toml_edit" "0.22.27"
                "16l15xm40404asih8vyjvnka9g0xs9i4hfb6ry3ph9g419k8rzj1"))

(define rust-toml-parser-1.0.2
  (crate-source "toml_parser" "1.0.2"
                "042wp5ni22yqcbrfqq9c63g2vbbp4m59zamxw97hvacs8ipqhldm"))

(define rust-toml-parser-1.0.3
  (crate-source "toml_parser" "1.0.3"
                "09x6i0b57lwc7yn6w1kbd2ypm4vpcrgd2vdax7h745g77g1r7y2c"))

(define rust-toml-span-0.4.1
  (crate-source "toml-span" "0.4.1"
                "1ifdm3lyqssp1zsfixy3pvm2857b1dx2ssgvv5ga5cz7j3s3czvm"))

(define rust-toml-write-0.1.1
  (crate-source "toml_write" "0.1.1"
                "0xhhyipq7vivp8pfmjkhma4zn9cs5fyybggwwxz9mqnqw7gl5fdz"))

(define rust-toml-write-0.1.2
  (crate-source "toml_write" "0.1.2"
                "008qlhqlqvljp1gpp9rn5cqs74gwvdgbvs92wnpq8y3jlz4zi6ax"))

(define rust-toml-writer-1.0.2
  (crate-source "toml_writer" "1.0.2"
                "0r7x3m050c66s9lssaq965vmrsxvxj131db4fq0m5vrd3w4l5j7w"))

(define rust-toml-writer-1.0.3
  (crate-source "toml_writer" "1.0.3"
                "0281l7bgchmlbvxmci01p9x2w5br9p61ylns5ji65rbc24yacqyi"))

(define rust-tonic-0.12.3
  (crate-source "tonic" "0.12.3"
                "0ljd1lfjpw0vrm5wbv15x6nq2i38llsanls5rkzmdn2n0wrmnz47"))

(define rust-tonic-build-0.12.3
  (crate-source "tonic-build" "0.12.3"
                "04baqblgrlc0g8scnhpky5s0n4cljaixrrdrr6cv6wx7kq8cwmwm"))

(define rust-topological-sort-0.2.2
  (crate-source "topological-sort" "0.2.2"
                "0gcxahg24c058izagz642vs0kfb2zja48my3qrd0kkaf2d730s7a"))

(define rust-totp-lite-2.0.1
  (crate-source "totp-lite" "2.0.1"
                "1hvnpv7nl79jp96w6g2j7l6xskl5qlx3h0qqf9zry68pvcs33r7q"))

(define rust-totp-rs-5.7.0
  (crate-source "totp-rs" "5.7.0"
                "0wmz0ri3d9ml04wrd5qk5a6xq70g8zlzc2g9k61gi3pm10hka97i"))

(define rust-tower-0.4.13
  (crate-source "tower" "0.4.13"
                "073wncyqav4sak1p755hf6vl66njgfc1z1g1di9rxx3cvvh9pymq"))

(define rust-tower-0.5.2
  (crate-source "tower" "0.5.2"
                "1ybmd59nm4abl9bsvy6rx31m4zvzp5rja2slzpn712y9b68ssffh"))

(define rust-tower-http-0.6.5
  (crate-source "tower-http" "0.6.5"
                "18hdkgznns83jhd98adkcsj9jfhjr0hza9j3718s84m4hvhdkhjw"))

(define rust-tower-http-0.6.6
  (crate-source "tower-http" "0.6.6"
                "1wh51y4rf03f91c6rvli6nwzsarx7097yx6sqlm75ag27pbjzj5d"))

(define rust-tower-layer-0.3.3
  (crate-source "tower-layer" "0.3.3"
                "03kq92fdzxin51w8iqix06dcfgydyvx7yr6izjq0p626v9n2l70j"))

(define rust-tower-lsp-server-0.22.0
  (crate-source "tower-lsp-server" "0.22.0"
                "1fcxi53451jv7b8xkpwcq9nl6n9dbxqik9cjfkxflx2i1261dkbn"))

(define rust-tower-service-0.3.3
  (crate-source "tower-service" "0.3.3"
                "1hzfkvkci33ra94xjx64vv3pp0sq346w06fpkcdwjcid7zhvdycd"))

(define rust-tracing-0.1.36
  (crate-source "tracing" "0.1.36"
                "01s3qsm1jfz4h4l401lhy8j2yfds45kpb234l447v9k0pmkrbkig"))

(define rust-tracing-0.1.41
  (crate-source "tracing" "0.1.41"
                "1l5xrzyjfyayrwhvhldfnwdyligi1mpqm8mzbi2m1d6y6p2hlkkq"))

(define rust-tracing-appender-0.2.3
  (crate-source "tracing-appender" "0.2.3"
                "1kq69qyjvb4dxch5c9zgii6cqhy9nkk81z0r4pj3y2nc537fhrim"))

(define rust-tracing-attributes-0.1.22
  (crate-source "tracing-attributes" "0.1.22"
                "1qjf90dvc9jiw78bjzb3iqzrhaybypg8nm0n0zhwi6smmy9miiqi"))

(define rust-tracing-attributes-0.1.28
  (crate-source "tracing-attributes" "0.1.28"
                "0v92l9cxs42rdm4m5hsa8z7ln1xsiw1zc2iil8c6k7lzq0jf2nir"))

(define rust-tracing-attributes-0.1.30
  (crate-source "tracing-attributes" "0.1.30"
                "00v9bhfgfg3v101nmmy7s3vdwadb7ngc8c1iw6wai9vj9sv3lf41"))

(define rust-tracing-chrome-0.7.2
  (crate-source "tracing-chrome" "0.7.2"
                "0977zy46gpawva2laffigxr2pph8v0xa51kfp6ghlifnsn7762mz"))

(define rust-tracing-core-0.1.33
  (crate-source "tracing-core" "0.1.33"
                "170gc7cxyjx824r9kr17zc9gvzx89ypqfdzq259pr56gg5bwjwp6"))

(define rust-tracing-core-0.1.34
  (crate-source "tracing-core" "0.1.34"
                "0y3nc4mpnr79rzkrcylv5f5bnjjp19lsxwis9l4kzs97ya0jbldr"))

(define rust-tracing-durations-export-0.3.0
  (crate-source "tracing-durations-export" "0.3.0"
                "03ssam7j7isyis7khpyfmdg6zznpz5mgab6x8din9nz0z1g04biq"
                #:snippet '(delete-file-recursively "examples")))

(define rust-tracing-error-0.2.1
  (crate-source "tracing-error" "0.2.1"
                "1nzk6qcvhmxxy3lw1nj71anmfmvxlnk78l5lym1389vs1l1825cb"))

(define rust-tracing-forest-0.1.6
  (crate-source "tracing-forest" "0.1.6"
                "0bsr88f4shanjr86ajrx9p8dmsfxxmr24a8llhxixpadn5fq6h7f"))

(define rust-tracing-log-0.2.0
  (crate-source "tracing-log" "0.2.0"
                "1hs77z026k730ij1a9dhahzrl0s073gfa2hm5p0fbl0b80gmz1gf"))

(define rust-tracing-serde-0.2.0
  (crate-source "tracing-serde" "0.2.0"
                "1wbgzi364vzfswfkvy48a3p0z5xmv98sx342r57sil70ggmiljvh"))

(define rust-tracing-subscriber-0.3.18
  (crate-source "tracing-subscriber" "0.3.18"
                "12vs1bwk4kig1l2qqjbbn2nm5amwiqmkcmnznylzmnfvjy6083xd"))

(define rust-tracing-subscriber-0.3.19
  (crate-source "tracing-subscriber" "0.3.19"
                "0220rignck8072i89jjsh140vmh14ydwpdwnifyaf3xcnpn9s678"))

(define rust-tracing-subscriber-0.3.20
  (crate-source "tracing-subscriber" "0.3.20"
                "1m9447bxq7236avgl6n5yb2aqwplrghm61dgipw03mh7ad7s2m10"))

(define rust-tracing-test-0.2.5
  (crate-source "tracing-test" "0.2.5"
                "0s0x076wpga7k1a3cl8da76rrgvs45zzq9rl6q75w3gy6qa8jysm"))

(define rust-tracing-test-macro-0.2.5
  (crate-source "tracing-test-macro" "0.2.5"
                "0s3m7a3pycn8r4xyql5gv5b85sdrqp4w24k1aqy26zf80vdrsr84"))

(define rust-tracing-tree-0.4.0
  (crate-source "tracing-tree" "0.4.0"
                "175lqyfp6zq7jbj8m026xdp8p765pzgfdzfxahfggmdhy5wwlngl"))

(define rust-trackable-1.3.0
  (crate-source "trackable" "1.3.0"
                "1bk96cvr587cdhz8xl7jhkqn7vksyg41grbpx77gi7mrmcad2nxi"))

(define rust-trackable-derive-1.0.0
  (crate-source "trackable_derive" "1.0.0"
                "0bzqh11n1k29cghjmb4dn426hpqy3nbyn1qgzqngiqj7b1f27szb"))

(define rust-tracy-client-0.17.6
  (crate-source "tracy-client" "0.17.6"
                "0zkwz9aq97znyal3hz9wmxya97pj01ddpv92ha7l39a6fdw2s83k"))

(define rust-tracy-client-0.18.1
  (crate-source "tracy-client" "0.18.1"
                "0aikp22kk9ypk87ipwr8wf30al83ggnxw9hdjyd9g3qpjcnq69rr"))

(define rust-tracy-client-0.18.2
  (crate-source "tracy-client" "0.18.2"
                "1msy42lj0gxx7rbx2m7grlqmbkbyn9xlpb9dcvyqc0bn7mfh0m7g"))

(define rust-tracy-client-sys-0.24.3
  (crate-source "tracy-client-sys" "0.24.3"
                "0ps3iwb7q1fzs9pir6b0nqi8n7i67lci4jp6z4xrq8s8lmyz7zv9"))

(define rust-tracy-client-sys-0.25.0
  (crate-source "tracy-client-sys" "0.25.0"
                "1dx7zs5sxn01ffzpf13yr11ywyn19b3g3zk4m0990pfj965dccn0"))

(define rust-tracy-client-sys-0.26.1
  (crate-source "tracy-client-sys" "0.26.1"
                "1c3jn4ka61l4qw0g8w9lw5l0fwid4wz76x2cvdb3za81a4cp171i"))

(define rust-transpose-0.2.3
  (crate-source "transpose" "0.2.3"
                "0zp74v7jrjg4jr654dncdj6hqvacicsywyhc62jawgxwhvnimmhs"))

(define rust-trash-5.2.2
  (crate-source "trash" "5.2.2"
                "0wdq8ax8z47cabs7j9hkghli3pyzaxq8z18diw5dd1bd1imnqx12"))

(define rust-tree-house-0.3.0
  (crate-source "tree-house" "0.3.0"
                "16drgxag73ba809hrgrscfaxj2fx5sv17p84w0d1fbrr499aa3nh"))

(define rust-tree-house-bindings-0.2.1
  (crate-source "tree-house-bindings" "0.2.1"
                "11k03s0yqcn5adqacrbnhpc36wyzb3sx1dciw7wdpbz0ixw4c5iz"))

(define rust-tree-magic-mini-3.1.6
  (crate-source "tree_magic_mini" "3.1.6"
                "0qwx2b0xfr00vdskl951cvh3m040zj5n8vm7ln4k6p143ybyiida"))

(define rust-tree-magic-mini-3.2.0
  (crate-source "tree_magic_mini" "3.2.0"
                "0b2ncw376snr5lwdnmpqgnz4bm1j0c8xf11sq3pfipvci4fkjhzr"))

(define rust-tree-sitter-0.22.6
  (crate-source "tree-sitter" "0.22.6"
                "1jkda5n43m7cxmx2h7l20zxc74nf9v1wpm66gvgxrm5drscw8z6z"))

(define rust-tree-sitter-0.24.7
  (crate-source "tree-sitter" "0.24.7"
                "0xbg9i3kvz316mhwciq147b7kzxqmbvwcl1bw7dd5izzlzzpsf55"))

(define rust-tree-sitter-bash-0.23.3
  (crate-source "tree-sitter-bash" "0.23.3"
                "0bm5chcqq5fvfb505h87d6ab5ny9l60lxy0x5ga3ghrsc944v6ij"))

(define rust-tree-sitter-c-0.23.4
  (crate-source "tree-sitter-c" "0.23.4"
                "0wcwdvp8k9qsyfb5zpa9cq05kc5dp0fx11wysvv2xp452nzv3lmg"))

(define rust-tree-sitter-c-sharp-0.23.1
  (crate-source "tree-sitter-c-sharp" "0.23.1"
                "1c7w6wvjc54k6kh0qrlspm9ksr4y10aq4fv6b0bkaibvrb66mw37"))

(define rust-tree-sitter-cpp-0.23.4
  (crate-source "tree-sitter-cpp" "0.23.4"
                "0hs7p45av437iw8rzsyw46qs06axbam7wadr655apd27kpm9c8fz"))

(define rust-tree-sitter-css-0.23.2
  (crate-source "tree-sitter-css" "0.23.2"
                "17mjy7f1s3cq8dacxaj3ixhqixlra4755gkz5b8m04yljjblimjs"))

(define rust-tree-sitter-elixir-0.3.4
  (crate-source "tree-sitter-elixir" "0.3.4"
                "0grdkbx6bqw3s1w3mkk94sibmhgdicdlqirjzpc57zdl8x348pg4"))

(define rust-tree-sitter-go-0.23.4
  (crate-source "tree-sitter-go" "0.23.4"
                "0cc4w4p12inxpsn2hgpmbvw1nyf5cm0l9pa705hbw3928milfgdi"))

(define rust-tree-sitter-haskell-0.23.1
  (crate-source "tree-sitter-haskell" "0.23.1"
                "19057d99kaq7bn8k86baf7v4q4mjv8p5mjr7zh9vm32l0kjm2z4p"))

(define rust-tree-sitter-highlight-0.24.7
  (crate-source "tree-sitter-highlight" "0.24.7"
                "15axx9avb1cb3mjyhkf5h36n2lsxwqy0z2xrj79qgg4y98z824b4"))

(define rust-tree-sitter-html-0.23.2
  (crate-source "tree-sitter-html" "0.23.2"
                "1vk3xyxnf3xv19qisyj2knd346dq4yjamawv6bg1w1ljbn7706r6"))

(define rust-tree-sitter-java-0.23.5
  (crate-source "tree-sitter-java" "0.23.5"
                "1mlh3skj2nasrwdz0v865r4hxnk7v8037z8nwqab4yf6r36wp9ha"))

(define rust-tree-sitter-javascript-0.23.1
  (crate-source "tree-sitter-javascript" "0.23.1"
                "1cf19p9rl96yqjjhzimhp0dpvp2xxq8fqg2w29nc25h4krcvyh5z"))

(define rust-tree-sitter-json-0.24.8
  (crate-source "tree-sitter-json" "0.24.8"
                "0wf4gsa5mcrcprg8wh647n76rwv4cx8kbky6zw605h06lk67lwjd"))

(define rust-tree-sitter-julia-0.23.1
  (crate-source "tree-sitter-julia" "0.23.1"
                "08z33mq5n5z3xgjjcjrha8b4rrci7f5ykc8rfs3fw4l82wd76i21"))

(define rust-tree-sitter-language-0.1.5
  (crate-source "tree-sitter-language" "0.1.5"
                "1f14p8i09yrfsf9708jmayy0km78n9pqzxmf31xzd0vk45q3j0f4"))

(define rust-tree-sitter-lua-0.2.0
  (crate-source "tree-sitter-lua" "0.2.0"
                "07k4753h1nz3pbffcnclxjz2xcfvb6hb7jv0fs7cbzk517grmnsw"))

(define rust-tree-sitter-make-1.1.1
  (crate-source "tree-sitter-make" "1.1.1"
                "0101h5ilrv2aqjffdlnq2d2m9mpj5fcfzvwamsgv3nnbrg3qv6f5"))

(define rust-tree-sitter-md-0.3.2
  (crate-source "tree-sitter-md" "0.3.2"
                "0c5h0j9cp5rq27cig1iqnsz8vc4xfap5a14nzj1hn0815b16iy8p"))

(define rust-tree-sitter-nix-0.0.2
  (crate-source "tree-sitter-nix" "0.0.2"
                "0160v6rqal8lsw9slx7x52ccq7cc5lfk6xd088rdcxyk0n3lz39s"))

(define rust-tree-sitter-objc-3.0.2
  (crate-source "tree-sitter-objc" "3.0.2"
                "1lp1570h6lwhknzq3nn9sf26cfkqbx99vrrm0mpigz13ciavpa4w"))

(define rust-tree-sitter-ocaml-0.23.2
  (crate-source "tree-sitter-ocaml" "0.23.2"
                "0xz3dkvb40b5anir8ld7547w2kibbms75y7i1kfhcn8p7ni09hck"))

(define rust-tree-sitter-php-0.23.11
  (crate-source "tree-sitter-php" "0.23.11"
                "030kiknyk2lw54yj7mzj92kfr5v0qr81qymhvkqy9kvjj97fjrph"))

(define rust-tree-sitter-python-0.23.6
  (crate-source "tree-sitter-python" "0.23.6"
                "012bgzycya91lpdbrrr8xnw9xjz116nf1w61c2pwxapk4ym5l1ix"))

(define rust-tree-sitter-ruby-0.23.1
  (crate-source "tree-sitter-ruby" "0.23.1"
                "15cz4h1sfgf838r2pmf7vg9ahh0kwgkvvnjgbdbrrfzn9vm8815y"))

(define rust-tree-sitter-rust-0.23.3
  (crate-source "tree-sitter-rust" "0.23.3"
                "12lf0v0a8zrwca3n1wglswglfk40qcjgvhzn8flwi59l78zcp36a"))

(define rust-tree-sitter-scala-0.23.4
  (crate-source "tree-sitter-scala" "0.23.4"
                "0bsxq5ihmi4qp1g3cfrnmgznp8h4y739d8mz2yn9wvkknil5xppg"))

(define rust-tree-sitter-toml-ng-0.6.0
  (crate-source "tree-sitter-toml-ng" "0.6.0"
                "0mblvbhba1lg59jy6b2f7n41jfxlsxv3xw3kqw16rwdchg6j0pb9"))

(define rust-tree-sitter-toml-ng-0.7.0
  (crate-source "tree-sitter-toml-ng" "0.7.0"
                "1cgbwl6x33d033ws4dwf3nw2pyd37m0bwxbxhl776jdfk34c5bg9"))

(define rust-tree-sitter-typescript-0.23.2
  (crate-source "tree-sitter-typescript" "0.23.2"
                "1zsyaxx3v1sd8gx2zkscwv6z1sq2nvccqpvd8k67ayllipnpcpvc"))

(define rust-tree-sitter-xml-0.7.0
  (crate-source "tree-sitter-xml" "0.7.0"
                "1cxnr3q72fvld0ia8xjc5hl0x4xw9s7wvpcpsma4z68xb4gh8w76"))

(define rust-tree-sitter-yaml-0.7.0
  (crate-source "tree-sitter-yaml" "0.7.0"
                "0phdym735blwnb8aff4225c5gyws6aljy8vbifhz2xxnj8mrzjfh"))

(define rust-trim-in-place-0.1.7
  (crate-source "trim-in-place" "0.1.7"
                "1z04g79xkrpf3h4g3cc8wax72dn6h6v9l4m39zg8rg39qrpr4gil"))

(define rust-triomphe-0.1.14
  (crate-source "triomphe" "0.1.14"
                "11fciha522hrz6pkafy3xlq20w405w9dqvy9ln7ba1s8v8k7g3zg"))

(define rust-triple-accel-0.3.4
  (crate-source "triple_accel" "0.3.4"
                "0v795l496crk3h6yff9zh1cjyrh5s9v23fbgccc4dpz25z70jav2"))

(define rust-triple-accel-0.4.0
  (crate-source "triple_accel" "0.4.0"
                "0qqyhl1pdvmfbx9fgw5jc15j42d0j1i7b6pzn42zsbzvbp4qn112"))

(define rust-trotter-1.0.2
  (crate-source "trotter" "1.0.2"
                "14gksihjm4bv2paqg22ym7n63adb69zac1qkp51yxz84572bhmkw"))

(define rust-trust-dns-proto-0.22.0
  (crate-source "trust-dns-proto" "0.22.0"
                "09mwv7fnjrkdpf82qqvsbby5xnnpwn0kcp2cqn53br50wk8q6zsg"))

(define rust-try-lock-0.2.5
  (crate-source "try-lock" "0.2.5"
                "0jqijrrvm1pyq34zn1jmy2vihd4jcrjlvsh4alkjahhssjnsn8g4"))

(define rust-trybuild-1.0.104
  (crate-source "trybuild" "1.0.104"
                "161qw9ajl3njzp3lq7wyxmn7bjkwf8hdhvbc1rpvjv85ikk8pq3a"))

(define rust-trycmd-0.15.9
  (crate-source "trycmd" "0.15.9"
                "1r5a5r22j7gi69y0zdbwhb6d2hp8r34plnfncp0alql870lwzdd8"))

(define rust-ttf-parser-0.15.2
  (crate-source "ttf-parser" "0.15.2"
                "1pfqn06vjlr6pvlljjmkmcb2kb8ind09q5f78nvxc3nqp74hcgkv"
                #:snippet '(for-each delete-file-recursively '("examples" "tests"))))

(define rust-ttf-parser-0.24.1
  (crate-source "ttf-parser" "0.24.1"
                "0jmcif537g3smzgcx0vfr4wwq4pnaypa7dnklasfhf2xzy813qjv"
                #:snippet '(for-each delete-file-recursively '("examples" "tests" "testing-tools"))))

(define rust-tui-react-0.23.2
  (crate-source "tui-react" "0.23.2"
                "1inc78yngwkambrpcs1alallndysnzdq6p8pd7c12cqgw00sf519"))

(define rust-tui-textarea-0.7.0
  (crate-source "tui-textarea" "0.7.0"
                "1bnb3andd8gcqkf5rdvz530zykkj8s8d2yj1m593rmwyc7fihlqa"))

(define rust-tuikit-0.5.0
  (crate-source "tuikit" "0.5.0"
                "1n2kiq6kjl1g89v8f9v2dwcv5g8hp7w2zhf8a0yyxawb0fmwc6ay"))

(define rust-tungstenite-0.24.0
  (crate-source "tungstenite" "0.24.0"
                "12nsxnxazk4nisgsqpywi6za0hsbc2rd15r1scb5pag7dqvbir8q"))

(define rust-tungstenite-0.26.2
  (crate-source "tungstenite" "0.26.2"
                "04rwwcxx95m3avi46rmn0kmpb6nynqimnla3v2qwn3k8argcp4s7"))

(define rust-two-face-0.4.3
  ;; TODO: Generate from source.
  (crate-source "two-face" "0.4.3"
                "0lpqra6ryq4q47iavmcabbgxknajv59485wsyg3f4qnzim1xlkiq"))

(define rust-twofish-0.7.1
  (crate-source "twofish" "0.7.1"
                "04w0ii2c0c9ws08aw6c7illh9zql22il9lbwjk1mgir30aiq73m7"))

(define rust-twox-hash-1.6.3
  (crate-source "twox-hash" "1.6.3"
                "0xgn72j36a270l5ls1jk88n7bmq2dhlfkbhdh5554hbagjsydzlp"))

(define rust-twox-hash-2.1.0
  (crate-source "twox-hash" "2.1.0"
                "022rwrv24rl6g32nqv1mywf6vdnkn7vq34fg793vll1hgccpzcg7"))

(define rust-type-map-0.5.1
  (crate-source "type-map" "0.5.1"
                "143v32wwgpymxfy4y8s694vyq0wdi7li4s5dmms5w59nj2yxnc6b"))

(define rust-typed-arena-2.0.2
  (crate-source "typed-arena" "2.0.2"
                "0shj0jpmglhgw2f1i4b33ycdzwd1z205pbs1rd5wx7ks2qhaxxka"))

(define rust-typed-sled-0.2.3
  (crate-source "typed-sled" "0.2.3"
                "03ah6i05rwkcwwq6ccbr42k96ni29yq522d6imsmpv2h8idg0q0h"))

(define rust-typeid-1.0.3
  (crate-source "typeid" "1.0.3"
                "0727ypay2p6mlw72gz3yxkqayzdmjckw46sxqpaj08v0b0r64zdw"))

(define rust-typenum-1.18.0
  (crate-source "typenum" "1.18.0"
                "0gwgz8n91pv40gabrr1lzji0b0hsmg0817njpy397bq7rvizzk0x"))

(define rust-typenum-1.19.0
  (crate-source "typenum" "1.19.0"
                "1fw2mpbn2vmqan56j1b3fbpcdg80mz26fm53fs16bq5xcq84hban"))

(define rust-typetag-0.2.20
  (crate-source "typetag" "0.2.20"
                "0vyyqg2ard4jz8jxdcim0j4j2r1nh4h71y8c4ca8rzkvvm02pwkk"))

(define rust-typetag-impl-0.2.20
  (crate-source "typetag-impl" "0.2.20"
                "0lnrad6qrzgw77p8ra9cm8ba0xbihxmzkgglwi6iphzz144kix9m"))

(define rust-typst-0.13.1
  (crate-source "typst" "0.13.1"
                "1x29iwm4p5f8964ylphh8nc1bj35f7azm9cfdwk19c3jijy5h10l"))

(define rust-typst-assets-0.13.1
  ;; NB: The ICC color profiles and postcard files within are open formats.
  ;; TODO: Un-bundle fonts from this and upstream typst-cli by disabling the
  ;; "font" feature here and the embed-fonts feature there.
  (crate-source "typst-assets" "0.13.1"
                "00mrmc0vkc1a7jwrawahjjvcwlf9qxy18wyb3ysh4m96qb1hrgxm"))

(define rust-typst-eval-0.13.1
  (crate-source "typst-eval" "0.13.1"
                "1b05298979xk0q12l7s1y913s6kdn1wyjl7xac0ipnzcbg4k61w5"))

(define rust-typst-html-0.13.1
  (crate-source "typst-html" "0.13.1"
                "1k770plp5p41ddrr23865bvnvv4adfqnw5i3jyqzj14iiflfakb5"))

(define rust-typst-kit-0.13.1
  (crate-source "typst-kit" "0.13.1"
                "141azrb7bp8axg2fckccsqk41k0nasa5jqszs4hgwqqn9vask3w1"))

(define rust-typst-layout-0.13.1
  (crate-source "typst-layout" "0.13.1"
                "15n6xrnvdqdaq04fzlyz3dd8lbm80dsaij23k57fa8s4qhlp2zyw"))

(define rust-typst-library-0.13.1
  (crate-source "typst-library" "0.13.1"
                "0j1hpkrrjzhdmh83mkc5ybim5x14ys32aa3zvavmpah5zmnhsavj"))

(define rust-typst-macros-0.13.1
  (crate-source "typst-macros" "0.13.1"
                "1yx6amz6jdg990s08xkllgczxmdyvfzjmrs4azi12cgzy9xnd9y5"))

(define rust-typst-pdf-0.13.1
  (crate-source "typst-pdf" "0.13.1"
                "1965prn6b1vyibld1jf1pjmq2jh0jnlnz0p9krgfvd989jlii1x7"))

(define rust-typst-realize-0.13.1
  (crate-source "typst-realize" "0.13.1"
                "1bxqllpj9dqzdy1r1n1ba69093xaqf12sa953z06961v631j9ivv"))

(define rust-typst-render-0.13.1
  (crate-source "typst-render" "0.13.1"
                "0kphv5vh7b4q3vpznpgqw0jxvnkc0l1y98m41y8rhyf3viz2fxif"))

(define rust-typst-svg-0.13.1
  (crate-source "typst-svg" "0.13.1"
                "1bpy0dmnqq373wxkyr36aw3hddsfnvkar2y4f3wcj5a0p5f8wk37"))

(define rust-typst-syntax-0.12.0
  (crate-source "typst-syntax" "0.12.0"
                "14vn68h92xas9kj0hh14paq7v3d293alfna9r8wwpcnnds5vxdq5"))

(define rust-typst-syntax-0.13.1
  (crate-source "typst-syntax" "0.13.1"
                "1a13bsxqz65s5j5zf5lx2zk1if5ccdn2xlv15xmylx53fnn4kaav"))

(define rust-typst-timing-0.13.1
  (crate-source "typst-timing" "0.13.1"
                "0xm9vjwim3k6khia0qrhj3mr0q90my97s9mj7l1f52z9chb5994b"))

(define rust-typst-utils-0.12.0
  (crate-source "typst-utils" "0.12.0"
                "0h3s9y3bvkkiqvfgn9bmbrq3by3bz0l74j3ihijhnzyr7r20a0wg"))

(define rust-typst-utils-0.13.1
  (crate-source "typst-utils" "0.13.1"
                "15px12a3qrdfhqkclp3hcc655nnqrr0lzagq84iv63vyjrcivdqf"))

(define rust-typstyle-core-0.13.10
  (crate-source "typstyle-core" "0.13.10"
                "1vfhvf1wpyjx925dwd3n70kbw6k7mm84b6qqp2vj96xi2a8s1ysh"))

(define rust-tz-rs-0.7.0
  (crate-source "tz-rs" "0.7.0"
                "18bi7k9zgwbm0ch049c1mj901a6aza4mr4z7f0hfg5wkp7r0nig1"))

(define rust-ucd-trie-0.1.7
  (crate-source "ucd-trie" "0.1.7"
                "0wc9p07sqwz320848i52nvyjvpsxkx3kv5bfbmm6s35809fdk5i8"))

(define rust-udev-0.7.0
  (crate-source "udev" "0.7.0"
                "06hr927z0fdn7ay0p817b9x19i5fagmpmvz95yhl4d1pf3bbpgaf"))

(define rust-udev-0.8.0
  (crate-source "udev" "0.8.0"
                "18f5vkhjcljw1p5gfvy61p1837n27d7h2l3x45pywa5y49p1q1ah"))

(define rust-udev-0.9.3
  (crate-source "udev" "0.9.3"
                "17vy1yc6ipb5m2kc2d4lx2qpj45yr7grsjzm3y2gq0a4xblkfkmg"))

(define rust-udevrs-0.3.0
  (crate-source "udevrs" "0.3.0"
                "028s1sk1ym919z0w6p8mnvgh9d35c9zzrlaxlmvfrppi6fx5kjdx"
                #:snippet '(delete-file-recursively "tests")))

(define rust-uds-windows-1.1.0
  (crate-source "uds_windows" "1.1.0"
                "1fb4y65pw0rsp0gyfyinjazlzxz1f6zv7j4zmb20l5pxwv1ypnl9"))

(define rust-uluru-3.1.0
  (crate-source "uluru" "3.1.0"
                "1njp6vvy1mm8idnsp6ljyxx5znfsk3xkmk9cr2am0vkfwmlj92kw"))

(define rust-umask-2.1.0
  (crate-source "umask" "2.1.0"
                "071xszsd6znk0ik11pxl7mwhf07clsiq3qpzw1ac0dcyak14d6pc"))

(define rust-uname-0.1.1
  (crate-source "uname" "0.1.1"
                "1j1xd1rryml4j1hf07kahva9d5ym8m9jz9z20hfdpr1jrbq8jbxp"))

(define rust-unarray-0.1.4
  (crate-source "unarray" "0.1.4"
                "154smf048k84prsdgh09nkm2n0w0336v84jd4zikyn6v6jrqbspa"))

(define rust-uncased-0.9.10
  (crate-source "uncased" "0.9.10"
                "15q6r6g4fszr8c2lzg9z9k9g52h8g29h24awda3d72cyw37qzf71"))

(define rust-unescape-0.1.0
  (crate-source "unescape" "0.1.0"
                "0vlgws15n4kz8xq4igzr1f80nbiyr838k687hn6ly8a36an7vffc"))

(define rust-unescaper-0.1.6
  (crate-source "unescaper" "0.1.6"
                "09hzbayg38dvc298zygrx7wvs228bz197winnjl34i3alpii47f0"))

(define rust-unic-char-property-0.9.0
  (crate-source "unic-char-property" "0.9.0"
                "08g21dn3wwix3ycfl0vrbahn0835nv2q3swm8wms0vwvgm07mid8"))

(define rust-unic-char-range-0.9.0
  (crate-source "unic-char-range" "0.9.0"
                "1g0z7iwvjhqspi6194zsff8vy6i3921hpqcrp3v1813hbwnh5603"))

(define rust-unic-common-0.9.0
  (crate-source "unic-common" "0.9.0"
                "1g1mm954m0zr497dl4kx3vr09yaly290zs33bbl4wrbaba1gzmw0"))

(define rust-unic-langid-0.9.6
  (crate-source "unic-langid" "0.9.6"
                "01bx59sqsx2jz4z7ppxq9kldcjq9dzadkmb2dr7iyc85kcnab2x2"))

(define rust-unic-langid-impl-0.9.6
  (crate-source "unic-langid-impl" "0.9.6"
                "0n66kdan4cz99n8ra18i27f7w136hmppi4wc0aa7ljsd0h4bzqfw"))

(define rust-unic-segment-0.9.0
  (crate-source "unic-segment" "0.9.0"
                "08wgz2q6vrdvmbd23kf9pbg8cyzm5q8hq9spc4blzy2ppqk5vvg4"))

(define rust-unic-ucd-block-0.9.0
  (crate-source "unic-ucd-block" "0.9.0"
                "1mzinqvb5qfjz2bmwp56264vd8bzdskaag05l4jm7lpcszr1cakb"))

(define rust-unic-ucd-category-0.9.0
  (crate-source "unic-ucd-category" "0.9.0"
                "1h4ixzplc2s441vc8mc4zxliw6qfqh1ziaiv8pa1pzpwyn8lb38v"))

(define rust-unic-ucd-segment-0.9.0
  (crate-source "unic-ucd-segment" "0.9.0"
                "0027lczcg0r401g6fnzm2bq9fxhgxvri1nlryhhv8192lqic2y90"))

(define rust-unic-ucd-version-0.9.0
  (crate-source "unic-ucd-version" "0.9.0"
                "1i5hnzpfnxkp4ijfk8kvhpvj84bij575ybqx1b6hyigy6wi2zgcn"))

(define rust-unicase-2.7.0
  (crate-source "unicase" "2.7.0"
                "12gd74j79f94k4clxpf06l99wiv4p30wjr0qm04ihqk9zgdd9lpp"))

(define rust-unicase-2.8.1
  (crate-source "unicase" "2.8.1"
                "0fd5ddbhpva7wrln2iah054ar2pc1drqjcll0f493vj3fv8l9f3m"))

(define rust-unicode-bidi-0.3.17
  (crate-source "unicode-bidi" "0.3.17"
                "14vqdsnrm3y5anj6h5zz5s32w88crraycblb88d9k23k9ns7vcas"))

(define rust-unicode-bidi-0.3.18
  (crate-source "unicode-bidi" "0.3.18"
                "1xcxwbsqa24b8vfchhzyyzgj0l6bn51ib5v8j6krha0m77dva72w"))

(define rust-unicode-bidi-mirroring-0.3.0
  (crate-source "unicode-bidi-mirroring" "0.3.0"
                "0bq832hqj8qbk14sk4qxyi3zk2ldhdhvw9ii2759ar26sxx0bbv4"))

(define rust-unicode-bom-2.0.3
  (crate-source "unicode-bom" "2.0.3"
                "05s2sqyjanqrbds3fxam35f92npp5ci2wz9zg7v690r0448mvv3y"))

(define rust-unicode-categories-0.1.1
  (crate-source "unicode_categories" "0.1.1"
                "0kp1d7fryxxm7hqywbk88yb9d1avsam9sg76xh36k5qx2arj9v1r"))

(define rust-unicode-ccc-0.3.0
  (crate-source "unicode-ccc" "0.3.0"
                "0hnyjqjaxxnsqfhc7capa9dbi5jzl41hhdk80slsk4rqgdjcc2r6"))

(define rust-unicode-display-width-0.3.0
  (crate-source "unicode-display-width" "0.3.0"
                "1wqb96ir19flkf557ch6lzng0mf2hzz52lz6p0mslh31clxjfhws"))

(define rust-unicode-ellipsis-0.2.0
  (crate-source "unicode-ellipsis" "0.2.0"
                "1zsdzmy6x1p8s35rgfmc7nx1qcs6j4bcfbfyiimrdngyqfwbajlj"))

(define rust-unicode-general-category-1.0.0
  (crate-source "unicode-general-category" "1.0.0"
                "1ls6j9faq62kqnvzcxjdjgjfcqcif8jz3pvag83nfja3261zxb94"))

(define rust-unicode-id-0.3.5
  (crate-source "unicode-id" "0.3.5"
                "0qdmwsyi72c17v64h1is28f0gc2d3bvhjz264i998c270ibkq40h"))

(define rust-unicode-ident-1.0.11
  (crate-source "unicode-ident" "1.0.11"
                "0g7wmn39nl9yzhjwn9ihacd22ymli8r4nlc2xf3idaas8ypbl6ih"
                #:snippet '(delete-file-recursively "tests")))

(define rust-unicode-ident-1.0.12
  (crate-source "unicode-ident" "1.0.12"
                "0jzf1znfpb2gx8nr8mvmyqs1crnv79l57nxnbiszc7xf7ynbjm1k"
                #:snippet '(delete-file-recursively "tests")))

(define rust-unicode-ident-1.0.13
  (crate-source "unicode-ident" "1.0.13"
                "1zm1xylzsdfvm2a5ib9li3g5pp7qnkv4amhspydvgbmd9k6mc6z9"))

(define rust-unicode-ident-1.0.17
  (crate-source "unicode-ident" "1.0.17"
                "1gpdxvaskz04whays5igg4zyca0dl7vdy2arsfxb13kpjcx4gqh0"
                #:snippet '(delete-file-recursively "tests")))

(define rust-unicode-ident-1.0.18
  (crate-source "unicode-ident" "1.0.18"
                "04k5r6sijkafzljykdq26mhjpmhdx4jwzvn1lh90g9ax9903jpss"
                #:snippet '(delete-file-recursively "tests")))

(define rust-unicode-ident-1.0.19
  (crate-source "unicode-ident" "1.0.19"
                "17bx1j1zf6b9j3kpyf74mraary7ava3984km0n8kh499h5a58fpn"
                #:snippet '(delete-file-recursively "tests")))

(define rust-unicode-joining-type-0.7.0
  (crate-source "unicode-joining-type" "0.7.0"
                "1xgn8cjv3r3s9f3zqs3v2k6hv8d43l3z6nkm1047bg5qri3wpy12"))

(define rust-unicode-linebreak-0.1.5
  (crate-source "unicode-linebreak" "0.1.5"
                "07spj2hh3daajg335m4wdav6nfkl0f6c0q72lc37blr97hych29v"))

(define rust-unicode-math-class-0.1.0
  (crate-source "unicode-math-class" "0.1.0"
                "0rbxcjirldpdrpxv1l7qiadbib8rnl7b413fsp4f7ynmk7snq93x"))

(define rust-unicode-normalization-0.1.23
  (crate-source "unicode-normalization" "0.1.23"
                "1x81a50h2zxigj74b9bqjsirxxbyhmis54kg600xj213vf31cvd5"))

(define rust-unicode-normalization-0.1.24
  (crate-source "unicode-normalization" "0.1.24"
                "0mnrk809z3ix1wspcqy97ld5wxdb31f3xz6nsvg5qcv289ycjcsh"))

(define rust-unicode-normalization-alignments-0.1.12
  (crate-source "unicode-normalization-alignments" "0.1.12"
                "1pk2f3arh3qvdsmrsiri0gr5y5vqpk2gv1yjin0njvh4zbj17xj3"))

(define rust-unicode-properties-0.1.3
  (crate-source "unicode-properties" "0.1.3"
                "1l3mbgzwz8g14xcs09p4ww3hjkjcf0i1ih13nsg72bhj8n5jl3z7"))

(define rust-unicode-script-0.5.7
  (crate-source "unicode-script" "0.5.7"
                "07vwr9iddw5xwrj57hc6ig0mwmlzjdajj9lyfxqz9by9a2rj3d4z"))

(define rust-unicode-segmentation-1.10.1
  (crate-source "unicode-segmentation" "1.10.1"
                "0dky2hm5k51xy11hc3nk85p533rvghd462b6i0c532b7hl4j9mhx"))

(define rust-unicode-segmentation-1.12.0
  (crate-source "unicode-segmentation" "1.12.0"
                "14qla2jfx74yyb9ds3d2mpwpa4l4lzb9z57c6d2ba511458z5k7n"))

(define rust-unicode-truncate-1.1.0
  (crate-source "unicode-truncate" "1.1.0"
                "1gr7arjjhrhy8dww7hj8qqlws97xf9d276svr4hs6pxgllklcr5k"))

(define rust-unicode-truncate-2.0.0
  (crate-source "unicode-truncate" "2.0.0"
                "0c33c29m85wb5c0vbvz724f3rinsz0l5zjhh74pp0f7l1y307gwg"))

(define rust-unicode-vo-0.1.0
  (crate-source "unicode-vo" "0.1.0"
                "151sha088v9jyfvbg5164xh4dk72g53b82xm4zzbf5dlagzqdlxi"))

(define rust-unicode-width-0.1.10
  (crate-source "unicode-width" "0.1.10"
                "12vc3wv0qwg8rzcgb9bhaf5119dlmd6lmkhbfy1zfls6n7jx3vf0"))

(define rust-unicode-width-0.1.12
  (crate-source "unicode-width" "0.1.12"
                "1mk6mybsmi5py8hf8zy9vbgs4rw4gkdqdq3gzywd9kwf2prybxb8"))

(define rust-unicode-width-0.1.14
  (crate-source "unicode-width" "0.1.14"
                "1bzn2zv0gp8xxbxbhifw778a7fc93pa6a1kj24jgg9msj07f7mkx"))

(define rust-unicode-width-0.2.0
  (crate-source "unicode-width" "0.2.0"
                "1zd0r5vs52ifxn25rs06gxrgz8cmh4xpra922k0xlmrchib1kj0z"))

(define rust-unicode-width-0.2.1
  (crate-source "unicode-width" "0.2.1"
                "0k0mlq7xy1y1kq6cgv1r2rs2knn6rln3g3af50rhi0dkgp60f6ja"))

(define rust-unicode-xid-0.1.0
  (crate-source "unicode-xid" "0.1.0"
                "1z57lqh4s18rr4x0j4fw4fmp9hf9346h0kmdgqsqx0fhjr3k0wpw"))

(define rust-unicode-xid-0.2.6
  (crate-source "unicode-xid" "0.2.6"
                "0lzqaky89fq0bcrh6jj6bhlz37scfd8c7dsj5dq7y32if56c1hgb"))

(define rust-unindent-0.1.11
  (crate-source "unindent" "0.1.11"
                "171may3v15wzc10z64i8sahdz49d031v7424mjsifa205ml6sxp1"))

(define rust-unindent-0.2.3
  (crate-source "unindent" "0.2.3"
                "1km2iy6fr6gsh2wvr1mxz86pm4wrlh3fjkinb35qfi3mw5rpvpn7"))

(define rust-unindent-0.2.4
  (crate-source "unindent" "0.2.4"
                "1wvfh815i6wm6whpdz1viig7ib14cwfymyr1kn3sxk2kyl3y2r3j"))

(define rust-unit-prefix-0.5.1
  (crate-source "unit-prefix" "0.5.1"
                "05rq0asf2f1q5vrcv4bwf0c3y6q20asqkiqpr8wqyrfxyb7h4d1j"))

(define rust-universal-hash-0.5.1
  (crate-source "universal-hash" "0.5.1"
                "1sh79x677zkncasa95wz05b36134822w6qxmi1ck05fwi33f47gw"))

(define rust-unrar-0.5.8 #f)

(define rust-unrar-sys-0.5.8 #f)

(define rust-unreachable-0.1.1
  (crate-source "unreachable" "0.1.1"
                "14gzpm329nrfgjvxf6zh77sp7plxakcnsq8p8xk9474fn7fyaahz"))

(define rust-unsafe-libyaml-0.2.11
  (crate-source "unsafe-libyaml" "0.2.11"
                "0qdq69ffl3v5pzx9kzxbghzn0fzn266i1xn70y88maybz9csqfk7"))

(define rust-unsafe-libyaml-norway-0.2.15
  (crate-source "unsafe-libyaml-norway" "0.2.15"
                "0111lbq845fwqv8cn89m02v7bjd2lq2jvd814dziqlijpxcvv6mk"))

(define rust-unscanny-0.1.0
  (crate-source "unscanny" "0.1.0"
                "0ivbipc1rnq15fhzgna41p1h01ncq4shycii72f3x5d7czq2mpz9"))

(define rust-untrusted-0.7.1
  (crate-source "untrusted" "0.7.1"
                "0jkbqaj9d3v5a91pp3wp9mffvng1nhycx6sh4qkdd9qyr62ccmm1"))

(define rust-untrusted-0.9.0
  (crate-source "untrusted" "0.9.0"
                "1ha7ib98vkc538x0z60gfn0fc5whqdd85mb87dvisdcaifi6vjwf"
                #:snippet '(delete-file-recursively "mk")))

(define rust-unwinding-0.2.5
  (crate-source "unwinding" "0.2.5"
                "0g5bp3xqpsd4zwiswx0znqk24qcnzqjzafzgmi30jrcghh2nmw2i"))

(define rust-unwinding-0.2.6
  (crate-source "unwinding" "0.2.6"
                "0idkswxp6vnb33gj895sz435588w1iw57wvz6c3shq305dwg54w3"))

(define rust-uom-0.30.0
  (crate-source "uom" "0.30.0"
                "1vg59hnb7hh0p8kjjhgmrsnn3597722lkfdkp481wksq6vk06rg7"))

(define rust-uom-0.36.0
  (crate-source "uom" "0.36.0"
                "176dszv02ki1a49hvv2nkszjcn4m8ccfjgh5hhji2pd6a19nxlzz"))

(define rust-update-informer-1.2.0
  (crate-source "update-informer" "1.2.0"
                "19w2s4wqpjrw70b8v5b6h0lrxk11jpllik7qjh3l7n7hspskp0ak"))

(define rust-upon-0.8.1
  (crate-source "upon" "0.8.1"
                "1h0ky0lvilja2bw3337w3jyxhgajbwdafpm3m57i0kv2s40rdqlz"))

(define rust-ureq-2.10.0.1cad58f
  (origin
    (method git-fetch)
    (uri (git-reference (url "https://github.com/algesten/ureq")
                        (commit "1cad58f5a4f359e318858810de51666d63de70e8")))
    (file-name (git-file-name "rust-ureq" "2.10.0.1cad58f"))
    (sha256 (base32 "1ryn499kbv44h3lzibk9568ln13yi10frbpjjnrn7dz0lkrdin2w"))))

(define rust-ureq-2.12.1
  (crate-source "ureq" "2.12.1"
                "07f0qdn6459k4rmdnkivkz0y7j28vxh5c8q8sr0gcxgdfxiadl82"))

(define rust-ureq-3.0.10
  (crate-source "ureq" "3.0.10"
                "07979dkp96fdrbgbwzhqq9i7zxjmjxfnrfzrwjl42yswcb5520sb"))

(define rust-ureq-proto-0.3.5
  (crate-source "ureq-proto" "0.3.5"
                "195g5y87f6y9f9kq3g5y3ka8gkngczfd254l4nabrbj16c59s8xf"
                #:snippet '(delete-file-recursively "doc")))

(define rust-urid-0.1.0
  (crate-source "urid" "0.1.0"
                "195672gs136vczn1r4hkjg5vfa7vdzr26bzv6lwhk0z7cvbvaa38"))

(define rust-urid-derive-0.1.1
  (crate-source "urid-derive" "0.1.1"
                "0i1nf0sgq4ai051h17s9msaavl3jfzdmdlsy8455pr88y0pfx7l1"))

(define rust-url-2.5.3
  (crate-source "url" "2.5.3"
                "1nis27cv09039nds9n83di92a23q5rqvl40zvkzh0ifijqdpy5cd"))

(define rust-url-2.5.4
  (crate-source "url" "2.5.4"
                "0q6sgznyy2n4l5lm16zahkisvc9nip9aa5q1pps7656xra3bdy1j"))

(define rust-url-2.5.6
  (crate-source "url" "2.5.6"
                "1i574rp7k34vdx9rnkcfbmy3782124zjyl486svkjwdg9s1kqyhk"))

(define rust-url-2.5.7
  (crate-source "url" "2.5.7"
                "0nzghdv0kcksyvri0npxbjzyx2ihprks5k590y77bld355m17g08"))

(define rust-urlencoding-1.3.3
  (crate-source "urlencoding" "1.3.3"
                "1yqgq2qigm1s8zyv23j0422j4vn20ppnnizx9b7p629sw1sh27ss"))

(define rust-urlencoding-2.1.3
  (crate-source "urlencoding" "2.1.3"
                "1nj99jp37k47n0hvaz5fvz7z6jd0sb4ppvfy3nphr1zbnyixpy6s"))

(define rust-usb-ids-1.2025.2
  (crate-source "usb-ids" "1.2025.2"
                "10rk0iqf847w54238znyhcwlkx4d6vx1pj71z8kvm1rjk41lsihz"))

(define rust-ustr-0.9.0
  (crate-source "ustr" "0.9.0"
                "177sfj2gcxcxbyw756k4r7dbpsnkjrb62nrqvibdx019kw4kc51p"))

(define rust-usvg-0.43.0
  (crate-source "usvg" "0.43.0"
                "1z9mfx1nw00kpjz3rfh5qz91vjmlm7ric8nfp2gnwhmvbixha0v8"))

(define rust-utf-8-0.7.6
  (crate-source "utf-8" "0.7.6"
                "1a9ns3fvgird0snjkd3wbdhwd3zdpc2h5gpyybrfr6ra5pkqxk09"))

(define rust-utf16-iter-1.0.5
  (crate-source "utf16_iter" "1.0.5"
                "0ik2krdr73hfgsdzw0218fn35fa09dg2hvbi1xp3bmdfrp9js8y8"))

(define rust-utf8-iter-1.0.4
  (crate-source "utf8_iter" "1.0.4"
                "1gmna9flnj8dbyd8ba17zigrp9c4c3zclngf5lnb5yvz1ri41hdn"))

(define rust-utf8-width-0.1.7
  (crate-source "utf8-width" "0.1.7"
                "1qwj8c0fg8cpn8hq7c9xzz26kdz6ci32bf0madz57a2xi578vgc6"))

(define rust-utf8parse-0.2.1
  (crate-source "utf8parse" "0.2.1"
                "02ip1a0az0qmc2786vxk2nqwsgcwf17d3a38fkf0q7hrmwh9c6vi"))

(define rust-utf8parse-0.2.2
  (crate-source "utf8parse" "0.2.2"
                "088807qwjq46azicqwbhlmzwrbkz7l4hpw43sdkdyyk524vdxaq6"))

(define rust-uu-cp-0.0.29
  (crate-source "uu_cp" "0.0.29"
                "1a2s7z41fr5lv74fys8m7j8cwpc4n6sjr6chvg5rkxygxzw3jm13"))

(define rust-uu-mkdir-0.0.29
  (crate-source "uu_mkdir" "0.0.29"
                "0lk96l3lh0lajc0dwfnnbzq8f7xnxdizxq92nlbfm9w81mi302pa"))

(define rust-uu-mktemp-0.0.29
  (crate-source "uu_mktemp" "0.0.29"
                "1k3k0i3shdkpvs1vfw52xx6f60rvvr39bvz6l3ldyk06zx47dnz1"))

(define rust-uu-mv-0.0.29
  (crate-source "uu_mv" "0.0.29"
                "0wb0fxzb7zdc7wl60hlfyh629yvwh8cl74kn06maslvimhbqrawj"))

(define rust-uu-touch-0.0.29
  (crate-source "uu_touch" "0.0.29"
                "0rsbh2mixpnh4ybz9g2zrn67p3qb5szz55gbzv92q9v1g567gp1s"))

(define rust-uu-uname-0.0.29
  (crate-source "uu_uname" "0.0.29"
                "1f5gzlh5msmgrip0fn1njg8wxkfxz400xcbjv7sd80zcd9i45acm"))

(define rust-uu-whoami-0.0.29
  (crate-source "uu_whoami" "0.0.29"
                "066hqqcq0r4cnamksqgcqcd2qqrcvx461dhkxlamqdy972lgf4m8"))

(define rust-uucore-0.0.29
  (crate-source "uucore" "0.0.29"
                "0pivc0vzsai7hdk7905csi5qgjppjblxk96f4j7s0nfrk0axrq2h"))

(define rust-uucore-procs-0.0.29
  (crate-source "uucore_procs" "0.0.29"
                "0pqwka50344s00f39dla2dnwaihxxcm4l26agi1w0mibmcrxxlr7"))

(define rust-uuhelp-parser-0.0.29
  (crate-source "uuhelp_parser" "0.0.29"
                "1g2cj9lhr1cmrfmdw73i70ps8shs180ahv0djiwxrddz3arwix0c"))

(define rust-uuid-1.16.0
  (crate-source "uuid" "1.16.0"
                "1a9dkv6jm4lz7ip9l9i1mcx7sh389xjsr03l6jgwqjpmkdvpm3s5"))

(define rust-uuid-1.17.0
  (crate-source "uuid" "1.17.0"
                "07ckq4fdiygy02gmislzfp727hx9zw6lskk9dbcds5ax3sfikx1w"))

(define rust-uuid-1.18.0
  (crate-source "uuid" "1.18.0"
                "1gn1vlggiwrdpizqcpc5hyxsqz9s5215bbay1b182mqn7rj9ccgk"))

(define rust-uuid-1.18.1
  (crate-source "uuid" "1.18.1"
                "18kh01qmfayn4psap52x8xdjkzw2q8bcbpnhhxjs05dr22mbi1rg"))

(define rust-uuid-simd-0.8.0
  (crate-source "uuid-simd" "0.8.0"
                "1n0b40m988h52xj03dkcp4plrzvz56r7xha1d681jrjg5ci85c13"))

(define rust-uutils-term-grid-0.7.0
  (crate-source "uutils_term_grid" "0.7.0"
                "0qb3ghkvvvs3i5i7svr8yhgjw1vicnbh4gxlh27d1fhiwlf19fpw"))

(define rust-uzers-0.11.3
  (crate-source "uzers" "0.11.3"
                "0qrzbhncbv4s52lgyzs2pxn1b6gmx9k7h1rdwdwix44cgvf87lkn"))

(define rust-uzers-0.12.1
  (crate-source "uzers" "0.12.1"
                "1pcpi9v90nr3q2y3i4pkac9c20r1nzaimvcm7vajmn770ksizy2d"))

(define rust-v-frame-0.3.8
  (crate-source "v_frame" "0.3.8"
                "0az9nd6qi1gyikh9yb3lhm453kf7d5isd6xai3j13kds4jm2mwyn"))

(define rust-v-frame-0.3.9
  (crate-source "v_frame" "0.3.9"
                "1qkvb4ks33zck931vzqckjn36hkngj6l2cwmvfsnlpc7r0kpfsv6"))

(define rust-v-htmlescape-0.15.8
  (crate-source "v_htmlescape" "0.15.8"
                "135inp4x7cc32k0hzrymlz1baf0rj0ah5h82nrpa9w0hqpxmg0jf"))

(define rust-validator-0.15.0
  (crate-source "validator" "0.15.0"
                "1mgfpwry40fcsfabx7vziy0a7j9kxn6b49mvbv1hy770j09hlyzh"))

(define rust-valuable-0.1.1
  (crate-source "valuable" "0.1.1"
                "0r9srp55v7g27s5bg7a2m095fzckrcdca5maih6dy9bay6fflwxs"))

(define rust-value-bag-1.11.1
  (crate-source "value-bag" "1.11.1"
                "1i89r6z2dxydybh43mxpc2qx3y943f35sm42c06v2gkliadf4g4l"))

(define rust-varisat-0.2.2
  (crate-source "varisat" "0.2.2"
                "1bvwh2bk80a5nci3sd3p205200c6pmbgd5f299krd48y3n2hkrpb"))

(define rust-varisat-checker-0.2.2
  (crate-source "varisat-checker" "0.2.2"
                "0fvndkgd2ypgr9rygnj0glxk0492696qw6xqysc6xv8kb5y9fp0k"))

(define rust-varisat-dimacs-0.2.2
  (crate-source "varisat-dimacs" "0.2.2"
                "0di68140imf7nfhhipkqllrp5m6f1iqsxxrrm70087xy457fw79x"))

(define rust-varisat-formula-0.2.2
  (crate-source "varisat-formula" "0.2.2"
                "08256rfjdmvfxjjw162r6l5ipfd46in9vx1sdmnhgndzp51map1r"))

(define rust-varisat-internal-macros-0.2.2
  (crate-source "varisat-internal-macros" "0.2.2"
                "13a0297kq3qhk6wa59sd44zjlhn0qs358ia8g2m6dl236mvwwbk0"))

(define rust-varisat-internal-proof-0.2.2
  (crate-source "varisat-internal-proof" "0.2.2"
                "01yj4zalzp6x6wa0yr3xl8v1q51xh1vgjr3dnxvz12h1r5xvnqv1"))

(define rust-vcpkg-0.2.15
  (crate-source "vcpkg" "0.2.15"
                "09i4nf5y8lig6xgj3f7fyrvzd3nlaw4znrihw8psidvv5yk4xkdc"
                #:snippet '(delete-file-recursively "test-data")))

(define rust-vec-map-0.8.2
  (crate-source "vec_map" "0.8.2"
                "1481w9g1dw9rxp3l6snkdqihzyrd2f8vispzqmwjwsdyhw8xzggi"))

(define rust-vec-mut-scan-0.3.0
  (crate-source "vec_mut_scan" "0.3.0"
                "1lkz66l8z13lvjll69s23vrca12inpyyh00kwg0djqsyil563vb8"))

(define rust-vergen-8.3.2
  (crate-source "vergen" "8.3.2"
                "0ri5n4k1g4z6gnllkjx9zny3vaa2bjma84zlrjh6w9k7b7mdk419"))

(define rust-vergen-9.0.6
  (crate-source "vergen" "9.0.6"
                "0xs7drxw7jlklnyrc8mifkpvag8nckja5ly60i0l3j8zw65zaavb"))

(define rust-vergen-git2-1.0.7
  (crate-source "vergen-git2" "1.0.7"
                "1hfam6z3c4wicg2n14rdzgkp3ipffq77b878mnmqw2a5xh8yavjg"))

(define rust-vergen-gitcl-1.0.8
  (crate-source "vergen-gitcl" "1.0.8"
                "1zmffxim0560czbfd5gaq3nqnqr9jlbinbqmvx6qmq5jdvgc3pxr"))

(define rust-vergen-lib-0.1.6
  (crate-source "vergen-lib" "0.1.6"
                "0rn1x40xwx4zlj62nkl63y6sczar6hw1dq34n7y5jghg1h0yc1wv"))

(define rust-version-check-0.1.5
  (crate-source "version_check" "0.1.5"
                "1pf91pvj8n6akh7w6j5ypka6aqz08b3qpzgs0ak2kjf4frkiljwi"))

(define rust-version-check-0.9.4
  (crate-source "version_check" "0.9.4"
                "0gs8grwdlgh0xq660d7wr80x14vxbizmd8dbp29p2pdncx8lp1s9"))

(define rust-version-check-0.9.5
  (crate-source "version_check" "0.9.5"
                "0nhhi4i5x89gm911azqbn7avs9mdacw2i3vcz3cnmz3mv4rqz4hb"))

(define rust-version-compare-0.0.10
  (crate-source "version-compare" "0.0.10"
                "18ack6rx18rp700h1dncljmpzchs3p2dfh76a8ds6vmfbfi5cdfn"))

(define rust-version-compare-0.1.1
  (crate-source "version-compare" "0.1.1"
                "0acg4pmjdbmclg0m7yhijn979mdy66z3k8qrcnvn634f1gy456jp"))

(define rust-version-compare-0.2.0
  (crate-source "version-compare" "0.2.0"
                "12y9262fhjm1wp0aj3mwhads7kv0jz8h168nn5fb8b43nwf9abl5"))

(define rust-version-ranges-0.1.1.b70cf70 rust-pubgrub-0.3.0.b70cf70)

(define rust-version-track-0.1.0
  (crate-source "version-track" "0.1.0"
                "0bz1b96iqh4z4r90xc0ya5gmywps072g5w6q1vrv2qsfdgnzz6kd"))

(define rust-versions-6.3.2
  (crate-source "versions" "6.3.2"
                "0ff12avdiqhiv6nanikkjl1x3s2y7amkj3r5nivb7zficf5ljpgj"))

(define rust-vlq-0.5.1
  (crate-source "vlq" "0.5.1"
                "1zygijgl47gasi0zx34ak1jq2n4qmk0cx2zpn13shba157npxpb5"))

(define rust-void-1.0.2
  (crate-source "void" "1.0.2"
                "0zc8f0ksxvmhvgx4fdg0zyn6vdnbxd2xv9hfx4nhzg6kbs4f80ka"))

(define rust-vsimd-0.8.0
  (crate-source "vsimd" "0.8.0"
                "0r4wn54jxb12r0x023r5yxcrqk785akmbddqkcafz9fm03584c2w"))

(define rust-vsort-0.2.0
  (crate-source "vsort" "0.2.0"
                "1y7b0x18a2mpn7hk3qfybl2ksy08klwl3bps4yxsd7pr90dilcqi"))

(define rust-vswhom-0.1.0
  (crate-source "vswhom" "0.1.0"
                "12v0fjjzxdc3y5c0lcwycfhphz7zf2s06hl5krwhawah0xzrp5xy"
                #:snippet '(delete-file "gh_rsa.enc")))

(define rust-vswhom-sys-0.1.3
  (crate-source "vswhom-sys" "0.1.3"
                "0l0i4fijapsybmfckfqh53yqxsg0bm5ikja6vz8ngw0zpm67w1pv"
                #:snippet '(delete-file "gh_rsa.enc")))

(define rust-vte-0.10.1
  (crate-source "vte" "0.10.1"
                "10srmy9ssircrwsb5lpx3fbhx71460j77kvz0krz38jcmf9fdg3c"
                #:snippet '(delete-file-recursively "tests")))

(define rust-vte-0.13.1
  (crate-source "vte" "0.13.1"
                "19fayaipg6s5hmw38m3an3m6n7vmj51v2i9py0ghfr7g40xnh2ws"
                #:snippet '(delete-file-recursively "tests")))

(define rust-vte-0.14.1
  (crate-source "vte" "0.14.1"
                "0xy01fgkzb2080prh2ncd8949hm2248fc5wf1lryhdrhxzbxq7r3"
                #:snippet '(delete-file-recursively "tests")))

(define rust-vte-0.15.0
  (crate-source "vte" "0.15.0"
                "1g9xgnw7q7zdwgfqa6zfcfsp92wn0j0h13kzsqy0dq3c80c414m5"
                #:snippet '(delete-file-recursively "tests")))

(define rust-vte-generate-state-changes-0.1.2
  (crate-source "vte_generate_state_changes" "0.1.2"
                "0biwgpcji3w4llz7h4bi8c2rwqchm9gmyr7dnjki1m853gp9ndif"))

(define rust-vtparse-0.6.2
  (crate-source "vtparse" "0.6.2"
                "1l5yz9650zhkaffxn28cvfys7plcw2wd6drajyf41pshn37jm6vd"))

(define rust-wait-timeout-0.2.1
  (crate-source "wait-timeout" "0.2.1"
                "04azqv9mnfxgvnc8j2wp362xraybakh2dy1nj22gj51rdl93pb09"))

(define rust-waker-fn-1.2.0
  (crate-source "waker-fn" "1.2.0"
                "1dvk0qsv88kiq22x8w0qz0k9nyrxxm5a9a9czdwdvvhcvjh12wii"))

(define rust-walkdir-2.3.3
  (crate-source "walkdir" "2.3.3"
                "16768hy32kcvghq7v4ci8llfjvdiwrwg6sj9nzcdiisnv9699prn"))

(define rust-walkdir-2.5.0
  (crate-source "walkdir" "2.5.0"
                "0jsy7a710qv8gld5957ybrnc07gavppp963gs32xk4ag8130jy99"
                #:snippet '(for-each delete-file-recursively '("compare" "src/tests"))))

(define rust-wallust-themes-1.1.0
  (crate-source "wallust_themes" "1.1.0"
                "1f27r2xd4gn5af5g8f2yws7ph7807cyg0d12ycmsvjhmk15pr6cf"))

(define rust-want-0.3.1
  (crate-source "want" "0.3.1"
                "03hbfrnvqqdchb5kgxyavb9jabwza0dmh2vw5kg0dq8rxl57d9xz"))

(define rust-wasi-0.11.0+wasi-snapshot-preview1
  (crate-source "wasi" "0.11.0+wasi-snapshot-preview1"
                "08z4hxwkpdpalxjps1ai9y7ihin26y9f476i53dv98v45gkqg3cw"))

(define rust-wasi-0.11.1+wasi-snapshot-preview1
  (crate-source "wasi" "0.11.1+wasi-snapshot-preview1"
                "0jx49r7nbkbhyfrfyhz0bm4817yrnxgd3jiwwwfv0zl439jyrwyc"))

(define rust-wasi-0.13.3+wasi-0.2.2
  (crate-source "wasi" "0.13.3+wasi-0.2.2"
                "1lnapbvdcvi3kc749wzqvwrpd483win2kicn1faa4dja38p6v096"))

(define rust-wasi-0.14.2+wasi-0.2.4
  (crate-source "wasi" "0.14.2+wasi-0.2.4"
                "1cwcqjr3dgdq8j325awgk8a715h0hg0f7jqzsb077n4qm6jzk0wn"))

(define rust-wasi-0.14.3+wasi-0.2.4
  (crate-source "wasi" "0.14.3+wasi-0.2.4"
                "158c0cy561zlncw71hjh1pficw60p1nj7ki8kqm2gpbv0f1swlba"))

(define rust-wasi-0.14.4+wasi-0.2.4
  (crate-source "wasi" "0.14.4+wasi-0.2.4"
                "0jpcy0qg3bvsclwyk1d11r3isd320rpickrl5hy9rx7s4jjg99c8"))

(define rust-wasi-0.14.7+wasi-0.2.4
  (crate-source "wasi" "0.14.7+wasi-0.2.4"
                "133fq3mq7h65mzrsphcm7bbbx1gsz7srrbwh01624zin43g7hd48"))

(define rust-wasi-0.9.0+wasi-snapshot-preview1
  (crate-source "wasi" "0.9.0+wasi-snapshot-preview1"
                "06g5v3vrdapfzvfq662cij7v8a1flwr2my45nnncdv2galrdzkfc"))

(define rust-wasip2-1.0.1+wasi-0.2.4
  (crate-source "wasip2" "1.0.1+wasi-0.2.4"
                "1rsqmpspwy0zja82xx7kbkbg9fv34a4a2if3sbd76dy64a244qh5"))

(define rust-wasite-0.1.0
  (crate-source "wasite" "0.1.0"
                "0nw5h9nmcl4fyf4j5d4mfdjfgvwi1cakpi349wc4zrr59wxxinmq"))

(define rust-wasm-bindgen-0.2.100
  (crate-source "wasm-bindgen" "0.2.100"
                "1x8ymcm6yi3i1rwj78myl1agqv2m86i648myy3lc97s9swlqkp0y"))

(define rust-wasm-bindgen-0.2.101
  (crate-source "wasm-bindgen" "0.2.101"
                "0fv0yrfx170gf7i4dds4c69dxh8axp247wyip2dm4nylmmf9253y"))

(define rust-wasm-bindgen-0.2.103
  (crate-source "wasm-bindgen" "0.2.103"
                "069qhf7yrl4jymzjzvwsmcmw96al639xim4scigpy5qapngsc45b"))

(define rust-wasm-bindgen-0.2.104
  (crate-source "wasm-bindgen" "0.2.104"
                "0b8f4l6pqm0bz0lj5xgwmchb6977n71vmh7srd0axwg93b011nn1"))

(define rust-wasm-bindgen-0.2.84
  (crate-source "wasm-bindgen" "0.2.84"
                "0fx5gh0b4n6znfa3blz92wn1k4bbiysyq9m95s7rn3gk46ydry1i"
                #:snippet '(delete-file-recursively "guide")))

(define rust-wasm-bindgen-0.2.93
  (crate-source "wasm-bindgen" "0.2.93"
                "1dfr7pka5kwvky2fx82m9d060p842hc5fyyw8igryikcdb0xybm8"))

(define rust-wasm-bindgen-backend-0.2.100
  (crate-source "wasm-bindgen-backend" "0.2.100"
                "1ihbf1hq3y81c4md9lyh6lcwbx6a5j0fw4fygd423g62lm8hc2ig"))

(define rust-wasm-bindgen-backend-0.2.101
  (crate-source "wasm-bindgen-backend" "0.2.101"
                "1fwkzc2z701g2rm2jq4m20a0lkc6qqq5r3a407yj6yfahalip3g2"))

(define rust-wasm-bindgen-backend-0.2.103
  (crate-source "wasm-bindgen-backend" "0.2.103"
                "070x7fjnnvzm2y3a5j29wmss4z547cjdx3rnpixh19j56m105dqb"))

(define rust-wasm-bindgen-backend-0.2.104
  (crate-source "wasm-bindgen-backend" "0.2.104"
                "069vnhhn2j4w2gwd8rch6g8d3iwkrgi45fas6i3qm7glcrd9l737"))

(define rust-wasm-bindgen-backend-0.2.84
  (crate-source "wasm-bindgen-backend" "0.2.84"
                "1ffc0wb293ha56i66f830x7f8aa2xql69a21lrasy1ncbgyr1klm"))

(define rust-wasm-bindgen-backend-0.2.93
  (crate-source "wasm-bindgen-backend" "0.2.93"
                "0yypblaf94rdgqs5xw97499xfwgs1096yx026d6h88v563d9dqwx"))

(define rust-wasm-bindgen-futures-0.4.34
  (crate-source "wasm-bindgen-futures" "0.4.34"
                "0m0lnnnhs9ni4dn9vz74prsjz8bdcf8dvnznd5ljch5s279f06gj"))

(define rust-wasm-bindgen-futures-0.4.50
  (crate-source "wasm-bindgen-futures" "0.4.50"
                "0q8ymi6i9r3vxly551dhxcyai7nc491mspj0j1wbafxwq074fpam"))

(define rust-wasm-bindgen-futures-0.4.51
  (crate-source "wasm-bindgen-futures" "0.4.51"
                "1znz8i8kyrlpq6q2fals223zrwwixmn6s7a16s1v6sdlm4wm1a0c"))

(define rust-wasm-bindgen-futures-0.4.53
  (crate-source "wasm-bindgen-futures" "0.4.53"
                "0rrglk9fjwk69adwca1skn10v9qlhacmbfxd16aq6mhj8bzj3cm0"))

(define rust-wasm-bindgen-futures-0.4.54
  (crate-source "wasm-bindgen-futures" "0.4.54"
                "0p5c10vfd7p7c607x3cgyfw9h77z1k33d6zzw2x77k3qwi0qs0vy"))

(define rust-wasm-bindgen-macro-0.2.100
  (crate-source "wasm-bindgen-macro" "0.2.100"
                "01xls2dvzh38yj17jgrbiib1d3nyad7k2yw9s0mpklwys333zrkz"))

(define rust-wasm-bindgen-macro-0.2.101
  (crate-source "wasm-bindgen-macro" "0.2.101"
                "038vxk2yg11c3qv9iyasqcm70dw8sr2xmyaxqjq7bxzgwcx4cgbw"))

(define rust-wasm-bindgen-macro-0.2.103
  (crate-source "wasm-bindgen-macro" "0.2.103"
                "18481jkmczv4j0m747ypb0k1acq093hhbdhpb4sr856r27sg8rgw"))

(define rust-wasm-bindgen-macro-0.2.104
  (crate-source "wasm-bindgen-macro" "0.2.104"
                "06d1m5bg272h6jabq0snm7c50fifjz6r20f5hqlmz7y5wivh99kw"))

(define rust-wasm-bindgen-macro-0.2.84
  (crate-source "wasm-bindgen-macro" "0.2.84"
                "1idlq28awqhq8rclb22rn5xix82w9a4rgy11vkapzhzd1dygf8ac"))

(define rust-wasm-bindgen-macro-0.2.93
  (crate-source "wasm-bindgen-macro" "0.2.93"
                "1kycd1xfx4d9xzqknvzbiqhwb5fzvjqrrn88x692q1vblj8lqp2q"))

(define rust-wasm-bindgen-macro-support-0.2.100
  (crate-source "wasm-bindgen-macro-support" "0.2.100"
                "1plm8dh20jg2id0320pbmrlsv6cazfv6b6907z19ys4z1jj7xs4a"))

(define rust-wasm-bindgen-macro-support-0.2.101
  (crate-source "wasm-bindgen-macro-support" "0.2.101"
                "1ajjqmdbi7ybdpw41avskjfdqnxpc9v547gmr8izj4c2n24wxd3v"))

(define rust-wasm-bindgen-macro-support-0.2.103
  (crate-source "wasm-bindgen-macro-support" "0.2.103"
                "0clsx611pday95s6wg8pndvrd8xknsaf20d40kk8x2irj6lh7h7z"))

(define rust-wasm-bindgen-macro-support-0.2.104
  (crate-source "wasm-bindgen-macro-support" "0.2.104"
                "1mr18kx7ima1pmsqlkk982q4a0vf3r8s1x6901jb59sd1prd41wz"))

(define rust-wasm-bindgen-macro-support-0.2.84
  (crate-source "wasm-bindgen-macro-support" "0.2.84"
                "1xm56lpi0rihh8ny7x085dgs3jdm47spgqflb98wghyadwq83zra"))

(define rust-wasm-bindgen-macro-support-0.2.93
  (crate-source "wasm-bindgen-macro-support" "0.2.93"
                "0dp8w6jmw44srym6l752nkr3hkplyw38a2fxz5f3j1ch9p3l1hxg"))

(define rust-wasm-bindgen-shared-0.2.100
  (crate-source "wasm-bindgen-shared" "0.2.100"
                "0gffxvqgbh9r9xl36gprkfnh3w9gl8wgia6xrin7v11sjcxxf18s"))

(define rust-wasm-bindgen-shared-0.2.101
  (crate-source "wasm-bindgen-shared" "0.2.101"
                "1h94nvm5p8zyr3718x4zhdz7rcmd0rir0b46a1ljqx8k7d58ahzi"))

(define rust-wasm-bindgen-shared-0.2.103
  (crate-source "wasm-bindgen-shared" "0.2.103"
                "1kx13fvmlxxaxf04vm3b14437hyq92zdy89pvcaclc54xzs3fg19"))

(define rust-wasm-bindgen-shared-0.2.104
  (crate-source "wasm-bindgen-shared" "0.2.104"
                "1la1xj9v3gmawnlyi7lc3mb3xi447r6frb98hi2fb9m1nb47vmms"))

(define rust-wasm-bindgen-shared-0.2.84
  (crate-source "wasm-bindgen-shared" "0.2.84"
                "0pcvk1c97r1pprzfaxxn359r0wqg5bm33ylbwgjh8f4cwbvzwih0"))

(define rust-wasm-bindgen-shared-0.2.93
  (crate-source "wasm-bindgen-shared" "0.2.93"
                "1104bny0hv40jfap3hp8jhs0q4ya244qcrvql39i38xlghq0lan6"))

(define rust-wasm-bindgen-test-0.3.50
  (crate-source "wasm-bindgen-test" "0.3.50"
                "1hsjc60wynlhgw02p32pgb93303pqmsdfxj67gxdkdm37kixbj36"))

(define rust-wasm-bindgen-test-macro-0.3.50
  (crate-source "wasm-bindgen-test-macro" "0.3.50"
                "16znd6wz79v2i3b2sf5n4ld2kcci8br3wcx7z5c9c07sqln09m8p"))

(define rust-wasm-encoder-0.221.3
  (crate-source "wasm-encoder" "0.221.3"
                "1r9jv84kbjf6z18rgf3666vvcn5fybzn9daszsj81pi097z4916w"))

(define rust-wasm-encoder-0.229.0
  (crate-source "wasm-encoder" "0.229.0"
                "1lk5w8602dn56sdzrc83ppfgv9kmlq85q0jj4md0ijya3r4ivfiq"))

(define rust-wasm-streams-0.4.2
  (crate-source "wasm-streams" "0.4.2"
                "0rddn007hp6k2cm91mm9y33n79b0jxv0c3znzszcvv67hn6ks18m"))

(define rust-wasm-timer-0.2.5
  (crate-source "wasm-timer" "0.2.5"
                "0zsyijv3wgj9p4q47a4awla8j4kw33jd7da2fsd1wml0nh6wn3my"))

(define rust-wasmer-2.3.0
  (crate-source "wasmer" "2.3.0"
                "1pxyaynmh5kcvb7k0952j47zsbs9n7b7pplpfwyyl1nhr5hq73ga"))

(define rust-wasmer-artifact-2.3.0
  (crate-source "wasmer-artifact" "2.3.0"
                "09g3ghphyl2y9sb5il1zwcsshs55m2dqhi8fmk98l7cwq8l99bvs"))

(define rust-wasmer-cache-2.3.0
  (crate-source "wasmer-cache" "2.3.0"
                "1931jp26a8ywiiq49zfhda9wnp3sq0cyjv0ymk2yl7b3w4g3kvqd"))

(define rust-wasmer-compiler-2.3.0
  (crate-source "wasmer-compiler" "2.3.0"
                "0b4xw4g1xic55d47hrb77z9vpalbq6bfmkxjdmjmdm5fcvc6qyp6"))

(define rust-wasmer-compiler-cranelift-2.3.0
  (crate-source "wasmer-compiler-cranelift" "2.3.0"
                "1q6s4ffsj14mgx2s2jk5bapry4dyrfi4dfgqwi4qdw4mcjgjzgj8"))

(define rust-wasmer-derive-2.3.0
  (crate-source "wasmer-derive" "2.3.0"
                "0lab50wlf236135f5k2afj9mq79cbw573125azzp8brarh2h9r80"))

(define rust-wasmer-engine-2.3.0
  (crate-source "wasmer-engine" "2.3.0"
                "0icfaiihg4iaix7k6hr3hah7xpv10smsx4mk66s4s94cjw8g161z"))

(define rust-wasmer-engine-dylib-2.3.0
  (crate-source "wasmer-engine-dylib" "2.3.0"
                "0lvy5dkpiqv5s607hrgnp2ndjj2nh19ma5rifxc28iqmkjpmh0xd"))

(define rust-wasmer-engine-universal-2.3.0
  (crate-source "wasmer-engine-universal" "2.3.0"
                "05ggvcr4p11df4yrpdcnawxrigw1x8vx1vgllijpi94w7kcw63a4"))

(define rust-wasmer-engine-universal-artifact-2.3.0
  (crate-source "wasmer-engine-universal-artifact" "2.3.0"
                "0hddz5md13d4gmkzwa43mlfq1xsza9k4pi46dvmmf9hmahzxpwb8"))

(define rust-wasmer-object-2.3.0
  (crate-source "wasmer-object" "2.3.0"
                "12zsm0iqdb14xgn2rf9nh1445ikm264nygrha7sfqi1szwsi70wd"))

(define rust-wasmer-types-2.3.0
  (crate-source "wasmer-types" "2.3.0"
                "0zv6nb84wmna5v80pfbvbarm67jj075wfm70cymrn2nw0pm03prr"))

(define rust-wasmer-vm-2.3.0
  (crate-source "wasmer-vm" "2.3.0"
                "1pa7c937ky1kak1scm21p7xn6dgcxkvslkd56pdlrp7lc7x6bn9h"))

(define rust-wasmi-0.40.0
  (crate-source "wasmi" "0.40.0"
                "0qq069di8lpb2vqckifv4cbyifxx98mkxlmlsv8ms14nrdzzk6m1"))

(define rust-wasmi-collections-0.40.0
  (crate-source "wasmi_collections" "0.40.0"
                "0qimh7lb0pdicdlirdp2794i6rippxs1amlxjchj14hwbcknn3g8"))

(define rust-wasmi-core-0.40.0
  (crate-source "wasmi_core" "0.40.0"
                "0cvw16b7l3his0p6m8bv0nywawyxpv91q8gzqz132bf35i45331s"))

(define rust-wasmi-ir-0.40.0
  (crate-source "wasmi_ir" "0.40.0"
                "16i8nmkjhlj2ywfqvq51gbw53vb8pn46flc858hmknw6q4a1lhvf"))

(define rust-wasmparser-0.207.0
  (crate-source "wasmparser" "0.207.0"
                "0b694q3frf4xvavj0rw7xk3j852gqljdp2pghajnsq87mgwbk6z1"))

(define rust-wasmparser-0.221.3
  (crate-source "wasmparser" "0.221.3"
                "11ficyz79dcypkxxg1c8vl8bm0avg8a80csnxq6vxhismcvglsyh"))

(define rust-wasmparser-0.224.1
  (crate-source "wasmparser" "0.224.1"
                "04cf332ajrccf1nfkw61p7bjkfnnw3x63il4ks0x7pf22xcpmw84"))

(define rust-wasmparser-0.229.0
  (crate-source "wasmparser" "0.229.0"
                "132sazrlzqi0ilqcw66i3129x2hvdndzm8a0asjimm7magqb3hqc"))

(define rust-wasmparser-0.83.0
  (crate-source "wasmparser" "0.83.0"
                "0yhx2kq7da4sdglh1x1di4xxg33k7lwddpd3ri46bp9abk2xg3ki"))

(define rust-wasmprinter-0.221.3
  (crate-source "wasmprinter" "0.221.3"
                "10xjs2bzvppwr4qdsgfqqmafjah9290bd0gz35w6r4pjjwmc8hvk"))

(define rust-wasmtime-29.0.1
  (crate-source "wasmtime" "29.0.1"
                "0s8f05h19hnnhx2jmi7dr4m1kjbacpbxbih49hf6smbj0qjnm5qi"))

(define rust-wasmtime-asm-macros-29.0.1
  (crate-source "wasmtime-asm-macros" "29.0.1"
                "1wk4rwn8085nllba4nz4z68ihl9ypm4zisjvyzlzn0aj286qn5qz"))

(define rust-wasmtime-c-api-impl-29.0.1
  (crate-source "wasmtime-c-api-impl" "29.0.1"
                "037dcqkjwl8vxmnfh6pqlzcpcdjgpb0r9fvvgiwyabcgc3rwwc7a"))

(define rust-wasmtime-c-api-macros-29.0.1
  (crate-source "wasmtime-c-api-macros" "29.0.1"
                "0nxndnwfwx490yw5rwc1gn2xvd66c5z3sil29lw5sk8jw7mpjah2"))

(define rust-wasmtime-component-macro-29.0.1
  (crate-source "wasmtime-component-macro" "29.0.1"
                "1byzawnz8myzmlwbc0g5y415sk9h5ac464pp0akd0ifr5rcyckfp"))

(define rust-wasmtime-component-util-29.0.1
  (crate-source "wasmtime-component-util" "29.0.1"
                "0vh5dzz2nn7clnbdy5igd07nm3igbgxy5krhdcv5maqjq6rwfzbh"))

(define rust-wasmtime-cranelift-29.0.1
  (crate-source "wasmtime-cranelift" "29.0.1"
                "11wyks7pdzshspw6ks5k5z45r28ngm6vrg0g559z2jsdcwiffsrn"))

(define rust-wasmtime-environ-29.0.1
  (crate-source "wasmtime-environ" "29.0.1"
                "1bclk4g44v8bcqf9fik9njsjdxqhk290iw54fsi7ld4pf2pw3bfd"))

(define rust-wasmtime-fiber-29.0.1
  (crate-source "wasmtime-fiber" "29.0.1"
                "05qis8ry2z3mqkw2hlpz3yz1g9ih0xjqa10rj751n24z27a91fnc"))

(define rust-wasmtime-jit-icache-coherence-29.0.1
  (crate-source "wasmtime-jit-icache-coherence" "29.0.1"
                "1la7aw71kavh11d9k9nig7nbv2pd9xq72c99wp1fd4hnw198appc"))

(define rust-wasmtime-math-29.0.1
  (crate-source "wasmtime-math" "29.0.1"
                "05yy3npnfk6h199bvibjm00yqf8g12pysp30ai6hzq15mb10w899"))

(define rust-wasmtime-slab-29.0.1
  (crate-source "wasmtime-slab" "29.0.1"
                "1kswbqx53kcnsgd78nnvafh2jxydsnrqn5bvphaaq17sjqd85dgw"))

(define rust-wasmtime-versioned-export-macros-29.0.1
  (crate-source "wasmtime-versioned-export-macros" "29.0.1"
                "0nvvm5xh6yclnf7511cnvzfi79l710lwhh6yc8229h3d47dqdzw6"))

(define rust-wasmtime-winch-29.0.1
  (crate-source "wasmtime-winch" "29.0.1"
                "13yg8dx7nf37fp6syyj8i2cyaak8sgmbk4hhv3hxa0h5yawbzfpx"))

(define rust-wasmtime-wit-bindgen-29.0.1
  (crate-source "wasmtime-wit-bindgen" "29.0.1"
                "1rh6ds0kcslrwks8k7rz6hand5c4myjx7if1wdwxpr6i5nf32n43"))

(define rust-wast-229.0.0
  (crate-source "wast" "229.0.0"
                "1nghmmz2z2xap3s0zzqf63ll1ywglffgfqxiddlja8n12gvazz33"
                #:snippet '(delete-file-recursively "tests")))

(define rust-wat-1.229.0
  (crate-source "wat" "1.229.0"
                "1q1k9inxwpbsfi3103x88g0zm4fg5mni5p37ksg5libhig8bm2a1"))

(define rust-watchexec-2.3.2
  (crate-source "watchexec" "2.3.2"
                "120g73qpywbrzlcma9nhgaymm44aw2516xwl0fima8yy2ig22car"))

(define rust-watchexec-6.0.0
  (crate-source "watchexec" "6.0.0"
                "0hzqxi0a5ifc4j3xlfcy9kfb4078184x9p9y8a14r6gid8w8lnr8"))

(define rust-watchexec-8.0.1
  (crate-source "watchexec" "8.0.1"
                "0c9klljnpmynhiffdphyq67y8pr23adkwf8jlnn6140k4557jddw"))

(define rust-watchexec-events-1.0.0
  (crate-source "watchexec-events" "1.0.0"
                "1qra2hpkdf5ln1vb5jpxp17s3va7dm2svb8d067r2xgx0az3nq01"))

(define rust-watchexec-events-5.0.0
  (crate-source "watchexec-events" "5.0.0"
                "0lazzqwk9dhlv0i4q14fc286n0nzacwqm01im6515q01p2dx9yvw"))

(define rust-watchexec-events-6.0.0
  (crate-source "watchexec-events" "6.0.0"
                "1lb0p1spn0hzkpn96xk8ifxajs1i2qcja9qji8ck0z8cl9rqjjlw"))

(define rust-watchexec-filterer-globset-1.2.0
  (crate-source "watchexec-filterer-globset" "1.2.0"
                "045j0cp83yl6csn04qi19d5jfgfzylgcnvgziksqjcdgrv6b1c54"))

(define rust-watchexec-filterer-globset-7.0.0
  (crate-source "watchexec-filterer-globset" "7.0.0"
                "063lvz0r4mwrhqv6dygh03rfxj2h28vvdzw03jx688c77r4qrbgi"))

(define rust-watchexec-filterer-globset-8.0.0
  (crate-source "watchexec-filterer-globset" "8.0.0"
                "103hhrdrmcdp5jicdjviap2kid0w4rz0xah0qba9v6sip1yjjn7j"))

(define rust-watchexec-filterer-ignore-1.2.1
  (crate-source "watchexec-filterer-ignore" "1.2.1"
                "18sfshgvi8d68xc9vc9pz7dar6cfbgzclnfdkyhvgkbw6qhg0igk"))

(define rust-watchexec-filterer-ignore-6.0.0
  (crate-source "watchexec-filterer-ignore" "6.0.0"
                "0rf76g6iw2yra8dwm0x92mlpzfp4bsspnkhyclhf05d72zn2818a"))

(define rust-watchexec-filterer-ignore-7.0.0
  (crate-source "watchexec-filterer-ignore" "7.0.0"
                "193xgpp4i7vi2790301k4nzv36g99hpyyhdgkk1kx2a0zm2h6fwj"))

(define rust-watchexec-signals-1.0.0
  (crate-source "watchexec-signals" "1.0.0"
                "1rd7sqfvd96lbf05w74p7v59ch8xspy5ah509k4h329qdkwmsanc"))

(define rust-watchexec-signals-4.0.1
  (crate-source "watchexec-signals" "4.0.1"
                "06l6ba21gsks0d8gvzkavk3m2f3kxbxdmjscbsl8xq8wiz8dsd48"))

(define rust-watchexec-signals-5.0.0
  (crate-source "watchexec-signals" "5.0.0"
                "10zxblq24yf7q40i712r5cbxpbmphh53d3x2wqklr5k2j9kjjxrp"))

(define rust-watchexec-supervisor-4.0.0
  (crate-source "watchexec-supervisor" "4.0.0"
                "0zbmk998hb01zac9z931glc151slnrkdw6f7vl1pn97npzk7yvb7"))

(define rust-watchexec-supervisor-5.0.1
  (crate-source "watchexec-supervisor" "5.0.1"
                "0v245z7f5s788x4fphkbiywpnwy830b65bbhs3rra9vbx985r94j"))

(define rust-watchman-client-0.9.0
  (crate-source "watchman_client" "0.9.0"
                "0lpvdkvf500cr6454c722m9q15y3zxxq19sg1phsm9s3njdlrg48"))

(define rust-wax-0.6.0
  (crate-source "wax" "0.6.0"
                "0mqk9qxsjlbwnjnj0gkaa29qm3mmgbgrc6pd4qpjvcmsl25af4ld"))

(define rust-waybackend-0.4.3
  (crate-source "waybackend" "0.4.3"
                "1w3kdb303bsjc73gbahw3wygkaha2gi8g42g0mhlga83dqcidam2"))

(define rust-waybackend-scanner-0.4.3
  (crate-source "waybackend-scanner" "0.4.3"
                "1slslxif6jhxyj5wkjg135p2zpaila9yqxs9w3zpyjqd4qg1kd6n"))

(define rust-wayland-backend-0.1.2
  (crate-source "wayland-backend" "0.1.2"
                "1n1yi6vna23wfkrpk1j46sx5qbsijh50viha4sra73by8lkqxd21"))

(define rust-wayland-backend-0.3.10
  (crate-source "wayland-backend" "0.3.10"
                "08a1l8ya2siwvr9jhdz43nmmm7brnw848zra7sfwfpry8a0h2xzy"))

(define rust-wayland-backend-0.3.11
  (crate-source "wayland-backend" "0.3.11"
                "0dcvwkhz45gsm7f9dwr31pxijkhpza09a4vb3blsv9a8631k6fk7"))

(define rust-wayland-backend-0.3.8
  (crate-source "wayland-backend" "0.3.8"
                "1gs7dw6s3lp9g6g0rhk4bh66wl41jnbkd27c6ynhv1x3xac8j85p"))

(define rust-wayland-client-0.29.5
  (crate-source "wayland-client" "0.29.5"
                "05b7qikqj22rjy17kqw5ar7j2chpy18dr0gqapvwjfd00n60cfrz"))

(define rust-wayland-client-0.30.2
  (crate-source "wayland-client" "0.30.2"
                "1j3as2g1znrs2lpkksqcvx8pag85yiwwbcv6wb3lyrqgfxa9d728"))

(define rust-wayland-client-0.31.10
  (crate-source "wayland-client" "0.31.10"
                "0qcsgnfvgyxpvda70ww2836p8j8pd4jwll7km7bdniq8gg3ag3wp"))

(define rust-wayland-client-0.31.11
  (crate-source "wayland-client" "0.31.11"
                "17a4vl5qw4jnnh2azm0d3kcpajyb9qz4psv448zpj86w83l4fsn6"))

(define rust-wayland-client-0.31.8
  (crate-source "wayland-client" "0.31.8"
                "0gzpr9gdd8yk1crflxngg5iwa1szyyzp4i4zbgpslf1nsgihs4n2"))

(define rust-wayland-commons-0.29.5
  (crate-source "wayland-commons" "0.29.5"
                "00m90bnxqy0d6lzqlyazc1jh18jgbjwigmyr0rk3m8w4slsg34c6"))

(define rust-wayland-csd-frame-0.3.0
  (crate-source "wayland-csd-frame" "0.3.0"
                "0zjcmcqprfzx57hlm741n89ssp4sha5yh5cnmbk2agflvclm0p32"))

(define rust-wayland-cursor-0.29.5
  (crate-source "wayland-cursor" "0.29.5"
                "0qbn6wqmjibkx3lb3ggbp07iabzgx2zhrm0wxxxjbmhkdyvccrb8"))

(define rust-wayland-cursor-0.31.10
  (crate-source "wayland-cursor" "0.31.10"
                "10piw152cm3c290ksdckhd7yy2h76icp05nb2q2d0a7cilaiflx6"))

(define rust-wayland-cursor-0.31.11
  (crate-source "wayland-cursor" "0.31.11"
                "0agb7ryc8p7bnylh00jvp44sl3yn4rbzg2crksqp24l8192cqz24"))

(define rust-wayland-cursor-0.31.8
  (crate-source "wayland-cursor" "0.31.8"
                "0pccjqiln1izjbdzprqiijhad4k00wmr5r003a44h1v5nv5jjc59"))

(define rust-wayland-egl-0.32.7
  (crate-source "wayland-egl" "0.32.7"
                "15a38p34m1b98sm9zfinnwijh3d35rqs0n2f8gxmwyh5xz1nribx"))

(define rust-wayland-egl-0.32.8
  (crate-source "wayland-egl" "0.32.8"
                "1zdr05lydq648xqrrkyxdqywmlwimyhqrmimd15a6gms4gp34qnk"))

(define rust-wayland-protocols-0.29.5
  ;; TODO: Use our packaged protocols.
  (crate-source "wayland-protocols" "0.29.5"
                "1ihbjyd0w460gd7w22g9qabbwd4v8x74f8vsh7p25csljcgn4l5r"))

(define rust-wayland-protocols-0.30.1
  (crate-source "wayland-protocols" "0.30.1"
                "0kcvvli38gdjb9c7dpa2s0ix4nnqfq7n2bbc39370kx9bhg10a1v"))

(define rust-wayland-protocols-0.31.2
  (crate-source "wayland-protocols" "0.31.2"
                "1x310l1p6p3p3l76nl1l2yava9408dy77s605917zadlp1jz70cg"))

(define rust-wayland-protocols-0.32.6
  (crate-source "wayland-protocols" "0.32.6"
                "1z0yahh48x8qzdbcallmxn5am5897hkk5d7p51ly6dwvhr3cz087"))

(define rust-wayland-protocols-0.32.8
  (crate-source "wayland-protocols" "0.32.8"
                "0aj2209i962k1www23wb7zdgx81y6a35ilgyjhbm56hy9r2pb43p"))

(define rust-wayland-protocols-0.32.9
  (crate-source "wayland-protocols" "0.32.9"
                "00cripl4m7hzhl0gzp4bqayal8n0zlf1llnj7cl73zgvfpnr19zg"))

(define rust-wayland-protocols-misc-0.3.8
  (crate-source "wayland-protocols-misc" "0.3.8"
                "0hgwskrih84cpzj09vcs9yq3nijxcppyx8i5bd29k1fqifbg4p33"))

(define rust-wayland-protocols-misc-0.3.9
  (crate-source "wayland-protocols-misc" "0.3.9"
                "176i0xirlaaif6arj6l4ay8r2qxr9fsahrzh7z8gz2zba7ak7zid"))

(define rust-wayland-protocols-plasma-0.3.6
  (crate-source "wayland-protocols-plasma" "0.3.6"
                "1cs6gpgxybjvr60j6j824blsvz4hnmjwaah2cdkzvzh3cz3srjkw"))

(define rust-wayland-protocols-plasma-0.3.8
  (crate-source "wayland-protocols-plasma" "0.3.8"
                "0xg1xa1hq32pz7qzimgr9bpy5idcypxw3g662gjaqmlvsvd8rlsg"))

(define rust-wayland-protocols-plasma-0.3.9
  (crate-source "wayland-protocols-plasma" "0.3.9"
                "0cn0qbr99snsh41p34qv11bvylbqk2xzii47k4kv6yh7ghji8ym0"))

(define rust-wayland-protocols-wlr-0.1.0
  (crate-source "wayland-protocols-wlr" "0.1.0"
                "12jqi7n77l8a13hc5w5fkdgs4kdjk9i6nvl74njsdr106c4r3sgw"))

(define rust-wayland-protocols-wlr-0.2.0
  (crate-source "wayland-protocols-wlr" "0.2.0"
                "1mjww9psk2nc5hm2q4s3qas30rbzfg1sb6qgw518fbbcdfvn27xd"))

(define rust-wayland-protocols-wlr-0.3.6
  (crate-source "wayland-protocols-wlr" "0.3.6"
                "1cpqb0d4ryf87x2wgca5n71wilhvc0jjva0zasbdgalmypk052i4"))

(define rust-wayland-protocols-wlr-0.3.8
  (crate-source "wayland-protocols-wlr" "0.3.8"
                "1gwbd9nv71ahaqylfm2lvml5bwl6z0ygwdy40ijy1h4r6g3wvdhw"))

(define rust-wayland-protocols-wlr-0.3.9
  (crate-source "wayland-protocols-wlr" "0.3.9"
                "1v3qbg18vsb3i62c6042xhjm7dcflmylzjlhl0w9kks3xmilkngg"))

(define rust-wayland-scanner-0.29.5
  (crate-source "wayland-scanner" "0.29.5"
                "0lxx3i2kxnmsk421qx87lqqc9kd2y1ksjxcyg0pqbar2zbc06hwg"))

(define rust-wayland-scanner-0.30.1
  (crate-source "wayland-scanner" "0.30.1"
                "03ikmfwacsgbym2y4raf05knl1qjlgg81sy0174jxhzvayr77f5r"))

(define rust-wayland-scanner-0.31.6
  (crate-source "wayland-scanner" "0.31.6"
                "110ldnyfxjqvjssir1jf3ndlci7xy9lpv4aqg775y518bpyxlvw9"))

(define rust-wayland-scanner-0.31.7
  (crate-source "wayland-scanner" "0.31.7"
                "1qqalp551blcxjzx80zvs7ckc19k966892zxpm81kacxqjfixjsl"))

(define rust-wayland-server-0.31.10
  (crate-source "wayland-server" "0.31.10"
                "08k23abw3vav8f71x6s5m70mc0zgq22a9ljs8iqbm7vcp8x4zggw"))

(define rust-wayland-server-0.31.9
  (crate-source "wayland-server" "0.31.9"
                "11apclvwsp43l24gvdlrg455b7pr2nnfcd2xc8s0vahdry6gnpa8"))

(define rust-wayland-sys-0.29.5
  (crate-source "wayland-sys" "0.29.5"
                "1m79qqmr1hx7jlyrvnrxjma5s6dk5js9fjsr4nx7vv1r7hdcw4my"))

(define rust-wayland-sys-0.30.1
  (crate-source "wayland-sys" "0.30.1"
                "01man4ll2kyxp9x2934rhnf98522pzwsd2c6jwr73q08qqma1cln"))

(define rust-wayland-sys-0.31.6
  (crate-source "wayland-sys" "0.31.6"
                "05b6i4lg2qrrz7l4h2b5fd7blkkvxq34i1yvlngsmmbpkhwvpknv"))

(define rust-wayland-sys-0.31.7
  (crate-source "wayland-sys" "0.31.7"
                "0hk157yawv9y7aj7fxbldhlvv8p33c65v3nv85mq4m91h919p51l"))

(define rust-wayrs-client-1.3.0
  (crate-source "wayrs-client" "1.3.0"
                "017pw6p6wzqr41kc4z18npkirwjb9f8vavhsmjnn5sziy9kkky05"))

(define rust-wayrs-core-1.0.4
  (crate-source "wayrs-core" "1.0.4"
                "1jf5gf60rmwky1q8g09adkl6gm8wssixnhm8vc2p1b2ksh6y78mn"))

(define rust-wayrs-proto-parser-3.0.1
  (crate-source "wayrs-proto-parser" "3.0.1"
                "0g74n84xhvjjbhz158lb5fqx9lyynh5yawxvkf6mzlca7l1rqrs1"))

(define rust-wayrs-protocols-0.14.9+1.43
  (crate-source "wayrs-protocols" "0.14.9+1.43"
                "09aj8nzmvickvvqxq5ky5qfpbz71wy91fb82h7k9cybzgi24bn8j"))

(define rust-wayrs-scanner-0.15.4
  (crate-source "wayrs-scanner" "0.15.4"
                "0a4d6szwgj6shm674n550b6nlf7va6z46i7idb5cpmx8c32zgchn"))

(define rust-web-atoms-0.1.3
  (crate-source "web_atoms" "0.1.3"
                "056lg00xm67d2yiyi1fh3x15jpi3idk0acifk7wvsh0jq0fxxzsp"))

(define rust-web-sys-0.3.61
  (crate-source "web-sys" "0.3.61"
                "15qbbdbhyz02srvm01p0cvlh0pvmbbin9hislp0yx8rvnbs9jfz3"))

(define rust-web-sys-0.3.77
  (crate-source "web-sys" "0.3.77"
                "1lnmc1ffbq34qw91nndklqqm75rasaffj2g4f8h1yvqqz4pdvdik"))

(define rust-web-sys-0.3.78
  (crate-source "web-sys" "0.3.78"
                "04lbcdr74pilsrf1g76lbw9bwg7zghgslqxdiwmxkw4zfhvvdr3p"))

(define rust-web-sys-0.3.80
  (crate-source "web-sys" "0.3.80"
                "1p4nwlrjvaymzas40x1i0n6mzn8pilzl6jwcswlw97c6bs4k9rzv"))

(define rust-web-sys-0.3.81
  (crate-source "web-sys" "0.3.81"
                "0871ifd79ni9813sp5amk7wb3avznkijlsly2ap4r9r4m4bw8rwk"))

(define rust-web-time-1.1.0
  (crate-source "web-time" "1.1.0"
                "1fx05yqx83dhx628wb70fyy10yjfq1jpl20qfqhdkymi13rq0ras"))

(define rust-web-view-0.7.3.82d7cbc rust-web-view-0.7.3.82d7cbc)

(define rust-webbrowser-1.0.4
  (crate-source "webbrowser" "1.0.4"
                "0pjmpl6p09y837ikdh4v8dymisfz0sk8dgdiaqc2y52ihigjkpym"))

(define rust-webbrowser-1.0.5
  (crate-source "webbrowser" "1.0.5"
                "166yrfz20a5qzxq881acw973537v0dq1bi6cwns853l3pb0g7x5a"))

(define rust-webkit2gtk-sys-0.12.0
  (crate-source "webkit2gtk-sys" "0.12.0"
                "0686iy2jrq8h2i2p4zb53mg32ql5zagba1fskcdi23asr0w537iq"))

(define rust-webp-0.3.0
  (crate-source "webp" "0.3.0"
                "169dihaimcr3ih3fqg8z3n47aqd9rhb3vhw4hk0g0npva4pialwg"
                #:snippet '(delete-file-recursively "assets")))

(define rust-webpki-0.22.4
  (crate-source "webpki" "0.22.4"
                "0lwv7jdlcqjjqqhxcrapnyk5bz4lvr12q444b50gzl3krsjswqzd"
                #:snippet '(delete-file-recursively "tests")))

(define rust-webpki-roots-0.25.4
  (crate-source "webpki-roots" "0.25.4"
                "1qgqa615gc1cgklls4bkjp9jv9pvv3jnl82lc6wd7dkximywa82z"
                #:snippet '(delete-file-recursively "tests")))

(define rust-webpki-roots-0.26.11
  (crate-source "webpki-roots" "0.26.11"
                "1agpayg5zzf7m1a01q30jahlgmn5nwggbabdhq0in008pf5c66sj"
                #:snippet '(delete-file-recursively "tests")))

(define rust-webpki-roots-0.26.8
  (crate-source "webpki-roots" "0.26.8"
                "1jf54brni9lk4ak5pkma2pn18hli22gr7i7wp9zn2lzayy8v4412"
                #:snippet '(delete-file-recursively "tests")))

(define rust-webpki-roots-1.0.0
  (crate-source "webpki-roots" "1.0.0"
                "1nyg365shpxkbazrsvh9c4cv7ar16xnfq62w48xdmwn43j6p6lr8"
                #:snippet '(delete-file-recursively "tests")))

(define rust-webpki-roots-1.0.1
  (crate-source "webpki-roots" "1.0.1"
                "00mm4bhkvis59pm2a7yz3ak6q8rykcj0ddj09wxfskm285ddv0l7"
                #:snippet '(delete-file-recursively "tests")))

(define rust-webview-sys-0.6.2.82d7cbc rust-web-view-0.7.3.82d7cbc)

(define rust-weezl-0.1.10
  (crate-source "weezl" "0.1.10"
                "1wqnxqn8n90bgazs6djlibf58ppdxki4slblwp9lgnq0fwkv6ld7"))

(define rust-weezl-0.1.8
  (crate-source "weezl" "0.1.8"
                "10lhndjgs6y5djpg3b420xngcr6jkmv70q8rb1qcicbily35pa2k"))

(define rust-wezterm-bidi-0.2.3
  (crate-source "wezterm-bidi" "0.2.3"
                "1v7kwmnxfplv9kgdmamn6csbn2ag5xjr0y6gs797slk0alsnw2hc"))

(define rust-wezterm-blob-leases-0.1.1
  (crate-source "wezterm-blob-leases" "0.1.1"
                "1dwf8bm3cwdi37fandwbk7nsfhn9spv4wm0l86gf551xv7vaybb9"))

(define rust-wezterm-color-types-0.3.0
  (crate-source "wezterm-color-types" "0.3.0"
                "15j29f60p1dc0msx50x940niyv9d5zpynavpcc6jf44hbkrixs3x"))

(define rust-wezterm-dynamic-0.2.1
  (crate-source "wezterm-dynamic" "0.2.1"
                "1b6mrk09xxiz66dj3912kmiq8rl7dqig6rwminkfmmhg287bcajz"))

(define rust-wezterm-dynamic-derive-0.1.1
  (crate-source "wezterm-dynamic-derive" "0.1.1"
                "0nspip7gwzmfn66fbnbpa2yik2sb97nckzmgir25nr4wacnwzh26"))

(define rust-wezterm-input-types-0.1.0
  (crate-source "wezterm-input-types" "0.1.0"
                "0zp557014d458a69yqn9dxfy270b6kyfdiynr5p4algrb7aas4kh"))

(define rust-which-2.0.1
  (crate-source "which" "2.0.1"
                "0r7i793sc0xqnd2fxnqbksj7j1kx65bwn81b8z49750v4c8cnymm"))

(define rust-which-4.4.0
  (crate-source "which" "4.4.0"
                "0sd24r17q4j3hc2yjjrg9q4qya1y4n9zq0bj9c2rla1bqn2cfh94"))

(define rust-which-4.4.2
  (crate-source "which" "4.4.2"
                "1ixzmx3svsv5hbdvd8vdhd3qwvf6ns8jdpif1wmwsy10k90j9fl7"))

(define rust-which-6.0.3
  (crate-source "which" "6.0.3"
                "07yg74dsq644hq5a35546c9mja6rsjdsg92rykr9hkflxf7r5vml"))

(define rust-which-7.0.2
  (crate-source "which" "7.0.2"
                "10qjdz8cld2ljd62kk3yaqhn7b96dj4bmy02vjmb6wphw5hwhx17"))

(define rust-which-7.0.3
  (crate-source "which" "7.0.3"
                "0qj7lx7v98hcs0rfw4xqw1ssn47v6h7hhak0ai4bbrfk7z747mi4"))

(define rust-which-8.0.0
  (crate-source "which" "8.0.0"
                "07dsqyvvyaqp3dbj4cdl3ib5fxhdf29l6vihm3pcihq666avpynk"))

(define rust-whoami-1.6.0
  (crate-source "whoami" "1.6.0"
                "19q2vm5ax3bgwffbywn4ad62anc1f4l1ky61h0y2qjdb30qx3539"))

(define rust-whoami-1.6.1
  (crate-source "whoami" "1.6.1"
                "0zg9sz669vhqyxysn4lymnianj29jxs2vl6k2lqcl0kp0yslsjjx"))

(define rust-wide-0.7.32
  (crate-source "wide" "0.7.32"
                "08mb6iqdscqiqrbfkjrnfr876ah4cc0cx5pjilz3yqw1k9mmgda1"))

(define rust-widestring-0.4.3
  (crate-source "widestring" "0.4.3"
                "0z3ba8qrxb62vpfgk7n2xs2grm9kdaj1cz4q8s0gs8fx8h0r8s61"))

(define rust-widestring-1.2.0
  (crate-source "widestring" "1.2.0"
                "0zg04qvpk2xysbbwhdgyilgn4p9igvyj9fhzl7pckam1khvz6z6x"))

(define rust-wild-2.2.1
  (crate-source "wild" "2.2.1"
                "1q8hnhmv3fvgx0j7bv8qig00599a15mfsdhgx3hq2ljpiky1l4x3"))

(define rust-wildmatch-2.4.0
  (crate-source "wildmatch" "2.4.0"
                "1zbi3q4djhlli496mg45sfacy1g5c6f5h3rmw7mma9n6z2qimkk8"))

(define rust-wildmatch-2.5.0
  (crate-source "wildmatch" "2.5.0"
                "10nk5s864iczsdrxn38iypixp5mk36pwyv2k0xkfzfka4dxd1drr"))

(define rust-win-crypto-ng-0.5.1
  (crate-source "win-crypto-ng" "0.5.1"
                "14mv2wsvji8x7ds72zsjpz6hdq57y4r8r38xjyr4mrbib91zpawr"))

(define rust-win32job-2.0.3
  (crate-source "win32job" "2.0.3"
                "070r3fhg2m26c4r7pd7plcnbxlpwn1ldh6v9m1a43wzvrhj6fslc"))

(define rust-winapi-0.2.8
  (crate-source "winapi" "0.2.8"
                "0yh816lh6lf56dpsgxy189c2ai1z3j8mw9si6izqb6wsjkbcjz8n"))

(define rust-winapi-0.3.9
  (crate-source "winapi" "0.3.9"
                "06gl025x418lchw1wxj64ycr7gha83m44cjr5sarhynd9xkrm0sw"))

(define rust-winapi-build-0.1.1
  (crate-source "winapi-build" "0.1.1"
                "1g4rqsgjky0a7530qajn2bbfcrl2v0zb39idgdws9b1l7gp5wc9d"))

(define rust-winapi-i686-pc-windows-gnu-0.4.0
  (crate-source "winapi-i686-pc-windows-gnu" "0.4.0"
                "1dmpa6mvcvzz16zg6d5vrfy4bxgg541wxrcip7cnshi06v38ffxc"
                #:snippet '(delete-file-recursively "lib")))

(define rust-winapi-util-0.1.10
  (crate-source "winapi-util" "0.1.10"
                "08hb8rj3aq9lcrfmliqs4l7v9zh6srbcn0376yn0pndkf5qvyy09"))

(define rust-winapi-util-0.1.11
  (crate-source "winapi-util" "0.1.11"
                "08hdl7mkll7pz8whg869h58c1r9y7in0w0pk8fm24qc77k0b39y2"))

(define rust-winapi-util-0.1.5
  (crate-source "winapi-util" "0.1.5"
                "0y71bp7f6d536czj40dhqk0d55wfbbwqfp2ymqf1an5ibgl6rv3h"))

(define rust-winapi-util-0.1.9
  (crate-source "winapi-util" "0.1.9"
                "1fqhkcl9scd230cnfj8apfficpf5c9vhwnk4yy9xfc1sw69iq8ng"))

(define rust-winapi-wsapoll-0.1.2
  (crate-source "winapi-wsapoll" "0.1.2"
                "0a1zxmpvxaw75y4lwavi6qbq95cnrz83a5p84rarjxn5g7vcbbqy"))

(define rust-winapi-x86-64-pc-windows-gnu-0.4.0
  (crate-source "winapi-x86_64-pc-windows-gnu" "0.4.0"
                "0gqq64czqb64kskjryj8isp62m2sgvx25yyj3kpc2myh85w24bki"
                #:snippet '(delete-file-recursively "lib")))

(define rust-winch-codegen-29.0.1
  (crate-source "winch-codegen" "29.0.1"
                "1745a92ni167lczz20rc5vjld613m9x4id7l1aib0v7lqpr9x11g"))

(define rust-windows-0.39.0
  (crate-source "windows" "0.39.0"
                "0jkr4rxj6jn7brqxycr1pjgvnixsimsiywl4a7v20q5ca05bvi7i"
                #:snippet '(delete-file-recursively "src/Windows")))

(define rust-windows-0.43.0
  (crate-source "windows" "0.43.0"
                "0i32alvc4n4l7abmv1fsvnd1lzw17f1cpr16kgx0sqz5wg82wrh4"
                #:snippet '(delete-file-recursively "src/Windows")))

(define rust-windows-0.44.0
  (crate-source "windows" "0.44.0"
                "0ax1ip82dhszxz4hhsrdj3b0681xw6axahnfldxcgi506nmmsx4y"
                #:snippet '(delete-file-recursively "src/Windows")))

(define rust-windows-0.48.0
  (crate-source "windows" "0.48.0"
                "03vh89ilnxdxdh0n9np4ns4m10fvm93h3b0cc05ipg3qq1mqi1p6"
                #:snippet '(delete-file-recursively "src/Windows")))

(define rust-windows-0.52.0
  (crate-source "windows" "0.52.0"
                "1gnh210qjlprpd1szaq04rjm1zqgdm9j7l9absg0kawi2rwm72p4"))

(define rust-windows-0.54.0
  (crate-source "windows" "0.54.0"
                "0j8vd8sg2rbln6g3a608qg1a7r2lwxcga78mmxjjin5ybmrfallj"))

(define rust-windows-0.56.0
  (crate-source "windows" "0.56.0"
                "0cp10nzrqgrlk91dpwxjcpzyy6imr5vxr5f898pss7nz3gq9vrhx"))

(define rust-windows-0.57.0
  (crate-source "windows" "0.57.0"
                "0hqid10bqvxa3pbpgvrh2cilf950lxsd9zqfv3rldc73v2s2qd0j"))

(define rust-windows-0.58.0
  (crate-source "windows" "0.58.0"
                "1dkjj94b0gn91nn1n22cvm4afsj98f5qrhcl3112v6f4jcfx816x"))

(define rust-windows-0.59.0
  (crate-source "windows" "0.59.0"
                "189ji8zy8mmy31k4vs0xrdmskfqvh4km13ifyvklnc4k1bp9m4bz"))

(define rust-windows-0.61.1
  (crate-source "windows" "0.61.1"
                "06d4ahj0lns53cgza2w73r82fqwabyxqp1npp81cnf2p08yqzvn5"))

(define rust-windows-0.61.3
  (crate-source "windows" "0.61.3"
                "14v8dln7i4ccskd8danzri22bkjkbmgzh284j3vaxhd4cykx7awv"))

(define rust-windows-aarch64-gnullvm-0.42.2
  (crate-source "windows_aarch64_gnullvm" "0.42.2"
                "1y4q0qmvl0lvp7syxvfykafvmwal5hrjb4fmv04bqs0bawc52yjr"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-aarch64-gnullvm-0.48.0
  (crate-source "windows_aarch64_gnullvm" "0.48.0"
                "1g71yxi61c410pwzq05ld7si4p9hyx6lf5fkw21sinvr3cp5gbli"))

(define rust-windows-aarch64-gnullvm-0.48.5
  (crate-source "windows_aarch64_gnullvm" "0.48.5"
                "1n05v7qblg1ci3i567inc7xrkmywczxrs1z3lj3rkkxw18py6f1b"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-aarch64-gnullvm-0.52.6
  (crate-source "windows_aarch64_gnullvm" "0.52.6"
                "1lrcq38cr2arvmz19v32qaggvj8bh1640mdm9c2fr877h0hn591j"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-aarch64-gnullvm-0.53.0
  (crate-source "windows_aarch64_gnullvm" "0.53.0"
                "0r77pbpbcf8bq4yfwpz2hpq3vns8m0yacpvs2i5cn6fx1pwxbf46"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-aarch64-msvc-0.33.0
  (crate-source "windows_aarch64_msvc" "0.33.0"
                "01q80v2zzfc144xsqj3yhd62rn1dy1kyamhyv0gcrf4sxg9iyxnd"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-aarch64-msvc-0.39.0
  (crate-source "windows_aarch64_msvc" "0.39.0"
                "1wj0nfmyli4bn5af243r4s3zncxv0n4j6dd8zyb41gcnc1k12xzc"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-aarch64-msvc-0.42.2
  (crate-source "windows_aarch64_msvc" "0.42.2"
                "0hsdikjl5sa1fva5qskpwlxzpc5q9l909fpl1w6yy1hglrj8i3p0"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-aarch64-msvc-0.48.0
  (crate-source "windows_aarch64_msvc" "0.48.0"
                "1wvwipchhywcjaw73h998vzachf668fpqccbhrxzrz5xszh2gvxj"))

(define rust-windows-aarch64-msvc-0.48.5
  (crate-source "windows_aarch64_msvc" "0.48.5"
                "1g5l4ry968p73g6bg6jgyvy9lb8fyhcs54067yzxpcpkf44k2dfw"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-aarch64-msvc-0.52.6
  (crate-source "windows_aarch64_msvc" "0.52.6"
                "0sfl0nysnz32yyfh773hpi49b1q700ah6y7sacmjbqjjn5xjmv09"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-aarch64-msvc-0.53.0
  (crate-source "windows_aarch64_msvc" "0.53.0"
                "0v766yqw51pzxxwp203yqy39ijgjamp54hhdbsyqq6x1c8gilrf7"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-acl-0.3.0
  (crate-source "windows-acl" "0.3.0"
                "1hyfng4dagfndxpwxynjk9zird8lhrp7zrsqc1h4rjvbk0iifyqp"
                #:snippet '(delete-file-recursively "docs")))

(define rust-windows-collections-0.2.0
  (crate-source "windows-collections" "0.2.0"
                "1s65anr609qvsjga7w971p6iq964h87670dkfqfypnfgwnswxviv"))

(define rust-windows-core-0.52.0
  (crate-source "windows-core" "0.52.0"
                "1nc3qv7sy24x0nlnb32f7alzpd6f72l4p24vl65vydbyil669ark"))

(define rust-windows-core-0.54.0
  (crate-source "windows-core" "0.54.0"
                "0r8x2sgl4qq1h23ldf4z7cj213k0bz7479m8a156h79mi6f1nrhj"))

(define rust-windows-core-0.56.0
  (crate-source "windows-core" "0.56.0"
                "19pj57bm0rzhlk0ghrccd3i5zvh0ghm52f8cmdc8d3yhs8pfb626"))

(define rust-windows-core-0.57.0
  (crate-source "windows-core" "0.57.0"
                "0bc3jxw2jw76xkk3ddvnp5b2m76qmbzv1qncgvb6qrlhl8wj9vfj"))

(define rust-windows-core-0.58.0
  (crate-source "windows-core" "0.58.0"
                "16czypy425jzmiys4yb3pwsh7cm6grxn9kjp889iqnf2r17d99kb"))

(define rust-windows-core-0.59.0
  (crate-source "windows-core" "0.59.0"
                "1kn1f0bf4rqynczkfgn543i170arbwp04pg1sjq8890isa7f2341"))

(define rust-windows-core-0.61.0
  (crate-source "windows-core" "0.61.0"
                "104915nsby2cgp322pqqkmj2r82v5sg4hil0hxddg1hc67gc2qs7"))

(define rust-windows-core-0.61.2
  (crate-source "windows-core" "0.61.2"
                "1qsa3iw14wk4ngfl7ipcvdf9xyq456ms7cx2i9iwf406p7fx7zf0"))

(define rust-windows-core-0.62.0
  (crate-source "windows-core" "0.62.0"
                "0z294cblga0dl2dg9s9080xyglkh33b7zc05i8nqsmyyyxl73zjp"))

(define rust-windows-core-0.62.1
  (crate-source "windows-core" "0.62.1"
                "1aa94x61q0x39xnlzxjmahwck9i5p51xgzrz7m6hi1dj2rafwi38"))

(define rust-windows-future-0.2.0
  (crate-source "windows-future" "0.2.0"
                "0cms355fawdyz32r1573lwccyvyscp4ip3l2376hmdmpzjz6n7bs"))

(define rust-windows-future-0.2.1
  (crate-source "windows-future" "0.2.1"
                "13mdzcdn51ckpzp3frb8glnmkyjr1c30ym9wnzj9zc97hkll2spw"))

(define rust-windows-i686-gnu-0.33.0
  (crate-source "windows_i686_gnu" "0.33.0"
                "03nsi8h0yd4n9wgpxcpynzwlnacihisgmh021kfb5fln79qczc6a"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-gnu-0.39.0
  (crate-source "windows_i686_gnu" "0.39.0"
                "06wynhxklmh3s1ril9bh00rhv1npppcyirsp60p09xx501qwagvn"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-gnu-0.42.2
  (crate-source "windows_i686_gnu" "0.42.2"
                "0kx866dfrby88lqs9v1vgmrkk1z6af9lhaghh5maj7d4imyr47f6"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-gnu-0.48.0
  (crate-source "windows_i686_gnu" "0.48.0"
                "0hd2v9kp8fss0rzl83wzhw0s5z8q1b4875m6s1phv0yvlxi1jak2"))

(define rust-windows-i686-gnu-0.48.5
  (crate-source "windows_i686_gnu" "0.48.5"
                "0gklnglwd9ilqx7ac3cn8hbhkraqisd0n83jxzf9837nvvkiand7"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-gnu-0.52.6
  (crate-source "windows_i686_gnu" "0.52.6"
                "02zspglbykh1jh9pi7gn8g1f97jh1rrccni9ivmrfbl0mgamm6wf"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-gnu-0.53.0
  (crate-source "windows_i686_gnu" "0.53.0"
                "1hvjc8nv95sx5vdd79fivn8bpm7i517dqyf4yvsqgwrmkmjngp61"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-gnullvm-0.52.6
  (crate-source "windows_i686_gnullvm" "0.52.6"
                "0rpdx1537mw6slcpqa0rm3qixmsb79nbhqy5fsm3q2q9ik9m5vhf"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-gnullvm-0.53.0
  (crate-source "windows_i686_gnullvm" "0.53.0"
                "04df1in2k91qyf1wzizvh560bvyzq20yf68k8xa66vdzxnywrrlw"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-msvc-0.33.0
  (crate-source "windows_i686_msvc" "0.33.0"
                "1l3kwxgdfg4lnx2j5bkcx6cnvhxnpcsbqjm3idhwxmwsrj4vxzcc"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-msvc-0.39.0
  (crate-source "windows_i686_msvc" "0.39.0"
                "01hiv9msxssy5iqxs7bczvf094k4mz56yi4z1bhj32c2b3zcpivv"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-msvc-0.42.2
  (crate-source "windows_i686_msvc" "0.42.2"
                "0q0h9m2aq1pygc199pa5jgc952qhcnf0zn688454i7v4xjv41n24"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-msvc-0.48.0
  (crate-source "windows_i686_msvc" "0.48.0"
                "004fkyqv3if178xx9ksqc4qqv8sz8n72mpczsr2vy8ffckiwchj5"))

(define rust-windows-i686-msvc-0.48.5
  (crate-source "windows_i686_msvc" "0.48.5"
                "01m4rik437dl9rdf0ndnm2syh10hizvq0dajdkv2fjqcywrw4mcg"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-msvc-0.52.6
  (crate-source "windows_i686_msvc" "0.52.6"
                "0rkcqmp4zzmfvrrrx01260q3xkpzi6fzi2x2pgdcdry50ny4h294"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-i686-msvc-0.53.0
  (crate-source "windows_i686_msvc" "0.53.0"
                "0pcvb25fkvqnp91z25qr5x61wyya12lx8p7nsa137cbb82ayw7sq"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-implement-0.48.0
  (crate-source "windows-implement" "0.48.0"
                "1764n853zd7bb0wn94i0qxfs6kdy7wrz7v9qhdn7x7hvk64fabjy"))

(define rust-windows-implement-0.56.0
  (crate-source "windows-implement" "0.56.0"
                "16rgkvlx4syqmajfdwmkcvn6nvh126wjj8sg3jvsk5fdivskbz7n"))

(define rust-windows-implement-0.57.0
  (crate-source "windows-implement" "0.57.0"
                "1mqs7qypclnmx5r8yq5jy3g2d8i27vzag9yzzzxzpdnmb70ds1wi"))

(define rust-windows-implement-0.58.0
  (crate-source "windows-implement" "0.58.0"
                "16spr5z65z21qyv379rv2mb1s5q2i9ibd1p2pkn0dr9qr535pg9b"))

(define rust-windows-implement-0.59.0
  (crate-source "windows-implement" "0.59.0"
                "1ha89vc4v4vabwdjkz46hdsnl2jp7ckpzw88qdca0j9g3q2pnmw3"))

(define rust-windows-implement-0.60.0
  (crate-source "windows-implement" "0.60.0"
                "0dm88k3hlaax85xkls4gf597ar4z8m5vzjjagzk910ph7b8xszx4"))

(define rust-windows-implement-0.60.1
  (crate-source "windows-implement" "0.60.1"
                "1q2lfwdqrkfzsrlshvvyr2cj7ckq4rqxj0ispzlnvyvl5bj0gczd"))

(define rust-windows-interface-0.48.0
  (crate-source "windows-interface" "0.48.0"
                "1iqcilw0hfyzwhk12xfmcy40r10406sgf4xmdansijlv1kr8vyz6"))

(define rust-windows-interface-0.56.0
  (crate-source "windows-interface" "0.56.0"
                "1k2prfxna0mw47f8gi8qhw9jfpw66bh2cqzs67sgipjfpx30b688"))

(define rust-windows-interface-0.57.0
  (crate-source "windows-interface" "0.57.0"
                "19zwlzr0q1z9s692681yb5w2lhvwcyx4v95s25hfdkd3isry9gi9"))

(define rust-windows-interface-0.58.0
  (crate-source "windows-interface" "0.58.0"
                "059mxmfvx3x88q74ms0qlxmj2pnidmr5mzn60hakn7f95m34qg05"))

(define rust-windows-interface-0.59.1
  (crate-source "windows-interface" "0.59.1"
                "1a4zr8740gyzzhq02xgl6vx8l669jwfby57xgf0zmkcdkyv134mx"))

(define rust-windows-interface-0.59.2
  (crate-source "windows-interface" "0.59.2"
                "19a6if8dfnazjgjw4hm0kayk9vrjclyj3iqivcaaqr39pkfx3ay0"))

(define rust-windows-link-0.1.0
  (crate-source "windows-link" "0.1.0"
                "1qr0srnkw148wbrws3726pm640h2vxgcdlxn0cxpbcg27irzvk3d"))

(define rust-windows-link-0.1.1
  (crate-source "windows-link" "0.1.1"
                "0f2cq7imbrppsmmnz8899hfhg07cp5gq6rh0bjhb1qb6nwshk13n"))

(define rust-windows-link-0.1.3
  (crate-source "windows-link" "0.1.3"
                "12kr1p46dbhpijr4zbwr2spfgq8i8c5x55mvvfmyl96m01cx4sjy"))

(define rust-windows-link-0.2.0
  (crate-source "windows-link" "0.2.0"
                "0r9w2z96d5phmm185aq92z54jp9h2nqisa4wgc71idxbc436rr25"))

(define rust-windows-numerics-0.2.0
  (crate-source "windows-numerics" "0.2.0"
                "1cf2j8nbqf0hqqa7chnyid91wxsl2m131kn0vl3mqk3c0rlayl4i"))

(define rust-windows-registry-0.4.0
  (crate-source "windows-registry" "0.4.0"
                "18wbgr6z6765qdnasi8mmvxhvp82xd1zlvd6s7pp2l5lvn8av1j2"))

(define rust-windows-registry-0.5.1
  (crate-source "windows-registry" "0.5.1"
                "0bl461w9qj942sg27c181sxhkzrbw8r779ixvzgm6xnw6vja67dd"))

(define rust-windows-registry-0.5.2
  (crate-source "windows-registry" "0.5.2"
                "0848hyj0irwljppy5j9lkphrx9m8isx8m6xr1cja20ykpn9v1fmk"))

(define rust-windows-registry-0.5.3
  (crate-source "windows-registry" "0.5.3"
                "17j9cxlnksdypanazss6cnh36v3rwvs86j4mpixwkvv5hz99x2jv"))

(define rust-windows-result-0.1.2
  (crate-source "windows-result" "0.1.2"
                "1y274q1v0vy21lhkgslpxpq1m08hvr1mcs2l88h1b1gcx0136f2y"))

(define rust-windows-result-0.2.0
  (crate-source "windows-result" "0.2.0"
                "03mf2z1xcy2slhhsm15z24p76qxgm2m74xdjp8bihyag47c4640x"))

(define rust-windows-result-0.3.2
  (crate-source "windows-result" "0.3.2"
                "0li2f76anf0rg7i966d9qs5iprsg555g9rgyzj7gcpfr9wdd2ky6"))

(define rust-windows-result-0.3.4
  (crate-source "windows-result" "0.3.4"
                "1il60l6idrc6hqsij0cal0mgva6n3w6gq4ziban8wv6c6b9jpx2n"))

(define rust-windows-result-0.4.0
  (crate-source "windows-result" "0.4.0"
                "0zqn8kmmf7y9yw9g7q6pbcg9dbry9m03fqi0b92q767q0v1xr13h"))

(define rust-windows-service-0.7.0
  (crate-source "windows-service" "0.7.0"
                "12lwc04rji06p8g8dhc5aps6cw3xbx6admzqxj8l0jkkgz66nkfj"))

(define rust-windows-strings-0.1.0
  (crate-source "windows-strings" "0.1.0"
                "042dxvi3133f7dyi2pgcvknwkikk47k8bddwxbq5s0l6qhjv3nac"))

(define rust-windows-strings-0.3.1
  (crate-source "windows-strings" "0.3.1"
                "06bkhkyclbfchcsv5bnhz77r290k20m15glj2xq60ra0bp64iyl7"))

(define rust-windows-strings-0.4.0
  (crate-source "windows-strings" "0.4.0"
                "15rg6a0ha1d231wwps2qlgyqrgkyj1r8v9vsb8nlbvih4ijajavs"))

(define rust-windows-strings-0.4.2
  (crate-source "windows-strings" "0.4.2"
                "0mrv3plibkla4v5kaakc2rfksdd0b14plcmidhbkcfqc78zwkrjn"))

(define rust-windows-strings-0.5.0
  (crate-source "windows-strings" "0.5.0"
                "1nld65azvms87rdm2bdm8gskwdmsswh4pxbc8babxc2klmawc63j"))

(define rust-windows-sys-0.33.0
  (crate-source "windows-sys" "0.33.0"
                "0xdym5hgf2pp5lmfdjk4lynad99w4j02v9yzn6752a9ncsbb1ns3"))

(define rust-windows-sys-0.42.0
  (crate-source "windows-sys" "0.42.0"
                "19waf8aryvyq9pzk0gamgfwjycgzk4gnrazpfvv171cby0h1hgjs"))

(define rust-windows-sys-0.45.0
  (crate-source "windows-sys" "0.45.0"
                "1l36bcqm4g89pknfp8r9rl1w4bn017q6a8qlx8viv0xjxzjkna3m"))

(define rust-windows-sys-0.48.0
  (crate-source "windows-sys" "0.48.0"
                "1aan23v5gs7gya1lc46hqn9mdh8yph3fhxmhxlw36pn6pqc28zb7"))

(define rust-windows-sys-0.52.0
  (crate-source "windows-sys" "0.52.0"
                "0gd3v4ji88490zgb6b5mq5zgbvwv7zx1ibn8v3x83rwcdbryaar8"))

(define rust-windows-sys-0.59.0
  (crate-source "windows-sys" "0.59.0"
                "0fw5672ziw8b3zpmnbp9pdv1famk74f1l9fcbc3zsrzdg56vqf0y"))

(define rust-windows-sys-0.60.2
  (crate-source "windows-sys" "0.60.2"
                "1jrbc615ihqnhjhxplr2kw7rasrskv9wj3lr80hgfd42sbj01xgj"))

(define rust-windows-sys-0.61.0
  (crate-source "windows-sys" "0.61.0"
                "1ajpwsmzfcsa1r7i0dxzvfn24dp3525rcd7aq95ydvdj8171h0g2"))

(define rust-windows-sys-0.61.1
  (crate-source "windows-sys" "0.61.1"
                "03vg2rxm0lyiyq64b5sm95lkg2x95sjdb0zb0y4q8g2avm0rw43g"))

(define rust-windows-targets-0.42.2
  (crate-source "windows-targets" "0.42.2"
                "0wfhnib2fisxlx8c507dbmh97kgij4r6kcxdi0f9nk6l1k080lcf"))

(define rust-windows-targets-0.48.1
  (crate-source "windows-targets" "0.48.1"
                "0pz9ad4mpp06h80hkmzlib78b5r9db7isycy1gr9j17pj1sb3m05"))

(define rust-windows-targets-0.48.5
  (crate-source "windows-targets" "0.48.5"
                "034ljxqshifs1lan89xwpcy1hp0lhdh4b5n0d2z4fwjx2piacbws"))

(define rust-windows-targets-0.52.6
  (crate-source "windows-targets" "0.52.6"
                "0wwrx625nwlfp7k93r2rra568gad1mwd888h1jwnl0vfg5r4ywlv"))

(define rust-windows-targets-0.53.0
  (crate-source "windows-targets" "0.53.0"
                "12yakpjizhfpppz1i3zgcwxlbar8axrp9j87fmywpydarvlcgr5i"))

(define rust-windows-targets-0.53.2
  (crate-source "windows-targets" "0.53.2"
                "1vwanhx2br7dh8mmrszdbcf01bccjr01mcyxcscxl4ffr7y6jvy6"))

(define rust-windows-targets-0.53.3
  (crate-source "windows-targets" "0.53.3"
                "14fwwm136dhs3i1impqrrip7nvkra3bdxa4nqkblj604qhqn1znm"))

(define rust-windows-targets-0.53.4
  (crate-source "windows-targets" "0.53.4"
                "0jxc6f032xb3bbb7mj9rlhky84w7vz7hkbsh8s2hcakdysvvfhid"))

(define rust-windows-threading-0.1.0
  (crate-source "windows-threading" "0.1.0"
                "19jpn37zpjj2q7pn07dpq0ay300w65qx7wdp13wbp8qf5snn6r5n"))

(define rust-windows-version-0.1.4
  (crate-source "windows-version" "0.1.4"
                "0p16g570yvgxiqyh0bgbck4n1wkxdhl8711ms8xa44734xk5qjp0"))

(define rust-windows-x86-64-gnu-0.33.0
  (crate-source "windows_x86_64_gnu" "0.33.0"
                "1127n961nib9338n0g0sp1464v8wnw0hvmw45sr7pkly1q69ppdl"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-gnu-0.39.0
  (crate-source "windows_x86_64_gnu" "0.39.0"
                "0r9b4lmapq66nn2dga7a0mkdv5sgbp184kfwx3hklrbxcdjw2s38"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-gnu-0.42.2
  (crate-source "windows_x86_64_gnu" "0.42.2"
                "0dnbf2xnp3xrvy8v9mgs3var4zq9v9yh9kv79035rdgyp2w15scd"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-gnu-0.48.0
  (crate-source "windows_x86_64_gnu" "0.48.0"
                "1cblz5m6a8q6ha09bz4lz233dnq5sw2hpra06k9cna3n3xk8laya"))

(define rust-windows-x86-64-gnu-0.48.5
  (crate-source "windows_x86_64_gnu" "0.48.5"
                "13kiqqcvz2vnyxzydjh73hwgigsdr2z1xpzx313kxll34nyhmm2k"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-gnu-0.52.6
  (crate-source "windows_x86_64_gnu" "0.52.6"
                "0y0sifqcb56a56mvn7xjgs8g43p33mfqkd8wj1yhrgxzma05qyhl"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-gnu-0.53.0
  (crate-source "windows_x86_64_gnu" "0.53.0"
                "1flh84xkssn1n6m1riddipydcksp2pdl45vdf70jygx3ksnbam9f"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-gnullvm-0.42.2
  (crate-source "windows_x86_64_gnullvm" "0.42.2"
                "18wl9r8qbsl475j39zvawlidp1bsbinliwfymr43fibdld31pm16"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-gnullvm-0.48.0
  (crate-source "windows_x86_64_gnullvm" "0.48.0"
                "0lxryz3ysx0145bf3i38jkr7f9nxiym8p3syklp8f20yyk0xp5kq"))

(define rust-windows-x86-64-gnullvm-0.48.5
  (crate-source "windows_x86_64_gnullvm" "0.48.5"
                "1k24810wfbgz8k48c2yknqjmiigmql6kk3knmddkv8k8g1v54yqb"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-gnullvm-0.52.6
  (crate-source "windows_x86_64_gnullvm" "0.52.6"
                "03gda7zjx1qh8k9nnlgb7m3w3s1xkysg55hkd1wjch8pqhyv5m94"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-gnullvm-0.53.0
  (crate-source "windows_x86_64_gnullvm" "0.53.0"
                "0mvc8119xpbi3q2m6mrjcdzl6afx4wffacp13v76g4jrs1fh6vha"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-msvc-0.33.0
  (crate-source "windows_x86_64_msvc" "0.33.0"
                "1akf81g0bh8mv8wjpiifd819r0hx3r0xrz9zgzn4hl298sk4l7pz"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-msvc-0.39.0
  (crate-source "windows_x86_64_msvc" "0.39.0"
                "02g8fy1sv8g0p4fi2yk62j9a5zwhcfknp9vwg1ifkjp97a440kay"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-msvc-0.42.2
  (crate-source "windows_x86_64_msvc" "0.42.2"
                "1w5r0q0yzx827d10dpjza2ww0j8iajqhmb54s735hhaj66imvv4s"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-msvc-0.48.0
  (crate-source "windows_x86_64_msvc" "0.48.0"
                "12ipr1knzj2rwjygyllfi5mkd0ihnbi3r61gag5n2jgyk5bmyl8s"))

(define rust-windows-x86-64-msvc-0.48.5
  (crate-source "windows_x86_64_msvc" "0.48.5"
                "0f4mdp895kkjh9zv8dxvn4pc10xr7839lf5pa9l0193i2pkgr57d"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-msvc-0.52.6
  (crate-source "windows_x86_64_msvc" "0.52.6"
                "1v7rb5cibyzx8vak29pdrk8nx9hycsjs4w0jgms08qk49jl6v7sq"
                #:snippet '(delete-file-recursively "lib")))

(define rust-windows-x86-64-msvc-0.53.0
  (crate-source "windows_x86_64_msvc" "0.53.0"
                "11h4i28hq0zlnjcaqi2xdxr7ibnpa8djfggch9rki1zzb8qi8517"
                #:snippet '(delete-file-recursively "lib")))

(define rust-winit-0.30.11
  (crate-source "winit" "0.30.11"
                "1x5aw47546mjfsvl65xwrhj41n2yr358hiwsqafpgy2d2w89qh54"
                #:snippet '(delete-file-recursively "examples")))

(define rust-winit-0.30.12
  (crate-source "winit" "0.30.12"
                "0cn7wvli4s0l3v5rf6s3rn4j72mdc5p2sxhz6bv0jh4wssg4nvf6"
                #:snippet '(delete-file-recursively "examples")))

(define rust-winit-0.30.9
  (crate-source "winit" "0.30.9"
                "1h1wmvhfcq2lg6c6715d7pgnv85508zm94ahcfvaiv68337yl2d8"
                #:snippet '(delete-file-recursively "examples")))

(define rust-winnow-0.5.15
  (crate-source "winnow" "0.5.15"
                "1z6fikri2xa8qkzf40xn58q7c964s0wk19vw2vajmsf4p6232bkw"))

(define rust-winnow-0.5.40
  (crate-source "winnow" "0.5.40"
                "0xk8maai7gyxda673mmw3pj1hdizy5fpi7287vaywykkk19sk4zm"))

(define rust-winnow-0.6.26
  (crate-source "winnow" "0.6.26"
                "0a4sjbbrkhbd0ba6dy0011hln1q3ry4iv6srqjjpi8hsmk9fv40y"))

(define rust-winnow-0.7.10
  (crate-source "winnow" "0.7.10"
                "1v69byry8fyarzl83wij6f1h3zxnw69assp9kdfb10cdfk42hsf0"))

(define rust-winnow-0.7.11
  (crate-source "winnow" "0.7.11"
                "1kfb2m5ahpqg037a34rgkm18x6nj2mw4fwn89ya0gdw06ipb5ivl"))

(define rust-winnow-0.7.12
  (crate-source "winnow" "0.7.12"
                "159y8inpy86xswmr4yig9hxss0v2fssyqy1kk12504n8jbsfpvgk"))

(define rust-winnow-0.7.13
  (crate-source "winnow" "0.7.13"
                "1krrjc1wj2vx0r57m9nwnlc1zrhga3fq41d8w9hysvvqb5mj7811"))

(define rust-winnow-0.7.3
  (crate-source "winnow" "0.7.3"
                "1c9bmhpdwbdmll6b4l6skabz0296dchnmnxw84hh2y3ggyllwzqf"))

(define rust-winnow-0.7.4
  (crate-source "winnow" "0.7.4"
                "0dmbsz6zfddcgsqzzqxw1h8f7zy19x407g7zl3hyp6vf2m2bb5qf"))

(define rust-winnow-0.7.6
  (crate-source "winnow" "0.7.6"
                "047abhm7qqgc32pf9a2arini5wsrx7p9wsbx3s106jx4pgczrlv3"))

(define rust-winpipe-0.1.1
  (crate-source "winpipe" "0.1.1"
                "0lv041gy68frc8ybi6l99ix1zdd6njdycnch7n8hrgfic8fngkqw"))

(define rust-winreg-0.10.1
  (crate-source "winreg" "0.10.1"
                "17c6h02z88ijjba02bnxi5k94q5cz490nf3njh9yypf8fbig9l40"))

(define rust-winreg-0.50.0
  (crate-source "winreg" "0.50.0"
                "1cddmp929k882mdh6i9f2as848f13qqna6czwsqzkh1pqnr5fkjj"))

(define rust-winreg-0.52.0
  (crate-source "winreg" "0.52.0"
                "19gh9vp7mp1ab84kc3ag48nm9y7xgjhh3xa4vxss1gylk1rsaxx2"))

(define rust-winres-0.1.12
  (crate-source "winres" "0.1.12"
                "0v2gvqnd8iwwvb6fs69nv0mmk1z96430527n0qlfbsarxxhv53dn"))

(define rust-winresource-0.1.20
  (crate-source "winresource" "0.1.20"
                "1xwkhnqdbr8xpkmm05ma9mj7jvicw73yn6xl3h62qy75iv3nfjms"
                #:snippet '(for-each delete-file '("test.ico" "winresource_embed_properties.png"))))

(define rust-winsafe-0.0.19
  (crate-source "winsafe" "0.0.19"
                "0169xy9mjma8dys4m8v4x0xhw2gkbhv2v1wsbvcjl9bhnxxd2dfi"))

(define rust-winsafe-0.0.23
  (crate-source "winsafe" "0.0.23"
                "1f94wsw4w13nkgb00q3iynkj8r40ak1s07207khh3imjiiigr5m0"))

(define rust-wio-0.2.2
  (crate-source "wio" "0.2.2"
                "199p404fp96w1f1c93bf1jrvaqwypxf3hmmldhww4jk4yhr9j4jx"))

(define rust-wiremock-0.6.3
  (crate-source "wiremock" "0.6.3"
                "00g3524s3nz6fbm56c6d7wqxsd1kx1j5mkvvx2cmhyyq9jvq25hh"))

(define rust-wit-bindgen-0.45.0
  (crate-source "wit-bindgen" "0.45.0"
                "053q28k1hn9qgm0l05gr9751d8q34zcz6lbzviwxiqxs3n1q68h5"
                #:snippet '(for-each delete-file (find-files "." "\\.(a|o|wasm)$"))))

(define rust-wit-bindgen-0.45.1
  (crate-source "wit-bindgen" "0.45.1"
                "0dndrimz95nwdv6m24ylj0vj1dvlc012xxrxs13mc1r5y5qk8msw"
                #:snippet '(for-each delete-file (find-files "." "\\.(a|o|wasm)$"))))

(define rust-wit-bindgen-0.46.0
  (crate-source "wit-bindgen" "0.46.0"
                "0ngysw50gp2wrrfxbwgp6dhw1g6sckknsn3wm7l00vaf7n48aypi"
                #:snippet '(for-each delete-file (find-files "." "\\.(a|o|wasm)$"))))

(define rust-wit-bindgen-rt-0.33.0
  (crate-source "wit-bindgen-rt" "0.33.0"
                "0g4lwfp9x6a2i1hgjn8k14nr4fsnpd5izxhc75zpi2s5cvcg6s1j"
                #:snippet '(for-each delete-file (find-files "." "\\.(a|o)$"))))

(define rust-wit-bindgen-rt-0.39.0
  (crate-source "wit-bindgen-rt" "0.39.0"
                "1hd65pa5hp0nl664m94bg554h4zlhrzmkjsf6lsgsb7yc4734hkg"
                #:snippet '(for-each delete-file (find-files "." "\\.(a|o)$"))))

(define rust-wit-parser-0.221.3
  (crate-source "wit-parser" "0.221.3"
                "1b5j2vpiww0crqavaijw4vv1y41darpd38q7id9llsymkrbi4qc9"))

(define rust-wl-clipboard-rs-0.9.2
  (crate-source "wl-clipboard-rs" "0.9.2"
                "1sxsaspzix3xiq6wi1l1g55acgi04sv6r7gxz94zar80wv8ghpwf"))

(define rust-wl-nl80211-0.2.0
  (crate-source "wl-nl80211" "0.2.0"
                "12cp8ar0wa0kvxz756dxnfvz357jqyznh9z2g5vlcf6a3lgczzsw"))

(define rust-wlcs-0.1.0
  (crate-source "wlcs" "0.1.0"
                "17k0nwn3f2z71rncb8glb4x15m5zmcbklnk71hpv739nrq2w769d"))

(define rust-wmi-0.12.2
  (crate-source "wmi" "0.12.2"
                "000sdzx8fnw913ws28ranf8bhm8dvvdpz89s4fhqfbkxpd5b9zys"))

(define rust-wmidi-3.1.0
  (crate-source "wmidi" "3.1.0"
                "1kxnbs18nmpzm2hfwaaa5h2s77cmk5w53srzxqmrqlkdpdcrjafa"))

(define rust-wrapcenum-derive-0.4.1
  (crate-source "wrapcenum-derive" "0.4.1"
                "0vi67wxkpw40cch5jx010wvwf0rjc5f13yxfzd60acimadcz4vx7"))

(define rust-write16-1.0.0
  (crate-source "write16" "1.0.0"
                "0dnryvrrbrnl7vvf5vb1zkmwldhjkf2n5znliviam7bm4900z2fi"))

(define rust-writeable-0.5.5
  (crate-source "writeable" "0.5.5"
                "0lawr6y0bwqfyayf3z8zmqlhpnzhdx0ahs54isacbhyjwa7g778y"))

(define rust-writeable-0.6.1
  (crate-source "writeable" "0.6.1"
                "1fx29zncvbrqzgz7li88vzdm8zvgwgwy2r9bnjqxya09pfwi0bza"))

(define rust-ws-0.9.2
  (crate-source "ws" "0.9.2"
                "0j2qkgshz258wyllvhyc0mxjskzad8i00nbx4h00lsi3bz3r1zi5"))

(define rust-ws2-32-sys-0.2.1
  (crate-source "ws2_32-sys" "0.2.1"
                "0ppscg5qfqaw0gzwv2a4nhn5bn01ff9iwn6ysqnzm4n8s3myz76m"))

(define rust-wu-diff-0.1.2
  (crate-source "wu-diff" "0.1.2"
                "14w8yacn4hk9k4rfzshlgir966xbwgbwk3fvf0l461nyzhsnfglf"
                #:snippet '(delete-file-recursively "tests")))

(define rust-wyz-0.5.1
  (crate-source "wyz" "0.5.1"
                "1vdrfy7i2bznnzjdl9vvrzljvs4s3qm8bnlgqwln6a941gy61wq5"))

(define rust-x11-clipboard-0.3.3
  (crate-source "x11-clipboard" "0.3.3"
                "1smwyr23jns0dncm6bwv00xfxxy99bv6qlx6df7dkdcydk04kgc9"))

(define rust-x11-clipboard-0.9.3
  (crate-source "x11-clipboard" "0.9.3"
                "18rmsm0lrcc4hy2wqs7gn90czv59sv5bj07bbf76nfbysyrp8bb6"))

(define rust-x11-dl-2.21.0
  (crate-source "x11-dl" "2.21.0"
                "0vsiq62xpcfm0kn9zjw5c9iycvccxl22jya8wnk18lyxzqj5jwrq"))

(define rust-x11rb-0.12.0
  (crate-source "x11rb" "0.12.0"
                "02h492k920mb1733cdmly138zfiwkspil6ssqcvi7inyshk1nr5i"))

(define rust-x11rb-0.13.1
  (crate-source "x11rb" "0.13.1"
                "04jyfm0xmc538v09pzsyr2w801yadsgvyl2p0p76hzzffg5gz4ax"))

(define rust-x11rb-0.13.2
  (crate-source "x11rb" "0.13.2"
                "053lvnaw9ycbl791mgwly2hw27q6vqgzrb1y5kz1as52wmdsm4wr"))

(define rust-x11rb-protocol-0.12.0
  (crate-source "x11rb-protocol" "0.12.0"
                "1g24qdvq0mbyl2npz7zqy5v6hpdxq2qakkpnp3x02rzvl3ww7ml2"))

(define rust-x11rb-protocol-0.13.1
  (crate-source "x11rb-protocol" "0.13.1"
                "0gfbxf2k7kbk577j3rjhfx7hm70kmwln6da7xyc4l2za0d2pq47c"))

(define rust-x11rb-protocol-0.13.2
  (crate-source "x11rb-protocol" "0.13.2"
                "1g81cznbyn522b0fbis0i44wh3adad2vhsz5pzf99waf3sbc4vza"))

(define rust-x25519-dalek-2.0.1
  (crate-source "x25519-dalek" "2.0.1"
                "0xyjgqpsa0q6pprakdp58q1hy45rf8wnqqscgzx0gyw13hr6ir67"
                #:snippet '(for-each delete-file-recursively '("docs" "res"))))

(define rust-xattr-1.5.0
  (crate-source "xattr" "1.5.0"
                "17nq2c23zcjciz8sxwhisqjkv4l7zcylx3yl2915c59cy7rcnr8d"))

(define rust-xattr-1.5.1
  (crate-source "xattr" "1.5.1"
                "0299mqwjfayn4g0aq156mc2myirw5bi2b8gqi6x85p2iff1ijfmg"))

(define rust-xcb-0.8.2
  (crate-source "xcb" "0.8.2"
                "1ph27r9nxyfy3hh1c7x85g6dciwxcinf6514pvw9ybhl4hzpm4ay"))

(define rust-xcb-1.5.0
  (crate-source "xcb" "1.5.0"
                "1gjj1481pf08b5fhpdncaphyv0k5ka11n1f8ma6cv4hspc9g5qpi"))

(define rust-xcb-1.6.0
  (crate-source "xcb" "1.6.0"
                "1sipp903dbcbqx9b55dshwiyacp112plwr9yc3hbyfb1g4xi4z7h"))

(define rust-xcb-util-cursor-0.3.3
  (crate-source "xcb-util-cursor" "0.3.3"
                "1b746cbp9g7lnakq0yk4m9909aq6vb38kbl4srq1bwdchinmdi5d"))

(define rust-xcb-util-cursor-sys-0.1.4
  (crate-source "xcb-util-cursor-sys" "0.1.4"
                "00d67fxdzvym65ln988a8x3aakg8bb0drki2bv4ac8s7avrmyrz2"))

(define rust-xcursor-0.3.10
  (crate-source "xcursor" "0.3.10"
                "0awgy98awg4ydcfmynqfcwvl4bnnfcm4i2vvnk2n926a02jy9jdy"))

(define rust-xcursor-0.3.8
  (crate-source "xcursor" "0.3.8"
                "0qazsl7h8nrbbzx84qrv39w8m2qc27g0mvrszgdls2v6n6k3vwqf"))

(define rust-xcursor-0.3.9
  (crate-source "xcursor" "0.3.9"
                "1d9aiiw2yl30ah8xbff53fkvz4hwgjyvsndh9rqwncss67s8fn33"))

(define rust-xdg-2.5.2
  (crate-source "xdg" "2.5.2"
                "0im5nzmywxjgm2pmb48k0cc9hkalarz57f1d9d0x4lvb6cj76fr1"))

(define rust-xdg-3.0.0
  (crate-source "xdg" "3.0.0"
                "1dc5jpkkylp7z54c4xwxzwxx1jb5cklwfjs5493k9y9d7wik7d1g"))

(define rust-xdg-home-1.3.0
  (crate-source "xdg-home" "1.3.0"
                "1xm122zz0wjc8p8cmchij0j9nw34hwncb39jc7dc0mgvb2rdl77c"))

(define rust-xenstore-rs-0.6.0
  (crate-source "xenstore-rs" "0.6.0"
                "1i8wdif8q2cxbrmcp2jxcqg95a4yfsp1m3j4yldcgad3ksl5l4qs"))

(define rust-xenstore-sys-0.2.0
  (crate-source "xenstore-sys" "0.2.0"
                "1xrjy0d3cg9qrznfsfjrgha8si1dvdyfx3g7c76hbb7lnim5vmj7"))

(define rust-xi-unicode-0.3.0
  (crate-source "xi-unicode" "0.3.0"
                "12mvjgrhr7557cib69wm4q5s4srba27pg2df9l1zihrxgnbh0wx6"))

(define rust-xkbcommon-0.7.0
  (crate-source "xkbcommon" "0.7.0"
                "07n9shhcls66wjvmk5pzqql46ipfdv7b8hbc384wgv9hk4jpv1hk"))

(define rust-xkbcommon-0.8.0
  (crate-source "xkbcommon" "0.8.0"
                "1j8s1sfwc6bw9phsca65rw3q3b5l2651v1s0pk5yxm6baa9wlrld"))

(define rust-xkbcommon-0.9.0
  (crate-source "xkbcommon" "0.9.0"
                "0bd0qkapxsvblfw42x6ryhi50d63v55g40awf2alx8b0h3s79ad7"))

(define rust-xkbcommon-dl-0.4.2
  (crate-source "xkbcommon-dl" "0.4.2"
                "1iai0r3b5skd9vbr8z5b0qixiz8jblzfm778ddm8ba596a0dwffh"))

(define rust-xkeysym-0.2.1
  (crate-source "xkeysym" "0.2.1"
                "0mksx670cszyd7jln6s7dhkw11hdfv7blwwr3isq98k22ljh1k5r"))

(define rust-xmas-elf-0.9.1
  (crate-source "xmas-elf" "0.9.1"
                "1inias7h1cv4zh3szk46byiqhnzm5zc7658q1brzfhl3wwbrii22"))

(define rust-xml-rs-0.8.25
  (crate-source "xml-rs" "0.8.25"
                "1i73ajf6scni5bi1a51r19xykgrambdx5fkks0fyg5jqqbml1ff5"))

(define rust-xml-rs-0.8.26
  (crate-source "xml-rs" "0.8.26"
                "1nkbzksai36pvx3ag1fn9b5cffwb1ldl66aglwcip42nkdnyfb56"))

(define rust-xml-rs-0.8.27
  (crate-source "xml-rs" "0.8.27"
                "1irplg223x6w3lvj0yig6czbiwci06495wc9xg3660kh6cvl1n3g"))

(define rust-xml5ever-0.17.0
  (crate-source "xml5ever" "0.17.0"
                "0l76v0c228c92sskiflpsy19c0bgc8q7flhlfanm32zrbb8f2d20"))

(define rust-xml5ever-0.18.1
  (crate-source "xml5ever" "0.18.1"
                "0sdz92vrcxfwv7yzai28y0wa9gswr6msjnksak0rp4cfbm02dfwv"))

(define rust-xmlparser-0.13.6
  (crate-source "xmlparser" "0.13.6"
                "1r796g21c70p983ax0j6rmhzmalg4rhx61mvd4farxdhfyvy1zk6"))

(define rust-xmlwriter-0.1.0
  (crate-source "xmlwriter" "0.1.0"
                "1fg0ldmkgiis6hnxpi1c9gy7v23y0lpi824bp8yp12fi3r82lypc"))

(define rust-xmp-writer-0.3.2
  (crate-source "xmp-writer" "0.3.2"
                "1mrd5yswm9b1j84skzvz7nx20wlha17wyclq1bybz7jd8152z7nf"))

(define rust-xshell-0.2.7
  (crate-source "xshell" "0.2.7"
                "0g9pd9bfp0f35rzichic55k7p1mn8mqp607y5rimhiq14g390wly"))

(define rust-xshell-macros-0.2.7
  (crate-source "xshell-macros" "0.2.7"
                "0irm50jxdc92r0kd6yvl5p28jsfzha59brxk7z9w3jcf7z6h1b1j"))

(define rust-xterm-color-1.0.1
  (crate-source "xterm-color" "1.0.1"
                "03x8p1yav22y96c5mb5ahxqpcib1w927b1jlhy8bgj4xzdbg1rad"))

(define rust-xxhash-rust-0.8.15
  (crate-source "xxhash-rust" "0.8.15"
                "1lrmffpn45d967afw7f1p300rsx7ill66irrskxpcm1p41a0rlpx"))

(define rust-xz2-0.1.7
  (crate-source "xz2" "0.1.7"
                "1qk7nzpblizvayyq4xzi4b0zacmmbqr6vb9fc0v1avyp17f4931q"))

(define rust-y4m-0.8.0
  (crate-source "y4m" "0.8.0"
                "0j24y2zf60lpxwd7kyg737hqfyqx16y32s0fjyi6fax6w4hlnnks"
                #:snippet '(delete-file-recursively "scripts")))

(define rust-yada-0.5.1
  (crate-source "yada" "0.5.1"
                "1pgzmm965f5396q1mj5rfbxmmd7hmnynr435hx8h5a28ksyi3ldf"))

(define rust-yaml-rust-0.4.5
  (crate-source "yaml-rust" "0.4.5"
                "118wbqrr4n6wgk5rjjnlrdlahawlxc1bdsx146mwk8f79in97han"))

(define rust-yaml-rust2-0.8.1
  (crate-source "yaml-rust2" "0.8.1"
                "1y0mf6n1jnf88xqfv5ppicnm5jg3fl57dmp9vd2v2bvg9q61c0l9"))

(define rust-yaml-rust2-0.9.0
  (crate-source "yaml-rust2" "0.9.0"
                "13ay8jdlk1192p6hhb53dx601b139ycizipqpnikhcw2r45iq6ia"))

(define rust-yansi-1.0.1
  (crate-source "yansi" "1.0.1"
                "0jdh55jyv0dpd38ij4qh60zglbw9aa8wafqai6m0wa7xaxk3mrfg"
                #:snippet '(delete-file-recursively ".github")))

(define rust-yansi-term-0.1.2
  (crate-source "yansi-term" "0.1.2"
                "1w8vjlvxba6yvidqdvxddx3crl6z66h39qxj8xi6aqayw2nk0p7y"))

(define rust-yasna-0.4.0
  (crate-source "yasna" "0.4.0"
                "0xgsvxqnycdakh6j3hg4dk3mylrpnba50w0d36vg5k311sfs4qp2"))

(define rust-yasna-0.5.2
  (crate-source "yasna" "0.5.2"
                "1ka4ixrplnrfqyl1kymdj8cwpdp2k0kdr73b57hilcn1kiab6yz1"))

(define rust-yeslogic-fontconfig-sys-5.0.0
  (crate-source "yeslogic-fontconfig-sys" "5.0.0"
                "0yiwnf2gapqaprp3icvv6b1jjh5d356vpis7pybskcd8k4wv5dpz"))

(define rust-yoke-0.7.4
  (crate-source "yoke" "0.7.4"
                "198c4jkh6i3hxijia7mfa4cpnxg1iqym9bz364697c3rn0a16nvc"))

(define rust-yoke-0.7.5
  (crate-source "yoke" "0.7.5"
                "0h3znzrdmll0a7sglzf9ji0p5iqml11wrj1dypaf6ad6kbpnl3hj"))

(define rust-yoke-0.8.0
  (crate-source "yoke" "0.8.0"
                "1k4mfr48vgi7wh066y11b7v1ilakghlnlhw9snzz8vi2p00vnhaz"))

(define rust-yoke-derive-0.7.4
  (crate-source "yoke-derive" "0.7.4"
                "15cvhkci2mchfffx3fmva84fpmp34dsmnbzibwfnzjqq3ds33k18"))

(define rust-yoke-derive-0.7.5
  (crate-source "yoke-derive" "0.7.5"
                "0m4i4a7gy826bfvnqa9wy6sp90qf0as3wps3wb0smjaamn68g013"))

(define rust-yoke-derive-0.8.0
  (crate-source "yoke-derive" "0.8.0"
                "1dha5jrjz9jaq8kmxq1aag86b98zbnm9lyjrihy5sv716sbkrniq"))

(define rust-z-base-32-0.1.4
  (crate-source "z-base-32" "0.1.4"
                "0fqk028mzvgnx9phqzvhzrx3fbzv4r736cm3x3hid136g157pgr1"))

(define rust-zbar-rust-0.0.24
  (crate-source "zbar-rust" "0.0.24"
                "09j65i3ic19j5mcjvwdzxqqylhx2gg4aaxhscfdblzxk2k6bmxzq"))

(define rust-zbus-1.9.3
  (crate-source "zbus" "1.9.3"
                "0jgwydwjgk16dyrzdbc1k0dnqj9kv9p3fwcv92a7l9np3hlv5glw"))

(define rust-zbus-3.15.2
  (crate-source "zbus" "3.15.2"
                "1ri5gklhh3kl9gywym95679xs7n3sw2j3ky80jcd8siacc5ifpb7"))

(define rust-zbus-4.4.0
  (crate-source "zbus" "4.4.0"
                "09f7916lp7haxv1y5zgcg99ny15whi6dn3waf1afcafxx8mh35xv"))

(define rust-zbus-5.10.0
  (crate-source "zbus" "5.10.0"
                "1gj3y25qzvmhlshi8bipvwbrik8mh0f714zmib2avqdck6z77837"))

(define rust-zbus-5.5.0
  (crate-source "zbus" "5.5.0"
                "0dmjaih7gi2d0fa37zzylvbmxqn80x4d7haxr5xn86za93v37hsr"))

(define rust-zbus-5.7.0
  (crate-source "zbus" "5.7.0"
                "06b7xbkyzm9hmpjxb8kxw45rcsbml7lbmh9f8y2hqyh5p9s2n8w8"))

(define rust-zbus-5.7.1
  (crate-source "zbus" "5.7.1"
                "0s3gvx605m4cf73ppfknrbj7j33mrf1ag3zllgy49l0kwg7cg9yk"))

(define rust-zbus-lockstep-0.5.1
  (crate-source "zbus-lockstep" "0.5.1"
                "1dj5h31mwnsy5hhwvdl9fi7fy2npi2w8ra3b1fwyl3nkvqw6xs99"))

(define rust-zbus-lockstep-macros-0.5.1
  (crate-source "zbus-lockstep-macros" "0.5.1"
                "0dmqq2zr40hv6qkg88nwla2vrjk830jgdfnbnh4bfh583y2j2s6w"))

(define rust-zbus-macros-1.9.3
  (crate-source "zbus_macros" "1.9.3"
                "19p0pdwdf52zkaknav0pj5qvgcf52xk8a4p3a4ymxybwhjkmjfgs"))

(define rust-zbus-macros-3.15.2
  (crate-source "zbus_macros" "3.15.2"
                "19g0d7d4b8l8ycw498sz8pwkplv300j31i9hnihq0zl81xxljcbi"))

(define rust-zbus-macros-4.4.0
  (crate-source "zbus_macros" "4.4.0"
                "0glqn6ddgv4ra734p343a41rrxb0phy1v13dljzhpsc1f10bjz96"))

(define rust-zbus-macros-5.10.0
  (crate-source "zbus_macros" "5.10.0"
                "0cdid1y6kraqm53hshmna4l4ypr6cczhaj5nvicri92579qwv00f"))

(define rust-zbus-macros-5.5.0
  (crate-source "zbus_macros" "5.5.0"
                "1h4zf0wh647fvv97bnsr3ah64cgcnz1r8d10c2q3w2hdxc8as9gk"))

(define rust-zbus-macros-5.7.0
  (crate-source "zbus_macros" "5.7.0"
                "1jallnh50kn5np2scwp49q1wyf85fisidnk0drkk8cr3k5lc0sb9"))

(define rust-zbus-macros-5.7.1
  (crate-source "zbus_macros" "5.7.1"
                "0sinpb3vs672qhwmzagnl0xq30ws3bw8s18swx3zfl0mxig7wzm1"))

(define rust-zbus-names-2.6.1
  (crate-source "zbus_names" "2.6.1"
                "13achs6jbrp4l0jy5m6nn7v89clfgb63qhldkg5ddgjh6y6p6za3"))

(define rust-zbus-names-3.0.0
  (crate-source "zbus_names" "3.0.0"
                "0v1f0ajwafj47bf11yp0xdgp26r93lslr9nb2v6624h2gppiz6sb"))

(define rust-zbus-names-4.2.0
  (crate-source "zbus_names" "4.2.0"
                "15ybdd6zic7simi9wjg0ywrxfq4sxg3z0wiyysadps3cpxj8xrkv"))

(define rust-zbus-xml-5.0.2
  (crate-source "zbus_xml" "5.0.2"
                "0agcgcvp063bwwvrcgf6hhk7g2gk70xrw2iln95pbfdgpw19m7jq"))

(define rust-zero-0.1.3
  (crate-source "zero" "0.1.3"
                "113pa9jj40x6bvxsw582ca9np7d53qkb2b6cavfyczya6k61pqig"))

(define rust-zerocopy-0.7.35
  (crate-source "zerocopy" "0.7.35"
                "1w36q7b9il2flg0qskapgi9ymgg7p985vniqd09vi0mwib8lz6qv"))

(define rust-zerocopy-0.8.24
  (crate-source "zerocopy" "0.8.24"
                "0yb8hyzfnwzr2wg4p7cnqmjps8fsw8xqnprafgpmfs8qisigx1i5"))

(define rust-zerocopy-0.8.25
  (crate-source "zerocopy" "0.8.25"
                "1jx07cd3b3456c9al9zjqqdzpf1abb0vf6z0fj8xnb93hfajsw51"))

(define rust-zerocopy-0.8.26
  (crate-source "zerocopy" "0.8.26"
                "0bvsj0qzq26zc6nlrm3z10ihvjspyngs7n0jw1fz031i7h6xsf8h"))

(define rust-zerocopy-0.8.27
  (crate-source "zerocopy" "0.8.27"
                "0b1870gf2zzlckca69v2k4mqwmf8yh2li37qldnzvvd3by58g508"))

(define rust-zerocopy-derive-0.7.35
  (crate-source "zerocopy-derive" "0.7.35"
                "0gnf2ap2y92nwdalzz3x7142f2b83sni66l39vxp2ijd6j080kzs"))

(define rust-zerocopy-derive-0.8.24
  (crate-source "zerocopy-derive" "0.8.24"
                "1gk9047pbq1yjj2jyiv0s37nqc53maqbmhcsjp6lhi2w7kvai5m9"))

(define rust-zerocopy-derive-0.8.25
  (crate-source "zerocopy-derive" "0.8.25"
                "1vsmpq0hp61xpqj9yk8b5jihkqkff05q1wv3l2568mhifl6y59i8"))

(define rust-zerocopy-derive-0.8.26
  (crate-source "zerocopy-derive" "0.8.26"
                "10aiywi5qkha0mpsnb1zjwi44wl2rhdncaf3ykbp4i9nqm65pkwy"))

(define rust-zerocopy-derive-0.8.27
  (crate-source "zerocopy-derive" "0.8.27"
                "0c9qrylm2p55dvaplxsl24ma48add9qk4y0d6kjbkllaqvcvill8"))

(define rust-zerofrom-0.1.4
  (crate-source "zerofrom" "0.1.4"
                "0mdbjd7vmbix2ynxbrbrrli47a5yrpfx05hi99wf1l4pwwf13v4i"))

(define rust-zerofrom-0.1.6
  (crate-source "zerofrom" "0.1.6"
                "19dyky67zkjichsb7ykhv0aqws3q0jfvzww76l66c19y6gh45k2h"))

(define rust-zerofrom-derive-0.1.4
  (crate-source "zerofrom-derive" "0.1.4"
                "19b31rrs2ry1lrq5mpdqjzgg65va51fgvwghxnf6da3ycfiv99qf"))

(define rust-zerofrom-derive-0.1.6
  (crate-source "zerofrom-derive" "0.1.6"
                "00l5niw7c1b0lf1vhvajpjmcnbdp2vn96jg4nmkhq2db0rp5s7np"))

(define rust-zeroize-1.3.0
  (crate-source "zeroize" "1.3.0"
                "1z8yix823b6lz878qwg6bvwhg3lb0cbw3c9yij9p8mbv7zdzfmj7"))

(define rust-zeroize-1.8.1
  (crate-source "zeroize" "1.8.1"
                "1pjdrmjwmszpxfd7r860jx54cyk94qk59x13sc307cvr5256glyf"))

(define rust-zeroize-derive-1.4.2
  (crate-source "zeroize_derive" "1.4.2"
                "0sczjlqjdmrp3wn62g7mw6p438c9j4jgp2f9zamd56991mdycdnf"))

(define rust-zerotrie-0.1.3
  (crate-source "zerotrie" "0.1.3"
                "07qa5ljss8j706iy0rd023naamwly4jfwz0pc1gmqcw7bpalsngv"))

(define rust-zerotrie-0.2.2
  (crate-source "zerotrie" "0.2.2"
                "15gmka7vw5k0d24s0vxgymr2j6zn2iwl12wpmpnpjgsqg3abpw1n"))

(define rust-zerovec-0.10.4
  (crate-source "zerovec" "0.10.4"
                "0yghix7n3fjfdppwghknzvx9v8cf826h2qal5nqvy8yzg4yqjaxa"))

(define rust-zerovec-0.11.2
  (crate-source "zerovec" "0.11.2"
                "0a2457fmz39k9vrrj3rm82q5ykdhgxgbwfz2r6fa6nq11q4fn1aa"))

(define rust-zerovec-0.11.4
  (crate-source "zerovec" "0.11.4"
                "0fz7j1ns8d86m2fqg2a4bzi5gnh5892bxv4kcr9apwc6a3ajpap7"))

(define rust-zerovec-derive-0.10.3
  (crate-source "zerovec-derive" "0.10.3"
                "1ik322dys6wnap5d3gcsn09azmssq466xryn5czfm13mn7gsdbvf"))

(define rust-zerovec-derive-0.11.1
  (crate-source "zerovec-derive" "0.11.1"
                "13zms8hj7vzpfswypwggyfr4ckmyc7v3di49pmj8r1qcz9z275jv"))

(define rust-zip-0.6.6
  (crate-source "zip" "0.6.6"
                "0qcjbqfvbwxi5g9wbymf2r05cvziic2qqj4xy64q3hp48vi980vn"
                #:snippet '(delete-file-recursively "tests")))

(define rust-zip-2.4.2
  (crate-source "zip" "2.4.2"
                "0l5s7sycj0w42hnkmj6vnrzcp3dn4smffg06a8f5my08x4j67gps"))

(define rust-zip-2.5.0
  (crate-source "zip" "2.5.0"
                "120zjj8rg5fzmvrb1lmznljmkxlcvi7lnmrpdwzy4r2g8qbkih17"))

(define rust-zip-2.6.1
  (crate-source "zip" "2.6.1"
                "0i276d0kracqv27f5r42p3ha2345f77isv5rp54sw9i52p829jqx"))

(define rust-zipsign-api-0.1.5
  (crate-source "zipsign-api" "0.1.5"
                "1h20rb71gcidgjclw19nz0hfb0dyk8v6vlddcnkxknrcz0zhd9nv"))

(define rust-zlib-rs-0.5.0
  (crate-source "zlib-rs" "0.5.0"
                "1a1vssif5m2hwsy574l1gb668q4k04ggqv88yvr9mq29g66r52w6"
                #:snippet '(delete-file-recursively "src/deflate/test-data")))

(define rust-zlib-rs-0.5.1
  (crate-source "zlib-rs" "0.5.1"
                "12nvshiq19nd4ksn3453ym9p0kcqf0hpaq301p2iyx9ljzxdjsv2"
                #:snippet '(delete-file-recursively "src/deflate/test-data")))

(define rust-zoneinfo-compiled-0.5.1
  (crate-source "zoneinfo_compiled" "0.5.1"
                "1pm50w4vv34r08mrajfvyhc1254gv8zv4q6p7gs315c9bvkfpyv4"))

(define rust-zopfli-0.8.1
  (crate-source "zopfli" "0.8.1"
                "0ip9azz9ldk19m0m1hdppz3n5zcz0cywbg1vx59g4p5c3cwry0g5"))

(define rust-zopfli-0.8.2
  (crate-source "zopfli" "0.8.2"
                "1mqc2np2l47l9r1qkdm41ykgsp6m096z3ipchi4ws17m0pj5xz7d"))

(define rust-zstd-0.11.2+zstd.1.5.2
  (crate-source "zstd" "0.11.2+zstd.1.5.2"
                "1r7xlmgnifhxbfyid8vkcnd5ip16gx9hf89d1l0lzrpc4q1rdk10"))

(define rust-zstd-0.13.3
  (crate-source "zstd" "0.13.3"
                "12n0h4w9l526li7jl972rxpyf012jw3nwmji2qbjghv9ll8y67p9"))

(define rust-zstd-safe-5.0.2+zstd.1.5.2
  (crate-source "zstd-safe" "5.0.2+zstd.1.5.2"
                "1nzl4q3xl68pq58g9xlym299bvjdii8cl7ix595ym7jgw22maahx"))

(define rust-zstd-safe-7.2.3
  (crate-source "zstd-safe" "7.2.3"
                "08i7h3nsakflxb8cpygkjc74gn3k1z32ip2468a1wbnwzf91f1gk"))

(define rust-zstd-safe-7.2.4
  (crate-source "zstd-safe" "7.2.4"
                "179vxmkzhpz6cq6mfzvgwc99bpgllkr6lwxq7ylh5dmby3aw8jcg"))

(define rust-zstd-seekable-0.1.23
  ;; TODO: Submit patches to use system zstd-seekable.
  (crate-source "zstd-seekable" "0.1.23"
                "1hn3lry2p1fzjlx6shwq1k5bcpx4lqckmvl16gqx326vbiy12jjp"))

(define rust-zstd-sys-2.0.14+zstd.1.5.7
  (crate-source "zstd-sys" "2.0.14+zstd.1.5.7"
                "1r8v64bx8vmfh5nvvx9ldcy38pzbkkln9n0mmniw6jkfjba61c4g"
                #:snippet '(delete-file-recursively "zstd")))

(define rust-zstd-sys-2.0.15+zstd.1.5.7
  (crate-source "zstd-sys" "2.0.15+zstd.1.5.7"
                "0dx2l7dyw1p7x7g3p1pfd25ip36hr22hvmgixm6cgl4pvlyii0gb"
                #:snippet '(delete-file-recursively "zstd")))

(define rust-zstd-sys-2.0.16+zstd.1.5.7
  (crate-source "zstd-sys" "2.0.16+zstd.1.5.7"
                "0j1pd2iaqpvaxlgqmmijj68wma7xwdv9grrr63j873yw5ay9xqci"
                #:snippet '(delete-file-recursively "zstd")))

(define rust-zune-core-0.4.12
  (crate-source "zune-core" "0.4.12"
                "0jj1ra86klzlcj9aha9als9d1dzs7pqv3azs1j3n96822wn3lhiz"))

(define rust-zune-inflate-0.2.54
  (crate-source "zune-inflate" "0.2.54"
                "00kg24jh3zqa3i6rg6yksnb71bch9yi1casqydl00s7nw8pk7avk"))

(define rust-zune-jpeg-0.4.14
  (crate-source "zune-jpeg" "0.4.14"
                "0a70sbnxxkgfm777i1xjkhyn8mx07swg5cabbi083pyysywbm9cr"))

(define rust-zune-jpeg-0.4.16
  (crate-source "zune-jpeg" "0.4.16"
                "0x1rrmgy1hh4kv7w4xydjy94igm7cxzrsd3nm16nymx21s652jiy"))

(define rust-zune-jpeg-0.4.17
  (crate-source "zune-jpeg" "0.4.17"
                "1b68jjrjh4byfs1zjpygwm1rrs9xvwb2x0138kk8xa827piy4vqg"))

(define rust-zune-jpeg-0.4.19
  (crate-source "zune-jpeg" "0.4.19"
                "0sisqjhwi6pwlnjyr80vl5s6p5w9z2vi8pz967h5i9m6y1d557ic"))

(define rust-zune-jpeg-0.4.20
  (crate-source "zune-jpeg" "0.4.20"
                "12fhlnyr254d3hys1xkrgk5qb5sqydamy76p7kdb57p7bhh7w7zw"))

(define rust-zvariant-2.10.0
  (crate-source "zvariant" "2.10.0"
                "0995d59vl8409mk3qrbshqrz5d76dq52szg0x2vqji07y9app356"))

(define rust-zvariant-3.15.2
  (crate-source "zvariant" "3.15.2"
                "1nxj9x187jl32fd32zvq8hfn6lyq3kjadb2q7f6kb6x0igl2pvsf"))

(define rust-zvariant-4.2.0
  (crate-source "zvariant" "4.2.0"
                "1zl1ika7zd9bxkd0bqc78h9bykvk6xc98965iz1p3i51p452k110"))

(define rust-zvariant-5.4.0
  (crate-source "zvariant" "5.4.0"
                "1b53qpb3q7j233512s2684iy7wyydra31pi5vkxwygw98kh9xpxj"))

(define rust-zvariant-5.5.3
  (crate-source "zvariant" "5.5.3"
                "1cgzkngj9nsmz5ql9fxiqm98s0sc9l3zkr0xl9iyx4z3fmpphc4x"))

(define rust-zvariant-5.7.0
  (crate-source "zvariant" "5.7.0"
                "1nwqji0y214dnchq372rsfqzl86d9wgai909s761yay5ffzd77cr"))

(define rust-zvariant-derive-2.10.0
  (crate-source "zvariant_derive" "2.10.0"
                "1s9xk9c4p9vl0j2vr1abqc12mgv500sjc3fnh8ij3d1yb4i5xjp4"))

(define rust-zvariant-derive-3.15.2
  (crate-source "zvariant_derive" "3.15.2"
                "1nbydrkawjwxan12vy79qsrn7gwc483mpfzqs685ybyppv04vhip"))

(define rust-zvariant-derive-4.2.0
  (crate-source "zvariant_derive" "4.2.0"
                "0jf408h0s83krxwm7wl62fnssin1kcklmb1bcn83ls6sddabmqkk"))

(define rust-zvariant-derive-5.4.0
  (crate-source "zvariant_derive" "5.4.0"
                "0bsbz68dsvkynnnpxpfchmhyl5bsgjjmcbazjg24rf5qhnm0q5vl"))

(define rust-zvariant-derive-5.5.3
  (crate-source "zvariant_derive" "5.5.3"
                "104m9pp4821mz2q6rgqnjq79q89j0iiv25w1sk63bms2rl1agzbm"))

(define rust-zvariant-derive-5.7.0
  (crate-source "zvariant_derive" "5.7.0"
                "13mnhlw8imn6garkqrq4gyxyj7rjnp0hfgqdv5mj4vd44q5zshv6"))

(define rust-zvariant-utils-1.0.1
  (crate-source "zvariant_utils" "1.0.1"
                "00625h3240rixvfhq6yhws1d4bwf3vrf74v8s69b97aq27cg0d3j"))

(define rust-zvariant-utils-2.1.0
  (crate-source "zvariant_utils" "2.1.0"
                "0h43h3jcw8rmjr390rdqnhkb9nn3913pgkvb75am1frxrkvwy6y5"))

(define rust-zvariant-utils-3.2.0
  (crate-source "zvariant_utils" "3.2.0"
                "0d7wllndiv7vwgapmji3q9sxmzbaqfdxjwkqnx9vbmz58gpdyvp1"))

(define rust-zvariant-utils-3.2.1
  (crate-source "zvariant_utils" "3.2.1"
                "16g5id3h9q85c5vcwdfwkwmwzyladbr2q8x2xinr3xl95wa9v566"))

(define ssss-separator 'end-of-crates)


;;;
;;; Cargo inputs.
;;;

;; TODO: Maintain this automatically too.
(define-public mesa-cargo-inputs
  `(("paste" ,rust-paste-1.0.15)
    ("proc-macro2" ,rust-proc-macro2-1.0.94)
    ("quote" ,rust-quote-1.0.40)
    ("rustc-hash" ,rust-rustc-hash-2.1.1)
    ("syn" ,rust-syn-2.0.100)
    ("unicode-ident" ,rust-unicode-ident-1.0.18)))

(define-cargo-inputs lookup-cargo-inputs
                     (aardvark-dns =>
                                   (list rust-addr2line-0.24.2
                                    rust-adler2-2.0.0
                                    rust-android-tzdata-0.1.1
                                    rust-android-system-properties-0.1.5
                                    rust-anstream-0.6.18
                                    rust-anstyle-1.0.10
                                    rust-anstyle-parse-0.2.6
                                    rust-anstyle-query-1.1.2
                                    rust-anstyle-wincon-3.0.7
                                    rust-arc-swap-1.7.1
                                    rust-async-trait-0.1.88
                                    rust-autocfg-1.4.0
                                    rust-backtrace-0.3.74
                                    rust-bitflags-2.9.0
                                    rust-bumpalo-3.17.0
                                    rust-bytes-1.10.1
                                    rust-cc-1.2.19
                                    rust-cfg-if-1.0.0
                                    rust-cfg-aliases-0.2.1
                                    rust-chrono-0.4.40
                                    rust-clap-4.5.36
                                    rust-clap-builder-4.5.36
                                    rust-clap-derive-4.5.32
                                    rust-clap-lex-0.7.4
                                    rust-colorchoice-1.0.3
                                    rust-core-foundation-sys-0.8.7
                                    rust-data-encoding-2.8.0
                                    rust-deranged-0.4.0
                                    rust-displaydoc-0.2.5
                                    rust-endian-type-0.1.2
                                    rust-enum-as-inner-0.6.1
                                    rust-flume-0.11.1
                                    rust-form-urlencoded-1.2.1
                                    rust-futures-channel-0.3.31
                                    rust-futures-core-0.3.31
                                    rust-futures-io-0.3.31
                                    rust-futures-sink-0.3.31
                                    rust-futures-task-0.3.31
                                    rust-futures-util-0.3.31
                                    rust-getrandom-0.2.15
                                    rust-gimli-0.31.1
                                    rust-heck-0.5.0
                                    rust-hickory-client-0.24.4
                                    rust-hickory-proto-0.24.4
                                    rust-hickory-server-0.24.4
                                    rust-hostname-0.4.1
                                    rust-iana-time-zone-0.1.63
                                    rust-iana-time-zone-haiku-0.1.2
                                    rust-icu-collections-1.5.0
                                    rust-icu-locid-1.5.0
                                    rust-icu-locid-transform-1.5.0
                                    rust-icu-locid-transform-data-1.5.1
                                    rust-icu-normalizer-1.5.0
                                    rust-icu-normalizer-data-1.5.1
                                    rust-icu-properties-1.5.1
                                    rust-icu-properties-data-1.5.1
                                    rust-icu-provider-1.5.0
                                    rust-icu-provider-macros-1.5.0
                                    rust-idna-1.0.3
                                    rust-idna-adapter-1.2.0
                                    rust-ipnet-2.11.0
                                    rust-is-terminal-polyfill-1.70.1
                                    rust-itoa-1.0.15
                                    rust-js-sys-0.3.77
                                    rust-libc-0.2.171
                                    rust-litemap-0.7.5
                                    rust-lock-api-0.4.12
                                    rust-log-0.4.27
                                    rust-memchr-2.7.4
                                    rust-memoffset-0.9.1
                                    rust-miniz-oxide-0.8.8
                                    rust-mio-1.0.3
                                    rust-nanorand-0.7.0
                                    rust-nibble-vec-0.1.0
                                    rust-nix-0.29.0
                                    rust-num-conv-0.1.0
                                    rust-num-traits-0.2.19
                                    rust-num-threads-0.1.7
                                    rust-object-0.36.7
                                    rust-once-cell-1.21.3
                                    rust-percent-encoding-2.3.1
                                    rust-pin-project-lite-0.2.16
                                    rust-pin-utils-0.1.0
                                    rust-powerfmt-0.2.0
                                    rust-ppv-lite86-0.2.21
                                    rust-proc-macro2-1.0.94
                                    rust-quote-1.0.40
                                    rust-radix-trie-0.2.1
                                    rust-rand-0.8.5
                                    rust-rand-chacha-0.3.1
                                    rust-rand-core-0.6.4
                                    rust-rustc-demangle-0.1.24
                                    rust-rustversion-1.0.20
                                    rust-scopeguard-1.2.0
                                    rust-serde-1.0.219
                                    rust-serde-derive-1.0.219
                                    rust-shlex-1.3.0
                                    rust-signal-hook-registry-1.4.2
                                    rust-slab-0.4.9
                                    rust-smallvec-1.15.0
                                    rust-socket2-0.5.9
                                    rust-spin-0.9.8
                                    rust-stable-deref-trait-1.2.0
                                    rust-strsim-0.11.1
                                    rust-syn-2.0.100
                                    rust-synstructure-0.13.1
                                    rust-syslog-7.0.0
                                    rust-thiserror-1.0.69
                                    rust-thiserror-impl-1.0.69
                                    rust-time-0.3.41
                                    rust-time-core-0.1.4
                                    rust-time-macros-0.2.22
                                    rust-tinystr-0.7.6
                                    rust-tinyvec-1.9.0
                                    rust-tinyvec-macros-0.1.1
                                    rust-tokio-1.44.2
                                    rust-tokio-macros-2.5.0
                                    rust-tokio-util-0.7.14
                                    rust-tracing-0.1.41
                                    rust-tracing-attributes-0.1.28
                                    rust-tracing-core-0.1.33
                                    rust-unicode-ident-1.0.18
                                    rust-url-2.5.4
                                    rust-utf16-iter-1.0.5
                                    rust-utf8-iter-1.0.4
                                    rust-utf8parse-0.2.2
                                    rust-wasi-0.11.0+wasi-snapshot-preview1
                                    rust-wasm-bindgen-0.2.100
                                    rust-wasm-bindgen-backend-0.2.100
                                    rust-wasm-bindgen-macro-0.2.100
                                    rust-wasm-bindgen-macro-support-0.2.100
                                    rust-wasm-bindgen-shared-0.2.100
                                    rust-windows-core-0.61.0
                                    rust-windows-implement-0.60.0
                                    rust-windows-interface-0.59.1
                                    rust-windows-link-0.1.1
                                    rust-windows-result-0.3.2
                                    rust-windows-strings-0.4.0
                                    rust-windows-sys-0.52.0
                                    rust-windows-sys-0.59.0
                                    rust-windows-targets-0.52.6
                                    rust-windows-aarch64-gnullvm-0.52.6
                                    rust-windows-aarch64-msvc-0.52.6
                                    rust-windows-i686-gnu-0.52.6
                                    rust-windows-i686-gnullvm-0.52.6
                                    rust-windows-i686-msvc-0.52.6
                                    rust-windows-x86-64-gnu-0.52.6
                                    rust-windows-x86-64-gnullvm-0.52.6
                                    rust-windows-x86-64-msvc-0.52.6
                                    rust-write16-1.0.0
                                    rust-writeable-0.5.5
                                    rust-yoke-0.7.5
                                    rust-yoke-derive-0.7.5
                                    rust-zerocopy-0.8.24
                                    rust-zerocopy-derive-0.8.24
                                    rust-zerofrom-0.1.6
                                    rust-zerofrom-derive-0.1.6
                                    rust-zerovec-0.10.4
                                    rust-zerovec-derive-0.10.3))
                     (agate =>
                            (list rust-addr2line-0.24.2
                                  rust-adler2-2.0.0
                                  rust-anstream-0.6.18
                                  rust-anstyle-1.0.10
                                  rust-anstyle-parse-0.2.6
                                  rust-anstyle-query-1.1.2
                                  rust-anstyle-wincon-3.0.7
                                  rust-autocfg-1.4.0
                                  rust-backtrace-0.3.74
                                  rust-bitflags-2.9.0
                                  rust-bytes-1.10.1
                                  rust-cc-1.2.19
                                  rust-cfg-if-1.0.0
                                  rust-colorchoice-1.0.3
                                  rust-configparser-3.1.0
                                  rust-deranged-0.4.0
                                  rust-displaydoc-0.2.5
                                  rust-env-filter-0.1.3
                                  rust-env-logger-0.11.8
                                  rust-foreign-types-0.3.2
                                  rust-foreign-types-shared-0.1.1
                                  rust-form-urlencoded-1.2.1
                                  rust-futures-core-0.3.31
                                  rust-futures-macro-0.3.31
                                  rust-futures-task-0.3.31
                                  rust-futures-util-0.3.31
                                  rust-getopts-0.2.21
                                  rust-getrandom-0.2.15
                                  rust-gimli-0.31.1
                                  rust-glob-0.3.2
                                  rust-icu-collections-1.5.0
                                  rust-icu-locid-1.5.0
                                  rust-icu-locid-transform-1.5.0
                                  rust-icu-locid-transform-data-1.5.1
                                  rust-icu-normalizer-1.5.0
                                  rust-icu-normalizer-data-1.5.1
                                  rust-icu-properties-1.5.1
                                  rust-icu-properties-data-1.5.1
                                  rust-icu-provider-1.5.0
                                  rust-icu-provider-macros-1.5.0
                                  rust-idna-1.0.3
                                  rust-idna-adapter-1.2.0
                                  rust-is-terminal-polyfill-1.70.1
                                  rust-jiff-0.2.6
                                  rust-jiff-static-0.2.6
                                  rust-libc-0.2.171
                                  rust-litemap-0.7.5
                                  rust-log-0.4.27
                                  rust-memchr-2.7.4
                                  rust-mime-0.3.17
                                  rust-mime-guess-2.0.5
                                  rust-miniz-oxide-0.8.8
                                  rust-mio-1.0.3
                                  rust-num-conv-0.1.0
                                  rust-object-0.36.7
                                  rust-once-cell-1.21.3
                                  rust-openssl-0.10.72
                                  rust-openssl-macros-0.1.1
                                  rust-openssl-sys-0.9.107
                                  rust-percent-encoding-2.3.1
                                  rust-pin-project-lite-0.2.16
                                  rust-pin-utils-0.1.0
                                  rust-pkg-config-0.3.32
                                  rust-portable-atomic-1.11.0
                                  rust-portable-atomic-util-0.2.4
                                  rust-powerfmt-0.2.0
                                  rust-proc-macro2-1.0.94
                                  rust-quote-1.0.40
                                  rust-rcgen-0.13.2
                                  rust-ring-0.17.14
                                  rust-rustc-demangle-0.1.24
                                  rust-rustls-0.23.26
                                  rust-rustls-pki-types-1.11.0
                                  rust-rustls-webpki-0.103.1
                                  rust-serde-1.0.219
                                  rust-serde-derive-1.0.219
                                  rust-shlex-1.3.0
                                  rust-slab-0.4.9
                                  rust-smallvec-1.15.0
                                  rust-socket2-0.5.9
                                  rust-stable-deref-trait-1.2.0
                                  rust-subtle-2.6.1
                                  rust-syn-2.0.100
                                  rust-synstructure-0.13.1
                                  rust-thiserror-1.0.69
                                  rust-thiserror-impl-1.0.69
                                  rust-time-0.3.41
                                  rust-time-core-0.1.4
                                  rust-tinystr-0.7.6
                                  rust-tokio-1.44.2
                                  rust-tokio-macros-2.5.0
                                  rust-tokio-openssl-0.6.5
                                  rust-tokio-rustls-0.26.2
                                  rust-trotter-1.0.2
                                  rust-unicase-2.8.1
                                  rust-unicode-ident-1.0.18
                                  rust-unicode-width-0.1.14
                                  rust-untrusted-0.9.0
                                  rust-url-2.5.4
                                  rust-urlencoding-2.1.3
                                  rust-utf16-iter-1.0.5
                                  rust-utf8-iter-1.0.4
                                  rust-utf8parse-0.2.2
                                  rust-vcpkg-0.2.15
                                  rust-wasi-0.11.0+wasi-snapshot-preview1
                                  rust-wildmatch-2.4.0
                                  rust-windows-sys-0.52.0
                                  rust-windows-sys-0.59.0
                                  rust-windows-targets-0.52.6
                                  rust-windows-aarch64-gnullvm-0.52.6
                                  rust-windows-aarch64-msvc-0.52.6
                                  rust-windows-i686-gnu-0.52.6
                                  rust-windows-i686-gnullvm-0.52.6
                                  rust-windows-i686-msvc-0.52.6
                                  rust-windows-x86-64-gnu-0.52.6
                                  rust-windows-x86-64-gnullvm-0.52.6
                                  rust-windows-x86-64-msvc-0.52.6
                                  rust-write16-1.0.0
                                  rust-writeable-0.5.5
                                  rust-yasna-0.5.2
                                  rust-yoke-0.7.5
                                  rust-yoke-derive-0.7.5
                                  rust-zerofrom-0.1.6
                                  rust-zerofrom-derive-0.1.6
                                  rust-zeroize-1.8.1
                                  rust-zerovec-0.10.4
                                  rust-zerovec-derive-0.10.3))
                     (alacritty =>
                                (list rust-adler2-2.0.0
                                 rust-ahash-0.8.11
                                 rust-aho-corasick-1.1.3
                                 rust-android-activity-0.6.0
                                 rust-android-properties-0.2.2
                                 rust-anstream-0.6.18
                                 rust-anstyle-1.0.10
                                 rust-anstyle-parse-0.2.6
                                 rust-anstyle-query-1.1.2
                                 rust-anstyle-wincon-3.0.7
                                 rust-arrayref-0.3.9
                                 rust-arrayvec-0.7.6
                                 rust-as-raw-xcb-connection-1.0.1
                                 rust-atomic-waker-1.1.2
                                 rust-autocfg-1.4.0
                                 rust-base64-0.22.1
                                 rust-bitflags-1.3.2
                                 rust-bitflags-2.9.0
                                 rust-block-0.1.6
                                 rust-block2-0.5.1
                                 rust-bumpalo-3.17.0
                                 rust-bytemuck-1.22.0
                                 rust-bytes-1.10.1
                                 rust-calloop-0.13.0
                                 rust-calloop-wayland-source-0.3.0
                                 rust-cc-1.2.19
                                 rust-cesu8-1.1.0
                                 rust-cfg-if-1.0.0
                                 rust-cfg-aliases-0.2.1
                                 rust-cgl-0.3.2
                                 rust-clap-4.5.35
                                 rust-clap-builder-4.5.35
                                 rust-clap-complete-4.5.47
                                 rust-clap-derive-4.5.32
                                 rust-clap-lex-0.7.4
                                 rust-clipboard-win-3.1.1
                                 rust-cocoa-0.25.0
                                 rust-cocoa-foundation-0.1.2
                                 rust-colorchoice-1.0.3
                                 rust-combine-4.6.7
                                 rust-concurrent-queue-2.5.0
                                 rust-copypasta-0.10.1
                                 rust-core-foundation-0.9.4
                                 rust-core-foundation-sys-0.8.7
                                 rust-core-graphics-0.23.2
                                 rust-core-graphics-types-0.1.3
                                 rust-core-text-20.1.0
                                 rust-crc32fast-1.4.2
                                 rust-crossbeam-channel-0.5.15
                                 rust-crossbeam-utils-0.8.21
                                 rust-crossfont-0.8.0
                                 rust-cstr-0.2.12
                                 rust-cursor-icon-1.1.0
                                 rust-dirs-5.0.1
                                 rust-dirs-sys-0.4.1
                                 rust-dispatch-0.2.0
                                 rust-dlib-0.5.2
                                 rust-downcast-rs-1.2.1
                                 rust-dpi-0.1.1
                                 rust-dwrote-0.11.3
                                 rust-embed-resource-2.5.1
                                 rust-equivalent-1.0.2
                                 rust-errno-0.3.11
                                 rust-fastrand-2.3.0
                                 rust-fdeflate-0.3.7
                                 rust-filetime-0.2.25
                                 rust-flate2-1.1.1
                                 rust-foreign-types-0.5.0
                                 rust-foreign-types-macros-0.2.3
                                 rust-foreign-types-shared-0.3.1
                                 rust-freetype-rs-0.36.0
                                 rust-freetype-sys-0.20.1
                                 rust-fsevent-sys-4.1.0
                                 rust-futures-io-0.3.31
                                 rust-gethostname-0.4.3
                                 rust-getrandom-0.2.15
                                 rust-getrandom-0.3.2
                                 rust-gl-generator-0.14.0
                                 rust-glutin-0.32.2
                                 rust-glutin-egl-sys-0.7.1
                                 rust-glutin-glx-sys-0.6.1
                                 rust-glutin-wgl-sys-0.6.1
                                 rust-hashbrown-0.15.2
                                 rust-heck-0.5.0
                                 rust-hermit-abi-0.4.0
                                 rust-home-0.5.11
                                 rust-indexmap-2.9.0
                                 rust-inotify-0.9.6
                                 rust-inotify-sys-0.1.5
                                 rust-is-terminal-polyfill-1.70.1
                                 rust-itoa-1.0.15
                                 rust-jni-0.21.1
                                 rust-jni-sys-0.3.0
                                 rust-jobserver-0.1.33
                                 rust-js-sys-0.3.77
                                 rust-khronos-api-3.1.0
                                 rust-kqueue-1.0.8
                                 rust-kqueue-sys-1.0.4
                                 rust-lazy-bytes-cast-5.0.1
                                 rust-lazy-static-1.5.0
                                 rust-libc-0.2.171
                                 rust-libloading-0.8.6
                                 rust-libredox-0.1.3
                                 rust-linux-raw-sys-0.4.15
                                 rust-linux-raw-sys-0.9.4
                                 rust-lock-api-0.4.12
                                 rust-log-0.4.27
                                 rust-malloc-buf-0.0.6
                                 rust-memchr-2.7.4
                                 rust-memmap2-0.9.5
                                 rust-miniz-oxide-0.8.8
                                 rust-mio-0.8.11
                                 rust-miow-0.6.0
                                 rust-ndk-0.9.0
                                 rust-ndk-context-0.1.1
                                 rust-ndk-sys-0.6.0+11769913
                                 rust-notify-6.1.1
                                 rust-num-enum-0.7.3
                                 rust-num-enum-derive-0.7.3
                                 rust-objc-0.2.7
                                 rust-objc-foundation-0.1.1
                                 rust-objc-sys-0.3.5
                                 rust-objc2-0.5.2
                                 rust-objc2-app-kit-0.2.2
                                 rust-objc2-cloud-kit-0.2.2
                                 rust-objc2-contacts-0.2.2
                                 rust-objc2-core-data-0.2.2
                                 rust-objc2-core-image-0.2.2
                                 rust-objc2-core-location-0.2.2
                                 rust-objc2-encode-4.1.0
                                 rust-objc2-foundation-0.2.2
                                 rust-objc2-link-presentation-0.2.2
                                 rust-objc2-metal-0.2.2
                                 rust-objc2-quartz-core-0.2.2
                                 rust-objc2-symbols-0.2.2
                                 rust-objc2-ui-kit-0.2.2
                                 rust-objc2-uniform-type-identifiers-0.2.2
                                 rust-objc2-user-notifications-0.2.2
                                 rust-objc-id-0.1.1
                                 rust-once-cell-1.21.3
                                 rust-option-ext-0.2.0
                                 rust-orbclient-0.3.48
                                 rust-parking-lot-0.12.3
                                 rust-parking-lot-core-0.9.10
                                 rust-percent-encoding-2.3.1
                                 rust-pin-project-1.1.10
                                 rust-pin-project-internal-1.1.10
                                 rust-pin-project-lite-0.2.16
                                 rust-piper-0.2.4
                                 rust-pkg-config-0.3.32
                                 rust-png-0.17.16
                                 rust-polling-3.7.4
                                 rust-proc-macro-crate-3.3.0
                                 rust-proc-macro2-1.0.94
                                 rust-quick-xml-0.37.4
                                 rust-quote-1.0.40
                                 rust-r-efi-5.2.0
                                 rust-raw-window-handle-0.6.2
                                 rust-redox-syscall-0.4.1
                                 rust-redox-syscall-0.5.11
                                 rust-redox-users-0.4.6
                                 rust-regex-automata-0.4.9
                                 rust-regex-syntax-0.8.5
                                 rust-rustc-version-0.4.1
                                 rust-rustix-0.38.44
                                 rust-rustix-1.0.5
                                 rust-rustix-openpty-0.1.1
                                 rust-rustversion-1.0.20
                                 rust-ryu-1.0.20
                                 rust-same-file-1.0.6
                                 rust-scoped-tls-1.0.1
                                 rust-scopeguard-1.2.0
                                 rust-sctk-adwaita-0.10.1
                                 rust-semver-1.0.26
                                 rust-serde-1.0.219
                                 rust-serde-derive-1.0.219
                                 rust-serde-json-1.0.140
                                 rust-serde-spanned-0.6.8
                                 rust-serde-yaml-0.9.34+deprecated
                                 rust-shlex-1.3.0
                                 rust-signal-hook-0.3.17
                                 rust-signal-hook-registry-1.4.2
                                 rust-simd-adler32-0.3.7
                                 rust-slab-0.4.9
                                 rust-smallvec-1.15.0
                                 rust-smithay-client-toolkit-0.19.2
                                 rust-smithay-clipboard-0.7.2
                                 rust-smol-str-0.2.2
                                 rust-strict-num-0.1.1
                                 rust-strsim-0.11.1
                                 rust-syn-2.0.100
                                 rust-tempfile-3.19.1
                                 rust-thiserror-1.0.69
                                 rust-thiserror-impl-1.0.69
                                 rust-tiny-skia-0.11.4
                                 rust-tiny-skia-path-0.11.4
                                 rust-toml-0.8.20
                                 rust-toml-datetime-0.6.8
                                 rust-toml-edit-0.22.24
                                 rust-tracing-0.1.41
                                 rust-tracing-core-0.1.33
                                 rust-unicode-ident-1.0.18
                                 rust-unicode-segmentation-1.12.0
                                 rust-unicode-width-0.1.14
                                 rust-unsafe-libyaml-0.2.11
                                 rust-utf8parse-0.2.2
                                 rust-version-check-0.9.5
                                 rust-vswhom-0.1.0
                                 rust-vswhom-sys-0.1.3
                                 rust-vte-0.13.1
                                 rust-vte-generate-state-changes-0.1.2
                                 rust-walkdir-2.5.0
                                 rust-wasi-0.11.0+wasi-snapshot-preview1
                                 rust-wasi-0.14.2+wasi-0.2.4
                                 rust-wasm-bindgen-0.2.100
                                 rust-wasm-bindgen-backend-0.2.100
                                 rust-wasm-bindgen-futures-0.4.50
                                 rust-wasm-bindgen-macro-0.2.100
                                 rust-wasm-bindgen-macro-support-0.2.100
                                 rust-wasm-bindgen-shared-0.2.100
                                 rust-wayland-backend-0.3.8
                                 rust-wayland-client-0.31.8
                                 rust-wayland-csd-frame-0.3.0
                                 rust-wayland-cursor-0.31.8
                                 rust-wayland-protocols-0.32.6
                                 rust-wayland-protocols-plasma-0.3.6
                                 rust-wayland-protocols-wlr-0.3.6
                                 rust-wayland-scanner-0.31.6
                                 rust-wayland-sys-0.31.6
                                 rust-web-sys-0.3.77
                                 rust-web-time-1.1.0
                                 rust-winapi-0.3.9
                                 rust-winapi-i686-pc-windows-gnu-0.4.0
                                 rust-winapi-util-0.1.9
                                 rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                 rust-windows-sys-0.45.0
                                 rust-windows-sys-0.48.0
                                 rust-windows-sys-0.52.0
                                 rust-windows-sys-0.59.0
                                 rust-windows-targets-0.42.2
                                 rust-windows-targets-0.48.5
                                 rust-windows-targets-0.52.6
                                 rust-windows-aarch64-gnullvm-0.42.2
                                 rust-windows-aarch64-gnullvm-0.48.5
                                 rust-windows-aarch64-gnullvm-0.52.6
                                 rust-windows-aarch64-msvc-0.42.2
                                 rust-windows-aarch64-msvc-0.48.5
                                 rust-windows-aarch64-msvc-0.52.6
                                 rust-windows-i686-gnu-0.42.2
                                 rust-windows-i686-gnu-0.48.5
                                 rust-windows-i686-gnu-0.52.6
                                 rust-windows-i686-gnullvm-0.52.6
                                 rust-windows-i686-msvc-0.42.2
                                 rust-windows-i686-msvc-0.48.5
                                 rust-windows-i686-msvc-0.52.6
                                 rust-windows-x86-64-gnu-0.42.2
                                 rust-windows-x86-64-gnu-0.48.5
                                 rust-windows-x86-64-gnu-0.52.6
                                 rust-windows-x86-64-gnullvm-0.42.2
                                 rust-windows-x86-64-gnullvm-0.48.5
                                 rust-windows-x86-64-gnullvm-0.52.6
                                 rust-windows-x86-64-msvc-0.42.2
                                 rust-windows-x86-64-msvc-0.48.5
                                 rust-windows-x86-64-msvc-0.52.6
                                 rust-winit-0.30.9
                                 rust-winnow-0.7.6
                                 rust-winreg-0.52.0
                                 rust-wio-0.2.2
                                 rust-wit-bindgen-rt-0.39.0
                                 rust-x11-clipboard-0.9.3
                                 rust-x11-dl-2.21.0
                                 rust-x11rb-0.13.1
                                 rust-x11rb-protocol-0.13.1
                                 rust-xcursor-0.3.8
                                 rust-xdg-2.5.2
                                 rust-xkbcommon-dl-0.4.2
                                 rust-xkeysym-0.2.1
                                 rust-xml-rs-0.8.26
                                 rust-yeslogic-fontconfig-sys-5.0.0
                                 rust-zerocopy-0.7.35
                                 rust-zerocopy-derive-0.7.35))
                     (alfis =>
                            (list rust-adler2-2.0.0
                                  rust-aead-0.5.2
                                  rust-aes-0.8.4
                                  rust-aes-gcm-0.10.3
                                  rust-allocator-api2-0.2.21
                                  rust-android-tzdata-0.1.1
                                  rust-android-system-properties-0.1.5
                                  rust-atk-sys-0.10.0
                                  rust-autocfg-1.4.0
                                  rust-base64-0.22.1
                                  rust-base64ct-1.7.3
                                  rust-bincode-1.3.3
                                  rust-bitflags-1.3.2
                                  rust-bitflags-2.9.0
                                  rust-blake2-0.9.2
                                  rust-blakeout-0.3.0
                                  rust-block-buffer-0.10.4
                                  rust-boxfnonce-0.1.1
                                  rust-bumpalo-3.17.0
                                  rust-byteorder-1.5.0
                                  rust-cairo-sys-rs-0.10.0
                                  rust-cc-1.2.19
                                  rust-cfg-if-1.0.0
                                  rust-chacha20-0.9.1
                                  rust-chacha20poly1305-0.10.1
                                  rust-chrono-0.4.40
                                  rust-cipher-0.4.4
                                  rust-const-oid-0.9.6
                                  rust-convert-case-0.4.0
                                  rust-core-foundation-sys-0.8.7
                                  rust-cpufeatures-0.2.17
                                  rust-crc32fast-1.4.2
                                  rust-crypto-common-0.1.6
                                  rust-crypto-mac-0.8.0
                                  rust-ctr-0.9.2
                                  rust-curve25519-dalek-4.1.3
                                  rust-curve25519-dalek-derive-0.1.1
                                  rust-der-0.7.9
                                  rust-deranged-0.4.0
                                  rust-derive-more-0.99.19
                                  rust-digest-0.9.0
                                  rust-digest-0.10.7
                                  rust-displaydoc-0.2.5
                                  rust-ecies-ed25519-ng-0.5.3.554ca29
                                  rust-ed25519-2.2.3
                                  rust-ed25519-dalek-2.1.1
                                  rust-equivalent-1.0.2
                                  rust-fiat-crypto-0.2.9
                                  rust-flate2-1.1.1
                                  rust-foldhash-0.1.5
                                  rust-form-urlencoded-1.2.1
                                  rust-gdk-pixbuf-sys-0.10.0
                                  rust-gdk-sys-0.10.0
                                  rust-generic-array-0.14.7
                                  rust-getopts-0.2.21
                                  rust-getrandom-0.2.15
                                  rust-getrandom-0.3.2
                                  rust-ghash-0.5.1
                                  rust-gio-sys-0.10.1
                                  rust-glib-sys-0.10.1
                                  rust-gobject-sys-0.10.0
                                  rust-gtk-sys-0.10.0
                                  rust-half-1.8.3
                                  rust-hashbrown-0.15.2
                                  rust-heck-0.3.3
                                  rust-hermit-abi-0.3.9
                                  rust-hex-0.4.3
                                  rust-hkdf-0.12.4
                                  rust-hmac-0.12.1
                                  rust-iana-time-zone-0.1.63
                                  rust-iana-time-zone-haiku-0.1.2
                                  rust-icu-collections-1.5.0
                                  rust-icu-locid-1.5.0
                                  rust-icu-locid-transform-1.5.0
                                  rust-icu-locid-transform-data-1.5.1
                                  rust-icu-normalizer-1.5.0
                                  rust-icu-normalizer-data-1.5.1
                                  rust-icu-properties-1.5.1
                                  rust-icu-properties-data-1.5.1
                                  rust-icu-provider-1.5.0
                                  rust-icu-provider-macros-1.5.0
                                  rust-idna-1.0.3
                                  rust-idna-adapter-1.2.0
                                  rust-indexmap-2.9.0
                                  rust-inout-0.1.4
                                  rust-is-docker-0.2.0
                                  rust-is-wsl-0.4.0
                                  rust-itoa-1.0.15
                                  rust-javascriptcore-rs-sys-0.2.0
                                  rust-js-sys-0.3.77
                                  rust-lazy-static-1.5.0
                                  rust-libc-0.2.171
                                  rust-litemap-0.7.5
                                  rust-log-0.4.27
                                  rust-lru-0.12.5
                                  rust-memchr-2.7.4
                                  rust-miniz-oxide-0.8.8
                                  rust-mio-1.0.3
                                  rust-num-bigint-0.4.6
                                  rust-num-conv-0.1.0
                                  rust-num-integer-0.1.46
                                  rust-num-traits-0.2.19
                                  rust-num-cpus-1.16.0
                                  rust-num-threads-0.1.7
                                  rust-once-cell-1.21.3
                                  rust-opaque-debug-0.3.1
                                  rust-open-5.3.2
                                  rust-pango-sys-0.10.0
                                  rust-pathdiff-0.2.3
                                  rust-percent-encoding-2.3.1
                                  rust-pkcs8-0.10.2
                                  rust-pkg-config-0.3.32
                                  rust-poly1305-0.8.0
                                  rust-polyval-0.6.2
                                  rust-powerfmt-0.2.0
                                  rust-ppv-lite86-0.2.21
                                  rust-proc-macro2-1.0.94
                                  rust-quote-1.0.40
                                  rust-r-efi-5.2.0
                                  rust-rand-0.8.5
                                  rust-rand-chacha-0.3.1
                                  rust-rand-core-0.6.4
                                  rust-ring-0.17.14
                                  rust-rustc-version-0.4.1
                                  rust-rustls-0.23.26
                                  rust-rustls-pki-types-1.11.0
                                  rust-rustls-webpki-0.103.1
                                  rust-rustversion-1.0.20
                                  rust-ryu-1.0.20
                                  rust-semver-1.0.26
                                  rust-serde-1.0.219
                                  rust-serde-bytes-0.11.17
                                  rust-serde-cbor-0.11.2
                                  rust-serde-derive-1.0.219
                                  rust-serde-json-1.0.140
                                  rust-serde-spanned-0.6.8
                                  rust-sha2-0.10.8
                                  rust-shlex-1.3.0
                                  rust-signature-2.2.0
                                  rust-simplelog-0.12.2
                                  rust-smallvec-1.15.0
                                  rust-soup-sys-0.10.0
                                  rust-spki-0.7.3
                                  rust-spmc-0.3.0
                                  rust-sqlite-0.36.2
                                  rust-sqlite3-src-0.6.1
                                  rust-sqlite3-sys-0.17.0
                                  rust-stable-deref-trait-1.2.0
                                  rust-strum-0.18.0
                                  rust-strum-macros-0.18.0
                                  rust-subtle-2.6.1
                                  rust-syn-1.0.109
                                  rust-syn-2.0.100
                                  rust-synstructure-0.13.1
                                  rust-system-deps-1.3.2
                                  rust-termcolor-1.4.1
                                  rust-thiserror-1.0.69
                                  rust-thiserror-impl-1.0.69
                                  rust-thread-priority-1.2.0
                                  rust-time-0.3.41
                                  rust-time-core-0.1.4
                                  rust-time-macros-0.2.22
                                  rust-tinyfiledialogs-3.9.1
                                  rust-tinystr-0.7.6
                                  rust-toml-0.5.11
                                  rust-toml-0.8.20
                                  rust-toml-datetime-0.6.8
                                  rust-toml-edit-0.22.24
                                  rust-typenum-1.18.0
                                  rust-unicode-ident-1.0.18
                                  rust-unicode-segmentation-1.12.0
                                  rust-unicode-width-0.1.14
                                  rust-universal-hash-0.5.1
                                  rust-untrusted-0.9.0
                                  rust-ureq-2.10.0.1cad58f
                                  rust-url-2.5.4
                                  rust-urlencoding-1.3.3
                                  rust-utf16-iter-1.0.5
                                  rust-utf8-iter-1.0.4
                                  rust-uuid-1.16.0
                                  rust-version-compare-0.0.10
                                  rust-version-check-0.9.5
                                  rust-wasi-0.11.0+wasi-snapshot-preview1
                                  rust-wasi-0.14.2+wasi-0.2.4
                                  rust-wasm-bindgen-0.2.100
                                  rust-wasm-bindgen-backend-0.2.100
                                  rust-wasm-bindgen-macro-0.2.100
                                  rust-wasm-bindgen-macro-support-0.2.100
                                  rust-wasm-bindgen-shared-0.2.100
                                  rust-web-view-0.7.3.82d7cbc
                                  rust-webkit2gtk-sys-0.12.0
                                  rust-webpki-roots-0.26.8
                                  rust-webview-sys-0.6.2.82d7cbc
                                  rust-widestring-1.2.0
                                  rust-winapi-0.3.9
                                  rust-winapi-i686-pc-windows-gnu-0.4.0
                                  rust-winapi-util-0.1.9
                                  rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                  rust-windows-core-0.61.0
                                  rust-windows-implement-0.60.0
                                  rust-windows-interface-0.59.1
                                  rust-windows-link-0.1.1
                                  rust-windows-result-0.3.2
                                  rust-windows-service-0.7.0
                                  rust-windows-strings-0.4.0
                                  rust-windows-sys-0.52.0
                                  rust-windows-sys-0.59.0
                                  rust-windows-targets-0.52.6
                                  rust-windows-aarch64-gnullvm-0.52.6
                                  rust-windows-aarch64-msvc-0.52.6
                                  rust-windows-i686-gnu-0.52.6
                                  rust-windows-i686-gnullvm-0.52.6
                                  rust-windows-i686-msvc-0.52.6
                                  rust-windows-x86-64-gnu-0.52.6
                                  rust-windows-x86-64-gnullvm-0.52.6
                                  rust-windows-x86-64-msvc-0.52.6
                                  rust-winnow-0.7.6
                                  rust-winres-0.1.12
                                  rust-wit-bindgen-rt-0.39.0
                                  rust-write16-1.0.0
                                  rust-writeable-0.5.5
                                  rust-x25519-dalek-2.0.1
                                  rust-yoke-0.7.5
                                  rust-yoke-derive-0.7.5
                                  rust-zerocopy-0.8.24
                                  rust-zerocopy-derive-0.8.24
                                  rust-zerofrom-0.1.6
                                  rust-zerofrom-derive-0.1.6
                                  rust-zeroize-1.8.1
                                  rust-zeroize-derive-1.4.2
                                  rust-zerovec-0.10.4
                                  rust-zerovec-derive-0.10.3))
                     (b3sum =>
                            (list rust-anstream-0.6.18
                                  rust-anstyle-1.0.10
                                  rust-anstyle-parse-0.2.6
                                  rust-anstyle-query-1.1.2
                                  rust-anstyle-wincon-3.0.7
                                  rust-anyhow-1.0.97
                                  rust-arrayref-0.3.9
                                  rust-arrayvec-0.7.6
                                  rust-bitflags-2.9.0
                                  rust-blake3-1.8.1
                                  rust-cc-1.2.18
                                  rust-cfg-if-1.0.0
                                  rust-clap-4.5.35
                                  rust-clap-builder-4.5.35
                                  rust-clap-derive-4.5.32
                                  rust-clap-lex-0.7.4
                                  rust-colorchoice-1.0.3
                                  rust-constant-time-eq-0.3.1
                                  rust-crossbeam-deque-0.8.6
                                  rust-crossbeam-epoch-0.9.18
                                  rust-crossbeam-utils-0.8.21
                                  rust-duct-0.13.7
                                  rust-errno-0.3.11
                                  rust-fastrand-2.3.0
                                  rust-getrandom-0.3.2
                                  rust-glob-0.3.2
                                  rust-heck-0.5.0
                                  rust-hex-0.4.3
                                  rust-is-terminal-polyfill-1.70.1
                                  rust-libc-0.2.171
                                  rust-linux-raw-sys-0.9.3
                                  rust-memmap2-0.9.5
                                  rust-once-cell-1.21.3
                                  rust-os-pipe-1.2.1
                                  rust-proc-macro2-1.0.94
                                  rust-quote-1.0.40
                                  rust-r-efi-5.2.0
                                  rust-rayon-core-1.12.1
                                  rust-rustix-1.0.5
                                  rust-shared-child-1.0.1
                                  rust-shlex-1.3.0
                                  rust-strsim-0.11.1
                                  rust-syn-2.0.100
                                  rust-tempfile-3.19.1
                                  rust-terminal-size-0.4.2
                                  rust-unicode-ident-1.0.18
                                  rust-utf8parse-0.2.2
                                  rust-wasi-0.14.2+wasi-0.2.4
                                  rust-wild-2.2.1
                                  rust-windows-sys-0.59.0
                                  rust-windows-targets-0.52.6
                                  rust-windows-aarch64-gnullvm-0.52.6
                                  rust-windows-aarch64-msvc-0.52.6
                                  rust-windows-i686-gnu-0.52.6
                                  rust-windows-i686-gnullvm-0.52.6
                                  rust-windows-i686-msvc-0.52.6
                                  rust-windows-x86-64-gnu-0.52.6
                                  rust-windows-x86-64-gnullvm-0.52.6
                                  rust-windows-x86-64-msvc-0.52.6
                                  rust-wit-bindgen-rt-0.39.0))
                     (bankstown-lv2 =>
                                    (list rust-biquad-0.4.2
                                          rust-libm-0.1.4
                                          rust-lv2-0.6.0
                                          rust-lv2-atom-2.0.0
                                          rust-lv2-core-3.0.0
                                          rust-lv2-core-derive-2.1.1
                                          rust-lv2-midi-1.2.0
                                          rust-lv2-sys-2.0.0
                                          rust-lv2-units-0.1.3
                                          rust-lv2-urid-2.1.0
                                          rust-proc-macro2-1.0.94
                                          rust-quote-1.0.40
                                          rust-syn-1.0.109
                                          rust-unicode-ident-1.0.18
                                          rust-urid-0.1.0
                                          rust-urid-derive-0.1.1
                                          rust-wmidi-3.1.0))
                     (bat =>
                          (list rust-adler2-2.0.0
                                rust-aho-corasick-1.1.3
                                rust-ansi-colours-1.2.3
                                rust-anstream-0.6.18
                                rust-anstyle-1.0.10
                                rust-anstyle-parse-0.2.6
                                rust-anstyle-query-1.1.2
                                rust-anstyle-wincon-3.0.7
                                rust-assert-cmd-2.0.16
                                rust-autocfg-1.4.0
                                rust-base64-0.22.1
                                rust-bincode-1.3.3
                                rust-bit-set-0.5.3
                                rust-bit-vec-0.6.3
                                rust-bitflags-1.3.2
                                rust-bitflags-2.9.0
                                rust-bstr-1.12.0
                                rust-bugreport-0.5.1
                                rust-bytemuck-1.22.0
                                rust-bytesize-1.3.3
                                rust-cc-1.2.19
                                rust-cfg-if-1.0.0
                                rust-clap-4.5.36
                                rust-clap-builder-4.5.36
                                rust-clap-lex-0.7.4
                                rust-clircle-0.4.0
                                rust-colorchoice-1.0.3
                                rust-console-0.15.11
                                rust-content-inspector-0.2.4
                                rust-core-foundation-sys-0.8.7
                                rust-crc32fast-1.4.2
                                rust-crossbeam-deque-0.8.6
                                rust-crossbeam-epoch-0.9.18
                                rust-crossbeam-utils-0.8.21
                                rust-dashmap-5.5.3
                                rust-deranged-0.4.0
                                rust-difflib-0.4.0
                                rust-displaydoc-0.2.5
                                rust-dissimilar-1.0.10
                                rust-doc-comment-0.3.3
                                rust-dunce-1.0.5
                                rust-either-1.15.0
                                rust-encode-unicode-1.0.0
                                rust-encoding-rs-0.8.35
                                rust-equivalent-1.0.2
                                rust-errno-0.3.11
                                rust-etcetera-0.8.0
                                rust-expect-test-1.5.1
                                rust-fancy-regex-0.11.0
                                rust-fastrand-2.3.0
                                rust-flate2-1.1.1
                                rust-float-cmp-0.10.0
                                rust-fnv-1.0.7
                                rust-form-urlencoded-1.2.1
                                rust-fsio-0.4.1
                                rust-getrandom-0.3.2
                                rust-git-version-0.3.9
                                rust-git-version-macro-0.3.9
                                rust-git2-0.18.3
                                rust-glob-0.3.2
                                rust-globset-0.4.16
                                rust-grep-cli-0.1.11
                                rust-hashbrown-0.14.5
                                rust-hashbrown-0.15.2
                                rust-home-0.5.11
                                rust-icu-collections-1.5.0
                                rust-icu-locid-1.5.0
                                rust-icu-locid-transform-1.5.0
                                rust-icu-locid-transform-data-1.5.1
                                rust-icu-normalizer-1.5.0
                                rust-icu-normalizer-data-1.5.1
                                rust-icu-properties-1.5.1
                                rust-icu-properties-data-1.5.1
                                rust-icu-provider-1.5.0
                                rust-icu-provider-macros-1.5.0
                                rust-idna-1.0.3
                                rust-idna-adapter-1.2.0
                                rust-indexmap-2.9.0
                                rust-is-terminal-polyfill-1.70.1
                                rust-itoa-1.0.15
                                rust-jobserver-0.1.33
                                rust-lazy-static-1.5.0
                                rust-libc-0.2.171
                                rust-libgit2-sys-0.16.2+1.7.2
                                rust-libz-sys-1.1.22
                                rust-linked-hash-map-0.5.6
                                rust-linux-raw-sys-0.9.4
                                rust-litemap-0.7.5
                                rust-lock-api-0.4.12
                                rust-log-0.4.27
                                rust-memchr-2.7.4
                                rust-miniz-oxide-0.8.8
                                rust-nix-0.26.4
                                rust-normalize-line-endings-0.3.0
                                rust-ntapi-0.4.1
                                rust-nu-ansi-term-0.49.0
                                rust-num-conv-0.1.0
                                rust-num-traits-0.2.19
                                rust-once-cell-1.21.3
                                rust-onig-6.4.0
                                rust-onig-sys-69.8.1
                                rust-os-str-bytes-6.6.1
                                rust-parking-lot-0.12.3
                                rust-parking-lot-core-0.9.10
                                rust-path-abs-0.5.1
                                rust-percent-encoding-2.3.1
                                rust-pkg-config-0.3.32
                                rust-plist-1.7.1
                                rust-powerfmt-0.2.0
                                rust-ppv-lite86-0.2.21
                                rust-predicates-3.1.3
                                rust-predicates-core-1.0.9
                                rust-predicates-tree-1.0.12
                                rust-proc-macro2-1.0.94
                                rust-quick-xml-0.32.0
                                rust-quote-1.0.40
                                rust-r-efi-5.2.0
                                rust-rand-0.9.0
                                rust-rand-chacha-0.9.0
                                rust-rand-core-0.9.3
                                rust-rayon-1.10.0
                                rust-rayon-core-1.12.1
                                rust-redox-syscall-0.5.11
                                rust-regex-1.11.1
                                rust-regex-automata-0.4.9
                                rust-regex-syntax-0.8.5
                                rust-rgb-0.8.50
                                rust-run-script-0.10.1
                                rust-rustix-1.0.5
                                rust-ryu-1.0.20
                                rust-same-file-1.0.6
                                rust-scopeguard-1.2.0
                                rust-semver-1.0.26
                                rust-serde-1.0.219
                                rust-serde-derive-1.0.219
                                rust-serde-json-1.0.140
                                rust-serde-yaml-0.9.34+deprecated
                                rust-serial-test-2.0.0
                                rust-serial-test-derive-2.0.0
                                rust-shell-escape-0.1.5
                                rust-shell-words-1.1.0
                                rust-shlex-1.3.0
                                rust-smallvec-1.15.0
                                rust-stable-deref-trait-1.2.0
                                rust-std-prelude-0.2.12
                                rust-strsim-0.11.1
                                rust-syn-2.0.100
                                rust-synstructure-0.13.1
                                rust-syntect-5.2.0
                                rust-sysinfo-0.33.1
                                rust-tempfile-3.19.1
                                rust-termcolor-1.4.1
                                rust-terminal-size-0.4.2
                                rust-termtree-0.5.1
                                rust-thiserror-1.0.69
                                rust-thiserror-impl-1.0.69
                                rust-time-0.3.41
                                rust-time-core-0.1.4
                                rust-time-macros-0.2.22
                                rust-tinystr-0.7.6
                                rust-unicode-ident-1.0.18
                                rust-unicode-width-0.1.14
                                rust-unicode-width-0.2.0
                                rust-unsafe-libyaml-0.2.11
                                rust-url-2.5.4
                                rust-utf16-iter-1.0.5
                                rust-utf8-iter-1.0.4
                                rust-utf8parse-0.2.2
                                rust-vcpkg-0.2.15
                                rust-wait-timeout-0.2.1
                                rust-walkdir-2.5.0
                                rust-wasi-0.14.2+wasi-0.2.4
                                rust-wild-2.2.1
                                rust-winapi-0.3.9
                                rust-winapi-i686-pc-windows-gnu-0.4.0
                                rust-winapi-util-0.1.9
                                rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                rust-windows-0.57.0
                                rust-windows-core-0.57.0
                                rust-windows-implement-0.57.0
                                rust-windows-interface-0.57.0
                                rust-windows-result-0.1.2
                                rust-windows-sys-0.48.0
                                rust-windows-sys-0.59.0
                                rust-windows-targets-0.48.5
                                rust-windows-targets-0.52.6
                                rust-windows-aarch64-gnullvm-0.48.5
                                rust-windows-aarch64-gnullvm-0.52.6
                                rust-windows-aarch64-msvc-0.48.5
                                rust-windows-aarch64-msvc-0.52.6
                                rust-windows-i686-gnu-0.48.5
                                rust-windows-i686-gnu-0.52.6
                                rust-windows-i686-gnullvm-0.52.6
                                rust-windows-i686-msvc-0.48.5
                                rust-windows-i686-msvc-0.52.6
                                rust-windows-x86-64-gnu-0.48.5
                                rust-windows-x86-64-gnu-0.52.6
                                rust-windows-x86-64-gnullvm-0.48.5
                                rust-windows-x86-64-gnullvm-0.52.6
                                rust-windows-x86-64-msvc-0.48.5
                                rust-windows-x86-64-msvc-0.52.6
                                rust-wit-bindgen-rt-0.39.0
                                rust-write16-1.0.0
                                rust-writeable-0.5.5
                                rust-yaml-rust-0.4.5
                                rust-yoke-0.7.5
                                rust-yoke-derive-0.7.5
                                rust-zerocopy-0.8.24
                                rust-zerocopy-derive-0.8.24
                                rust-zerofrom-0.1.6
                                rust-zerofrom-derive-0.1.6
                                rust-zerovec-0.10.4
                                rust-zerovec-derive-0.10.3))
                     (bcachefs-tools =>
                                     (list rust-aho-corasick-1.1.3
                                      rust-anstream-0.6.18
                                      rust-anstyle-1.0.10
                                      rust-anstyle-parse-0.2.6
                                      rust-anstyle-query-1.1.2
                                      rust-anstyle-wincon-3.0.7
                                      rust-anyhow-1.0.97
                                      rust-autocfg-1.4.0
                                      rust-bindgen-0.69.5
                                      rust-bitfield-0.14.0
                                      rust-bitflags-1.3.2
                                      rust-bitflags-2.9.0
                                      rust-byteorder-1.5.0
                                      rust-cc-1.2.18
                                      rust-cexpr-0.6.0
                                      rust-cfg-if-1.0.0
                                      rust-clang-sys-1.8.1
                                      rust-clap-4.5.35
                                      rust-clap-builder-4.5.35
                                      rust-clap-complete-4.5.47
                                      rust-clap-derive-4.5.32
                                      rust-clap-lex-0.7.4
                                      rust-colorchoice-1.0.3
                                      rust-either-1.15.0
                                      rust-env-logger-0.10.2
                                      rust-errno-0.2.8
                                      rust-errno-0.3.11
                                      rust-errno-dragonfly-0.1.2
                                      rust-glob-0.3.2
                                      rust-heck-0.5.0
                                      rust-home-0.5.11
                                      rust-is-terminal-polyfill-1.70.1
                                      rust-itertools-0.12.1
                                      rust-lazy-static-1.5.0
                                      rust-lazycell-1.3.0
                                      rust-libc-0.2.171
                                      rust-libloading-0.8.6
                                      rust-libudev-sys-0.1.4
                                      rust-linux-raw-sys-0.4.15
                                      rust-linux-raw-sys-0.9.3
                                      rust-log-0.4.27
                                      rust-memchr-2.7.4
                                      rust-memoffset-0.8.0
                                      rust-minimal-lexical-0.2.1
                                      rust-nom-7.1.3
                                      rust-once-cell-1.21.3
                                      rust-owo-colors-4.2.0
                                      rust-paste-1.0.15
                                      rust-pkg-config-0.3.32
                                      rust-prettyplease-0.2.32
                                      rust-proc-macro2-1.0.94
                                      rust-quote-1.0.40
                                      rust-regex-1.11.1
                                      rust-regex-automata-0.4.9
                                      rust-regex-syntax-0.8.5
                                      rust-rustc-hash-1.1.0
                                      rust-rustix-0.38.44
                                      rust-rustix-1.0.5
                                      rust-rustversion-1.0.20
                                      rust-shlex-1.3.0
                                      rust-strsim-0.11.1
                                      rust-strum-0.26.3
                                      rust-strum-macros-0.26.4
                                      rust-syn-2.0.100
                                      rust-terminal-size-0.4.2
                                      rust-udev-0.7.0
                                      rust-unicode-ident-1.0.18
                                      rust-utf8parse-0.2.2
                                      rust-uuid-1.16.0
                                      rust-which-4.4.2
                                      rust-winapi-0.3.9
                                      rust-winapi-i686-pc-windows-gnu-0.4.0
                                      rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                      rust-windows-sys-0.59.0
                                      rust-windows-targets-0.52.6
                                      rust-windows-aarch64-gnullvm-0.52.6
                                      rust-windows-aarch64-msvc-0.52.6
                                      rust-windows-i686-gnu-0.52.6
                                      rust-windows-i686-gnullvm-0.52.6
                                      rust-windows-i686-msvc-0.52.6
                                      rust-windows-x86-64-gnu-0.52.6
                                      rust-windows-x86-64-gnullvm-0.52.6
                                      rust-windows-x86-64-msvc-0.52.6
                                      rust-zeroize-1.8.1
                                      rust-zeroize-derive-1.4.2))
                     (bottom =>
                             (list rust-addr2line-0.24.2
                                   rust-adler2-2.0.0
                                   rust-ahash-0.8.11
                                   rust-aho-corasick-1.1.3
                                   rust-allocator-api2-0.2.21
                                   rust-anstream-0.6.18
                                   rust-anstyle-1.0.10
                                   rust-anstyle-parse-0.2.6
                                   rust-anstyle-query-1.1.2
                                   rust-anstyle-wincon-3.0.7
                                   rust-anyhow-1.0.97
                                   rust-assert-cmd-2.0.16
                                   rust-autocfg-1.4.0
                                   rust-backtrace-0.3.74
                                   rust-base64-0.22.1
                                   rust-bitflags-1.3.2
                                   rust-bitflags-2.9.0
                                   rust-bstr-1.12.0
                                   rust-byteorder-1.5.0
                                   rust-cargo-husky-1.5.0
                                   rust-cassowary-0.3.0
                                   rust-castaway-0.2.3
                                   rust-cfg-if-1.0.0
                                   rust-cfg-aliases-0.2.1
                                   rust-clap-4.5.36
                                   rust-clap-builder-4.5.36
                                   rust-clap-complete-4.5.47
                                   rust-clap-complete-fig-4.5.2
                                   rust-clap-complete-nushell-4.5.5
                                   rust-clap-derive-4.5.32
                                   rust-clap-lex-0.7.4
                                   rust-clap-mangen-0.2.26
                                   rust-colorchoice-1.0.3
                                   rust-compact-str-0.7.1
                                   rust-concat-string-1.0.1
                                   rust-core-foundation-0.9.4
                                   rust-core-foundation-sys-0.8.7
                                   rust-crossbeam-deque-0.8.6
                                   rust-crossbeam-epoch-0.9.18
                                   rust-crossbeam-utils-0.8.21
                                   rust-crossterm-0.27.0
                                   rust-crossterm-winapi-0.9.1
                                   rust-ctrlc-3.4.6
                                   rust-darling-0.20.11
                                   rust-darling-core-0.20.11
                                   rust-darling-macro-0.20.11
                                   rust-deranged-0.4.0
                                   rust-difflib-0.4.0
                                   rust-dirs-5.0.1
                                   rust-dirs-sys-0.4.1
                                   rust-doc-comment-0.3.3
                                   rust-downcast-rs-1.2.1
                                   rust-dyn-clone-1.0.19
                                   rust-either-1.15.0
                                   rust-enum-as-inner-0.6.1
                                   rust-equivalent-1.0.2
                                   rust-errno-0.3.11
                                   rust-fern-0.6.2
                                   rust-filedescriptor-0.8.3
                                   rust-float-cmp-0.10.0
                                   rust-fnv-1.0.7
                                   rust-foldhash-0.1.5
                                   rust-getrandom-0.2.15
                                   rust-gimli-0.31.1
                                   rust-hashbrown-0.14.5
                                   rust-hashbrown-0.15.2
                                   rust-heck-0.5.0
                                   rust-humantime-2.2.0
                                   rust-ident-case-1.0.1
                                   rust-indexmap-2.9.0
                                   rust-indoc-2.0.6
                                   rust-ioctl-rs-0.1.6
                                   rust-is-terminal-polyfill-1.70.1
                                   rust-itertools-0.13.0
                                   rust-itoa-1.0.15
                                   rust-lazy-static-1.5.0
                                   rust-lazycell-1.3.0
                                   rust-libc-0.2.171
                                   rust-libloading-0.8.6
                                   rust-libredox-0.1.3
                                   rust-linux-raw-sys-0.4.15
                                   rust-linux-raw-sys-0.9.4
                                   rust-lock-api-0.4.12
                                   rust-log-0.4.27
                                   rust-lru-0.12.5
                                   rust-mach2-0.4.2
                                   rust-memchr-2.7.4
                                   rust-memoffset-0.6.5
                                   rust-miniz-oxide-0.8.8
                                   rust-mio-0.8.11
                                   rust-nix-0.25.1
                                   rust-nix-0.29.0
                                   rust-normalize-line-endings-0.3.0
                                   rust-ntapi-0.4.1
                                   rust-num-conv-0.1.0
                                   rust-num-traits-0.2.19
                                   rust-num-threads-0.1.7
                                   rust-nvml-wrapper-0.10.0
                                   rust-nvml-wrapper-sys-0.8.0
                                   rust-object-0.36.7
                                   rust-once-cell-1.21.3
                                   rust-option-ext-0.2.0
                                   rust-parking-lot-0.12.3
                                   rust-parking-lot-core-0.9.10
                                   rust-paste-1.0.15
                                   rust-pin-utils-0.1.0
                                   rust-plist-1.7.1
                                   rust-portable-pty-0.8.1
                                   rust-powerfmt-0.2.0
                                   rust-predicates-3.1.3
                                   rust-predicates-core-1.0.9
                                   rust-predicates-tree-1.0.12
                                   rust-proc-macro2-1.0.94
                                   rust-quick-xml-0.32.0
                                   rust-quote-1.0.40
                                   rust-ratatui-0.27.0
                                   rust-rayon-1.10.0
                                   rust-rayon-core-1.12.1
                                   rust-redox-syscall-0.5.11
                                   rust-redox-users-0.4.6
                                   rust-regex-1.11.1
                                   rust-regex-automata-0.4.9
                                   rust-regex-syntax-0.8.5
                                   rust-roff-0.2.2
                                   rust-rustc-demangle-0.1.24
                                   rust-rustix-0.38.44
                                   rust-rustix-1.0.5
                                   rust-rustversion-1.0.20
                                   rust-ryu-1.0.20
                                   rust-same-file-1.0.6
                                   rust-schemars-0.8.22
                                   rust-schemars-derive-0.8.22
                                   rust-scopeguard-1.2.0
                                   rust-serde-1.0.219
                                   rust-serde-derive-1.0.219
                                   rust-serde-derive-internals-0.29.1
                                   rust-serde-json-1.0.140
                                   rust-serde-spanned-0.6.8
                                   rust-serial-0.4.0
                                   rust-serial-core-0.4.0
                                   rust-serial-unix-0.4.0
                                   rust-serial-windows-0.4.0
                                   rust-shared-library-0.1.9
                                   rust-shell-words-1.1.0
                                   rust-signal-hook-0.3.17
                                   rust-signal-hook-mio-0.2.4
                                   rust-signal-hook-registry-1.4.2
                                   rust-smallvec-1.15.0
                                   rust-stability-0.2.1
                                   rust-starship-battery-0.9.1
                                   rust-static-assertions-1.1.0
                                   rust-strsim-0.11.1
                                   rust-strum-0.26.3
                                   rust-strum-macros-0.26.4
                                   rust-syn-2.0.100
                                   rust-sysctl-0.5.5
                                   rust-sysinfo-0.30.13
                                   rust-terminal-size-0.4.2
                                   rust-termios-0.2.2
                                   rust-termtree-0.5.1
                                   rust-thiserror-1.0.69
                                   rust-thiserror-impl-1.0.69
                                   rust-time-0.3.41
                                   rust-time-core-0.1.4
                                   rust-time-macros-0.2.22
                                   rust-toml-datetime-0.6.8
                                   rust-toml-edit-0.22.24
                                   rust-typenum-1.18.0
                                   rust-unicode-ellipsis-0.2.0
                                   rust-unicode-ident-1.0.18
                                   rust-unicode-segmentation-1.12.0
                                   rust-unicode-truncate-1.1.0
                                   rust-unicode-width-0.1.14
                                   rust-uom-0.36.0
                                   rust-utf8parse-0.2.2
                                   rust-version-check-0.9.5
                                   rust-wait-timeout-0.2.1
                                   rust-walkdir-2.5.0
                                   rust-wasi-0.11.0+wasi-snapshot-preview1
                                   rust-winapi-0.3.9
                                   rust-winapi-i686-pc-windows-gnu-0.4.0
                                   rust-winapi-util-0.1.9
                                   rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                   rust-windows-0.52.0
                                   rust-windows-0.58.0
                                   rust-windows-core-0.52.0
                                   rust-windows-core-0.58.0
                                   rust-windows-implement-0.58.0
                                   rust-windows-interface-0.58.0
                                   rust-windows-result-0.2.0
                                   rust-windows-strings-0.1.0
                                   rust-windows-sys-0.48.0
                                   rust-windows-sys-0.59.0
                                   rust-windows-targets-0.48.5
                                   rust-windows-targets-0.52.6
                                   rust-windows-aarch64-gnullvm-0.48.5
                                   rust-windows-aarch64-gnullvm-0.52.6
                                   rust-windows-aarch64-msvc-0.48.5
                                   rust-windows-aarch64-msvc-0.52.6
                                   rust-windows-i686-gnu-0.48.5
                                   rust-windows-i686-gnu-0.52.6
                                   rust-windows-i686-gnullvm-0.52.6
                                   rust-windows-i686-msvc-0.48.5
                                   rust-windows-i686-msvc-0.52.6
                                   rust-windows-x86-64-gnu-0.48.5
                                   rust-windows-x86-64-gnu-0.52.6
                                   rust-windows-x86-64-gnullvm-0.48.5
                                   rust-windows-x86-64-gnullvm-0.52.6
                                   rust-windows-x86-64-msvc-0.48.5
                                   rust-windows-x86-64-msvc-0.52.6
                                   rust-winnow-0.7.6
                                   rust-winreg-0.10.1
                                   rust-wrapcenum-derive-0.4.1
                                   rust-zerocopy-0.7.35
                                   rust-zerocopy-derive-0.7.35))
                     (breezy =>
                             (list rust-aho-corasick-1.1.3
                                   rust-autocfg-1.4.0
                                   rust-cfg-if-1.0.0
                                   rust-heck-0.5.0
                                   rust-indoc-2.0.6
                                   rust-lazy-static-1.5.0
                                   rust-libc-0.2.171
                                   rust-memchr-2.7.4
                                   rust-memoffset-0.9.1
                                   rust-once-cell-1.21.3
                                   rust-portable-atomic-1.11.0
                                   rust-proc-macro2-1.0.94
                                   rust-pyo3-0.23.5
                                   rust-pyo3-build-config-0.23.5
                                   rust-pyo3-ffi-0.23.5
                                   rust-pyo3-macros-0.23.5
                                   rust-pyo3-macros-backend-0.23.5
                                   rust-quote-1.0.40
                                   rust-regex-1.11.1
                                   rust-regex-automata-0.4.9
                                   rust-regex-syntax-0.8.5
                                   rust-syn-2.0.100
                                   rust-target-lexicon-0.12.16
                                   rust-unicode-ident-1.0.18
                                   rust-unindent-0.2.4))
                     (c2rust =>
                             (list rust-addr2line-0.24.2
                                   rust-adler2-2.0.0
                                   rust-aho-corasick-1.1.3
                                   rust-anyhow-1.0.97
                                   rust-arc-swap-1.7.1
                                   rust-atty-0.2.14
                                   rust-autocfg-1.4.0
                                   rust-backtrace-0.3.74
                                   rust-bindgen-0.65.1
                                   rust-bitflags-1.3.2
                                   rust-bitflags-2.9.0
                                   rust-block-buffer-0.10.4
                                   rust-c2rust-ast-builder-0.20.0
                                   rust-c2rust-ast-exporter-0.20.0
                                   rust-c2rust-ast-printer-0.20.0
                                   rust-c2rust-bitfields-0.20.0
                                   rust-c2rust-bitfields-derive-0.20.0
                                   rust-c2rust-build-paths-0.20.0
                                   rust-c2rust-transpile-0.20.0
                                   rust-cc-1.2.18
                                   rust-cexpr-0.6.0
                                   rust-cfg-if-1.0.0
                                   rust-clang-sys-1.8.1
                                   rust-clap-3.2.25
                                   rust-clap-derive-3.2.25
                                   rust-clap-lex-0.2.4
                                   rust-cmake-0.1.54
                                   rust-colored-1.9.4
                                   rust-colored-2.2.0
                                   rust-cpufeatures-0.2.17
                                   rust-crypto-common-0.1.6
                                   rust-digest-0.10.7
                                   rust-dtoa-1.0.10
                                   rust-either-1.15.0
                                   rust-env-logger-0.10.2
                                   rust-errno-0.3.11
                                   rust-failure-0.1.8
                                   rust-failure-derive-0.1.8
                                   rust-fern-0.6.2
                                   rust-generic-array-0.14.7
                                   rust-gimli-0.31.1
                                   rust-glob-0.3.2
                                   rust-half-1.8.3
                                   rust-handlebars-4.5.0
                                   rust-hashbrown-0.12.3
                                   rust-heck-0.4.1
                                   rust-hermit-abi-0.1.19
                                   rust-hermit-abi-0.5.0
                                   rust-home-0.5.11
                                   rust-humantime-2.2.0
                                   rust-indexmap-1.9.3
                                   rust-is-terminal-0.4.16
                                   rust-is-executable-1.0.4
                                   rust-itertools-0.10.5
                                   rust-itoa-1.0.15
                                   rust-lazy-static-1.5.0
                                   rust-lazycell-1.3.0
                                   rust-libc-0.2.171
                                   rust-libloading-0.8.6
                                   rust-linked-hash-map-0.5.6
                                   rust-linux-raw-sys-0.4.15
                                   rust-log-0.4.27
                                   rust-log-reroute-0.1.8
                                   rust-memchr-2.7.4
                                   rust-minimal-lexical-0.2.1
                                   rust-miniz-oxide-0.8.7
                                   rust-nom-7.1.3
                                   rust-object-0.36.7
                                   rust-once-cell-1.21.3
                                   rust-os-str-bytes-6.6.1
                                   rust-pathdiff-0.2.3
                                   rust-peeking-take-while-0.1.2
                                   rust-pest-2.8.0
                                   rust-pest-derive-2.8.0
                                   rust-pest-generator-2.8.0
                                   rust-pest-meta-2.8.0
                                   rust-prettyplease-0.1.25
                                   rust-prettyplease-0.2.32
                                   rust-print-bytes-1.2.0
                                   rust-proc-macro-error-1.0.4
                                   rust-proc-macro-error-attr-1.0.4
                                   rust-proc-macro2-1.0.94
                                   rust-quote-1.0.40
                                   rust-regex-1.11.1
                                   rust-regex-automata-0.4.9
                                   rust-regex-syntax-0.8.5
                                   rust-rustc-demangle-0.1.24
                                   rust-rustc-hash-1.1.0
                                   rust-rustix-0.38.44
                                   rust-rustversion-1.0.20
                                   rust-ryu-1.0.20
                                   rust-serde-1.0.219
                                   rust-serde-bencode-0.2.4
                                   rust-serde-bytes-0.11.17
                                   rust-serde-cbor-0.11.2
                                   rust-serde-derive-1.0.219
                                   rust-serde-json-1.0.140
                                   rust-sha2-0.10.8
                                   rust-shlex-1.3.0
                                   rust-smallvec-1.15.0
                                   rust-strsim-0.10.0
                                   rust-strum-0.24.1
                                   rust-strum-macros-0.24.3
                                   rust-syn-1.0.109
                                   rust-syn-2.0.100
                                   rust-synstructure-0.12.6
                                   rust-termcolor-1.4.1
                                   rust-textwrap-0.16.2
                                   rust-thiserror-1.0.69
                                   rust-thiserror-2.0.12
                                   rust-thiserror-impl-1.0.69
                                   rust-thiserror-impl-2.0.12
                                   rust-typenum-1.18.0
                                   rust-ucd-trie-0.1.7
                                   rust-unicode-ident-1.0.18
                                   rust-unicode-xid-0.2.6
                                   rust-version-check-0.9.5
                                   rust-which-4.4.2
                                   rust-winapi-0.3.9
                                   rust-winapi-i686-pc-windows-gnu-0.4.0
                                   rust-winapi-util-0.1.9
                                   rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                   rust-windows-sys-0.48.0
                                   rust-windows-sys-0.59.0
                                   rust-windows-targets-0.48.5
                                   rust-windows-targets-0.52.6
                                   rust-windows-aarch64-gnullvm-0.48.5
                                   rust-windows-aarch64-gnullvm-0.52.6
                                   rust-windows-aarch64-msvc-0.48.5
                                   rust-windows-aarch64-msvc-0.52.6
                                   rust-windows-i686-gnu-0.48.5
                                   rust-windows-i686-gnu-0.52.6
                                   rust-windows-i686-gnullvm-0.52.6
                                   rust-windows-i686-msvc-0.48.5
                                   rust-windows-i686-msvc-0.52.6
                                   rust-windows-x86-64-gnu-0.48.5
                                   rust-windows-x86-64-gnu-0.52.6
                                   rust-windows-x86-64-gnullvm-0.48.5
                                   rust-windows-x86-64-gnullvm-0.52.6
                                   rust-windows-x86-64-msvc-0.48.5
                                   rust-windows-x86-64-msvc-0.52.6
                                   rust-yaml-rust-0.4.5))
                     (cargo-audit =>
                                  (list rust-abscissa-core-0.8.2
                                   rust-abscissa-derive-0.8.2
                                   rust-addr2line-0.21.0
                                   rust-adler-1.0.2
                                   rust-adler2-2.0.0
                                   rust-ahash-0.8.11
                                   rust-aho-corasick-1.1.3
                                   rust-allocator-api2-0.2.21
                                   rust-anstream-0.6.18
                                   rust-anstyle-1.0.10
                                   rust-anstyle-parse-0.2.6
                                   rust-anstyle-query-1.1.2
                                   rust-anstyle-wincon-3.0.7
                                   rust-arc-swap-1.7.1
                                   rust-arrayvec-0.7.6
                                   rust-async-compression-0.4.22
                                   rust-atomic-waker-1.1.2
                                   rust-auditable-extract-0.3.5
                                   rust-auditable-info-0.8.0
                                   rust-auditable-serde-0.7.0
                                   rust-autocfg-1.4.0
                                   rust-backtrace-0.3.71
                                   rust-base64-0.22.1
                                   rust-binfarce-0.2.1
                                   rust-bitflags-2.9.0
                                   rust-borsh-1.5.7
                                   rust-bstr-1.11.3
                                   rust-bumpalo-3.17.0
                                   rust-bytes-1.10.1
                                   rust-camino-1.1.9
                                   rust-canonical-path-2.0.2
                                   rust-cargo-lock-10.1.0
                                   rust-cc-1.2.18
                                   rust-cfg-if-1.0.0
                                   rust-cfg-aliases-0.2.1
                                   rust-clap-4.5.35
                                   rust-clap-builder-4.5.35
                                   rust-clap-derive-4.5.32
                                   rust-clap-lex-0.7.4
                                   rust-clru-0.6.2
                                   rust-color-eyre-0.6.3
                                   rust-colorchoice-1.0.3
                                   rust-core-foundation-0.9.4
                                   rust-core-foundation-0.10.0
                                   rust-core-foundation-sys-0.8.7
                                   rust-crc32fast-1.4.2
                                   rust-crossbeam-channel-0.5.14
                                   rust-crossbeam-deque-0.8.6
                                   rust-crossbeam-epoch-0.9.18
                                   rust-crossbeam-utils-0.8.21
                                   rust-cvss-2.0.0
                                   rust-deranged-0.4.1
                                   rust-display-error-chain-0.2.2
                                   rust-displaydoc-0.2.5
                                   rust-dunce-1.0.5
                                   rust-either-1.15.0
                                   rust-encoding-rs-0.8.35
                                   rust-equivalent-1.0.2
                                   rust-errno-0.3.11
                                   rust-eyre-0.6.12
                                   rust-faster-hex-0.9.0
                                   rust-fastrand-2.3.0
                                   rust-filetime-0.2.25
                                   rust-fixedbitset-0.4.2
                                   rust-flate2-1.1.1
                                   rust-fnv-1.0.7
                                   rust-form-urlencoded-1.2.1
                                   rust-fs-err-2.11.0
                                   rust-futures-channel-0.3.31
                                   rust-futures-core-0.3.31
                                   rust-futures-io-0.3.31
                                   rust-futures-sink-0.3.31
                                   rust-futures-task-0.3.31
                                   rust-futures-util-0.3.31
                                   rust-getrandom-0.2.15
                                   rust-getrandom-0.3.2
                                   rust-gimli-0.28.1
                                   rust-gix-0.70.0
                                   rust-gix-actor-0.33.2
                                   rust-gix-attributes-0.24.0
                                   rust-gix-bitmap-0.2.14
                                   rust-gix-chunk-0.4.11
                                   rust-gix-command-0.4.1
                                   rust-gix-commitgraph-0.26.0
                                   rust-gix-config-0.43.0
                                   rust-gix-config-value-0.14.12
                                   rust-gix-credentials-0.27.0
                                   rust-gix-date-0.9.4
                                   rust-gix-diff-0.50.0
                                   rust-gix-discover-0.38.0
                                   rust-gix-features-0.40.0
                                   rust-gix-filter-0.17.0
                                   rust-gix-fs-0.13.0
                                   rust-gix-glob-0.18.0
                                   rust-gix-hash-0.16.0
                                   rust-gix-hashtable-0.7.0
                                   rust-gix-ignore-0.13.0
                                   rust-gix-index-0.38.0
                                   rust-gix-lock-16.0.0
                                   rust-gix-negotiate-0.18.0
                                   rust-gix-object-0.47.0
                                   rust-gix-odb-0.67.0
                                   rust-gix-pack-0.57.0
                                   rust-gix-packetline-0.18.4
                                   rust-gix-packetline-blocking-0.18.3
                                   rust-gix-path-0.10.15
                                   rust-gix-pathspec-0.9.0
                                   rust-gix-prompt-0.9.1
                                   rust-gix-protocol-0.48.0
                                   rust-gix-quote-0.4.15
                                   rust-gix-ref-0.50.0
                                   rust-gix-refspec-0.28.0
                                   rust-gix-revision-0.32.0
                                   rust-gix-revwalk-0.18.0
                                   rust-gix-sec-0.10.12
                                   rust-gix-shallow-0.2.0
                                   rust-gix-submodule-0.17.0
                                   rust-gix-tempfile-16.0.0
                                   rust-gix-trace-0.1.12
                                   rust-gix-transport-0.45.0
                                   rust-gix-traverse-0.44.0
                                   rust-gix-url-0.29.0
                                   rust-gix-utils-0.1.14
                                   rust-gix-validate-0.9.4
                                   rust-gix-worktree-0.39.0
                                   rust-gix-worktree-state-0.17.0
                                   rust-h2-0.4.8
                                   rust-hashbrown-0.14.5
                                   rust-hashbrown-0.15.2
                                   rust-heck-0.5.0
                                   rust-home-0.5.11
                                   rust-http-1.3.1
                                   rust-http-body-1.0.1
                                   rust-http-body-util-0.1.3
                                   rust-httparse-1.10.1
                                   rust-hyper-1.6.0
                                   rust-hyper-rustls-0.27.5
                                   rust-hyper-util-0.1.11
                                   rust-icu-collections-1.5.0
                                   rust-icu-locid-1.5.0
                                   rust-icu-locid-transform-1.5.0
                                   rust-icu-locid-transform-data-1.5.1
                                   rust-icu-normalizer-1.5.0
                                   rust-icu-normalizer-data-1.5.1
                                   rust-icu-properties-1.5.1
                                   rust-icu-properties-data-1.5.1
                                   rust-icu-provider-1.5.0
                                   rust-icu-provider-macros-1.5.0
                                   rust-ident-case-1.0.1
                                   rust-idna-1.0.3
                                   rust-idna-adapter-1.2.0
                                   rust-indenter-0.3.3
                                   rust-indexmap-2.9.0
                                   rust-io-close-0.3.7
                                   rust-ipnet-2.11.0
                                   rust-is-terminal-polyfill-1.70.1
                                   rust-itoa-1.0.15
                                   rust-jiff-0.2.5
                                   rust-jiff-static-0.2.5
                                   rust-jiff-tzdb-0.1.4
                                   rust-jiff-tzdb-platform-0.1.3
                                   rust-js-sys-0.3.77
                                   rust-kstring-2.0.2
                                   rust-lazy-static-1.5.0
                                   rust-libc-0.2.171
                                   rust-libredox-0.1.3
                                   rust-linux-raw-sys-0.4.15
                                   rust-linux-raw-sys-0.9.3
                                   rust-litemap-0.7.5
                                   rust-lock-api-0.4.12
                                   rust-log-0.4.27
                                   rust-matchers-0.1.0
                                   rust-maybe-async-0.2.10
                                   rust-memchr-2.7.4
                                   rust-memmap2-0.9.5
                                   rust-mime-0.3.17
                                   rust-miniz-oxide-0.6.2
                                   rust-miniz-oxide-0.7.4
                                   rust-miniz-oxide-0.8.7
                                   rust-mio-1.0.3
                                   rust-nu-ansi-term-0.46.0
                                   rust-num-conv-0.1.0
                                   rust-object-0.32.2
                                   rust-once-cell-1.21.3
                                   rust-openssl-probe-0.1.6
                                   rust-overload-0.1.1
                                   rust-owo-colors-3.5.0
                                   rust-parking-lot-0.12.3
                                   rust-parking-lot-core-0.9.10
                                   rust-percent-encoding-2.3.1
                                   rust-petgraph-0.6.5
                                   rust-pin-project-lite-0.2.16
                                   rust-pin-utils-0.1.0
                                   rust-platforms-3.5.0
                                   rust-portable-atomic-1.11.0
                                   rust-portable-atomic-util-0.2.4
                                   rust-powerfmt-0.2.0
                                   rust-ppv-lite86-0.2.21
                                   rust-proc-macro2-1.0.94
                                   rust-prodash-29.0.1
                                   rust-quinn-0.11.7
                                   rust-quinn-proto-0.11.10
                                   rust-quinn-udp-0.5.11
                                   rust-quitters-0.1.0
                                   rust-quote-1.0.40
                                   rust-r-efi-5.2.0
                                   rust-rand-0.9.0
                                   rust-rand-chacha-0.9.0
                                   rust-rand-core-0.9.3
                                   rust-rayon-1.10.0
                                   rust-rayon-core-1.12.1
                                   rust-redox-syscall-0.5.10
                                   rust-regex-1.11.1
                                   rust-regex-automata-0.1.10
                                   rust-regex-automata-0.4.9
                                   rust-regex-syntax-0.6.29
                                   rust-regex-syntax-0.8.5
                                   rust-reqwest-0.12.15
                                   rust-ring-0.17.14
                                   rust-rustc-demangle-0.1.24
                                   rust-rustc-hash-2.1.1
                                   rust-rustc-stable-hash-0.1.2
                                   rust-rustix-0.38.44
                                   rust-rustix-1.0.5
                                   rust-rustls-0.23.25
                                   rust-rustls-native-certs-0.8.1
                                   rust-rustls-pemfile-2.2.0
                                   rust-rustls-pki-types-1.11.0
                                   rust-rustls-webpki-0.103.1
                                   rust-rustsec-0.30.2
                                   rust-rustversion-1.0.20
                                   rust-ryu-1.0.20
                                   rust-same-file-1.0.6
                                   rust-schannel-0.1.27
                                   rust-scopeguard-1.2.0
                                   rust-secrecy-0.10.3
                                   rust-security-framework-3.2.0
                                   rust-security-framework-sys-2.14.0
                                   rust-semver-1.0.26
                                   rust-serde-1.0.219
                                   rust-serde-derive-1.0.219
                                   rust-serde-json-1.0.140
                                   rust-serde-spanned-0.6.8
                                   rust-serde-urlencoded-0.7.1
                                   rust-sha1-smol-1.0.1
                                   rust-sharded-slab-0.1.7
                                   rust-shell-words-1.1.0
                                   rust-shlex-1.3.0
                                   rust-slab-0.4.9
                                   rust-smallvec-1.15.0
                                   rust-smol-str-0.3.2
                                   rust-socket2-0.5.9
                                   rust-stable-deref-trait-1.2.0
                                   rust-static-assertions-1.1.0
                                   rust-strsim-0.11.1
                                   rust-subtle-2.6.1
                                   rust-syn-1.0.109
                                   rust-syn-2.0.100
                                   rust-sync-wrapper-1.0.2
                                   rust-synstructure-0.12.6
                                   rust-synstructure-0.13.1
                                   rust-system-configuration-0.6.1
                                   rust-system-configuration-sys-0.6.0
                                   rust-tame-index-0.18.1
                                   rust-tempfile-3.19.1
                                   rust-termcolor-1.4.1
                                   rust-thiserror-1.0.69
                                   rust-thiserror-2.0.12
                                   rust-thiserror-impl-1.0.69
                                   rust-thiserror-impl-2.0.12
                                   rust-thread-local-1.1.8
                                   rust-time-0.3.41
                                   rust-time-core-0.1.4
                                   rust-time-macros-0.2.22
                                   rust-tinystr-0.7.6
                                   rust-tinyvec-1.9.0
                                   rust-tinyvec-macros-0.1.1
                                   rust-tokio-1.44.1
                                   rust-tokio-rustls-0.26.2
                                   rust-tokio-util-0.7.14
                                   rust-toml-0.8.20
                                   rust-toml-span-0.4.1
                                   rust-toml-datetime-0.6.8
                                   rust-toml-edit-0.22.24
                                   rust-topological-sort-0.2.2
                                   rust-tower-0.5.2
                                   rust-tower-layer-0.3.3
                                   rust-tower-service-0.3.3
                                   rust-tracing-0.1.41
                                   rust-tracing-attributes-0.1.28
                                   rust-tracing-core-0.1.33
                                   rust-tracing-log-0.2.0
                                   rust-tracing-subscriber-0.3.19
                                   rust-try-lock-0.2.5
                                   rust-twox-hash-2.1.0
                                   rust-uluru-3.1.0
                                   rust-unicode-bom-2.0.3
                                   rust-unicode-ident-1.0.18
                                   rust-unicode-normalization-0.1.24
                                   rust-unicode-xid-0.2.6
                                   rust-untrusted-0.9.0
                                   rust-url-2.5.4
                                   rust-utf16-iter-1.0.5
                                   rust-utf8-iter-1.0.4
                                   rust-utf8parse-0.2.2
                                   rust-valuable-0.1.1
                                   rust-version-check-0.9.5
                                   rust-wait-timeout-0.2.1
                                   rust-walkdir-2.5.0
                                   rust-want-0.3.1
                                   rust-wasi-0.11.0+wasi-snapshot-preview1
                                   rust-wasi-0.14.2+wasi-0.2.4
                                   rust-wasm-bindgen-0.2.100
                                   rust-wasm-bindgen-backend-0.2.100
                                   rust-wasm-bindgen-futures-0.4.50
                                   rust-wasm-bindgen-macro-0.2.100
                                   rust-wasm-bindgen-macro-support-0.2.100
                                   rust-wasm-bindgen-shared-0.2.100
                                   rust-wasmparser-0.207.0
                                   rust-web-sys-0.3.77
                                   rust-web-time-1.1.0
                                   rust-winapi-0.3.9
                                   rust-winapi-i686-pc-windows-gnu-0.4.0
                                   rust-winapi-util-0.1.9
                                   rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                   rust-windows-link-0.1.1
                                   rust-windows-registry-0.4.0
                                   rust-windows-result-0.3.2
                                   rust-windows-strings-0.3.1
                                   rust-windows-sys-0.52.0
                                   rust-windows-sys-0.59.0
                                   rust-windows-targets-0.52.6
                                   rust-windows-targets-0.53.0
                                   rust-windows-aarch64-gnullvm-0.52.6
                                   rust-windows-aarch64-gnullvm-0.53.0
                                   rust-windows-aarch64-msvc-0.52.6
                                   rust-windows-aarch64-msvc-0.53.0
                                   rust-windows-i686-gnu-0.52.6
                                   rust-windows-i686-gnu-0.53.0
                                   rust-windows-i686-gnullvm-0.52.6
                                   rust-windows-i686-gnullvm-0.53.0
                                   rust-windows-i686-msvc-0.52.6
                                   rust-windows-i686-msvc-0.53.0
                                   rust-windows-x86-64-gnu-0.52.6
                                   rust-windows-x86-64-gnu-0.53.0
                                   rust-windows-x86-64-gnullvm-0.52.6
                                   rust-windows-x86-64-gnullvm-0.53.0
                                   rust-windows-x86-64-msvc-0.52.6
                                   rust-windows-x86-64-msvc-0.53.0
                                   rust-winnow-0.6.26
                                   rust-winnow-0.7.4
                                   rust-wit-bindgen-rt-0.39.0
                                   rust-write16-1.0.0
                                   rust-writeable-0.5.5
                                   rust-yoke-0.7.5
                                   rust-yoke-derive-0.7.5
                                   rust-zerocopy-0.7.35
                                   rust-zerocopy-0.8.24
                                   rust-zerocopy-derive-0.7.35
                                   rust-zerocopy-derive-0.8.24
                                   rust-zerofrom-0.1.6
                                   rust-zerofrom-derive-0.1.6
                                   rust-zeroize-1.8.1
                                   rust-zerovec-0.10.4
                                   rust-zerovec-derive-0.10.3))
                     (cargo-bloat =>
                                  (list rust-aho-corasick-1.1.3
                                   rust-binfarce-0.2.1
                                   rust-bumpalo-3.17.0
                                   rust-cfg-if-1.0.0
                                   rust-fallible-iterator-0.2.0
                                   rust-js-sys-0.3.77
                                   rust-json-0.12.4
                                   rust-libc-0.2.172
                                   rust-log-0.4.27
                                   rust-memchr-2.7.4
                                   rust-memmap2-0.9.5
                                   rust-multimap-0.10.1
                                   rust-once-cell-1.21.3
                                   rust-pdb-0.8.0
                                   rust-pico-args-0.5.0
                                   rust-proc-macro2-1.0.95
                                   rust-quote-1.0.40
                                   rust-regex-1.11.1
                                   rust-regex-automata-0.4.9
                                   rust-regex-syntax-0.8.5
                                   rust-rustversion-1.0.21
                                   rust-scroll-0.11.0
                                   rust-syn-2.0.101
                                   rust-term-size-0.3.2
                                   rust-unicode-ident-1.0.18
                                   rust-uuid-1.17.0
                                   rust-wasm-bindgen-0.2.100
                                   rust-wasm-bindgen-backend-0.2.100
                                   rust-wasm-bindgen-macro-0.2.100
                                   rust-wasm-bindgen-macro-support-0.2.100
                                   rust-wasm-bindgen-shared-0.2.100
                                   rust-winapi-0.3.9
                                   rust-winapi-i686-pc-windows-gnu-0.4.0
                                   rust-winapi-x86-64-pc-windows-gnu-0.4.0))
                     (cargo-license =>
                                    (list rust-ansi-term-0.12.1
                                     rust-anstream-0.6.18
                                     rust-anstyle-1.0.10
                                     rust-anstyle-parse-0.2.6
                                     rust-anstyle-query-1.1.2
                                     rust-anstyle-wincon-3.0.7
                                     rust-anyhow-1.0.97
                                     rust-camino-1.1.9
                                     rust-cargo-platform-0.1.9
                                     rust-cargo-metadata-0.18.1
                                     rust-clap-4.5.35
                                     rust-clap-builder-4.5.35
                                     rust-clap-derive-4.5.32
                                     rust-clap-lex-0.7.4
                                     rust-colorchoice-1.0.3
                                     rust-csv-1.3.1
                                     rust-csv-core-0.1.12
                                     rust-either-1.15.0
                                     rust-equivalent-1.0.2
                                     rust-getopts-0.2.21
                                     rust-hashbrown-0.15.2
                                     rust-heck-0.5.0
                                     rust-indexmap-2.9.0
                                     rust-is-terminal-polyfill-1.70.1
                                     rust-itertools-0.12.1
                                     rust-itoa-1.0.15
                                     rust-memchr-2.7.4
                                     rust-once-cell-1.21.3
                                     rust-proc-macro2-1.0.94
                                     rust-quote-1.0.40
                                     rust-ryu-1.0.20
                                     rust-semver-1.0.26
                                     rust-serde-1.0.219
                                     rust-serde-derive-1.0.219
                                     rust-serde-json-1.0.140
                                     rust-serde-spanned-0.6.8
                                     rust-smallvec-1.15.0
                                     rust-spdx-0.10.8
                                     rust-strsim-0.11.1
                                     rust-syn-2.0.100
                                     rust-thiserror-1.0.69
                                     rust-thiserror-impl-1.0.69
                                     rust-toml-0.8.20
                                     rust-toml-datetime-0.6.8
                                     rust-toml-edit-0.22.24
                                     rust-unicode-ident-1.0.18
                                     rust-unicode-width-0.1.14
                                     rust-utf8parse-0.2.2
                                     rust-winapi-0.3.9
                                     rust-winapi-i686-pc-windows-gnu-0.4.0
                                     rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                     rust-windows-sys-0.59.0
                                     rust-windows-targets-0.52.6
                                     rust-windows-aarch64-gnullvm-0.52.6
                                     rust-windows-aarch64-msvc-0.52.6
                                     rust-windows-i686-gnu-0.52.6
                                     rust-windows-i686-gnullvm-0.52.6
                                     rust-windows-i686-msvc-0.52.6
                                     rust-windows-x86-64-gnu-0.52.6
                                     rust-windows-x86-64-gnullvm-0.52.6
                                     rust-windows-x86-64-msvc-0.52.6
                                     rust-winnow-0.7.4))
                     (cargo-machete =>
                                    (list rust-aho-corasick-1.1.3
                                          rust-anyhow-1.0.97
                                          rust-argh-0.1.13
                                          rust-argh-derive-0.1.13
                                          rust-argh-shared-0.1.13
                                          rust-bstr-1.12.0
                                          rust-camino-1.1.9
                                          rust-cargo-platform-0.1.9
                                          rust-cargo-metadata-0.19.2
                                          rust-cargo-toml-0.21.0
                                          rust-cfg-if-1.0.0
                                          rust-crossbeam-deque-0.8.6
                                          rust-crossbeam-epoch-0.9.18
                                          rust-crossbeam-utils-0.8.21
                                          rust-either-1.15.0
                                          rust-encoding-rs-0.8.35
                                          rust-encoding-rs-io-0.1.7
                                          rust-env-logger-0.10.2
                                          rust-equivalent-1.0.2
                                          rust-globset-0.4.16
                                          rust-grep-0.3.2
                                          rust-grep-cli-0.1.11
                                          rust-grep-matcher-0.1.7
                                          rust-grep-printer-0.2.2
                                          rust-grep-regex-0.1.13
                                          rust-grep-searcher-0.1.14
                                          rust-hashbrown-0.15.2
                                          rust-hermit-abi-0.5.0
                                          rust-humantime-2.2.0
                                          rust-ignore-0.4.23
                                          rust-indexmap-2.9.0
                                          rust-is-terminal-0.4.16
                                          rust-itoa-1.0.15
                                          rust-libc-0.2.171
                                          rust-log-0.4.27
                                          rust-memchr-2.7.4
                                          rust-memmap2-0.9.5
                                          rust-pretty-env-logger-0.5.0
                                          rust-proc-macro2-1.0.94
                                          rust-quote-1.0.40
                                          rust-rayon-1.10.0
                                          rust-rayon-core-1.12.1
                                          rust-regex-1.11.1
                                          rust-regex-automata-0.4.9
                                          rust-regex-syntax-0.8.5
                                          rust-rust-fuzzy-search-0.1.1
                                          rust-ryu-1.0.20
                                          rust-same-file-1.0.6
                                          rust-semver-1.0.26
                                          rust-serde-1.0.219
                                          rust-serde-derive-1.0.219
                                          rust-serde-json-1.0.140
                                          rust-serde-spanned-0.6.8
                                          rust-syn-2.0.100
                                          rust-termcolor-1.4.1
                                          rust-thiserror-2.0.12
                                          rust-thiserror-impl-2.0.12
                                          rust-toml-0.8.20
                                          rust-toml-datetime-0.6.8
                                          rust-toml-edit-0.22.24
                                          rust-unicode-ident-1.0.18
                                          rust-walkdir-2.5.0
                                          rust-winapi-util-0.1.9
                                          rust-windows-sys-0.59.0
                                          rust-windows-targets-0.52.6
                                          rust-windows-aarch64-gnullvm-0.52.6
                                          rust-windows-aarch64-msvc-0.52.6
                                          rust-windows-i686-gnu-0.52.6
                                          rust-windows-i686-gnullvm-0.52.6
                                          rust-windows-i686-msvc-0.52.6
                                          rust-windows-x86-64-gnu-0.52.6
                                          rust-windows-x86-64-gnullvm-0.52.6
                                          rust-windows-x86-64-msvc-0.52.6
                                          rust-winnow-0.7.6))
                     (cargo-nextest =>
                                    (list rust-addr2line-0.24.2
                                     rust-adler2-2.0.0
                                     rust-ahash-0.8.12
                                     rust-aho-corasick-1.1.3
                                     rust-android-tzdata-0.1.1
                                     rust-android-system-properties-0.1.5
                                     rust-anstream-0.6.18
                                     rust-anstyle-1.0.10
                                     rust-anstyle-parse-0.2.6
                                     rust-anstyle-query-1.1.2
                                     rust-anstyle-wincon-3.0.8
                                     rust-anyhow-1.0.98
                                     rust-async-scoped-0.9.0
                                     rust-async-stream-0.3.6
                                     rust-async-stream-impl-0.3.6
                                     rust-async-trait-0.1.88
                                     rust-atomic-waker-1.1.2
                                     rust-atomicwrites-0.4.4
                                     rust-autocfg-1.4.0
                                     rust-axum-0.7.9
                                     rust-axum-core-0.4.5
                                     rust-backtrace-0.3.75
                                     rust-backtrace-ext-0.2.1
                                     rust-base64-0.21.7
                                     rust-base64-0.22.1
                                     rust-base64ct-1.7.3
                                     rust-bitflags-2.9.1
                                     rust-block-buffer-0.10.4
                                     rust-borsh-1.5.7
                                     rust-bstr-1.12.0
                                     rust-bumpalo-3.17.0
                                     rust-byteorder-1.5.0
                                     rust-bytes-1.10.1
                                     rust-camino-1.1.9
                                     rust-camino-tempfile-1.4.1
                                     rust-cargo-platform-0.2.0
                                     rust-cargo-util-schemas-0.2.0
                                     rust-cargo-metadata-0.20.0
                                     rust-cc-1.2.25
                                     rust-cfg-expr-0.18.0
                                     rust-cfg-if-1.0.0
                                     rust-cfg-aliases-0.2.1
                                     rust-chrono-0.4.41
                                     rust-clap-4.5.39
                                     rust-clap-builder-4.5.39
                                     rust-clap-derive-4.5.32
                                     rust-clap-lex-0.7.4
                                     rust-color-eyre-0.6.5
                                     rust-color-spantrace-0.3.0
                                     rust-colorchoice-1.0.3
                                     rust-config-0.15.11
                                     rust-console-0.15.11
                                     rust-console-api-0.8.1
                                     rust-console-subscriber-0.4.1
                                     rust-const-oid-0.9.6
                                     rust-convert-case-0.7.1
                                     rust-core-foundation-0.9.4
                                     rust-core-foundation-sys-0.8.7
                                     rust-cpufeatures-0.2.17
                                     rust-crc32fast-1.4.2
                                     rust-crossbeam-channel-0.5.15
                                     rust-crossbeam-utils-0.8.21
                                     rust-crossterm-0.29.0
                                     rust-crossterm-winapi-0.9.1
                                     rust-crypto-common-0.1.6
                                     rust-curve25519-dalek-4.1.3
                                     rust-curve25519-dalek-derive-0.1.1
                                     rust-debug-ignore-1.0.5
                                     rust-der-0.7.10
                                     rust-derive-where-1.4.0
                                     rust-derive-more-2.0.1
                                     rust-derive-more-impl-2.0.1
                                     rust-dialoguer-0.11.0
                                     rust-digest-0.10.7
                                     rust-document-features-0.2.11
                                     rust-duct-0.13.7
                                     rust-dunce-1.0.5
                                     rust-ed25519-2.2.3
                                     rust-ed25519-dalek-2.1.1
                                     rust-either-1.15.0
                                     rust-enable-ansi-support-0.2.1
                                     rust-encode-unicode-1.0.0
                                     rust-equivalent-1.0.2
                                     rust-erased-serde-0.4.6
                                     rust-errno-0.3.12
                                     rust-eyre-0.6.12
                                     rust-fastrand-2.3.0
                                     rust-fiat-crypto-0.2.9
                                     rust-filetime-0.2.25
                                     rust-fixedbitset-0.5.7
                                     rust-flate2-1.1.1
                                     rust-fnv-1.0.7
                                     rust-foreign-types-0.3.2
                                     rust-foreign-types-shared-0.1.1
                                     rust-form-urlencoded-1.2.1
                                     rust-future-queue-0.4.0
                                     rust-futures-0.3.31
                                     rust-futures-channel-0.3.31
                                     rust-futures-core-0.3.31
                                     rust-futures-executor-0.3.31
                                     rust-futures-io-0.3.31
                                     rust-futures-macro-0.3.31
                                     rust-futures-sink-0.3.31
                                     rust-futures-task-0.3.31
                                     rust-futures-util-0.3.31
                                     rust-generic-array-0.14.7
                                     rust-getrandom-0.2.16
                                     rust-getrandom-0.3.3
                                     rust-gimli-0.31.1
                                     rust-globset-0.4.16
                                     rust-guppy-0.17.19
                                     rust-guppy-workspace-hack-0.1.0
                                     rust-h2-0.4.10
                                     rust-hashbrown-0.12.3
                                     rust-hashbrown-0.15.3
                                     rust-hdrhistogram-7.5.4
                                     rust-heck-0.5.0
                                     rust-hex-0.4.3
                                     rust-home-0.5.11
                                     rust-http-1.3.1
                                     rust-http-body-1.0.1
                                     rust-http-body-util-0.1.3
                                     rust-httparse-1.10.1
                                     rust-httpdate-1.0.3
                                     rust-humantime-2.2.0
                                     rust-humantime-serde-1.1.1
                                     rust-hyper-1.6.0
                                     rust-hyper-rustls-0.27.6
                                     rust-hyper-timeout-0.5.2
                                     rust-hyper-tls-0.6.0
                                     rust-hyper-util-0.1.13
                                     rust-iana-time-zone-0.1.63
                                     rust-iana-time-zone-haiku-0.1.2
                                     rust-idna-1.0.3
                                     rust-idna-adapter-1.0.0
                                     rust-indent-write-2.2.0
                                     rust-indenter-0.3.3
                                     rust-indexmap-1.9.3
                                     rust-indexmap-2.9.0
                                     rust-indicatif-0.17.11
                                     rust-ipnet-2.11.0
                                     rust-iri-string-0.7.8
                                     rust-is-ci-1.2.0
                                     rust-is-terminal-polyfill-1.70.1
                                     rust-itertools-0.14.0
                                     rust-itoa-1.0.15
                                     rust-jobserver-0.1.33
                                     rust-js-sys-0.3.77
                                     rust-lazy-static-1.5.0
                                     rust-libc-0.2.172
                                     rust-libredox-0.1.3
                                     rust-linux-raw-sys-0.4.15
                                     rust-linux-raw-sys-0.9.4
                                     rust-litrs-0.4.1
                                     rust-lock-api-0.4.13
                                     rust-log-0.4.27
                                     rust-lru-slab-0.1.2
                                     rust-matchers-0.1.0
                                     rust-matchit-0.7.3
                                     rust-memchr-2.7.4
                                     rust-miette-7.6.0
                                     rust-miette-derive-7.6.0
                                     rust-mime-0.3.17
                                     rust-minimal-lexical-0.2.1
                                     rust-miniz-oxide-0.8.8
                                     rust-mio-1.0.4
                                     rust-mukti-metadata-0.3.0
                                     rust-native-tls-0.2.14
                                     rust-nested-0.1.1
                                     rust-newtype-uuid-1.2.2
                                     rust-nextest-filtering-0.15.0
                                     rust-nextest-metadata-0.12.2
                                     rust-nextest-runner-0.80.0
                                     rust-nextest-workspace-hack-0.1.0
                                     rust-nix-0.30.1
                                     rust-nom-7.1.3
                                     rust-num-traits-0.2.19
                                     rust-number-prefix-0.4.0
                                     rust-object-0.36.7
                                     rust-once-cell-1.21.3
                                     rust-once-cell-polyfill-1.70.1
                                     rust-openssl-0.10.73
                                     rust-openssl-macros-0.1.1
                                     rust-openssl-probe-0.1.6
                                     rust-openssl-sys-0.9.109
                                     rust-ordered-float-2.10.1
                                     rust-os-pipe-1.2.2
                                     rust-owo-colors-4.2.1
                                     rust-parking-lot-0.12.4
                                     rust-parking-lot-core-0.9.11
                                     rust-pathdiff-0.2.3
                                     rust-percent-encoding-2.3.1
                                     rust-petgraph-0.7.1
                                     rust-pin-project-1.1.10
                                     rust-pin-project-internal-1.1.10
                                     rust-pin-project-lite-0.2.16
                                     rust-pin-utils-0.1.0
                                     rust-pkcs8-0.10.2
                                     rust-pkg-config-0.3.32
                                     rust-portable-atomic-1.11.0
                                     rust-ppv-lite86-0.2.21
                                     rust-proc-macro2-1.0.95
                                     rust-prost-0.13.5
                                     rust-prost-derive-0.13.5
                                     rust-prost-types-0.13.5
                                     rust-pulldown-cmark-0.13.0
                                     rust-quick-junit-0.5.1
                                     rust-quick-xml-0.37.5
                                     rust-quinn-0.11.8
                                     rust-quinn-proto-0.11.12
                                     rust-quinn-udp-0.5.12
                                     rust-quote-1.0.40
                                     rust-r-efi-5.2.0
                                     rust-rand-0.8.5
                                     rust-rand-0.9.1
                                     rust-rand-chacha-0.3.1
                                     rust-rand-chacha-0.9.0
                                     rust-rand-core-0.6.4
                                     rust-rand-core-0.9.3
                                     rust-recursion-0.5.2
                                     rust-redox-syscall-0.5.12
                                     rust-regex-1.11.1
                                     rust-regex-automata-0.1.10
                                     rust-regex-automata-0.4.9
                                     rust-regex-syntax-0.6.29
                                     rust-regex-syntax-0.8.5
                                     rust-reqwest-0.12.18
                                     rust-ring-0.17.14
                                     rust-ron-0.8.1
                                     rust-rustc-demangle-0.1.24
                                     rust-rustc-hash-2.1.1
                                     rust-rustc-version-0.4.1
                                     rust-rustix-0.38.44
                                     rust-rustix-1.0.7
                                     rust-rustls-0.23.27
                                     rust-rustls-pki-types-1.12.0
                                     rust-rustls-webpki-0.103.3
                                     rust-rustversion-1.0.21
                                     rust-ryu-1.0.20
                                     rust-schannel-0.1.27
                                     rust-scopeguard-1.2.0
                                     rust-security-framework-2.11.1
                                     rust-security-framework-sys-2.14.0
                                     rust-self-replace-1.5.0
                                     rust-self-update-0.42.0
                                     rust-semver-1.0.26
                                     rust-serde-1.0.219
                                     rust-serde-untagged-0.1.7
                                     rust-serde-value-0.7.0
                                     rust-serde-derive-1.0.219
                                     rust-serde-ignored-0.1.12
                                     rust-serde-json-1.0.140
                                     rust-serde-path-to-error-0.1.17
                                     rust-serde-spanned-0.6.8
                                     rust-serde-urlencoded-0.7.1
                                     rust-sha2-0.10.9
                                     rust-sharded-slab-0.1.7
                                     rust-shared-child-1.0.2
                                     rust-shell-words-1.1.0
                                     rust-shlex-1.3.0
                                     rust-signal-hook-0.3.18
                                     rust-signal-hook-mio-0.2.4
                                     rust-signal-hook-registry-1.4.5
                                     rust-signature-2.2.0
                                     rust-slab-0.4.9
                                     rust-smallvec-1.15.0
                                     rust-smol-str-0.3.2
                                     rust-socket2-0.5.10
                                     rust-spki-0.7.3
                                     rust-static-assertions-1.1.0
                                     rust-strip-ansi-escapes-0.2.1
                                     rust-strsim-0.11.1
                                     rust-subtle-2.6.1
                                     rust-supports-color-3.0.2
                                     rust-supports-hyperlinks-3.1.0
                                     rust-supports-unicode-3.0.0
                                     rust-swrite-0.1.0
                                     rust-syn-2.0.101
                                     rust-sync-wrapper-1.0.2
                                     rust-tar-0.4.44
                                     rust-target-lexicon-0.13.2
                                     rust-target-spec-3.4.2
                                     rust-target-spec-miette-0.4.4
                                     rust-tempfile-3.20.0
                                     rust-terminal-size-0.4.2
                                     rust-textwrap-0.16.2
                                     rust-thiserror-1.0.69
                                     rust-thiserror-2.0.12
                                     rust-thiserror-impl-1.0.69
                                     rust-thiserror-impl-2.0.12
                                     rust-thread-local-1.1.8
                                     rust-tinyvec-1.9.0
                                     rust-tinyvec-macros-0.1.1
                                     rust-tokio-1.45.1
                                     rust-tokio-macros-2.5.0
                                     rust-tokio-native-tls-0.3.1
                                     rust-tokio-rustls-0.26.2
                                     rust-tokio-stream-0.1.17
                                     rust-tokio-util-0.7.15
                                     rust-toml-0.8.22
                                     rust-toml-datetime-0.6.9
                                     rust-toml-edit-0.22.26
                                     rust-toml-write-0.1.1
                                     rust-tonic-0.12.3
                                     rust-tower-0.4.13
                                     rust-tower-0.5.2
                                     rust-tower-http-0.6.5
                                     rust-tower-layer-0.3.3
                                     rust-tower-service-0.3.3
                                     rust-tracing-0.1.41
                                     rust-tracing-attributes-0.1.28
                                     rust-tracing-core-0.1.33
                                     rust-tracing-error-0.2.1
                                     rust-tracing-log-0.2.0
                                     rust-tracing-subscriber-0.3.19
                                     rust-try-lock-0.2.5
                                     rust-typeid-1.0.3
                                     rust-typenum-1.18.0
                                     rust-unicase-2.8.1
                                     rust-unicode-ident-1.0.18
                                     rust-unicode-linebreak-0.1.5
                                     rust-unicode-normalization-0.1.24
                                     rust-unicode-segmentation-1.12.0
                                     rust-unicode-width-0.1.14
                                     rust-unicode-width-0.2.0
                                     rust-unicode-xid-0.2.6
                                     rust-untrusted-0.9.0
                                     rust-url-2.5.4
                                     rust-urlencoding-2.1.3
                                     rust-utf8-iter-1.0.4
                                     rust-utf8parse-0.2.2
                                     rust-uuid-1.17.0
                                     rust-valuable-0.1.1
                                     rust-vcpkg-0.2.15
                                     rust-version-check-0.9.5
                                     rust-vte-0.14.1
                                     rust-want-0.3.1
                                     rust-wasi-0.11.0+wasi-snapshot-preview1
                                     rust-wasi-0.14.2+wasi-0.2.4
                                     rust-wasm-bindgen-0.2.100
                                     rust-wasm-bindgen-backend-0.2.100
                                     rust-wasm-bindgen-futures-0.4.50
                                     rust-wasm-bindgen-macro-0.2.100
                                     rust-wasm-bindgen-macro-support-0.2.100
                                     rust-wasm-bindgen-shared-0.2.100
                                     rust-web-sys-0.3.77
                                     rust-web-time-1.1.0
                                     rust-webpki-roots-1.0.0
                                     rust-win32job-2.0.3
                                     rust-winapi-0.3.9
                                     rust-winapi-i686-pc-windows-gnu-0.4.0
                                     rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                     rust-windows-0.61.1
                                     rust-windows-collections-0.2.0
                                     rust-windows-core-0.61.2
                                     rust-windows-future-0.2.1
                                     rust-windows-implement-0.60.0
                                     rust-windows-interface-0.59.1
                                     rust-windows-link-0.1.1
                                     rust-windows-numerics-0.2.0
                                     rust-windows-result-0.3.4
                                     rust-windows-strings-0.4.2
                                     rust-windows-sys-0.42.0
                                     rust-windows-sys-0.52.0
                                     rust-windows-sys-0.59.0
                                     rust-windows-targets-0.52.6
                                     rust-windows-threading-0.1.0
                                     rust-windows-aarch64-gnullvm-0.42.2
                                     rust-windows-aarch64-gnullvm-0.52.6
                                     rust-windows-aarch64-msvc-0.42.2
                                     rust-windows-aarch64-msvc-0.52.6
                                     rust-windows-i686-gnu-0.42.2
                                     rust-windows-i686-gnu-0.52.6
                                     rust-windows-i686-gnullvm-0.52.6
                                     rust-windows-i686-msvc-0.42.2
                                     rust-windows-i686-msvc-0.52.6
                                     rust-windows-x86-64-gnu-0.42.2
                                     rust-windows-x86-64-gnu-0.52.6
                                     rust-windows-x86-64-gnullvm-0.42.2
                                     rust-windows-x86-64-gnullvm-0.52.6
                                     rust-windows-x86-64-msvc-0.42.2
                                     rust-windows-x86-64-msvc-0.52.6
                                     rust-winnow-0.7.10
                                     rust-wit-bindgen-rt-0.39.0
                                     rust-xattr-1.5.0
                                     rust-xxhash-rust-0.8.15
                                     rust-zerocopy-0.8.25
                                     rust-zerocopy-derive-0.8.25
                                     rust-zeroize-1.8.1
                                     rust-zipsign-api-0.1.5
                                     rust-zstd-0.13.3
                                     rust-zstd-safe-7.2.4
                                     rust-zstd-sys-2.0.15+zstd.1.5.7))
                     (cargo-readme =>
                                   (list rust-addr2line-0.24.2
                                    rust-adler2-2.0.0
                                    rust-aho-corasick-1.1.3
                                    rust-anstream-0.6.18
                                    rust-anstyle-1.0.10
                                    rust-anstyle-parse-0.2.6
                                    rust-anstyle-query-1.1.2
                                    rust-anstyle-wincon-3.0.8
                                    rust-assert-cli-0.6.3
                                    rust-backtrace-0.3.75
                                    rust-cfg-if-1.0.0
                                    rust-clap-4.5.39
                                    rust-clap-builder-4.5.39
                                    rust-clap-derive-4.5.32
                                    rust-clap-lex-0.7.4
                                    rust-colorchoice-1.0.3
                                    rust-colored-1.9.4
                                    rust-difference-2.0.0
                                    rust-environment-0.1.1
                                    rust-equivalent-1.0.2
                                    rust-failure-0.1.8
                                    rust-failure-derive-0.1.8
                                    rust-gimli-0.31.1
                                    rust-hashbrown-0.15.3
                                    rust-heck-0.5.0
                                    rust-hermit-abi-0.5.1
                                    rust-indexmap-2.9.0
                                    rust-is-terminal-0.4.16
                                    rust-is-terminal-polyfill-1.70.1
                                    rust-itoa-1.0.15
                                    rust-lazy-static-1.5.0
                                    rust-libc-0.2.172
                                    rust-memchr-2.7.4
                                    rust-miniz-oxide-0.8.8
                                    rust-object-0.36.7
                                    rust-once-cell-polyfill-1.70.1
                                    rust-percent-encoding-2.3.1
                                    rust-proc-macro2-1.0.95
                                    rust-quote-1.0.40
                                    rust-regex-1.11.1
                                    rust-regex-automata-0.4.9
                                    rust-regex-syntax-0.8.5
                                    rust-rustc-demangle-0.1.24
                                    rust-ryu-1.0.20
                                    rust-serde-1.0.219
                                    rust-serde-derive-1.0.219
                                    rust-serde-json-1.0.140
                                    rust-serde-spanned-0.6.8
                                    rust-strsim-0.11.1
                                    rust-syn-1.0.109
                                    rust-syn-2.0.101
                                    rust-synstructure-0.12.6
                                    rust-toml-0.8.22
                                    rust-toml-datetime-0.6.9
                                    rust-toml-edit-0.22.26
                                    rust-toml-write-0.1.1
                                    rust-unicode-ident-1.0.18
                                    rust-unicode-xid-0.2.6
                                    rust-utf8parse-0.2.2
                                    rust-winapi-0.3.9
                                    rust-winapi-i686-pc-windows-gnu-0.4.0
                                    rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                    rust-windows-sys-0.59.0
                                    rust-windows-targets-0.52.6
                                    rust-windows-aarch64-gnullvm-0.52.6
                                    rust-windows-aarch64-msvc-0.52.6
                                    rust-windows-i686-gnu-0.52.6
                                    rust-windows-i686-gnullvm-0.52.6
                                    rust-windows-i686-msvc-0.52.6
                                    rust-windows-x86-64-gnu-0.52.6
                                    rust-windows-x86-64-gnullvm-0.52.6
                                    rust-windows-x86-64-msvc-0.52.6
                                    rust-winnow-0.7.10))
                     (cargo-remark =>
                                   (list rust-ahash-0.8.12
                                    rust-aho-corasick-1.1.3
                                    rust-anstream-0.6.18
                                    rust-anstyle-1.0.10
                                    rust-anstyle-parse-0.2.6
                                    rust-anstyle-query-1.1.2
                                    rust-anstyle-wincon-3.0.8
                                    rust-anyhow-1.0.98
                                    rust-askama-0.12.1
                                    rust-askama-derive-0.12.5
                                    rust-askama-escape-0.10.3
                                    rust-askama-parser-0.2.1
                                    rust-autocfg-1.4.0
                                    rust-basic-toml-0.1.10
                                    rust-bitflags-2.9.1
                                    rust-block-buffer-0.10.4
                                    rust-bstr-1.12.0
                                    rust-bumpalo-3.17.0
                                    rust-byteorder-1.5.0
                                    rust-camino-1.1.9
                                    rust-cargo-platform-0.1.9
                                    rust-cargo-metadata-0.15.4
                                    rust-cc-1.2.25
                                    rust-cfg-if-1.0.0
                                    rust-clap-4.5.39
                                    rust-clap-builder-4.5.39
                                    rust-clap-derive-4.5.32
                                    rust-clap-lex-0.7.4
                                    rust-colorchoice-1.0.3
                                    rust-colored-2.2.0
                                    rust-console-0.15.11
                                    rust-cpufeatures-0.2.17
                                    rust-crossbeam-deque-0.8.6
                                    rust-crossbeam-epoch-0.9.18
                                    rust-crossbeam-utils-0.8.21
                                    rust-crypto-common-0.1.6
                                    rust-digest-0.10.7
                                    rust-either-1.15.0
                                    rust-encode-unicode-1.0.0
                                    rust-env-logger-0.10.2
                                    rust-equivalent-1.0.2
                                    rust-errno-0.3.12
                                    rust-fastrand-2.3.0
                                    rust-fxhash-0.2.1
                                    rust-generic-array-0.14.7
                                    rust-getrandom-0.3.3
                                    rust-hashbrown-0.13.2
                                    rust-hashbrown-0.15.3
                                    rust-heck-0.5.0
                                    rust-hermit-abi-0.5.1
                                    rust-html-escape-0.2.13
                                    rust-humansize-2.1.3
                                    rust-humantime-2.2.0
                                    rust-indexmap-2.9.0
                                    rust-indicatif-0.17.11
                                    rust-insta-1.43.1
                                    rust-is-terminal-0.4.16
                                    rust-is-terminal-polyfill-1.70.1
                                    rust-itoa-1.0.15
                                    rust-js-sys-0.3.77
                                    rust-lazy-static-1.5.0
                                    rust-libc-0.2.172
                                    rust-libm-0.2.15
                                    rust-libmimalloc-sys-0.1.42
                                    rust-linux-raw-sys-0.9.4
                                    rust-log-0.4.27
                                    rust-memchr-2.7.4
                                    rust-mimalloc-0.1.46
                                    rust-mime-0.3.17
                                    rust-mime-guess-2.0.5
                                    rust-minimal-lexical-0.2.1
                                    rust-nom-7.1.3
                                    rust-normpath-1.3.0
                                    rust-num-traits-0.2.19
                                    rust-number-prefix-0.4.0
                                    rust-once-cell-1.21.3
                                    rust-once-cell-polyfill-1.70.1
                                    rust-opener-0.6.1
                                    rust-percent-encoding-2.3.1
                                    rust-portable-atomic-1.11.0
                                    rust-proc-macro2-1.0.95
                                    rust-quote-1.0.40
                                    rust-r-efi-5.2.0
                                    rust-rayon-1.10.0
                                    rust-rayon-core-1.12.1
                                    rust-regex-1.11.1
                                    rust-regex-automata-0.4.9
                                    rust-regex-syntax-0.8.5
                                    rust-rust-embed-6.8.1
                                    rust-rust-embed-impl-6.8.1
                                    rust-rust-embed-utils-7.8.1
                                    rust-rustc-demangle-0.1.24
                                    rust-rustix-1.0.7
                                    rust-ryu-1.0.20
                                    rust-same-file-1.0.6
                                    rust-semver-1.0.26
                                    rust-serde-1.0.219
                                    rust-serde-derive-1.0.219
                                    rust-serde-json-1.0.140
                                    rust-serde-yaml-0.9.34+deprecated
                                    rust-sha2-0.10.9
                                    rust-shlex-1.3.0
                                    rust-similar-2.7.0
                                    rust-strsim-0.11.1
                                    rust-syn-2.0.101
                                    rust-tempfile-3.20.0
                                    rust-termcolor-1.4.1
                                    rust-thiserror-1.0.69
                                    rust-thiserror-impl-1.0.69
                                    rust-typenum-1.18.0
                                    rust-unicase-2.8.1
                                    rust-unicode-ident-1.0.18
                                    rust-unicode-width-0.2.0
                                    rust-unsafe-libyaml-0.2.11
                                    rust-utf8-width-0.1.7
                                    rust-utf8parse-0.2.2
                                    rust-version-check-0.9.5
                                    rust-walkdir-2.5.0
                                    rust-wasi-0.14.2+wasi-0.2.4
                                    rust-wasm-bindgen-0.2.100
                                    rust-wasm-bindgen-backend-0.2.100
                                    rust-wasm-bindgen-macro-0.2.100
                                    rust-wasm-bindgen-macro-support-0.2.100
                                    rust-wasm-bindgen-shared-0.2.100
                                    rust-web-time-1.1.0
                                    rust-winapi-0.3.9
                                    rust-winapi-i686-pc-windows-gnu-0.4.0
                                    rust-winapi-util-0.1.9
                                    rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                    rust-windows-sys-0.59.0
                                    rust-windows-targets-0.52.6
                                    rust-windows-aarch64-gnullvm-0.52.6
                                    rust-windows-aarch64-msvc-0.52.6
                                    rust-windows-i686-gnu-0.52.6
                                    rust-windows-i686-gnullvm-0.52.6
                                    rust-windows-i686-msvc-0.52.6
                                    rust-windows-x86-64-gnu-0.52.6
                                    rust-windows-x86-64-gnullvm-0.52.6
                                    rust-windows-x86-64-msvc-0.52.6
                                    rust-wit-bindgen-rt-0.39.0
                                    rust-zerocopy-0.8.25
                                    rust-zerocopy-derive-0.8.25))
                     (cargo-show-asm =>
                                     (list rust-adler2-2.0.0
                                      rust-aho-corasick-1.1.3
                                      rust-anyhow-1.0.98
                                      rust-ar-0.9.0
                                      rust-bpaf-0.9.20
                                      rust-bpaf-derive-0.5.17
                                      rust-camino-1.1.9
                                      rust-capstone-0.13.0
                                      rust-capstone-sys-0.17.0
                                      rust-cargo-platform-0.1.9
                                      rust-cargo-metadata-0.19.2
                                      rust-cc-1.2.25
                                      rust-cfg-if-1.0.0
                                      rust-crc32fast-1.4.2
                                      rust-flate2-1.1.1
                                      rust-hermit-abi-0.5.1
                                      rust-is-terminal-0.4.16
                                      rust-is-ci-1.2.0
                                      rust-itoa-1.0.15
                                      rust-libc-0.2.172
                                      rust-line-span-0.1.5
                                      rust-memchr-2.7.4
                                      rust-minimal-lexical-0.2.1
                                      rust-miniz-oxide-0.8.8
                                      rust-nom-7.1.3
                                      rust-object-0.36.7
                                      rust-owo-colors-4.2.1
                                      rust-proc-macro2-1.0.95
                                      rust-quote-1.0.40
                                      rust-regex-1.11.1
                                      rust-regex-automata-0.4.9
                                      rust-regex-syntax-0.8.5
                                      rust-rustc-demangle-0.1.24
                                      rust-ruzstd-0.7.3
                                      rust-ryu-1.0.20
                                      rust-same-file-1.0.6
                                      rust-semver-1.0.26
                                      rust-serde-1.0.216
                                      rust-serde-derive-1.0.216
                                      rust-serde-json-1.0.140
                                      rust-shlex-1.3.0
                                      rust-static-assertions-1.1.0
                                      rust-supports-color-2.1.0
                                      rust-supports-color-3.0.2
                                      rust-syn-2.0.101
                                      rust-thiserror-2.0.12
                                      rust-thiserror-impl-2.0.12
                                      rust-twox-hash-1.6.3
                                      rust-unicode-ident-1.0.18
                                      rust-winapi-util-0.1.9
                                      rust-windows-sys-0.59.0
                                      rust-windows-targets-0.52.6
                                      rust-windows-aarch64-gnullvm-0.52.6
                                      rust-windows-aarch64-msvc-0.52.6
                                      rust-windows-i686-gnu-0.52.6
                                      rust-windows-i686-gnullvm-0.52.6
                                      rust-windows-i686-msvc-0.52.6
                                      rust-windows-x86-64-gnu-0.52.6
                                      rust-windows-x86-64-gnullvm-0.52.6
                                      rust-windows-x86-64-msvc-0.52.6))
                     (cargo-with =>
                                 (list rust-addr2line-0.24.2
                                  rust-adler2-2.0.0
                                  rust-aho-corasick-1.1.3
                                  rust-ansi-term-0.12.1
                                  rust-atty-0.2.14
                                  rust-backtrace-0.3.75
                                  rust-bitflags-1.3.2
                                  rust-cfg-if-1.0.0
                                  rust-clap-2.34.0
                                  rust-env-logger-0.6.2
                                  rust-failure-0.1.8
                                  rust-failure-derive-0.1.8
                                  rust-gimli-0.31.1
                                  rust-hermit-abi-0.1.19
                                  rust-humantime-1.3.0
                                  rust-itoa-1.0.15
                                  rust-libc-0.2.172
                                  rust-log-0.4.27
                                  rust-memchr-2.7.4
                                  rust-miniz-oxide-0.8.8
                                  rust-object-0.36.7
                                  rust-proc-macro2-1.0.95
                                  rust-quick-error-1.2.3
                                  rust-quote-1.0.40
                                  rust-regex-1.11.1
                                  rust-regex-automata-0.4.9
                                  rust-regex-syntax-0.8.5
                                  rust-rustc-demangle-0.1.24
                                  rust-ryu-1.0.20
                                  rust-serde-1.0.219
                                  rust-serde-derive-1.0.219
                                  rust-serde-json-1.0.140
                                  rust-strsim-0.8.0
                                  rust-syn-1.0.109
                                  rust-syn-2.0.101
                                  rust-synstructure-0.12.6
                                  rust-termcolor-1.4.1
                                  rust-textwrap-0.11.0
                                  rust-unicode-ident-1.0.18
                                  rust-unicode-width-0.1.14
                                  rust-unicode-xid-0.2.6
                                  rust-vec-map-0.8.2
                                  rust-void-1.0.2
                                  rust-winapi-0.3.9
                                  rust-winapi-i686-pc-windows-gnu-0.4.0
                                  rust-winapi-util-0.1.9
                                  rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                  rust-windows-sys-0.59.0
                                  rust-windows-targets-0.52.6
                                  rust-windows-aarch64-gnullvm-0.52.6
                                  rust-windows-aarch64-msvc-0.52.6
                                  rust-windows-i686-gnu-0.52.6
                                  rust-windows-i686-gnullvm-0.52.6
                                  rust-windows-i686-msvc-0.52.6
                                  rust-windows-x86-64-gnu-0.52.6
                                  rust-windows-x86-64-gnullvm-0.52.6
                                  rust-windows-x86-64-msvc-0.52.6))
                     (castor =>
                             (list rust-aho-corasick-0.7.20
                                   rust-ansi-parser-0.6.5
                                   rust-atk-0.8.0
                                   rust-atk-sys-0.9.1
                                   rust-autocfg-0.1.8
                                   rust-autocfg-1.4.0
                                   rust-bitflags-1.3.2
                                   rust-bitflags-2.9.0
                                   rust-cairo-rs-0.8.1
                                   rust-cairo-sys-rs-0.9.2
                                   rust-cc-1.2.19
                                   rust-cfg-if-0.1.10
                                   rust-cfg-if-1.0.0
                                   rust-cloudabi-0.0.3
                                   rust-core-foundation-0.7.0
                                   rust-core-foundation-sys-0.7.0
                                   rust-dirs-3.0.2
                                   rust-dirs-sys-0.3.7
                                   rust-displaydoc-0.2.5
                                   rust-foreign-types-0.3.2
                                   rust-foreign-types-shared-0.1.1
                                   rust-form-urlencoded-1.2.1
                                   rust-fuchsia-cprng-0.1.1
                                   rust-futures-channel-0.3.31
                                   rust-futures-core-0.3.31
                                   rust-futures-executor-0.3.31
                                   rust-futures-io-0.3.31
                                   rust-futures-macro-0.3.31
                                   rust-futures-task-0.3.31
                                   rust-futures-util-0.3.31
                                   rust-gdk-0.12.1
                                   rust-gdk-pixbuf-0.8.0
                                   rust-gdk-pixbuf-sys-0.9.1
                                   rust-gdk-sys-0.9.1
                                   rust-getrandom-0.2.15
                                   rust-gio-0.8.1
                                   rust-gio-sys-0.9.1
                                   rust-glib-0.9.3
                                   rust-glib-sys-0.9.1
                                   rust-gobject-sys-0.9.1
                                   rust-gtk-0.8.1
                                   rust-gtk-sys-0.9.2
                                   rust-icu-collections-1.5.0
                                   rust-icu-locid-1.5.0
                                   rust-icu-locid-transform-1.5.0
                                   rust-icu-locid-transform-data-1.5.1
                                   rust-icu-normalizer-1.5.0
                                   rust-icu-normalizer-data-1.5.1
                                   rust-icu-properties-1.5.1
                                   rust-icu-properties-data-1.5.1
                                   rust-icu-provider-1.5.0
                                   rust-icu-provider-macros-1.5.0
                                   rust-idna-1.0.3
                                   rust-idna-adapter-1.2.0
                                   rust-lazy-static-1.4.0
                                   rust-libc-0.2.171
                                   rust-libredox-0.1.3
                                   rust-linkify-0.7.0
                                   rust-litemap-0.7.5
                                   rust-log-0.4.27
                                   rust-memchr-2.7.4
                                   rust-native-tls-0.2.4
                                   rust-nom-4.2.3
                                   rust-once-cell-1.21.3
                                   rust-open-2.0.3
                                   rust-openssl-0.10.72
                                   rust-openssl-macros-0.1.1
                                   rust-openssl-probe-0.1.6
                                   rust-openssl-sys-0.9.107
                                   rust-pango-0.8.0
                                   rust-pango-sys-0.9.1
                                   rust-pathdiff-0.2.3
                                   rust-percent-encoding-2.3.1
                                   rust-pin-project-lite-0.2.16
                                   rust-pin-utils-0.1.0
                                   rust-pkg-config-0.3.32
                                   rust-proc-macro2-1.0.94
                                   rust-quote-1.0.40
                                   rust-rand-0.6.5
                                   rust-rand-chacha-0.1.1
                                   rust-rand-core-0.3.1
                                   rust-rand-core-0.4.2
                                   rust-rand-hc-0.1.0
                                   rust-rand-isaac-0.1.1
                                   rust-rand-jitter-0.1.4
                                   rust-rand-os-0.1.3
                                   rust-rand-pcg-0.1.2
                                   rust-rand-xorshift-0.1.1
                                   rust-rdrand-0.4.0
                                   rust-redox-syscall-0.1.57
                                   rust-redox-users-0.4.6
                                   rust-regex-1.5.6
                                   rust-regex-syntax-0.6.29
                                   rust-remove-dir-all-0.5.3
                                   rust-schannel-0.1.27
                                   rust-security-framework-0.4.4
                                   rust-security-framework-sys-0.4.3
                                   rust-serde-1.0.219
                                   rust-serde-derive-1.0.219
                                   rust-shlex-1.3.0
                                   rust-slab-0.4.9
                                   rust-smallvec-1.15.0
                                   rust-smawk-0.3.2
                                   rust-stable-deref-trait-1.2.0
                                   rust-syn-2.0.100
                                   rust-synstructure-0.13.1
                                   rust-tempfile-3.0.8
                                   rust-textwrap-0.14.2
                                   rust-thiserror-1.0.69
                                   rust-thiserror-impl-1.0.69
                                   rust-tinystr-0.7.6
                                   rust-toml-0.5.11
                                   rust-unicode-ident-1.0.18
                                   rust-unicode-linebreak-0.1.5
                                   rust-unicode-width-0.1.14
                                   rust-url-2.5.4
                                   rust-utf16-iter-1.0.5
                                   rust-utf8-iter-1.0.4
                                   rust-vcpkg-0.2.15
                                   rust-version-check-0.1.5
                                   rust-wasi-0.11.0+wasi-snapshot-preview1
                                   rust-winapi-0.3.9
                                   rust-winapi-i686-pc-windows-gnu-0.4.0
                                   rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                   rust-windows-sys-0.59.0
                                   rust-windows-targets-0.52.6
                                   rust-windows-aarch64-gnullvm-0.52.6
                                   rust-windows-aarch64-msvc-0.52.6
                                   rust-windows-i686-gnu-0.52.6
                                   rust-windows-i686-gnullvm-0.52.6
                                   rust-windows-i686-msvc-0.52.6
                                   rust-windows-x86-64-gnu-0.52.6
                                   rust-windows-x86-64-gnullvm-0.52.6
                                   rust-windows-x86-64-msvc-0.52.6
                                   rust-write16-1.0.0
                                   rust-writeable-0.5.5
                                   rust-yoke-0.7.5
                                   rust-yoke-derive-0.7.5
                                   rust-zerofrom-0.1.6
                                   rust-zerofrom-derive-0.1.6
                                   rust-zerovec-0.10.4
                                   rust-zerovec-derive-0.10.3))
                     (catppuccin-whiskers =>
                                          (list rust-aho-corasick-1.1.3
                                           rust-android-tzdata-0.1.1
                                           rust-android-system-properties-0.1.5
                                           rust-anstream-0.6.19
                                           rust-anstyle-1.0.11
                                           rust-anstyle-parse-0.2.7
                                           rust-anstyle-query-1.1.3
                                           rust-anstyle-wincon-3.0.9
                                           rust-anyhow-1.0.98
                                           rust-assert-cmd-2.0.17
                                           rust-autocfg-1.5.0
                                           rust-base64-0.22.1
                                           rust-bitflags-2.9.1
                                           rust-block-buffer-0.10.4
                                           rust-bstr-1.12.0
                                           rust-bumpalo-3.19.0
                                           rust-byteorder-1.5.0
                                           rust-catppuccin-2.5.1
                                           rust-cc-1.2.29
                                           rust-cfg-if-1.0.1
                                           rust-chrono-0.4.41
                                           rust-chrono-tz-0.9.0
                                           rust-chrono-tz-build-0.3.0
                                           rust-clap-4.5.41
                                           rust-clap-stdin-0.5.1
                                           rust-clap-builder-4.5.41
                                           rust-clap-derive-4.5.41
                                           rust-clap-lex-0.7.5
                                           rust-colorchoice-1.0.4
                                           rust-core-foundation-sys-0.8.7
                                           rust-cpufeatures-0.2.17
                                           rust-crossbeam-deque-0.8.6
                                           rust-crossbeam-epoch-0.9.18
                                           rust-crossbeam-utils-0.8.21
                                           rust-crypto-common-0.1.6
                                           rust-css-colors-1.0.1
                                           rust-detect-newline-style-0.1.2
                                           rust-deunicode-1.6.2
                                           rust-difflib-0.4.0
                                           rust-digest-0.10.7
                                           rust-doc-comment-0.3.3
                                           rust-either-1.15.0
                                           rust-encoding-rs-0.8.35
                                           rust-encoding-rs-io-0.1.7
                                           rust-equivalent-1.0.2
                                           rust-errno-0.3.13
                                           rust-fastrand-2.3.0
                                           rust-float-cmp-0.10.0
                                           rust-generic-array-0.14.7
                                           rust-getrandom-0.2.16
                                           rust-getrandom-0.3.3
                                           rust-globset-0.4.16
                                           rust-globwalk-0.9.1
                                           rust-hashbrown-0.15.4
                                           rust-heck-0.5.0
                                           rust-humansize-2.1.3
                                           rust-iana-time-zone-0.1.63
                                           rust-iana-time-zone-haiku-0.1.2
                                           rust-ignore-0.4.23
                                           rust-indexmap-2.10.0
                                           rust-is-terminal-polyfill-1.70.1
                                           rust-itertools-0.13.0
                                           rust-itertools-0.14.0
                                           rust-itoa-1.0.15
                                           rust-js-sys-0.3.77
                                           rust-lazy-static-1.5.0
                                           rust-libc-0.2.174
                                           rust-libm-0.2.15
                                           rust-linux-raw-sys-0.9.4
                                           rust-log-0.4.27
                                           rust-lzma-rust-0.1.7
                                           rust-memchr-2.7.5
                                           rust-normalize-line-endings-0.3.0
                                           rust-num-traits-0.2.19
                                           rust-once-cell-1.21.3
                                           rust-once-cell-polyfill-1.70.1
                                           rust-parse-zoneinfo-0.3.1
                                           rust-paste-1.0.15
                                           rust-percent-encoding-2.3.1
                                           rust-pest-2.8.1
                                           rust-pest-derive-2.8.1
                                           rust-pest-generator-2.8.1
                                           rust-pest-meta-2.8.1
                                           rust-phf-0.11.3
                                           rust-phf-codegen-0.11.3
                                           rust-phf-generator-0.11.3
                                           rust-phf-shared-0.11.3
                                           rust-ppv-lite86-0.2.21
                                           rust-predicates-3.1.3
                                           rust-predicates-core-1.0.9
                                           rust-predicates-tree-1.0.12
                                           rust-prettyplease-0.2.35
                                           rust-proc-macro2-1.0.95
                                           rust-quote-1.0.40
                                           rust-r-efi-5.3.0
                                           rust-rand-0.8.5
                                           rust-rand-chacha-0.3.1
                                           rust-rand-core-0.6.4
                                           rust-regex-1.11.1
                                           rust-regex-automata-0.4.9
                                           rust-regex-syntax-0.8.5
                                           rust-rmp-0.8.14
                                           rust-rmp-serde-1.3.0
                                           rust-rustix-1.0.7
                                           rust-rustversion-1.0.21
                                           rust-ryu-1.0.20
                                           rust-same-file-1.0.6
                                           rust-semver-1.0.26
                                           rust-serde-1.0.219
                                           rust-serde-derive-1.0.219
                                           rust-serde-json-1.0.140
                                           rust-serde-yaml-0.9.34+deprecated
                                           rust-sha2-0.10.9
                                           rust-shlex-1.3.0
                                           rust-siphasher-1.0.1
                                           rust-slug-0.1.6
                                           rust-strsim-0.11.1
                                           rust-syn-2.0.104
                                           rust-tempfile-3.20.0
                                           rust-tera-1.20.0
                                           rust-termtree-0.5.1
                                           rust-thiserror-1.0.69
                                           rust-thiserror-2.0.12
                                           rust-thiserror-impl-1.0.69
                                           rust-thiserror-impl-2.0.12
                                           rust-typenum-1.18.0
                                           rust-ucd-trie-0.1.7
                                           rust-unic-char-property-0.9.0
                                           rust-unic-char-range-0.9.0
                                           rust-unic-common-0.9.0
                                           rust-unic-segment-0.9.0
                                           rust-unic-ucd-segment-0.9.0
                                           rust-unic-ucd-version-0.9.0
                                           rust-unicode-ident-1.0.18
                                           rust-unsafe-libyaml-0.2.11
                                           rust-utf8parse-0.2.2
                                           rust-version-check-0.9.5
                                           rust-wait-timeout-0.2.1
                                           rust-walkdir-2.5.0
                                           rust-wasi-0.11.1+wasi-snapshot-preview1
                                           rust-wasi-0.14.2+wasi-0.2.4
                                           rust-wasm-bindgen-0.2.100
                                           rust-wasm-bindgen-backend-0.2.100
                                           rust-wasm-bindgen-macro-0.2.100
                                           rust-wasm-bindgen-macro-support-0.2.100
                                           rust-wasm-bindgen-shared-0.2.100
                                           rust-winapi-util-0.1.9
                                           rust-windows-core-0.61.2
                                           rust-windows-implement-0.60.0
                                           rust-windows-interface-0.59.1
                                           rust-windows-link-0.1.3
                                           rust-windows-result-0.3.4
                                           rust-windows-strings-0.4.2
                                           rust-windows-sys-0.59.0
                                           rust-windows-sys-0.60.2
                                           rust-windows-targets-0.52.6
                                           rust-windows-targets-0.53.2
                                           rust-windows-aarch64-gnullvm-0.52.6
                                           rust-windows-aarch64-gnullvm-0.53.0
                                           rust-windows-aarch64-msvc-0.52.6
                                           rust-windows-aarch64-msvc-0.53.0
                                           rust-windows-i686-gnu-0.52.6
                                           rust-windows-i686-gnu-0.53.0
                                           rust-windows-i686-gnullvm-0.52.6
                                           rust-windows-i686-gnullvm-0.53.0
                                           rust-windows-i686-msvc-0.52.6
                                           rust-windows-i686-msvc-0.53.0
                                           rust-windows-x86-64-gnu-0.52.6
                                           rust-windows-x86-64-gnu-0.53.0
                                           rust-windows-x86-64-gnullvm-0.52.6
                                           rust-windows-x86-64-gnullvm-0.53.0
                                           rust-windows-x86-64-msvc-0.52.6
                                           rust-windows-x86-64-msvc-0.53.0
                                           rust-wit-bindgen-rt-0.39.0
                                           rust-zerocopy-0.8.26
                                           rust-zerocopy-derive-0.8.26))
                     (circtools =>
                                (list rust-adler2-2.0.0
                                      rust-aho-corasick-1.1.3
                                      rust-alga-0.9.3
                                      rust-android-tzdata-0.1.1
                                      rust-android-system-properties-0.1.5
                                      rust-anyhow-1.0.97
                                      rust-approx-0.3.2
                                      rust-array-macro-1.0.5
                                      rust-autocfg-1.4.0
                                      rust-bio-0.33.0
                                      rust-bio-types-1.0.4
                                      rust-bit-set-0.5.3
                                      rust-bit-vec-0.6.3
                                      rust-buffer-redux-1.0.2
                                      rust-bumpalo-3.17.0
                                      rust-bv-0.11.1
                                      rust-bytecount-0.6.8
                                      rust-byteorder-1.5.0
                                      rust-cc-1.2.18
                                      rust-cfg-if-1.0.0
                                      rust-chrono-0.4.40
                                      rust-cmake-0.1.54
                                      rust-console-0.15.11
                                      rust-core-foundation-sys-0.8.7
                                      rust-crc32fast-1.4.2
                                      rust-crossbeam-utils-0.8.21
                                      rust-csv-1.3.1
                                      rust-csv-core-0.1.12
                                      rust-custom-derive-0.1.7
                                      rust-derive-new-0.6.0
                                      rust-docopt-1.1.1
                                      rust-either-1.15.0
                                      rust-encode-unicode-1.0.0
                                      rust-enum-map-0.6.6
                                      rust-enum-map-derive-0.4.6
                                      rust-feature-probe-0.1.1
                                      rust-fixedbitset-0.2.0
                                      rust-flate2-1.1.1
                                      rust-fnv-1.0.7
                                      rust-fxhash-0.2.1
                                      rust-generic-array-0.13.3
                                      rust-getrandom-0.1.16
                                      rust-getset-0.0.9
                                      rust-hashbrown-0.12.3
                                      rust-heck-0.3.3
                                      rust-heck-0.5.0
                                      rust-iana-time-zone-0.1.63
                                      rust-iana-time-zone-haiku-0.1.2
                                      rust-indexmap-1.9.3
                                      rust-indicatif-0.15.0
                                      rust-itertools-0.9.0
                                      rust-itertools-num-0.1.3
                                      rust-itoa-1.0.15
                                      rust-js-sys-0.3.77
                                      rust-lazy-static-1.5.0
                                      rust-libc-0.2.171
                                      rust-libm-0.2.11
                                      rust-log-0.4.27
                                      rust-matrixmultiply-0.2.4
                                      rust-memchr-2.7.4
                                      rust-miniz-oxide-0.8.7
                                      rust-multimap-0.8.3
                                      rust-nalgebra-0.19.0
                                      rust-ndarray-0.14.0
                                      rust-newtype-derive-0.1.6
                                      rust-num-complex-0.2.4
                                      rust-num-complex-0.3.1
                                      rust-num-integer-0.1.46
                                      rust-num-rational-0.2.4
                                      rust-num-traits-0.2.19
                                      rust-number-prefix-0.3.0
                                      rust-once-cell-1.21.3
                                      rust-ordered-float-1.1.1
                                      rust-petgraph-0.5.1
                                      rust-ppv-lite86-0.2.21
                                      rust-proc-macro2-1.0.94
                                      rust-quote-1.0.40
                                      rust-rand-0.7.3
                                      rust-rand-chacha-0.2.2
                                      rust-rand-core-0.5.1
                                      rust-rand-distr-0.2.2
                                      rust-rand-hc-0.2.0
                                      rust-rawpointer-0.2.1
                                      rust-regex-1.11.1
                                      rust-regex-automata-0.4.9
                                      rust-regex-syntax-0.8.5
                                      rust-rustc-version-0.1.7
                                      rust-rustversion-1.0.20
                                      rust-ryu-1.0.20
                                      rust-scoped-threadpool-0.1.9
                                      rust-semver-0.1.20
                                      rust-seq-io-0.3.4
                                      rust-serde-1.0.219
                                      rust-serde-derive-1.0.219
                                      rust-shlex-1.3.0
                                      rust-statrs-0.13.0
                                      rust-strsim-0.10.0
                                      rust-strum-0.20.0
                                      rust-strum-macros-0.20.1
                                      rust-strum-macros-0.26.4
                                      rust-syn-1.0.109
                                      rust-syn-2.0.100
                                      rust-thiserror-1.0.69
                                      rust-thiserror-impl-1.0.69
                                      rust-triple-accel-0.3.4
                                      rust-typenum-1.18.0
                                      rust-unicode-ident-1.0.18
                                      rust-unicode-segmentation-1.12.0
                                      rust-unicode-width-0.2.0
                                      rust-vec-map-0.8.2
                                      rust-wasi-0.9.0+wasi-snapshot-preview1
                                      rust-wasm-bindgen-0.2.100
                                      rust-wasm-bindgen-backend-0.2.100
                                      rust-wasm-bindgen-macro-0.2.100
                                      rust-wasm-bindgen-macro-support-0.2.100
                                      rust-wasm-bindgen-shared-0.2.100
                                      rust-windows-core-0.61.0
                                      rust-windows-implement-0.60.0
                                      rust-windows-interface-0.59.1
                                      rust-windows-link-0.1.1
                                      rust-windows-result-0.3.2
                                      rust-windows-strings-0.4.0
                                      rust-windows-sys-0.59.0
                                      rust-windows-targets-0.52.6
                                      rust-windows-aarch64-gnullvm-0.52.6
                                      rust-windows-aarch64-msvc-0.52.6
                                      rust-windows-i686-gnu-0.52.6
                                      rust-windows-i686-gnullvm-0.52.6
                                      rust-windows-i686-msvc-0.52.6
                                      rust-windows-x86-64-gnu-0.52.6
                                      rust-windows-x86-64-gnullvm-0.52.6
                                      rust-windows-x86-64-msvc-0.52.6
                                      rust-zerocopy-0.8.24
                                      rust-zerocopy-derive-0.8.24))
                     (clamav =>
                             (list rust-adler2-2.0.0
                                   rust-adler32-1.2.0
                                   rust-aho-corasick-1.1.3
                                   rust-android-tzdata-0.1.1
                                   rust-android-system-properties-0.1.5
                                   rust-autocfg-1.4.0
                                   rust-base64-0.21.7
                                   rust-bindgen-0.65.1
                                   rust-bit-field-0.10.2
                                   rust-bitflags-1.3.2
                                   rust-bitflags-2.9.0
                                   rust-block-buffer-0.10.4
                                   rust-bumpalo-3.17.0
                                   rust-bytemuck-1.22.0
                                   rust-byteorder-1.5.0
                                   rust-bytes-1.10.1
                                   rust-bzip2-rs-0.1.2
                                   rust-cbindgen-0.25.0
                                   rust-cc-1.2.18
                                   rust-cexpr-0.6.0
                                   rust-cfg-if-1.0.0
                                   rust-chrono-0.4.40
                                   rust-clang-sys-1.8.1
                                   rust-color-quant-1.1.0
                                   rust-core-foundation-sys-0.8.7
                                   rust-cpufeatures-0.2.17
                                   rust-crc32fast-1.4.2
                                   rust-crossbeam-deque-0.8.6
                                   rust-crossbeam-epoch-0.9.18
                                   rust-crossbeam-utils-0.8.21
                                   rust-crunchy-0.2.3
                                   rust-crypto-common-0.1.6
                                   rust-delharc-0.6.1
                                   rust-digest-0.10.7
                                   rust-either-1.15.0
                                   rust-encoding-rs-0.8.35
                                   rust-enum-primitive-derive-0.2.2
                                   rust-errno-0.3.11
                                   rust-exr-1.73.0
                                   rust-fastrand-2.3.0
                                   rust-fdeflate-0.3.7
                                   rust-flate2-1.1.1
                                   rust-generic-array-0.14.7
                                   rust-getrandom-0.3.2
                                   rust-gif-0.13.1
                                   rust-glob-0.3.2
                                   rust-half-2.5.0
                                   rust-hashbrown-0.12.3
                                   rust-heck-0.4.1
                                   rust-hex-0.4.3
                                   rust-hex-literal-0.4.1
                                   rust-home-0.5.11
                                   rust-iana-time-zone-0.1.63
                                   rust-iana-time-zone-haiku-0.1.2
                                   rust-image-0.24.9
                                   rust-indexmap-1.9.3
                                   rust-inflate-0.4.5
                                   rust-itertools-0.10.5
                                   rust-itoa-1.0.15
                                   rust-jpeg-decoder-0.3.1
                                   rust-js-sys-0.3.77
                                   rust-lazy-static-1.5.0
                                   rust-lazycell-1.3.0
                                   rust-lebe-0.5.2
                                   rust-libc-0.2.171
                                   rust-libloading-0.8.6
                                   rust-linux-raw-sys-0.4.15
                                   rust-linux-raw-sys-0.9.3
                                   rust-log-0.4.27
                                   rust-memchr-2.7.4
                                   rust-minimal-lexical-0.2.1
                                   rust-miniz-oxide-0.8.7
                                   rust-nom-7.1.3
                                   rust-num-complex-0.4.6
                                   rust-num-integer-0.1.46
                                   rust-num-traits-0.2.19
                                   rust-once-cell-1.21.3
                                   rust-onenote-parser-0.3.1.29c0853
                                   rust-paste-1.0.15
                                   rust-peeking-take-while-0.1.2
                                   rust-png-0.17.16
                                   rust-prettyplease-0.2.32
                                   rust-primal-check-0.3.4
                                   rust-proc-macro2-1.0.94
                                   rust-qoi-0.4.1
                                   rust-quote-1.0.40
                                   rust-r-efi-5.2.0
                                   rust-rayon-1.10.0
                                   rust-rayon-core-1.12.1
                                   rust-regex-1.11.1
                                   rust-regex-automata-0.4.9
                                   rust-regex-syntax-0.8.5
                                   rust-rustc-hash-1.1.0
                                   rust-rustdct-0.7.1
                                   rust-rustfft-6.2.0
                                   rust-rustix-0.38.44
                                   rust-rustix-1.0.5
                                   rust-rustversion-1.0.20
                                   rust-ryu-1.0.20
                                   rust-serde-1.0.219
                                   rust-serde-derive-1.0.219
                                   rust-serde-json-1.0.140
                                   rust-sha1-0.10.6
                                   rust-sha2-0.10.8
                                   rust-shlex-1.3.0
                                   rust-simd-adler32-0.3.7
                                   rust-smallvec-1.15.0
                                   rust-strength-reduce-0.2.4
                                   rust-syn-1.0.109
                                   rust-syn-2.0.100
                                   rust-tempfile-3.19.1
                                   rust-thiserror-1.0.69
                                   rust-thiserror-impl-1.0.69
                                   rust-tiff-0.9.1
                                   rust-tinyvec-1.9.0
                                   rust-toml-0.5.11
                                   rust-transpose-0.2.3
                                   rust-typenum-1.18.0
                                   rust-unicode-ident-1.0.18
                                   rust-unicode-segmentation-1.12.0
                                   rust-uuid-1.16.0
                                   rust-version-check-0.9.5
                                   rust-wasi-0.14.2+wasi-0.2.4
                                   rust-wasm-bindgen-0.2.100
                                   rust-wasm-bindgen-backend-0.2.100
                                   rust-wasm-bindgen-macro-0.2.100
                                   rust-wasm-bindgen-macro-support-0.2.100
                                   rust-wasm-bindgen-shared-0.2.100
                                   rust-weezl-0.1.8
                                   rust-which-4.4.2
                                   rust-widestring-1.2.0
                                   rust-windows-core-0.61.0
                                   rust-windows-implement-0.60.0
                                   rust-windows-interface-0.59.1
                                   rust-windows-link-0.1.1
                                   rust-windows-result-0.3.2
                                   rust-windows-strings-0.4.0
                                   rust-windows-sys-0.59.0
                                   rust-windows-targets-0.52.6
                                   rust-windows-aarch64-gnullvm-0.52.6
                                   rust-windows-aarch64-msvc-0.52.6
                                   rust-windows-i686-gnu-0.52.6
                                   rust-windows-i686-gnullvm-0.52.6
                                   rust-windows-i686-msvc-0.52.6
                                   rust-windows-x86-64-gnu-0.52.6
                                   rust-windows-x86-64-gnullvm-0.52.6
                                   rust-windows-x86-64-msvc-0.52.6
                                   rust-wit-bindgen-rt-0.39.0
                                   rust-zune-inflate-0.2.54))
                     (codeberg-cli =>
                                   (list rust-addr2line-0.24.2
                                    rust-adler2-2.0.1
                                    rust-android-system-properties-0.1.5
                                    rust-ansi-str-0.9.0
                                    rust-ansitok-0.3.0
                                    rust-anstream-0.6.20
                                    rust-anstyle-1.0.11
                                    rust-anstyle-parse-0.2.7
                                    rust-anstyle-query-1.1.4
                                    rust-anstyle-wincon-3.0.10
                                    rust-anyhow-1.0.99
                                    rust-arrayvec-0.7.6
                                    rust-atomic-waker-1.1.2
                                    rust-autocfg-1.5.0
                                    rust-backtrace-0.3.75
                                    rust-base64-0.22.1
                                    rust-base64ct-1.8.0
                                    rust-bitflags-1.3.2
                                    rust-bitflags-2.9.4
                                    rust-bumpalo-3.19.0
                                    rust-byteorder-1.5.0
                                    rust-bytes-1.10.1
                                    rust-cc-1.2.37
                                    rust-cesu8-1.1.0
                                    rust-cfg-if-1.0.3
                                    rust-chrono-0.4.42
                                    rust-clap-4.5.47
                                    rust-clap-builder-4.5.47
                                    rust-clap-complete-4.5.58
                                    rust-clap-derive-4.5.47
                                    rust-clap-lex-0.7.5
                                    rust-colorchoice-1.0.4
                                    rust-combine-4.6.7
                                    rust-comfy-table-7.2.1
                                    rust-config-0.15.16
                                    rust-console-0.16.1
                                    rust-core-foundation-0.9.4
                                    rust-core-foundation-0.10.1
                                    rust-core-foundation-sys-0.8.7
                                    rust-crossterm-0.25.0
                                    rust-crossterm-0.29.0
                                    rust-crossterm-winapi-0.9.1
                                    rust-deranged-0.5.3
                                    rust-derive-new-0.7.0
                                    rust-dirs-6.0.0
                                    rust-dirs-sys-0.5.0
                                    rust-displaydoc-0.2.5
                                    rust-document-features-0.2.11
                                    rust-dyn-clone-1.0.20
                                    rust-either-1.15.0
                                    rust-encode-unicode-1.0.0
                                    rust-encoding-rs-0.8.35
                                    rust-equivalent-1.0.2
                                    rust-errno-0.3.14
                                    rust-fastrand-2.3.0
                                    rust-find-msvc-tools-0.1.1
                                    rust-fnv-1.0.7
                                    rust-foreign-types-0.3.2
                                    rust-foreign-types-shared-0.1.1
                                    rust-forgejo-api-0.7.0
                                    rust-form-urlencoded-1.2.2
                                    rust-futures-channel-0.3.31
                                    rust-futures-core-0.3.31
                                    rust-futures-sink-0.3.31
                                    rust-futures-task-0.3.31
                                    rust-futures-util-0.3.31
                                    rust-fxhash-0.2.1
                                    rust-getrandom-0.2.16
                                    rust-getrandom-0.3.3
                                    rust-gimli-0.31.1
                                    rust-git2-0.20.2
                                    rust-h2-0.4.12
                                    rust-hashbrown-0.15.5
                                    rust-heck-0.5.0
                                    rust-http-1.3.1
                                    rust-http-body-1.0.1
                                    rust-http-body-util-0.1.3
                                    rust-httparse-1.10.1
                                    rust-hyper-1.7.0
                                    rust-hyper-rustls-0.27.7
                                    rust-hyper-tls-0.6.0
                                    rust-hyper-util-0.1.17
                                    rust-iana-time-zone-0.1.64
                                    rust-iana-time-zone-haiku-0.1.2
                                    rust-icu-collections-2.0.0
                                    rust-icu-locale-core-2.0.0
                                    rust-icu-normalizer-2.0.0
                                    rust-icu-normalizer-data-2.0.0
                                    rust-icu-properties-2.0.1
                                    rust-icu-properties-data-2.0.1
                                    rust-icu-provider-2.0.0
                                    rust-idna-1.1.0
                                    rust-idna-adapter-1.2.1
                                    rust-indexmap-2.11.3
                                    rust-indicatif-0.18.0
                                    rust-inquire-0.7.5
                                    rust-insta-1.43.2
                                    rust-io-uring-0.7.10
                                    rust-ipnet-2.11.0
                                    rust-iri-string-0.7.8
                                    rust-is-terminal-polyfill-1.70.1
                                    rust-itertools-0.14.0
                                    rust-itoa-1.0.15
                                    rust-jni-0.21.1
                                    rust-jni-sys-0.3.0
                                    rust-jobserver-0.1.34
                                    rust-js-sys-0.3.78
                                    rust-lazy-static-1.5.0
                                    rust-libc-0.2.175
                                    rust-libgit2-sys-0.18.2+1.9.1
                                    rust-libredox-0.1.10
                                    rust-libz-sys-1.1.22
                                    rust-linux-raw-sys-0.11.0
                                    rust-litemap-0.8.0
                                    rust-litrs-0.4.2
                                    rust-lock-api-0.4.13
                                    rust-log-0.4.28
                                    rust-memchr-2.7.5
                                    rust-mime-0.3.17
                                    rust-mime-guess-2.0.5
                                    rust-minimal-lexical-0.2.1
                                    rust-miniz-oxide-0.8.9
                                    rust-mio-0.8.11
                                    rust-mio-1.0.4
                                    rust-native-tls-0.2.14
                                    rust-ndk-context-0.1.1
                                    rust-newline-converter-0.3.0
                                    rust-nom-7.1.3
                                    rust-num-conv-0.1.0
                                    rust-num-traits-0.2.19
                                    rust-objc2-0.6.2
                                    rust-objc2-encode-4.1.0
                                    rust-objc2-foundation-0.3.1
                                    rust-object-0.36.7
                                    rust-once-cell-1.21.3
                                    rust-once-cell-polyfill-1.70.1
                                    rust-openssl-0.10.73
                                    rust-openssl-macros-0.1.1
                                    rust-openssl-probe-0.1.6
                                    rust-openssl-sys-0.9.109
                                    rust-option-ext-0.2.0
                                    rust-parking-lot-0.12.4
                                    rust-parking-lot-core-0.9.11
                                    rust-pathdiff-0.2.3
                                    rust-percent-encoding-2.3.2
                                    rust-pin-project-lite-0.2.16
                                    rust-pin-utils-0.1.0
                                    rust-pkg-config-0.3.32
                                    rust-portable-atomic-1.11.1
                                    rust-potential-utf-0.1.3
                                    rust-powerfmt-0.2.0
                                    rust-proc-macro2-1.0.101
                                    rust-quote-1.0.40
                                    rust-r-efi-5.3.0
                                    rust-redox-syscall-0.5.17
                                    rust-redox-users-0.5.2
                                    rust-reqwest-0.12.23
                                    rust-ring-0.17.14
                                    rust-rustc-demangle-0.1.26
                                    rust-rustix-1.1.2
                                    rust-rustls-0.23.31
                                    rust-rustls-pki-types-1.12.0
                                    rust-rustls-webpki-0.103.6
                                    rust-rustversion-1.0.22
                                    rust-ryu-1.0.20
                                    rust-same-file-1.0.6
                                    rust-schannel-0.1.28
                                    rust-scopeguard-1.2.0
                                    rust-security-framework-2.11.1
                                    rust-security-framework-sys-2.15.0
                                    rust-serde-1.0.225
                                    rust-serde-core-1.0.225
                                    rust-serde-derive-1.0.225
                                    rust-serde-json-1.0.145
                                    rust-serde-spanned-1.0.1
                                    rust-serde-urlencoded-0.7.1
                                    rust-sharded-slab-0.1.7
                                    rust-shlex-1.3.0
                                    rust-signal-hook-0.3.18
                                    rust-signal-hook-mio-0.2.4
                                    rust-signal-hook-registry-1.4.6
                                    rust-similar-2.7.0
                                    rust-slab-0.4.11
                                    rust-smallvec-1.15.1
                                    rust-socket2-0.6.0
                                    rust-soft-assert-0.1.1
                                    rust-stable-deref-trait-1.2.0
                                    rust-strsim-0.11.1
                                    rust-strum-0.27.2
                                    rust-strum-macros-0.27.2
                                    rust-subtle-2.6.1
                                    rust-syn-2.0.106
                                    rust-sync-wrapper-1.0.2
                                    rust-synstructure-0.13.2
                                    rust-system-configuration-0.6.1
                                    rust-system-configuration-sys-0.6.0
                                    rust-tempfile-3.22.0
                                    rust-terminal-size-0.4.3
                                    rust-termsize-0.1.9
                                    rust-thiserror-1.0.69
                                    rust-thiserror-2.0.16
                                    rust-thiserror-impl-1.0.69
                                    rust-thiserror-impl-2.0.16
                                    rust-thread-local-1.1.9
                                    rust-time-0.3.43
                                    rust-time-core-0.1.6
                                    rust-time-macros-0.2.24
                                    rust-tinystr-0.8.1
                                    rust-tokio-1.47.1
                                    rust-tokio-macros-2.5.0
                                    rust-tokio-native-tls-0.3.1
                                    rust-tokio-rustls-0.26.2
                                    rust-tokio-util-0.7.16
                                    rust-toml-0.9.6
                                    rust-toml-datetime-0.7.1
                                    rust-toml-parser-1.0.2
                                    rust-toml-writer-1.0.2
                                    rust-tower-0.5.2
                                    rust-tower-http-0.6.6
                                    rust-tower-layer-0.3.3
                                    rust-tower-service-0.3.3
                                    rust-tracing-0.1.41
                                    rust-tracing-core-0.1.34
                                    rust-tracing-subscriber-0.3.20
                                    rust-try-lock-0.2.5
                                    rust-unicase-2.8.1
                                    rust-unicode-ident-1.0.19
                                    rust-unicode-segmentation-1.12.0
                                    rust-unicode-width-0.1.14
                                    rust-unicode-width-0.2.1
                                    rust-unit-prefix-0.5.1
                                    rust-untrusted-0.9.0
                                    rust-url-2.5.7
                                    rust-utf8-iter-1.0.4
                                    rust-utf8parse-0.2.2
                                    rust-vcpkg-0.2.15
                                    rust-vte-0.14.1
                                    rust-walkdir-2.5.0
                                    rust-want-0.3.1
                                    rust-wasi-0.11.1+wasi-snapshot-preview1
                                    rust-wasi-0.14.7+wasi-0.2.4
                                    rust-wasip2-1.0.1+wasi-0.2.4
                                    rust-wasm-bindgen-0.2.101
                                    rust-wasm-bindgen-backend-0.2.101
                                    rust-wasm-bindgen-futures-0.4.51
                                    rust-wasm-bindgen-macro-0.2.101
                                    rust-wasm-bindgen-macro-support-0.2.101
                                    rust-wasm-bindgen-shared-0.2.101
                                    rust-web-sys-0.3.78
                                    rust-web-time-1.1.0
                                    rust-webbrowser-1.0.5
                                    rust-winapi-0.3.9
                                    rust-winapi-i686-pc-windows-gnu-0.4.0
                                    rust-winapi-util-0.1.11
                                    rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                    rust-windows-core-0.62.0
                                    rust-windows-implement-0.60.0
                                    rust-windows-interface-0.59.1
                                    rust-windows-link-0.1.3
                                    rust-windows-link-0.2.0
                                    rust-windows-registry-0.5.3
                                    rust-windows-result-0.3.4
                                    rust-windows-result-0.4.0
                                    rust-windows-strings-0.4.2
                                    rust-windows-strings-0.5.0
                                    rust-windows-sys-0.45.0
                                    rust-windows-sys-0.48.0
                                    rust-windows-sys-0.52.0
                                    rust-windows-sys-0.59.0
                                    rust-windows-sys-0.60.2
                                    rust-windows-sys-0.61.0
                                    rust-windows-targets-0.42.2
                                    rust-windows-targets-0.48.5
                                    rust-windows-targets-0.52.6
                                    rust-windows-targets-0.53.3
                                    rust-windows-aarch64-gnullvm-0.42.2
                                    rust-windows-aarch64-gnullvm-0.48.5
                                    rust-windows-aarch64-gnullvm-0.52.6
                                    rust-windows-aarch64-gnullvm-0.53.0
                                    rust-windows-aarch64-msvc-0.42.2
                                    rust-windows-aarch64-msvc-0.48.5
                                    rust-windows-aarch64-msvc-0.52.6
                                    rust-windows-aarch64-msvc-0.53.0
                                    rust-windows-i686-gnu-0.42.2
                                    rust-windows-i686-gnu-0.48.5
                                    rust-windows-i686-gnu-0.52.6
                                    rust-windows-i686-gnu-0.53.0
                                    rust-windows-i686-gnullvm-0.52.6
                                    rust-windows-i686-gnullvm-0.53.0
                                    rust-windows-i686-msvc-0.42.2
                                    rust-windows-i686-msvc-0.48.5
                                    rust-windows-i686-msvc-0.52.6
                                    rust-windows-i686-msvc-0.53.0
                                    rust-windows-x86-64-gnu-0.42.2
                                    rust-windows-x86-64-gnu-0.48.5
                                    rust-windows-x86-64-gnu-0.52.6
                                    rust-windows-x86-64-gnu-0.53.0
                                    rust-windows-x86-64-gnullvm-0.42.2
                                    rust-windows-x86-64-gnullvm-0.48.5
                                    rust-windows-x86-64-gnullvm-0.52.6
                                    rust-windows-x86-64-gnullvm-0.53.0
                                    rust-windows-x86-64-msvc-0.42.2
                                    rust-windows-x86-64-msvc-0.48.5
                                    rust-windows-x86-64-msvc-0.52.6
                                    rust-windows-x86-64-msvc-0.53.0
                                    rust-winnow-0.7.13
                                    rust-wit-bindgen-0.46.0
                                    rust-writeable-0.6.1
                                    rust-yoke-0.8.0
                                    rust-yoke-derive-0.8.0
                                    rust-zerofrom-0.1.6
                                    rust-zerofrom-derive-0.1.6
                                    rust-zeroize-1.8.1
                                    rust-zerotrie-0.2.2
                                    rust-zerovec-0.11.4
                                    rust-zerovec-derive-0.11.1))
                     (complgen =>
                               (list rust-ahash-0.7.8
                                     rust-ahash-0.8.11
                                     rust-aho-corasick-1.1.3
                                     rust-annotate-snippets-0.6.1
                                     rust-anstream-0.6.18
                                     rust-anstyle-1.0.10
                                     rust-anstyle-parse-0.2.6
                                     rust-anstyle-query-1.1.2
                                     rust-anstyle-wincon-3.0.7
                                     rust-anyhow-1.0.97
                                     rust-autocfg-1.4.0
                                     rust-bit-set-0.8.0
                                     rust-bit-vec-0.8.0
                                     rust-bitflags-1.3.2
                                     rust-bitflags-2.9.0
                                     rust-bumpalo-3.17.0
                                     rust-bytecount-0.6.8
                                     rust-bytemuck-1.22.0
                                     rust-byteorder-1.5.0
                                     rust-cfg-if-1.0.0
                                     rust-chic-1.2.2
                                     rust-clap-4.5.36
                                     rust-clap-builder-4.5.36
                                     rust-clap-derive-4.5.32
                                     rust-clap-lex-0.7.4
                                     rust-colorchoice-1.0.3
                                     rust-either-1.15.0
                                     rust-env-logger-0.10.2
                                     rust-equivalent-1.0.2
                                     rust-errno-0.3.11
                                     rust-fastrand-2.3.0
                                     rust-fnv-1.0.7
                                     rust-getrandom-0.2.15
                                     rust-getrandom-0.3.2
                                     rust-hashbrown-0.13.2
                                     rust-hashbrown-0.15.2
                                     rust-heck-0.5.0
                                     rust-hermit-abi-0.5.0
                                     rust-htmlescape-0.3.1
                                     rust-humantime-2.2.0
                                     rust-indexmap-2.9.0
                                     rust-instant-0.1.13
                                     rust-is-terminal-0.4.16
                                     rust-is-terminal-polyfill-1.70.1
                                     rust-itertools-0.10.5
                                     rust-lazy-static-1.5.0
                                     rust-libc-0.2.171
                                     rust-linux-raw-sys-0.9.4
                                     rust-lock-api-0.4.12
                                     rust-log-0.4.27
                                     rust-memchr-2.7.4
                                     rust-minimal-lexical-0.2.1
                                     rust-nom-7.1.3
                                     rust-nom-locate-4.2.0
                                     rust-num-traits-0.2.19
                                     rust-once-cell-1.21.3
                                     rust-parking-lot-0.11.2
                                     rust-parking-lot-core-0.8.6
                                     rust-ppv-lite86-0.2.21
                                     rust-proc-macro2-1.0.94
                                     rust-proptest-1.6.0
                                     rust-quick-error-1.2.3
                                     rust-quote-1.0.40
                                     rust-r-efi-5.2.0
                                     rust-railroad-0.2.0
                                     rust-rand-0.8.5
                                     rust-rand-chacha-0.3.1
                                     rust-rand-core-0.6.4
                                     rust-rand-xorshift-0.3.0
                                     rust-redox-syscall-0.2.16
                                     rust-regex-1.11.1
                                     rust-regex-automata-0.4.9
                                     rust-regex-syntax-0.8.5
                                     rust-roaring-0.10.12
                                     rust-rustix-1.0.5
                                     rust-rusty-fork-0.3.0
                                     rust-scopeguard-1.2.0
                                     rust-slice-group-by-0.3.1
                                     rust-smallvec-1.15.0
                                     rust-strsim-0.11.1
                                     rust-syn-2.0.100
                                     rust-tempfile-3.19.1
                                     rust-termcolor-1.4.1
                                     rust-thiserror-1.0.69
                                     rust-thiserror-impl-1.0.69
                                     rust-unarray-0.1.4
                                     rust-unicode-ident-1.0.18
                                     rust-unicode-width-0.1.14
                                     rust-ustr-0.9.0
                                     rust-utf8parse-0.2.2
                                     rust-version-check-0.9.5
                                     rust-wait-timeout-0.2.1
                                     rust-wasi-0.11.0+wasi-snapshot-preview1
                                     rust-wasi-0.14.2+wasi-0.2.4
                                     rust-winapi-0.3.9
                                     rust-winapi-i686-pc-windows-gnu-0.4.0
                                     rust-winapi-util-0.1.9
                                     rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                     rust-windows-sys-0.59.0
                                     rust-windows-targets-0.52.6
                                     rust-windows-aarch64-gnullvm-0.52.6
                                     rust-windows-aarch64-msvc-0.52.6
                                     rust-windows-i686-gnu-0.52.6
                                     rust-windows-i686-gnullvm-0.52.6
                                     rust-windows-i686-msvc-0.52.6
                                     rust-windows-x86-64-gnu-0.52.6
                                     rust-windows-x86-64-gnullvm-0.52.6
                                     rust-windows-x86-64-msvc-0.52.6
                                     rust-wit-bindgen-rt-0.39.0
                                     rust-zerocopy-0.7.35
                                     rust-zerocopy-0.8.24
                                     rust-zerocopy-derive-0.7.35
                                     rust-zerocopy-derive-0.8.24))
                     (cyme =>
                           (list rust-android-tzdata-0.1.1
                                 rust-android-system-properties-0.1.5
                                 rust-anstream-0.6.18
                                 rust-anstyle-1.0.10
                                 rust-anstyle-parse-0.2.6
                                 rust-anstyle-query-1.1.2
                                 rust-anstyle-wincon-3.0.7
                                 rust-assert-json-diff-2.0.2
                                 rust-atomic-polyfill-1.0.3
                                 rust-autocfg-1.4.0
                                 rust-base64-0.13.1
                                 rust-bitflags-1.3.2
                                 rust-bitflags-2.9.0
                                 rust-bumpalo-3.17.0
                                 rust-byteorder-1.5.0
                                 rust-cc-1.2.19
                                 rust-cfg-if-1.0.0
                                 rust-chrono-0.4.40
                                 rust-clap-4.5.36
                                 rust-clap-builder-4.5.36
                                 rust-clap-complete-4.5.47
                                 rust-clap-derive-4.5.32
                                 rust-clap-lex-0.7.4
                                 rust-clap-mangen-0.2.26
                                 rust-colorchoice-1.0.3
                                 rust-colored-2.2.0
                                 rust-core-foundation-sys-0.8.7
                                 rust-critical-section-1.2.0
                                 rust-darling-0.20.11
                                 rust-darling-core-0.20.11
                                 rust-darling-macro-0.20.11
                                 rust-deranged-0.4.0
                                 rust-diff-0.1.13
                                 rust-dirs-4.0.0
                                 rust-dirs-sys-0.3.7
                                 rust-either-1.15.0
                                 rust-errno-0.3.11
                                 rust-fnv-1.0.7
                                 rust-getrandom-0.2.15
                                 rust-glob-0.3.2
                                 rust-hash32-0.2.1
                                 rust-hashbrown-0.12.3
                                 rust-heapless-0.7.17
                                 rust-heck-0.4.1
                                 rust-heck-0.5.0
                                 rust-hermit-abi-0.3.9
                                 rust-hex-0.4.3
                                 rust-iana-time-zone-0.1.63
                                 rust-iana-time-zone-haiku-0.1.2
                                 rust-ident-case-1.0.1
                                 rust-indexmap-1.9.3
                                 rust-io-lifetimes-1.0.11
                                 rust-is-terminal-polyfill-1.70.1
                                 rust-itertools-0.10.5
                                 rust-itoa-1.0.15
                                 rust-js-sys-0.3.77
                                 rust-lazy-static-1.5.0
                                 rust-libc-0.2.171
                                 rust-libredox-0.1.3
                                 rust-libudev-sys-0.1.4
                                 rust-libusb1-sys-0.7.0
                                 rust-linux-raw-sys-0.3.8
                                 rust-linux-raw-sys-0.9.4
                                 rust-lock-api-0.4.12
                                 rust-log-0.4.27
                                 rust-memchr-2.7.4
                                 rust-minimal-lexical-0.2.1
                                 rust-nix-0.27.1
                                 rust-nom-7.1.3
                                 rust-num-conv-0.1.0
                                 rust-num-traits-0.2.19
                                 rust-num-threads-0.1.7
                                 rust-once-cell-1.21.3
                                 rust-phf-0.11.3
                                 rust-phf-codegen-0.11.3
                                 rust-phf-generator-0.11.3
                                 rust-phf-shared-0.11.3
                                 rust-pkg-config-0.3.32
                                 rust-powerfmt-0.2.0
                                 rust-ppv-lite86-0.2.21
                                 rust-proc-macro2-1.0.94
                                 rust-quote-1.0.40
                                 rust-rand-0.8.5
                                 rust-rand-chacha-0.3.1
                                 rust-rand-core-0.6.4
                                 rust-redox-users-0.4.6
                                 rust-roff-0.2.2
                                 rust-rusb-0.9.4
                                 rust-rustc-version-0.4.1
                                 rust-rustix-0.37.28
                                 rust-rustix-1.0.5
                                 rust-rustversion-1.0.20
                                 rust-ryu-1.0.20
                                 rust-scopeguard-1.2.0
                                 rust-semver-1.0.26
                                 rust-serde-1.0.219
                                 rust-serde-derive-1.0.219
                                 rust-serde-json-1.0.140
                                 rust-serde-with-2.3.3
                                 rust-serde-with-macros-2.3.3
                                 rust-shlex-1.3.0
                                 rust-simple-logger-4.3.3
                                 rust-siphasher-1.0.1
                                 rust-spin-0.9.8
                                 rust-stable-deref-trait-1.2.0
                                 rust-strsim-0.11.1
                                 rust-strum-0.24.1
                                 rust-strum-macros-0.24.3
                                 rust-syn-1.0.109
                                 rust-syn-2.0.100
                                 rust-terminal-size-0.2.6
                                 rust-terminal-size-0.4.2
                                 rust-thiserror-1.0.69
                                 rust-thiserror-impl-1.0.69
                                 rust-time-0.3.41
                                 rust-time-core-0.1.4
                                 rust-time-macros-0.2.22
                                 rust-udev-0.8.0
                                 rust-udevrs-0.3.0
                                 rust-unicode-ident-1.0.18
                                 rust-usb-ids-1.2025.2
                                 rust-utf8parse-0.2.2
                                 rust-vcpkg-0.2.15
                                 rust-wasi-0.11.0+wasi-snapshot-preview1
                                 rust-wasm-bindgen-0.2.100
                                 rust-wasm-bindgen-backend-0.2.100
                                 rust-wasm-bindgen-macro-0.2.100
                                 rust-wasm-bindgen-macro-support-0.2.100
                                 rust-wasm-bindgen-shared-0.2.100
                                 rust-winapi-0.3.9
                                 rust-winapi-i686-pc-windows-gnu-0.4.0
                                 rust-winapi-x86-64-pc-windows-gnu-0.4.0
                                 rust-windows-core-0.61.0
                                 rust-windows-implement-0.60.0
                                 rust-windows-interface-0.59.1
                                 rust-windows-link-0.1.1
                                 rust-windows-result-0.3.2
                                 rust-windows-strings-0.4.0
                                 rust-windows-sys-0.48.0
                                 rust-windows-sys-0.59.0
                                 rust-windows-targets-0.48.5
                                 rust-windows-targets-0.52.6
                                 rust-windows-aarch64-gnullvm-0.48.5
                                 rust-windows-aarch64-gnullvm-0.52.6
                                 rust-windows-aarch64-msvc-0.48.5
                                 rust-windows-aarch64-msvc-0.52.6
                                 rust-windows-i686-gnu-0.48.5
                                 rust-windows-i686-gnu-0.52.6
                                 rus